/* OGMDvd - A wrapper library around libdvdread
 * Copyright (C) 2004-2006 Olivier Rolland <billl@users.sf.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "ogmdvd.h"

#include <fcntl.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>

#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/types.h>

#include <glib/gstdio.h>
#include <glib/gi18n-lib.h>

#if defined(HAVE_INTTYPES_H)
#include <inttypes.h>
#elif defined(HAVE_STDINT_H)
#include <stdint.h>
#endif

#include <dvdread/ifo_types.h>
#include <dvdread/ifo_read.h>

#if defined(__FreeBSD__) || defined(__NetBSD__)
#include <sys/cdio.h>
#define CDROMEJECT CDIOCEJECT
#if __FreeBSD_version < 500000
#undef MIN
#undef MAX
#include <sys/param.h>
#include <sys/mount.h>
#endif /* __FreeBSD_version */
#endif /* __FreeBSD__ */

#ifdef __linux__
#include <linux/cdrom.h>
#endif /* __linux__ */

struct _OGMDvdDisc
{
  guint ref;
  gchar *device;
  dvd_reader_t *reader;
  ifo_handle_t *vmg_file;
};

struct _OGMDvdTitle
{
  guint ref;
  guint nr;

  guint8 ttn;
  guint16 pgcn;

  OGMDvdDisc *disc;

  ifo_handle_t *vts_file;
};

struct _OGMDvdAudioStream
{
  guint ref;
  guint nr;

  guint16 id;

  audio_attr_t *attr;

  OGMDvdTitle *title;
};

struct _OGMDvdSubpStream
{
  guint ref;
  guint nr;

  guint16 id;

  subp_attr_t *attr;

  OGMDvdTitle *title;
};

#if DVDREAD_VERSION < 905
unsigned int UDFFindFile (dvd_reader_t *, const char *, unsigned int *);
#endif /* DVDREAD_VERSION */

const gchar *ogmdvd_languages[][3] = 
{
  { "  ", "und", "Undetermined" },
  { "??", "und", "Undetermined" },
  { "ab", "abk", "Abkhazian" },
  { "aa", "aar", "Afar" },
  { "af", "afr", "Afrikaans" },
  { "ak", "aka", "Akan" },
  { "sq", "alb", "Albanian" },
  { "am", "amh", "Amharic" },
  { "ar", "ara", "Arabic" },
  { "an", "arg", "Aragonese" },
  { "hy", "arm", "Armenian" },
  { "as", "asm", "Assamese" },
  { "av", "ava", "Avaric" },
  { "ae", "ave", "Avestan" },
  { "ay", "aym", "Aymara" },
  { "az", "aze", "Azerbaijani" },
  { "bm", "bam", "Bambara" },
  { "ba", "bak", "Bashkir" },
  { "eu", "baq", "Basque" },
  { "be", "bel", "Belarusian" },
  { "bn", "ben", "Bengali" },
  { "bh", "bih", "Bihari" },
  { "bi", "bis", "Bislama" },
  { "bs", "bos", "Bosnian" },
  { "br", "bre", "Breton" },
  { "bg", "bul", "Bulgarian" },
  { "my", "bur", "Burmese" },
  { "ca", "cat", "Catalan" },
  { "ch", "cha", "Chamorro" },
  { "ce", "che", "Chechen" },
  { "ny", "nya", "Chewa; Chichewa; Nyanja" },
  { "zh", "chi", "Chinese" },
  { "za", "zha", "Chuang; Zhuang" },
  { "cu", "chu", "Church Slavonic; Old Bulgarian; Church Slavic;" },
  { "cv", "chv", "Chuvash" },
  { "kw", "cor", "Cornish" },
  { "co", "cos", "Corsican" },
  { "cr", "cre", "Cree" },
  { "hr", "scr", "Croatian" },
  { "cs", "cze", "Czech" },
  { "da", "dan", "Danish" },
  { "de", "ger", "Deutsch" },
  { "dv", "div", "Divehi" },
  { "dz", "dzo", "Dzongkha" },
  { "en", "eng", "English" },
  { "es", "spa", "Espanol" },
  { "eo", "epo", "Esperanto" },
  { "et", "est", "Estonian" },
  { "ee", "ewe", "Ewe" },
  { "fo", "fao", "Faroese" },
  { "fj", "fij", "Fijian" },
  { "fi", "fin", "Finnish" },
  { "fr", "fre", "Francais" },
  { "fy", "fry", "Frisian" },
  { "ff", "ful", "Fulah" },
  { "gd", "gla", "Gaelic; Scottish Gaelic" },
  { "gl", "glg", "Gallegan" },
  { "lg", "lug", "Ganda" },
  { "ka", "geo", "Georgian" },
  { "ki", "kik", "Gikuyu; Kikuyu" },
  { "el", "ell", "Greek" },
  { "kl", "kal", "Greenlandic; Kalaallisut" },
  { "gn", "grn", "Guarani" },
  { "gu", "guj", "Gujarati" },
  { "ha", "hau", "Hausa" },
  { "he", "heb", "Hebrew" },
  { "hz", "her", "Herero" },
  { "hi", "hin", "Hindi" },
  { "ho", "hmo", "Hiri Motu" },
  { "hu", "hun", "Hungarian" },
  { "is", "ice", "Icelandic" },
  { "io", "ido", "Ido" },
  { "ig", "ibo", "Igbo" },
  { "id", "ind", "Indonesian" },
  { "ia", "ina", "Interlingua (International)" },
  { "ie", "ile", "Interlingue" },
  { "iu", "iku", "Inuktitut" },
  { "ik", "ipk", "Inupiaq" },
  { "ga", "gle", "Irish" },
  { "it", "ita", "Italiano" },
  { "ja", "jpn", "Japanese" },
  { "jv", "jav", "Javanese" },
  { "kn", "kan", "Kannada" },
  { "kr", "kau", "Kanuri" },
  { "ks", "kas", "Kashmiri" },
  { "kk", "kaz", "Kazakh" },
  { "km", "khm", "Khmer" },
  { "rw", "kin", "Kinyarwanda" },
  { "ky", "kir", "Kirghiz" },
  { "kv", "kom", "Komi" },
  { "kg", "kon", "Kongo" },
  { "ko", "kor", "Korean" },
  { "ku", "kur", "Kurdish" },
  { "kj", "kua", "Kwanyama, Kuanyama" },
  { "lo", "lao", "Lao" },
  { "la", "lat", "Latin" },
  { "lv", "lav", "Latvian" },
  { "li", "lim", "Limburgish; Limburger; Limburgan" },
  { "ln", "lin", "Lingala" },
  { "lt", "lit", "Lithuanian" },
  { "lu", "lub", "Luba-Katanga" },
  { "lb", "ltz", "Luxembourgish; Letzeburgesch" },
  { "mk", "mac", "Macedonian" },
  { "mg", "mlg", "Malagasy" },
  { "ml", "mal", "Malayalam" },
  { "ms", "may", "Malay" },
  { "mt", "mlt", "Maltese" },
  { "gv", "glv", "Manx" },
  { "mi", "mao", "Maori" },
  { "mr", "mar", "Marathi" },
  { "mh", "mah", "Marshallese" },
  { "mo", "mol", "Moldavian" },
  { "mn", "mon", "Mongolian" },
  { "na", "nau", "Nauru" },
  { "nv", "nav", "Navajo; Navaho" },
  { "ng", "ndo", "Ndonga" },
  { "nl", "nld", "Nederlands" },
  { "ne", "nep", "Nepali" },
  { "se", "sme", "Northern Sami" },
  { "nd", "nde", "North Ndebele" },
  { "nb", "nob", "Norwegian Bokmal; Bokmal, Norwegian" },
  { "no", "nor", "Norwegian" },
  { "nn", "nno", "Norwegian Nynorsk; Nynorsk, Norwegian" },
  { "oj", "oji", "Ojibwa" },
  { "or", "ori", "Oriya" },
  { "om", "orm", "Oromo" },
  { "os", "oss", "Ossetian; Ossetic" },
  { "pi", "pli", "Pali" },
  { "pa", "pan", "Panjabi" },
  { "fa", "per", "Persian" },
  { "pl", "pol", "Polish" },
  { "pt", "por", "Portuguese" },
  { "oc", "oci", "Provencal; Occitan (post 1500)" },
  { "ps", "pus", "Pushto" },
  { "qu", "que", "Quechua" },
  { "rm", "roh", "Raeto-Romance" },
  { "ro", "ron", "Romanian" },
  { "rn", "run", "Rundi" },
  { "ru", "rus", "Russian" },
  { "sm", "smo", "Samoan" },
  { "sg", "sag", "Sango" },
  { "sa", "san", "Sanskrit" },
  { "sc", "srd", "Sardinian" },
  { "sr", "srp", "Serbian" },
  { "sn", "sna", "Shona" },
  { "ii", "iii", "Sichuan Yi" },
  { "sd", "snd", "Sindhi" },
  { "si", "sin", "Sinhalese" },
  { "sk", "slo", "Slovak" },
  { "sl", "slv", "Slovenian" },
  { "so", "som", "Somali" },
  { "st", "sot", "Sotho, Southern" },
  { "nr", "nbl", "South Ndebele" },
  { "su", "sun", "Sundanese" },
  { "sw", "swa", "Swahili" },
  { "ss", "ssw", "Swati" },
  { "sv", "swe", "Swedish" },
  { "tl", "tgl", "Tagalog" },
  { "ty", "tah", "Tahitian" },
  { "tg", "tgk", "Tajik" },
  { "ta", "tam", "Tamil" },
  { "tt", "tat", "Tatar" },
  { "te", "tel", "Telugu" },
  { "th", "tha", "Thai" },
  { "bo", "tib", "Tibetan" },
  { "ti", "tir", "Tigrinya" },
  { "to", "ton", "Tonga (Tonga Islands)" },
  { "ts", "tso", "Tsonga" },
  { "tn", "tsn", "Tswana" },
  { "tr", "tur", "Turkish" },
  { "tk", "tuk", "Turkmen" },
  { "tw", "twi", "Twi" },
  { "ug", "uig", "Uighur" },
  { "uk", "ukr", "Ukrainian" },
  { "ur", "urd", "Urdu" },
  { "uz", "uzb", "Uzbek" },
  { "ve", "ven", "Venda" },
  { "vi", "vie", "Vietnamese" },
  { "vo", "vol", "Volapük" },
  { "wa", "wln", "Walloon" },
  { "cy", "wel", "Welsh" },
  { "wo", "wol", "Wolof" },
  { "xh", "xho", "Xhosa" },
  { "yi", "yid", "Yiddish" },
  { "yo", "yor", "Yoruba" },
  { "zu", "zul", "Zulu" },
  { NULL, NULL,  NULL },
};

guint ogmdvd_nlanguages = sizeof (ogmdvd_languages) / sizeof (ogmdvd_languages[0]) - 1;

/*
 * We keep a trace of all opened drives because, with libdvdread >= 0.9.5
 * a same drive cannot be opened multiple times at the same time.
 */
static GHashTable *opened_drives;

static glong
ogmdvd_time_to_msec (dvd_time_t *dtime)
{
  guint hour, min, sec, frames;
  gfloat fps;

  hour   = ((dtime->hour    & 0xf0) >> 4) * 10 + (dtime->hour    & 0x0f);
  min    = ((dtime->minute  & 0xf0) >> 4) * 10 + (dtime->minute  & 0x0f);
  sec    = ((dtime->second  & 0xf0) >> 4) * 10 + (dtime->second  & 0x0f);
  frames = ((dtime->frame_u & 0x30) >> 4) * 10 + (dtime->frame_u & 0x0f);

  if (((dtime->frame_u & 0xc0) >> 6) == 1)
    fps = 25.0;
  else
    fps = 30000 / 1001.0;

  return hour * 60 * 60 * 1000 + min * 60 * 1000 + sec * 1000 + (gfloat) frames * 1000.0 / fps;
}

static gint64
ogmdvd_get_ifo_size (OGMDvdDisc *disc, guint vts)
{
#if DVDREAD_VERSION >= 905
  dvd_file_t *file;
  gint size;

  file = DVDOpenFile (disc->reader, vts, DVD_READ_INFO_FILE);
  size = DVDFileSize (file);
  DVDCloseFile (file);

  size *= DVD_VIDEO_LB_LEN;
#else /* DVDREAD_VERSION */
  gchar filename[FILENAME_MAX];
  guint size;

  if (vts == 0)
    strncpy (filename, "/VIDEO_TS/VIDEO_TS.IFO", FILENAME_MAX);
  else
    snprintf (filename, FILENAME_MAX, "/VIDEO_TS/VTS_%02u_0.IFO", vts);

  if (!UDFFindFile (disc->reader, filename, &size))
    return -1;
#endif /* DVDREAD_VERSION */

  if (size < 0)
    return 0;

#ifdef DEBUG_CFLAGS
  printf ("IFO size: %d\n", size);
#endif /* DEBUG_CFLAGS */

  return (gint64) size;
}

static gint64
ogmdvd_get_bup_size (OGMDvdDisc *disc, guint vts)
{
#if DVDREAD_VERSION >= 905
  dvd_file_t *file;
  gint size;

  file = DVDOpenFile (disc->reader, vts, DVD_READ_INFO_BACKUP_FILE);
  size = DVDFileSize (file);
  DVDCloseFile (file);

  size *= DVD_VIDEO_LB_LEN;
#else /* DVDREAD_VERSION */
  gchar filename[FILENAME_MAX];
  guint size;

  if (vts == 0)
    strncpy (filename, "/VIDEO_TS/VIDEO_TS.BUP", FILENAME_MAX);
  else
    snprintf (filename, FILENAME_MAX, "/VIDEO_TS/VTS_%02u_0.BUP", vts);

  if (!UDFFindFile (disc->reader, filename, &size))
    return 0;
#endif /* DVDREAD_VERSION */

  if (size < 0)
    return 0;

#ifdef DEBUG_CFLAGS
  printf ("BUP size: %d\n", size);
#endif /* DEBUG_CFLAGS */

  return (gint64) size;
}

static gint64
ogmdvd_get_menu_size (OGMDvdDisc *disc, guint vts)
{
#if DVDREAD_VERSION >= 905
  dvd_file_t *file;
  gint size;

  file = DVDOpenFile (disc->reader, vts, DVD_READ_MENU_VOBS);
  size = DVDFileSize (file);
  DVDCloseFile (file);

  size *= DVD_VIDEO_LB_LEN;
#else /* DVDREAD_VERSION */
  gchar filename[FILENAME_MAX];
  guint size;

  if (vts == 0)
    strncpy (filename, "/VIDEO_TS/VIDEO_TS.VOB", FILENAME_MAX);
  else
    snprintf (filename, FILENAME_MAX, "/VIDEO_TS/VTS_%02u_0.VOB", vts);

  if (!UDFFindFile (disc->reader, filename, &size))
    return 0;
#endif /* DVDREAD_VERSION */

  if (size < 0)
    return 0;

#ifdef DEBUG_CFLAGS
  printf ("Menu size: %d\n", size);
#endif /* DEBUG_CFLAGS */

  return (gint64) size;
}

static gint64
ogmdvd_get_vob_size (OGMDvdDisc *disc, guint vts)
{
  gint64 fullsize;

#if DVDREAD_VERSION >= 905
  dvd_file_t *file;

  file = DVDOpenFile (disc->reader, vts, DVD_READ_TITLE_VOBS);
  fullsize = DVDFileSize (file);
  DVDCloseFile (file);

  fullsize *= DVD_VIDEO_LB_LEN;
#else /* DVDREAD_VERSION */
  gchar filename[FILENAME_MAX];
  guint vob, size;

  if (vts == 0)
    return 0;

  vob = 1; fullsize = 0;
  while (1)
  {
    snprintf (filename, FILENAME_MAX, "/VIDEO_TS/VTS_%02u_%u.VOB", vts, vob++);
    if (!UDFFindFile (disc->reader, filename, &size))
      break;
    fullsize += size;
  }

  if (vob == 1)
    return 0;
#endif /* DVDREAD_VERSION */

  if (fullsize < 0)
    return 0;

#ifdef DEBUG_CFLAGS
  printf ("VOB size: %" G_GINT64_FORMAT "\n", fullsize);
#endif /* DEBUG_CFLAGS */

  return fullsize;
}

static gboolean
ogmdvd_device_tray_is_open (const gchar *device)
{
  gint fd;
#if defined(__linux__) || defined(__FreeBSD__)
  gint status = 0;
#endif
#ifdef __FreeBSD__
  struct ioc_toc_header h;
#endif

  g_return_val_if_fail (device && *device, FALSE);

  fd = g_open (device, O_RDONLY | O_NONBLOCK | O_EXCL, 0);

  if (fd < 0)
    return FALSE;

#if defined(__linux__)
  status = ioctl (fd, CDROM_DRIVE_STATUS, CDSL_CURRENT);
#elif defined(__FreeBSD__)
  status = ioctl (fd, CDIOREADTOCHEADER, &h);
#endif

  close (fd);

#if defined(__linux__)
  return status < 0 ? FALSE : status == CDS_TRAY_OPEN;
#elif defined(__FreeBSD__)
  return status < 0;
#else
  return FALSE;
#endif
}

GQuark
ogmdvd_error_quark (void)
{
  static GQuark quark = 0;

  if (quark == 0)
    quark = g_quark_from_static_string ("ogmdvd-error-quark");

  return quark;
}

OGMDvdDisc *
ogmdvd_disc_open (const gchar *device, GError **error)
{
  OGMDvdDisc *disc;

  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  dvd_reader_t *reader;
  ifo_handle_t *vmg_file;

  if (!device || !*device)
    device = "/dev/dvd";

  if (!opened_drives)
    opened_drives = g_hash_table_new (g_str_hash, g_str_equal);

  disc = g_hash_table_lookup (opened_drives, device);
  if (disc)
  {
    ogmdvd_disc_ref (disc);
    return disc;
  }

  reader = DVDOpen (device);
  if (!reader)
  {
    struct stat buf;

    if (g_stat (device, &buf))
      g_set_error (error, OGMDVD_ERROR, OGMDVD_ERROR_UNKNOWN, _("No such file or directory"));
    else
    {
      if (S_ISBLK(buf.st_mode))
      {
        if (ogmdvd_device_tray_is_open (device))
          g_set_error (error, OGMDVD_ERROR, OGMDVD_ERROR_UNKNOWN, _("Tray seems to be opened"));
        /*
        else if (ogmdvd_drive_has_valid_media (device))
          g_set_error (error, OGMDVD_ERROR, OGMDVD_ERROR_UNKNOWN, _("Device does not contain a valid DVD video"));
        else
          g_set_error (error, OGMDVD_ERROR, OGMDVD_ERROR_UNKNOWN, _("Unknown error"));
        */
          g_set_error (error, OGMDVD_ERROR, OGMDVD_ERROR_UNKNOWN, _("Device does not contain a valid DVD video"));
      }
      else if (S_ISDIR(buf.st_mode))
        g_set_error (error, OGMDVD_ERROR, OGMDVD_ERROR_UNKNOWN, _("Path does not contain a valid DVD structure"));
      else if (S_ISREG(buf.st_mode))
        g_set_error (error, OGMDVD_ERROR, OGMDVD_ERROR_UNKNOWN, _("Directory does not contain a valid DVD structure"));
      else
        g_set_error (error, OGMDVD_ERROR, OGMDVD_ERROR_UNKNOWN, _("No such directory, block device or iso file"));
    }

    return NULL;
  }

  vmg_file = ifoOpen (reader, 0);
  if (!vmg_file)
  {
    DVDClose (reader);
    return NULL;
  }

  disc = g_new0 (OGMDvdDisc, 1);
  disc->device = g_strdup (device);
  disc->vmg_file = vmg_file;
  disc->reader = reader;
  disc->ref = 1;

  g_hash_table_insert (opened_drives, disc->device, disc);

  return disc;
}

void
ogmdvd_disc_ref (OGMDvdDisc *disc)
{
  g_return_if_fail (disc != NULL);

  disc->ref ++;
}

void
ogmdvd_disc_unref (OGMDvdDisc *disc)
{
  g_return_if_fail (disc != NULL);

  if (disc->ref > 0)
  {
    disc->ref --;
    if (disc->ref == 0)
    {
      g_hash_table_remove (opened_drives, disc->device);

      g_free (disc->device);
      disc->device = NULL;

      if (disc->vmg_file)
        ifoClose (disc->vmg_file);
      disc->vmg_file = NULL;

      if (disc->reader)
        DVDClose (disc->reader);
      disc->reader = NULL;

      g_free (disc);
    }
  }
}

gchar *
ogmdvd_disc_get_label (OGMDvdDisc *disc)
{
  gchar label[32];

  g_return_val_if_fail (disc != NULL, NULL);

  if (DVDUDFVolumeInfo (disc->reader, label, 32, NULL, 0) < 0)
    return g_strdup ("Unknown");

  return g_strdup (label);
}

G_CONST_RETURN gchar *
ogmdvd_disc_get_device (OGMDvdDisc *disc)
{
  g_return_val_if_fail (disc != NULL, NULL);

  return disc->device;
}

gint64
ogmdvd_disc_get_vmg_size (OGMDvdDisc *disc)
{
  gint64 fullsize;

  g_return_val_if_fail (disc != NULL, -1);

  fullsize  = ogmdvd_get_ifo_size (disc, 0);
  fullsize += ogmdvd_get_bup_size (disc, 0);
  fullsize += ogmdvd_get_menu_size (disc, 0);

  return fullsize;
}

gint
ogmdvd_disc_get_n_titles (OGMDvdDisc *disc)
{
  g_return_val_if_fail (disc != NULL, -1);

  return disc->vmg_file->tt_srpt->nr_of_srpts;
}

OGMDvdTitle *
ogmdvd_disc_get_nth_title (OGMDvdDisc *disc, guint nr)
{
  OGMDvdTitle *title;
  ifo_handle_t *vts_file;

  g_return_val_if_fail (disc != NULL, NULL);
  g_return_val_if_fail (nr < disc->vmg_file->tt_srpt->nr_of_srpts, NULL);

  if (nr >= disc->vmg_file->tt_srpt->nr_of_srpts)
    return NULL;

  vts_file = ifoOpen (disc->reader, disc->vmg_file->tt_srpt->title[nr].title_set_nr);
  if (!vts_file)
    return NULL;

  ogmdvd_disc_ref (disc);

  title = g_new0 (OGMDvdTitle, 1);
  title->vts_file = vts_file;
  title->disc = disc;
  title->ref = 1;
  title->nr = nr;

  title->ttn = disc->vmg_file->tt_srpt->title[nr].vts_ttn;
  title->pgcn = title->vts_file->vts_ptt_srpt->title[title->ttn - 1].ptt[0].pgcn;

  return title;
}

void
ogmdvd_title_ref (OGMDvdTitle *title)
{
  g_return_if_fail (title != NULL);

  title->ref ++;
}

void
ogmdvd_title_unref (OGMDvdTitle *title)
{
  g_return_if_fail (title != NULL);

  if (title->ref > 0)
  {
    title->ref --;

    if (title->ref == 0)
    {
      if (title->vts_file)
        ifoClose (title->vts_file);
      title->vts_file = NULL;

      if (title->disc)
        ogmdvd_disc_unref (title->disc);
      title->disc = NULL;

      g_free (title);
    }
  }
}

OGMDvdDisc *
ogmdvd_title_get_disc (OGMDvdTitle *title)
{
  g_return_val_if_fail (title != NULL, NULL);

  return title->disc;
}

gint
ogmdvd_title_get_nr (OGMDvdTitle *title)
{
  g_return_val_if_fail (title != NULL, -1);

  return title->nr;
}

gint64
ogmdvd_title_get_vts_size (OGMDvdTitle *title)
{
  OGMDvdDisc *disc;
  gint64 size, fullsize;
  guint vts;

  g_return_val_if_fail (title != NULL, -1);

  disc = ogmdvd_title_get_disc (title);

  vts = disc->vmg_file->tt_srpt->title[title->nr].title_set_nr;

  fullsize  = ogmdvd_get_ifo_size (disc, vts);
  fullsize += ogmdvd_get_bup_size (disc, vts);
  fullsize += ogmdvd_get_menu_size (disc, vts);

  if (vts > 0)
  {
    if ((size = ogmdvd_get_vob_size (disc, vts)) == 0)
      return 0;
    fullsize += size;
  }

  return fullsize;
}

glong
ogmdvd_title_get_length (OGMDvdTitle *title, OGMDvdTime  *length)
{
  dvd_time_t *dtime;
  pgc_t *pgc;

  gfloat fps;
  guint dhour, dmin, dsec, dframes;

  g_return_val_if_fail (title != NULL, -1);

  pgc = title->vts_file->vts_pgcit->pgci_srp[title->pgcn - 1].pgc;

  dtime = &pgc->playback_time;

  dhour   = ((dtime->hour    & 0xf0) >> 4) * 10 + (dtime->hour    & 0x0f);
  dmin    = ((dtime->minute  & 0xf0) >> 4) * 10 + (dtime->minute  & 0x0f);
  dsec    = ((dtime->second  & 0xf0) >> 4) * 10 + (dtime->second  & 0x0f);
  dframes = ((dtime->frame_u & 0x30) >> 4) * 10 + (dtime->frame_u & 0x0f);

  if (((dtime->frame_u & 0xc0) >> 6) == 1)
    fps = 25.0;
  else
    fps = 30000 / 1001.0;

  if (length)
  {
    length->hour = dhour;
    length->min = dmin;
    length->sec = dsec;
    length->frames = dframes;
  }

  return (glong) (dhour * 3600 * fps + dmin * 60 * fps + dsec * fps + dframes);
}

glong
ogmdvd_title_get_chapters_length (OGMDvdTitle *title, guint start, gint end, OGMDvdTime *length)
{
  pgc_t *pgc;
  guint pgn, pgcn;
  glong last, cell, total;
  gfloat fps;

  g_return_val_if_fail (title != NULL, -1);
  g_return_val_if_fail (end < 0 || start <= end, -1);

  pgc = title->vts_file->vts_pgcit->pgci_srp[title->pgcn - 1].pgc;

  g_return_val_if_fail (start < pgc->nr_of_programs, -1);

  pgcn = title->vts_file->vts_ptt_srpt->title[title->ttn - 1].ptt[start].pgcn;
  pgc = title->vts_file->vts_pgcit->pgci_srp[pgcn - 1].pgc;

  pgn = title->vts_file->vts_ptt_srpt->title[title->ttn - 1].ptt[start].pgn;
  cell = pgc->program_map[pgn - 1] - 1;

  last = pgc->nr_of_cells;
  if (end > -1 && end < pgc->nr_of_programs - 1)
  {
    pgn = title->vts_file->vts_ptt_srpt->title[title->ttn - 1].ptt[end + 1].pgn;
    last = pgc->program_map[pgn - 1] - 1;
  }

  if (start == 0 && last == pgc->nr_of_cells)
    return ogmdvd_title_get_length (title, length);

  for (total = 0; cell < last; cell ++)
    total += ogmdvd_time_to_msec (&pgc->cell_playback[cell].playback_time);

  if (length)
  {
    length->hour = total / (60 * 60 * 1000);
    length->min = total / (60 * 1000) % 60;
    length->sec = total / 1000 % 60;
    length->frames = total % 1000;
  }

  if (((title->vts_file->vts_pgcit->pgci_srp[title->pgcn - 1].pgc->playback_time.frame_u & 0xc0) >> 6) == 1)
    fps = 25.0;
  else
    fps = 30000 / 1001.0;

  return (glong) (total * fps / 1000.0);
}

void
ogmdvd_title_get_framerate (OGMDvdTitle *title, guint *numerator, guint *denominator)
{
  pgc_t *pgc;

  g_return_if_fail (title != NULL);
  g_return_if_fail (numerator != NULL);
  g_return_if_fail (denominator != NULL);

  pgc = title->vts_file->vts_pgcit->pgci_srp[title->pgcn - 1].pgc;

  switch ((pgc->playback_time.frame_u & 0xc0) >> 6)
  {
    case 1:
      *numerator = 25;
      *denominator = 1;
      break;
    case 3:
      *numerator = 30000;
      *denominator = 1001;
      break;
    default:
      g_assert_not_reached ();
      break;
  }
}

void
ogmdvd_title_get_size (OGMDvdTitle *title, guint *width, guint *height)
{
  g_return_if_fail (title != NULL);
  g_return_if_fail (width != NULL);
  g_return_if_fail (height != NULL);

  *width = 0;
  *height = 480;
  if (title->vts_file->vtsi_mat->vts_video_attr.video_format != 0)
    *height = 576;

  switch (title->vts_file->vtsi_mat->vts_video_attr.picture_size)
  {
    case 0:
      *width = 720;
      break;
    case 1:
      *width = 704;
      break;
    case 2:
      *width = 352;
      break;
    case 3:
      *width = 352;
      *width /= 2;
      break;
    default:
      g_assert_not_reached ();
      break;
  }
}

gint
ogmdvd_title_get_video_format (OGMDvdTitle *title)
{
  g_return_val_if_fail (title != NULL, -1);

  return title->vts_file->vtsi_mat->vts_video_attr.video_format;
}

gint
ogmdvd_title_get_display_aspect (OGMDvdTitle *title)
{
  g_return_val_if_fail (title != NULL, -1);

  switch (title->vts_file->vtsi_mat->vts_video_attr.display_aspect_ratio)
  {
    case 0:
      return OGMDVD_DISPLAY_ASPECT_4_3;
    case 1:
    case 3:
      return OGMDVD_DISPLAY_ASPECT_16_9;
    default:
      return -1;
  }
}

gint
ogmdvd_title_get_display_format (OGMDvdTitle *title)
{
  g_return_val_if_fail (title != NULL, -1);

  return title->vts_file->vtsi_mat->vts_video_attr.permitted_df;
}

G_CONST_RETURN guint *
ogmdvd_title_get_palette (OGMDvdTitle *title)
{
  pgc_t *pgc;
  guint pgn;

  g_return_val_if_fail (title != NULL, NULL);

  pgn = title->vts_file->vts_ptt_srpt->title[title->ttn - 1].ptt[0].pgn;
  pgc = title->vts_file->vts_pgcit->pgci_srp[title->pgcn - 1].pgc;

  return pgc->palette;
}

gint
ogmdvd_title_get_n_angles (OGMDvdTitle *title)
{
  g_return_val_if_fail (title != NULL, -1);

  return title->disc->vmg_file->tt_srpt->title[title->nr].nr_of_angles;
}

gint
ogmdvd_title_get_n_chapters (OGMDvdTitle *title)
{
  pgc_t *pgc;

  g_return_val_if_fail (title != NULL, -1);

  pgc = title->vts_file->vts_pgcit->pgci_srp[title->pgcn - 1].pgc;

  return pgc->nr_of_programs;
}

gint
ogmdvd_title_get_n_audio_streams (OGMDvdTitle *title)
{
  gint i, naudio;

  g_return_val_if_fail (title != NULL, -1);

  for (naudio = i = 0; i < title->vts_file->vtsi_mat->nr_of_vts_audio_streams; i++)
    if (title->vts_file->vts_pgcit->pgci_srp[title->ttn - 1].pgc->audio_control[i] & 0x8000)
      naudio ++;

  return naudio;
}

OGMDvdAudioStream *
ogmdvd_title_get_nth_audio_stream (OGMDvdTitle *title, guint nr)
{
  OGMDvdAudioStream *audio;

  g_return_val_if_fail (title != NULL, NULL);
  g_return_val_if_fail (nr < title->vts_file->vtsi_mat->nr_of_vts_audio_streams, NULL);
  g_return_val_if_fail (title->vts_file->vts_pgcit->pgci_srp[title->ttn - 1].pgc->audio_control[nr] & 0x8000, NULL);

  ogmdvd_title_ref (title);

  audio = g_new0 (OGMDvdAudioStream, 1);
  audio->attr = &title->vts_file->vtsi_mat->vts_audio_attr[nr];
  audio->title = title;
  audio->nr = nr;
  audio->ref = 1;
  /*
  audio->aid = video->vts_file->vts_pgcit->pgci_srp[video->ttn - 1].pgc->audio_control[aid] >> 8 & 7;
  */
  return audio;
}

gint
ogmdvd_title_get_n_subp_streams (OGMDvdTitle *title)
{
  gint i, nsubp;

  g_return_val_if_fail (title != NULL, -1);

  for (nsubp = i = 0; i < title->vts_file->vtsi_mat->nr_of_vts_subp_streams; i++)
    if (title->vts_file->vts_pgcit->pgci_srp[title->ttn - 1].pgc->subp_control[i] & 0x80000000)
      nsubp ++;

  return nsubp;
}

OGMDvdSubpStream *
ogmdvd_title_get_nth_subp_stream (OGMDvdTitle *title, guint nr)
{
  OGMDvdSubpStream *subp;

  g_return_val_if_fail (title != NULL, NULL);
  g_return_val_if_fail (nr < title->vts_file->vtsi_mat->nr_of_vts_subp_streams, NULL);
  g_return_val_if_fail (title->vts_file->vts_pgcit->pgci_srp[title->ttn - 1].pgc->subp_control[nr] & 0x80000000, NULL);

  ogmdvd_title_ref (title);

  subp = g_new0 (OGMDvdSubpStream, 1);
  subp->attr = &title->vts_file->vtsi_mat->vts_subp_attr[nr];
  subp->title = title;
  subp->ref = 1;
  subp->nr = nr;

  return subp;
}

void
ogmdvd_title_get_aspect_ratio  (OGMDvdTitle *title, guint *numerator, guint *denominator)
{
  g_return_if_fail (title != NULL);
  g_return_if_fail (numerator != NULL);
  g_return_if_fail (denominator != NULL);

  switch (title->vts_file->vtsi_mat->vts_video_attr.display_aspect_ratio)
  {
    case 0:
      *numerator = 4;
      *denominator = 3;
      break;
    case 1:
    case 3:
      *numerator = 16;
      *denominator = 9;
      break;
    default:
      g_assert_not_reached ();
      break;
  }
}

void
ogmdvd_audio_stream_ref (OGMDvdAudioStream *audio)
{
  g_return_if_fail (audio != NULL);

  audio->ref ++;
}

void
ogmdvd_audio_stream_unref (OGMDvdAudioStream *audio)
{
  g_return_if_fail (audio != NULL);

  if (audio->ref > 0)
  {
    audio->ref --;

    if (audio->ref == 0)
    {
      ogmdvd_title_unref (audio->title);
      audio->title = NULL;

      g_free (audio);
    }
  }
}

OGMDvdTitle *
ogmdvd_audio_stream_get_title (OGMDvdAudioStream *audio)
{
  g_return_val_if_fail (audio != NULL, NULL);

  return audio->title;
}

gint
ogmdvd_audio_stream_get_nr (OGMDvdAudioStream *audio)
{
  g_return_val_if_fail (audio != NULL, -1);

  return audio->nr;
}

gint
ogmdvd_audio_stream_get_format (OGMDvdAudioStream *audio)
{
  g_return_val_if_fail (audio != NULL, -1);

  return audio->attr->audio_format;
}

gint
ogmdvd_audio_stream_get_channels (OGMDvdAudioStream *audio)
{
  g_return_val_if_fail (audio != NULL, -1);

  return audio->attr->channels;
}

gint
ogmdvd_audio_stream_get_language (OGMDvdAudioStream *audio)
{
  g_return_val_if_fail (audio != NULL, -1);
/*
  if (audio->attr->lang_type == 1)
*/
  return audio->attr->lang_code;
/*
  return -1;
*/
}

gint
ogmdvd_audio_stream_get_quantization (OGMDvdAudioStream *audio)
{
  g_return_val_if_fail (audio != NULL, -1);

  return audio->attr->quantization;
}

gint
ogmdvd_audio_stream_get_content (OGMDvdAudioStream *audio)
{
  g_return_val_if_fail (audio != NULL, -1);

  return audio->attr->code_extension;
}

void
ogmdvd_subp_stream_ref (OGMDvdSubpStream *subp)
{
  g_return_if_fail (subp != NULL);

  subp->ref ++;
}

void
ogmdvd_subp_stream_unref (OGMDvdSubpStream *subp)
{
  g_return_if_fail (subp != NULL);

  if (subp->ref > 0)
  {
    subp->ref --;

    if (subp->ref == 0)
    {
      ogmdvd_title_unref (subp->title);
      subp->title = NULL;

      g_free (subp);
    }
  }
}

OGMDvdTitle *
ogmdvd_subp_stream_get_title (OGMDvdSubpStream *subp)
{
  g_return_val_if_fail (subp != NULL, NULL);

  return subp->title;
}

gint
ogmdvd_subp_stream_get_nr (OGMDvdSubpStream *subp)
{
  g_return_val_if_fail (subp != NULL, -1);

  return subp->nr;
}

gint
ogmdvd_subp_stream_get_language (OGMDvdSubpStream *subp)
{
  g_return_val_if_fail (subp != NULL, -1);

  return subp->attr->lang_code;
}

gint
ogmdvd_subp_stream_get_content (OGMDvdSubpStream *subp)
{
  g_return_val_if_fail (subp != NULL, -1);

  return subp->attr->lang_extension;
}
/*
gboolean
ogmdvd_drive_tray_open (const gchar *device)
{
#if defined(__linux__) || defined(__FreeBSD__) || defined(__NetBSD__)
  gint fd, status;

  g_return_val_if_fail (device != NULL, FALSE);

  fd = g_open (device, O_RDONLY | O_NONBLOCK, 0);
  if (fd < 0)
    return FALSE;

#ifdef __FreeBSD__
  ioctl (fd, CDIOCALLOW);
#endif

  status = ioctl (fd, CDROMEJECT);
  if (status < 0)
  {
    close (fd);
    return FALSE;
  }

  close (fd);
#endif

  return TRUE;
}

gboolean
ogmdvd_drive_has_valid_media (const gchar *device)
{
  NautilusBurnDrive *drive;
  NautilusBurnMediaType type;
#ifdef __linux__
  gint fd, status;
#endif

  g_return_val_if_fail (device && *device, FALSE);

  drive = nautilus_burn_drive_new_from_path (device);
  if (!drive)
    return FALSE;

  type = nautilus_burn_drive_get_media_type (drive);
  g_object_unref (drive);

  if (type != NAUTILUS_BURN_MEDIA_TYPE_UNKNOWN && type < NAUTILUS_BURN_MEDIA_TYPE_DVD)
    return FALSE;

  if (type == NAUTILUS_BURN_MEDIA_TYPE_UNKNOWN)
    g_warning ("Unknown media type. Assuming it is valid.");

#ifdef __linux__

  fd = g_open (device, O_RDONLY | O_NONBLOCK | O_EXCL, 0);
  if (fd < 0)
    return FALSE;

  status = ioctl (fd, CDROM_DISC_STATUS, CDSL_CURRENT);
  close (fd);

  if (status < 0)
    return FALSE;

  return status != CDS_NO_INFO && status != CDS_NO_DISC;

#endif

  return TRUE;
}
*/
G_CONST_RETURN gchar *
ogmdvd_get_video_format_label (gint format)
{
  static gchar *video_format[] = 
  {
    "NTSC",
    "PAL",
    "Error",
    "Error"
  };

  return format < 0 ? NULL : video_format[format];
}

G_CONST_RETURN gchar *
ogmdvd_get_display_aspect_label (gint aspect)
{
  static gchar *display_aspect[] =
  {
    "4/3", 
    "16/9", 
    "?:?", 
    "16/9"
  };

  return display_aspect[aspect];
}

G_CONST_RETURN gchar *
ogmdvd_get_audio_format_label (gint format)
{
  static gchar *audio_format[] =
  {
    "AC3",
    "Unknown",
    "MPEG1",
    "MPEG2EXT",
    "LPCM",
    "SDDS",
    "DTS",
    "Error"
  };

  return format < 0 ? NULL : audio_format[format];
}

G_CONST_RETURN gchar *
ogmdvd_get_audio_channels_label (gint channels)
{
  static gchar *audio_channels[] =
  {
    "Mono",
    "Stereo",
    "Unknown",
    "Surround",
    "Unknown",
    "5.1",
    "6.1",
    "7.1"
  };

  return channels < 0 ? NULL : audio_channels[channels];
}

G_CONST_RETURN gchar *
ogmdvd_get_audio_quantization_label (gint quantization)
{
  static gchar *audio_quantization[] = 
  {
    "16bit", 
    "20bit", 
    "24bit", 
    "drc"
  };

  return audio_quantization[quantization];
}

G_CONST_RETURN gchar *
ogmdvd_get_audio_content_label (gint content)
{
  static gchar *audio_content[] = 
  {
    "Undefined", 
    "Normal", 
    "Impaired", 
    "Comments 1", 
    "Comments 2"
  };

  return content < 0 ? NULL : audio_content[content];
}

G_CONST_RETURN gchar *
ogmdvd_get_subp_content_label (gint content)
{
  static gchar *subp_content[] = 
  {
    "Undefined", 
    "Normal", 
    "Large", 
    "Children", 
    "Reserved", 
    "Normal_CC", 
    "Large_CC", 
    "Children_CC",
    "Reserved", 
    "Forced", 
    "Reserved", 
    "Reserved", 
    "Reserved", 
    "Director", 
    "Large_Director", 
    "Children_Director"
  };

  return content < 0 ? NULL : subp_content[content];
}

G_CONST_RETURN gchar *
ogmdvd_get_language_label (gint code)
{
  const gchar *lang;
  guint i;

  lang = ogmdvd_get_language_iso639_1 (code);

  for (i = 0; ogmdvd_languages[i][OGMDVD_LANGUAGE_ISO639_1]; i++)
    if (strcmp (ogmdvd_languages[i][OGMDVD_LANGUAGE_ISO639_1], lang) == 0)
      return ogmdvd_languages[i][OGMDVD_LANGUAGE_NAME];

  return NULL;
}

G_CONST_RETURN gchar *
ogmdvd_get_language_iso639_1 (gint code)
{
  static gchar lang[3];

  if (code < 1)
  {
    lang[0] = '?';
    lang[1] = '?';
    lang[2] = 0;
  }
  else
  {
    lang[0] = code >> 8;
    lang[1] = code & 0xff;
    lang[2] = 0;
  }

  return lang;
}

G_CONST_RETURN gchar *
ogmdvd_get_language_iso639_2 (gint code)
{
  const gchar *lang;
  guint i;

  lang = ogmdvd_get_language_iso639_1 (code);

  for (i = 0; ogmdvd_languages[i][OGMDVD_LANGUAGE_ISO639_1]; i++)
    if (strcmp (ogmdvd_languages[i][OGMDVD_LANGUAGE_ISO639_1], lang) == 0)
      return ogmdvd_languages[i][OGMDVD_LANGUAGE_ISO639_2];

  return NULL;
}

