/*  snmp.h - monitor SNMP UPS (RFC 1628 compliant) for NUT
 *
 *  Based on NetSNMP API (Simple Network Management Protocol V1-2)
 *  
 *  Copyright (C) 2002 Arnaud Quette <arnaud.quette@free.fr>
 *  some parts are Copyright (C) :
 *                Russel Kroll <rkroll@exploits.org>
 *                Hans Ekkehard Plesser <hans.plesser@itf.nlh.no>
 *
 *  Sponsored by MGE UPS SYSTEMS <http://www.mgeups.com>
 *
 *  All rights reserved.
 *
 */

/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

/* Defines related to NUT and SNMP UPS */
#include <shared-tables.h> 

#define SNMP_INFOMAX 12

/* SNMP OIDs set */
#define OID_UPS_MIB      ".1.3.6.1.2.1.33"
#define OID_MFR_NAME     ".1.3.6.1.2.1.33.1.1.1.0"     /* INFO_MFR      */
#define OID_MODEL_NAME   ".1.3.6.1.2.1.33.1.1.2.0"     /* INFO_MODEL    */
#define OID_FIRMREV      ".1.3.6.1.2.1.33.1.1.4.0"     /* INFO_FIRMREV  */
#define OID_BATT_STATUS  ".1.3.6.1.2.1.33.1.2.1.0"     /* INFO_STATUS (2) */
#define OID_BATT_RUNTIME ".1.3.6.1.2.1.33.1.2.3.0"     /* INFO_RUNTIME  */
#define OID_BATT_CHARGE  ".1.3.6.1.2.1.33.1.2.4.0"     /* INFO_BATTPCT  */
#define OID_BATT_VOLTAGE ".1.3.6.1.2.1.33.1.2.5.0"     /* INFO_BATTVOLT */
#define OID_POWER_STATUS ".1.3.6.1.2.1.33.1.4.1.0"     /* INFO_STATUS (1) */
#define OID_OUT_VOLTAGE  ".1.3.6.1.2.1.33.1.9.3.0"     /* INFO_UTILITY  */
#define OID_UTIL_VOLTAGE ".1.3.6.1.2.1.33.1.4.4.1.2.0" /* INFO_UTILITY  */
#define OID_LOAD_LEVEL   ".1.3.6.1.2.1.33.1.4.4.1.5.0" /* INFO_LOADPCT */
#define OID_OUTPUT_TAB   ".1.3.6.1.2.1.33.1.4.4"

/* Defines for OID_POWER_STATUS (1) */
#define PWR_OTHER   1 /* other -> INFO_STATUS/? */
#define PWR_NONE    2 /* none -> INFO_STATUS/OFF */
#define PWR_NORMAL  3 /* normal -> INFO_STATUS/OL */
#define PWR_BYPASS  4 /* bypass -> INFO_STATUS/? */
#define PWR_BATTERY 5 /* battery -> INFO_STATUS/OB */
#define PWR_BOOSTER 6 /* booster -> INFO_STATUS/BOOST */
#define PWR_REDUCER 7 /* reducer -> INFO_STATUS/TRIM */

/* Defines for OID_BATT_STATUS (2) */
#define BATT_UNKNOWN  1 /* unknown -> INFO_STATUS/? */
#define BATT_NORMAL   2 /* batteryNormal -> INFO_STATUS/? */
#define BATT_LOW      3 /* batteryLow -> INFO_STATUS/LB */
#define BATT_DEPLETED 4 /* batteryDepleted -> INFO_STATUS/? */

/* Missing data
   LB    - UPS battery is low (with OB = shutdown situation)
   CAL   - UPS is performing calibration
   OVER  - UPS is overloaded
   RB    - UPS battery needs to be replaced
   FSD   - UPS is in forced shutdown state (slaves take note)
*/

/* Objectified function and struct from Hans Ekkehard Plesser */

#define MAX_CMD_LENGTH 48
#define MAX_DFL_LENGTH 32
#define MAX_FMT_LENGTH 10

/* use explicit booleans */
#ifdef FALSE
#  undef FALSE
#endif /* FALSE */
#ifdef TRUE
#  undef TRUE
#endif /* TRUE */
typedef enum ebool { FALSE=0, TRUE } bool;

/* special functions to interpret items: 
   take UPS answer, return string to set in INFO, max len  */
typedef void (*interpreter)(char *, char *, int);

/* Structure containing info about one item that can be requested
   from UPS and set in INFO.  If no interpreter functions is defined,
   use sprintf with given format string.  If unit is not NONE, values
   are converted according to the multiplier table  
*/
typedef struct {
  int  type;                       /* INFO_ element */
  int  flags;                      /* flags to set in addinfo */
  int  length;                     /* length of strings (if STR), else multiplier (if INT) */  
  char cmd[MAX_CMD_LENGTH];        /* ups command string to requets element */
  char cmd_fmt[MAX_FMT_LENGTH];    /* OID format for complex values */
  interpreter interpret;           /* interpreter fct, NULL if not needed  */
  char dfl[MAX_DFL_LENGTH];        /* default value */
  bool  ok;                        /* flag indicating if item is available */
} snmp_info_item;

/* Array containing all possible INFO items
   NOTE: - Array is terminated by element with type INFO_UNUSED.
	 - Essential INFO items (_MFR, _MODEL, _FIRMWARE, _STATUS) are
	   handled separately.
         - Array is NOT const, since "ok" can be changed.
*/

static snmp_info_item snmp_info[] = {
  { 
    INFO_MFR,                  /* INFO_ element */
    0,			       /* Flag          */
    32,			       /* max length    */
    OID_MFR_NAME,              /* UPS command   */
    "",		               /* format        */
    NULL,		       /* interpreter   */
    "Generic",	               /* default value */
    TRUE		       /* ok?           */
  },
  { 
    INFO_MODEL,                /* INFO_ element */
    0,			       /* Flag          */
    16,			       /* max length    */
    OID_MODEL_NAME,            /* UPS command   */
    "", 		       /* format        */
    NULL,		       /* interpreter   */
    "Generic SNMP UPS",        /* default value */
    TRUE		       /* ok?           */
  },
  { 
    INFO_FIRMREV,              /* INFO_ element */
    0,			       /* Flag          */
    16,			       /* max length    */
    OID_FIRMREV,               /* UPS command   */
    "", 		       /* format        */
    NULL,		       /* interpreter   */
    "",                        /* default value */
    TRUE		       /* ok?           */
  },
  { 
    INFO_STATUS,               /* INFO_ element */
    FLAG_STRING | FLAG_RW,     /* Flag          */
    1,			       /* max length    */
    OID_POWER_STATUS,          /* UPS command   */
    "",		               /* format        */
    NULL,		       /* interpreter   */
    "OL",                      /* default value */
    TRUE		       /* ok?           */
  },
  { 
    INFO_STATUS,               /* INFO_ element */
    FLAG_STRING | FLAG_RW,     /* Flag          */
    1,			       /* max length    */
    OID_BATT_STATUS,           /* UPS command   */
    "",		               /* format        */
    NULL,		       /* interpreter   */
    "OL",                      /* default value */
    TRUE		       /* ok?           */
  },
  { 
    INFO_BATTPCT,              /* INFO_ element */
    0,			       /* Flag          */
    1,			       /* multiplier    */
    OID_BATT_CHARGE,           /* UPS command   */
    "",	                       /* format        */
    NULL,		       /* interpreter   */
    "",		               /* output format */
    TRUE		       /* ok?           */
  },
  { 
    INFO_RUNTIME,              /* INFO_ element */
    0,			       /* Flag          */
    60,			       /* multiplier    */
    OID_BATT_RUNTIME,	       /* UPS command   */
    "",	        	       /* format        */
    NULL,		       /* interpreter   */
    "",  		       /* output format */
    TRUE		       /* ok?           */
  },
  { 
    INFO_BATTVOLT,             /* INFO_ element */
    0,			       /* Flag          */
    1,			       /* multiplier    */
    OID_BATT_VOLTAGE,	       /* UPS command   */
    "", 		       /* format        */
    NULL,		       /* interpreter   */
    "",  		       /* output format */
    TRUE		       /* ok?           */
  },
  { 
    INFO_OUTVOLT,              /* INFO_ element */
    0,			       /* Flag          */
    1,			       /* multiplier    */
    OID_OUT_VOLTAGE,	       /* UPS command   */
    "", 		       /* format        */
    NULL,		       /* interpreter   */
    "",  		       /* output format */
    TRUE		       /* ok?           */
  },
  /* terminating element */
  { 
    INFO_UNUSED,               /* INFO_ element */
    0,			       /* Flag          */
    0,			       /* multiplier    */
    "\0",		       /* UPS command   */
    "\0",    		       /* format        */
    NULL,		       /* interpreter   */
    "",  		       /* output format */
    FALSE		       /* ok?           */
  }
};
