/*
 * vim: ts=4 sw=4
 */

/*
 * powernet.c -- SNMP interface driver for APC SNMP devices.
 * Tested on AP9606 APC Web/SNMP managenent card.
 * Dmitry Frolov <frolov@riss-telecom.ru>
 *
 * $RISS: nut/drivers/powernet.c,v 1.4 2002/10/21 07:19:39 frol Exp $
 *
 * Based on NUT snmp-ups driver:
 * Copyright (C) 2002 Arnaud Quette <arnaud.quette@free.fr>
 * some parts are Copyright (C) :
 *               Russell Kroll <rkroll@exploits.org>
 *               Hans Ekkehard Plesser <hans.plesser@itf.nlh.no>
 *
 * References: PowerNet MIB v 3.3.0
 * You can get it from http://apcc.com/tools/download/
 */

/*
 * Suggested entry in the makefile:
 *
 * COMMON_LIBDEP=	main.o upscommon.o ../common/upsconf.o ../common/parseconf.o
 * COMMON_INCDEP=	../include/shared.h ../include/shared-tables.h
 *
 * powernet: powernet.o $(COMMON_LIBDEP) $(LIBDEP)
 * 		$(CC) $(CFLAGS) -o $@ $@.o $(COMMON_LIBDEP) $(LIBOBJ) \
 * 			-lsnmp -lcrypto
 *
 * powernet.o: powernet.c powernet.h $(COMMON_INCDEP)
 * 		$(CC) $(CFLAGS) -c powernet.c
 */

/*
 *		       GNU GENERAL PUBLIC LICENSE
 *			  Version 2, June 1991
 *
 *  Copyright (C) 1989, 1991 Free Software Foundation, Inc.
 *            59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *  Everyone is permitted to copy and distribute verbatim copies
 *  of this license document, but changing it is not allowed.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

/* UCD SNMP includes and global data */
#include <arpa/inet.h> /* for ucd-snmp include bug */
#include <ucd-snmp/ucd-snmp-config.h>
#include <ucd-snmp/ucd-snmp-includes.h>
#include <ucd-snmp/system.h>

/* NUT includes and global data */
#include "main.h"
#include "powernet.h"

static char upsdrv_version[] = PN_UPSDRV_VERSION;
static char upsdrv_cvsid[] = "$RISS: nut/drivers/powernet.c,v 1.4 2002/10/21 07:19:39 frol Exp $";

unsigned long g_ups_status[PN_STATUS_NUM_ELEM];

struct snmp_session g_snmp_sess, *g_snmp_sess_p;

/* -----------------------------------------------------------
 * SNMP functions.
 * ----------------------------------------------------------- */

void pn_snmp_init(const char *type, const char *hostname, const char *community)
{
	/* Initialize the SNMP library */
	init_snmp(type);

	/* Initialize session */
	snmp_sess_init(&g_snmp_sess);

	g_snmp_sess.peername = (char *)hostname;
	g_snmp_sess.community = (char *)community;
	g_snmp_sess.community_len = strlen(community);
	g_snmp_sess.version = SNMP_VERSION_1;

	/* Open the session */
	SOCK_STARTUP;
	g_snmp_sess_p = snmp_open(&g_snmp_sess);	/* establish the session */
	if (g_snmp_sess_p == NULL) {
		pn_snmp_perror(&g_snmp_sess, "pn_snmp_init: snmp_open");
		exit(1);
	}
}

void pn_snmp_cleanup(void)
{
	/* close snmp session. */
	if (g_snmp_sess_p)
		snmp_close(g_snmp_sess_p);
	SOCK_CLEANUP;
}

struct snmp_pdu *pn_snmp_get(const char *OID)
{
	int status;
	struct snmp_pdu *pdu, *response = NULL;
	oid name[MAX_OID_LEN];
	size_t name_len = MAX_OID_LEN;

	/* create and send request. */

	if (!read_objid(OID, name, &name_len)) {
		upslogx(LOG_ERR, "pn_snmp_get: %s: %s",
			OID, snmp_api_errstring(snmp_errno));
		return NULL;
	}

	pdu = snmp_pdu_create(SNMP_MSG_GET);
	if (pdu == NULL)
		fatalx("Not enough memory");
	snmp_add_null_var(pdu, name, name_len);

	status = snmp_synch_response(g_snmp_sess_p, pdu,
		&response);

	if (!((status == STAT_SUCCESS) && (response->errstat == SNMP_ERR_NOERROR)))
	{
		pn_snmp_perror(g_snmp_sess_p, "pn_snmp_get: %s", OID);
		snmp_free_pdu(response);
		response = NULL;
	}

	return response;
}

bool pn_snmp_get_str(const char *OID, char *buf, size_t buf_len)
{
	size_t len = 0;
	struct snmp_pdu *pdu;

	/* zero out buffer. */
	memset(buf, 0, buf_len);

	pdu = pn_snmp_get(OID);
	if (pdu == NULL)
		return FALSE;

	switch (pdu->variables->type) {
	case ASN_OCTET_STR:
	case ASN_OPAQUE:
		len = pdu->variables->val_len > buf_len - 1 ?
			buf_len - 1 : pdu->variables->val_len;
		memcpy(buf, pdu->variables->val.string, len);
		buf[len] = '\0';
		break;
	case ASN_INTEGER:
	case ASN_GAUGE:
	case ASN_TIMETICKS:
		len = snprintf(buf, buf_len, "%ld", *pdu->variables->val.integer);
		break;
	default:
		upslogx(LOG_ERR, "unhandled ASN 0x%x recieved from %s",
			pdu->variables->type, OID);
		return FALSE;
		break;
	}

	snmp_free_pdu(pdu);

	return TRUE;
}

bool pn_snmp_get_int(const char *OID, long *pval)
{
	struct snmp_pdu *pdu;
	long value;
	char *buf;

	pdu = pn_snmp_get(OID);
	if (pdu == NULL)
		return FALSE;

	switch (pdu->variables->type) {
	case ASN_OCTET_STR:
	case ASN_OPAQUE:
		buf = xmalloc(pdu->variables->val_len + 1);
		memcpy(buf, pdu->variables->val.string, pdu->variables->val_len);
		buf[pdu->variables->val_len] = '\0';
		value = strtol(buf, NULL, 0);
		free(buf);
		break;
	case ASN_INTEGER:
	case ASN_GAUGE:
	case ASN_TIMETICKS:
		value = *pdu->variables->val.integer;
		break;
	default:
		upslogx(LOG_ERR, "unhandled ASN 0x%x recieved from %s",
			pdu->variables->type, OID);
		return FALSE;
		break;
	}

	snmp_free_pdu(pdu);

	if (pval != NULL)
		*pval = value;

	return TRUE;
}

bool pn_snmp_set(const char *OID, char type, const char *value)
{
	int status;
	bool ret = FALSE;
	struct snmp_pdu *pdu, *response = NULL;
	oid name[MAX_OID_LEN];
	size_t name_len = MAX_OID_LEN;

	upsdebugx(2, "in pn_snmp_set");

	if (!read_objid(OID, name, &name_len)) {
		upslogx(LOG_ERR, "pn_snmp_set: %s: %s",
			OID, snmp_api_errstring(snmp_errno));
		return FALSE;
	}

	pdu = snmp_pdu_create(SNMP_MSG_SET);
	if (pdu == NULL)
		fatalx("Not enough memory");
	snmp_add_var(pdu, name, name_len, type, value);

	status = snmp_synch_response(g_snmp_sess_p, pdu, &response);

	if ((status == STAT_SUCCESS) && (response->errstat == SNMP_ERR_NOERROR))
		ret = TRUE;
	else
		pn_snmp_perror(g_snmp_sess_p, "pn_snmp_set: %s", OID);

	snmp_free_pdu(response);

	return ret;
}

bool pn_snmp_set_str(const char *OID, const char *value)
{
	return pn_snmp_set(OID, 's', value);
}

bool pn_snmp_set_int(const char *OID, long value)
{
	bool ret;
	char *buf = xmalloc(PN_BUFSIZE);

	snprintf(buf, PN_BUFSIZE, "%ld", value);
	ret = pn_snmp_set(OID, 'i', buf);
	free(buf);
	return ret;
}

/* log descriptive SNMP error message. */
void pn_snmp_perror(struct snmp_session *sess, const char *fmt, ...)
{
	va_list va;
	int cliberr, snmperr;
	char *snmperrstr;
	char *buf;

	buf = xmalloc(PN_LARGEBUF);

	va_start(va, fmt);
	vsnprintf(buf, PN_LARGEBUF, fmt, va);
	va_end(va);

	snmp_error(sess, &cliberr, &snmperr, &snmperrstr);
	upslogx(LOG_ERR, "%s: %s",
		buf, snmperrstr);
	free(snmperrstr);

	free(buf);
}

/* -----------------------------------------------------------
 * utility functions.
 * ----------------------------------------------------------- */

/* called on startup. */
void pn_startup(void)
{
	pn_snmp_init(progname, device_path,
		(testvar(PN_VAR_COMMUNITY) ? getval(PN_VAR_COMMUNITY) : "public"));
}

/* clean up before exit. */
void pn_cleanup(void)
{
	pn_snmp_cleanup();
}

/* add instant commands into info database. */
void pn_init_instcmds(void)
{
	pn_info_t *pn_info_p;

	for (pn_info_p = &pn_info[0]; pn_info_p->info_type; pn_info_p++)
		if (pn_info_p->info_type & PN_CMD_MASK)
			addinfo(INFO_INSTCMD, "", 0, pn_info_p->info_type);
}

/* install pointers to functions for msg handlers called from msgparse */
void pn_setuphandlers(void)
{
	upsh.instcmd = pn_ups_instcmd;
	upsh.setvar = pn_ups_set;
}

/* universal function to add or update info element. */
void pn_setinfo(int type, const char *value, int flags, int auxdata)
{
	if (type == INFO_INSTCMD)
		return;
	if (getdata(type))
		setinfo(type, "%s", value);
	else
		addinfo(type, value, flags, auxdata);
}

void pn_status_init(void)
{
	memset(g_ups_status, 0, sizeof(g_ups_status));
}

void pn_status_set(int type, long value)
{
	g_ups_status[PN_STATUS_TYPE(type)] = value;
}

void pn_status_commit(void)
{
	char buf[PN_INFOSIZE];
	pn_info_t *pn_info_p;

	switch (g_ups_status[PN_STATUS_TYPE(PN_STATUS_PWR)]) {
	case PWR_OTHER:			/*  INFO_STATUS/? */
	case PWR_REBOOTING:
	case PWR_SOFT_BYPASS:
	case PWR_HARD_BYPASS:
	case PWR_FAIL_BYPASS:
		break;
	case PWR_NONE:			/* none -> INFO_STATUS/OFF */
	case PWR_SLEEPING:
	case PWR_SLEEPING2:
		snprintf(buf, sizeof(buf), "OFF");
		break;
	case PWR_NORMAL:		/* normal -> INFO_STATUS/OL */
		snprintf(buf, sizeof(buf), "OL");
		break;
	case PWR_BATTERY:		/* battery -> INFO_STATUS/OB */
		snprintf(buf, sizeof(buf), "OB");
		break;
	case PWR_BOOSTER:		/* booster -> INFO_STATUS/BOOST */
		snprintf(buf, sizeof(buf), "BOOST");
		break;
	case PWR_REDUCER:		/* reducer -> INFO_STATUS/TRIM */
		snprintf(buf, sizeof(buf), "TRIM");
		break;
	}

	switch (g_ups_status[PN_STATUS_TYPE(PN_STATUS_BATT)]) {
	case BATT_UNKNOWN:		/* unknown -> INFO_STATUS/? */
	case BATT_NORMAL:		/* batteryNormal -> INFO_STATUS/? */
		break;
	case BATT_LOW:			/* batteryLow -> INFO_STATUS/LB */
		snprintfcat(buf, sizeof(buf),
			(strlen(buf) == 0 ? "%s" : " %s"), "LB");
		break;
	}

	switch (g_ups_status[PN_STATUS_TYPE(PN_STATUS_CAL)]) {
	case CAL_INPROGRESS:
		snprintfcat(buf, sizeof(buf),
			(strlen(buf) == 0 ? "%s" : " %s"), "CAL");
		break;
	}

	switch (g_ups_status[PN_STATUS_TYPE(PN_STATUS_RB)]) {
	case RB_NEED:
		snprintfcat(buf, sizeof(buf),
			(strlen(buf) == 0 ? "%s" : " %s"), "RB");
		break;
	}

	pn_info_p = pn_find_info(INFO_STATUS);

	pn_setinfo(pn_info_p->info_type, buf, pn_info_p->info_flags,
		pn_info_p->info_len);
}

/* find info element definition in my info array. */
pn_info_t *pn_find_info(int type)
{
	pn_info_t *pn_info_p;

	for (pn_info_p = &pn_info[0]; pn_info_p->info_type; pn_info_p++)
		if (pn_info_p->info_type == type)
			return pn_info_p;
	fatalx("pn_find_info: unknown info type: 0x%x", type);
	return NULL;
}

/* find description of INFO_ element. */
struct netvars_t *pn_find_netvar(int type)
{
	struct netvars_t *netvar;

	for (netvar = &netvars[0]; netvar->name; netvar++) {
		if (netvar->type == type)
			return netvar;
	}
	fatalx("pn_find_netvar: unknown netvar type: 0x%x", type);
	return NULL;
}

/* find description of CMD_ element. */
struct instcmds_t *pn_find_instcmd(int cmd)
{
	struct instcmds_t *instcmd;

	for (instcmd = &instcmds[0]; instcmd->name; instcmd++) {
		if (instcmd->cmd == cmd)
			return instcmd;
	}
	fatalx("pn_find_instcmd: unknown instcmd: 0x%x", cmd);
	return NULL;
}

/* -----------------------------------------------------------
 * some logic here.
 * ----------------------------------------------------------- */

/* walk ups variables and set elements of the info array. */
void pn_ups_walk(int mode)
{
	static long iterations = 0;

	pn_info_t *pn_info_p;
	bool status;
	struct netvars_t *netvar;
	
	upsdebugx(2, "in pn_ups_walk()");

	pn_status_init();

	for (pn_info_p = &pn_info[0]; pn_info_p->info_type; pn_info_p++) {

		/* skip instcmd. */
		if (pn_info_p->info_type & PN_CMD_MASK)
			continue;

		netvar = pn_find_netvar(pn_info_p->info_type);

		upsdebugx(3, "pn_ups_walk: processing info %s", netvar->name);

		/* skip elements we shouldn't show. */
		if (!(pn_info_p->flags & PN_FLAG_OK))
			continue;

		/* skip static elements in update mode. */
		if (mode == PN_WALKMODE_UPDATE && 
				pn_info_p->flags & PN_FLAG_STATIC)
			continue;

		/* set default value if we cannot fetch it */
		/* and set static flag on this element. */
		if (pn_info_p->flags & PN_FLAG_ABSENT) {
			if (mode == PN_WALKMODE_INIT) {
				if (pn_info_p->dfl) {
					/* Set default value if we cannot fetch it from ups. */
					pn_setinfo(pn_info_p->info_type, pn_info_p->dfl,
						pn_info_p->info_flags, pn_info_p->info_len);
				}
				pn_info_p->flags |= PN_FLAG_STATIC;
			}
			continue;
		}

		/* check stale elements only on each PN_STALE_RETRY iteration. */
		if ((pn_info_p->flags & PN_FLAG_STALE) &&
				(iterations % PN_STALE_RETRY) != 0)
			continue;

		/* ok, update this element. */

		status = pn_ups_get(pn_info_p);

		/* set stale flag if data is stale, clear if not. */
		if (status == TRUE) {
			if (pn_info_p->flags & PN_FLAG_STALE)
				upslogx(LOG_INFO, "pn_ups_walk: data resumed for %s",
					netvar->name);
			pn_info_p->flags &= ~PN_FLAG_STALE;
		} else {
			if (!(pn_info_p->flags & PN_FLAG_STALE))
				upslogx(LOG_INFO, "pn_ups_walk: data stale for %s",
					netvar->name);
			pn_info_p->flags |= PN_FLAG_STALE;
		}

	}	/* for (pn_info_p... */

	pn_status_commit();

	iterations++;
}

bool pn_ups_get(pn_info_t *pn_info_p)
{
	static char buf[PN_INFOSIZE];
	bool status;
	long value;

	switch (pn_info_p->info_type) {

	case INFO_STATUS:
		status = pn_snmp_get_int(pn_info_p->OID, &value);
		if (status == TRUE)
			pn_status_set(pn_info_p->flags, value);

		break;

	case INFO_SLFTSTRES:
		status = pn_snmp_get_int(pn_info_p->OID, &value);
		if (status == TRUE) {
			static char *slftstres[] = 
				{ "OK", "FAILED", "TEST INVALID", "INPROGRESS" };

			if (value >= 1 && value <= 4)
				pn_setinfo(pn_info_p->info_type, slftstres[value - 1],
					pn_info_p->info_flags, pn_info_p->info_len);
		}
		break;

	case INFO_RUNTIME:
		status = pn_snmp_get_int(pn_info_p->OID, &value);
		if (status == TRUE) {
			/* convert milliseconds to minutes. */
			snprintf(buf, sizeof(buf), "%ld", value / (60 * 100));
			pn_setinfo(pn_info_p->info_type, buf,
				pn_info_p->info_flags, pn_info_p->info_len);
		}
		break;

	default:
		/* get snmp OID value. */
		status = pn_snmp_get_str(pn_info_p->OID,
			buf, sizeof(buf));

		if (status == TRUE) {
			pn_setinfo(pn_info_p->info_type, buf,
				pn_info_p->info_flags, pn_info_p->info_len);
		}
		break;
	}	/* switch (pn_info_p->info_type) */

	return status;
}

/* set r/w INFO_ element to a value. */
void pn_ups_set(int type, int data_len, char *data)
{
	pn_info_t *pn_info_p;
	bool ret;
	struct netvars_t *netvar;

	upsdebugx(2, "in pn_ups_set()");

	pn_info_p = pn_find_info(type);
	netvar = pn_find_netvar(type);

	if (pn_info_p == NULL || pn_info_p->info_type == 0 ||
		!(pn_info_p->flags & PN_FLAG_OK))
	{
		upslogx(LOG_ERR, "pn_ups_set: info element unavailable %s",
			netvar->name);
		return;
	}

	if (! (pn_info_p->info_flags & FLAG_RW) || pn_info_p->OID == NULL) {
		upslogx(LOG_ERR, "pn_ups_set: not writable %s", netvar->name);
		return;
	}

	/* set value. */

	if (PN_TYPE(pn_info_p->info_flags) == PN_TYPE_STRING) {
		ret = pn_snmp_set_str(pn_info_p->OID, data);
	} else {
		ret = pn_snmp_set_int(pn_info_p->OID, strtol(data, NULL, 0));
	}

	if (ret == FALSE)
		upslogx(LOG_ERR, "pn_ups_set: cannot set value %s for %s",
			data, pn_info_p->OID);
	else
		upsdebugx(1, "pn_ups_set: sucessfully set %s to \"%s\"",
			netvar->name, data);

	/* update info array. */
	pn_setinfo(type, data, pn_info_p->info_flags, pn_info_p->info_len);
	writeinfo();
}

/* process instant command and take action. */
void pn_ups_instcmd(int auxcmd, int data_len, char *data)
{
	pn_info_t *pn_info_p;
	int status;
	struct instcmds_t *instcmd;

	upsdebugx(2, "in pn_ups_instcmd()");

	pn_info_p = pn_find_info(auxcmd);
	instcmd = pn_find_instcmd(auxcmd);

	if (pn_info_p->info_type == 0 || !(pn_info_p->flags & PN_FLAG_OK) ||
		pn_info_p->OID == NULL)
	{
		upslogx(LOG_ERR, "pn_ups_instcmd: %s unavailable",
			instcmd->name);
		return;
	}

	/* set value. */

	if (pn_info_p->info_flags & FLAG_STRING) {
		status = pn_snmp_set_str(pn_info_p->OID, pn_info_p->dfl);
	} else {
		status = pn_snmp_set_int(pn_info_p->OID, pn_info_p->info_len);
	}

	if (status == FALSE)
		upslogx(LOG_ERR, "pn_ups_instcmd: cannot set value for %s",
			instcmd->name);
	else
		upsdebugx(1, "pn_ups_instcmd: successfully sent command %s",
			instcmd->name);
}

/* do the shutdown immediately. */
void pn_shutdown_ups(void)
{
	int sdtype = 0;
	long pwr_status;

	if (pn_snmp_get_int(OID_POWER_STATUS, &pwr_status) == FALSE)
		fatalx("cannot determine UPS status");

	if (testvar(PN_VAR_SDTYPE))
		sdtype = atoi(getval(PN_VAR_SDTYPE));

	/* logic from newapc.c */
	switch (sdtype) {
	case 3:		/* shutdown with grace period */
		upslogx(LOG_INFO, "sending delayed power off command to UPS");
		pn_ups_instcmd(CMD_SHUTDOWN, 0, 0);
		break;
	case 2:		/* instant shutdown */
		upslogx(LOG_INFO, "sending power off command to UPS");
		pn_ups_instcmd(CMD_OFF, 0, 0);
		break;
	case 1:
		/* Send a combined set of shutdown commands which can work better */
		/* if the UPS gets power during shutdown process */
		/* Specifically it sends both the soft shutdown 'S' */
		/* and the powerdown after grace period - '@000' commands */
		upslogx(LOG_INFO, "UPS - sending shutdown/powerdown");
		if (pwr_status == PWR_BATTERY)
			pn_ups_instcmd(CMD_SOFTDOWN, 0, 0);
		pn_ups_instcmd(CMD_SDRET, 0, 0);
		break;
	default:
		/* if on battery... */
		if (pwr_status == PWR_BATTERY) {
			upslogx(LOG_INFO,
				"UPS is on battery, sending shutdown command...");
			pn_ups_instcmd(CMD_SOFTDOWN, 0, 0);
		} else {
			upslogx(LOG_INFO,
				"UPS is online, sending shutdown+return command...");
			pn_ups_instcmd(CMD_SDRET, 0, 0);
		}
		break;
	}
}

/* ---------------------------------------------
 * driver functions implementations
 * --------------------------------------------- */

void upsdrv_initinfo(void)
{
	upsdebugx(2, "in upsdrv_initinfo()");

	/* add instant commands to the info database. */
	pn_init_instcmds();

	/* set habdlers for instcmds and set commands. */
	pn_setuphandlers();

	/* initialize all INFO_ fields */
	pn_ups_walk(PN_WALKMODE_INIT);
}

void upsdrv_updateinfo(void)
{
	upsdebugx(2, "in upsdrv_updateinfo()");
  
	/* update all dynamic info fields */
	pn_ups_walk(PN_WALKMODE_UPDATE);

	writeinfo();
}

/* do the shutdown immediately and cleanup for exit. */
/* upsdrv_initups already called, but upsdrv_initinfo isn't. */
void upsdrv_shutdown(void)
{
	upsdebugx(2, "in upsdrv_shutdown");

	pn_shutdown_ups();
	pn_cleanup();
}

void upsdrv_help(void)
{
	printf("\nShutdown types:\n");
	printf("  0: soft shutdown or powerdown, depending on battery status\n");
	printf("  1: soft shutdown followed by powerdown\n");
	printf("  2: instant power off\n");
	printf("  3: power off with grace period\n");
	printf("Modes 0-1 will make the UPS come back when power returns\n");
	printf("Modes 2-3 will make the UPS stay turned off when power returns\n");
}

/* list flags and values that you want to receive via -x */
void upsdrv_makevartable(void)
{
	upsdebugx(2, "in upsdrv_makevartable()");

	addvar(VAR_VALUE, PN_VAR_SDTYPE,
		"Specify shutdown type (1-3, default 0)");
	addvar(VAR_VALUE, PN_VAR_COMMUNITY,
		"Set SNMP community name (default=public)");
}

void upsdrv_banner(void)
{
	upslogx(1,"Network UPS Tools version %s", UPS_VERSION);
	upslogx(1,"APC Powernet SNMP driver version %s", upsdrv_version);
	upslogx(1,"Revision %s\n", upsdrv_cvsid);

	/* main.c prints a warning. */
	experimental_driver = 1;
}

void upsdrv_initups(void)
{
	pn_info_t *pn_info_p;
	char model[PN_INFOSIZE];
	bool status;

	upsdebugx(2, "in upsdrv_initups()");

	/* init SNMP library, etc... */
	pn_startup();

	/* get value of UPS model OID */
	pn_info_p = pn_find_info(INFO_MODEL);
	status = pn_snmp_get_str(pn_info_p->OID, model, sizeof(model));
	if (status == TRUE)
		upslogx(0, "detected %s on host %s", model, device_path);
	else
		fatalx("Powernet MIB %s wasn't found on %s",
			OID_POWERNET_MIB, g_snmp_sess.peername);   
}

/* tell main how many items you need */
int upsdrv_infomax(void)
{
	return NUM_INFO_ITEMS;
}
