/*
 * powercom.c - model specific routines for following units:
 *  -Trust 425/625
 *  -Powercom
 *  -Advice Partner/King PR750
 *    See http://www.advice.co.il/product/inter/ups.html for its specifications.
 *    This model is based on PowerCom (www.powercom.com) models.
 *  -Socomec Sicon Egys 420
 *
 * Read docs/powercom.txt for other models and manufactures
 *
 * $Id: - will be filled in on next CVS add/update $
 *
 * Status:
 *  20011022/Revision 0.1 - Peter Bieringer <pb@bieringer.de>
 *   - porting done, looks like working
 *  20011208/Revision 0.2 - Peter Bieringer <pb@bieringer.de>
 *   - add some debug code
 *   - add new option "subtype"
 *   - add support for a 16 byte sending UPS (KP625AP)
 *   - shutdown function checked, but 'Trust' wakes up after a few seconds...
 *  20020629/Revision 0.3 - Simon Rozman <simon@rozman.net>
 *   - add different serial port init. for each subtype
 *   - add support for Socomec Sycon Egys 420
 *   - add support for different calculation parameters for each subtype
 *   - add support for output voltage and output frequency monitoring
 *  20020701/Revision 0.4 - Simon Rozman <simon@rozman.net> proposed by Shaul Karl
 *   - subtypes specific info moved to struct subtype
 *
 *   	
 * Copyrights:
 * (C) 2002 Simon Rozman <simon@rozman.net>
 *  Added support for Egys
 
 * (C) 2001 Peter Bieringer <pb@bieringer.de>
 *  Porting old style "powercom" to new style "newpowercom"
 *  
 * (C) 2000  Shaul Karl <shaulk@israsrv.net.il>
 *  Creating old style "powercom"
 *   Heavily based on
 *    ups-trust425+625.c - model specific routines for Trust UPS 425/625
 *    Copyright (C) 1999  Peter Bieringer <pb@bieringer.de>
 *                              
 * A technical description of the interface to the UPS that this program
 *  implements is found in docs/\*trust* files.
 *                                     
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *                            
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *    
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 * 
 */ 

#include "main.h"
#include "powercom.h"

#define POWERCOM_DRIVER_VERSION      "$Revision: 0.4 $"
#define NUM_OF_SUBTYPES              (sizeof (subtypes) / sizeof (*subtypes))

/* variables used by module */
static unsigned char powercom_raw_data[MAX_NUM_OF_BYTES_FROM_UPS]; /* raw data reveived from UPS */
static unsigned int  powercom_linevoltage = 230U; /* line voltage, can be defined via command line option */
static char *powercom_manufacturer = "PowerCom";
static char *powercom_modelname    = "Unknown";
static char *powercom_serialnumber = "Unknown";
static char *powercom_subtype_name = "Trust";
static unsigned int subtype_index = 0;

/* forward declaration of functions used to setup flow control */
static void dtr0rts1 ();
static void no_flow_control ();

/* forward declaration of functions used to validate raw data */
static int validate_Trust ();
static int validate_KP625AP ();

/* struct defining subtypes */
static struct subtype subtypes[] = {
        {
                "Trust",
                11,
                dtr0rts1,
                validate_Trust,
                {  0.00020997, 0.00020928 },
                {  6.1343, -0.3808,  4.3110,  0.1811 },
                {  5.0000,  0.3268,  -825.00,  4.5639, -835.82 },
                {  1.9216, -0.0977,  0.9545,  0.0000 },
        },
        {
                "KP625AP",
                16,
                dtr0rts1,
                validate_KP625AP,
                {  0.00020997, 0.00020928 },
                {  6.1343, -0.3808,  4.3110,  0.1811 },
                {  5.0000,  0.3268,  -825.00,  4.5639, -835.82 },
                {  1.9216, -0.0977,  0.9545,  0.0000 },
        },
        {
                "Egys",
                16,
                no_flow_control,
                validate_KP625AP,
                {  0.00020997, 0.00020928 },
                {  6.1343, -0.3808,  1.3333,  0.6667 },
                {  5.0000,  0.3268,  -825.00,  2.2105, -355.37 },
                {  1.9216, -0.0977,  0.9545,  0.0000 },
        },
};


/* used external variables */
extern int sddelay;          /* shutdown delay, set by "-d $delay" in main.c */
extern int do_forceshutdown; /* shutdown delay, set by "-k" in main.c */


/*
 * local used functions
 */

static void powercom_shutdown(void)
{
        printf ("Initiating UPS shutdown!\n");
        
        upssendchar (SHUTDOWN);
	upssendchar (SEC_FOR_POWERKILL);
			
        exit (0);
}


/* registered instant commands */
static void instcmd (int auxcmd, int dlen, char *data)
{
        /* TODO: reply to upsd? */

        switch (auxcmd) {
                case CMD_BTEST1:        /* start battery test */
                        upssendchar (BATTERY_TEST);
                        break;
                default:
                        upslogx(LOG_INFO, "instcmd: unknown type 0x%04x", auxcmd);
        }
}


/* wait for buflen bytes from upsfd and store in buf */
static int read_raw_data (int upsfd, unsigned char *buf, int buflen)
{
        int     counter = 0;
        struct  sigaction sa;
        sigset_t pow_sigmask;

	sa.sa_handler = timeout;
        sigemptyset (&pow_sigmask);
        sa.sa_mask = pow_sigmask;
        sa.sa_flags = 0;
        sigaction (SIGALRM, &sa, NULL);

        alarm (3);

        for (counter = 0; counter < buflen; counter++) 

		if (!(read (upsfd, buf + counter, 1))) {

			alarm (0);
		        signal (SIGALRM, SIG_IGN);
			return counter;
			
		}

	nolongertimeout();
		
	return counter;
}
			

/* set DTR and RTS lines on a serial port to supply a passive
 * serial interface: DTR to 0 (-V), RTS to 1 (+V)
 */
static void dtr0rts1 ()
{
	int dtr_bit = TIOCM_DTR;
	int rts_bit = TIOCM_RTS;

	upsdebugx(2, "DTR => 0, RTS => 1");

	/* set DTR to low and RTS to high */
	ioctl(upsfd, TIOCMBIC, &dtr_bit);
	ioctl(upsfd, TIOCMBIS, &rts_bit);
}


/* clear any flow control */
static void no_flow_control ()
{
	struct termios tio;
	
	tcgetattr (upsfd, &tio);
	
	tio.c_iflag &= ~ (IXON | IXOFF);
	tio.c_cc[VSTART] = _POSIX_VDISABLE;
	tio.c_cc[VSTOP] = _POSIX_VDISABLE;
				
	upsdebugx(2, "Flow control disable");

	/* disable any flow control */
	tcsetattr(upsfd, TCSANOW, &tio);
}


/* sane check for returned buffer from Trust */
static int validate_Trust ()
{
	int flag = 0;
	
	/* Trust */
	if (	(powercom_raw_data[5] != 0) 
		|| (powercom_raw_data[7] != 0) 
		|| (powercom_raw_data[8] != 0)
	)
		flag = 1;
		
	return flag;
}


/* sane check for returned buffer from KP625AP */
static int validate_KP625AP ()
{
	int flag = 0;

	/* KP625AP */
	if (	(powercom_raw_data[5] != 0x80) 
		|| (powercom_raw_data[7] != 0) 
		|| (powercom_raw_data[8] != 0)
	)
		flag = 1;
	
	return flag;
}

/* install pointers to functions for msg handlers called from msgparse */
static void setuphandlers(void)
{
        upsh.instcmd = instcmd;
}


/*
 * global used functions
 */

/* update information */
void upsdrv_updateinfo(void)
{
	char	val[32];
	float	tmp;
	int	i, c;
	
	/* send trigger char to UPS */
	if (upssendchar (SEND_DATA) != 1) {
		upslogx(LOG_NOTICE, "writing error");
		return;
	} else {
		upsdebugx(5, "Num of bytes requested for reading from UPS: %d", subtypes[subtype_index].num_of_bytes_from_ups);

		c = read_raw_data(upsfd, powercom_raw_data, subtypes[subtype_index].num_of_bytes_from_ups);
	
		if (c != subtypes[subtype_index].num_of_bytes_from_ups) {
			upslogx(LOG_NOTICE, "data receiving error (%d instead of %d bytes)", c, subtypes[subtype_index].num_of_bytes_from_ups);
			return;
		} else
			upsdebugx(5, "Num of bytes received from UPS: %d", c);

	};

	/* optional dump of raw data */
	if (nut_debug_level > 4) {
		printf("Raw data from UPS:\n");
		for (i = 0; i < subtypes[subtype_index].num_of_bytes_from_ups; i++) {
	 		printf("%2d 0x%02x (%c)\n", i, powercom_raw_data[i], powercom_raw_data[i]>=0x20 ? powercom_raw_data[i] : ' ');
		};
	};

	/* validate raw data for correctness */
	if (subtypes[subtype_index].validate_raw_data() != 0) {
		upslogx(LOG_NOTICE, "data receiving error (validation check)");
		return;
	};
	
	/* set ACFREQ */
	upsdebugx(3, "ACFREQ   (raw data): [raw: %u]", powercom_raw_data[IN_AC_FREQ]);
	setinfo(INFO_ACFREQ, "%02.2f", powercom_raw_data[IN_AC_FREQ] ? 
			1.0 / (subtypes[subtype_index].freq[0] * powercom_raw_data[IN_AC_FREQ] + subtypes[subtype_index].freq[1]) :
			0);
	upsdebugx(2, "ACFREQ: %s", getdata(INFO_ACFREQ));

	/* set OUT_FREQ */
	upsdebugx(3, "OUT_FREQ   (raw data): [raw: %u]", powercom_raw_data[OUT_AC_FREQ]);
	setinfo(INFO_OUT_FREQ, "%02.2f", powercom_raw_data[OUT_AC_FREQ] ? 
			1.0 / (subtypes[subtype_index].freq[0] * powercom_raw_data[OUT_AC_FREQ] + subtypes[subtype_index].freq[1]) :
			0);
	upsdebugx(2, "OUT_FREQ: %s", getdata(INFO_OUT_FREQ));

	/* set LOADPCT */	
	upsdebugx(3, "LOADPCT  (raw data): [raw: %u]", powercom_raw_data[LOAD_LEVEL]);
	setinfo(INFO_LOADPCT, "%03.1f", tmp = powercom_raw_data[STATUS_A] & MAINS_FAILURE ?
	 		subtypes[subtype_index].loadpct[0] * powercom_raw_data[LOAD_LEVEL] + subtypes[subtype_index].loadpct[1] :
			subtypes[subtype_index].loadpct[2] * powercom_raw_data[LOAD_LEVEL] + subtypes[subtype_index].loadpct[3]);
	upsdebugx(2, "LOADPCT: %s", getdata(INFO_LOADPCT));

	/* set BATTPCT */
	upsdebugx(3, "BATTPCT (raw data): [raw: %u]", powercom_raw_data[BAT_LEVEL]);
	setinfo(INFO_BATTPCT, "%03.1f", powercom_raw_data[STATUS_A] & MAINS_FAILURE ?
			subtypes[subtype_index].battpct[0] * powercom_raw_data[BAT_LEVEL] + subtypes[subtype_index].battpct[1] * tmp + subtypes[subtype_index].battpct[2] :
			subtypes[subtype_index].battpct[3] * powercom_raw_data[BAT_LEVEL] + subtypes[subtype_index].battpct[4]);
	upsdebugx(2, "BATTPCT: %s", getdata(INFO_BATTPCT));

	/* set UTILITY */	
	upsdebugx(3, "UTILITY (raw data): [raw: %u]", powercom_raw_data[IN_AC_VOLT]);
	setinfo(INFO_UTILITY, "%03.1f", tmp = powercom_linevoltage >= 220 ?
			subtypes[subtype_index].voltage[0] * powercom_raw_data[IN_AC_VOLT] + subtypes[subtype_index].voltage[1] :
			subtypes[subtype_index].voltage[2] * powercom_raw_data[IN_AC_VOLT] + subtypes[subtype_index].voltage[3]);
	upsdebugx(2, "UTILITY: %s", getdata(INFO_UTILITY));
	
	/* set OUTVOLT */	
	upsdebugx(3, "OUTVOLT (raw data): [raw: %u]", powercom_raw_data[OUT_AC_VOLT]);
	setinfo(INFO_OUTVOLT, "%03.1f", powercom_linevoltage >= 220 ?
			subtypes[subtype_index].voltage[0] * powercom_raw_data[OUT_AC_VOLT] + subtypes[subtype_index].voltage[1] :
			subtypes[subtype_index].voltage[2] * powercom_raw_data[OUT_AC_VOLT] + subtypes[subtype_index].voltage[3]);
	upsdebugx(2, "OUTVOLT: %s", getdata(INFO_OUTVOLT));
	
	*val = 0;
	if (!(powercom_raw_data[STATUS_A] & MAINS_FAILURE))
		!(powercom_raw_data[STATUS_A] & OFF) ? 
			strcat(val, "OL ") : strcat(val, "OFF ");
	else strcat(val, "OB ");
	if (powercom_raw_data[STATUS_A] & LOW_BAT)
		strcat(val, "LB ");
	if (powercom_raw_data[STATUS_A] & AVR_ON)
		tmp < powercom_linevoltage ?
			strcat(val, "BOOST ") : strcat(val, "TRIM ");
	if (powercom_raw_data[STATUS_A] & OVERLOAD)
		strcat (val, "OVER ");
	if (powercom_raw_data[STATUS_B] & BAD_BAT)
		strcat (val, "RB ");
	if (powercom_raw_data[STATUS_B] & TEST)
		strcat (val, "TEST ");
	*(val + strlen(val)) = 0;
	setinfo(INFO_STATUS, "%s", val);
	upsdebugx(2, "STATUS: %s", getdata(INFO_STATUS));
	
	writeinfo();
}


/* shutdown UPS */
void upsdrv_shutdown(void)
{
	if (do_forceshutdown == 1) {
		/* power down the attached load immediately */
		printf("Forced UPS shutdown triggered, do it...\n");
		powercom_shutdown();
	} else {
		/* power down the attached load after the given delay */
		printf("UPS shutdown with '%d' seconds delay triggered, wait now...\n", sddelay);
		sleep(sddelay);
		powercom_shutdown();
	};
}


/* initialize UPS */
void upsdrv_initups(void)
{
	int tmp, i;

	/* check manufacturer name from arguments */
	if (getval("manufacturer") != NULL) {
		powercom_manufacturer = getval("manufacturer");
	};
	
	/* check model name from arguments */
	if (getval("modelname") != NULL) {
		powercom_modelname = getval("modelname");
	};
	
	/* check serial number from arguments */
	if (getval("serialnumber") != NULL) {
		powercom_serialnumber = getval("serialnumber");
	};
	
	/* get and check subtype */
	if (getval("subtype") != NULL) {
		powercom_subtype_name = getval("subtype");
		for (i = 0; i < NUM_OF_SUBTYPES; i++) {
			if (strcmp(subtypes[i].name, powercom_subtype_name) == 0)
				break;
		};
		if (i < NUM_OF_SUBTYPES) {
			/* we found a subtype */
			subtype_index = i;	
		} else {
			/* no such subtype exists */
			printf("Given UPS subtype '%s' isn't valid!\n", powercom_subtype_name);
			exit (1);
		};
	};
	
	/* check line voltage from arguments */
	if (getval("linevoltage") != NULL) {
		tmp = atoi(getval("linevoltage"));
		if (! ( (tmp >= 220 && tmp <= 240) || (tmp >= 110 && tmp <= 120) ) ) {
			printf("Given line voltage '%d' is out of range (110-120 or 220-240 V)\n", tmp);
			exit (1);
		};
		powercom_linevoltage  = (unsigned int) tmp;
	};

	upsdebugx(1, "Values of arguments:");
	upsdebugx(1, " manufacturer : '%s'", powercom_manufacturer);
	upsdebugx(1, " model name   : '%s'", powercom_modelname);
	upsdebugx(1, " serial number: '%s'", powercom_serialnumber);
	upsdebugx(1, " line voltage : '%u'", powercom_linevoltage);
	upsdebugx(1, " subtype      : '%s'", powercom_subtype_name);

	/* open serial port */
	open_serial (device_path, B1200);
	
	/* setup flow control */
	subtypes[subtype_index].setup_flow_control();
}


/* display help */
void upsdrv_help(void)
{
	printf("\n");
	return;
}


/* display banner */
void upsdrv_banner(void)
{
	printf ("Network UPS Tools (version %s) - PowerCom and similars protocol driver\n",
		UPS_VERSION);
	printf ("\tDriver %s\n",
		POWERCOM_DRIVER_VERSION);
}


/* tell main how many entries we need */
int upsdrv_infomax(void)
{
	/* TODO: define constant elsewhere! */
	/* must be equal or greater than numer of addinfo calls below */
        return (16);
}


/* initialize information */
void upsdrv_initinfo(void)
{
        /* write constant data for this model */
	addinfo (INFO_MFR,    powercom_manufacturer, FLAG_STRING, 8);
	addinfo (INFO_MODEL,  powercom_modelname,    FLAG_STRING, strlen(powercom_modelname));
	addinfo (INFO_SERIAL, powercom_serialnumber, FLAG_STRING, strlen(powercom_serialnumber));

        /* add other things to monitor */
	addinfo (INFO_ACFREQ,   "", 0, 0);
	addinfo (INFO_OUT_FREQ, "", 0, 0);
	addinfo (INFO_BATTPCT,  "", 0, 0);
	addinfo (INFO_LOADPCT,  "", 0, 0);
	addinfo (INFO_STATUS,   "", 0, 0);
	addinfo (INFO_UTILITY,  "", 0, 0);
	addinfo (INFO_OUTVOLT,  "", 0, 0);

	/* now add the instant commands */
        addinfo (INFO_INSTCMD, "", 0, CMD_BTEST1);

	setuphandlers();
}


/* define possible arguments */
void upsdrv_makevartable(void)
{
	addvar(VAR_VALUE, "manufacturer",  "Specify manufacturer name (default: 'PowerCom')");
	addvar(VAR_VALUE, "linevoltage",   "Specify line voltage (110-120 or 220-240 V), because it cannot detect automagically (default: 230 V)");
	addvar(VAR_VALUE, "modelname",     "Specify model name, because it cannot detect automagically (default: Unknown)");
	addvar(VAR_VALUE, "serialnumber",  "Specify serial number, because it cannot detect automagically (default: Unknown)");
	addvar(VAR_VALUE, "subtype",       "Subtype of UPS like 'Trust', 'KP625AP' or 'Egys' (default: 'Trust')");
}


