/* liebert.c - support for Liebert UPS models via MultiLink cable.

   Copyright (C) 2002  Russell Kroll <rkroll@exploits.org>

   Based on old-style multilink.c driver:

   Copyright (C) 2001  Rick Lyons <rick@powerup.com.au>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#include "main.h"
#include <sys/ioctl.h>

#define	ML_ONBATTERY	0x55

void upsdrv_shutdown(void)
{
	/* XXX: replace with a proper shutdown function (raise DTR) */

	/* worse yet: stock cables don't support shutdown at all */

	fatalx("shutdown not supported");
}

void upsdrv_initinfo(void)
{
	char	*tmp;

	tmp = getval("mfr");

	if (!tmp)
		addinfo(INFO_MFR, "Liebert", 0, 0);
	else
		addinfo(INFO_MFR, tmp, 0, 0);

	tmp = getval("model");

	if (!tmp)
		addinfo(INFO_MODEL, "MultiLink", 0, 0);
	else
		addinfo(INFO_MODEL, tmp, 0, 0);

	addinfo(INFO_STATUS, "", 0, 0);
}

/* normal idle loop - keep up with the current state of the UPS */
void upsdrv_updateinfo(void)
{
	int	flags, ob, lb, ret;
	char	c;

	ob = 0;

	/* the UPS connects RX to TX when on battery, so test for loopback */

	c = ML_ONBATTERY;
	write(upsfd, &c, 1);
	if (read(upsfd, &c, 1) == 1) {
		while (read(upsfd, &c, 1) == 1)
			continue;
		if (c == ML_ONBATTERY)
			ob = 1;
	}
	
	ret = ioctl(upsfd, TIOCMGET, &flags);

	if (ret != 0) {
		upslog(LOG_INFO, "ioctl failed");
		return;
	}

	lb = (flags & TIOCM_CD);

	status_init();

	if (ob)
		status_set("OB");	/* on battery */
	else
		status_set("OL");	/* on line */

	if (lb)
		status_set("LB");	/* low battery */

	status_commit();
	writeinfo();
}

/*  Open the serial port.  The upscommon routines aren't used because
    they lose bits off the MCR required to make TX/RX work, and we
    set up the port to make use of VMIN/VTIME.
*/
static void open_serial_ml(const char *port)
{
	int	flags;
	struct termios tio;

	signal(SIGALRM, openfail);
	alarm(3);

	if ((upsfd = open(port, O_RDWR)) == -1)
		fatal("Unable to open %s", port);

	alarm(0);

	lockport(upsfd, port);

	if (ioctl(upsfd, TIOCMGET, &flags))
		fatal("get ioctl");
	flags |= TIOCM_RTS;
	if (ioctl(upsfd, TIOCMSET, &flags))
		fatal("set ioctl");

	tcgetattr(upsfd, &tio);
	cfmakeraw(&tio);
	tio.c_cc[VMIN] = 0;
	tio.c_cc[VTIME] = 1;
	tcsetattr(upsfd, TCSANOW, &tio);
}

void upsdrv_banner(void)
{
	printf("Network UPS Tools - Liebert MultiLink UPS driver 1.0 (%s)\n", 
		UPS_VERSION);

	experimental_driver = 1;
}

void upsdrv_makevartable(void)
{
	addvar(VAR_VALUE, "mfr", "Override manufacturer name");
	addvar(VAR_VALUE, "model", "Override model name");
}

void upsdrv_help(void)
{
}

void upsdrv_initups(void)
{
	open_serial_ml(device_path);
}

int upsdrv_infomax(void)
{
	return 8;
}
