/* etapro.c - model specific routines for ETA UPS

   Copyright (C) 2002  Marek Michalkiewicz  <marekm@amelek.gda.pl>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

/*
   This driver is for the ETA UPS (http://www.eta.com.pl/) with the
   "PRO" option (available at small added cost, highly recommended).
   All units (even without that option) should also work in "dumb"
   mode with the genericups driver (type 7 or 10), but in that mode
   shutdown only works when running on battery.

   Tested with ETA mini+UPS 720 PRO.  Thanks to ETA for help with
   protocol documentation, no free UPS though (but they still can
   send me another one if they like me ;-).

   Shutdown should work even when on line, so this should help avoid
   power races (system remaining in halted or "ATX standby" state,
   requiring manual intervention).  Delay from power off to power on
   can be set in software, currently hardcoded to 15 seconds.

   Instant commands CMD_OFF and CMD_ON should work (not tested yet).
   Be careful with CMD_OFF - it turns off the load after one second.

   Known issues:
    - larger units (>= 1000VA) have a 24V battery, so the battery
      voltage reported should be multiplied by 2 if the INFO_MODEL
      string indicates such a larger unit.
    - load percentage is only measured when running on battery, and
      is reported as 0 when on line.  This seems to be a hardware
      limitation of the UPS, so we can't do much about it...
    - UPS does not provide any remaining battery charge (or time at
      current load) information, but we should be able to estimate it
      based on battery voltage, load percentage and UPS model.
    - error handling not tested (we assume that the UPS is always
      correctly connected to the serial port).
 */

#include <sys/ioctl.h>
#include "main.h"

static int
etapro_get_response(const char *resp_type)
{
	char tmp[256];
	char *cp;
	int ret;
	unsigned int val;

	ret = upsrecv(tmp, sizeof(tmp), '\n', "");
	if (ret < 0) {
		upslogx(LOG_ERR, "no response from UPS");
		return -1;
	}
	/* Search for start of response (skip any echoed back command).  */
	cp = strstr(tmp, resp_type);
	if (!cp || *cp == '\0' || cp[strlen(cp) - 1] != '\r') {
		upslogx(LOG_ERR, "bad response (%s)", tmp);
		return -1;
	}
	cp[strlen(cp) - 1] = '\0';  /* remove the CR */

	switch (cp[1]) {
		/* Handle ASCII text responses directly here.  */
	case 'R':
		setinfo(INFO_MFR, "%s", cp + 2);
		return 0;
	case 'S':
		setinfo(INFO_MODEL, "%s", cp + 2);
		return 0;
	case 'T':
		setinfo(INFO_MFRDATE, "%s", cp + 2);
		return 0;
	}
	/* Handle all other responses as hexadecimal numbers.  */
	val = 0;
	if (sscanf(cp + 2, "%x", &val) != 1) {
		upslogx(LOG_ERR, "bad response format (%s)", tmp);
		return -1;
	}
	return val;
}

static void
etapro_set_on_timer(int seconds)
{
	int x;

	if (seconds == 0) {  /* cancel the running timer */
		upssend("RS\r");
		x = etapro_get_response("SV");
		if (x == 0x30)
			return;  /* OK */
	} else {
		if (seconds > 0x7fff) {  /* minutes */
			seconds = (seconds + 59) / 60;
			if (seconds > 0x7fff)
				seconds = 0x7fff;
			printf("UPS on in %d minutes\n", seconds);
			seconds |= 0x8000;
		} else {
			printf("UPS on in %d seconds\n", seconds);
		}

		upssend("RN%04X\r", seconds);
		x = etapro_get_response("SV");
		if (x == 0x20)
			return;  /* OK */
	}
	upslogx(LOG_ERR, "etapro_set_on_timer: error, status=0x%02x", x);
}

static void
etapro_set_off_timer(int seconds)
{
	int x;

	if (seconds == 0) {  /* cancel the running timer */
		upssend("RR\r");
		x = etapro_get_response("SV");
		if (x == 0x10)
			return;  /* OK */
	} else {
		if (seconds > 0x7fff) {  /* minutes */
			seconds /= 60;
			if (seconds > 0x7fff)
				seconds = 0x7fff;
			printf("UPS off in %d minutes\n", seconds);
			seconds |= 0x8000;
		} else {
			printf("UPS off in %d seconds\n", seconds);
		}

		upssend("RO%04X\r", seconds);
		x = etapro_get_response("SV");
		if (x == 0)
			return;  /* OK */
	}
	upslogx(LOG_ERR, "etapro_set_off_timer: error, status=0x%02x", x);
}

void
instcmd(int auxcmd, int dlen, char *data)
{
	switch (auxcmd) {
	case CMD_OFF:
		etapro_set_off_timer(1);
		break;
	case CMD_ON:
		etapro_set_on_timer(1);
		break;
	case CMD_SHUTDOWN:
		upsdrv_shutdown();
		break;
	default:
		upslogx(LOG_INFO, "instcmd: unknown type 0x%04x", auxcmd);
	}
}

void
upsdrv_initinfo(void)
{
	addinfo(INFO_MFR, "", 0, 0);
	addinfo(INFO_MODEL, "", 0, 0);
	addinfo(INFO_MFRDATE, "", 0, 0);

	addinfo(INFO_STATUS, "", 0, 0);
	addinfo(INFO_UTILITY, "", 0, 0);
	addinfo(INFO_OUTVOLT, "", 0, 0);
	addinfo(INFO_BATTVOLT, "", 0, 0);
	addinfo(INFO_LOADPCT, "", 0, 0);

	addinfo(INFO_INSTCMD, "", 0, CMD_OFF);
	addinfo(INFO_INSTCMD, "", 0, CMD_ON);
	addinfo(INFO_INSTCMD, "", 0, CMD_SHUTDOWN);

	/* First command after power on returns junk - ignore it.  */
	upssend("RI\r");
	sleep(1);
	upsflushin(1, nut_debug_level, "");

	upsdrv_updateinfo();

	upsh.instcmd = instcmd;
}

void
upsdrv_updateinfo(void)
{
	char status[256];
	int x, flags;
	double utility, outvolt, battvolt, loadpct;

	upssend("RI\r");  /* identify */

	x = etapro_get_response("SR");  /* manufacturer */
	if (x < 0)
		return;
	x = etapro_get_response("SS");  /* model */
	if (x < 0)
		return;
	x = etapro_get_response("ST");  /* mfr date */
	if (x < 0)
		return;
	x = etapro_get_response("SU");  /* UPS ident */
	if (x < 0)
		return;

	upssend("RP\r");  /* read measurements */

	x = etapro_get_response("SO");  /* status flags */
	if (x < 0)
		return;
	flags = x;

	x = etapro_get_response("SG");  /* input voltage, 0xFF = 270V */
	if (x < 0)
		return;
	utility = (270.0 / 255) * x;

	x = etapro_get_response("SH");  /* output voltage, 0xFF = 270V */
	if (x < 0)
		return;
	outvolt = (270.0 / 255) * x;

	x = etapro_get_response("SI");  /* battery voltage, 0xFF = 14V */
	if (x < 0)
		return;
	/* TODO: >= 1000VA models have a 24V battery (max 28V) - check
	   the model string returned by the RI command.  */
	battvolt = (14.0 / 255) * x;

	x = etapro_get_response("SL");  /* load (on battery), 0xFF = 150% */
	if (x < 0)
		return;
	loadpct = (150.0 / 255) * x;

	x = etapro_get_response("SN");  /* time running on battery */
	if (x < 0)
		return;
	/* This is the time how long the UPS has been running on battery
	   (in seconds, reset to zero after power returns), but there
	   seems to be no INFO_xxx variable defined for this yet...  */

	status[0] = '\0';
	if (!(flags & 0x02))
		strcat(status, "OFF");
	else if (flags & 0x01)
		strcat(status, "OL");
	else
		strcat(status, "OB");

	if (!(flags & 0x04))
		strcat(status, " LB");

	/* TODO bit 3: 1 = ok, 0 = fault */

	if (flags & 0x10)
		strcat(status, " BOOST");

	if (loadpct > 100.0)
		strcat(status, " OVER");

	/* Battery voltage out of range (lower than LB, or too high).  */
	if (flags & 0x20)
		strcat(status, " RB");

	/* TODO bit 6: 1 = charging, 0 = full */

	setinfo(INFO_STATUS, "%s", status);
	setinfo(INFO_UTILITY, "%03.1f", utility);
	setinfo(INFO_OUTVOLT, "%03.1f", outvolt);
	setinfo(INFO_BATTVOLT, "%02.2f", battvolt);
	setinfo(INFO_LOADPCT, "%03.1f", loadpct);

	writeinfo();
}

/* TODO: delays should be tunable, the UPS supports max 32767 minutes.  */

/* Shutdown command to off delay in seconds.  */
#define SHUTDOWN_GRACE_TIME 10

/* Shutdown to return delay in seconds.  */
#define SHUTDOWN_TO_RETURN_TIME 15

void
upsdrv_shutdown(void)
{
	etapro_set_on_timer(SHUTDOWN_GRACE_TIME + SHUTDOWN_TO_RETURN_TIME);
	etapro_set_off_timer(SHUTDOWN_GRACE_TIME);
}

void
upsdrv_help(void)
{
}

void
upsdrv_makevartable(void)
{
}

void
upsdrv_banner(void)
{
	printf("Network UPS Tools - ETA PRO driver 0.01 (%s)\n\n", UPS_VERSION);
}

void
upsdrv_initups(void)
{
	int dtr_bit = TIOCM_DTR;
	int rts_bit = TIOCM_RTS;

	open_serial(device_path, B1200);
	ioctl(upsfd, TIOCMBIC, &dtr_bit);
	ioctl(upsfd, TIOCMBIS, &rts_bit);
}

int
upsdrv_infomax(void)
{
	return 20;
}
