/* bestferrups801-807.c - model specific routines for 
 *                        Best FerrUPS 801-807 models
 *
 * Copyright (C) 2002  Jonathan Corbin <jonboy@rocketmonkey.org>
 * Copyright (C) 1999  Russell Kroll <rkroll@exploits.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "main.h"

#define ENDCHAR	'>'
#define IGNCHARS	"f\n\r"

#define ACTIVE_ALARM "1"

#define TIME "0"
#define DATE "10"
#define LOAD_PERCENTAGE "16"
#define POWER "17"

#define USER_PASS "377"
/* disclaimer: this password yields a higher access to the ups than is
 * normally available.  This driver only uses this password to change the date
 * and time (which, according to the spec, don't need this password to change,
 * but at least with firmware version 8.01, the spec is wrong and you need the
 * password).  Use this password at your own risk.  You could very easily break
 * the ups with this.
 *
 * This password can be changed, so if you have changed it on the ups, you will
 * need to change it on the ups as well.
 *
 * TODO: allow password change as command line option
 */

/* This function clears the buffer on the ups.  It takes several seconds to
 * execute, so it should only be called when necessary
 */
void clearUpsBuffer() {
	char data[1];

	/* Turn off timeout failure logging */
	flag_timeoutfailure = -1;
	
	/* clear the happy buffer */
	while (upsrecv (data, 1, ENDCHAR, IGNCHARS) != -1);
	
	/* Turn timeout failure logging back on */
	flag_timeoutfailure = 0;
}

void hw_set (char *paramNum, char *newval) {
	char go1[20];
	char go2[20];
	
	/* send a carriage return to clear anything extraneous from the ups keyboard
	 * buffer
	 */
	upssend ("\r");

	/* send the user password to the ups */
	strcpy(go1, "pw ");
	upssend (strcat(strcat(go1, USER_PASS), "\r"));

	/* change the parameter */
	strcpy(go2, "pr ");
	upssend (strcat(strcat(strcat(strcat(go2, paramNum), " "), newval), "\r"));

	/* logout */
	upssend ("pw\r");
	
	clearUpsBuffer();
}

/* data should be long enough to hold whatever data will be returned.  Longest
 * currently is 9.  This is not checked in the function, so consider it a
 * precondition.
 *
 * This function can easily be expanded to handle any value the ups can provide.
 * Simply add the value as a constant at the top of this file, then add the
 * appropriate length and offset values to this function.
 *
 * The ups buffer should be clear before this function is called.  This can be
 * accomplished by calling clearUpsBuffer() before calling this function.  This
 * function will leave the buffer cleared, so clearUpsBuffer does not have to be
 * called between calls of this function.
 */
char *hw_get (char *paramNum, char *data) {
	char go[20];
	char temp[30];
	int i; /* for loop counter */
	int length = 0;
	int offset = 0;

	/* Get the value for length and offset
	 *
	 * TODO: Get these values from some sort of table
	 */
	if (!strcmp(paramNum, TIME)) {
		length = 8;
		offset = 11;
	} else if (!strcmp(paramNum, DATE)) {
		length = 8;
		offset = 12;
	} else if (!strcmp(paramNum, LOAD_PERCENTAGE)) {
		length = 3;
		offset = 17;
	} else if (!strcmp(paramNum, POWER)) {
		length = 3;
		offset = 17;
	}

	/* ask the ups for the parameter's value */
	strcpy(go, "p ");
	upssend (strcat(strcat(go, paramNum), "\r"));

	/* get the value from the ups */
	upsrecv (temp, sizeof(temp), ENDCHAR, IGNCHARS);
	for (i = 0; i < length; i++) {
		data[i] = temp[offset + i];
	}
	data[length] = '\0';

	return data;
}

void setvar (int auxcmd, int dlen, char *data)
{
	upslog(LOG_INFO, "test10");
	/* TODO: look this up from a table? */
	switch (auxcmd) {
		case INFO_TIME:
			upslog(LOG_INFO, "test11");
			hw_set(TIME, data);
			setinfo(INFO_TIME, "%s", hw_get(TIME, data));
			break;
		case INFO_DATE:
			upslog(LOG_INFO, "test12");
			hw_set(DATE, data);
			setinfo(INFO_DATE, "%s", hw_get(DATE, data));
			break;
		default:
			upslogx(LOG_ERR, "setvar: unknown type 0x%04x\n", auxcmd);
	}
	upslog(LOG_INFO, "test13");
}

void upsdrv_initinfo(void)
{
	addinfo(INFO_MFR, "Best Power", 0, 0);
	addinfo(INFO_MODEL, "FERRUPS 8.01 - 8.07", 0, 0);

	addinfo(INFO_ALRM_OVERLOAD, "", 0, 0);
	addinfo(INFO_ALRM_UPSSHUT, "", 0, 0);
	addinfo(INFO_ALRM_GENERAL, "", 0, 0);
	addinfo(INFO_ALRM_BADINPUT, "", 0, 0);
	addinfo(INFO_ALRM_TEMP, "", 0, 0);
	addinfo(INFO_STATUS, "", 0, 0);
	addinfo(INFO_UTILITY, "", 0, 0);
	addinfo(INFO_OUTVOLT, "", 0, 0);
	addinfo(INFO_CURRENT, "", 0, 0);
	addinfo(INFO_BATT_CURRENT, "", 0, 0);
	addinfo(INFO_BATTVOLT, "", 0, 0);
	addinfo(INFO_ACFREQ, "", 0, 0);
	addinfo(INFO_RUNTIME, "", 0, 0);
	addinfo(INFO_UPSTEMP, "", 0, 0);
	addinfo(INFO_LOADPCT, "", 0, 0);
	addinfo(INFO_OUT_VA, "", 0, 0);
	addinfo(INFO_OUT_POWER, "", 0, 0);
	addinfo(INFO_TIME, "", FLAG_RW | FLAG_STRING, 8); /* This info variable can be changed */
	addinfo(INFO_DATE, "", FLAG_RW | FLAG_STRING, 8); /* This info variable can also be changed */

	/* upsh.instcmd = instcmd; */

	/* Install handler for changing variables. */
	upsh.setvar = setvar;
}

void upsdrv_updateinfo(void)
{
	/* int flags; */
	char data[9]; /* necessary for hw_get calls */
	char temp[86];
	char status[16];
	char byte, byte1, byte2;
	int digit[4];
	char digit6[7];
	char time[9];
	int voltageIn, voltageOut, runtimeMinutesRemaining, i;
	int ambientTemperature, batteryCurrent;
	int offset;
	float currentOut, batteryVoltage, frequency;
	
	strcpy(status, "");

	clearUpsBuffer();

	/*upssendchar('');*/
	upssend("f\r");

	upsrecv (temp, sizeof(temp), ENDCHAR, IGNCHARS);

	/* Parse the format line */

	/* Time */
	offset = 0;
	for (i = 0; i < 8; i++) {
		if (i == 2 || i == 5) {
			time[i] = ':';
			offset++;
		} else {
			time[i] = temp[4 + i - offset];
		}
	}

	time[8] = '\0';
	setinfo(INFO_TIME, "%s", time);
	
	/* Check if ups is running on the battery */
	if (temp[17] == '1')
		strcat(status, "OB ");
	else
		strcat(status, "OL ");

	/* Check alarms (first register) */
	byte = 16 * temp[20] + temp[21];

	if ((byte & 0x80) == 0x80) {
		/* high ambient temperature alarm is on */
		setinfo(INFO_ALRM_TEMP, "%s", ACTIVE_ALARM);
	}
	
	if ((byte & 0x40) == 0x40) {
		/* Output overload alarm is active */
		setinfo(INFO_ALRM_OVERLOAD, "%s", ACTIVE_ALARM);
	}
	
	if ((byte & 0x08) == 0x08) {
		/* Low runtime left alarm */
		strcat(status, "LB ");
	} else if ((byte & 0x02) == 0x02) {
		/* Near Low Battery alarm */
		strcat(status, "LB ");
	} else if ((byte & 0x01) == 0x01) {
		/* Low Battery alarm */
		strcat(status, "LB ");
	}

	byte1 = temp[22];
	byte2 = temp[23];
	byte1 = byte1 - '0';
	if (byte1 > 9) {
		byte1 = byte1 - ('A' - '0');
	}

	byte2 = byte2 - '0';
	if (byte2 > 9) {
		byte2 = byte2 - ('A' - '0');
	}
	
	byte = 16 * byte1 + byte2;
	

	if ((byte & 0x80) == 0x80) {
		/* Emergency Power Off */
		setinfo(INFO_ALRM_UPSSHUT, "%s", ACTIVE_ALARM);
	}

	if ((byte & 0x40) == 0x40) {
		/* Check memory */
		setinfo(INFO_ALRM_GENERAL, "%s", ACTIVE_ALARM);
	}

	if ((byte & 0x20) == 0x20) {
		/* Check inverter */
		setinfo(INFO_ALRM_GENERAL, "%s", ACTIVE_ALARM);
	}

	if ((byte & 0x10) == 0x10) {
		/* Check battery */
		setinfo(INFO_ALRM_GENERAL, "%s", ACTIVE_ALARM);
	}

	if ((byte & 0x08) == 0x08) {
		/* Check charger */
		setinfo(INFO_ALRM_GENERAL, "%s", ACTIVE_ALARM);
	}

	if ((byte * 0x04) == 0x04) {
		/* High transformer temperature */
		setinfo(INFO_ALRM_TEMP, "%s", ACTIVE_ALARM);
	}

	if ((byte * 0x02) == 0x02) {
		/* User Test Alarm */
		setinfo(INFO_ALRM_GENERAL, "%s", ACTIVE_ALARM);
	}

	if ((byte * 0x01) == 0x01) {
		/* High heatsink temperature */
		setinfo(INFO_ALRM_TEMP, "%s", ACTIVE_ALARM);
	}

	/* Third alarm register */
	byte1 = temp[66];
	byte2 = temp[67];

	byte1 -= '0';
	if (byte1 > 9) {
		byte1 -= 'A' - '0';
	}

	byte2 -= '0';
	if (byte2 > 9) {
		byte2 -= 'A' - '0';
	}

	byte = 16 * byte1 + byte2;

	if ((byte & 0x08) == 0x08) {
		/* Call service */
		setinfo(INFO_ALRM_GENERAL, "%s", ACTIVE_ALARM);
	}

	if ((byte & 0x04) == 0x04) {
		/* High AC Input */
		setinfo(INFO_ALRM_BADINPUT, "%s", ACTIVE_ALARM);
	}

	if ((byte & 0x02) == 0x02) {
		/* Probe missing */
		setinfo(INFO_ALRM_GENERAL, "%s", ACTIVE_ALARM);
	}

	if ((byte & 0x01) == 0x01) {
		/* Check PFM temperature */
		setinfo(INFO_ALRM_TEMP, "%s", ACTIVE_ALARM);
	}
	
	setinfo(INFO_STATUS, "%s", status);

	/* Yeah Binary Coded Decimal */
	
	/* Utility Voltage */
	for (i = 0; i < 4; i++) {
		digit[i] = temp[i + 24] - '0';
	}

	voltageIn = digit[0] * 1000 + digit[1] * 100 + digit[2] * 10 + digit[3];
	setinfo(INFO_UTILITY, "%d", voltageIn);

	/* Output Voltage */
	for (i = 0; i < 4; i++) {
		digit[i] = temp[i + 28] - '0';
	}

	voltageOut = digit[0] * 1000 + digit[1] * 100 + digit[2] * 10 + digit[3];
	setinfo(INFO_OUTVOLT, "%d", voltageOut);

	/* Output Current */
	for (i = 0; i < 4; i++) {
		digit[i] = temp[i + 36] - '0';
	}

	currentOut = digit[0] * 100 + digit[1] * 10 + digit[2] + digit[3] / 10.0;
	setinfo(INFO_CURRENT, "%f", currentOut);

	/* Battery Current */
	for (i = 0; i < 4; i++) {
		digit[i] = temp[i + 46] - '0';
	}

	batteryCurrent = digit[0] * 1000 + digit[1] * 100 + digit[2] * 10  + digit[3];
	setinfo(INFO_BATT_CURRENT, "%d", batteryCurrent);

	/* Battery Voltage */
	for (i = 0; i < 4; i++) {
		digit[i] = temp[i + 50] - '0';
	}

	batteryVoltage = digit[0] * 100 + digit[1] * 10 + digit[2] + digit[3] / 10.0;
	setinfo(INFO_BATTVOLT, "%f", batteryVoltage);

	/* Frequency */
	for (i = 0; i < 4; i++) {
		digit[i] = temp[i + 54] - '0';
	}

	frequency = digit[0] * 10 + digit[1] + digit[2] / 10.0 + digit[3] / 100.0;
	setinfo(INFO_ACFREQ, "%f", frequency);

	/* Runtime Minutes Remaining (or estimated runtime on battery) */
	for (i = 0; i < 4; i++) {
		digit[i] = temp[i + 58] -'0';
	}

	runtimeMinutesRemaining = digit[0] * 1000 + digit[1] * 100 + digit[2] * 10 + digit[3];
	setinfo(INFO_RUNTIME, "%d", runtimeMinutesRemaining);

	/* Ambient Temperature */
	for (i = 0; i < 4; i++) {
		digit[i] = temp[i + 62] -'0';
	}

	ambientTemperature = digit[0] * 1000 + digit[1] * 100 + digit[2] * 10 + digit[3];
	setinfo(INFO_UPSTEMP, "%d", ambientTemperature);

	/* Output Volt Amps */
	for (i = 0; i < 6; i++) {
		digit6[i] = temp[i + 40];
	}
	digit6[6] = '\0';

	setinfo(INFO_OUT_VA, "%s", digit6);

	clearUpsBuffer();

	/* Time to get some more happy info from the ups */

	/* such as load percentage */
	setinfo(INFO_LOADPCT, "%s", hw_get(LOAD_PERCENTAGE, data));

	/* and power */
	setinfo(INFO_OUT_POWER, "%s", hw_get(POWER, data));

	/* and the all-important date */
	setinfo(INFO_DATE, "%s", hw_get(DATE, data));

	/* if (ioctl(upsfd, TIOCMGET, &flags)) {
	 *	upslog(LOG_ERR, "TIOCMGET");
	 *	return;
	 * }
	 */

	/* setinfo(INFO_STATUS, "%s%s",
	 *	(util < lownorm) ? "BOOST ", "",
	 *	(util > highnorm) ? "TRIM ", "",
	 *	((flags & TIOCM_CD) == 0) ? "" : "LB ",
	 *	((flags & TIOCM_CTS) == TIOCM_CTS) ? "OB" : "OL");
	 */

	writeinfo();
}

void upsdrv_shutdown(void)
{
	upssend("o 0\r");
}

void upsdrv_help(void)
{
}

/* list flags and values that you want to receive via -x */
void upsdrv_makevartable(void)
{
	/* allow '-x xyzzy' */
	/* addvar(VAR_FLAG, "xyzzy", "Enable xyzzy mode"); */

	/* allow '-x foo=<some value>' */
	/* addvar(VAR_VALUE, "foo", "Override foo setting"); */
}

void upsdrv_banner(void)
{
	printf("Network UPS Tools - Ferrups 8.01-8.07 UPS driver 0.00 (%s)\n\n", UPS_VERSION);
}


static void sync_serial(void) {
	char buffer[10];

	upssend("\r");
	upsrecv(buffer, sizeof(buffer), ENDCHAR, IGNCHARS);

	while (buffer[0] != '=') {
		upssend("\r");
		upsrecv(buffer, sizeof(buffer), ENDCHAR, IGNCHARS);
	}
}

/* Begin code stolen from bestups.c */
static void setup_serial(void)
{  
	struct   termios  tio;
			     
	if (tcgetattr(upsfd, &tio) == -1)
		fatal("tcgetattr");
				     
	tio.c_iflag = IXON | IXOFF;
   tio.c_oflag = 0;
   tio.c_cflag = (CS8 | CREAD | HUPCL | CLOCAL);
	tio.c_lflag = 0;
	tio.c_cc[VMIN] = 1;
	tio.c_cc[VTIME] = 0;

#ifdef HAVE_CFSETISPEED
	cfsetispeed(&tio, B1200); /* baud change here */
	cfsetospeed(&tio, B1200);
#else
#error This system lacks cfsetispeed() and has no other means to set the speed
#endif

	if (tcsetattr(upsfd, TCSANOW, &tio) == -1)
		fatal("tcsetattr");
/* end code stolen from bestups.c */

	sync_serial();
}

void upsdrv_initups(void)
{
	open_serial(device_path, B1200);
	setup_serial();

	/* to get variables and flags from the command line, use this:
	 *
	 * first populate with upsdrv_buildvartable above, then...
	 *
	 *                   set flag foo : /bin/driver -x foo
	 * set variable 'cable' to '1234' : /bin/driver -x cable=1234
	 *
	 * to test flag foo in your code:
	 *
	 * 	if (testvar("foo"))
	 * 		do_something();
	 *
	 * to show the value of cable:
	 *
	 *      if ((cable == getval("cable")))
	 *		printf("cable is set to %s\n", getval("cable"));
	 *	else
	 *		printf("cable is not set!\n");
	 *
	 * don't use NULL pointers - test the return result first!
	 */

	/* the upsh handlers can't be done here, as they get initialized
	 * shortly after upsdrv_initups returns to main.
	 */
}

/* tell main how many items you need */
int upsdrv_infomax(void)
{
	/* every addinfo() consumes an entry.  plan ahead, but don't go
	 * overboard, since this is an easy way to waste memory.
	 */

	return 40;
}
