/* upslog - log ups values to a file for later collection and analysis

   Copyright (C) 1998  Russell Kroll <rkroll@exploits.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "common.h"
#include "upsclient.h"

#include "config.h"
#include "timehead.h"
#include "upslog.h"

	static	int	port;
	static	char	*upsname, *hostname;
	static	UPSCONN	ups;

	static	FILE	*logfile;
	static	const	char *logfn, *monhost;
	static	sigset_t	upsl_sigmask;
	static	char	logformat[LARGEBUF], logbuffer[LARGEBUF];

#define DEFAULT_LOGFORMAT "%TIME @Y@m@d @H@M@S% %VAR battpct% " \
		"%VAR utility% %VAR loadpct% [%VAR status%] " \
		"%VAR upstemp% %VAR acfreq%"

void sig_hup(int signal)
{
	if (logfile == stdout) {
		upslogx(LOG_INFO, "SIGHUP: logging to stdout");
		return;
	}

	upslogx(LOG_INFO, "SIGHUP: Reopening log file");

	fclose(logfile);
	logfile = fopen(logfn, "a");
	if (logfile == NULL)
		fatal("could not reopen logfile %s", logfn);

	return;
}

/* install handler for reloading config on SIGHUP */
static void instsighandler(void)
{
	struct	sigaction	sa;

	sigemptyset(&upsl_sigmask);
	sigaddset(&upsl_sigmask, SIGHUP);
	sa.sa_mask = upsl_sigmask;
	sa.sa_handler = sig_hup;
	sa.sa_flags = 0;
	if (sigaction(SIGHUP, &sa, NULL) < 0)
		fatal("Can't install SIGHUP handler");
}

static void help(char *prog)
{
	printf("usage: %s <ups> <logfile> <interval> [format string]\n\n",
		prog);

	printf("UPS status logger.\n");
	printf("\n");

	printf("  <ups>		- UPS to monitor - [<upsname>@]host[:port]\n");
	printf("  <logfile>	- Log filename, or - for stdout\n");
	printf("  <interval>	- Time between updates, in seconds\n");
	printf("\n");
	printf("Some valid format string escapes:\n");
	printf("\t%%%% insert a single %%\n");
	printf("\t%%TIME format%% insert the time with strftime formatting\n");
	printf("\t%%HOST%% insert the local hostname\n");
	printf("\t%%UPSHOST%% insert the host of the ups being monitored\n");
	printf("\t%%PID%% insert the pid of upslog\n");
	printf("\t%%VAR varname%% insert the value of ups variable varname\n\n");
	printf("format string defaults to:\n");
	printf("%s\n", DEFAULT_LOGFORMAT);

	printf("\n");
	printf("See the upslog(8) man page for more information\n");

	exit(0);
}

/* print current host name */
static void do_host(const char *arg)
{
	int	ret;
	char	hn[LARGEBUF];

	ret = gethostname(hn, sizeof(hn));

	if (ret != 0) {
		upslog(LOG_ERR, "gethostname failed");
		return;
	}

	snprintfcat(logbuffer, sizeof(logbuffer), "%s", hn);
}

static void do_upshost(const char *arg)
{
	snprintfcat(logbuffer, sizeof(logbuffer), "%s", monhost);
}

static void do_pid(const char *arg)
{
	snprintfcat(logbuffer, sizeof(logbuffer), "%d", getpid());
}

static void do_time(const char *arg)
{
	int	i;
	char	timebuf[SMALLBUF], *format;
	time_t	tod;

	format = xstrdup(arg);

	/* @s are used on the command line since % is taken */
	for (i = 0; i < strlen(format); i++)
		if (format[i] == '@')
			format[i] = '%';

	time(&tod);
	strftime(timebuf, sizeof(timebuf), format, localtime(&tod));

	snprintfcat(logbuffer, sizeof(logbuffer), "%s", timebuf);

	free(format);
}

static void do_var(const char *arg)
{
	char	varval[16];
	int	res;

	if ((!arg) || (strlen(arg) < 1)) {
		snprintfcat(logbuffer, sizeof(logbuffer), "INVALID");
		return;
	}

	res = upscli_getvar(&ups, upsname, arg, varval, sizeof(varval));
	if (res < 0)
		strcpy(varval, "NA");

	snprintfcat(logbuffer, sizeof(logbuffer), "%s", varval);
}

static void do_etime(const char *arg)
{
	time_t	tod;

	time(&tod);
	snprintfcat(logbuffer, sizeof(logbuffer), "%ld", (unsigned long) tod);
}

/* FUTURE: "compile" these results so it only needs to be parsed once */
static void print_status(void)
{
	int	i, j, found, ofs;
	char	*cmd, *arg, *ptr;

	/* parse the format string */
	memset(logbuffer, 0, sizeof(logbuffer));

	for (i = 0; i < strlen(logformat); i++) {

		/* if not a % sequence, append character and start over */
		if (logformat[i] != '%') {
			snprintfcat(logbuffer, sizeof(logbuffer), "%c", 
				logformat[i]);
			continue;
		}

		/* if a %%, append % and start over */
		if (logformat[i+1] == '%') {
			snprintfcat(logbuffer, sizeof(logbuffer), "%%");
			i++;
			continue;
		}

		/* it must start with a % now - %<cmd>[ <arg>]%*/

		cmd = xstrdup(&logformat[i+1]);
		ptr = strchr(cmd, '%');

		/* no trailing % = broken */
		if (!ptr) {
			snprintfcat(logbuffer, sizeof(logbuffer), "INVALID");
			continue;
		}

		*ptr = '\0';

		/* remember length (plus first %) so we can skip over it */
		ofs = strlen(cmd) + 1;

		/* jump out to argument (if any) */
		arg = strchr(cmd, ' ');
		if (arg)
			*arg++ = '\0';

		found = 0;

		/* see if we know how to handle this command */

		for (j = 0; logcmds[j].name != NULL; j++) {
			if (strncasecmp(cmd, logcmds[j].name, 
				strlen(logcmds[j].name)) == 0) {
				logcmds[j].func(arg);
				found = 1;
				break;
			}
		}

		free(cmd);

		if (!found)
			snprintfcat(logbuffer, sizeof(logbuffer), "INVALID");

		/* now do the skip ahead saved from before */
		i += ofs;

	} /* for (i = 0; i < strlen(logformat); i++) */
		
	fprintf(logfile, "%s\n", logbuffer);
	fflush(logfile);
}

int main(int argc, char **argv)
{
	int	interval, i;

	snprintf(logformat, sizeof(logformat), "%s", DEFAULT_LOGFORMAT);

	printf("Network UPS Tools upslog %s\n", UPS_VERSION);

	if (argc < 4)
		help(argv[0]);

	monhost = argv[1];
	logfn = argv[2];
	interval = atoi(argv[3]);

	if (argc != 4) {
		/* read out the remaining argv entries to the format string */
		logformat[0] = logformat[sizeof(logformat)-1] = 0;
		for (i = 4; i < argc; i++)
			snprintfcat(logformat, sizeof(logformat), "%s ",
				argv[i]);
	}

	printf("logging status of %s to %s (%is intervals)\n", 
		monhost, logfn, interval);

	upscli_splitname(monhost, &upsname, &hostname, &port);

	if (upscli_connect(&ups, hostname, port, UPSCLI_CONN_TRYSSL) < 0)
		fprintf(stderr, "Warning: initial connect failed: %s\n", 
			upscli_strerror(&ups));

	if (strcmp(logfn, "-") == 0)
		logfile = stdout;
	else
		logfile = fopen(logfn, "a");

	if (logfile == NULL)
		fatal("could not open logfile %s", logfn);

	openlog("upslog", LOG_PID, LOG_FACILITY); 

	if (logfile != stdout) {
		background();
		instsighandler();
	}

	writepid("upslog");

	for (;;) {

		/* reconnect if necessary */
		if (ups.fd == -1) {
			upscli_disconnect(&ups);
			upscli_connect(&ups, hostname, port, 0);
		}

		print_status();
		sleep(interval);
	}

	/* notreached */

	exit(1);
}
