/* mustekups.c - model specific routines for Mustek Electronics units

   Copyright (C) 2000  Jeremy Maccelari <visualn@iafrica.com>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              

   This is a direct hack of Russell Kroll's fentonups.c to support
   Mustek UPSes. Credit for the bulk of the work here goes to him -
   I just did the port.

   I have checked it on a 1400 and a 425 with a straght through 9 pin
   RS-232 cable. The -k option also works. I have not checked the instcmd
   stuff.

*/
#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <signal.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <unistd.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/termios.h>

#include "config.h"
#include "proto.h"
#include "shared.h"
#include "version.h"
#include "upscommon.h"
#include "mustekups.h"
#include "common.h"

#define INFOMAX 16
#define ENDCHAR 13   /* replies end with CR */

int        shmok = 1, cap_upstemp = 0;
char       statefn[256];
itype      *info;
extern int flag_timeoutfailure;
float      lowvolt = 0, voltrange;
int        infomax = 16;
int        lownorm, highnorm;

void initinfo (void)
{
   info = create_info(INFOMAX,shmok);

   /* setup variables that should always exist */

   addinfo(INFO_UTILITY,"",0,0);
   addinfo(INFO_BATTPCT,"",0,0);
   addinfo(INFO_STATUS,"",0,0);
   addinfo(INFO_ACFREQ,"",0,0);
   addinfo(INFO_LOADPCT,"",0,0);
}

char *rtrim(char *in)
{
   int  i;
   char tmp[256];

   strncpy(tmp,in,sizeof(tmp)-1);

   for (i = 0; i < strlen(tmp); i++)
      if (tmp [i] == ' ') tmp[i] = 0;

   return(strdup(tmp));
}

void guessmodel(char *raw)
{
   char temp[256], mch, *mstr;
   int  i;

   fprintf(stdout,
      "Guessing at UPS capabilities - please report this ID string:\n");

   mch = raw[17];

   mstr = strdup(&raw[18]);
   mstr[10] = '\0';   /* 10 chars max, per the protocol */

   /* trim whitespace  - we expect "Pro ????   " */
   for (i = 4; i < 10; i++)
   {
      if (mstr[i] == ' ')
      {
         mstr[i] = '\0';
         break;
      }
   }

   /* use Mustek model-chars to attempt UPS detection */
   switch (mch)
   {
      case 'P':
         snprintf(temp,sizeof(temp),"P%s", mstr);
         cap_upstemp = 1;
         break;
      default:
         snprintf(temp,sizeof(temp),"Unknown %s",mstr);
         break;
   }
   addinfo (INFO_MODEL, temp, 0, 0);
}

void getbaseinfo(char *port)
{
   char temp[256], model[32], *raw;
   int  modelnum, i;

   /* dummy read attempt to sync - throw it out */
   upssendchar('I');
   upssendchar(13);
   upsrecv(temp,sizeof(temp),ENDCHAR,"");

   /* now retrieve information and parse */
   upssendchar('I');
   upssendchar(13);
   upsrecv(temp,sizeof(temp),ENDCHAR,"");
   raw = strdup(temp);

   if (temp[0] != '#')
   {
      fprintf(stdout,"Bad UPS info start character\n");
      fprintf(stdout,"Got: [%s]\n",temp);
      exit(1);
   }

   temp[11] = 0;
   temp[27] = 0;

   /* manufacturer */
   // setinfo(INFO_MFR,rtrim(&temp[1]));
   setinfo(INFO_MFR,"Mustek");

   /* grab full model string */
   snprintf(model,sizeof(model),"Pro %s",rtrim(&temp[21]));

   modelnum = -1;

   /* figure out official model name and voltage info from table */
   for (i = 0; modeltab[i].mtext != NULL; i++)
   {
      if (!strcmp(modeltab[i].mtext,model))
      {
         modelnum = i;
         lowvolt = modeltab[i].lowvolt;
         voltrange = modeltab[i].voltrange;
         cap_upstemp = modeltab[i].has_temp;
         break;
      }
   }

   /* table lookup fails -> guess */
   if (modelnum == -1)
      guessmodel(raw);
   else
   {
      addinfo(INFO_MODEL,modeltab[modelnum].desc,0,0);
      snprintf(temp,sizeof(temp),"%i",modeltab[modelnum].lowxfer);
      addinfo(INFO_LOWXFER,temp,0,0);

      snprintf(temp,sizeof(temp),"%i",modeltab[modelnum].highxfer);
      addinfo(INFO_HIGHXFER,temp,0,0);

      lownorm = modeltab[modelnum].lownorm;
      highnorm = modeltab[modelnum].highnorm;
   }

   if (cap_upstemp == 1) addinfo(INFO_UPSTEMP,"",0,0);

   /* now add instant command support info */
   addinfo (INFO_INSTCMD, "", 0, CMD_BTEST0);
   addinfo (INFO_INSTCMD, "", 0, CMD_BTEST1);

   fprintf(stdout,"Detected %s on %s\n",getdata(INFO_MODEL),port);
}

/* normal idle loop - keep up with the current state of the UPS */
void updateinfo(void)
{
   char   temp[256], utility[16], loadpct[16], acfreq[16], battvolt[16],
      upstemp[16], stat[16];
   int    util;
   double bvoltp;

   upssendchar('Q');
   upssendchar('1');
   upssendchar(13);

   upsrecv(temp,sizeof(temp),ENDCHAR,"");

   sscanf(temp,"%*c%s %*s %*s %s %s %s %s %s",utility,loadpct, 
      acfreq,battvolt,upstemp,stat);
//   {
//      FILE *out = fopen("/tmp/dump.dat","a");
//      fprintf(out,"%s\n",temp);
//      fclose(out);
//   }

   setinfo(INFO_UTILITY,utility);

   bvoltp = ((atof(battvolt)-lowvolt)/voltrange)*100.0;

   if (bvoltp > 100.0) bvoltp = 100.0;

   snprintf(temp,sizeof(temp),"%02.1f",bvoltp);
   setinfo(INFO_BATTPCT,temp);

   strcpy(temp,"");

   if (stat[0] == '0')
      strcat(temp,"OL ");      /* on line */
   else
      strcat(temp,"OB ");      /* on battery */

   if (stat[1] == '1')
      strcat(temp,"LB ");      /* low battery */

   util = atoi(utility);

   if (stat[2] == '1')
   {      /* boost or trim in effect */
      if (util < lownorm)
         strcat(temp,"BOOST ");
      if (util > highnorm)
         strcat(temp,"TRIM ");
   }

   /* lose trailing space if present */
   if (temp[strlen(temp)-1] == ' ') temp[strlen(temp)-1] = 0;

   setinfo(INFO_STATUS,temp);

   if (cap_upstemp == 1) setinfo(INFO_UPSTEMP,upstemp);

   setinfo(INFO_ACFREQ,acfreq);
   setinfo(INFO_LOADPCT,loadpct);

   writeinfo(info);
}

/* power down the attached load immediately */
void forceshutdown(char *port)
{
   char temp[256], stat[32];

   syslog(LOG_INFO,"Initiating UPS shutdown\n");
   fprintf(stdout,"Initiating forced UPS shutdown!\n");

   open_serial(port,B2400);

   /* basic idea: find out line status and send appropriate command */
   upssendchar('Q');
   upssendchar('1');
   upssendchar(13);
   upsrecv(temp,sizeof(temp),ENDCHAR,"");
   sscanf(temp,"%*s %*s %*s %*s %*s %*s %*s %s",stat);

   /* on battery: send S01<cr>, ups will return by itself on utility */
   /* on line: send S01R0003<cr>, ups will cycle and return soon */

   upssendchar('S');
   upssendchar('0');
   upssendchar('1');

   if (stat[0] == '0')
   {         /* on line */
      fprintf(stdout,"On line, sending shutdown+return command...\n");
      upssendchar('R');
      upssendchar('0');
      upssendchar('0');
      upssendchar('0');
      upssendchar('3');
   }
   else
      fprintf(stdout,"On battery, sending normal shutdown command...\n");

   upssendchar(13);   /* end sequence */

   fprintf(stdout,"Waiting for poweroff...\n");
   sleep(90);
   fprintf(stdout,"Hmm, did the shutdown fail?  Oh well...\n");
   exit(1);                               
}

void instcmd(int auxcmd, int dlen, char *data)
{
   /* TODO: reply to upsd? */

   switch (auxcmd)
   {
      case CMD_BTEST0:   /* stop battery test */
         upssendchar('C');
         upssendchar('T');
         upssendchar(13);
         break;
      case CMD_BTEST1:   /* start battery test */
         upssendchar('T');
         upssendchar(13);
         break;
      default:
         syslog(LOG_INFO,"instcmd: unknown type 0x%04x\n",
            auxcmd);
   }
}

/* install pointers to functions for msg handlers called from msgparse */
void setuphandlers(void)
{
   upsh.instcmd = instcmd;
   /* TODO: future */
}

void usage(char *prog)
{
   fprintf(stdout,"usage: %s [-h] [-k] <device>\n",prog);
   fprintf(stdout,"Example: %s /dev/ttyS0\n",prog);
   exit(1);
}

void help(char *prog)
{
   fprintf(stdout,"usage: %s [-h] [-k] <device>\n",prog);
   fprintf(stdout,"\n");
   fprintf(stdout,"-h       - display this help\n");
   fprintf(stdout,"-k       - force shutdown\n");
   fprintf(stdout,"<device> - /dev entry corresponding to UPS port\n");
}

int main(int argc, char *argv[])
{
   char *portname, *prog;
   int  i;

   fprintf(stdout,"Network UPS Tools - Mustek UPS driver 0.10 (%s)\n",
      UPS_VERSION);
   openlog("mustekups", LOG_PID, LOG_FACILITY);

   prog = argv[0];

   while ((i = getopt(argc,argv,"+hk:")) != EOF)
   {
      switch (i)
      {
         case 'k':
            forceshutdown(optarg);
            break;
         case 'h':
            help(prog);
            break;
         default:
            usage(prog);
            break;
      }
   }

   argc -= optind;
   argv += optind;

   if (argc != 1)
   {
      help(prog);
      exit(1);
   }

   droproot();

   portname = NULL;
   for (i = strlen(argv[0]); i >= 0; i--)
   {
      if (argv[0][i] == '/')
      {
         portname = &argv[0][i+1];
         break;
      }
   }

   if (portname == NULL)
   {
      fprintf(stdout,"Unable to abbreviate %s\n",argv[0]);
      exit(1);
   }

   snprintf(statefn,sizeof(statefn),"%s/mustekups-%s",STATEPATH,
      portname);

   open_serial(argv[0],B2400);

   initinfo();

   createmsgq();   /* try to create IPC message queue */

   getbaseinfo(argv[0]);

   setuphandlers();

   background();

   for (;;)
   {
      updateinfo();
      /* wait up to 2 seconds for a message from the upsd */
      if (getupsmsg(2))
         syslog (LOG_INFO,"Received a message from upsd\n");
   }
}
