/* upssched.c - upsmon's scheduling helper for offset timers

   Copyright (C) 2000  Russell Kroll <rkroll@exploits.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              
*/

/* design notes for the curious: 
 *
 * 1. we get called with a upsname and notifytype from upsmon
 * 2. the config file is searched for an AT condition that matches
 * 3. the condition on the first matching line is parsed
 * 
 * starting a timer: the timer is added to the daemon's timer queue
 * cancelling a timer: the timer is removed from that queue
 *
 * if the daemon is not already running and is required (to start a timer)
 * it will be started automatically
 *
 * when the time arrives, the command associated with a timer will be
 * executed by the daemon (via the cmdscript)
 *
 * timers can be cancelled at any time before they trigger
 *
 * the daemon will shut down automatically when no more timers are active
 * 
 */

#include <stdio.h>
#include <stdlib.h>
#include <syslog.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/socket.h>

#include "common.h"
#include "upssched.h"

typedef struct {
	char	*name;
	time_t	etime;
	void	*next;
}	ttype;

	ttype	*thead = NULL;
	char	*cmdscript = NULL, *pipefn = NULL;
	int	verbose = 0;		/* use for debugging */

/* --- server functions --- */

void removetimer (ttype *tfind)
{
	ttype	*tmp, *last;

	last = NULL;
	tmp = thead;

	while (tmp) {
		if (tmp == tfind) {	/* found it */
			if (tmp->name)
				free (tmp->name);

			if (last == NULL)	/* deleting first */
				thead = tmp->next;
			else
				last->next = tmp->next;

			free (tmp);
			return;
		}

		last = tmp;
		tmp = tmp->next;
	}

	/* this is not necessarily an error */
	if (verbose)
		syslog (LOG_INFO, "Tried to delete nonexistent timer!\n");
	return;
}

void checktimers (void)
{
	ttype	*tmp, *tmpnext;
	time_t	now;
	char	exec[LARGEBUF];

	/* if the queue is empty then we're done */
	if (!thead) {
		if (verbose) {
			syslog (LOG_INFO, "Timer queue empty, exiting\n");
			unlink (pipefn);
		}

		exit (0);
	}

	/* flip through LL, look for activity */
	tmp = thead;

	time (&now);
	while (tmp) {
		tmpnext = tmp->next;

		if (now >= tmp->etime) {
			if (verbose)
				syslog (LOG_INFO, "Event: %s \n", tmp->name);

			/* build the call to the cmdscript and run it */
			snprintf (exec, sizeof(exec), "%s %s", cmdscript,
			          tmp->name);
			
			system (exec);

			/* delete from queue */
			removetimer (tmp);
		}

		tmp = tmpnext;
	}
}

void start_timer (char *name, char *ofsstr)
{
	time_t	now;
	int	ofs;
	ttype	*tmp, *last;

	/* get the time */
	time (&now);

	/* add an event for <now> + <time> */
	ofs = strtol (ofsstr, (char **) NULL, 10);

	if (ofs < 0) {
		syslog (LOG_INFO, "bogus offset for timer, ignoring\n");
		return;
	}

	if (verbose)
		syslog (LOG_INFO, "New timer: %s (%d seconds)\n", name, ofs);

	/* now add to the queue */
	tmp = last = thead;

	while (tmp) {
		last = tmp;
		tmp = tmp->next;
	}

	tmp = malloc (sizeof(ttype));
	tmp->name = strdup (name);
	tmp->etime = now + ofs;
	tmp->next = NULL;

	if (last)
		last->next = tmp;
	else
		thead = tmp;
}

void cancel_timer (char *name)
{
	ttype	*tmp;

	for (tmp = thead; tmp != NULL; tmp = tmp->next) {
		if (!strcmp(tmp->name, name)) {		/* match */
			if (verbose)
				syslog (LOG_INFO, "Cancelling timer: %s\n", 
				        name);
			removetimer (tmp);
			return;
		}
	}

	/* this is not necessarily an error */
	if (verbose)	
		syslog (LOG_INFO, "Tried to cancel nonexistent timer %s\n", name);
}

/* handle received messages from the unix domain socket */
void parsecmd (cmdtype cbuf)
{
	/* basic sanity check */
	if (cbuf.magic != CMDMAGIC) {
		syslog (LOG_INFO, "Received packet with bad magic\n");
		return;
	}

	switch (cbuf.cmd) {
		case CMD_START_TIMER:
			start_timer (cbuf.data1, cbuf.data2);
			break;

		case CMD_CANCEL_TIMER:
			cancel_timer (cbuf.data1);
			break;

		default:
			syslog (LOG_INFO, "unknown command num %d\n", 
			        cbuf.cmd);
	}
}

int openlistenpipe (void)
{
	int	ss, ret;
	struct sockaddr ssaddr;

	ss = socket (AF_UNIX, SOCK_DGRAM, 0);

	if (!ss) {
		syslog (LOG_ERR, "socket failed: %m\n");
		exit (1);
	}

	ssaddr.sa_family = AF_UNIX;
	strncpy (ssaddr.sa_data, pipefn, sizeof(ssaddr.sa_data));
	unlink (pipefn);

	ret = bind (ss, (struct sockaddr *) &ssaddr, sizeof(struct sockaddr));

	if (ret < 0) {
		char	err[256];
		snprintf (err, sizeof(err), "bind %s failed: %s\n",
		          pipefn, strerror(errno));
		syslog (LOG_ERR, err);
		exit (1);
	}

	ret = chmod (pipefn, 0600);

	if (ret < 0) {
		syslog (LOG_ERR, "chmod on pipe failed: %m\n");
		exit (1);
	}

	return (ss);
}

void start_daemon (cmdtype inbuf)
{
	int	pid, pipefd, ret;
	struct	timeval	tv;
	fd_set	rfds;
	cmdtype	cbuf;

	if ((pid = fork()) < 0) {
		perror ("Unable to enter background");

		exit (1);
	}

	if (pid != 0)
		return;			/* parent */

	/* child */

	close (0);
	close (1);
	close (2);

	(void) open ("/dev/null", O_RDWR);
	dup (0);
	dup (0);

	openlog ("upssched", LOG_PID, LOG_DAEMON);

	pipefd = openlistenpipe();

	if (verbose)
		syslog (LOG_INFO, "Timer daemon running\n");

	/* handle whatever it was that caused us to start */
	parsecmd (inbuf);

	/* now watch for activity */

	for (;;) {
		/* wait at most 1s so we can check our timers regularly */
		tv.tv_sec = 1;
		tv.tv_usec = 0;

		FD_ZERO (&rfds);
		FD_SET (pipefd, &rfds);

		ret = select (pipefd + 1, (void *) &rfds, (void *) NULL,
		             (void *) NULL, &tv);

		if (ret) {
			memset (&cbuf, '\0', sizeof(cbuf));
			ret = read (pipefd, &cbuf, sizeof(cbuf));

			if (ret < 0)
				syslog (LOG_ERR, "read error: %m\n");
			else
				parsecmd (cbuf);
		}
 
		checktimers();
	}
}	

/* --- 'client' functions --- */

void sendcmd (int type, char *arg1, char *arg2)
{
	int	pipefd, ret;
	cmdtype	cbuf;
	struct sockaddr pipeaddr;

	cbuf.magic = CMDMAGIC;
	cbuf.cmd = type;

	if (arg1)
		strncpy(cbuf.data1, arg1, sizeof(cbuf.data1));
	else
		strncpy(cbuf.data1, "", sizeof(cbuf.data1));

	if (arg2)
		strncpy(cbuf.data2, arg2, sizeof(cbuf.data2));
	else
		strncpy(cbuf.data2, "", sizeof(cbuf.data2));

	pipefd = socket (AF_UNIX, SOCK_DGRAM, 0);

	if (!pipefd) {
		perror ("socket");
		exit (1);
	}

	pipeaddr.sa_family = AF_UNIX;
	strncpy (pipeaddr.sa_data, pipefn, sizeof(pipeaddr.sa_data));

	ret = connect (pipefd, &pipeaddr, sizeof(pipeaddr.sa_family) +
	               strlen(pipeaddr.sa_data));

	/* if the connect fails, start our own daemon with this info */

	if (ret < 0) {
		close (pipefd);

		/* don't start a daemon just to cancel something */
		if (cbuf.cmd == CMD_CANCEL_TIMER) {
			if (verbose)
				printf ("No daemon running, but none needed...\n");
			exit (0);
		}

		if (verbose)
			printf ("Starting background process...\n");

		/* child returns from this call */
		start_daemon(cbuf);

		exit (0);
	}

	ret = write (pipefd, &cbuf, sizeof(cbuf));

	if (ret < 1) {
		perror ("write");
		return;
	}

	close (pipefd);
}

void parse_at (char *ntype, char *upsname, char *ntstr, char *un,
               char *cmd, char *ca1, char *ca2)
{
	int	ct;

	/* do these in the syslog too since we don't normally have a tty */
	if (!cmdscript) {
		printf ("CMDSCRIPT must be set before any ATs in the config file\n");
		syslog (LOG_ERR, "CMDSCRIPT must be set before any ATs in the config file\n");
		exit (1);
	}

	if (!pipefn) {
		printf ("PIPEFN must be set before any ATs in the config file!\n");
		syslog (LOG_ERR, "PIPEFN must be set before any ATs in the config file!\n");
		exit (1);
	}

	/* check upsname: does this apply to us? */
	if (strcmp(upsname, un) != 0)
		if (strcmp(un, "*") != 0)
			return;		/* not for us, and not the wildcard */

	/* see if the current notify type matches the one from the .conf */
	if (strcasecmp(ntstr, ntype) != 0)
		return;

	/* now see if the command is valid */

	ct = 0;

	if (!strcmp(cmd, "START-TIMER"))
		ct = CMD_START_TIMER;

	if (!strcmp(cmd, "CANCEL-TIMER"))
		ct = CMD_CANCEL_TIMER;

	if (ct == 0) {
		printf ("Invalid command: %s\n", cmd);
		return;
	}

	/* send command to the daemon (starting it if necessary) */
	sendcmd (ct, ca1, ca2);
}

void checkconf (char *ntstr, char *upsname)
{
	char	buf[SMALLBUF], *arg[7], cfn[SMALLBUF];
	FILE	*conf;
	int	i, ln;

	snprintf (cfn, sizeof(cfn), "%s/upssched.conf", CONFPATH);

	conf = fopen (cfn, "r");

	if (!conf) {
		printf ("Can't open %s: %s\n", cfn, strerror(errno));
		exit (1);
	}

	ln = 0;
	while (fgets(buf, sizeof(buf), conf)) {
		buf[strlen(buf) - 1] = '\0';

		ln++;
		i = parseconf ("upssched.conf", ln, buf, arg, 7);

		if (i == 0)
			continue;

		/* AT <notifytype> <upsname> <command> <cmdarg1> <cmdarg2> */
		if (!strcmp(arg[0], "AT")) {
			parse_at (ntstr, upsname, arg[1], arg[2], arg[3], 
			          arg[4], arg[5]);
			continue;
		}

		/* CMDSCRIPT <scriptname> */
		if (!strcmp(arg[0], "CMDSCRIPT"))
			cmdscript = strdup (arg[1]);

		/* PIPEFN <pipename> */
		if (!strcmp(arg[0], "PIPEFN"))
			pipefn = strdup (arg[1]);

		/* ... other config directives ... */
	}

	fclose (conf);
}

int main (int argc, char **argv)
{
	char	*upsname, *ntstr;

	verbose = 1;		/* TODO: remove when done testing */

	upsname = ntstr = NULL;

	upsname = getenv("UPSNAME");
	ntstr = getenv("NOTIFYTYPE");

	if ((!upsname) || (!ntstr)) {
		printf ("Error: UPSNAME and NOTIFYTYPE must be set.\n");
		printf ("This program should only be run from upsmon.\n");
		exit (1);
	}

	/* see if this matches anything in the config file */
	checkconf(ntstr, upsname);

	return 0;
}
