/* multimon - CGI program to monitor several UPSes from one page

   Copyright (C) 1998  Russell Kroll <rkroll@exploits.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              
 */

#include <unistd.h>
#include <stdio.h>
#include <syslog.h>
#include <stdlib.h>
#include <string.h>
#include <sys/time.h>

#include "common.h"
#include "multimon.h"
#include "upsfetch.h"
#include "config.h"
#include "version.h"
#include "status.h"
#include "cgilib.h"
#include "proto.h"

typedef struct {
	char	*var;
	char	*name;
	char	*suffix;
	void	*next;
}	ftype;

	ftype	*firstfield = NULL;
	int	numfields = 0, use_celsius = 1, refreshdelay = -1;
	char	*desc;

void report_error(char *str) 
{
    printf("%s</BODY></HTML>\n", str);
    exit(1);
}

void noresp()
{
	printf ("<TD BGCOLOR=\"#FF0000\">Unavailable</TD>\n");
}

/* handler for "STATUS" */
void do_status(char *monhost)
{
	char	status[64], *stat, color[16], stattxt[128], *ptr;
	int	i, severity;

 	if (getupsvar (monhost, "status", status, sizeof(status)) < 0) {
		noresp();
		return;
	}

	strcpy (stattxt, "");
	severity = 0;

	stat = status;

	while (stat) {
		ptr = strchr (stat, ' ');
		if (ptr)
			*ptr++ = '\0';

		for (i = 0; stattab[i].name != NULL; i++) {
			if (!strcmp(stattab[i].name, stat)) {
				snprintf (&stattxt[strlen(stattxt)],
				          sizeof(stattxt)-strlen(stattxt),
				          "%s<BR>", stattab[i].desc);
				if (stattab[i].severity > severity)
					severity = stattab[i].severity;
			}
		}

		stat = ptr;
	}

	/* severity: green/yellow/red for ok, warning, bad */

	/* obviously this is not universal for all cultures! */

	switch (severity) {
		case 0:	strcpy (color, "#00FF00"); break;
		case 1: strcpy (color, "#FFFF00"); break;
		case 2: strcpy (color, "#FF0000"); break;
	}

	printf ("<TD BGCOLOR=\"%s\">%s</TD>\n", color, stattxt);
}

/* handler for "MODEL" */
void do_model (char *monhost)
{
	char	model[SMALLBUF];

	if (getupsvar (monhost, "model", model, sizeof(model)) < 0) {
		noresp();
		return;
	}

	printf ("<TD BGCOLOR=\"#00FFFF\">%s</TD>\n", model);
	return;
}

/* handler for "UPSTEMP" */
void do_upstemp (char *monhost)
{
	char	upstemp[64];
	float	tempf;

	if (getupsvar (monhost, "upstemp", upstemp, sizeof(upstemp)) > 0) {
		if (use_celsius)
			printf ("<TD BGCOLOR=\"#00FF00\">%s C</TD>\n", upstemp); 
		else {
			tempf = (strtod (upstemp, 0) * 1.8) + 32;
			printf ("<TD BGCOLOR=\"#00FF00\">%.1f F</TD>\n", tempf); 
		}
	}
	else
		printf ("<TD></TD>\n");
}

/* handler for "UPSTEMPC" */
void do_upstempc (char *monhost)
{
	char	upstemp[64];

	if (getupsvar (monhost, "upstemp", upstemp, sizeof(upstemp)) > 0)
		printf ("<TD BGCOLOR=\"#00FF00\">%s C</TD>\n", upstemp); 
	else
		printf ("<TD></TD>\n");
}

/* handler for "UPSTEMPF" */
void do_upstempf (char *monhost)
{
	char	upstemp[64];
	float	tempf;

	if (getupsvar (monhost, "upstemp", upstemp, sizeof(upstemp)) > 0) {
		tempf = (strtod (upstemp, 0) * 1.8) + 32;
		printf ("<TD BGCOLOR=\"#00FF00\">%.1f F</TD>\n", tempf); 
	}
	else
		printf ("<TD></TD>\n");
}

/* handler for "AMBTEMP" */
void do_ambtemp (char *monhost)
{
	char	ambtemp[64];
	float	tempf;

	if (getupsvar (monhost, "ambtemp", ambtemp, sizeof(ambtemp)) > 0) {
		if (use_celsius) 
			printf ("<TD BGCOLOR=\"#00FF00\">%s C</TD>\n", ambtemp); 
		else {
			tempf = (strtod (ambtemp, 0) * 1.8) + 32;
			printf ("<TD BGCOLOR=\"#00FF00\">%.1f F</TD>\n", tempf); 
		}
	}
	else
		printf ("<TD></TD>\n");
}

/* handler for "AMBTEMPC" */
void do_ambtempc (char *monhost)
{
	char	ambtemp[64];

	if (getupsvar (monhost, "ambtemp", ambtemp, sizeof(ambtemp)) > 0)
		printf ("<TD BGCOLOR=\"#00FF00\">%s C</TD>\n", ambtemp); 
	else
		printf ("<TD></TD>\n");
}

/* handler for "AMBTEMPF" */
void do_ambtempf (char *monhost)
{
	char	ambtemp[64];
	float	tempf;

	if (getupsvar (monhost, "ambtemp", ambtemp, sizeof(ambtemp)) > 0) {
		tempf = (strtod (ambtemp, 0) * 1.8) + 32;
		printf ("<TD BGCOLOR=\"#00FF00\">%.1f F</TD>\n", tempf); 
	}
	else
		printf ("<TD></TD>\n");
}

/* handler for "UTILITY" */
void do_utility (char *monhost)
{
	char	utility[64], lowxfer[64], highxfer[64];
	int	lowx, highx, util;

	if (getupsvar (monhost, "utility", utility, sizeof(utility)) > 0) {
		/* try to get low and high transfer points for color codes */

		lowx = highx = 0;

		if (getupsvar (monhost, "lowxfer", lowxfer, sizeof(lowxfer)) > 0)
			lowx = atoi(lowxfer);
		if (getupsvar (monhost, "highxfer", highxfer, sizeof(highxfer)) > 0)
			highx = atoi(highxfer);

		printf ("<TD BGCOLOR=\"#");

		/* only do this if we got both values */
		if ((lowx != 0) && (highx != 0)) {
			util = atoi(utility);

			if ((util < lowx) || (util > highx))
				printf ("FF0000");
			else
				printf ("00FF00");
		}
		else
			printf ("00FF00");

		printf ("\">%s VAC</TD>\n", utility);
	}
	else
		printf ("<TD></TD>\n");
}


void getinfo (char *monhost)
{
	ftype	*tmp;
	char	tmpbuf[SMALLBUF];
	int	i, found;

	printf ("<TR ALIGN=CENTER>\n");

	/* provide system name and link to upsstats */
	printf ("<TD BGCOLOR=\"#00FFFF\">");
	printf ("<A HREF=\"upsstats.cgi?host=%s&amp;use_celsius=%i", monhost, 
	        use_celsius);
	if (refreshdelay > 0)
		printf ("&refresh=%d", refreshdelay);
	printf("\">%s</A></TD>\n", desc);

	/* grab a dummy variable to see if the host is up */
	if (getupsvar(monhost, "model", tmpbuf, sizeof(tmpbuf)) < 0) {
		printf ("<TD COLSPAN=%i BGCOLOR=\"#FF0000\">Error: %s</TD></TR>\n",
		        numfields, upsstrerror(upserror));
		return;
	}

	/* process each field one by one */
	for (tmp = firstfield; tmp != NULL; tmp = tmp->next) {
		found = 0;

		/* search for a recognized special field name */
		for (i = 0; fields[i].name != NULL; i++) {
			if (!strcmp(fields[i].name, tmp->var)) {
				fields[i].func(monhost);
				found = 1;
			}
		}

		if (found)
			continue;

		if (getupsvar(monhost, tmp->var, tmpbuf, sizeof(tmpbuf)) > 0) {
			if (tmp->suffix == NULL)
				printf ("<TD BGCOLOR=\"#00FF00\">%s</TD>\n", 
				        tmpbuf);
			else
				printf ("<TD BGCOLOR=\"#00FF00\">%s %s</TD>\n", 
				        tmpbuf, tmp->suffix);
		}
		else
			printf ("<TD></TD>\n");
	}

	printf ("</TR>\n");

	return;
}

/* add a field to the linked list */
void addfield (char *var, char *name, char *suffix)
{
	ftype	*tmp, *last;

	tmp = last = firstfield;

	while (tmp != NULL) {
		last = tmp;
		tmp = tmp->next;
	}

	tmp = malloc (sizeof (ftype));
	tmp->var = var;
	tmp->name = name;
	tmp->suffix = suffix;
	tmp->next = NULL;

	if (last == NULL)
		firstfield = tmp;
	else
		last->next = tmp;

	numfields++;
}

/* parse a FIELD line from the buf and call addfield */
void parsefield (char *buf)
{
	char *ptr, *var, *name = NULL, *suffix = NULL, *tmp;
	int i = 0, in_string = 0;

	tmp = malloc(strlen(buf) + 1);

	/* <variable> "<field name>" "<field suffix>" */

	var = strdup (buf);
	ptr = strchr (var, ' ');
	if (!ptr)
		report_error("multimon.conf: "
		             "No separating space in FIELD line");

	*ptr++ = '\0';
	while (*ptr) {
		if (*ptr == '"') {
			in_string = !in_string;
			if (!in_string) {
				tmp[i] = '\0';
				i = 0;
				if (suffix) {
					sprintf(tmp, "multimon.conf: "
						"More than two strings "
						"in field %s.", var);
					report_error(tmp);
				}
				else if (name) {
					suffix = strdup(tmp);
				}
				else {
					name = strdup(tmp);
				}
			}
		}
		else if (in_string) {
			if (*ptr == '\\') {
				++ptr;
				if (*ptr == '\0') {
					sprintf(tmp, "multimon.conf: "
						"Backslash at end of line "
						"in field %s.", var);
					report_error(tmp);
				}
			}
			tmp[i++] = *ptr;
		}
		++ptr;
	}

	if (in_string) {
		sprintf(tmp, "multimon.conf: "
			"Unbalanced quotes in field %s!", var);
		report_error(tmp);
	}

	free(tmp);
	addfield (var, name, suffix);
}	

void readconf()
{
	FILE	*conf;
	char	buf[LARGEBUF], fn[SMALLBUF];

	snprintf (fn, sizeof(fn), "%s/multimon.conf", CONFPATH);
	conf = fopen (fn, "r");

	/* the config file is not required */
	if (conf == NULL)
		return;

	while (fgets (buf, sizeof(buf), conf)) {
		buf[strlen(buf) - 1] = 0;
		if (!strncmp (buf, "FIELD", 5))
			parsefield (&buf[6]);
		if (!strncmp (buf, "TEMPC", 5))
			use_celsius = 1;
		if (!strncmp (buf, "TEMPF", 5))
			use_celsius = 0;
	}

	fclose (conf);
}	

/* create default field configuration */
void defaultfields()
{
	addfield ("MODEL", "Model", "");
	addfield ("STATUS", "Status", "");
	addfield ("battpct", "Batt Chg", "%");
	addfield ("UTILITY", "Utility", "VAC");
	addfield ("loadpct", "UPS Load", "%");
	addfield ("UPSTEMP", "UPS Temperature", "");
}	

void parsearg(char *var, char *value)
{
	if (!strcmp(var, "refresh"))
		refreshdelay = (int) strtol(value, (char **) NULL, 10);
}

int main() 
{
	FILE	*conf;
	time_t	tod;
	char	buf[SMALLBUF], fn[SMALLBUF], addr[SMALLBUF], timestr[SMALLBUF], *ptr;
	int	restofs;
	ftype	*tmp;

        /* print out header first so we can give error reports */
	printf ("Content-type: text/html\n");
	printf ("\n");

	printf ("<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.0 Transitional//EN\"\n");
	printf ("	\"http://www.w3.org/TR/REC-html40/loose.dtd\">\n");

	extractcgiargs();

	printf ("<HTML>\n");
	printf ("<HEAD>\n");
	if (refreshdelay > 0)
		printf ("<META HTTP-EQUIV=\"Refresh\" CONTENT=\"%d\">\n", refreshdelay);
	
	printf ("<TITLE>Multimon: UPS Status Page</TITLE>\n");
	printf ("</HEAD>\n");
	printf ("<BODY BGCOLOR=\"#FFFFFF\" TEXT=\"#000000\" LINK=\"#0000EE\" VLINK=\"#551A8B\">\n");

	readconf();

	if (firstfield == NULL)		/* nothing from config file? */
		defaultfields();

	printf ("<TABLE BGCOLOR=\"#50A0A0\" ALIGN=CENTER>\n");
	printf ("<TR><TD>\n");
	printf ("<TABLE CELLPADDING=5>\n");
	printf ("<TR>\n");
	
	time (&tod);
	strftime (timestr, 100, "%a %b %d %X %Z %Y", localtime(&tod));

	printf ("<TH COLSPAN=%i BGCOLOR=\"#60B0B0\">\n", numfields + 1);
	printf ("<FONT SIZE=\"+2\">Network UPS Tools multimon %s</FONT>\n", 
	        UPS_VERSION);
	printf ("<BR>%s</TH>\n", timestr);
	printf ("</TR>\n"); 

	printf ("<TR BGCOLOR=\"#60B0B0\">\n");

	/* first column is always 'system' */

	printf ("<TH COLSPAN=1>System</TH>\n");

	/* print column names */
	for (tmp = firstfield; tmp != NULL; tmp = tmp->next)
		printf ("<TH COLSPAN=1>%s</TH>\n", tmp->name);

	printf ("</TR>\n"); 

	/* ups status */

	snprintf (fn, sizeof(fn), "%s/hosts.conf", CONFPATH);
	conf = fopen (fn, "r");

	if (!conf) {
		printf ("</TABLE>\n");
		printf ("</TD></TR>\n");
		printf ("</TABLE>\n");
		printf ("Error: can't open %s/hosts.conf\n", CONFPATH);
		printf ("</BODY></HTML>\n");
		exit (0);
	}

	while (fgets (buf, sizeof(buf), conf)) 
		if (strncmp("MONITOR", buf, 7) == 0) {
			sscanf (buf, "%*s %s %n", addr, &restofs);
			desc = buf + restofs + 1;

			ptr = strchr (desc, '\"');
			if (ptr)
				*ptr++ = '\0';

			getinfo (addr);
		}

	fclose (conf);
	
	printf ("</TABLE>\n"); 
	printf ("</TD></TR>\n"); 
	printf ("</TABLE></BODY></HTML>\n");

	return (0);
}
