/*
 *  Copyright (C) 1998-99 Luca Deri <deri@unipi.it>
 *                        Stefano Suin <stefano@unipi.it>
 *                        (Stefano contributed to version 1.0)
 *                      
 * 			  Centro SERRA, University of Pisa
 * 			  http://www-serra.unipi.it/
 *  					
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 * Copyright (c) 1994, 1996
 *	The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that: (1) source code distributions
 * retain the above copyright notice and this paragraph in its entirety, (2)
 * distributions including binary code include the above copyright notice and
 * this paragraph in its entirety in the documentation or other materials
 * provided with the distribution, and (3) all advertising materials mentioning
 * features or use of this software display the following acknowledgement:
 * ``This product includes software developed by the University of California,
 * Lawrence Berkeley Laboratory and its contributors.'' Neither the name of
 * the University nor the names of its contributors may be used to endorse
 * or promote products derived from this software without specific prior
 * written permission.
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */

#define _UTIL_C_

/* #define DEBUG  */

#include "ntop.h"


#ifdef HAVE_LSOF
ProcessInfo *processes[MAX_NUM_PROCESSES];
u_short numProcesses;
ProcessInfoList *localPorts[TOP_IP_PORT];
#endif



/* Extern */
extern unsigned int localnet, netmask;
extern char* intoa(struct in_addr addr);
extern struct timeval lastPktTime;
extern FILE *logd;
extern int32_t thisZone;
extern FlowFilterList *flowsList;
#ifndef WIN32
extern pcap_t *pcapPtr;
#endif
extern int getAllPortByName(char* portName);
extern time_t actTime;
#ifdef HAVE_LSOF
extern u_short updateLsof;
#ifdef MULTITHREADED
extern pthread_mutex_t lsofMutex;
#endif
#endif

/* Forward */
int32_t gmt2local(time_t t);
unsigned char isPseudoBroadcastAddress(struct in_addr *addr);
unsigned char isPseudoLocalAddress(struct in_addr *addr);

/* Local */
#define MAX_NUM_NETWORKS      32
#define NETWORK                0
#define NETMASK                1
#define BROADCAST              2
#define INVALIDNETMASK        -1

static short     numLocalNets=0;
static u_int32_t networks[MAX_NUM_NETWORKS][3]; /* [0]=network, [1]=mask, [2]=broadcast */


/*
 * Copy arg vector into a new buffer, concatenating arguments with spaces.
 */
char* copy_argv(register char **argv)
{
  register char **p;
  register u_int len = 0;
  char *buf;
  char *src, *dst;

  p = argv;
  if (*p == 0)
    return 0;

  while (*p)
    len += strlen(*p++) + 1;

  buf = (char *)malloc(len);
  if (buf == NULL) {
    printf("copy_argv: malloc");
	exit(-1);
  }

  p = argv;
  dst = buf;
  while ((src = *p++) != NULL) {
    while ((*dst++ = *src++) != '\0')
      ;
    dst[-1] = ' ';
  }
  dst[-1] = '\0';

  return buf;
}

/* ********************************* */

unsigned char isBroadcastAddress(struct in_addr *addr) {
  if((addr == NULL) 
     || (addr->s_addr == 0x0)
     || ((addr->s_addr | netmask) == 0xffffffff)
     || ((addr->s_addr & 0x000000ff) == 0x000000ff)
     || ((addr->s_addr & 0x000000ff) == 0x00000000) /* Network address */
     ) {   
    return 1;
  } else
    return(isPseudoBroadcastAddress(addr));
}

/* ********************************* */

short isMulticastAddress(struct in_addr *addr) {
  if((addr->s_addr & MULTICAST_MASK) == MULTICAST_MASK) {
#ifdef DEBUG
    printf("%s is multicast [%X/%X]\n", 
	   intoa(*addr),
	   ((unsigned long)(addr->s_addr) & MULTICAST_MASK),
	   MULTICAST_MASK	   
	   );
#endif
    return 1;
  } else
    return 0;
}

/* ********************************* */

short isLocalAddress(struct in_addr *addr) {
  if((addr->s_addr & netmask) == localnet) {
#ifdef DEBUG
    printf("%s is local\n", intoa(*addr));
#endif
    return 1;
  } else {
#ifdef DEBUG
    printf("%s is %s\n", intoa(*addr),
	   isPseudoLocalAddress (addr) ? "pseudolocal" : "remote");
#endif
    /* Broadcast is considered a local address */
    return (isBroadcastAddress(addr));
  }
}

/* **********************************************
 * Description:
 *  Converts an integer in the range
 *  from  0 to 255 in number of bits
 *  useful for netmask  calculation.
 *  The conversion is  valid if there
 *  is an uninterrupted sequence of
 *  bits set to 1 at the most signi-
 *  ficant positions. Example:
 *
 *     1111 1000 -> valid
 *     1110 1000 -> invalid
 *
 * Return values:
 *     0 - 8 (number of subsequent
 *            bits set to 1)
 *    -1     (INVALIDNETMASK)
 *
 *
 * Courtesy of Antonello Maiorca <marty@tai.it>
 *
 *********************************************** */

int int2bits (int number)
{
  int bits = 8;
  int test;

  if ((number > 255) || (number < 0))
    {
#ifdef DEBUG
      printf ("int2bits (%3d) = %d\n", number, INVALIDNETMASK);
#endif
      return (INVALIDNETMASK);
    }
  else
    {
      test = ~number & 0xff;
      while (test & 0x1)
	{
	  bits --;
	  test = test >> 1;
	}
      if (number != ((~(0xff >> bits)) & 0xff))
	{
#ifdef DEBUG
	  printf ("int2bits (%3d) = %d\n", number, INVALIDNETMASK);
#endif
	  return (INVALIDNETMASK);
	}
      else
	{
#ifdef DEBUG
	  printf ("int2bits (%3d) = %d\n", number, bits);
#endif
	  return (bits);
	}
    }
}

/* ***********************************************
 * Description:
 *  Converts a dotted quad notation
 *  netmask  specification  to  the 
 *  equivalent number of bits.
 *  from  0 to 255 in number of bits
 *  useful for netmask  calculation.
 *  The converion is  valid if there
 *  is an  uninterrupted sequence of
 *  bits set to 1 at the most signi-
 *  ficant positions. Example:
 *
 *     1111 1000 -> valid
 *     1110 1000 -> invalid
 *
 * Return values:
 *     0 - 32 (number of subsequent
 *             bits set to 1)
 *    -1      (INVALIDNETMASK)
 *
 *
 * Courtesy of Antonello Maiorca <marty@tai.it>
 *
 *********************************************** */
int dotted2bits (char * mask)
{
  int		fields[4];
  int		fields_num, field_bits;
  int		bits = 0;
  int		i;

  fields_num = sscanf(mask, "%d.%d.%d.%d",
		      &fields[0], &fields[1], &fields[2], &fields[3]);
  if ((fields_num == 1) && (fields[0] <= 32) && (fields[0] >= 0)) 
    {
#ifdef DEBUG
      printf ("dotted2bits (%s) = %d\n", mask, fields[0]);
#endif
      return (fields[0]);
    }
  for (i=0; i < fields_num; i++)
    {
      /* We are in a dotted quad notation. */
      field_bits = int2bits (fields[i]);
      switch (field_bits)
	{
	case INVALIDNETMASK:
	  return (INVALIDNETMASK);

	case 0:
	  /* whenever a 0 bits field is reached there are no more */
	  /* fields to scan                                       */        
#ifdef DEBUG
	  printf ("dotted2bits (%15s) = %d\n", mask, bits);
#endif
	  /* In this case we are in a bits (not dotted quad) notation */
	  return (fields[0]);

	default:
	  bits += field_bits;
	}
    }
#ifdef DEBUG
  printf ("dotted2bits (%15s) = %d\n", mask, bits);
#endif
  return (bits);
}

/* ********************************* */
/* Example: "131.114.0.0/16,193.43.104.0/255.255.255.0" */
void handleLocalAddresses(char* addresses) {
  char *address = strtok(addresses, ",");

  while(address != NULL) {
    char *mask = strchr(address, '/');
    
    if(mask == NULL)
      printf("Unknown network '%s' (empty mask!). It has been ignored.\n", address);
    else {
      u_int32_t network, networkMask, broadcast;
      int bits, a, b, c, d;

      mask[0] = '\0';
      mask++;
      bits = dotted2bits (mask);

      if(sscanf(address, "%d.%d.%d.%d", &a, &b, &c, &d) != 4) {
	  printf ("Unknown network '%s' .. skipping. Check network numbers.\n",
		  address);
	  address = strtok(NULL, ",");
	  continue;
      }
      
      if(bits == INVALIDNETMASK)
	{
	  /* malformed netmask specification */
          printf ("The specified netmask %s is not valid. Skipping it..\n",
		  mask);
	  address = strtok(NULL, ",");
	  continue;
	}

      network = ((a & 0xff) << 24) + ((b & 0xff) << 16) 
	+ ((c & 0xff) << 8) + (d & 0xff);
	
      networkMask = 0xffffffff >> bits;
      networkMask = ~networkMask;

#ifdef DEBUG
      printf ("Nw=%08X - Mask: %08X [%08X]\n", network, networkMask, (network & networkMask));
#endif
      if ((network & networkMask) != network)  {
          /* malformed network specification */
          printf ("WARNING: %d.%d.%d.%d/%d is not a valid network number\n",
		  a, b, c, d, bits);

	  /* correcting network numbers as specified in the netmask */
          network &= networkMask;

	  a = (int) ((network >> 24) & 0xff);
	  b = (int) ((network >> 16) & 0xff);
	  c = (int) ((network >>  8) & 0xff);
	  d = (int) ((network >>  0) & 0xff);

	  printf ("Assuming %d.%d.%d.%d/%d [0x%08x/0x%08x]\n\n",
		  a, b, c, d, bits, network, networkMask);
      }
#ifdef DEBUG
      printf("%d.%d.%d.%d/%d [0x%08x/0x%08x]\n", a, b, c, d, bits, network, networkMask);
#endif

      broadcast = network | (~networkMask);

#ifdef DEBUG
      a = (int) ((broadcast >> 24) & 0xff);
      b = (int) ((broadcast >> 16) & 0xff);
      c = (int) ((broadcast >>  8) & 0xff);
      d = (int) ((broadcast >>  0) & 0xff);

      printf("Broadcast: [net=0x%08x] [broadcast=%d.%d.%d.%d]\n", 
	     network, a, b, c, d);
#endif

      if(numLocalNets < MAX_NUM_NETWORKS) {

	if((network == localnet) && (netmask == networkMask)) {
	  a = (int) ((network >> 24) & 0xff);
	  b = (int) ((network >> 16) & 0xff);
	  c = (int) ((network >>  8) & 0xff);
	  d = (int) ((network >>  0) & 0xff);
	  
	  printf("WARNING: Discarded network %d.%d.%d.%d/%d: this is the local network.\n",
		 a, b, c, d, bits);	  
	} else {
	  networks[numLocalNets][NETWORK]   = network;
	  networks[numLocalNets][NETMASK]   = networkMask;
	  networks[numLocalNets][BROADCAST] = broadcast;
	  numLocalNets++;
	}
      } else
	printf("Unable to handle network (too many entries!).\n");
    }

    address = strtok(NULL, ",");
  }
}

/* ********************************* */

/* This function returns true when a host is considered local
   as specified using the 'm' flag */
unsigned char isPseudoLocalAddress(struct in_addr *addr) {
  if((addr->s_addr & netmask) == localnet) {
    return 1;
  } else {
    int i;

    for(i=0; i<numLocalNets; i++) {
#ifdef DEBUG
      printf("%s comparing [0x%08x/0x%08x]\n", 
	     intoa(*addr),
	     (addr->s_addr & networks[i][NETMASK]),
	     networks[i][NETWORK]);
#endif
      if((addr->s_addr & networks[i][NETMASK]) == networks[i][NETWORK]) {
#ifdef DEBUG
	printf ("%s is pseudolocal\n", intoa(*addr));
#endif
	return 1;
      }
    }

    /* Broadcast is considered a local address */
    return(isBroadcastAddress(addr));
  }
}

/* ********************************* */

/* This function returns true when an address is the broadcast
   for the specified (-m flag subnets */

unsigned char isPseudoBroadcastAddress(struct in_addr *addr) {
  int i;
  
#ifdef DEBUG
  printf ("Checking %8X (pseudo broadcast)\n", addr->s_addr);
#endif

  for(i=0; i<numLocalNets; i++) {
    if(addr->s_addr == networks[i][BROADCAST]) {
#ifdef DEBUG
      printf ("--> %8X is pseudo broadcast\n", addr->s_addr);
#endif
      return 1;
    } else
#ifdef DEBUG
      printf ("%8X/%8X is NOT pseudo broadcast\n", addr->s_addr, networks[i][BROADCAST]);
#else
    ;
#endif    
  }
  
  return(0);
}

/* ********************************* */

/*
 * Print the log timestamp
 */
void printLogTime()
{
  /* Unix timeval style */
  register int s;
  
  s = (lastPktTime.tv_sec + thisZone) % 86400;
  fprintf(logd, "%02d:%02d:%02d.%06u ",
	  s/3600, (s%3600)/60, s%60,
	  (u_int32_t)lastPktTime.tv_usec);
}

/* ********************************* */

/*
 * Returns the difference between gmt and local time in seconds.
 * Use gmtime() and localtime() to keep things simple.
 * [Borrowed from tcpdump]
 */
int32_t gmt2local(time_t t)
{
  register int dt, dir;
  register struct tm *gmt, *loc;

  if (t == 0)
    t = time(NULL);
  /*
    gmt = &sgmt;
    *gmt = *gmtime(&t);
    */
  gmt = gmtime(&t);
  loc = localtime(&t);
  dt = (loc->tm_hour - gmt->tm_hour) * 60 * 60 +
    (loc->tm_min - gmt->tm_min) * 60;

  /*
   * If the year or julian day is different, we span 00:00 GMT
   * and must add or subtract a day. Check the year first to
   * avoid problems when the julian day wraps.
   */
  dir = loc->tm_year - gmt->tm_year;
  if (dir == 0)
    dir = loc->tm_yday - gmt->tm_yday;
  dt += dir * 24 * 60 * 60;

  return (dt);
}

/* ********************************* */

#ifndef WIN32
/* Example: "flow1='host jake',flow2='dst host born2run'" */
void handleFlowsSpecs(char* flows) {
  char *flow = strtok(flows, ",");

  while(flow != NULL) {
    char *flowSpec = strchr(flow, '=');
    
    if(flowSpec == NULL)
      printf("Missing flow spec '%s'. It has been ignored.\n", flow);
    else {
      struct bpf_program fcode;
      int rc, len;
      char *flowName = flow;

      flowSpec[0] = '\0';
      flowSpec++;
      /* flowSpec should now point to 'host jake' */
      len = strlen(flowSpec);
      
      if((len <= 2)
	 || (flowSpec[0] != '\'')
	 || (flowSpec[len-1] != '\''))
	printf("Wrong flow specification \"%s\" (missing \'). "
	       "It has been ignored.\n", flowSpec);
      else {	
	flowSpec[len-1] = '\0';
	flowSpec++;
	rc = pcap_compile(pcapPtr, &fcode, flowSpec, 1, netmask);

	if(rc < 0)
	  printf("Wrong flow specification \"%s\" (syntax error). "
		 "It has been ignored.\n", flowSpec);  
	else {
	  FlowFilterList *newFlow;

	  newFlow = (FlowFilterList*)calloc(1, sizeof(FlowFilterList));

	  if(newFlow == NULL) {
	    printf("Fatal error: not enough memory. Bye!\n");
	    exit(-1);
	  } else {
	    newFlow->flowName = strdup(flowName);
	    newFlow->filterCode = fcode.bf_insns;
	    newFlow->next = flowsList;
	    flowsList = newFlow;
	  }
	}
      }
    }

    flow = strtok(NULL, ",");
  }
}
#endif /* WIN32 */

/* ********************************* */

void getLocalHostAddress(struct in_addr *hostAddress, char* device)
{
#ifdef WIN32
	printf("MISSING getLocalHostAddress:util.c\n");
#else
	register int fd;
  register struct sockaddr_in *sin;
  struct ifreq ifr;
#ifdef DEBUG
  int a, b, c, d;
#endif

  fd = socket(AF_INET, SOCK_DGRAM, 0);
  if (fd < 0) {
    printf("socket error: %d", errno);
    return;
  }
  memset(&ifr, 0, sizeof(ifr));

#ifdef linux
  /* XXX Work around Linux kernel bug */
  ifr.ifr_addr.sa_family = AF_INET;
#endif
  strncpy(ifr.ifr_name, device, sizeof(ifr.ifr_name));
  if (ioctl(fd, SIOCGIFADDR, (char *)&ifr) < 0) {
    printf("SIOCGIFADDR error: %s/errno=%d",
		 device, errno);
  } else {
    sin = (struct sockaddr_in *)&ifr.ifr_addr;   
    hostAddress->s_addr = ntohl(sin->sin_addr.s_addr);
  }

#ifdef DEBUG
  printf("Local address is: %s\n", intoa(*hostAddress));
#endif

  close(fd);
#endif
}

/* ********************************* */

#ifndef WIN32
#ifdef MULTITHREADED

/* *********** MULTITHREAD STUFF *********** */

int createThread(pthread_t *threadId,
		 void *(*__start_routine) (void *)) {
  int rc;

  rc = pthread_create(threadId, NULL, __start_routine, NULL);

  return(rc);
}

/* ************************************ */

void killThread(pthread_t *threadId) {
  pthread_detach(*threadId);
}

/* ************************************ */

int createMutex(pthread_mutex_t *mutexId) {
  int rc = pthread_mutex_init(mutexId, NULL);
  return(rc);
}

/* ************************************ */

void deleteMutex(pthread_mutex_t *mutexId) {
  pthread_mutex_unlock(mutexId);
  pthread_mutex_destroy(mutexId);
}

/* ************************************ */

int accessMutex(pthread_mutex_t *mutexId) {
  return(pthread_mutex_lock(mutexId));
}

/* ************************************ */

int releaseMutex(pthread_mutex_t *mutexId) {
  return(pthread_mutex_unlock(mutexId));
}

/* ************************************ */

int createCondvar(ConditionalVariable *condvarId) {
  int rc;

  rc = pthread_mutex_init(&condvarId->mutex, NULL);
  rc = pthread_cond_init(&condvarId->condvar, NULL);
  condvarId->predicate = 0;

  return(rc);
}

/* ************************************ */

void deleteCondvar(ConditionalVariable *condvarId) {
  pthread_mutex_destroy(&condvarId->mutex);
  pthread_cond_destroy(&condvarId->condvar);
}

/* ************************************ */

int waitCondvar(ConditionalVariable *condvarId) {
  int rc;

  if((rc = pthread_mutex_lock(&condvarId->mutex)) != 0)
    return rc; 

  while(condvarId->predicate <= 0) {
    rc = pthread_cond_wait(&condvarId->condvar, &condvarId->mutex);    
  }

  condvarId->predicate--;

  rc = pthread_mutex_unlock(&condvarId->mutex);
  
  return rc;
}

/* ************************************ */

int signalCondvar(ConditionalVariable *condvarId) {
  int rc;

  rc = pthread_mutex_lock(&condvarId->mutex);

  condvarId->predicate++;

  rc = pthread_mutex_unlock(&condvarId->mutex);
  rc = pthread_cond_signal(&condvarId->condvar); 

  return rc;
}

/* ************************************ */

#ifdef HAVE_SEMAPHORE_H

int createSem(sem_t *semId, int initialValue) {
  int rc;

  rc = sem_init(semId, 0, initialValue);
  return(rc);
}

/* ************************************ */

void waitSem(sem_t *semId) {
  sem_wait(semId);
}

/* ************************************ */

int incrementSem(sem_t *semId) {
  return(sem_post(semId));
}

/* ************************************ */

int decrementSem(sem_t *semId) {
  return(sem_trywait(semId));
}

/* ************************************ */

int deleteSem(sem_t *semId) {
  return(sem_destroy(semId));
}
#endif

#endif /* MULTITHREADED */
#endif /* WIN32 */

/* ************************************ */

#ifdef HAVE_LSOF 
/*
  #define DEBUG 
  #define USE_LSOF_DUMP
  */

void readLsofInfo() {
  char line[384];
  FILE *fd;
  int i, j, found, portNumber, idx, processesIdx;
  ProcessInfoList *listElement;
  ProcessInfo *tmpProcesses[MAX_NUM_PROCESSES];
  
#ifdef MULTITHREADED
  accessMutex(&lsofMutex);
#endif

  for(i=0; i<numProcesses; i++) 
    processes[i]->marker = 0;

  for(idx=0; idx<TOP_IP_PORT; idx++) {
    while(localPorts[idx] != NULL) {
      listElement = localPorts[idx]->next;
      free(localPorts[idx]);
      localPorts[idx] = listElement;
    }
  }

  memset(localPorts, 0, sizeof(localPorts)); /* Just to be sure... */

#ifdef USE_LSOF_DUMP
  fd = fopen("lsof.dump", "r");
#else
  fd = popen(HAVE_LSOF" -i -n -w", "r");
#endif

  if(fd == NULL) {
    printf("FATAL ERROR: Unable to run lsof [%s].\n", HAVE_LSOF);
    exit(-1);
  }

  fgets(line, 383, fd); /* Ignore 1st line */

  while(fgets(line, 383, fd) != NULL) {
    int pid, i, len;
    char command[32], user[32], *portNr;
    char *trailer, *thePort;

    /* printf("%s\n", line); */

    sscanf(line, "%s %d %s", command, &pid, user);

    if(strcmp(command, "lsof") == 0)
      continue;

    /* Either UDP or TCP */
    for(i=10; (line[i] != '\0'); i++)
      if((line[i] == 'P') && (line[i+1] == ' '))
	break;

    if(line[i] == '\0')
      continue;
    else
      trailer = &line[i+2];

    portNr = (char*)strtok(trailer, ":");

    if(portNr[0] == '*')
      portNr = &portNr[2];
    else
      portNr = (char*)strtok(NULL, "-");

    if((portNr == NULL) || (portNr[0] == '*'))
      continue;

    for(i=0, found = 0; i<numProcesses; i++) {
      if(processes[i]->pid == pid) {
	found = 1;
	processes[i]->marker = 1;
	break;
      }
    }
    
    thePort = strtok(portNr, " ");    

    for(j=0; portNr[j] != '\0'; j++)
      if(!isalnum(portNr[j])) {
	portNr[j] = '\0';
	break;
      }

    if(isdigit(portNr[0])) {
      portNumber = atoi(thePort);
    } else {
      portNumber = getAllPortByName(thePort);
    }

#ifdef DEBUG
    printf("%s - %s - %s (%s/%d)\n", command, user, thePort, portNr, portNumber);
#endif

    if(portNumber == -1)
      continue;

    if(!found) {
#ifdef DEBUG      
      printf("%3d) %s %s %s/%d\n", numProcesses, command, user, portNr, portNumber);
#endif
      processes[numProcesses] = (ProcessInfo*)malloc(sizeof(ProcessInfo));
      processes[numProcesses]->command             = strdup(command);
      processes[numProcesses]->user                = strdup(user);
      processes[numProcesses]->pid                 = pid;
      processes[numProcesses]->firstSeen           = actTime;
      processes[numProcesses]->lastSeen            = actTime;
      processes[numProcesses]->marker              = 1;
      processes[numProcesses]->bytesSent           = 0;
      processes[numProcesses]->bytesReceived       = 0;
      processes[numProcesses]->contactedIpPeersIdx = 0;

      memset(processes[numProcesses]->contactedIpPeersIndexes, NO_PEER, 
	     sizeof(processes[i]->contactedIpPeersIndexes));

      idx = numProcesses;
      numProcesses++;
    } else
      idx = i;

    listElement = (ProcessInfoList*)malloc(sizeof(ProcessInfoList));
    listElement->element = processes[idx];
    listElement->next = localPorts[portNumber];
    localPorts[portNumber] = listElement;
  }

#ifdef USE_LSOF_DUMP
  fclose(fd);
#else
  pclose(fd);
#endif

  memcpy(tmpProcesses, processes, sizeof(processes));
  memset(processes, 0, sizeof(processes));

  for(i=0, processesIdx=0; i<numProcesses; i++) {
    if(tmpProcesses[i]->marker == 0) {
      /* Free the process */
      free(tmpProcesses[i]->command);
      free(tmpProcesses[i]->user);
      free(tmpProcesses[i]);
    } else {
      processes[processesIdx++] = tmpProcesses[i];
    }
  }

  numProcesses = processesIdx;

  updateLsof = 0;

#ifdef MULTITHREADED
  releaseMutex(&lsofMutex);
#endif
}

#endif

#ifndef WIN32
/*
 * An os independent signal() with BSD semantics, e.g. the signal
 * catcher is restored following service of the signal.
 *
 * When sigset() is available, signal() has SYSV semantics and sigset()
 * has BSD semantics and call interface. Unfortunately, Linux does not
 * have sigset() so we use the more complicated sigaction() interface
 * there.
 *
 * Did I mention that signals suck?
 */
RETSIGTYPE (*setsignal (int sig, RETSIGTYPE (*func)(int)))(int)
{
#ifdef HAVE_SIGACTION
        struct sigaction old, new;

        memset(&new, 0, sizeof(new));
        new.sa_handler = func;
#ifdef SA_RESTART
        new.sa_flags |= SA_RESTART;
#endif
        if (sigaction(sig, &new, &old) < 0)
                return (SIG_ERR);
        return (old.sa_handler);

#else
#ifdef HAVE_SIGSET
        return (sigset(sig, func));
#else
        return (signal(sig, func));
#endif
#endif
}
#endif /* WIN32 */
