/*
 * Copyright (c) 1996 University College London
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by the Computer Science
 *      Department at University College London
 * 4. Neither the name of the University nor of the Department may be used
 *    to endorse or promote products derived from this software without
 *    specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE UNIVERSITY OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
#include <string.h>
#include <search.h>
#include <sys/types.h>
#include <stdlib.h>

#include "prototypes.h"
block *parse_file_block(char *line);
line *parse_file_line(char *line, block * cur_block);
extern page *p;
extern int tx_fd;
extern queue sendq;
extern Tcl_Interp *interp;

#define MAXLINELEN 240

int save_struct_file(FILE * file, page * p)
{				/* save file in structured text format */
    block *b;

    fprintf(file, "NT1\n");	/* write header to file */

    if (p == NULL)
	return -1;		/* if page is empty, quit */

    if (p->first_block == NULL)
	return -1;		/* if page has no blocks, quit */

    b = p->first_block;		/* get the first block of the page */

    while (b != NULL) {		/* while we have blocks to deal with */
	line *l;
	if (b->status != DELETED) {	/* if block has not been deleted */

/* block header file structure"B:<id>:<o-username>:<o-hostaddress>:<m-username>:<m-hostaddress>:<xpos>,<ypos>: */
/*                      <colour>,<font>:<lastmodsec>,<lastmodusec>\n" */

	    fprintf(file, "B:%s:", b->blockid);
	    fprintf(file, "%s:%s:", b->originator.username, inet_ntoa(b->originator.hostaddr));
	    fprintf(file, "%s:%s:", b->modifier.username, inet_ntoa(b->modifier.hostaddr));
	    fprintf(file, "%u,%u:%u,%u:", b->xpos, b->ypos, b->face.colindex, b->face.fontindex);
	    fprintf(file, "%lu,%lu\n", (unsigned long) b->last_mod.tv_sec, (unsigned long) b->last_mod.tv_usec);

	    l = b->first_line;	/* point l at the first line of the block */

	    while (l != NULL) {	/* while we still have lines to deal with */
		/*if(l->status!=DELETED) *//* if line hasn't been deleted - this is wrong */
		/* replaced with */
		if (l->linenum != LINE_DELETED) {
		    fprintf(file, "L:%s:", l->lineid);
		    if (l->linenum == LINE_DELETED) {
			debug("saving a deleted line\n");
		    } else {
			debug("saving a normal line\n");
		    }

		    /*hopefully we shouldn't need this anymore, but leave
		       it for backwards compatibility */
		    if (strcmp(l->nextid, "OK") == 0) {
			if (l->next_line != NULL)
			    strcpy(l->nextid, l->next_line->lineid);
		    }
		    if (strcmp(l->previd, "OK") == 0) {
			if (l->prev_line != NULL)
			    strcpy(l->previd, l->prev_line->lineid);
		    }
		    fprintf(file, "%s:%s:", l->previd, l->nextid);
		    fprintf(file, "%s:%s:", l->modifier.username, inet_ntoa(l->modifier.hostaddr));
		    fprintf(file, "%lu,%lu:", (unsigned long) l->last_mod.tv_sec, (unsigned long) l->last_mod.tv_usec);
		    fprintf(file, "%s\n", l->line_data);
		}
		l = l->next_line;
	    }
	}
	b = b->next_block;
    }
    return 0;

}

int load_struct_file(FILE * file)
{
    block *cur_block = NULL;
    line *cur_line;
    char line[MAXLINELEN];
    fgets(line, MAXLINELEN - 1, file);
    if (strncmp(line, "NT1", 3) != 0) {
	printf("Unknown file format: %s\n", line);
	return -1;
    }
    while (!feof(file)) {
	if (fgets(line, MAXLINELEN - 1, file) == NULL)
	    break;

	if (strncmp(line, "B:", 2) == 0) {
	    if (cur_block != NULL)
		cur_block->new_data=1;
		redisplay_block(cur_block);

	    cur_block = parse_file_block(line + 2);
	} else if (strncmp(line, "L:", 2) == 0) {
	    cur_line = parse_file_line(line + 2, cur_block);
	} else {
	    printf("Unknown file entry: %s\n", line);
	}
    }
    if (cur_block != NULL) {
	renumber_block(cur_block);
	cur_block->new_data=1;
	redisplay_block(cur_block);
    }
    return 0;
}

block *parse_file_block(char *line)
{
    char *block_str, *crname, *craddr, *modname, *modaddr, *xpos, *ypos;
    char *col, *font, *sec, *usec;
    char command[80];

    block *cur_block, *prev, *next;
    struct timeval modtime;
    user_data orig;
    typeface tface;

    debug("parsing file block: %s", line);

    block_str = line;		/* set pointers to each section of the string correctly */
    crname = strchr(line, ':');
    *crname++ = '\0';
    craddr = strchr(crname, ':');
    *craddr++ = '\0';
    modname = strchr(craddr, ':');
    *modname++ = '\0';
    modaddr = strchr(modname, ':');
    *modaddr++ = '\0';
    xpos = strchr(modaddr, ':');
    *xpos++ = '\0';
    ypos = strchr(xpos, ',');
    *ypos++ = '\0';
    col = strchr(ypos, ':');
    *col++ = '\0';
    font = strchr(col, ',');
    *font++ = '\0';
    sec = strchr(font, ':');
    *sec++ = '\0';
    usec = strchr(sec, ',');
    *usec++ = '\0';

    modtime.tv_sec = atoi(sec);	/* convert ascii -> integer the mod time parts */
    modtime.tv_usec = atoi(usec);

    strcpy(orig.username, modname);	/* set name and hostaddress of modifier */
    orig.hostaddr.s_addr = htonl(inet_addr(modaddr));

    tface.colindex = atoi(col);	/* set colour and font for block */
    tface.fontindex = atoi(font);

    cur_block = find_block_by_id(p, block_str);		/* try to find this block */

    if (cur_block == NULL) {	/* if we don't have it */
	if (p->first_block == NULL) {	/* if this is the first block */
	    /*this is the first block in the page */
	    prev = NULL;
	    next = NULL;
	} else {
	    /*there's at least one block there */
	    prev = p->last_block;
	    next = NULL;
	}

	cur_block = init_block(block_str, NORMAL, atoi(xpos), atoi(ypos),	/* create block */
			     &modtime, prev, next, &orig, &orig, &tface);

	if (p->first_block == NULL)	/* *** DUPLICATED ABOVE? *** */
	    p->first_block = cur_block;

	p->last_block = cur_block;	/* set page's last block correctly */

	sprintf(command, "set_block_posn %s %d %d",	/* TCL Stuff */
		block_str, atoi(xpos), atoi(ypos));
	Tcl_Eval(interp, command);
	sprintf(command, "set_block_colour %s %d",
		block_str, tface.colindex);
	Tcl_Eval(interp, command);
	sprintf(command, "set_block_font %s %d",
		block_str, tface.fontindex);
	Tcl_Eval(interp, command);

	queue_packet_for_sending(tx_fd, cur_block, cur_block->blockid,	/* send block */
				 &sendq, BLOCKMSG);
	add_to_recent_list(cur_block->blockid, &(cur_block->last_mod), BLOCKMSG);	/* add to recent list */
    } else {
	printf("loading an existing block\n");
    }
    return cur_block;
}

line *parse_file_line(char *lne, block * cur_block)
{
    line *cur_line, *prev, *next;
    char *line_id, *crname, *craddr, *prev_id, *next_id;
    char *sec, *usec, *ldata;
    struct timeval modtime;
    user_data orig;

    line_id = lne;
    prev_id = strchr(lne, ':');
    *prev_id++ = '\0';
    next_id = strchr(prev_id, ':');
    *next_id++ = '\0';
    crname = strchr(next_id, ':');
    *crname++ = '\0';
    craddr = strchr(crname, ':');
    *craddr++ = '\0';
    sec = strchr(craddr, ':');
    *sec++ = '\0';
    usec = strchr(sec, ',');
    *usec++ = '\0';
    ldata = strchr(usec, ':') + 1;
    if (strchr(ldata, '\n') != NULL)
	*strchr(ldata, '\n') = '\0';
    modtime.tv_sec = atoi(sec);
    modtime.tv_usec = atoi(usec);

    strcpy(orig.username, crname);

    orig.hostaddr.s_addr = htonl(inet_addr(craddr));

    cur_line = find_line_by_id(line_id, cur_block);

    if (cur_line == NULL) {	/* if we don't already have this line */
	if (strcmp(prev_id, "OK") == 0) {	/* if the previous id is 'ok' */
	    prev = NULL;	/* leave a null pointer */
	} else {
	    prev = find_line_by_id(prev_id, cur_block);		/* else try to find the previous line */
	}

	next = NULL;		/* set next line as null */

	/* create the current line */
	cur_line = init_line(line_id, 0, &modtime, prev, prev_id, next, next_id);
	cur_line->block = cur_block;
	cur_line->no_of_chars = strlen(ldata);
	cur_line->status = NORMAL;
	cur_line->missing_lines = 0;
	user_copy(&orig, &cur_line->modifier);

	strcpy(cur_line->line_data, ldata);

	/* if the current block has no lines, then set this line as the first line */
	if (cur_block->first_line == NULL)
	    cur_block->first_line = cur_line;

	cur_block->no_of_lines = count_lines(cur_block);	/* set the line count for the block */

	if ((prev != NULL) && (strcmp(prev->lineid, line_id) == 0))	/* if previous line is correct */
	    prev->next_line = cur_line;		/* set it's next line to be this line */

	cur_line->prev_line = prev;	/* set our previous line */

	queue_packet_for_sending(tx_fd, cur_line, cur_line->lineid,	/* send this line and add it to */
				 &sendq, LINEMSG);	/*  the recent list */
	add_to_recent_list(cur_line->lineid, &(cur_line->last_mod), LINEMSG);
    }
    return cur_line;
}
