#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <string.h>
#include <stdio.h>
#include <time.h>
#include <errno.h>
#include <stdlib.h>
#include <ctype.h>
#include <gtk/gtk.h>

#include "indexdb.h"
#include "apassign.h"
#include "nls.h"

enum {
	SEL_TYPE_NONE,
	COMPOUND_TEXT,
	STRING,
	TEXT,
	LAST_SEL_TYPE
};

extern void apassign_run();
extern int apassign_cmp();
extern void load_rc();
extern void save_rc();
extern int cmd_setup(int, char **);

static GtkWidget *tl,*tlist,*status,*entry;
static GtkWidget *iw,*ilabel;
static GtkWidget *smi,*rmi,*err,*omi,*wmi,*umi,*fmi,*cmi;
static GSList *elist = NULL;
static GHashTable *proth;
static GHashTable *servh;
static guint tid = 0, etid = 0;
char *db_name = NULL;
static int recnum;
static int sitenum;
static long totsize;
static char *url_selection;
static int cvt_to_lower;

#ifndef O_BINARY
#define O_BINARY	0
#endif

static char *str_to_lower(str)
char *str;
{
	char *p = str;

	while(*p)
	{
		*p = tolower(*p);
		p++;
	}

	return str;
}

static void ClrMsg(data)
gpointer data;
{
	gtk_label_set_text(GTK_LABEL(data) , "");
	gtk_timeout_remove(tid);
	tid = 0;
}

void gui_msg(str)
char *str;
{
	gtk_label_set_text(GTK_LABEL(status), str);
	if (tid) gtk_timeout_remove(tid);
	tid = gtk_timeout_add(10000 , (GtkFunction) ClrMsg , status);
}

static void ClrErr(data)
gpointer data;
{
	gtk_label_set_text(GTK_LABEL(data) , "");
	gtk_timeout_remove(etid);
	etid = 0;
}

void gui_err(str)
char *str;
{
	gtk_label_set_text(GTK_LABEL(err), str);
	if (etid) gtk_timeout_remove(etid);
	etid = gtk_timeout_add(10000 , (GtkFunction) ClrErr , err);
	gdk_beep();
}

static void Quit()
{
	gtk_exit(0);
}

static void PopdownW(w , data)
GtkWidget *w;
gpointer data;
{
	gtk_widget_hide(GTK_WIDGET(data));
}

static void Help(w , data)
GtkWidget *w;
gpointer *data;
{
	static GtkWidget *sw = NULL;

	if (!sw)
	{
		GtkWidget *box,*label,*button;

		sw = gtk_window_new(GTK_WINDOW_TOPLEVEL);
		gtk_signal_connect(GTK_OBJECT(sw), "destroy",
			GTK_SIGNAL_FUNC(gtk_widget_destroyed), &sw);

		box = gtk_vbox_new(FALSE, 3);
		gtk_container_add(GTK_CONTAINER(sw) , box);
		gtk_widget_show(box);

		label = gtk_label_new(gettext("Cache viewer for Netscape browsers disc cache\n\nAuthor: Stefan Ondrejicka <ondrej@idata.sk>\nURL: http://www.idata.sk/~ondrej/nscache"));
		gtk_label_set_justify(GTK_LABEL(label) , GTK_JUSTIFY_LEFT);
		gtk_misc_set_alignment(GTK_MISC(label) , 0.1 , 0.5);
		gtk_misc_set_padding(GTK_MISC(label) , 15 , 15);
		gtk_box_pack_start(GTK_BOX(box), label , TRUE , FALSE , 10);
		gtk_widget_show(label);

		button = gtk_button_new_with_label(gettext("Close"));
		gtk_box_pack_start(GTK_BOX(box), button , FALSE , FALSE , 1);
		gtk_widget_show(button);

		gtk_signal_connect(GTK_OBJECT(button), "clicked",
			GTK_SIGNAL_FUNC(PopdownW) , sw);
	}

	gtk_widget_show(sw);
	if (GTK_WIDGET_REALIZED(sw))
		gdk_window_raise(sw->window);
}

static void BrowseOK(w , data)
GtkWidget *w;
gpointer *data;
{
	GtkFileSelection *fs;
	char *filename, *raw_filename;
	struct stat buf;
	int err;

	fs = GTK_FILE_SELECTION(data);
	filename = gtk_file_selection_get_filename(fs);
	raw_filename = gtk_entry_get_text(GTK_ENTRY(fs->selection_entry));

	g_assert(filename && raw_filename);

	if (strlen (raw_filename) == 0)
		return;

	err = stat(filename, &buf);

	if (err == 0 && (buf.st_mode & S_IFDIR))
	{
		GString *s = g_string_new (filename);
		if (s->str[s->len - 1] != G_DIR_SEPARATOR)
		{
			g_string_append_c (s, G_DIR_SEPARATOR);
		}
		gtk_file_selection_set_filename(fs, s->str);
		g_string_free(s, TRUE);
		return;
	}

	if (err)
	{
		char pom2[2348];

		sprintf(pom2 , gettext("No such File or Directory - %s") , filename);
		gui_err(pom2);
		return;
	}

	gtk_entry_set_text(GTK_ENTRY(entry) , filename);
	PopdownW(w, data);
}

static void Browse(w , data)
GtkWidget *w;
gpointer *data;
{
	static GtkWidget *fs = NULL;

	if (!fs)
	{
		char *fn;

		fs = gtk_file_selection_new(gettext("Select cache index ..."));

		gtk_signal_connect (GTK_OBJECT(GTK_FILE_SELECTION(fs)->cancel_button),
        	        "clicked", GTK_SIGNAL_FUNC(PopdownW), fs);

		gtk_signal_connect (GTK_OBJECT(GTK_FILE_SELECTION(fs)->ok_button),
        	        "clicked", GTK_SIGNAL_FUNC(BrowseOK), fs);

		/*gtk_signal_connect (GTK_OBJECT(GTK_FILE_SELECTION(fs)->ok_button), */
        	/*        "clicked", GTK_SIGNAL_FUNC(PopdownW), fs);                 */

		fn = gtk_entry_get_text(GTK_ENTRY(entry));

		if (fn && *fn)
			gtk_file_selection_set_filename(GTK_FILE_SELECTION(fs) , fn);
	}
	gtk_widget_show(fs);
}

static int copy_file(sf , df)
char *sf;
char *df;
{
	char pom[8192];
	int len;
	int dfd,sfd;

	if ((sfd = open(sf , O_BINARY | O_RDONLY | O_CREAT , 0644)) < 0)
	{
		gui_err(gettext("Error opening source file"));
		return -1;
	}

	if ((dfd = open(df , O_BINARY | O_WRONLY | O_CREAT , 0644)) < 0)
	{
		gui_err(gettext("Error opening destination file"));
		close(sfd);
		return -1;
        }

	while((len = read(sfd , pom , sizeof(pom))) > 0)
	{
		if (len != write(dfd , pom , len))
		{
			gui_err(gettext("Error writing to destinantion file"));
			return -1;
		}
	}

	close(dfd);
	close(sfd);

	return 0;
}

static void Save(w , data)
GtkWidget *w;
gpointer data;
{
	char *fn = gtk_file_selection_get_filename(GTK_FILE_SELECTION(data));
	char pom[2048];
	char *p;
	ns_cache_record *rec;

	rec = (ns_cache_record *) gtk_ctree_node_get_row_data( GTK_CTREE(tlist) , 
		(GtkCTreeNode *)(GTK_CLIST(tlist)->selection->data));

	strcpy(pom , ns_cache_get_db_name());

	p = strrchr(pom , '/');

	if (p)
	{
		strcpy(p+1 , rec->filename);
		if (cvt_to_lower)
			str_to_lower(p+1);
	}
	else
	{
		strcpy(pom , rec->filename);
		if (cvt_to_lower)
			str_to_lower(pom);
	}

	if (access(pom , R_OK))
	{
		gui_err(gettext("Source file can't be readed"));
	}
	else if (link(pom , fn))
	{
		if (errno == EPERM || errno == EXDEV)
		{
			copy_file(pom , fn);
		}
		else
		{
			gui_err(gettext("Error linking files"));
		}
	}
}

static void SaveAs()
{
	ns_cache_record *rec;
	char *p;

	static GtkWidget *fs = NULL;

	if (!fs)
	{
		fs = gtk_file_selection_new(gettext("Save cache file as ..."));

		gtk_signal_connect (GTK_OBJECT(GTK_FILE_SELECTION(fs)->cancel_button),
        	        "clicked", GTK_SIGNAL_FUNC(PopdownW), fs);

		gtk_signal_connect (GTK_OBJECT(GTK_FILE_SELECTION(fs)->ok_button),
        	        "clicked", GTK_SIGNAL_FUNC(Save), fs);
	}

	rec = (ns_cache_record *)gtk_ctree_node_get_row_data(GTK_CTREE(tlist),
		(GtkCTreeNode *)(GTK_CLIST(tlist)->selection->data));

	p = strrchr(rec->urlstr, '/');
	if (p)
	{
		if (!*(p+1)) p = "index.html";

		gtk_file_selection_set_filename(GTK_FILE_SELECTION(fs) , p);
	}

	gtk_window_set_modal(GTK_WINDOW(fs) , TRUE);
	gtk_widget_show(fs);
}

static void unlink_file(name)
char *name;
{
	if (unlink(name))
	{
		char pom2[2348];

		sprintf(pom2 , gettext("Error removing cache file - %s") , name);
		gui_err(pom2);
	}
}

static void Remove()
{
	char pom[2048];
	char *p;
	ns_cache_record *rec;
	struct stat estat;
	GtkCTreeNode *prn = (GtkCTreeNode *)(GTK_CLIST(tlist)->selection->data);
	GtkCTreeRow *rd = (GtkCTreeRow *) prn->list.data;

	rec = (ns_cache_record *)gtk_ctree_node_get_row_data(GTK_CTREE(tlist), prn);

	if (ns_cache_open_db(db_name))
	{
		sprintf(pom , gettext("Error opening cache index file : %s"), strerror(errno));
		gui_err(pom);
		return;
	}

	if (!rec)
	{
		if (rd->level == 2)
		{
			GtkCTreeNode *rn = rd->children;
			GtkCTreeNode *nrn;
			gboolean allrm = TRUE;

			gtk_clist_freeze(GTK_CLIST(tlist));
			while(rn)
			{
				rec = (ns_cache_record *)gtk_ctree_node_get_row_data(GTK_CTREE(tlist), rn);
				rd = rn->list.data;
				nrn = rd->sibling;

				strcpy(pom , db_name);

				p = strrchr(pom , '/');

				if (!p) p = pom;
				else p++;

				strcpy(p , rec->filename);
				if (cvt_to_lower)
					str_to_lower(p);
				if (!stat(pom , &estat))
				{
					if (estat.st_mode & S_IWUSR)
					{
						if (ns_cache_del_rec(rec->urlstr))
						{
							sprintf(pom, gettext("Unable to remove cache index entry - %s") , rec->urlstr);
							allrm = FALSE;
							gui_err(pom);
						}
						else
						{
							gtk_ctree_remove_node(GTK_CTREE(tlist), rn);
							totsize -= rec->content_length;
							recnum --;
							unlink_file(pom);
						}
					}
					else
					{
						char pom2[2048];

						sprintf(pom2 , gettext("read only file : %s") , pom);
						gui_err(pom2);
					}
				}
				else
				{
					char pom2[2048];

					sprintf(pom2 , "%s : %s" , pom , strerror(errno));
					gui_err(pom2);
				}
				rn = nrn;
			}
			if (allrm)
			{
				sitenum --;
				gtk_ctree_remove_node(GTK_CTREE(tlist), prn);
				gtk_clist_thaw(GTK_CLIST(tlist));
			}
		}
	}
	else
	{
		GtkCTreeNode *parent = rd->parent;

		rec = (ns_cache_record *)gtk_ctree_node_get_row_data(GTK_CTREE(tlist), prn);
		gtk_clist_freeze(GTK_CLIST(tlist));
		strcpy(pom , db_name);

		p = strrchr(pom , '/');

		if (!p) p = pom;
		else p++;

		strcpy(p , rec->filename);
		if (cvt_to_lower)
			str_to_lower(p);
		if (!stat(pom , &estat))
		{
			if (estat.st_mode & S_IWUSR)
			{
				if (ns_cache_del_rec(rec->urlstr))
				{
					sprintf(pom, gettext("Unable to remove cache index entry - %s") , rec->urlstr);
					gui_err(pom);
				}
				else
				{
					gtk_ctree_remove_node(GTK_CTREE(tlist), prn);
					totsize -= rec->content_length;
					recnum --;
					unlink_file(pom);
				}
			}
			else
			{
				char pom2[2048];

				sprintf(pom2 , gettext("read only file : %s") , pom);
				gui_err(pom2);
			}
		}
		else
		{
			char pom2[2048];

			sprintf(pom2 , "%s : %s" , pom , strerror(errno));
			gui_err(pom2);
		}

		if (parent)
		{
			rd = (GtkCTreeRow *)parent->list.data;

			if (!rd->children)
			{
				sitenum --;
				gtk_ctree_remove_node(GTK_CTREE(tlist), parent);
			}
		}
		gtk_clist_thaw(GTK_CLIST(tlist));

	}
	ns_cache_close_db();
	sprintf(pom , gettext("Records: %d , Sites: %d , Size: %ld kB"),
		recnum , sitenum , totsize / 1024);

	gui_msg(pom);
}

static void compute_info(tree , node , data)
GtkCTree *tree;
GtkCTreeNode *node;
gpointer data;
{
	ns_cache_record *rec;
	struct {
		long sz;
		int num;
	} *nfo;

	nfo = data;

	rec = (ns_cache_record *)gtk_ctree_node_get_row_data(GTK_CTREE(tree), node);

	if (rec)
	{
		nfo->num ++;
		nfo->sz += rec->content_length;
	}
}

static void NodeInfo()
{
	char pom[256];
	struct {
		long sz;
		int num;
	} nfo;
	
	nfo.sz = 0;
	nfo.num = 0;

	gtk_ctree_post_recursive(GTK_CTREE(tlist) , 
		(GtkCTreeNode *)(GTK_CLIST(tlist)->selection->data) ,
		GTK_CTREE_FUNC(compute_info) , &nfo);

	sprintf(pom , gettext("Records: %d , Size: %ld kB"),
		nfo.num , nfo.sz / 1024);

	gui_msg(pom);
}

static void chmod_recursive(tree , node , data)
GtkCTree *tree;
GtkCTreeNode *node;
gpointer data;
{
	ns_cache_record *rec;
	struct {
		int set;
		mode_t mask;
	} *nfo;

	nfo = data;

	rec = (ns_cache_record *)gtk_ctree_node_get_row_data(GTK_CTREE(tree), node);

	if (rec)
	{
		char pom[2048];
		char *p;
		struct stat estat;

		strcpy(pom , db_name);
		p = strrchr(pom , '/');

		if (!p) p = pom;
		else p++;

		strcpy(p , rec->filename);
		if (cvt_to_lower)
			str_to_lower(p);

		if (!stat(pom , &estat))
		{
			if (nfo->set)
				estat.st_mode |= nfo->mask;
			else
				estat.st_mode &= ~nfo->mask;

			if (chmod(pom, estat.st_mode))
			{
				gui_err(gettext("Can't change mode for any of cache files"));
			}
		}
		else
		{
			gui_err(gettext("Some of cache file doesn't exist"));
		}
	}
}
	
static void RdOnly()
{
	struct {
		int set;
		mode_t mask;
	} nfo;
	nfo.set = 0;
	nfo.mask = S_IWUSR;

	gtk_ctree_post_recursive(GTK_CTREE(tlist) , 
		(GtkCTreeNode *)(GTK_CLIST(tlist)->selection->data) ,
		GTK_CTREE_FUNC(chmod_recursive) , &nfo);
}

static void Writable()
{
	struct {
		int set;
		mode_t mask;
	} nfo;
	nfo.set = 1;
	nfo.mask = S_IWUSR;

	gtk_ctree_post_recursive(GTK_CTREE(tlist) , 
		(GtkCTreeNode *)(GTK_CLIST(tlist)->selection->data) ,
		GTK_CTREE_FUNC(chmod_recursive) , &nfo);
}

static void ViewURL()
{
	ns_cache_record *rec = (ns_cache_record *)
		gtk_ctree_node_get_row_data(GTK_CTREE(tlist), (GtkCTreeNode *)(GTK_CLIST(tlist)->selection->data));
	apassign *e = NULL;

	if (rec && rec->content_type)
	{
		GSList *ptr = g_slist_find_custom(apassign_data , rec->content_type , apassign_cmp);
		if (ptr) e = (apassign *) ptr->data;
	}

	if (e && e->uapplication)
	{
		char pom[4098];
		char *p,*r;

		for (p = e->uapplication, r = pom; *p ; p++)
		{
			if (*p == '%' && *(p+1) == 'u')
			{
				p++;
				*r = '\"';
				r++;
				strcpy(r , rec->urlstr);
				r += strlen(r);
				*r = '\"';
				r++;
			}
			else
			{
				*r = *p;
				r++;
			}
		}
		*r = '&';
		r++;
		*r = '\0';

		system(pom);
		gui_msg(gettext("URL viewer launched"));
	}
}

static void ViewFile()
{
	ns_cache_record *rec = (ns_cache_record *)
		gtk_ctree_node_get_row_data(GTK_CTREE(tlist), (GtkCTreeNode *)(GTK_CLIST(tlist)->selection->data));
	apassign *e = NULL;

	if (rec && rec->content_type)
	{
		GSList *ptr = g_slist_find_custom(apassign_data , rec->content_type , apassign_cmp);
		if (ptr) e = (apassign *) ptr->data;
	}

	if (e && e->fapplication)
	{
		char pom[4098];
		char *p,*r;

		for (p = e->fapplication, r = pom; *p ; p++)
		{
			if (*p == '%' && *(p+1) == 'f')
			{
				char *pt;

				p++;
				*r = '\"';
				r++;

				pt = strrchr(db_name , '/');

				if (pt)		
				{
					strncpy(r , db_name , pt - db_name + 1);
					r += pt - db_name + 1;
					strcpy(r , rec->filename);
				}
				else strcpy(r , rec->filename);
				if (cvt_to_lower)
					str_to_lower(r);
				r += strlen(r);
				*r = '\"';
				r++;
			}
			else
			{
				*r = *p;
				r++;
			}
		}
		*r = '&';
		r++;
		*r = '\0';

		system(pom);
		gui_msg(gettext("File viewer launched"));
	}
}

static void SetSelectionURL()
{
	ns_cache_record *rec = (ns_cache_record *)
		gtk_ctree_node_get_row_data(GTK_CTREE(tlist), (GtkCTreeNode *)(GTK_CLIST(tlist)->selection->data));

	if (!url_selection)
		gtk_selection_owner_set(tl , GDK_SELECTION_PRIMARY ,GDK_CURRENT_TIME);

	if (rec && rec->urlstr)
	{
		g_free(url_selection);
		url_selection = g_strdup(rec->urlstr);
	}
}

static gint SelectionClear(widget , event)
GtkWidget *widget;
GdkEventSelection *event;
{
	g_free(url_selection);
	url_selection = NULL;
	return TRUE;
}

static void SendSelection(widget , selection_data , info , time , data)
GtkWidget *widget;
GtkSelectionData *selection_data;
guint      info;
guint      time;
gpointer   data;
{
	GdkAtom type = GDK_NONE;

	switch (info)
	{
		case COMPOUND_TEXT:
			type = gdk_atom_intern("COMPOUND_TEXT",FALSE);
			break;
		case TEXT:
			type = gdk_atom_intern("TEXT",FALSE);
			break;
		case STRING:
			type = gdk_atom_intern("STRING",FALSE);
			break;
	}
	gtk_selection_data_set(selection_data , type , 8*sizeof(gchar) ,
		url_selection , strlen(url_selection)); 
}

static void UpdateEInfo()
{
	ns_cache_record *rec = (ns_cache_record *)
		gtk_ctree_node_get_row_data(GTK_CTREE(tlist), (GtkCTreeNode *)(GTK_CLIST(tlist)->selection->data));

	if (rec && iw)
	{
		char pom[100];
		char *p,*p1;

		p = g_strconcat(gettext("Source URL: "), rec->urlstr, "\n",
			gettext("Local cache file: "), rec->filename, "\n", NULL);

		if (rec->content_type)
		{	p1 = p;
			p = g_strconcat(p, gettext("MIME type: "), 
				rec->content_type, "\n", NULL);
			g_free(p1);
		}
			
		sprintf(pom, gettext("%d bytes"), rec->content_length);
		p1 = p;
		p = g_strconcat(p, gettext("Size:"), pom, "\n", NULL);
		g_free(p1);

		if (rec->content_encoding)
		{
			p1 = p;
			p = g_strconcat(p, gettext("Encoding: "), 
				rec->content_encoding, "\n", NULL);
			g_free(p1);
		}

		if (rec->charset)
		{
			p1 = p;
			p = g_strconcat(p, gettext("Charset: "), 
				rec->charset, "\n", NULL);
			g_free(p1);
		}

		if (rec->last_modified)
		{
			struct tm *tm =  localtime(&rec->last_modified);
			strftime(pom , sizeof(pom) , "%H:%M %d.%m.%Y" , tm);
			p1 = p;
			p = g_strconcat(p, gettext("Modification time: "), 
				pom, "\n", NULL);
			g_free(p1);
		}

		if (rec->last_accessed)
		{
			char pom[100];
			struct tm *tm =  localtime(&rec->last_accessed);
			strftime(pom , sizeof(pom) , "%H:%M %d.%m.%Y" , tm);
			p1 = p;
			p = g_strconcat(p, gettext("Access time: "), 
				pom, "\n", NULL);
			g_free(p1);
		}

		if (rec->expires)
		{
			char pom[100];
			struct tm *tm =  localtime(&rec->expires);
			strftime(pom , sizeof(pom) , "%H:%M %d.%m.%Y" , tm);
			p1 = p;
			p = g_strconcat(p, gettext("Expiration time: "), 
				pom, "\n", NULL);
			g_free(p1);
		}

		gtk_label_set_text(GTK_LABEL(ilabel), p);

		g_free(p);
	}
	else if (iw)
	{
		char pom[256];
		struct {
			long sz;
			int num;
		} nfo;
	
		nfo.sz = 0;
		nfo.num = 0;

		gtk_ctree_post_recursive(GTK_CTREE(tlist) , 
			(GtkCTreeNode *)(GTK_CLIST(tlist)->selection->data) ,
			GTK_CTREE_FUNC(compute_info) , &nfo);

		sprintf(pom , gettext("Records: %d\nSize: %ld kB"),
			nfo.num , nfo.sz / 1024);

		gtk_label_set_text(GTK_LABEL(ilabel), pom);
	}
}

static void ViewInfo()
{
	if (!iw)
	{
		GtkWidget *box,*button;

		iw = gtk_window_new(GTK_WINDOW_TOPLEVEL);
		gtk_window_set_title(GTK_WINDOW(iw) , gettext("NScache: entry info window"));
		gtk_signal_connect(GTK_OBJECT(iw), "destroy",
			GTK_SIGNAL_FUNC(gtk_widget_destroyed), &iw);

		box = gtk_vbox_new(FALSE , 5);
		gtk_container_add(GTK_CONTAINER(iw) , box);
		gtk_widget_show(box);

		ilabel = gtk_label_new("");
		gtk_misc_set_alignment(GTK_MISC(ilabel) , 0.0, 0.0);
		gtk_misc_set_padding(GTK_MISC(ilabel), 5, 5);
		gtk_label_set_justify(GTK_LABEL(ilabel), GTK_JUSTIFY_LEFT);
		gtk_widget_set_usize(ilabel, 400, 150);
		gtk_box_pack_start(GTK_BOX(box), ilabel, TRUE, TRUE, 5);
		gtk_widget_show(ilabel);

		button = gtk_button_new_with_label(gettext("Close"));
		gtk_box_pack_start(GTK_BOX(box), button, FALSE, FALSE, 2);
		gtk_widget_show(button);
	
		gtk_signal_connect(GTK_OBJECT(button) , "clicked" ,
			GTK_SIGNAL_FUNC(PopdownW) , iw);

		UpdateEInfo();
	}
	gtk_widget_show(iw);
	if (GTK_WIDGET_REALIZED(iw)) gdk_window_raise(iw->window);
}

GtkCTreeNode *get_parent_node(urlstr , dnamep)
char *urlstr;
char **dnamep;
{
	GtkCTreeNode *pn;
	GtkCTreeNode *sn;
	char *proto;
	char *serv;
	char *p,*p2;
	char *centry[9];

	p = strchr(urlstr , ':');

	if (!p) return NULL;

	proto = g_strndup(urlstr , p - urlstr);

	pn = (GtkCTreeNode *)g_hash_table_lookup(proth , proto);

	if (!pn)
	{
		centry[0] = proto;
		centry[1] = "";
		centry[2] = "";
		centry[3] = "";
		centry[4] = "";
		centry[5] = "";
		centry[6] = "";
		centry[7] = "";
		centry[8] = "";
		pn = gtk_ctree_insert_node(GTK_CTREE(tlist) , NULL , NULL , centry ,
			8 , NULL , NULL , NULL , NULL , FALSE , TRUE);

		g_hash_table_insert(proth , proto , pn);
	}

	p += 3;
	p2 = strchr(p , '/');

	if (!p2) return pn;

	serv = g_strndup(p , p2 - p);

	sn = (GtkCTreeNode *)g_hash_table_lookup(servh , serv);

	if (!sn)
	{
		centry[0] = serv;
		centry[1] = "";
		centry[2] = "";
		centry[3] = "";
		centry[4] = "";
		centry[5] = "";
		centry[6] = "";
		centry[7] = "";
		centry[8] = "";
		sn = gtk_ctree_insert_node(GTK_CTREE(tlist) , pn , NULL , centry ,
			8 , NULL , NULL , NULL , NULL , FALSE , FALSE);

		g_hash_table_insert(servh , serv , sn);

		sitenum++;
	}

	*dnamep = p2;
	return sn;
}

static void Open(w , data)
GtkWidget *w;
gpointer data;
{
	char *fname = gtk_entry_get_text(GTK_ENTRY(data));
	GSList *ptr;
	GtkCTreeNode *parent;
	char pom[2048];

	if (ns_cache_open_db(fname))
	{
		sprintf(pom , gettext("Error opening cache index file : %s"), strerror(errno));
		gui_err(pom);
		return;
	}

	totsize = 0;
	recnum = 0;
	sitenum = 0;

	if (db_name) g_free(db_name);
	db_name = g_strdup(fname);

	gtk_clist_freeze(GTK_CLIST(tlist));
	gtk_clist_clear(GTK_CLIST(tlist));

	while(elist)
	{
		ns_cache_free_record((ns_cache_record *)elist->data);
		elist = g_slist_remove_link(elist , elist);
	}

	if (proth)  g_hash_table_destroy(proth);
	if (servh)  g_hash_table_destroy(servh);

	proth = g_hash_table_new(g_str_hash , g_str_equal);
	servh = g_hash_table_new(g_str_hash , g_str_equal);

	ptr = elist = ns_cache_read_db();

	ns_cache_close_db();
	parent = NULL;
	while(ptr)
	{
		char *centry[10];
		char *dname;
		GtkCTreeNode *n;
		char size[15];
		char mdtm[30];
		char atm[30];
		char exptm[30];
		struct tm *tm;

		recnum ++;
		totsize += ((ns_cache_record *)ptr->data)->content_length;

		parent = get_parent_node(((ns_cache_record *)ptr->data)->urlstr , &dname);

		centry[0] = dname;
		centry[1] = ((ns_cache_record *)ptr->data)->filename;
		centry[2] = ((ns_cache_record *)ptr->data)->content_type;
		sprintf(size , "%d" , ((ns_cache_record *)ptr->data)->content_length);
		centry[3] = size;
		centry[4] = ((ns_cache_record *)ptr->data)->content_encoding;
		centry[5] = ((ns_cache_record *)ptr->data)->charset;
		tm =  localtime(&((ns_cache_record *)ptr->data)->last_modified); 
		strftime(mdtm , sizeof(mdtm) , "%H:%M %d.%m.%Y" , tm);
		centry[6] = mdtm;
		tm = localtime(&((ns_cache_record *)ptr->data)->last_accessed); 
		strftime(atm , sizeof(atm) , "%H:%M %d.%m.%Y" , tm);
		centry[7] = atm;
		if (((ns_cache_record *)ptr->data)->expires)
		{
			tm = localtime(&((ns_cache_record *)ptr->data)->expires); 
			strftime(exptm , sizeof(exptm) , "%H:%M %d.%m.%Y" , tm);
		}
		else
			exptm[0] = '\0';
		centry[8] = exptm;
		centry[9] = "";


		n = gtk_ctree_insert_node(GTK_CTREE(tlist) , parent , NULL , centry ,
			8 , NULL , NULL , NULL , NULL , FALSE , FALSE);
		gtk_ctree_node_set_row_data(GTK_CTREE(tlist) , n , ptr->data);

		ptr = ptr->next;
	}

	gtk_ctree_pre_recursive_to_depth(GTK_CTREE(tlist) , NULL , 1 ,
		GTK_CTREE_FUNC(gtk_ctree_sort_recursive) , NULL);

	gtk_clist_thaw(GTK_CLIST(tlist));

	sprintf(pom , gettext("Records: %d , Sites: %d , Size: %ld kB"),
		recnum , sitenum , totsize / 1024);

	gui_msg(pom);
}

static void ToggleBool(w , fdata)
GtkWidget *w;
gpointer fdata;
{
	int *v = (int *)fdata;

	*v = GTK_CHECK_MENU_ITEM(w)->active;
}

void build_menu(w)
GtkWidget *w;
{
	GtkWidget *mbar,*menu,*mbb,*mi;

	mbar = gtk_menu_bar_new();
	gtk_box_pack_start(GTK_BOX(w) , mbar , FALSE , TRUE , 1);
	gtk_widget_show(mbar);

	menu = gtk_menu_new();
	gtk_widget_realize(menu);

	mi = gtk_menu_item_new_with_label(gettext("Viewers setup ..."));
	gtk_menu_append(GTK_MENU(menu) , mi);
	gtk_widget_show(mi);

	gtk_signal_connect(GTK_OBJECT(mi) , "activate" ,
		GTK_SIGNAL_FUNC(apassign_run) , NULL);

	mi = gtk_menu_item_new_with_label(gettext("Entry informations ..."));
	gtk_menu_append(GTK_MENU(menu) , mi);
	gtk_widget_show(mi);

	gtk_signal_connect(GTK_OBJECT(mi) , "activate" ,
		GTK_SIGNAL_FUNC(ViewInfo) , NULL);

	mi = gtk_menu_item_new_with_label(gettext("Save rc file"));
	gtk_menu_append(GTK_MENU(menu) , mi);
	gtk_widget_show(mi);

	gtk_signal_connect(GTK_OBJECT(mi) , "activate" ,
		GTK_SIGNAL_FUNC(save_rc) , NULL);


	mi = gtk_menu_item_new();
	gtk_menu_append(GTK_MENU(menu) , mi);
	gtk_widget_show(mi);

	mi = gtk_menu_item_new_with_label(gettext("Quit"));
	gtk_menu_append(GTK_MENU(menu) , mi);
	gtk_widget_show(mi);

	gtk_signal_connect(GTK_OBJECT(mi) , "activate" ,
		GTK_SIGNAL_FUNC(Quit) , NULL);

	mbb = gtk_menu_item_new_with_label(gettext("File"));
	gtk_menu_bar_append(GTK_MENU_BAR(mbar) , mbb);
	gtk_menu_item_set_submenu(GTK_MENU_ITEM(mbb) , menu);
	gtk_widget_show(mbb);

	menu = gtk_menu_new();
	gtk_widget_realize(menu);

	mi = gtk_check_menu_item_new_with_label(gettext("Convert filenames to lowercase"));
	gtk_menu_append(GTK_MENU(menu) , mi);
	gtk_widget_show(mi);

	gtk_signal_connect(GTK_OBJECT(mi) , "activate" ,
		GTK_SIGNAL_FUNC(ToggleBool) , &cvt_to_lower);

	mbb = gtk_menu_item_new_with_label(gettext("Config"));
	gtk_menu_item_right_justify(GTK_MENU_ITEM(mbb));
	gtk_menu_bar_append(GTK_MENU_BAR(mbar) , mbb);
	gtk_menu_item_set_submenu(GTK_MENU_ITEM(mbb) , menu);
	gtk_widget_show(mbb);

	menu = gtk_menu_new();
	gtk_widget_realize(menu);

	mi = gtk_menu_item_new_with_label(gettext("About"));
	gtk_menu_append(GTK_MENU(menu) , mi);
	gtk_widget_show(mi);

	gtk_signal_connect(GTK_OBJECT(mi) , "activate" ,
		GTK_SIGNAL_FUNC(Help) , NULL);

	mbb = gtk_menu_item_new_with_label(gettext("Help"));
	gtk_menu_item_right_justify(GTK_MENU_ITEM(mbb));
	gtk_menu_bar_append(GTK_MENU_BAR(mbar) , mbb);
	gtk_menu_item_set_submenu(GTK_MENU_ITEM(mbb) , menu);
	gtk_widget_show(mbb);
}

void build_seldb(w)
GtkWidget *w;
{
	GtkWidget *box,*button,*label;

	box = gtk_hbox_new(FALSE , 5);
	gtk_box_pack_start(GTK_BOX(w) , box , FALSE , TRUE , 1);
	gtk_widget_show(box);

	label = gtk_label_new(gettext("Cache index file: "));
	gtk_box_pack_start(GTK_BOX(box) , label , FALSE , FALSE , 5);
	gtk_widget_show(label);

	entry = gtk_entry_new();
	gtk_box_pack_start(GTK_BOX(box) , entry , TRUE , TRUE , 1);
	gtk_widget_show(entry);

	button = gtk_button_new_with_label(gettext("Browse ..."));
	gtk_box_pack_start(GTK_BOX(box) , button , FALSE , FALSE , 1);
	gtk_widget_show(button);

	gtk_signal_connect(GTK_OBJECT(button), "clicked" ,
		GTK_SIGNAL_FUNC(Browse) , NULL);

	button = gtk_button_new_with_label(gettext("Load"));
	gtk_box_pack_start(GTK_BOX(box) , button , FALSE , FALSE , 1);
	gtk_widget_show(button);

	gtk_signal_connect(GTK_OBJECT(button), "clicked" ,
		GTK_SIGNAL_FUNC(Open) , entry);
}


static gint popup_tmenu(widget , event , data)
GtkWidget *widget;
GdkEvent  *event;
gpointer data;
{
	GdkEventButton *bevent;

	switch (event->type)
	{
	    case GDK_2BUTTON_PRESS:
		bevent = (GdkEventButton *) event;
		if (bevent->button == 1)
		{
			ViewFile();
		}
	    case GDK_BUTTON_PRESS:
		bevent = (GdkEventButton *) event;
		if (bevent->button == 3)
		{
			if (GTK_CLIST(tlist)->selection)
			{
				ns_cache_record *rec = (ns_cache_record *)
					gtk_ctree_node_get_row_data(GTK_CTREE(tlist), (GtkCTreeNode *)(GTK_CLIST(tlist)->selection->data));
				apassign *e = NULL;

				if (rec && rec->content_type)
				{
					GSList *ptr = g_slist_find_custom(apassign_data , rec->content_type , apassign_cmp);

					if (ptr) e = (apassign *) ptr->data;
				}
				gtk_widget_set_sensitive(smi , (int)rec);
				gtk_widget_set_sensitive(cmi , (int)rec);
				gtk_widget_set_sensitive(rmi , (int)rec ||
					((GtkCTreeRow *)((GtkCTreeNode *)(GTK_CLIST(tlist)->selection->data))->list.data)->level == 2);
				gtk_widget_set_sensitive(omi , (int)rec ||
					((GtkCTreeRow *)((GtkCTreeNode *)(GTK_CLIST(tlist)->selection->data))->list.data)->level == 2);
				gtk_widget_set_sensitive(wmi , (int)rec ||
					((GtkCTreeRow *)((GtkCTreeNode *)(GTK_CLIST(tlist)->selection->data))->list.data)->level == 2);
				gtk_widget_set_sensitive(fmi , (e && e->fapplication));
				gtk_widget_set_sensitive(umi , (e && e->uapplication));
				gtk_menu_popup (GTK_MENU(data), NULL, NULL,
					 NULL, NULL, 3, bevent->time);
			}
		}
	    break;
	    default: break;
	}

	return FALSE;
}

void build_tlist(w)
GtkWidget *w;
{
	GtkWidget *swin;
	GtkWidget *menu;
	GtkWidget *mi;

	swin = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (swin),
		GTK_POLICY_ALWAYS, GTK_POLICY_ALWAYS);
	gtk_widget_set_usize(swin , 600 , 300);
	gtk_box_pack_start(GTK_BOX(w) , swin , TRUE , TRUE , 1);
	gtk_widget_show (swin);

	tlist = gtk_ctree_new(9 , 0);
	gtk_clist_set_column_title (GTK_CLIST(tlist), 0 , gettext("Cache URL entry"));
	gtk_clist_set_column_title (GTK_CLIST(tlist), 1 , gettext("Local file"));
	gtk_clist_set_column_title (GTK_CLIST(tlist), 2 , gettext("Type"));
	gtk_clist_set_column_title (GTK_CLIST(tlist), 3 , gettext("Size"));
	gtk_clist_set_column_title (GTK_CLIST(tlist), 4 , gettext("Encoding"));
	gtk_clist_set_column_title (GTK_CLIST(tlist), 5 , gettext("Charset"));
	gtk_clist_set_column_title (GTK_CLIST(tlist), 6 , gettext("Mod. time"));
	gtk_clist_set_column_title (GTK_CLIST(tlist), 7 , gettext("Acces time"));
	gtk_clist_set_column_title (GTK_CLIST(tlist), 8 , gettext("Expires"));
	gtk_clist_column_titles_show (GTK_CLIST(tlist));
	gtk_ctree_set_line_style (GTK_CTREE(tlist), GTK_CTREE_LINES_DOTTED);
	gtk_clist_set_column_auto_resize (GTK_CLIST(tlist), 0 , TRUE);
	gtk_clist_set_column_auto_resize (GTK_CLIST(tlist), 1 , TRUE);
	gtk_clist_set_column_auto_resize (GTK_CLIST(tlist), 2 , TRUE);
	gtk_clist_set_column_auto_resize (GTK_CLIST(tlist), 3 , TRUE);
	gtk_clist_set_column_auto_resize (GTK_CLIST(tlist), 4 , TRUE);
	gtk_clist_set_column_auto_resize (GTK_CLIST(tlist), 5 , TRUE);
	gtk_clist_set_column_auto_resize (GTK_CLIST(tlist), 6 , TRUE);
	gtk_clist_set_column_auto_resize (GTK_CLIST(tlist), 7 , TRUE);
	gtk_clist_set_column_auto_resize (GTK_CLIST(tlist), 8 , TRUE);
	gtk_clist_set_selection_mode (GTK_CLIST(tlist), GTK_SELECTION_BROWSE);
	gtk_container_add(GTK_CONTAINER(swin) , tlist);
	gtk_widget_show(tlist);

	gtk_signal_connect(GTK_OBJECT(tlist) , "tree_select_row" , 
		GTK_SIGNAL_FUNC(UpdateEInfo) , NULL);


	menu = gtk_menu_new();
	gtk_widget_realize(menu);

	smi = gtk_menu_item_new_with_label(gettext("Save As ..."));
	gtk_menu_append(GTK_MENU(menu) , smi);
	gtk_widget_show(smi);

	gtk_signal_connect(GTK_OBJECT(smi) , "activate" ,
		GTK_SIGNAL_FUNC(SaveAs) , NULL);

	rmi = gtk_menu_item_new_with_label(gettext("Remove"));
	gtk_menu_append(GTK_MENU(menu) , rmi);
	gtk_widget_show(rmi);

	gtk_signal_connect(GTK_OBJECT(rmi) , "activate" ,
		GTK_SIGNAL_FUNC(Remove) , NULL);

	cmi = gtk_menu_item_new_with_label(gettext("Copy url to clipboard"));
	gtk_menu_append(GTK_MENU(menu) , cmi);
	gtk_widget_show(cmi);

	gtk_signal_connect(GTK_OBJECT(cmi) , "activate" ,
		GTK_SIGNAL_FUNC(SetSelectionURL) , NULL);

	mi = gtk_menu_item_new_with_label(gettext("Node info"));
	gtk_menu_append(GTK_MENU(menu) , mi);
	gtk_widget_show(mi);

	gtk_signal_connect(GTK_OBJECT(mi) , "activate" ,
		GTK_SIGNAL_FUNC(NodeInfo) , NULL);

	mi = gtk_menu_item_new();
	gtk_menu_append(GTK_MENU(menu) , mi);
	gtk_widget_show(mi);

	omi = gtk_menu_item_new_with_label(gettext("Make read-only"));
	gtk_menu_append(GTK_MENU(menu) , omi);
	gtk_widget_show(omi);

	gtk_signal_connect(GTK_OBJECT(omi) , "activate" ,
		GTK_SIGNAL_FUNC(RdOnly) , NULL);

	wmi = gtk_menu_item_new_with_label(gettext("Make writable"));
	gtk_menu_append(GTK_MENU(menu) , wmi);
	gtk_widget_show(wmi);

	gtk_signal_connect(GTK_OBJECT(wmi) , "activate" ,
		GTK_SIGNAL_FUNC(Writable) , NULL);

	mi = gtk_menu_item_new();
	gtk_menu_append(GTK_MENU(menu) , mi);
	gtk_widget_show(mi);

	fmi = gtk_menu_item_new_with_label(gettext("View file"));
	gtk_menu_append(GTK_MENU(menu) , fmi);
	gtk_widget_show(fmi);

	gtk_signal_connect(GTK_OBJECT(fmi) , "activate" ,
		GTK_SIGNAL_FUNC(ViewFile) , NULL);

	umi = gtk_menu_item_new_with_label(gettext("View URL"));
	gtk_menu_append(GTK_MENU(menu) , umi);
	gtk_widget_show(umi);

	gtk_signal_connect(GTK_OBJECT(umi) , "activate" ,
		GTK_SIGNAL_FUNC(ViewURL) , NULL);

	gtk_signal_connect(GTK_OBJECT(tlist) , "button_press_event" , 
		GTK_SIGNAL_FUNC(popup_tmenu) , menu);

}

int main(argc , argv)
int argc;
char **argv;
{
	GtkWidget *box,*sep,*frame;
	GdkColor rc;
	GtkStyle *rs,*my_style;
	static GtkTargetEntry targetlist[] = {
		{"STRING", 0, STRING},
                {"TEXT", 0, TEXT},
                {"COMPOUND_TEXT", 0, COMPOUND_TEXT}
        };



	gtk_init(&argc , &argv);

	load_rc();

	if (cmd_setup(argc, argv)) return 1;

#ifdef HAVE_GETTEXT
#ifdef LC_MESSAGES
        setlocale(LC_MESSAGES , "");
#endif
        textdomain(PACKAGE);
#endif
	gtk_set_locale();

	tl = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_window_set_title(GTK_WINDOW(tl), gettext("NScache"));
	gtk_signal_connect (GTK_OBJECT (tl), "destroy",
		GTK_SIGNAL_FUNC(Quit), NULL);
	gtk_signal_connect(GTK_OBJECT(tl) , "selection_clear_event" ,
		GTK_SIGNAL_FUNC(SelectionClear) , NULL);
	gtk_selection_add_targets (tl, GDK_SELECTION_PRIMARY,
		targetlist , sizeof(targetlist)/sizeof(targetlist[0]));
	gtk_signal_connect(GTK_OBJECT(tl) , "selection_get" ,
		GTK_SIGNAL_FUNC(SendSelection) , NULL);

	box = gtk_vbox_new(FALSE , 1);
	gtk_container_add(GTK_CONTAINER(tl) , box);
	gtk_widget_show(box);

	build_menu(box);

	sep = gtk_hseparator_new();
	gtk_widget_show(sep);
	gtk_box_pack_start(GTK_BOX (box), sep , FALSE, TRUE, 4);

	build_seldb(box);
	build_tlist(box);

	frame = gtk_frame_new(NULL);
	gtk_box_pack_start(GTK_BOX (box), frame , FALSE, TRUE, 4);
	gtk_widget_show(frame);

	status = gtk_label_new("");
	gtk_misc_set_alignment(GTK_MISC(status) , 0.0 , 0.5);
	gtk_container_add(GTK_CONTAINER(frame) , status);
	gtk_widget_show(status);

	frame = gtk_frame_new(NULL);
	gtk_box_pack_start(GTK_BOX (box), frame , FALSE, TRUE, 4);
	gtk_widget_show(frame);

	err = gtk_label_new("");
	my_style = gtk_rc_get_style(err);
	if (!my_style)
		my_style = gtk_widget_get_style(err);
	rs = gtk_style_copy(my_style);
	gdk_color_parse("#ff0000" , &rc);
	gdk_color_alloc(gdk_colormap_get_system() , &rc);
	rs->fg[GTK_STATE_NORMAL] = rc;
	gtk_widget_set_style(err , rs);
	gtk_misc_set_alignment(GTK_MISC(err) , 0.0 , 0.5);
	gtk_container_add(GTK_CONTAINER(frame) , err);
	gtk_widget_show(err);

	if (db_name)
	{
		gtk_entry_set_text(GTK_ENTRY(entry) , db_name);
		Open(NULL , entry);
	}
	else
	{
		char *p;
		
		if ((p = getenv("HOME")))
		{
			char pom[2048];

			sprintf(pom , "%s/.netscape/cache/index.db" , p);
			db_name = g_strdup(pom);
			gtk_entry_set_text(GTK_ENTRY(entry) , db_name);
			Open(NULL , entry);
		}
	}

	gtk_widget_show(tl);

	gtk_main();
	return 0;
}

