
/***********************************************************************
 * nbase.h -- The main include file exposing the external API for      *
 * libnbase, a library of base (often compatability) routines.         *
 * Programs using libnbase can guarantee the availability of functions *
 * like (v)snprintf and inet_aton.  This library also provides         *
 * consistency and extended features for some functions.  It was       *
 * originally written for use in the Nmap Security Scanner             *
 * ( http://www.insecure.org/nmap/ ).                                  *
 *                                                                     *
 ***********************************************************************
 *                                                                     *
 *  Many of the files contained in libnbase are compatability          *
 *  functions written by others.  License conditions for those files   *
 *  may vary and is generally included at the top of the files.   Be   *
 *  sure to read that information before you redistribute or           *
 *  contents of those files.                                           *
 *                                                                     *   
 *  Source is provided to this software because we believe users have  *
 *  a right to know exactly what a program is going to do before they  *
 *  run it.  This also allows you to audit the software for security   *
 *  holes (none have been found so far).                               *
 *                                                                     *
 *  Source code also allows you to port libnbase to new platforms, fix *
 *  bugs, and add new features.  You are highly encouraged to send     *
 *  your changes to fyodor@insecure.org for possible incorporation     *
 *  into the main distribution.  By sending these changes to Fyodor or *
 *  one the insecure.org development mailing lists, it is assumed that *
 *  you are offering Fyodor the unlimited, non-exclusive right to      *
 *  reuse, modify, and relicense the code.  This is important because  *
 *  the inability to relicense code has caused devastating problems    *
 *  for other Free Software projects (such as KDE and NASM).  Nmap     *
 *  will always be available Open Source.  If you wish to specify      *
 *  special license conditions of your contributions, just say so      *
 *  when you send them.                                                *
 *                                                                     *
 *  This program is distributed in the hope that it will be useful,    *
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of     *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.               *
 *                                                                     *
 ***********************************************************************/

/* $Id: nbase.h,v 1.3 2000/10/05 05:46:19 fyodor Exp $ */

#ifndef NBASE_H
#define NBASE_H

/* NOTE -- libnbase offers the following features that you should probably
 * be aware of:
 *
 * * 'inline' is defined to what is neccessary for the C compiler being
 *   used (which may be nothing)
 *
 * * snprintf, inet_aton, inet_pton, inet_aton, memcpy, and bzero are 
 *   provided if you don't have them (prototypes for these are 
 *   included either way).
 *
 * * WORDS_BIGENDIAN is defined if platform is big endian
 *
 * * Definitions included which give the operating system type.  They
 *   will generally be one of the following: LINUX, FREEBSD, NETBSD,
 *   OPENBSD, SOLARIS, SUNOS, BSDI, IRIX, NETBSD
 *
 * * Insures that GNU getopt_* functions exist (such as getopt_long_only
 *
 * * Various string functions such as Strncpy() and strcasestr() see protos 
 *   for more info.
 */

#include "nbase_config.h"

#include <unistd.h>
#include <stdlib.h>
#include <ctype.h>
#include <sys/types.h>



#ifndef HAVE___ATTRIBUTE__
#define __attribute__(x)
#endif


#include <stdarg.h>


/* Insure memcpy() and bzero() are available */
#ifndef HAVE_MEMCPY
#define memcpy(d, s, n) bcopy((s), (d), (n))
#endif

#ifndef HAVE_BZERO
#define bzero(s, n) memset((s), 0, (n))
#endif





/* sprintf family */
#if !defined(HAVE_SNPRINTF)
int snprintf (char *str, size_t sz, const char *format, ...)
     __attribute__ ((format (printf, 3, 4)));
#endif

#if !defined(HAVE_VSNPRINTF)
int vsnprintf (char *str, size_t sz, const char *format, va_list ap)
     __attribute__((format (printf, 3, 0)));
#endif

#if !defined(HAVE_ASPRINTF)
int asprintf (char **ret, const char *format, ...)
     __attribute__ ((format (printf, 2, 3)));
#endif

#if !defined(HAVE_VASPRINTF)
int vasprintf (char **ret, const char *format, va_list ap)
     __attribute__((format (printf, 2, 0)));
#endif

#if !defined(HAVE_ASNPRINTF)
int asnprintf (char **ret, size_t max_sz, const char *format, ...)
     __attribute__ ((format (printf, 3, 4)));
#endif

#if !defined(HAVE_VASNPRINTF)
int vasnprintf (char **ret, size_t max_sz, const char *format, va_list ap)
     __attribute__((format (printf, 3, 0)));
#endif

/* GNU getopt replacements ... Anyone have a BSD licensed version of these? */
#ifdef HAVE_GETOPT_H
#include <getopt.h>
#else
/* The next half-dozen lines are from gcc-2.95 ... -Fyodor */
/* Include getopt.h for the sake of getopt_long.
   We don't need the declaration of getopt, and it could conflict
   with something from a system header file, so effectively nullify that.  */
#define getopt getopt_loser
#include "getopt.h"
#undef getopt
#endif /* HAVE_GETOPT_H */

/* The usleep() function is important as well */
#ifndef HAVE_USLEEP
#ifdef HAVE_NANOSLEEP
void usleep(unsigned long usec);
#endif
#endif


/***************** String functions -- See nbase_str.c ******************/
#ifndef HAVE_STRCASESTR
/* strcasestr is like strstr() except case insensitive */
char *strcasestr(char *haystack, char *pneedle);
#endif
/* Strncpy is like strcpy() except it ALWAYS zero-terminates, even if
   it must truncate */
int Strncpy(char *dest, const char *src, size_t n);



#endif /* NBASE_H */


