/*
 * net6emu r1: net6emu.h
 * Emulation code for the IPv6 socket API extensions.
 *
 * Copyright (c) 2002 Christoph Pfisterer
 * Based on emulation code from OpenSSH, reused and improved under
 * OpenSSH's BSD-style license.
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use, copy,
 * modify, merge, publish, distribute, sublicense, and/or sell copies
 * of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE. 
 */

#ifndef __NET6EMU_H__
#define __NET6EMU_H__

#include "net6emuconf.h"

/*
 * address families (just to be sure)
 */

#ifndef PF_INET
# error PF_INET not defined -- you must include the system headers first
#endif /* !PF_INET */
#ifndef PF_UNSPEC
# define PF_UNSPEC 0
# define AF_UNSPEC PF_UNSPEC
#endif /* !PF_UNSPEC */
#ifndef PF_INET6
# define PF_INET6 999
# define AF_INET6 PF_INET6
#endif /* !PF_INET6 */

/*
 * error codes returned from getaddrinfo()
 */

#ifndef EAI_NODATA
# define EAI_NODATA        1
# define EAI_MEMORY        2
#endif /* !EAI_NODATA */

#if !defined(EAI_MEMORY)
# error EAI_ constants present but incomplete in system headers
#endif

/*
 * constants to pass into getaddrinfo()
 */

#ifndef AI_PASSIVE
# define AI_PASSIVE        1
# define AI_CANONNAME      2
# define AI_NUMERICHOST    4
#endif /* !AI_PASSIVE */

#if !defined(AI_CANONNAME) || !defined(AI_NUMERICHOST)
# error AI_ constants present but incomplete in system headers
#endif

/*
 * recommended buffer sizes for getnameinfo()
 */

#ifndef NI_MAXHOST
# define NI_MAXHOST        1025
#endif /* !NI_MAXHOST */
#ifndef NI_MAXSERV
# define NI_MAXSERV        32
#endif /* !NI_MAXSERV */

/*
 * constants to pass into getnameinfo()
 */

#ifndef NI_NUMERICHOST
# define NI_NUMERICHOST    1
# define NI_NUMERICSERV    2
# define NI_NOFQDN         4
# define NI_NAMEREQD       8
# define NI_DGRAM          16
#endif /* !NI_NUMERICHOST */

#if !defined(NI_NUMERICSERV) || !defined(NI_NOFQDN) || !defined(NI_NAMEREQD) || !defined(NI_DGRAM)
# error NI_ constants present but incomplete in system headers
#endif

/*
 * padded protocol-independent address structure
 *
 * NOTE: There seem to be differences between systems; some have a short
 *  giving the family, others have two chars: one for the length and one
 *  for the family. If you use net6emu, you should never access the
 *  ss_family member directly. Rather, cast to a struct sockaddr and use
 *  that, as it will know the system conventions.
 */

#ifndef N6E_HAVE_SOCKADDR_STORAGE
struct sockaddr_storage {
  unsigned short int ss_family;
  char ss_padding[126];
};
#endif /* !N6E_HAVE_SOCKADDR_STORAGE */

/*
 * data structure for getaddrinfo()
 */

#ifndef N6E_HAVE_STRUCT_ADDRINFO
struct addrinfo {
  int ai_flags;			/* AI_PASSIVE, AI_CANONNAME */
  int ai_family;		/* PF_xxx */
  int ai_socktype;		/* SOCK_xxx */
  int ai_protocol;		/* 0 or IPPROTO_xxx for IPv4 and IPv6 */
  size_t ai_addrlen;		/* length of ai_addr */
  struct sockaddr *ai_addr;	/* binary address */
  char *ai_canonname;		/* canonical name for hostname */
  struct addrinfo *ai_next;	/* next structure in linked list */
};
#endif /* !N6E_HAVE_STRUCT_ADDRINFO */

/*
 * getaddrinfo() and her friends
 */

#ifndef N6E_HAVE_GETADDRINFO
int getaddrinfo(const char *hostname, const char *servname, 
                const struct addrinfo *hints, struct addrinfo **res);
char *gai_strerror(int ecode);
void freeaddrinfo(struct addrinfo *ai);
#endif /* !N6E_HAVE_GETADDRINFO */

/*
 * getnameinfo() herself
 */

#ifndef N6E_HAVE_GETNAMEINFO
int getnameinfo(const struct sockaddr *sa, size_t salen, char *host, 
                size_t hostlen, char *serv, size_t servlen, int flags);
#endif /* !N6E_HAVE_GETNAMEINFO */


#endif /* __NET6EMU_H__ */
