/*\
 * $Id: bmptoppm.c,v 1.10 1992/11/24 19:38:17 dws Exp dws $
 * 
 * bmptoppm.c - Converts from a Microsoft Windows or OS/2 .BMP file to a
 * PPM file.
 * 
 * The current implementation is probably not complete, but it works for
 * all the BMP files I have.  I welcome feedback.
 * 
 * Copyright (C) 1992 by David W. Sanderson.
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation.  This software is provided "as is"
 * without express or implied warranty.
 * 
 * $Log: bmptoppm.c,v $
 * Revision 1.10  1992/11/24  19:38:17  dws
 * Added code to verify that reading occurred at the correct offsets.
 * Added copyright.
 *
 * Revision 1.9  1992/11/17  02:15:24  dws
 * Changed to include bmp.h.
 * Eliminated need for fseek(), and therefore the need for a
 * temporary file.
 *
 * Revision 1.8  1992/11/13  23:48:57  dws
 * Made definition of Seekable() static, to match its prototype.
 *
 * Revision 1.7  1992/11/11  00:17:50  dws
 * Generalized to use bitio routines.
 *
 * Revision 1.6  1992/11/10  23:51:44  dws
 * Enhanced command-line handling.
 *
 * Revision 1.5  1992/11/08  00:38:46  dws
 * Changed some names to help w/ addition of ppmtobmp.
 *
 * Revision 1.4  1992/10/27  06:28:28  dws
 * Corrected stupid typo.
 *
 * Revision 1.3  1992/10/27  06:17:10  dws
 * Removed a magic constant value.
 *
 * Revision 1.2  1992/10/27  06:09:58  dws
 * Made stdin seekable.
 *
 * Revision 1.1  1992/10/27  05:31:41  dws
 * Initial revision
\*/

#include    "bmp.h"
#include    "ppm.h"
#include    "bitio.h"

/* MAXCOLORS is the maximum size of a color map in a BMP image */
#define MAXCOLORS       256
/* BMPCOLORDEPTH is the number of intensities of a color channel (be it
   truecolor or in a color map) that are represented in a BMP image.
*/
#define BMPCOLORDEPTH   256

struct cmdline_info {
    /* All the information the user supplied in the command line,
       in a form easy for the program to use.
    */
    char *input_filespec;  /* Filespecs of input files */
    int verbose;    /* -verbose option */
};

static char    *ifname;

/*
 * Utilities
 */

static int GetByte ARGS((FILE * fp));
static short GetShort ARGS((FILE * fp));
static long GetLong ARGS((FILE * fp));
static void BMPreadfileheader ARGS((FILE *fp, unsigned long *ppos,
    unsigned long *poffBits));
static void BMPreadinfoheader ARGS((FILE *fp, unsigned long *ppos,
    unsigned long *pcx, unsigned long *pcy, unsigned short *pcBitCount,
    int *pclass, int *pcmapsize));
static int BMPreadcolormap ARGS((FILE *fp, unsigned long *ppos,
    unsigned short cBitCount, int class, pixval *R, pixval *G, pixval *B,
    int cmapsize));
static int
BMPreadrow(FILE *fp, 
    unsigned long * const ppos,     /* number of bytes read from fp */
    pixel * const row, const unsigned long cx, const unsigned short cBitCount, 
    const pixval R[], const pixval G[], const pixval B[]);
static pixel ** BMPreadbits ARGS((FILE *fp, unsigned long *ppos,
    unsigned long offBits, unsigned long cx, unsigned long cy,
    unsigned short cBitCount, int class, pixval *R, pixval *G, pixval *B));



static void
parse_command_line(int argc, char ** argv,
                   struct cmdline_info *cmdline_p) {
/*----------------------------------------------------------------------------
   Note that the file spec array we return is stored in the storage that
   was passed to us as the argv array.
-----------------------------------------------------------------------------*/
    optStruct *option_def = malloc(100*sizeof(optStruct));
        /* Instructions to OptParseOptions2 on how to parse our options.
         */
    optStruct2 opt;

    unsigned int option_def_index;

    option_def_index = 0;   /* incremented by OPTENTRY */
    OPTENTRY(0,   "verbose",     OPT_FLAG,   &cmdline_p->verbose,         0);

    /* Set the defaults */
    cmdline_p->verbose = FALSE;

    opt.opt_table = option_def;
    opt.short_allowed = FALSE;  /* We have no short (old-fashioned) options */
    opt.allowNegNum = FALSE;  /* We have no parms that are negative numbers */

    pm_optParseOptions2(&argc, argv, opt, 0);
        /* Uses and sets argc, argv, and some of *cmdline_p and others. */

    if (argc-1 == 0) 
        cmdline_p->input_filespec = "-";
    else if (argc-1 != 1)
        pm_error("Program takes zero or one argument (filename).  You "
                 "specified %d", argc-1);
    else
        cmdline_p->input_filespec = argv[1];

}



static char     er_read[] = "%s: read error";

static int
GetByte(fp)
    FILE           *fp;
{
    int             v;

    if ((v = getc(fp)) == EOF)
    {
        pm_error(er_read, ifname);
    }

    return v;
}

static short
GetShort(fp)
    FILE           *fp;
{
    short           v;

    if (pm_readlittleshort(fp, &v) == -1)
    {
        pm_error(er_read, ifname);
    }

    return v;
}

static long
GetLong(fp)
    FILE           *fp;
{
    long            v;

    if (pm_readlittlelong(fp, &v) == -1)
    {
        pm_error(er_read, ifname);
    }

    return v;
}


static void
readto(FILE *fp, unsigned long * const ppos, const unsigned long dst) {
/*----------------------------------------------------------------------------
   Read as many bytes as necessary to position the file 'fp' at offset 
   'dst', assuming we have already ready *ppos bytes from it.
-----------------------------------------------------------------------------*/
    unsigned long   pos;

    if(!fp || !ppos)
        return;

    pos = *ppos;

    if(pos > dst)
        pm_error("%s: internal error in readto()", ifname);

    for(; pos < dst; pos++)
    {
        if (getc(fp) == EOF)
        {
            pm_error(er_read, ifname);
        }
    }

    *ppos = pos;
}

/*
 * BMP reading routines
 */

static void
BMPreadfileheader(fp, ppos, poffBits)
    FILE           *fp;
    unsigned long  *ppos;   /* number of bytes read from fp */
    unsigned long  *poffBits;
{
    unsigned long   cbSize;
    unsigned short  xHotSpot;
    unsigned short  yHotSpot;
    unsigned long   offBits;

    if (GetByte(fp) != 'B')
    {
        pm_error("%s is not a BMP file", ifname);
    }
    if (GetByte(fp) != 'M')
    {
        pm_error("%s is not a BMP file", ifname);
    }

    cbSize = GetLong(fp);
    xHotSpot = GetShort(fp);
    yHotSpot = GetShort(fp);
    offBits = GetLong(fp);

    *poffBits = offBits;

    *ppos += 14;
}

static void
BMPreadinfoheader(fp, ppos, pcx, pcy, pcBitCount, pclass, pcmapsize)
    FILE           *fp;
    unsigned long  *ppos;   /* number of bytes read from fp */
    unsigned long  *pcx;
    unsigned long  *pcy;
    unsigned short *pcBitCount;
    int            *pclass;
    int        *pcmapsize;
{
    unsigned long   cbFix;
    unsigned short  cPlanes;

    unsigned long   cx;
    unsigned long   cy;
    unsigned short  cBitCount;
    unsigned long   compression;
    int             class;

    cbFix = GetLong(fp);

    switch (cbFix)
    {
    case 12:
        class = C_OS2;

        cx = GetShort(fp);
        cy = GetShort(fp);
        cPlanes = GetShort(fp);
        cBitCount = GetShort(fp);
        /* I actually don't know if the OS/2 BMP format allows
           cBitCount > 8 or if it does, what it means, but ppmtobmp
           creates such BMPs, more or less as a byproduct of creating
           the same for Windows BMP, so we interpret cBitCount > 8 the
           same as for Windows.
        */
        if (cBitCount <= 8)
            *pcmapsize = 1 << cBitCount;
        else if (cBitCount == 24)
            *pcmapsize = 0;
        /* There is a 16 bit truecolor format, but we don't know how the
           bits are divided among red, green, and blue, so we can't handle it.
           */
        else
            pm_error("Unrecognized bits per pixel in BMP file header: %d",
                     cBitCount);

        break;
    case 40: {
        int colorsimportant;   /* ColorsImportant value from header */
        int colorsused;        /* ColorsUsed value from header */

        class = C_WIN;

        cx = GetLong(fp);
        cy = GetLong(fp);
        cPlanes = GetShort(fp);
        cBitCount = GetShort(fp);
 
        compression = GetLong(fp);
        if (compression != 0) 
            pm_error("Input is compressed.  This program doesn't handle "
                     "compressed images.  Compression type code = %ld",
                     compression);

        /* And read the rest of the junk in the 40 byte header */
        GetLong(fp);   /* ImageSize */
        GetLong(fp);   /* XpixelsPerM */
        GetLong(fp);   /* YpixelsPerM */
        colorsused = GetLong(fp);   /* ColorsUsed */
        /* See comments in bmp.h for info about the definition of the following
           word and its relationship to the color map size (*pcmapsize).
           */
        colorsimportant = GetLong(fp);  /* ColorsImportant */

        if (colorsused != 0)
            *pcmapsize = colorsused;
        else {
            if (cBitCount <= 8)
                *pcmapsize = 1 << cBitCount;
            else if (cBitCount == 24)
                *pcmapsize = 0;
            /* There is a 16 bit truecolor format, but we don't know
               how the bits are divided among red, green, and blue, so
               we can't handle it.  */
            else
                pm_error("Unrecognized bits per pixel in BMP file header: %d",
                         cBitCount);
        }
    }
        break;
    default:
        pm_error("%s: unknown cbFix: %ld", ifname, cbFix);
        break;
    }

    if (cPlanes != 1)
    {
        pm_error("%s: don't know how to handle cPlanes = %d"
             ,ifname
             ,cPlanes);
    }

    switch (class)
    {
    case C_WIN:
        pm_message("Windows BMP, %ldx%ldx%d"
               ,cx
               ,cy
               ,cBitCount);
        break;
    case C_OS2:
        pm_message("OS/2 BMP, %ldx%ldx%d"
               ,cx
               ,cy
               ,cBitCount);
        break;
    }

#ifdef DEBUG
    pm_message("cbFix: %d", cbFix);
    pm_message("cx: %d", cx);
    pm_message("cy: %d", cy);
    pm_message("cPlanes: %d", cPlanes);
    pm_message("cBitCount: %d", cBitCount);
#endif

    *pcx = cx;
    *pcy = cy;
    *pcBitCount = cBitCount;
    *pclass = class;

    *ppos += cbFix;
}

/*
 * returns the number of bytes read, or -1 on error.
 */
static int
BMPreadcolormap(FILE *fp, 
                unsigned long *ppos, /* number of bytes read from fp */
                unsigned short cBitCount, 
                int class, pixval *R, pixval *G, pixval *B, int cmapsize)
{
    int             i;
    int     nbyte = 0;

    for (i = 0; i < cmapsize; i++)
    {
        /* There is a document that says the bytes or ordered R,G,B,Z,
           but in practice it appears to be the following instead:
           */
        B[i] = (pixval) GetByte(fp);
        G[i] = (pixval) GetByte(fp);
        R[i] = (pixval) GetByte(fp);
        nbyte += 3;

        if (class == C_WIN)
        {
            (void) GetByte(fp);
            nbyte++;
        }
    }

    *ppos += nbyte;
    return nbyte;
}

static int
BMPreadrow(FILE *fp, 
    unsigned long * const ppos,     /* number of bytes read from fp */
    pixel * const row, const unsigned long cx, const unsigned short cBitCount, 
    const pixval R[], const pixval G[], const pixval B[])
{
/*
 * returns the number of bytes read, or -1 on error.
 */
    BITSTREAM       b;
    unsigned        nbyte = 0;
    int             rc;
    unsigned int    col;

    if ((b = pm_bitinit(fp, "r")) == (BITSTREAM) 0)
    {
        return -1;
    }

    for (col = 0; col < cx; col++)
    {
        unsigned long   v;

        /* There is a document that gives a much different format for
           24 bit BMPs.  But this seems to be the de facto standard.
        */
        if ((rc = pm_bitread(b, cBitCount, &v)) == -1)
        {
            return -1;
        }
        nbyte += rc;

        if (cBitCount <= 8)
            /* It's colormapped, so look up v in the colormap */
            PPM_ASSIGN(row[col], R[v], G[v], B[v]);
        else if (cBitCount == 24)
            /* It's truecolor, so just unpack v */
            PPM_ASSIGN(row[col], 
                       (v >>  0) & 0xff,
                       (v >>  8) & 0xff,
                       (v >> 16) & 0xff);

    }

    if ((rc = pm_bitfini(b)) != 0)
    {
        return -1;
    }

    /*
     * Make sure we read a multiple of 4 bytes.
     */
    while (nbyte % 4)
    {
        GetByte(fp);
        nbyte++;
    }

    *ppos += nbyte;
    return nbyte;
}

static pixel **
BMPreadbits(FILE *fp, 
            unsigned long *ppos, /* number of bytes read from fp already */
            unsigned long offBits, /* Offset into file where raster begins */
            unsigned long cx, unsigned long cy, 
            unsigned short cBitCount, int class, 
            pixval *R, pixval *G, pixval *B)
{
    pixel         **pixels; /* output */
    long            y;

    readto(fp, ppos, offBits);

    pixels = ppm_allocarray(cx, cy);

    /*
     * The picture is stored bottom line first, top line last
     */

    for (y = cy - 1; y >= 0; y--)
    {
        int rc;
        rc = BMPreadrow(fp, ppos, pixels[y], cx, cBitCount, R, G, B);

        if(rc == -1)
        {
            pm_error("%s: couldn't read row %ld"
                     ,ifname
                     ,y);
        }
        if(rc%4)
        {
            pm_error("%s: row had bad number of bytes: %d"
                     ,ifname
                     ,rc);
        }
    }

    return pixels;
}

int
main(argc, argv)
    int             argc;
    char          **argv;
{
    struct cmdline_info cmdline;
    FILE* ifp;

    int             rc;
    unsigned long   pos = 0;

    /* The following are all information from the BMP headers */
    unsigned long   offBits;
      /* Byte offset into file of raster */
    unsigned long   cx;
    unsigned long   cy;
    unsigned short  cBitCount;
    int             class;

    int     cmapsize;
      /* Number of colors in the colormap in the BMP image */

    /* The colormap */
    pixval          R[MAXCOLORS];   /* reds */
    pixval          G[MAXCOLORS];   /* greens */
    pixval          B[MAXCOLORS];   /* blues */

    /* The image */
    pixel         **pixels;


    ppm_init(&argc, argv);

    parse_command_line(argc, argv, &cmdline);

    ifp = pm_openr(cmdline.input_filespec);
    if (strcmp(cmdline.input_filespec, "-"))
        ifname = "Standard Input";
    else 
        ifname = cmdline.input_filespec;

    BMPreadfileheader(ifp, &pos, &offBits);
    BMPreadinfoheader(ifp, &pos, &cx, &cy, &cBitCount, &class, &cmapsize);

    if (cmdline.verbose) {
        pm_message("BMP image header says:");
        pm_message("  Class of BMP: %s", 
                   class == C_WIN ? "Windows" : class == C_OS2 ? "OS/2" :
                   "???");
        pm_message("  Width: %ld pixels", cx);
        pm_message("  Height: %ld pixels", cy);
        pm_message("  Byte offset of raster within file: %ld", offBits);
        pm_message("  Bits per pixel in raster: %d", cBitCount);
        pm_message("  Colors in color map: %d", cmapsize);
    }        

    if(offBits != BMPoffbits(class, cBitCount, cmapsize))
    {
        pm_message("warning: offBits is %ld, expected %ld"
                   , offBits
                   , BMPoffbits(class, cBitCount, cmapsize));
    }

    rc = BMPreadcolormap(ifp, &pos, cBitCount, class, R, G, B, cmapsize);

    if(rc != BMPlencolormap(class, cBitCount, cmapsize))
    {
        pm_message("warning: %d-byte RGB table, expected %ld bytes"
                   , rc
                   , BMPlencolormap(class, cBitCount, cmapsize));
    }


    pixels = BMPreadbits(ifp, &pos, offBits, cx, cy
            , cBitCount, class, R, G, B);

    if(pos != BMPlenfile(class, cBitCount, cmapsize, cx, cy))
    {
        pm_message("warning: read %ld bytes, expected to read %ld bytes"
                   , pos
                   , BMPlenfile(class, cBitCount, cmapsize, cx, cy));
    }

    pm_close(ifp);
    ppm_writeppm(stdout, pixels, cx, cy, (pixval) BMPCOLORDEPTH-1, 0);
    pm_close(stdout);

    exit(0);
}

