/* Bryan got this from mm.ftp-cs.berkeley.edu from the package
   mpeg-encode-1.5b-src under the name eyuvtoppm.c on March 30, 2000.  
   The file was dated April 14, 1995.  

   Bryan rewrote the program entirely to match Netpbm coding style,
   use the Netpbm libraries and also to output to stdout and ignore
   any specification of an output file on the command line and not
   segfault when called with no arguments.

   There was no attached documentation except for this:  Encoder/Berkeley
   YUV format is merely the concatenation of Y, U, and V data in order.
   Compare with Abekda YUV, which interlaces Y, U, and V data.  */

/*
 * Copyright (c) 1995 The Regents of the University of California.
 * All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose, without fee, and without written agreement is
 * hereby granted, provided that the above copyright notice and the following
 * two paragraphs appear in all copies of this software.
 *
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY FOR
 * DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
 * OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE UNIVERSITY OF
 * CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND THE UNIVERSITY OF CALIFORNIA HAS NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.  */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include "ppm.h"

typedef unsigned char uint8;

#define CHOP(x)     ((x < 0) ? 0 : ((x > 255) ? 255 : x))



struct cmdline_info {
    /* All the information the user supplied in the command line,
       in a form easy for the program to use.
    */
    char *input_filespec;  /* Filespecs of input file */
    unsigned int width;
    unsigned int height;
};



static void
parseCommandLine(int argc, char ** argv,
                 struct cmdline_info *cmdlineP) {

    optStruct3 opt;
    unsigned int option_def_index = 0;
    optEntry *option_def = malloc(100*sizeof(optEntry));

    OPTENT3('w', "width",     OPT_UINT,  &cmdlineP->width,   NULL,         0);
    OPTENT3('h', "height",    OPT_UINT,  &cmdlineP->height,  NULL,         0);
    
    /* DEFAULTS */
    cmdlineP->width = 352;
    cmdlineP->height = 240;

    opt.opt_table = option_def;
    opt.short_allowed = TRUE;
    opt.allowNegNum = FALSE;

    optParseOptions3(&argc, argv, opt, sizeof(opt), 0);

    if (cmdlineP->width == 0)
        pm_error("The width cannot be zero.");
    if (cmdlineP->width % 2 != 0)
        pm_error("The width of an eyuv image must be an even number.  "
                 "You specified %u.", cmdlineP->width);
    if (cmdlineP->height == 0)
        pm_error("The height cannot be zero.");
    if (cmdlineP->height % 2 != 0)
        pm_error("The height of an eyuv image must be an even number.  "
                 "You specified %u.", cmdlineP->height);


    if (argc-1 == 0) 
        cmdlineP->input_filespec = "-";
    else if (argc-1 != 1)
        pm_error("Program takes zero or one argument (filename).  You "
                 "specified %d", argc-1);
    else
        cmdlineP->input_filespec = argv[1];

}



static uint8 ** 
AllocUint8Array(unsigned int const cols, unsigned int const rows) {

    uint8 **retval;
    unsigned int y;

    retval = (uint8 **) malloc(sizeof(uint8 *) * rows);
    if (retval == NULL)
        pm_error("Unable to allocate storage for %d x %d byte array.",
                 cols, rows);

    for (y = 0; y < rows; y++) {
        retval[y] = (uint8 *) malloc(sizeof(uint8) * cols);
        if (retval[y] == NULL)
            pm_error("Unable to allocate storage for %d x %d byte array.",
                     cols, rows);
    }
    return retval;
}



static int ** 
AllocIntArray(unsigned int const cols, unsigned int const rows) {

    int **retval;
    unsigned int y;

    retval = (int **) malloc(sizeof(int *) * rows);
    if (retval == NULL)
        pm_error("Unable to allocate storage for %d x %d byte array.",
                 cols, rows);

    for (y = 0; y < rows; y++) {
        retval[y] = (int *) malloc(sizeof(int) * cols);
        if (retval[y] == NULL)
            pm_error("Unable to allocate storage for %d x %d byte array.",
                     cols, rows);
    }
    return retval;
}



static void 
FreeArray(void ** const array, unsigned int const rows) {

    unsigned int y;

    for (y = 0; y < rows; y++)
        free(array[y]);
    free(array);
}



static void 
YUVtoPPM(unsigned int const cols, unsigned int const rows,
         uint8 ** const orig_y, uint8 ** const orig_cb, uint8 ** const orig_cr,
         pixel *** pixelsP ) {
    
    int ** const Y = AllocIntArray(cols, rows);
    int ** const U = AllocIntArray(cols, rows);
    int ** const V = AllocIntArray(cols, rows);

    int y;

    *pixelsP = ppm_allocarray(cols, rows);

    for ( y = 0; y < rows/2; y ++ ) {
        int x;
        for ( x = 0; x < cols/2; x ++ ) {
            U[y][x] = orig_cb[y][x] - 128;
            V[y][x] = orig_cr[y][x] - 128;
        }
    }

    for ( y = 0; y < rows; y ++ ) {
        int x;
        for ( x = 0; x < cols; x ++ ) 
            Y[y][x] = orig_y[y][x] - 16;
    }

    for ( y = 0; y < rows; y++ ) {
        int x;
        for ( x = 0; x < cols; x++ ) {
            long   tempR, tempG, tempB;
            int     r, g, b;
            /* look at yuvtoppm source for explanation */

            tempR = 104635*V[y/2][x/2];
            tempG = -25690*U[y/2][x/2] + -53294 * V[y/2][x/2];
            tempB = 132278*U[y/2][x/2];

            tempR += (Y[y][x]*76310);
            tempG += (Y[y][x]*76310);
            tempB += (Y[y][x]*76310);
            
            r = CHOP((int)(tempR >> 16));
            g = CHOP((int)(tempG >> 16));
            b = CHOP((int)(tempB >> 16));
            
            PPM_ASSIGN((*pixelsP)[y][x], r, g, b);
        }
    }
}



static void 
ReadYUV(FILE * const yuvfile,
        unsigned int const cols, unsigned int const rows,
        uint8 *** const orig_yP, 
        uint8 *** const orig_cbP, 
        uint8 *** const orig_crP,
        bool * const eofP) {

    unsigned int y;
    int c;

    *orig_yP  = AllocUint8Array(cols, rows);
    *orig_cbP = AllocUint8Array(cols, rows);
    *orig_crP = AllocUint8Array(cols, rows);

    c = fgetc(yuvfile);
    if (c < 0)
        *eofP = TRUE;
    else {
        *eofP = FALSE;
        ungetc(c, yuvfile);
    }
    if (!*eofP) {
        for (y = 0; y < rows; y++)            /* Y */
            fread((*orig_yP)[y], 1, cols, yuvfile);
        
        for (y = 0; y < rows / 2; y++)            /* U */
            fread((*orig_cbP)[y], 1, cols / 2, yuvfile);
        
        for (y = 0; y < rows / 2; y++)            /* V */
            fread((*orig_crP)[y], 1, cols / 2, yuvfile);
        if (feof(yuvfile))
            pm_error("Premature end of file reading EYUV input file");
    }
}



int
main(int argc, char **argv) {

    FILE *ifp;
    struct cmdline_info cmdline;
    unsigned int frameSeq;

    ppm_init(&argc, argv);

    parseCommandLine(argc, argv, &cmdline);

    ifp = pm_openr(cmdline.input_filespec);

    for (frameSeq = 0; !feof(ifp); frameSeq++) {
        uint8 **orig_y, **orig_cb, **orig_cr;
        pixel ** pixels;
        bool eof;

        ReadYUV(ifp, cmdline.width, cmdline.height, 
                &orig_y, &orig_cb, &orig_cr, &eof);
        if (!eof) {
            pm_message("Converting Frame %u", frameSeq);

            YUVtoPPM(cmdline.width, cmdline.height, orig_y, orig_cb, orig_cr, 
                     &pixels);
            FreeArray((void**) orig_y, cmdline.height); 
            FreeArray((void**) orig_cb, cmdline.height); 
            FreeArray((void**) orig_cr, cmdline.height);
            ppm_writeppm(stdout, pixels, cmdline.width, cmdline.height, 
                         255, FALSE);
            ppm_freearray(pixels, cmdline.height);
        }
    }
    pm_close(ifp);
    exit(0);
}
