/*

Copyright (C) 2000 - 2007 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#ifdef LINUX
#define __FAVOR_BSD
#endif
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include <gtk/gtk.h>
#include <callbacks.h>
#include <nd.h>
#include <nd_dialog.h>
#include <nd_gui.h>
#include <nd_packet.h>
#include <nd_trace.h>
#include <nd_trace_registry.h>
#include <nd_tp.h>
#include <nd_protocol_inst.h>
#include <nd_raw_protocol.h>
#include <support.h>



static void
packet_ob_insert_post(LND_Packet *packet)
{
  D_ENTER;

  if (! packet->next)
    nd_tp_packet_insert_at_index(packet->part, packet,
				 libnd_packet_get_index(packet));
  else
    {
      nd_gui_list_update(libnd_packet_get_trace(packet));
      nd_gui_num_packets_set();
    }
  
  D_RETURN;
}

static void
packet_ob_delete_pre(LND_Packet *packet)
{
  D_ENTER;
  nd_packet_delete(packet);
  D_RETURN;
}

static void
packet_ob_delete_post(LND_Packet *packet)
{
  D_ENTER;
  if (libnd_packet_get_trace(packet) == nd_trace_registry_get_current())
    nd_gui_num_packets_set();
  D_RETURN;
}

static void
packet_ob_visibility(LND_Packet *packet)
{
  D_ENTER;
  nd_packet_set_filtered_state(packet);
  D_RETURN;
}

static void
packet_ob_fix_gui(LND_Packet *packet)
{
  LND_Trace *trace;

  return_if_no_current_trace(trace);

  /* If this is not our currently-selected packet, just update
   * the packet list widget. Updating the notebook will only
   * slow things down and cause flickering.
   */
  if (!packet || packet != nd_trace_get_current_packet(trace))
    {
      nd_gui_list_update_packet(packet);    
      return;
    }

  nd_packet_set_gui(packet);
}

void            
nd_packet_init(void)
{
  LND_PacketObserver *ob = libnd_packet_observer_new();
  
  ob->packet_insert_post    = packet_ob_insert_post;
  ob->packet_delete_pre     = packet_ob_delete_pre;
  ob->packet_delete_post    = packet_ob_delete_post;
  ob->packet_visibility     = packet_ob_visibility;
  ob->packet_modified       = packet_ob_fix_gui;
  ob->packet_updated        = packet_ob_fix_gui;
  ob->packet_fixed          = packet_ob_fix_gui;
  /* Packet len/caplen change will trigger packet_fixed anyway */
  
  libnd_packet_add_observer(ob);
}


void
nd_packet_delete(LND_Packet *packet)
{
  LND_Trace *trace;

  D_ENTER;

  if (!packet)
    D_RETURN;

  trace = libnd_packet_get_trace(packet);

  if (packet == nd_trace_get_current_packet(trace))
    nd_trace_set_current_packet(trace, NULL);

  nd_gui_list_remove_row(trace, libnd_packet_get_index(packet));
  
  D_RETURN;
}



void            
nd_packet_set_filtered_state(LND_Packet *packet)
{
  LND_Trace *trace;
  int index;

  D_ENTER;

  if (!packet)
    D_RETURN;

  if (! (trace = libnd_packet_get_trace(packet)))
    D_RETURN;

  /* If this is not one of the in-memory packets, don't update */
  if (!packet->prev &&                               /* no previous packet */
      !packet->next &&                               /* AND no next packet */
      (!packet->part || packet->part->pl != packet)) /* AND it's not the only packet */
    D_RETURN;
    
  index = libnd_packet_get_index(packet);
  nd_gui_list_set_row_filtered(trace, index, libnd_packet_is_filtered(packet));
  D_RETURN;
}

static gboolean
packet_set_gui_cb(LND_Packet *packet, LND_ProtoData *pd, void *user_data)
{
  LND_Trace *trace = (LND_Trace *) user_data;
  nd_trace_add_proto_tab(trace, pd->inst.proto, pd->inst.nesting);
  return TRUE;
  TOUCH(packet);
}

void    
nd_packet_set_gui(const LND_Packet *packet)
{
  GtkWidget      *page;
  GtkNotebook    *notebook;
  ND_Protocol    *proto_gui;
  ND_Trace       *trace_gui;
  LND_Trace      *trace;
  LND_Protocol   *raw_proto, *pcap_proto = NULL;
  LND_ProtoData  *pd;
  LND_ProtoInfo  *pinf;
  int             i;
  int             real_proto_start = 0; /* the tab at which "real" protocols start */
  GList          *l;
  gboolean        has_sel_proto = FALSE;

  D_ENTER;

  D(("Setting gui to packet %p\n", packet));

  /* Whatever we do in this function does not affect the user's preferred
   * protocol, so make sure we take no switching of the current notebook
   * tabs as the signal to update that preferential protocol. We do this
   * by unhooking the signal handler that takes care of updating this
   * preferential choice.
   */  

  if (!packet)
    {
      LND_Trace *trace;

      return_if_no_current_trace(trace);
      notebook = nd_trace_get_notebook(trace);
      D_ASSERT_PTR(notebook);
     
      /* Hide all tabs */      
      i = 0;
      page = gtk_notebook_get_nth_page(notebook, 0);
      
      while (page)
	{
	  gtk_widget_hide(page);
	  page = gtk_notebook_get_nth_page(notebook, ++i);
	}
      
      goto flush;
    }

  trace = libnd_packet_get_trace(packet);
  if (! (trace_gui = nd_trace_get(trace)))
    D_RETURN;

  notebook = nd_trace_get_notebook(trace);
  D_ASSERT_PTR(notebook);

  /* See if we have the pcap "protocol" plugin; we always should. */
  pcap_proto = libnd_proto_registry_find(LND_PROTO_LAYER_LINK, 0xa1b2c3d4);

  /* First ensure that the notebook actually has tabs for all the
   * protocols contained in the packet. After that, we hide any
   * tabs for the protocols in the notebook that are not in the packet.
   * We can just attempt to add new tabs via nd_trace_add_proto_tab(),
   * since it'll detect when a protocol is already registered and return.
   */
  libnd_packet_foreach_proto((LND_Packet*) packet, packet_set_gui_cb, trace);

  /* Now show/hide the protocol tabs for this packet, and also gray
   * out the protocol menus that aren't applicable for this packet.
   */
  page = gtk_notebook_get_nth_page(notebook, 0);
  
  for (i = 0; gtk_notebook_get_nth_page(notebook, i); i++)
    {  
      page = gtk_notebook_get_nth_page(notebook, i);
      D_ASSERT_PTR(page);

      ND_GTK_GET(pinf, page, "pinf");
      proto_gui = nd_proto_get(pinf->inst.proto);
      
      /* Always show the pcap "protocol" if we have it. */
      if (pcap_proto && pinf->inst.proto == pcap_proto)
	{
	  gtk_notebook_reorder_child(notebook, nd_proto_info_get(pinf)->proto_tab, 0);
	  proto_gui->set_gui(packet, pinf);
	  real_proto_start = 1;
	  gtk_widget_show(page);
	  continue;
	}
      
      /* Whereas for the tab itself, we want to keep an
       * eye on that nesting level.
       */
      if (libnd_packet_has_proto_nested(packet, pinf->inst.proto, pinf->inst.nesting))
	{
	  gtk_widget_show(page);
	}
      else
	{
	  gtk_widget_hide(page);
	  D(("Packet doesn't have %s at level %i\n", pinf->inst.proto->name, pinf->inst.nesting));
	}
    }

  /* Set has_sel_proto to TRUE if the packet has the currently
   * selected protocol. This is so we can try to keep that
   * protocol selected for the new packet.
   */
  if (trace_gui->cur_pi_sel && trace_gui->cur_pi_sel->proto != pcap_proto)
    {
      has_sel_proto =
	libnd_packet_has_proto_nested(packet,
				      trace_gui->cur_pi_sel->proto,
				      trace_gui->cur_pi_sel->nesting);
    }

  /* If the packet uses the raw protocol data displayer,
   * reset its label for now -- it may get improved upon
   * in the loop below:
   */
  raw_proto = libnd_raw_proto_get();
  if (libnd_packet_has_proto(packet, raw_proto) && trace)
    {
      ND_ProtoInfo *pinf_raw = nd_raw_proto_get_gui(trace);      
      gtk_label_set_text(GTK_LABEL(pinf_raw->proto_label), raw_proto->name);
    }

  /* Now, update each of the packet's protocol header's contents in the GUI,
   * and make sure they're sorted correctly.
   */
  for (l = packet->pd, i = real_proto_start; l; l = g_list_next(l), i++)
    {
      pd = (LND_ProtoData *) l->data;
      
      if (i == 0 && !trace_gui->cur_pi_sel)
	nd_trace_set_current_proto_selection(trace, &pd->inst);
      
      /* only update the gui for the first protocol, and only if it
	 does not have the selected protocol (which we set below)
	 anyway. Kills a lot of flickering :)
      */
      nd_trace_set_current_proto(trace, &pd->inst, (i == 0 && !has_sel_proto));
      
      pinf = nd_trace_get_proto_info(trace, pd->inst.proto, pd->inst.nesting);

      if (!pinf)
	continue;

      nd_gui_proto_table_block_events(trace, pinf);
      proto_gui = nd_proto_get(pd->inst.proto);
      proto_gui->set_gui(packet, pinf);
      nd_gui_proto_table_unblock_events(trace, pinf);

      gtk_notebook_reorder_child(notebook, nd_proto_info_get(pinf)->proto_tab, i);
    }
  
  gtk_signal_handler_block_by_data(GTK_OBJECT(notebook), notebook);
  if (trace_gui->cur_pi_sel)
    nd_trace_set_current_proto(trace, trace_gui->cur_pi_sel, TRUE);
  gtk_signal_handler_unblock_by_data(GTK_OBJECT(notebook), notebook);

  nd_gui_list_update_packet(packet);    
  
 flush:
  
  /* Okay, this one took forever. The problem is that we do NOT want
   * any protocol tab switches in the notbook to be understood as user
   * input here (which would cause the user's preferred protocol to be
   * switched, likely to the pcap tab). Since all the relevant signal
   * handlers normally are triggered *after* we leave this function,
   * we force an event flugh at this point, so we can make sure we
   * block the signal handler that'd update the preferred-protocol
   * selection.
   */
  
  gtk_signal_handler_block_by_data(GTK_OBJECT(notebook), notebook);
  while(gtk_events_pending())
    gtk_main_iteration();
  gtk_signal_handler_unblock_by_data(GTK_OBJECT(notebook), notebook);
  
  D_RETURN;
}
