/* 
 * Copyright (C) 2004 Michel Arboi <mikhail@nessus.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2,
 * as published by the Free Software Foundation
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
#include <includes.h>

#define GRAB_MAX_SOCK	2048
#define GRAB_MIN_SOCK	32
#define GRAB_MAX_SOCK_SAFE	(GRAB_MIN_SOCK * 2)

PlugExport int plugin_init(struct arglist * desc)
{
 plug_set_id(desc, 10335);
 plug_set_version(desc, "$Revision: 1.25 $");
   
         
 plug_set_name(desc, "Nessus TCP scanner", NULL);
 plug_set_summary(desc, "Look for open TCP ports & services banners", NULL);
 plug_set_description(desc, "\
This plugin is a classical TCP port scanner\n\
It shall be reasonably quick even against a firewalled target.\n\
\n\
Once a TCP connection is open, it grabs any available banner\n\
for the service identification plugins\n\
\n\
Note that TCP scanners are more intrusive than \n\
SYN (half open) scanners\
", NULL);
 
 plug_set_copyright(desc, "(C) 2004 Michel Arboi <mikhail@nessus.org>", NULL);
 plug_set_category(desc, ACT_SCANNER);
 plug_set_family(desc, "Scanners de ports", "francais");
 plug_set_family(desc, "Port scanners", NULL);

 plug_set_dep(desc, "ping_host.nasl");
 return(0);
}
 

typedef struct {
  int			fd;
  struct timeval	tictac;		/* open time */
  unsigned short	port;
  unsigned char		state;
} grab_socket_t;

#define DIFFTV(t1,t2)	(t1.tv_sec - t2.tv_sec + (t1.tv_usec - t2.tv_usec) / 1000000)
#define DIFFTVu(t1,t2)	((t1.tv_sec - t2.tv_sec) * 1000000.0 + (t1.tv_usec - t2.tv_usec))

#define GRAB_SOCKET_UNUSED	0
#define GRAB_SOCKET_OPENING	1
#define GRAB_SOCKET_OPEN	2

#define GRAB_PORT_UNKNOWN	0
#define GRAB_PORT_CLOSED	1
#define GRAB_PORT_OPEN		2
#define GRAB_PORT_FILTERED	3
#define GRAB_PORT_NOT_TESTED	254
#define GRAB_PORT_TESTING	255
  
#ifdef DEBUG
# define DISPLAY
#endif
#undef STATISTICS
/*
 * If you want to enable the "statistics", define STATISTICS and link 
 * the plugin with libm (we need sqrt)
 */
#undef BLACKLIST_FILTERED_PORTS
/*
 * MA 2005-05-08: Too dangerous, as nessus_tcp_scanner may miss connections
 * against slow machines.
 * By the way, nessus_tcp_scanner was the only port scanner that did this.
 */


static int
my_socket_close(int s)
{
#ifndef SO_LINGER
  if (shutdown(s, 2) < 0)
#ifdef DEBUG
    perror("shutdown")
#endif
      ;
#endif
  return close(s);
}

#ifdef BLACKLIST_FILTERED_PORTS
static void
remember_filtered_port(struct arglist* desc, int port)
{
  char		k[80];
  snprintf(k, sizeof(k), "/tmp/ConnectTimeout/TCP/%d", port);
  plug_set_key(desc, k, ARG_INT, (void*)1); 
}
#endif

static int
banner_grab(const struct in_addr *pia, const char* portrange, 
	    const int read_timeout,
	    int		min_cnx,
	    int		max_cnx,
	    struct arglist *globals, 
	    struct arglist *desc,
	    struct arglist *hostinfos)
{
  char			buf[2048], kb[64];
  int			s, tcpproto;
  struct protoent	*proto;
  fd_set		rfs, wfs, efs;
  struct timeval	timeout, ti;
  struct sockaddr_in	sa;
  int			port = 23;
  int			imax, i, j, scanned_ports, x, opt;
  unsigned int		optsz;
  int			minport;
  unsigned char		ports_states[65536];
  grab_socket_t		sockets[GRAB_MAX_SOCK];
  int			open_sock_nb, open_sock_max, open_sock_max2;
  int			unfiltered_ports_nb, filtered_ports_nb, timeout_nb;
  int			open_ports_nb, closed_ports_nb;
  int			untested_ports_nb, total_ports_nb;
#ifdef STATISTICS
  double		rtt_sum[3], rtt_sum2[3];
  int			rtt_nb[3], rtt_max[3], rtt_min[3];
  static const char	*rtt_type[] = {"unfiltered", "open", "closed" };
#endif

  proto = getprotobyname("tcp");
  if (proto == NULL)
    {
      perror("tcp");
      return -1;
    }
  tcpproto = proto->p_proto;

  for (i = 0; i < sizeof(ports_states) / sizeof(*ports_states); i ++)
    ports_states[i] = GRAB_PORT_NOT_TESTED;
  scanned_ports = 0;
#ifdef STATISTICS
  for (i = 0; i < 3; i ++)
    {
      rtt_sum[i] = rtt_sum2[i] = 0.0;
      rtt_max[i] = rtt_nb[i] = 0;
      rtt_min[i] = 1 << (sizeof(int) * 4) - 1;
    }
#endif

  {
    char	*p, *q;
    int		po1, po2;
    p = (char*)portrange;
    untested_ports_nb = 0;

    if (p == NULL || *p == '\0' || strcmp(p, "default") == 0)
      {
	int	last_num = 0;
	unsigned short * nums = (unsigned short*)get_tcp_svcs(&last_num);

	if (nums == NULL)
	  {
	    fprintf(stderr, "Cannot get list of default services\n");
	    return -1;
	  }
	for (i = 0; i < last_num; i ++)
	    {
	      ports_states[nums[i]] = GRAB_PORT_UNKNOWN;
	      untested_ports_nb ++;
	    }
	efree(&nums);
      }
    else
      while (*p != '\0')
	{
	  while (*p == ',')
	    p ++;

	  if (*p == '-')
	    {
	      po1 = 1;
	      q = p + 1;
	      po2 = strtol(q, &p, 10);
	      if (q == p)
		{
		  fprintf(stderr, "Cannot parse '%s'\n", p);
		  return -1;
		}
	    }
	  else
	    {
	      po1 = strtol(p, &q, 10);
	      if (q == p)
		{
		  fprintf(stderr, "Cannot parse '%s'\n", p);
		  return -1;
		}
	      if (*q == ',')
		{
		  p = q + 1;
		  po2 = po1;
		}
	      else if (*q == '\0')
		{
		  p = q;
		  po2 = po1;
		}
	      else if (*q == '-')
		{
		  if (q[1] == '\0')
		    {
		      po2 = 65535;
		      p = q+1;
		    }
		  else
		    {
		      po2 = strtol(q+1, &p, 10);
		      if (q+1 == p)
			{
			  fprintf(stderr, "Cannot parse '%s'\n", p);
			  return -1;
			}
		    }
		}
	    }
	  for (i = po1; i <= po2; i ++)
	    {
	      ports_states[i] = GRAB_PORT_UNKNOWN;
	      untested_ports_nb ++;
	    }
	}
  }

  for (i = 0; i < max_cnx; i ++)
    {
      sockets[i].state = GRAB_SOCKET_UNUSED;
      sockets[i].fd = -1;
    }

  open_sock_nb = 0; 
  open_sock_max = min_cnx; open_sock_max2 = max_cnx;

  open_ports_nb = closed_ports_nb = filtered_ports_nb = unfiltered_ports_nb = 0;

  minport = 1;
  while (scanned_ports < 65535)
    {
      total_ports_nb = unfiltered_ports_nb + filtered_ports_nb + untested_ports_nb;
      comm_send_status(globals, arg_get_value(hostinfos, "NAME"),"portscan", 
		       unfiltered_ports_nb + filtered_ports_nb, 
		       total_ports_nb);
#ifdef DEBUG
      fprintf(stderr, "%d / %d = %02d%% - %d ports remaining\n", 
	      unfiltered_ports_nb + filtered_ports_nb,
	      total_ports_nb,
	      (unfiltered_ports_nb + filtered_ports_nb) * 100 / 
	      (total_ports_nb > 0 ? total_ports_nb : 1),
	      untested_ports_nb);
#endif
      while (open_sock_nb < open_sock_max)
	{
	  for (port = minport; port <= 65535 && ports_states[port] != GRAB_PORT_UNKNOWN; port ++)
	    ;
	  if (port > 65535)
	    break;
	  minport = port;

	  ports_states[port] = GRAB_PORT_TESTING;
#ifdef DEBUG
	  fprintf(stderr, "Trying %d\n", port);
#endif
	  s = socket(PF_INET, SOCK_STREAM, tcpproto);
	  if (s < 0)
	    {
	      if (errno == ENFILE) /* File table overflow */
		{
		  open_sock_max = open_sock_max2 = open_sock_nb / 2 - 1;
		  /* NB: if open_sock_max2 < 0, the scanner aborts */
#ifdef DEBUG
		  /* DEBUG: otherwise, we print a less frigthtening message */
		  perror("socket");
#endif
		  fprintf(stderr, "Reducing the number of maximum open connections to %d [ENFILE]\n", open_sock_max);
		  continue;
		}
	      else if (errno == EMFILE)	/* Too many open files */
		{
		  x = open_sock_nb  / 16;	/* 6.25% */
		  open_sock_max = open_sock_max2 = 
		    open_sock_nb - (x > 0 ? x : 1);
		  /* NB: if open_sock_max2 < 0, the scanner aborts */
#ifdef DEBUG
		  /* DEBUG: otherwise, we print a less frigthtening message */
		  perror("socket");
#endif
		  fprintf(stderr, "Reducing the number of maximum open connections to %d [EMFILE]\n", open_sock_max);
		  continue;
		}
	      else
		{
		  perror("socket");
		  return -1;
		}
	    }

	  if (fcntl(s, F_SETFL, O_NONBLOCK) < 0)
	    {
	      perror("fcntl");
	      return -1;
	    }

#ifdef SO_LINGER
	  {
	    struct linger	l;

	    l.l_onoff = 0; l.l_linger = 0;
	    if (setsockopt(s, SOL_SOCKET,  SO_LINGER,  &l, sizeof(l)) < 0)
	      perror("setsockopt(SO_LINGER)");
	  }
#endif

	  sa.sin_addr = *pia;
	  sa.sin_family = AF_INET;
	  sa.sin_port = htons(port);

	  if (connect(s, (struct sockaddr*)&sa, sizeof(sa)) < 0)
	    {
	      switch (errno)
		{
		case EINPROGRESS:
		case EALREADY:
		  sockets[open_sock_nb].fd = s;
		  sockets[open_sock_nb].port = port;
		  sockets[open_sock_nb].state = GRAB_SOCKET_OPENING;
		  (void) gettimeofday(&sockets[open_sock_nb].tictac, NULL);
		  open_sock_nb ++;
		  break;
		  
		case EAGAIN:
		  x = open_sock_nb  / 16;	/* 6.25% */
		  open_sock_max = open_sock_max2 = 
		    open_sock_nb - (x > 0 ? x : 1);
		  /* If open_sock_max2 < 0, the scanner aborts */
		  fprintf(stderr, "Reducing the number of maximum open connections to %d [EAGAIN]\n", open_sock_max);
		  continue;

		case ECONNREFUSED:
		  ports_states[port] = GRAB_PORT_CLOSED;
		  my_socket_close(s);
		  unfiltered_ports_nb ++;
		  closed_ports_nb ++;
		  untested_ports_nb --;
		  continue;
		  
		case ENETUNREACH:
		  ports_states[port] = GRAB_PORT_FILTERED;
		  my_socket_close(s);
		  filtered_ports_nb ++;
		  untested_ports_nb --;
		  continue;

		default:
		  perror("connect");
		  return -1;
		}
	    }
	  else			/* This shoud not happen! */
	    {
	      sockets[open_sock_nb].fd = s;
	      sockets[open_sock_nb].port = port;
	      sockets[open_sock_nb].state = GRAB_SOCKET_OPEN;
	      (void) gettimeofday(&sockets[open_sock_nb].tictac, NULL);
	      open_sock_nb ++;
	      ports_states[port] = GRAB_PORT_OPEN;
	      unfiltered_ports_nb ++;
	      open_ports_nb ++;
	      untested_ports_nb --;
	      scanner_add_port(desc, port, "tcp");
	    }
	}

      if (open_sock_max2 <= 0)	/* file table is full */
	return -1;

      if (open_sock_nb == 0)
	{
	  if (untested_ports_nb > 0)
	    {
#if 1
	      fprintf(stderr, "No more open socket?\n");
#endif
	      return -1;
	    }
	  else
	    goto end;
	}

      FD_ZERO(&rfs); FD_ZERO(&wfs); FD_ZERO(&efs);
      imax = -1;
      for (i = 0; i < open_sock_nb; i ++)
	{
	  if (sockets[i].fd >= 0)
	    {
	      switch (sockets[i].state)
		{
		case GRAB_SOCKET_OPEN:
		  FD_SET(sockets[i].fd, &rfs);
		  break;
		case GRAB_SOCKET_OPENING:
		  FD_SET(sockets[i].fd, &wfs);
		  break;
		default:
#if 1
		  fprintf(stderr, "Bad status %d - s=%d\n", 
			  sockets[i].state, sockets[i].fd);
#endif
		  break;
		}
	      if (sockets[i].fd > imax)
		imax = sockets[i].fd;
	    }
	}

      if (imax < 0)
	{
	  if (untested_ports_nb > 0)
	    {
#if 1
	      fprintf(stderr, "No socket! %d ports remaining\n", untested_ports_nb);
#endif
	      return -1;
	    }
	  else
	    {
#ifdef DEBUG
	      fprintf(stderr, "No socket! %d ports remaining\n", untested_ports_nb);
#endif
	      goto end;
	    }
	}

      timeout.tv_sec = read_timeout; /* * 2 ? */
      /*
       * Some randomness so that not all scanners fire at the same time
       * when several firewalled machines are scanned in parallel.
       */
      timeout.tv_usec = (unsigned)(lrand48() & 0x7FFFFFFF) % 1000000;
      i = 0;
      do
	x = select(imax + 1, &rfs, &wfs, NULL, &timeout);
      while (i ++ < 10 && x < 0 && errno == EINTR);

      if (x < 0)
	{
	  perror("select");
	  return -1;
	}
      else if (x == 0)		/* timeout */
	{
#ifdef DEBUG
	  fprintf(stderr, "select: timeout on all (%d) sockets!\n", imax - 1);
#endif
	  for (i = 0; i < open_sock_nb; i ++)
	    {
	      if (sockets[i].fd > 0)
		{
		  my_socket_close(sockets[i].fd);
		  sockets[i].fd = -1;
		  if (sockets[i].state == GRAB_SOCKET_OPENING)
		    {
		      ports_states[sockets[i].port] = GRAB_PORT_FILTERED;
		      filtered_ports_nb ++;
		      untested_ports_nb --;
#ifdef BLACKLIST_FILTERED_PORTS
		      remember_filtered_port(desc, sockets[i].port);
#endif
		    }
		  
		}
	      sockets[i].state = GRAB_SOCKET_UNUSED;
	    }
	}
      else			/* something to do */
	{
	  (void) gettimeofday(&ti, NULL);
	  for (i = 0; i < open_sock_nb; i ++)
	    {
	      if (sockets[i].fd > 0)
		if (FD_ISSET(sockets[i].fd, &wfs))
		  {
		    opt = 0; optsz = sizeof(opt);
		    if (getsockopt(sockets[i].fd, SOL_SOCKET, SO_ERROR, &opt, &optsz) < 0)
		      {
			perror("getsockopt");
			return -1;
		      }

		    x = DIFFTVu(ti, sockets[i].tictac);
#if defined DEBUG || defined STATISTICS
		    fprintf(stderr, "RTT to %s:%d: %g s\n", 
			    inet_ntoa(*pia), sockets[i].port, x / 1e6);
#endif
		    if (opt != 0)
		      {
			errno = opt;
#ifdef DEBUG
			perror("select->getsockopt");
#endif
#ifdef STATISTICS
			rtt_nb[2] ++;
			rtt_sum[2] += (double)x;
			rtt_sum2[2] += (double)x * (double)x;
			if (x > rtt_max[2]) rtt_max[2] = x;
			if (x < rtt_min[2]) rtt_min[2] = x;
#endif
			ports_states[sockets[i].port] = GRAB_PORT_CLOSED;
			my_socket_close(sockets[i].fd);
			sockets[i].fd = -1;
			sockets[i].state = GRAB_SOCKET_UNUSED;
			unfiltered_ports_nb ++;
			closed_ports_nb ++;
			untested_ports_nb --;
#ifdef DISPLAY
			printf(">> %d: CLOSED\n", sockets[i].port);
#endif
		      }
		    else
		      {
			sockets[i].state = GRAB_SOCKET_OPEN;
#ifdef DISPLAY
			printf(">> %d: OPEN\n", sockets[i].port);
#endif
#ifdef STATISTICS
			rtt_nb[1] ++;
			rtt_sum[1] += (double)x;
			rtt_sum2[1] += (double)x * (double)x;
			if (x > rtt_max[1]) rtt_max[1] = x;
			if (x < rtt_min[1]) rtt_min[1] = x;
#endif
			unfiltered_ports_nb ++;
			open_ports_nb ++;
			untested_ports_nb --;
			ports_states[sockets[i].port] = GRAB_PORT_OPEN;
			scanner_add_port(desc, sockets[i].port, "tcp");
			snprintf(kb, sizeof(kb), "TCPScanner/CnxTime1000/%d", sockets[i].port);
			plug_set_key(desc, kb, ARG_INT, (void*)(x/1000));
			snprintf(kb, sizeof(kb), "TCPScanner/CnxTime/%d", sockets[i].port);
			plug_set_key(desc, kb, ARG_INT, (void*)((x + 500000) / 1000000));
			sockets[i].tictac = ti;
		      }
#ifdef STATISTICS
		    rtt_nb[0] ++;
		    rtt_sum[0] += (double)x;
		    rtt_sum2[0] += (double)x * (double)x;
		    if (x > rtt_max[0]) rtt_max[0] = x;
		    if (x < rtt_min[0]) rtt_min[0] = x;
#endif
		  }
		else if (FD_ISSET(sockets[i].fd, &rfs))
		  {
		    x = read(sockets[i].fd, buf, sizeof(buf)-1);
		    if (x > 0)
		      {
			char	buf2[sizeof(buf)*2+1];
			int y, flag = 0;

			for (y = 0; y < x; y ++)
			  {
			    sprintf(buf2 + 2*y, "%02x", (unsigned char) buf[y]);
			    if (buf[y] == '\0') flag = 1;
			  }
			buf2[2 * x - 1] = '\0';
			if (flag)
			  {
			    snprintf(kb, sizeof(kb),  "BannerHex/%d", sockets[i].port);
			    plug_set_key(desc, kb, ARG_STRING, buf2);
			  }

			buf[x] = '\0';
			snprintf(kb, sizeof(kb), "Banner/%d", sockets[i].port);
			plug_set_key(desc, kb, ARG_STRING, buf);
#ifdef DISPLAY
			printf("Banner for port %d: %s\n", sockets[i].port, buf);
#endif
			x = DIFFTVu(ti, sockets[i].tictac) / 1000;
			snprintf(kb, sizeof(kb), "TCPScanner/RwTime1000/%d", sockets[i].port);
			plug_set_key(desc, kb, ARG_INT, (void*) x);
			snprintf(kb, sizeof(kb), "TCPScanner/RwTime/%d", sockets[i].port);
			plug_set_key(desc, kb, ARG_INT, (void*)((x + 500) / 1000));
		      }
		    else
#ifdef DEBUG
		      perror("read");
#endif
		    my_socket_close(sockets[i].fd);
		    sockets[i].fd = -1;
		    sockets[i].state = GRAB_SOCKET_UNUSED;
		  }
	    }
	}

      (void) gettimeofday(&ti, NULL);
      timeout_nb = 0;
      for (i = 0; i < open_sock_nb; i ++)
	if (sockets[i].fd >= 0 && DIFFTV(ti, sockets[i].tictac) >= read_timeout)
	  {
	    switch(sockets[i].state)
	      {
	      case GRAB_SOCKET_OPEN:
#ifdef DISPLAY
		printf(">> %d: NO BANNER\n", sockets[i].port);
#endif
		timeout_nb ++;
		snprintf(kb, sizeof(kb), "/tmp/NoBanner/%d", sockets[i].port);
		plug_set_key(desc, kb, ARG_INT, (void *) 1);
		break;
	      case GRAB_SOCKET_OPENING:
#ifdef DISPLAY
		printf(">> %d: TIMEOUT\n", sockets[i].port);
#endif
		ports_states[sockets[i].port] = GRAB_PORT_FILTERED;
		filtered_ports_nb ++;
		untested_ports_nb --;
#ifdef BLACKLIST_FILTERED_PORTS
		remember_filtered_port(desc, sockets[i].port);
#endif
		break;
	      }
	    my_socket_close(sockets[i].fd); sockets[i].fd = -1;
	    sockets[i].state = GRAB_SOCKET_UNUSED;
	  }

      x = open_sock_max;
      open_sock_max += filtered_ports_nb;
      open_sock_max += timeout_nb;
      if (open_sock_max > open_sock_max2)
	open_sock_max = open_sock_max2;
#ifdef DEBUG
      if (x != open_sock_max)
	fprintf(stderr, "open_sock_max=%d\n", open_sock_max);
#endif
      for (i = 0; i < open_sock_nb; )
	if (sockets[i].state == GRAB_SOCKET_UNUSED || sockets[i].fd < 0)
	  {
	    for (j = i +1;  
		 j < open_sock_nb && (sockets[j].state == GRAB_SOCKET_UNUSED || sockets[j].fd < 0);
		 j ++)
	      ;
	    if (j < open_sock_nb)
	      memmove(sockets+i, sockets+j, sizeof(*sockets) * (max_cnx - j));
	    open_sock_nb -= j - i;
	  }
	else
	  i ++;
    }

 end:
#ifdef STATISTICS
  for (i = 0; i < 3; i ++)
    if (rtt_nb[i] > 0)
      {
	char	*key, rep[160];
	double	mean, sd = -1.0, emax = -1.0;

	/* Convert from micro-seconds to seconds */
	rtt_sum[i] /= 1e6; rtt_sum2[i] /= 1e12;

	mean = rtt_sum[i] / rtt_nb[i];
#if 0
	snprintf(rep, sizeof(rep), "%6g", mean);
	plug_set_key(desc, "TCPScanner/MeanRTT", ARG_STRING, rep);
	x = floor(mean * 1000 + 0.5);
	plug_set_key(desc, "TCPScanner/MeanRTT1000", ARG_INT, (void*) x);
	/* rtt_max is integer (uS) */
	plug_set_key(desc, "TCPScanner/MaxRTT1000", ARG_INT,
		     (void*)((rtt_max[i] + 500)/1000));
	snprintf(rep, sizeof(rep), "%6g", (rtt_max[i] + 500000) / 1000000);
	plug_set_key(desc, "TCPScanner/MaxRTT", ARG_STRING, rep);
#endif
	if (rtt_nb[i] > 1)
	  {
	    sd = sqrt((rtt_sum2[i] / rtt_nb[i] - mean * mean) * rtt_nb[i] / (rtt_nb[i] - 1));
	    emax = mean + 3 * sd;
#if 0
	    snprintf(rep, sizeof(rep), "%6g", sd);
	    plug_set_key(desc, "TCPScanner/SDRTT", ARG_STRING, rep);
	    x = floor(sd * 1000 + 0.5);
	    plug_set_key(desc, "TCPScanner/SDRTT1000", ARG_INT, (void*)x);
	    snprintf(rep, sizeof(rep), "%6g", emax);
	    plug_set_key(desc, "TCPScanner/EstimatedMaxRTT", ARG_STRING, rep);
	    x = floor(emax * 1000 + 0.5);
	    plug_set_key(desc, "TCPScanner/EstimatedMaxRTT1000", ARG_INT, (void*)x);
#endif
	  }
	fprintf(stderr, "Mean RTT to %s = %g - [%g, %g] - SD = %g - +3SD = %g [%d %s ports]\n", 
		inet_ntoa(*pia), mean, 
		rtt_min[i] / 1e6, rtt_max[i] / 1e6,
		sd, emax, rtt_nb[i], rtt_type[i]);
      }
#endif
#ifdef DISPLAY
  fprintf(stderr, "On %s: open_ports_nb=%d / closed_ports_nb = %d / filtered_ports_nb=%d\n", inet_ntoa(*pia), open_ports_nb, closed_ports_nb, filtered_ports_nb);
#endif
  plug_set_key(desc, "TCPScanner/OpenPortsNb", ARG_INT, (void*)open_ports_nb);
  plug_set_key(desc, "TCPScanner/ClosedPortsNb", ARG_INT, (void*)closed_ports_nb);
  plug_set_key(desc, "TCPScanner/FilteredPortsNb", ARG_INT, (void*)filtered_ports_nb);
#if 1
  if (total_ports_nb >= 65535)
    plug_set_key(desc, "Host/full_scan", ARG_INT, (void*) 1);
#else
  plug_set_key(desc, "Host/num_ports_scanned", ARG_INT, (void*)total_ports_nb);
#endif
  return 0;
}

PlugExport int plugin_run(struct arglist * desc)
{
  struct arglist * globals = arg_get_value(desc, "globals");
  struct arglist * preferences = arg_get_value(desc, "preferences");
  struct arglist * hostinfos = arg_get_value(desc, "HOSTNAME");
  char * port_range = arg_get_value(preferences, "port_range");
  char * p;
  struct in_addr *p_addr;
  int	timeout = 0, max_cnx, min_cnx, safe_checks = 0, x;


  p = arg_get_value(preferences, "safe_checks");
  if (p != NULL && strcmp(p, "yes") == 0) safe_checks = 1;

  p =  arg_get_value(preferences, "checks_read_timeout");
  if (p != NULL) timeout = atoi(p);
  if (timeout <= 0)
    timeout = 30;

  {
    int		max_host = 0, max_checks = 0, cur_sys_fd = 0, max_sys_fd = 0;
    struct rlimit	rlim;
    FILE	*fp;

    p = arg_get_value(preferences, "max_hosts");
    if (p != NULL) max_host = atoi(p);
    if (max_host <= 0) max_host = 15;

    p = arg_get_value(preferences, "max_checks");
    if (p != NULL) max_checks = atoi(p);
    if (max_checks <= 0) max_checks = 10;

    min_cnx = 8 * max_checks;
    if (safe_checks)
      max_cnx = 16 * max_checks;
    else
      max_cnx = 64 * max_checks;

    if (max_sys_fd <= 0)
      {
	if ( find_in_path("sysctl", 0) != NULL )
		fp = popen("sysctl fs.file-nr", "r");
  	else
		fp = NULL;

	if (fp != NULL)
	  {
	    if (fscanf(fp, "%*s = %*d %d %d", &cur_sys_fd, &max_sys_fd) == 2)
	      max_sys_fd -= cur_sys_fd;
	    else
	      max_sys_fd = 0;	    
	    pclose(fp);
	  }
      }
    if (max_sys_fd <= 0)
      {
	if ( find_in_path("sysctl", 0) )
		fp = popen("sysctl fs.file-max", "r");
	else
		fp = NULL;

	if (fp != NULL)
	  {
	    fscanf(fp, "%*s = %d", &max_sys_fd);
	    pclose(fp);
	  }
      }

    if (max_sys_fd <= 0)
      {
	if ( find_in_path("sysctl", 0) )
		fp = popen("sysctl kern.maxfiles", "r");
 	else
		fp = NULL;

	if (fp != NULL)
	  {
	    fscanf(fp, "%*s = %d", &max_sys_fd);
	    pclose(fp);
	  }
      }
#ifdef DEBUG
    fprintf(stderr, "max_sys_fd = %d\n", max_sys_fd);
#endif
    if (max_sys_fd <= 0) max_sys_fd = 16384; /* reasonable default */
    /* Let's leave at least 1024 FD for other processes */
    if (max_sys_fd < 1024)
      x = GRAB_MIN_SOCK;
    else
      {
	max_sys_fd -= 1024;
	x = max_sys_fd / max_host;
      }
    if (max_cnx > x) max_cnx = x;
#if 0
    fprintf(stderr, "min_cnx = %d ; max_cnx = %d\n", min_cnx, max_cnx);
#endif
    if (max_cnx > GRAB_MAX_SOCK) max_cnx = GRAB_MAX_SOCK;
    if (max_cnx < GRAB_MIN_SOCK) max_cnx = GRAB_MIN_SOCK;

    if (safe_checks && max_cnx > GRAB_MAX_SOCK_SAFE)
      max_cnx = GRAB_MAX_SOCK_SAFE;

    if (getrlimit(RLIMIT_NOFILE, &rlim) < 0)
      perror("getrlimit(RLIMIT_NOFILE)");
    else
      {
	/* value = one greater than the maximum  file  descriptor number */
	if (rlim.rlim_cur != RLIM_INFINITY && max_cnx >= rlim.rlim_cur)
	  max_cnx = rlim.rlim_cur - 1;
      }
    x = max_cnx / 2;
    if (min_cnx > x) min_cnx = x > 0 ? x : 1;
#ifdef DEBUG
    fprintf(stderr, "min_cnx = %d ; max_cnx = %d\n", min_cnx, max_cnx);
#endif
  }
  
  p_addr = arg_get_value(hostinfos, "IP");
  if( p_addr == NULL )
    return -1;
  if (banner_grab(p_addr, port_range, timeout, min_cnx, max_cnx, globals, desc, hostinfos) < 0)
    return -1;
  comm_send_status(globals, arg_get_value(hostinfos, "NAME"),"portscan", 65535, 65535);
  plug_set_key(desc, "Host/scanned", ARG_INT, (void*)1);
  plug_set_key(desc, "Host/scanners/nessus_tcp_scanner", ARG_INT, (void*)1);
  return 0;
}


