/* Copyright (c) 2000  Kevin Sullivan <nite@gis.net>
 *
 * Please refer to the COPYRIGHT file for more information.
 */

#include <stdio.h>
#include <string.h>
#include <ncurses.h>
#include <stdlib.h>
#include <ctype.h>
#include <sys/time.h>

#include "defines.h"
#include "colors.h"
#include "codes.h"
#include "scheck.h"
#include "nap.h"
#include "winio.h"
#include "sscr.h"
#include "lists.h"
#include "alias.h"

#ifdef MEMWATCH
  #include "memwatch.h"
#endif

extern char *conns[];
extern WINDOW *wchan;
extern int screen;
extern ssearch_t *search; /* search result list */
extern int srch;          /* 1 while search in progress */
extern info_t info;

/* curses windows and global state for search screen */
WINDOW *swin;
int cx;             /* horizontal shift of the status line */
int cy;             /* item number aligned with first line on screen */
int num;            /* item that the cursor is on */
int lk;             /* 0 if status line should be set to filename */
int max;            /* number of search results */
char *status;       /* status line */
int showconn;       /* show connection speed? */
int showping;       /* show ping? (Note this will only be shown, by
                       default, if pings are actually available) */
int showuser;       /* show user nickname? */
int showbrate;      /* show bitrate? */
int showlength;     /* show length? */
int showfreq;       /* show frequency? */
int showsize;       /* show size? */
char lastsort;      /* indicates what the most recent sort key was */
int usercols;       /* width of longest user name */

/* initialize result screen state. Note: this only needs to be called
   when a new list of search results was generated, not each time we
   switch to the result screen. This way, the state will be preserved
   while switching back and forth between screens */
void resetsscr(void)
{
  ssearch_t *cur;
  char *defaults;

  cx = 0;
  cy = 1;  /* note search items are numbered starting from 1 */
  num = 1;
  lastsort = 0;
  
  /* determine number of items in search list, and maximum size of
     username */
  
  max = 0;
  usercols = 4;
  list_forall(cur, search) {
    max++;
    if (strlen(cur->cmp) > usercols)
      usercols = strlen(cur->cmp);
  }

  showconn = 0;
  showping = 0;
  showuser = 0;
  showbrate = 0;
  showlength = 0;
  showfreq = 0;
  showsize = 0;

  defaults = getval("sdefaults");
  if (!defaults)
    defaults = "blsp";

  while (*defaults) {
    shandlechar(*defaults);
    defaults++;
  }

  /* turn off display of connection speed and/or pings if the
     information is not available */

  showconn = (showconn && search && search->conn != -1) ? 1 : 0;
  showping = (showping && search && search->ping > -2) ? 1 : 0;

  /* reset screen and cursor to first item (after possible sorting!) */
  cy = 1; 
  num = 1;

}

/* turn on result screen */
void searchscr(void)
{
  sock_t *sk;

  sk = findsock("input");
  if (sk) {
    sk->func = sinput;
  }

  swin = newwin("sscr", LINES, 0, 0, 0);
  wbkgdset(swin, COLOR_PAIR(CPW));
  keypad(swin, TRUE);
  werase(swin);
  drw(swin);
  
  status = strdup("Use F1 or 'q' to return to the main screen. 'h' for help.");
  lk = 1;

  plist();
}

/* turn off result screen */
void endsearchscr()
{
  delwin(swin);
  
  if (status)
    free(status);
}

/* print result screen */
void plist()
{
  int j, ln=0, min, sec, col, titlecols;
  ssearch_t *cur;
  float mb;
  char *t=NULL;
  
  if (screen != RESULT_SCREEN)
    return;
  
  if (srch)
  {
    werase(swin);
    mvwprintw(swin, LINES/2, (COLS-29)/2, "Waiting for search results...");
    wmove(swin, LINES-1, 0);
    wrefresh(swin);
    return;
  }
  
  if (!search)
  {
    werase(swin);
    mvwprintw(swin, LINES/2, (COLS-25)/2, "No search results to list");
    wmove(swin, LINES-1, 0);
    wrefresh(swin);
    return;
  }
  
  if (num > max)
    num = max;
  if (cy > max)
    cy = max-(LINES-3)+1;
  if (cy < 1)
    cy = 1;

  if (num >= (LINES-3)+cy)      /* screen follows cursor */
  {
    cy = num-(LINES-3)+1;
  }
  if (num < cy)
  {
    cy = num;
  }
  
  /* decide how many columns to use for filenames (=titlecols) */
  titlecols = COLS - 1;
  titlecols -= 8*showbrate;
  titlecols -= 9*showlength;
  titlecols -= (3+usercols)*showuser;
  titlecols -= 12*showconn;
  titlecols -= 7*showping;
  titlecols -= 8*showfreq;
  switch (showsize) {
  case 1:
    titlecols -= 9;
    break;
  case 2:
    titlecols -= 12;
    break;
  case 0: default:
    break;
  }
  if (titlecols < 14)
    titlecols = 14;
  
  werase(swin);
  
  msprintf(&t, "Filename (total = %d)", max);
  if (strlen(t) > titlecols)
    t[titlecols]=0;
  mvwprintw(swin, 0, 0, "%s", t);
  wmove(swin, 1, 0);
  whline(swin, ACS_HLINE, COLS);

  col = titlecols;
  if (showbrate) {
    mvwprintw(swin, 0, col+3, "BRate");
    wmove(swin, 0, col+1);
    wvline(swin, ACS_VLINE, LINES-1);
    col += 8;
  }
  if (showlength) {
    mvwprintw(swin, 0, col+3, "Length");
    wmove(swin, 0, col+1);
    wvline(swin, ACS_VLINE, LINES-1);
    col += 9;
  }
  switch (showsize) {
  case 1:
    mvwprintw(swin, 0, col+3, "MBytes");
    wmove(swin, 0, col+1);
    wvline(swin, ACS_VLINE, LINES-1);
    col += 9;
    break;
  case 2:
    mvwprintw(swin, 0, col+3, "  Bytes");
    wmove(swin, 0, col+1);
    wvline(swin, ACS_VLINE, LINES-1);
    col += 12;
    break;
  case 0: default:
    break;
  }
  if (showfreq) {
    mvwprintw(swin, 0, col+3, "Freq");
    wmove(swin, 0, col+1);
    wvline(swin, ACS_VLINE, LINES-1);
    col += 8;
  }
  if (showuser) {
    mvwprintw(swin, 0, col+3, "User");
    wmove(swin, 0, col+1);
    wvline(swin, ACS_VLINE, LINES-1);
    col += 3+usercols;
  }
  if (showconn) {
    mvwprintw(swin, 0, col+3, "Speed");
    wmove(swin, 0, col+1);
    wvline(swin, ACS_VLINE, LINES-1);
    col += 12;
  }
  if (showping) {
    mvwprintw(swin, 0, col+3, "Ping");
    wmove(swin, 0, col+1);
    wvline(swin, ACS_VLINE, LINES-1);
    col += 7;
  }

  /* find cy'th element of the list (remember list macros count from
     0, but cy counts from 1 */
  list_nth(cur, search, cy-1);
  for (j=cy, ln=0; cur && ln < LINES-3; j++, ln++, cur=cur->next)
  {
    if (num == j)
    {
      wattron(swin, COLOR_PAIR(1)|A_BOLD);
      wmove(swin, ln+2, 0);
      whline(swin, ' ', COLS);
      if (!lk)
      {
	if (status)
	  free(status);
	status = strdup(quote(cur->fn));
      }
    }
    
    msprintf(&t, "%i) %s", j, cur->song);
    if (strlen(t) > titlecols)
      t[titlecols]=0;
    mvwprintw(swin, ln+2, 0, "%s", t);
    col = titlecols;
    
    if (showbrate) {
      mvwprintw(swin, ln+2, col+3, "%4i", cur->brate);
      if (num==j) {
	wmove(swin, ln+2, col+1);
	wvline(swin, ACS_VLINE, 1);
      }
      col+=8;
    }
    if (showlength) {
      min = cur->time/60;
      sec = cur->time%60;
      mvwprintw(swin, ln+2, col+3, "%3i:%02i", min, sec);
      if (num==j) {
	wmove(swin, ln+2, col+1);
	wvline(swin, ACS_VLINE, 1);
      }
      col+=9;
    }
    switch (showsize) {
    case 1:
      mb = ((float)cur->sz)/1048576.0;
      mvwprintw(swin, ln+2, col+3, "%6.02f", mb);
      if (num == j) {
	wmove(swin, ln+2, col+1);
	wvline(swin, ACS_VLINE, 1);
      }
      col += 9;
      break;
    case 2:
      mvwprintw(swin, ln+2, col+3, "%9d", cur->sz);
      if (num == j) {
	wmove(swin, ln+2, col+1);
	wvline(swin, ACS_VLINE, 1);
      }
      col += 12;
      break;
    case 0: default:
      break;
    }
    if (showfreq) {
      mvwprintw(swin, ln+2, col+3, "%5i", cur->freq);
      if (num==j) {
	wmove(swin, ln+2, col+1);
	wvline(swin, ACS_VLINE, 1);
      }
      col+=8;
    }
    if (showuser) {
      mvwprintw(swin, ln+2, col+3, "%s", cur->cmp);
      if (num==j) {
	wmove(swin, ln+2, col+1);
	wvline(swin, ACS_VLINE, 1);
      }
      col+=3+usercols;
    }
    if (showconn) {
      mvwprintw(swin, ln+2, col+3, cur->conn != -1 ? conns[cur->conn] : "N/A");
      if (num==j) {
	wmove(swin, ln+2, col+1);
	wvline(swin, ACS_VLINE, 1);
      }
      col+=12;
    }
    if (showping) {
      mvwprintw(swin, ln+2, col+3, cur->ping>=0 ? "%4i" : " N/A", cur->ping);
      if (num==j) {
	wmove(swin, ln+2, col+1);
	wvline(swin, ACS_VLINE, 1);
      }
      col+=7;
    }
    
    if (num == j)
      wattroff(swin, COLOR_PAIR(1)|A_BOLD);
    
  } /* end for j */
  
  if (status)
  {
    if (cx+COLS > strlen(status))
      cx = strlen(status)-COLS;
    if (cx < 0)
      cx = 0;
  }
  
  wattron(swin, COLOR_PAIR(CPBR));
  if (!status)
    status = strdup("Status");
  wmove(swin, LINES-1, 0);
  whline(swin, ' ', COLS);
  waddstr(swin, status+cx);
  wattroff(swin, COLOR_PAIR(CPBR));
  
  drw(swin);

  if (t)
    free(t);
}

/* handle input on result screen */
int sinput(WINDOW *win, sock_t *m)
{
  chtype cbuf = 0;
  WINDOW *w = swin;

  cbuf = wgetch(w);

  shandlechar(cbuf);

  plist();

  return(1);
}

/* handle key stroke on result screen */
int shandlechar(chtype cbuf)
{
  sock_t *sk;
  ssearch_t *cur, *a, *b;
  int r;
  const char *help = 
    "HELP - use <right> and <left> to scroll this information. "
    "F1/q to return to main screen. <up> and <down> to move cursor. "
    "<return> to download item. "
    "l/b/m/f/u/s/p/a/n to toggle information displayed. "
    "N/D/L/B/M/F/U/S/P to sort. h for help. ";
  
  switch (cbuf) {

  case 'h': case 'H':
    free(status);
    status = strdup(help);
    cx = 0;
    lk = 1;
    break;

  case KEY_F(1):
  case 'q':
    switchtoscreen(MAIN_SCREEN);
    return(1);
    break;

  case KEY_UP:
    num--;
    if (num <= 0)
      num = 1;
    cx = lk = 0;
    break;

  case KEY_DOWN:
    num++;
    cx = lk = 0;
    break;
    
  case KEY_RIGHT:
    cx++;
    break;

  case KEY_LEFT:
    cx--;
    if (cx < 0)
      cx = 0;
    break;

  case KEY_PPAGE:                       /* PgUp */
  case 5:                               /* Ctrl-E */
  case 16:                              /* Ctrl-P */
    if (nvar_default("cursorfollowsscreen", 0)) {
      if (cy<=1) {
	free(status);
	status = strdup("Beginning of search results");
	lk = 1;
	cx = 0;
      } else {           /* one screen up, cursor follows screen */
	cy -= LINES-4;
	if (cy<1) {
	  cy = 1;
	}
	if (num >= (LINES-3)+cy) {
	  num = cy+(LINES-3)-1;
	}
	cx = lk = 0;
      }
    } else {             /* screen follows cursor policy */
      if (num <= 1) {
	free(status);
	status = strdup("Beginning of search results");
	lk = 1;
	cx = 0;
      } else {	
	num -= LINES-4;
	if (num <= 0)
	  num = 1;
	cx = lk = 0;
      }
    }
    break;

  case KEY_NPAGE:                       /* PgDn */
  case 4:                               /* Ctrl-D */
  case 14:                              /* Ctrl-N */
  case 22:                              /* Crtl-V */
    if (nvar_default("cursorfollowsscreen", 0)) {
      if (cy>=max-(LINES-3)+1) {
	free(status);
	status = strdup("End of search results");
	lk = 1;
	cx = 0;
      } else {           /* one screen down, cursor follows screen */
	cy += LINES-4;
	if (cy>max-(LINES-3)+1) {
	  cy = max-(LINES-3)+1;
	}
	if (num < cy) {
	  num = cy;
	}
	cx = lk = 0;
      }
    } else {              /* screen follows cursor policy */
      if (num>=max) {
	free(status);
	status = strdup("End of search results");
        lk = 1;
        cx = 0;
      } else {
	num += LINES-4;
	cx = lk = 0;
      }
    }
    break;

  case 12:                              /* Ctrl-L */
    clearok(swin, 1);
    wrefresh(swin);
    break;

  case '\n':
    if (search)
    {
      lk = 1;
      list_nth(cur, search, num-1);
      if (cur)
      {
	if (status)
	  free(status);
	status = NULL;
	sk = findsock("server");
	if (sk)
        {
	  r = requestdownload(sk->fd, cur);

	  switch(r) {
	  case 0:
	    msprintf(&status, "Getting %s", cur->song);
	    wp(wchan, "* Getting \"%s\" from %s\n", cur->song, cur->cmp);
	    break;
	  case 1: case 2:
	    msprintf(&status, "Queued %s", cur->song);
	    wp(wchan, "* Queued download of \"%s\" from %s\n", cur->song, cur->cmp);	    
	    break;
	  case -1:
	    msprintf(&status, "Already getting %s", cur->song);
	    break;
	  case -2:
	    msprintf(&status, "Already queued %s", cur->song);
	    break;
	  case -3:
	    msprintf(&status, "Can't get a file from yourself!");
	    break;
	  default:
	    break;
	  }
	}
	else /* !sk */
	{
	  msprintf(&status, "Not connected to server");
	}
      } 
      else /* !cur */
      {
	msprintf(&status, "Error: No matching item in search list??");
      }
    } /* end if(search) */
    break;

  case 'l':
    showlength = 1-showlength;
    break;

  case 'b':
    showbrate = 1-showbrate;
    break;

  case 'm':
    showsize = (showsize + 1) % 3;
    break;

  case 'f':
    showfreq = 1-showfreq;
    break;

  case 'u':
    showuser = 1-showuser;
    break;

  case 's':
    showconn = 1-showconn;
    break;

  case 'p':
    showping = 1-showping;
    break;

  case 'a':
    showlength = showbrate = showfreq = 1;
    showconn = showping = showuser = 1;
    showsize = showsize ? showsize : 1;
    break;

  case 'n':
    showlength = showbrate = showsize = showfreq = 0;
    showconn = showping = showuser = 0;
    break;

  case 'N':  /* various ways to sort */
  case 'D':
  case 'L':
  case 'B':
  case 'M':
  case 'F':
  case 'U':
  case 'S':
  case 'P':
    /* remember which item the cursor was on */
    list_nth(cur, search, num-1);
    
    if (cbuf == lastsort) {                 /* reverse? */

      list_reverse(ssearch_t, search);

    } else {                                /* sort */
      lastsort = cbuf;

      /* this looks awful, but list_mergesort is a fairly large macro,
	 and expanding it 7 times would probably be a worse option */
      list_mergesort(ssearch_t, search, a, b,
		     (cbuf=='N' && strcasecmp(a->song, b->song) <= 0)
		     || (cbuf=='D' && strcasecmp(a->fn, b->fn) <= 0)
		     || (cbuf=='L' && a->time <= b->time)
		     || (cbuf=='B' && a->brate <= b->brate)
		     || (cbuf=='M' && a->sz <= b->sz)
		     || (cbuf=='F' && a->freq <= b->freq)
		     || (cbuf=='U' && strcasecmp(a->cmp, b->cmp) <= 0)
		     || (cbuf=='S' && a->conn <= b->conn)
		     || (cbuf=='P' && (b->ping == -1 || 
				       (a->ping != -1 && a->ping <= b->ping)))
		     );
    }
    /* put the cursor back where it was */
    if (cur) {
      list_index(search, num, a, a==cur);
      num++;
      if (num >= (LINES-3)+cy || num < cy) {
	cy = num-(LINES-3)/2;
	if (cy<1)
	  cy = 1;
      }
    }
    break;
    
  default:
    break;
  }

  return(1);
}

