/* Copyright (c) 2000  Kevin Sullivan <nite@gis.net>
 *
 * Please refer to the COPYRIGHT file for more information.
 */

#include <stdio.h>
#include <time.h>
#include <stdarg.h>
#include <termios.h>
#include <dirent.h>
#include <fcntl.h>
#include <sys/types.h>
#include <arpa/inet.h>
#ifndef MCURSES
  #include <ncurses.h>
#endif
#include <stdlib.h>
#include <signal.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>
#include <netinet/in.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <sys/time.h>
#include <netdb.h>
#include <getopt.h>

#include "defines.h"
#include "codes.h"
#include "colors.h"
#include "title.h"
#include "mp3s.h"
#include "scheck.h"
#include "cmds.h"
#include "timer.h"
#include "alias.h"
#include "nap.h"

#ifdef MCURSES
#include "wind.h"
#endif


extern file_t *up, *down;
extern alias_t *alhead;
extern out_nap_cmd_t out[];
extern scount_t scount;
extern unsigned char ircin, ircmode, wmode, finp;
extern int ircsock;
extern char *mnick;
extern unsigned char ups[];
extern void *cbk;

chans_t *chanl = NULL, *curchan = NULL, *recent = NULL;
struct link_t *hlist = NULL;
ssearch_t *search = NULL;
scrls_t *mscroll, *scur;
phead_t *thdr = NULL;
cmds_t *cmdl, *ccmd;
time_t cct = 0;
info_t info;
WINDOW *wchan=NULL, *winput, *sep, *head, *swin;
unsigned char bspace, discard, lpbrk=0, noprint=0, noresolv=0;
unsigned char *tdbuf = NULL;
char tpbuf[512];
int curx, tpos, ipcs[2], tind=0, curr = 0;
unsigned char dresize = 0, notop = 0, force = 0;
unsigned char dx[MAXDOWNLOADS], ux[MAXUPLOADS];

colortab_t colortab[] = {
#ifndef MCURSES
  { RED, CPR, 1 },
  { GREEN, CPG, 1 },
  { WHITE, CPW, 1 },
  { BLUE, CPB, 1 },
  { YELLOW, CPY, 1 },
  { MAGENTA, CPM, 1 },
  { CYAN, CPC, 1, },
  { BOLD, A_BOLD, 0 },
  { DARK, A_DIM, 0 },
  { BLINK, A_BLINK, 0 },
  { UNDERLINE, A_UNDERLINE, 0 },
#endif
  { NULL, -1, 0 }
};


void ctrlc(int dummy)
{
  doquit();
}

void doquit()
{
  scur = NULL;
  if (wchan)
  {
    dscr(wchan);
    drw(wchan);
  }
  if (ircin)
  {
    ssock(ircsock, "QUIT :Leaving\n");
    close(ircsock);
    ircsock = 0;
  }
  endwin();
  if (nvar("savechannels") == 1)
    savechans(chanl);
  if (info.log)
    fclose(info.log);
  if (info.xfer)
    fclose(info.xfer);
  exit(1);
}

void tresize(int dummy)
{
  dresize = 1;
}

void sigint(int dummy)
{
  if ((time(NULL)-cct) <= 1)
    doquit();
  
  lpbrk = 1;
  cct = time(NULL);
}

void dochecks()
{
  if (dresize)
  {
    dresize = 0;
    resize();
  }
}

#ifndef MCURSES

void resize()
{
  sock_t *t;
  cmds_t *cur;
  struct winsize ws;

  memset(&ws, 0, sizeof(ws));
  ioctl(fileno(stdin), TIOCGWINSZ, &ws);
  
  resizeterm(ws.ws_row, ws.ws_col);
  winput->_begy = LINES-1;
  winput->_begx = 0;
  sep->_begy = LINES-2;
  sep->_begx = 0;
  if (notop)
    wresize(wchan, LINES-2, COLS);
  else
    wresize(wchan, LINES-3, COLS);
  wresize(winput, 1, COLS);
  if (!notop)
    wresize(head, 1, COLS);
  wresize(sep, 1, COLS);
  drw(wchan);
  drw(sep);
  
  dstatus();
  
  werase(winput);
  t = findsock("input");
  if (!t)
  {
    drw(winput);
    return;
  }
  cur = (cmds_t *)t->d;
  if (cur->cmd)
  {
    indraw(cur->cmd, curx, winput);
    drw(winput);
  }
}

#endif

void wstats(WINDOW *win)
{
  wp(win, "stdscr to %i - %i\n", stdscr->_maxy, stdscr->_maxx);
  wp(win, "winput to %i - %i\n", winput->_maxy, winput->_maxx);
  wp(win, "sep to %i - %i\n", sep->_maxy, sep->_maxx);
  wp(win, "wchan to %i - %i\n", wchan->_maxy, wchan->_maxx);
  wp(win, "LINES/COLS to %i - %i\n", LINES, COLS);
  drw(win);
}

void drw(WINDOW *win)
{
  if (!win)
    return;
  
  if (finp && win == swin)
    wrefresh(win);
  else if (!finp)
    wrefresh(win);
}

void dstatus()
{
  int j, lg;
  char *t = NULL;
  user_t *cur;
  
#ifdef MCURSES
  if (info.cloak)
    mvwprintw(sep, 0, 0, "\e[0;37;44m (%s) [%i songs in %i libraries (%i gigs)] [nap v%s] [U/%i D/%i]", info.user, scount.songs, scount.libraries, scount.gigs, VERSION, fcnt(up), fcnt(down));
  else
    mvwprintw(sep, 0, 0, "\e[0;37;44m [%s] [%i songs in %i libraries (%i gigs)] [nap v%s] [U/%i D/%i]", info.user, scount.songs, scount.libraries, scount.gigs, VERSION, fcnt(up), fcnt(down));
  mvwaddstr(sep, 0, COLS+10, "\e[0m");
  drw(sep);
#else
  werase(sep);
  if (!notop)
    werase(head);
  for (j=0;j<COLS;j++)
  {
    waddch(sep, ' ');
    if (!notop)
      waddch(head, ' ');
  }
  
  if ((!ircmode && !curchan) || (curchan && (!curchan->q || curchan->q == 1)))
  {
    msprintf(&t, "[U/%i D/%i]", fcnt(up), fcnt(down));
    mvwprintw(sep, 0, COLS-(strlen(t)+1), t);
    free(t);
    if (info.cloak)
      mvwprintw(sep, 0, 0, " (%s) [%lu songs in %lu libraries (%lu gigs)]", info.user, scount.songs, scount.libraries, scount.gigs);
    else
      mvwprintw(sep, 0, 0, " [%s] [%lu songs in %lu libraries (%lu gigs)]", info.user, scount.songs, scount.libraries, scount.gigs);
  }
  else
  {
    if (!curchan || !curchan->users)
      mvwprintw(sep, 0, 0, " [%s]", mnick);
    else if (curchan->flag)
    {
      t = strdup(curchan->nm);
      if (strlen(curchan->nm) > COLS/4)
        t[COLS/4] = 0;
      cur = finduser(curchan, mnick);
      if (cur->flag & F_OP)
        mvwprintw(sep, 0, 0, " [@%s] [%s +", mnick, t);
      else if (cur->flag & F_VOICE)
        mvwprintw(sep, 0, 0, " [+%s] [%s +", mnick, t);
      else
        mvwprintw(sep, 0, 0, " [%s] [%s +", mnick, t);
      if (curchan->flag & F_I)
        waddch(sep, 'i');
      if (curchan->flag & F_S)
        waddch(sep, 's');
      if (curchan->flag & F_P)
        waddch(sep, 'p');
      if (curchan->flag & F_T)
        waddch(sep, 't');
      if (curchan->flag & F_M)
        waddch(sep, 'm');
      if (curchan->flag & F_N)
        waddch(sep, 'n');
      if (curchan->flag & F_K)
        waddch(sep, 'k');
      if (curchan->flag & F_L)
        waddch(sep, 'l');
      if (curchan->flag & F_K || curchan->flag & F_L)
        waddch(sep, ' ');
      if (curchan->flag & F_K)
        wprintw(sep, "%s", curchan->key);
      if (curchan->flag & F_K && curchan->flag & F_L)
        waddch(sep, ' ');
      if (curchan->flag & F_L)
        wprintw(sep, "%i", (unsigned int)curchan->l);
      waddch(sep, ']');
      free(t);
      t = NULL;
    }
    else
    {
      t = strdup(curchan->nm);
      if (strlen(curchan->nm) > COLS/4)
        t[COLS/4] = 0;
      cur = finduser(curchan, mnick);
      if (cur->flag & F_OP)
        mvwprintw(sep, 0, 0, " [@%s] [%s]", mnick, t);
      else if (cur->flag & F_VOICE)
        mvwprintw(sep, 0, 0, " [+%s] [%s]", mnick, t);
      else
        mvwprintw(sep, 0, 0, " [%s] [%s]", mnick, t);
      free(t);
      t = NULL;
    }
  }
  
  if (!notop)
  {
    lg = COLS-(strlen(VERSION)+8)-4;
    mvwprintw(head, 0, COLS-(strlen(VERSION)+8), "[nap v%s]", VERSION);
    if (curchan && curchan->q != 1 && curchan->topic)
    {
      t = (char *)malloc(lg+1);
      memset(t, 0, lg+1);
      if (strlen(curchan->topic) <= lg)
        tind = 0;
      if (tind && (((int)strlen(curchan->topic+tind)-lg) < 0 || tind > strlen(curchan->topic)))
      {
        drw(wchan);
        tind = strlen(curchan->topic)-lg;
      }
      if (strlen(curchan->topic) > lg && tind != (strlen(curchan->topic)-lg))
      {
        strncpy(t, curchan->topic+tind, lg-3);
        mvwprintw(head, 0, 0, " [%s...]", t);
      }
      else
      {
        strncpy(t, curchan->topic+tind, lg);
        mvwprintw(head, 0, 0, " [%s]", t);
      }
      free(t);
    }
    else if (curchan && curchan->q == 1)
      mvwprintw(head, 0, 0, " [Query (%s)]", curchan->nm);
    if (!finp)
      wnoutrefresh(head);
  }
  if (!finp)
  {
    wnoutrefresh(sep);
    doupdate();
  }
#endif
}

int readcfg(char *fn)
{
  FILE *f;
  char *t;
  
  f = fopen(fn, "r");
  if (f == NULL)
  {
    wp(NULL, "Error opening file\n");
    return(-1);
  }
  
  info.user = findent(f, "user");
  if (info.user == NULL)
  {
    wp(NULL, "Couldn't find user entry\n");
    fclose(f);
    return(-1);
  }
  info.pass = findent(f, "pass");
  if (info.pass == NULL)
  {
    wp(NULL, "Couldn't find pass entry\n");
    fclose(f);
    return(-1);
  }
  info.up = findent(f, "upload");
  if (info.up == NULL)
  {
    wp(NULL, "Couldn't find upload entry\n");
    fclose(f);
    return(-1);
  }
  info.down = findent(f, "download");
  if (info.down == NULL)
  {
    wp(NULL, "Couldn't find download entry\n");
    fclose(f);
    return(-1);
  }
  info.email = findent(f, "email");
  if (info.email == NULL)
  {
    wp(NULL, "Couldn't find email entry\n");
    fclose(f);
    return(-1);
  }
  t = findent(f, "dataport");
  if (t == NULL)
  {
    wp(NULL, "Couldn't find dataport entry\n");
    fclose(f);
    return(-1);
  }
  info.port = atoi(t);
  free(t);
  t = findent(f, "connection");
  if (t == NULL)
  {
    wp(NULL, "Couldn't find connection entry\n");
    fclose(f);
    return(-1);
  }
  info.conn = atoi(t);
  free(t);
  
  if (info.pass[0] == '\0')
  {
    wp(NULL, "Your password entry is blank\n");
    fclose(f);
    return(-1);
  }
  
  t = findent(f, "maxusers");
  if (t)
  {
    MAXUSERS = atoi(t);
    free(t);
  }
  else
    MAXUSERS = ups[info.conn];
  
  fclose(f);
  
  return(1);
}

void checkhotlist(int s, char *fn)
{
  FILE *f;
  char rd[64];
  hotlist_t *cur;
  
  if (hlist)
    dellist(&hlist);
  
  f = fopen(fn, "r");
  if (!f)
    return;
  
  while (!feof(f))
  {
    memset(rd, 0, sizeof(rd));
    fgets(rd, sizeof(rd), f);
    if (!strlen(rd))
      break;
    rd[strlen(rd)-1] = 0;
    cur = (hotlist_t *)malloc(sizeof(hotlist_t));
    cur->nm = strdup(rd);
    cur->conn = 0;
    cur->on = 0;
    addlink(&hlist, cur);
    sendpack(s, NOTIFY_CHECK, rd);
  }
  fclose(f);
}

int isshared(char *nm)
{
  char *buf = (char *)malloc(1024);
  FILE *f;
  int i;
  
  f = fopen(SFILE, "r");
  if (f == NULL)
  {
    free(buf);
    return(0);
  }
  
  for (;;)
  {
    memset(buf, 0, 1024);
    if (fgets(buf, 1024, f) == NULL)
    {
      free(buf);
      fclose(f);
      return(0);
    }
    for (i=1;buf[i]!='\"';i++)
      if (buf[i] == '\\')
        buf[i] = '/';
    if (!strncasecmp(buf+1, nm, strlen(nm)) && buf[1+strlen(nm)] == '\"')
    {
      free(buf);
      fclose(f);
      return(1);
    }
  }
  
  return(0);
}

char *findent(FILE *f, char *key)
{
  char *buf = (char *)malloc(256);
  char *pt, *ret;
  int i, j;
  
  fseek(f, 0, SEEK_SET);

  while (1)
  {
    memset(buf, 0, 256);
    if (fgets(buf, 256, f) == NULL)
      return(NULL);
    if (*buf == '#')
      continue;
    pt = (char *)malloc(strlen(buf)+1);
    memset(pt, 0, strlen(buf)+1);
    for (i=0;buf[i]!='=';i++)
    {
      if (!buf[i])
      {
        free(buf);
        free(pt);
        return(NULL);
      }
      pt[i] = buf[i];
    }
    pt[i] = '\0';
    if (!strcasecmp(pt, key))
    {
      ret = (char *)malloc(strlen(buf)-strlen(pt));
      memset(ret, 0, strlen(buf)-strlen(pt));
      for (j=0,i++;buf[i]!='\n'&&buf[i]!='\0'&&buf[i]!='\r';i++,j++)
        ret[j] = buf[i];
      ret[j] = '\0';
      free(pt);
      free(buf);
      return(ret);
    }
    free(pt);
  }
}

int msprintf(char **str, char *fmt, ...)
{
  va_list args;
  
  *str = (char *)realloc(*str, 4096);
  memset(*str, 0, 4096);
  
  va_start(args, fmt);
  vsprintf(*str, fmt, args);
  va_end(args);
  
  *str = (char *)realloc(*str, strlen(*str)+1);
  
  return(strlen(*str));
}

int strcnt(char *buf, char t)
{
  int i,r;
  
  for (i=0,r=0;buf[i];i++)
    if (buf[i] == t)
      r++;
  
  return(r);
}

void addlog(FILE *f, char *str, unsigned char rp)
{
  char t[1024];
  int i, j, k;
  struct tm *r;
  time_t ct = time(NULL);
  
  memset(t, 0, sizeof(t));
  
  if (rp)
  {
    for (i=0,j=0;str[i];i++)
    {
      if (str[i] == '\e')
      {
        for (k=0;str[i]&&str[i]!='m'&&k<6;k++,i++);
        continue;
      }
      t[j] = str[i];
      j++;
    }
  }
  else
    strcpy(t, str);
  
  r = localtime(&ct);
  
  fprintf(f, "[%i:%i] %s", r->tm_hour, r->tm_min, t);
  fflush(f);
}

int wp(WINDOW *win, char *fmt, ...)
{
  va_list args;
  char *str;
  int i;
  
  if (noprint == 2)
    return(0);
  
  str = (char *)malloc(4096);
  memset(str, 0, 4096);
  
  va_start(args, fmt);
  vsprintf(str, fmt, args);
  va_end(args);
  
  if (info.log)
    addlog(info.log, str, 1);
    
  if (win)
  {
    addscroll(win, str);
    dscr(win);
  }
  else
    printf(str);
  
  i = strlen(str);
  
  free(str);
  
  return(i);
}

scrls_t *scrend(scrls_t *t)
{
  scrls_t *cur, *cur1 = NULL;
  
  for (cur=t;cur!=NULL;cur=cur->next)
    cur1 = cur;
  
  return(cur1);
}

void addscroll(WINDOW *win, char *buf)
{
  int i, j, k = 0, m = 0, bf = 0;
  scrls_t *cur = scrend(mscroll), *cur1, *par;
  char col[10];
  
  if (cur == NULL)
  {
    mscroll = (scrls_t *)malloc(sizeof(scrls_t));
    mscroll->next = NULL;
    mscroll->prev = NULL;
    mscroll->d = 0;
    mscroll->chan = recent;
    mscroll->own = NULL;
    cur = mscroll;
    memset(cur->ln, 0, sizeof(cur->ln));
  }
  else if (cur->d)
  {
    cur1 = cur;
    cur = (scrls_t *)malloc(sizeof(scrls_t));
    cur->prev = cur1;
    cur->next = NULL;
    cur->d = 0;
    cur->chan = recent;
    cur->own = NULL;
    cur1->next = cur;
    memset(cur->ln, 0, sizeof(cur->ln));
  }
  
  if (!cur->d)
  {
    for (;cur->ln[k];k++)
    {
      if (cur->ln[k] == '\e')
      {
        if (!strncmp(cur->ln+k, BOLD, strlen(BOLD)))
          bf = 1;
        else if (!strncmp(cur->ln+k, WHITE, strlen(WHITE)))
          bf = 0;
        if (cur->ln[k+3] == 'm')
          m+=4;
        else
          m+=5;
      }
    }
  }
  
  par = cur;
  
  memset(col, 0, sizeof(col));
  
  for (i=0,j=strlen(buf);i<j;i++,k++)
  {
    if (buf[i] == '\r')
      i++;
    if (k >= (COLS+m-1) || buf[i] == '\n')
    {
      if (k == (COLS+m-1) && strchr(cur->ln, ' '))
      {
        for (;!isspace(cur->ln[k])&&k>=0;k--,i--);
        i++;
      }
      cur->ln[k] = '\0';
      strcat(cur->ln, WHITE);
      cur->d = 1;
      k = 0;
      m = 0;
      if (buf[i] == '\n')
        i++;
      if (i == j)
        break;
      cur1 = cur;
      cur = (scrls_t *)malloc(sizeof(scrls_t));
      cur->prev = cur1;
      cur->next = NULL;
      cur->d = 0;
      cur->chan = recent;
      cur1->next = cur;
      if (buf[i-1] == '\n')
      {
        cur->own = NULL;
        par = cur;
        bf = 0;
      }
      else
        cur->own = par;
      memset(cur->ln, 0, sizeof(cur->ln));
      if (*col)
      {
        if (bf && strcmp(col, BOLD))
          strcpy(cur->ln, BOLD);
        strcat(cur->ln, col);
        m = k = strlen(cur->ln);
      }
      else if (bf)
      {
        strcpy(cur->ln, BOLD);
        m = k = strlen(cur->ln);
      }
      if (cur->own)
        m-=TABSIZE;
    }
    if (buf[i] == '\e')
    {
      if (buf[i+3] == 'm')
      {
        if (!strncmp(buf+i, BOLD, strlen(BOLD)))
          bf = 1;
        else if (!strncmp(buf+i, WHITE, strlen(WHITE)))
          bf = 0;
        memset(col, 0, sizeof(col));
        strncpy(col, buf+i, 4);
        strncpy(cur->ln+k, buf+i, 4);
        m+=4;
        i+=3;
        k+=3;
      }
      else
      {
        memset(col, 0, sizeof(col));
        strncpy(col, buf+i, 5);
        strncpy(cur->ln+k, buf+i, 5);
        m+=5;
        i+=4;
        k+=4;
      }
    }
    else
      cur->ln[k] = buf[i];
  }
}

void dscr(WINDOW *win)
{
  scrls_t *cur;
  int i, j, k, m, at=0, t, c;
  unsigned char bf = 0, uf = 0, gf = 0;
  char ebuf[6];
  
  werase(win);
  wmove(win, 0, 0);
  
  if (scur == NULL)
    cur = scrend(mscroll);
  else
    cur = scur;
  
  if (cur == NULL)
    return;
  
  i = win->_maxy-1;
  
  if (cur->chan && (wmode && cur->chan != curchan))
    i++;
  
  while (i >= 0)
  {
    if (cur->prev == NULL)
      break;
    cur = cur->prev;
    if (cur->chan && (wmode && cur->chan != curchan))
      continue;
    i--;
  }
  
  c = win->_maxy;
  
  for (j=0,i=0;i<=c;cur=cur->next)
  {
    if (cur == NULL)
      break;
    if (cur->chan && (wmode && cur->chan != curchan))
      continue;
    if (cur->own)
      wmove(win, j, TABSIZE);
    else
    {
      bf = 0;
      uf = 0;
      gf = 0;
      wmove(win, j, 0);
    }
#ifndef MCURSES
    for (k=0;cur->ln[k];k++)
    {
      if (cur->ln[k] == '\e' && cur->ln[k+1] == '[' && isdigit(cur->ln[k+2]))
      {
        for (m=0;cur->ln[k]&&cur->ln[k]!='m';m++,k++)
          ebuf[m] = cur->ln[k];
        ebuf[m] = 'm';
        ebuf[m+1] = '\0';
        t = doesc(win, ebuf);
        if (t == COLOR_PAIR(CPW))
          at = 0;
        else
          at |= t;
      }
      else if (cur->ln[k] == 2)
      {
        if (!bf)
        {
          at |= doesc(win, BOLD);
          bf = 1;
        }
        else
        {
          at &= ~doesc(win, BOLD);
          bf = 0;
        }
      }
      else if (cur->ln[k] == 31)
      {
        if (!uf)
        {
          at |= doesc(win, UNDERLINE);
          uf = 1;
        }
        else
        {
          at &= ~doesc(win, UNDERLINE);
          uf = 0;
        }
      }
      else if (cur->ln[k] == 7)
      {
        if (!gf && i == c)
          beep();
        gf = 1;
      }
      else if (cur->ln[k] == 15)
        uf = bf = at = 0;
      else
        waddch(win, cur->ln[k]|at);
    }
#else
    waddstr(win, cur->ln);
#endif
    i++;
    j++;
  }
}

void dscroll(WINDOW *win, int n)
{
  int i;
  scrls_t *cur;
  
  if (scur == NULL)
    cur = scrend(mscroll);
  else
    cur = scur;
  
  if (!cur)
    return;
  
  if (n > 0)
  {
    i = n;
    if (cur->chan && (wmode && cur->chan != curchan))
      i++;
  
    while (i > 0)
    {
      if (cur->prev == NULL)
        break;
      cur = cur->prev;
      if (cur->chan && (wmode && cur->chan != curchan))
        continue;
      i--;
    }
  }
  else if (n < 0)
  {
    i = n;
    if (cur->chan && (wmode && cur->chan != curchan))
      i--;
  
    while (i < 0)
    {
      if (cur->next == NULL)
      {
        cur = NULL;
        break;
      }
      cur = cur->next;
      if (cur->chan && (wmode && cur->chan != curchan))
        continue;
      i++;
    }
  }
  else
    return;
  
  scur = cur;
  
  dscr(win);
  drw(win);
}

int doesc(WINDOW *win, char *in)
{
  int i;
  
  for (i=0;;i++)
  {
    if (!colortab[i].in)
      return(0);
    if (!strcmp(colortab[i].in, in))
      break;
  }
  
  if (colortab[i].c)
    return(COLOR_PAIR(colortab[i].pair));
  else
    return(colortab[i].pair);
}

int cmdcnt(cmds_t *t)
{
  cmds_t *cur;
  int i;
  
  for (i=0,cur=t;cur!=NULL;cur=cur->next,i++);
  
  return(i);
}

cmds_t *cmdend(cmds_t *t)
{
  cmds_t *cur, *cur1 = NULL;
  
  for (cur=t;cur!=NULL;cur=cur->next)
    cur1 = cur;
  
  return(cur1);
}

void phelp(char *nm)
{
  printf("Usage: %s [-B] [-b] [-m] [-r] [-f fn] [-t] [-l] [-x fn] [-g fn]\n", nm);
  printf("Switches:\n");
  printf("-B - Builds your library and then exits\n");
  printf("-b - Builds your library\n");
  printf("-m - Creates a new account from the information in your config file\n");
  printf("-r - Keeps trying to reconnect\n");
  printf("-f fn - Specifies the config file to use\n");
  printf("-t - Tells the client not to display the title bar\n");
  printf("-l - Attempts to use a terminal which is compatible with most systems\n");
  printf("-x fn - Tells the client to log all transfers to a specific filename\n");
  printf("-g fn - Tells the client to log everything to a specific filename\n");
  
  exit(1);
}

void dopts(int argc, char **argv, char **cfg, char **srv, int *b, int *m, int *r, int *d)
{
  int c;
  
  info.xfer = NULL;
  info.log = NULL;
  
  while ((c = getopt(argc, argv, "h?Bbtrlmf:s:d:x:g:")) != -1)
  {
    switch (c)
    {
      case 'd':
        *d = atoi(optarg);
        break;
      case 'B':
        *b = 2;
        break;
      case 'b':
        *b = 1;
        break;
      case 'm':
        *m = 1;
        break;
      case 'r':
        *r = 1;
        break;
      case 'f':
        *cfg = strdup(optarg);
        break;
      case 't':
        notop = 1;
        break;
      case 's':
        *srv = strdup(optarg);
        break;
      case 'l':
        force = 1;
        break;
      case 'x':
        info.xfer = fopen(optarg, "a");
        break;
      case 'g':
        info.log = fopen(optarg, "a");
        break;
      case '?':
      case 'h':
        phelp(*argv);
        break;
      default:
        wp(NULL, "Invalid options\n");
        exit(1);
    }
  }

  if (!(*cfg))
    *cfg = CONF;
  if (!(*srv))
    *srv = NULL;
  
  return;
}

char *glistn(char *t)
{
  char *r = strdup(t);
  int i;
  
  for (i=0;r[i];i++)
    r[i] = tolower(r[i]);
  
  r = (char *)realloc(r, strlen(r)+9);
  strcat(r, ".hotlist");
  
  return(r);
}

int main(int argc, char *argv[])
{
  int s, b=0, m=0, r=0, n, f, i, d=0;
  char *pt = NULL, *srv = NULL, *p;
  char *t = NULL;
  FILE *fl;

  /*  memset(&ux, 0, );*/


  for (i=0;i<9;i++)
    ux[i] = 0;
  for (i=0;i<MAXDOWNLOADS;i++)
    dx[i] = 0;
  
  mscroll = scur = NULL;
  cmdl = ccmd = NULL;
  curx = 0;
  wmode = 0;
  recent = NULL;
  
  dopts(argc, argv, &pt, &srv, &b, &m, &r, &d);
  if (!srv)
    srv = SERV;
  info.d = d;
  
  
  wp(NULL, "Reading config...\n");
  
  if (readcfg(pt) == -1)
    return(1);
  
  if (!strcmp(info.pass, "?"))
  {
    p = getpass("Enter your password: ");
    info.pass = strdup(p);
    memset(p, 0, strlen(p));
  }
  
  if (b)
  {
    wp(NULL, "Building list...\n");
    if (buildflist(SFILE, info.up) == -1)
      return(1);
    if (b == 2)
      return(1);
  }
  
  fl = fopen(SFILE, "r");
  if (!fl)
  {
    wp(NULL, "Error reading %s (try running \"nap -b\")\n", SFILE);
    return(1);
  }
  fclose(fl);
  
  if (!opendir(info.down))
  {
    wp(NULL, "Error opening your download directory (%s)\n", info.down);
    return(1);
  }
  
  if (info.conn < 0 || info.conn > 10)
    info.conn = 0;
  
  if (info.down[strlen(info.down)-1] == '/')
    msprintf(&t, "%s.tst", info.down);
  else
    msprintf(&t, "%s/.tst", info.down);
  f = open(t, O_CREAT);
  if (f == -1)
  {
    wp(NULL, "Error writing to your download directory (%s)\n", t);
    free(t);
    return(1);
  }
  unlink(t);
  free(t);
  t = NULL;
  
  for (;;)
  {
    if (!strchr(srv, ':'))
      wp(NULL, "Getting best host...\n");
    s = 2;
    while (s == 2)
      s = conn(srv, PORT);
    if (s == -1 && !r)
      return(1);
    else if (s == -1 && r)
    {
      if (noresolv)
      {
        if (srv == SERV)
          srv = SERV2;
        else if (srv == SERV2)
          srv = SERV3;
        else if (srv == SERV3)
          srv = SERV4;
        else if (srv == SERV4)
          srv = SERV5;
        else if (srv == SERV5)
          srv = SERV2;
      }
      close(s);
      continue;
    }
  
    if (m)
    {
      wp(NULL, "Creating account...\n");
      n = makeact(s, info.user, info.pass, info.port, CLIENT, info.conn, info.email);
      if (n == -1 && !r)
      {
        close(s);
        return(1);
      }
      else if (n == -1 && r)
      {
        close(s);
        continue;
      }
      else
        break;
      m = 0;
    }
    else
    {
      wp(NULL, "Logging in...\n");
      n = login(s, info.user, info.pass, info.port, CLIENT, info.conn, info.email);
      if (n == -1 && !r)
      {
        close(s);
        return(1);
      }
      else if (n == -1 && r)
      {
        close(s);
        continue;
      }
      else
        break;
    }
  }
  
  signal(SIGCHLD, SIG_IGN);
  signal(SIGPIPE, SIG_IGN);
  signal(SIGWINCH, tresize);
  signal(SIGINT, sigint);
#ifdef MCURSES
  signal(SIGCONT, tresize);
#endif

  pipe(ipcs);
  
  addsock(ipcs[0], "ipc", S_R, inipc);
  if (initfserv(info.port) == -1)
    info.port = 0;
/*  initfserve(info.port); */
  addsock(s, "server", S_R, inserv);
  addsock(0, "input", S_R, input);
/*  initssock(LPORT); */
  initwin(force);
  t = glistn(info.user);
  checkhotlist(s, t);
  free(t);
  loadaliases("alias");
  loadhandlers("handler");
  loadsets(".naprc");
  if (nvar("savechannels") == 1)
    loadchans(s);
#ifndef __CYGWIN32__
  libload(USERLIB);
#endif
/*  checknv(SHOST, SPORT, "VERSION"); */
  wp(wchan, "%s\n", title);
  drw(wchan);
  drw(winput);
  usleep(400000);
  sockfunc(wchan, winput);
  
  shutdown(s, 2);
  close(s);
  
  doquit();
  
  return(1);
}

void savechans(chans_t *cl)
{
  chans_t *cur;
  FILE *f;
  char *fn = NULL;
  
  msprintf(&fn, "%s.channels", info.user);
  
  f = fopen(fn, "w");
  if (!f)
    return;
  free(fn);
  
  for (cur=cl;cur;cur=cur->next)
    if (!cur->q)
      fprintf(f, "%s\n", cur->nm);
  
  fclose(f);
}

void loadchans(int s)
{
  char buf[256], *fn = NULL;
  FILE *f;
  
  msprintf(&fn, "%s.channels", info.user);
  
  f = fopen(fn, "r");
  if (!f)
    return;
  free(fn);
  
  while (!feof(f))
  {
    memset(buf, 0, sizeof(buf));
    fgets(buf, sizeof(buf), f);
    
    if (buf[strlen(buf)-1] == '\n')
      buf[strlen(buf)-1] = 0;
    if (strlen(buf))
      sendpack(s, F_JOIN, buf);
  }
  
  fclose(f);
}

void initssock(int port)
{
  struct sockaddr_in frm;
  int s, on = 1;
  
  s = socket(AF_INET, SOCK_DGRAM, 0);
  
  frm.sin_addr.s_addr = INADDR_ANY;
  frm.sin_port = htons(port);
  frm.sin_family = AF_INET;
  
  setsockopt(s, SOL_SOCKET, SO_REUSEADDR, &on, sizeof(on));
  
  bind(s, (struct sockaddr *)&frm, sizeof(frm));
  
  addsock(s, "ssock", S_R, inssock);
}

int inssock(WINDOW *win, sock_t *m)
{
  char *buf;
  struct sockaddr_in frm;
  int frmlen = sizeof(frm);
  
  buf = (char *)malloc(1024);
  memset(buf, 0, 1024);
  
  recvfrom(m->s, buf, 1024, 0, (struct sockaddr *)&frm, &frmlen);
  if (!strncmp(buf, "VERSION", strlen("VERSION")))
  {
    if (strcmp(strchr(buf, ' ')+1, VERSION))
    {
      wp(win, "%s* Attention: Version %s of nap is out!%s\n", BRIGHT(BLUE), strchr(buf, ' ')+1, WHITE);
      wp(win, "%s* Attention: Please visit http://www.gis.net/~nite/ to upgrade immediately%s\n", BRIGHT(BLUE), WHITE);
      drw(win);
    }
  }
  else if (!strncmp(buf, "MSG", strlen("MSG")))
  {
    wp(win, "%s* %s%s\n", BRIGHT(BLUE), strchr(buf, ' ')+1, WHITE);
    drw(win);
  }
  
  free(buf);
  
  return(1);
}

void checknv(char *host, int port, char *req)
{
  int s;
  struct sockaddr_in dst;
  
  s = socket(AF_INET, SOCK_DGRAM, 0);
  
  dst.sin_addr.s_addr = resolve(host);
  dst.sin_port = htons(port);
  dst.sin_family = AF_INET;
  
  sendto(s, req, strlen(req), 0, (struct sockaddr *)&dst, sizeof(dst));
}

int rebuild(int s, char *sd, char *path)
{
  FILE *f;
  char rd[1024];
  int i;
  
  f = fopen(sd, "r");
  if (!f)
    return(-1);
  
  while (!feof(f))
  {
    memset(rd, 0, sizeof(rd));
    fgets(rd, sizeof(rd), f);
    for (i=1;rd[i];i++)
    {
      if (rd[i] == '\"')
      {
        rd[i] = 0;
        break;
      }
      if (rd[i] == '/')
        rd[i] = '\\';
    }
    if (strlen(rd))
      sendpack(s, F_DFILE, "\"%s\"", rd+1);
  }
  
  if (buildflist(sd, path) == -1)
    return(-1);
  
  if (lfiles(s, sd) == -1)
    return(-1);
  
  return(1);
}

int buildflist(char *sd, char *path)
{
  FILE *f;
  char *sv, *t, *n = strdup(path);
  
  f = fopen(sd, "w");
  if (f == NULL)
  {
    wp(NULL, "Error opening \"%s\"\n", sd);
    free(n);
    return(-1);
  }
  
  sv = (char *)malloc(513);
  memset(sv, 0, 513);
  getcwd(sv, 512);
  if (sv == NULL)
  {
    wp(NULL, "Error getting the current working directory\n");
    free(n);
    return(-1);
  }
  
  t = strtok(n, ";");
  
  while (t)
  {
    dir(t, f);
    t = strtok(NULL, ";");
  }
  
  free(n);
  
  fclose(f);
  
  chdir(sv);
  free(sv);
  
  return(1);
}

void dir(char *nm, FILE *f)
{
  DIR *t;
  char *p=NULL;
  
  if (nm[strlen(nm)-1] != '/')
    msprintf(&p, "%s/", nm);
  else
    p = strdup(nm);
  
  t = opendir(p);
  if (!t)
    return;
  
  ddir(p, t, f);
  closedir(t);
  free(p);
}

void ddir(char *bpath, DIR *p, FILE *f)
{
  struct dirent *ls;
  DIR *t;
  char *nbpath=NULL;
  mhdr_t *m;
  
  while ((ls = readdir(p)))
  {
    msprintf(&nbpath, "%s%s", bpath, ls->d_name);
    t = opendir(nbpath);
    if (t)
    {
      closedir(t);
      if (strcmp(ls->d_name, "..") && strcmp(ls->d_name, "."))
      {
        free(nbpath);
        nbpath=NULL;
        msprintf(&nbpath, "%s%s/", bpath, ls->d_name);
        dir(nbpath, f);
      }
    }
    else
    {
      m = gethdr(nbpath);
      if (m)
      {
        fprintf(f, "\"%s\" %s-%i %i %i %i %i\n", nbpath, m->check, m->sz, m->sz, m->bitrate, m->freq, m->len);
        free(m);
      }
    }
    free(nbpath);
    nbpath=NULL;
  }
}

int lfiles(int s, char *fn)
{
  FILE *f;
  int i;
  char *buf = (char *)malloc(512), *pt;
  
  f = fopen(fn, "r");
  if (f == NULL)
  {
    wp(NULL, "Error reading %s (try running \"nap -b\")\n", fn);
    free(buf);
    return(-1);
  }
  
  while(1)
  {
    memset(buf, 0, 512);
    if (fgets(buf, 512, f) == NULL)
      break;
    pt = strchr(buf, '\r');
    if (pt == NULL)
      pt = strchr(buf, '\n');
    if (pt)
      *pt = 0;
    for (i=0;buf[i];i++)
      if (buf[i] == '/')
        buf[i] = '\\';
    if (sendpack(s, F_SFILE, buf) == -1)
    {
      wp(NULL, "Connection reset by peer\n");
      free(buf);
      return(-1);
    }
  }

  free(buf);
  fclose(f);
  
  return(1);
}  

long resolve(char *host)
{
  struct hostent *hst;
  long ret;
  
  if ((ret = inet_addr(host)) != -1)
    return(ret);
  hst = gethostbyname(host);
  if (!hst)
    return(-1);
  
  return(*(long *)hst->h_addr);
}

int conn(char *host, unsigned short port)
{
  int s;
  struct sockaddr_in dst;
  char *buf, serv[16];
  int p, i;
  
  if (!strchr(host, ':'))
  {
    dst.sin_port = htons(port);
    dst.sin_family = AF_INET;
    dst.sin_addr.s_addr = resolve(host);
    if (dst.sin_addr.s_addr == -1)
    {
      noresolv = 1;
      wp(wchan, "Error resolving host\n");
      return(-1);
    }
    
    s = socket(AF_INET, SOCK_STREAM, 0);
    if (s == -1)
    {
      wp(wchan, "Error creating socket\n");
      return(-1);
    }
    
    if (connect(s, (struct sockaddr *)&dst, sizeof(dst)) == -1)
    {
      wp(wchan, "Error connecting socket\n");
      close(s);
      return(-1);
    }
  
    rsock(s, &buf);
    shutdown(s, 2);
    close(s);
    if (!buf)
    {
      wp(wchan, "Error finding best host\n");
      return(-1);
    }
  }
  else
    buf = strdup(host);

  if (!strncasecmp(buf, "wait", 4))
  {
    sscanf(buf, "%s %i", serv, &p);
    wp(wchan, "Waiting for %i seconds\n", p);
    sleep(p);
    return(2);
  }
  else if (!strncasecmp(buf, "busy", 4))
  {
    wp(wchan, "Busy\n");
    return(2);
  }
  p = atoi(strchr(buf, ':')+1);
  for (i=0;buf[i]!=':';i++)
    serv[i] = buf[i];
  serv[i] = '\0';
  free(buf);
  
  dst.sin_addr.s_addr = resolve(serv);
  if (dst.sin_addr.s_addr == -1)
  {
    wp(wchan, "Error resolving host\n");
    return(-1);
  }
  dst.sin_port = htons(p);
  dst.sin_family = AF_INET;
  
  s = socket(AF_INET, SOCK_STREAM, 0);
  if (s == -1)
  {
    wp(wchan, "Error creating socket\n");
    return(-1);
  }
  
  wp(wchan, "Connecting...\n");
  
  if (connect(s, (struct sockaddr *)&dst, sizeof(dst)) == -1)
  {
    wp(wchan, "Error connecting socket\n");
    close(s);
    return(-1);
  }
  
  return(s);
}

int login(int s, char *user, char *pass, int data, char *client, int conn, char *email)
{
  char *t1=NULL;
  phead_t *rcv;
  int r;
  
  msprintf(&t1, "nap v%s", client);
  sendpack(s, F_LOGIN, "%s %s %i \"%s\" %i", user, pass, data, t1, conn);
  free(t1);
  t1 = NULL;
  if ((r = recvpack(s, &t1, &rcv)) == -1)
  {
    wp(wchan, "Error: Connection reset by peer\n");
    return(-1);
  }
  
  if (r == -2)
    while (recvpack(s, &t1, &rcv) == -2);
  
  if (!rcv)
    return(-1);
  
  if (rcv->op == F_LOGERROR)
  {
    wp(wchan, "Error: %s\n", t1);
    free(t1);
    free(rcv);
    return(-1);
  }
  else if (strcasecmp(t1, email))
    wp(wchan, "Email addresses did not match (%s and %s), proceeding anyway...\n", email, t1);
  
  free(t1);
  free(rcv);
  
  return(lfiles(s, SFILE));
}

int makeact(int s, char *user, char *pass, int data, char *client, int conn, char *email)
{
  char *t1=NULL;
  phead_t *rcv;
  int r;
  
  sendpack(s, F_MKUSER, user);
  if ((r = recvpack(s, &t1, &rcv)) == -1)
  {
    wp(NULL, "Error: Connection reset by peer\n");
    return(-1);
  }
  
  if (r == -2)
    while (recvpack(s, &t1, &rcv) == -2);
  
  if (rcv->op == F_UNTK)
  {
    wp(NULL, "Error: Username taken\n");
    free(t1);
    free(rcv);
    return(-1);
  }
  else if (rcv->op == F_UNBAD)
  {
    wp(NULL, "Error: Invalid username\n");
    free(t1);
    free(rcv);
    return(-1);
  }
  else if (rcv->op == F_UNOK)
    wp(NULL, "Registered username\n");
  else
  {
    wp(NULL, "Unknown op: 0x%x\n", rcv->op);
    free(t1);
    free(rcv);
    return(-1);
  }
  
  free(t1);
  free(rcv);
  t1 = NULL;
  
  msprintf(&t1, "nap v%s", client);
  sendpack(s, F_REG, "%s %s %i \"%s\" %i %s", user, pass, data, t1, conn, email);
  free(t1);
  t1 = NULL;
  if ((r = recvpack(s, &t1, &rcv)) == -1)
  {
    wp(NULL, "Error: Connection reset by peer\n");
    return(-1);
  }
  
  if (r == -2)
    while (recvpack(s, &t1, &rcv) == -2);
  
  if (rcv->op == F_LOGERROR)
  {
    wp(NULL, "Error: %s\n", t1);
    free(t1);
    free(rcv);
    return(-1);
  }
  
  free(t1);
  free(rcv);
  
  return(lfiles(s, SFILE));
}

void initwin(unsigned char f)
{
  struct termios ts;
  WINDOW *w;
  
  tcgetattr(0, &ts);
  bspace = ts.c_cc[VERASE];
  discard = 21;

#ifdef __CYGWIN32__
  setenv("TERMINFO", "./", 1);
  
  if (!newterm("cygwin", stdout, stdin))
  {
    fprintf(stderr, "Error opening terminal\n");
    exit(-1);
  }
  def_prog_mode();
  w = stdscr;
#else
  if (f)
  {
    if (!newterm("nxterm", stdout, stdin))
    {
      fprintf(stderr, "Error opening terminal\n");
      exit(-1);
    }
    def_prog_mode();
    w = stdscr;
  }
  else
    w = initscr();
#endif

/*  COLS = w->_maxx+1;
  LINES = w->_maxy+1; */
/*  dolc(); */
#ifndef MCURSES
  start_color();
  cbreak();
  noecho();
  
  init_pair(1, COLOR_WHITE, COLOR_BLUE);
  init_pair(CPR, COLOR_RED, COLOR_BLACK);
  init_pair(CPG, COLOR_GREEN, COLOR_BLACK);
  init_pair(CPW, COLOR_WHITE, COLOR_BLACK);
  init_pair(CPB, COLOR_BLUE, COLOR_BLACK);
  init_pair(CPY, COLOR_YELLOW, COLOR_BLACK);
  init_pair(CPM, COLOR_MAGENTA, COLOR_BLACK);
  init_pair(CPC, COLOR_CYAN, COLOR_BLACK);
  init_pair(CPBR, COLOR_WHITE, COLOR_RED);
  winput = newwin("input", 1, 0, LINES-1, 0);
#else
  notop = 1;
  winput = newwin("input", 1, COLS-2, LINES-1, 0);
#endif
  sep = newwin("sep", 1, 0, LINES-2, 0);
  if (!notop)
  {
    wchan = newwin("chan", LINES-3, 0, 1, 0);
    head = newwin("head", 1, 0, 0, 0);
  }
  else
    wchan = newwin("chan", LINES-2, 0, 0, 0);

#ifndef MCURSES
  nodelay(winput, TRUE);
  wattrset(sep, COLOR_PAIR(1));
  if (!notop)
    wattrset(head, COLOR_PAIR(1));
  idlok(wchan, FALSE);
  scrollok(wchan, TRUE);
  wbkgdset(winput, COLOR_PAIR(CPW));
  wbkgdset(wchan, COLOR_PAIR(CPW));
/*  bkgd(COLOR_PAIR(1)); */
  keypad(winput, TRUE);
#endif
  wmove(winput, 0, 0);
  drw(winput);
  
/*  tcgetattr(0, &ts);
  printf("%i\n", ts.c_iflag&ISTRIP);
  exit(1); */
  
  dstatus();
  drw(winput);
}

int inipc(WINDOW *win, sock_t *m)
{
  int s = m->s;
  char *buf;
  
  rsock(s, &buf);
  if (!buf)
    return(1);
  
  addscroll(win, buf);
  dscr(win);
  drw(win);
  
  free(buf);
  
  return(1);
}

int inserv(WINDOW *win, sock_t *m)
{
  int s = m->s, j, n, r;
  phead_t *pk;
  char *data;
  sock_t *t;
  cmds_t *cur;
  
  r = recvpack(s, &data, &pk);
  
  if (r == -1)
  {
    wp(win, "%s* Error reading from socket%s\n", RED, WHITE);
    drw(win);
    return(0);
  }
  else if (r == -2)
    return(1);
  
  for (j=0;data[j];j++)
    if (data[j] == 10)
    {
      strncpy(data+j, data+j+1, strlen(data+j+1));
      data[strlen(data)-1] = 0;
    }
    
  n = parsein(s, pk->op, data, win);
  noprint = 0;
  if (n == -1)
    return(0);
  else if (n == -2)
    return(-1);
  
  free(pk);
  free(data);
  
  dstatus();

  t = findsock("input");
  if (!t)
    return(1);
  if (!cmdl)
  {
    cmdl = (cmds_t *)malloc(sizeof(cmds_t));
    cur = cmdl;
    cur->next = NULL;
    cur->prev = NULL;
    memset(cur->cmd, 0, sizeof(cur->cmd));
    curx = 0;
    t->d = (void *)cur;
  }

  cur = (cmds_t *)t->d;
  indraw(cur->cmd, curx, winput);
  drw(winput);
  
  return(1);
}

unsigned char gchr(WINDOW *win)
{
  fd_set fs;
  struct timeval tv;
  unsigned char r;
  
  FD_ZERO(&fs);
  FD_SET(0, &fs);
  
  tv.tv_sec = 0;
  tv.tv_usec = 50000;
  
  if (!select(1, &fs, NULL, NULL, &tv))
    return(0);
  
  r = wgetch(win);
  if (r == '\e')
  {
    ungetch('\e');
    return(0);
  }
  
  return(r);
}

int input(WINDOW *win, sock_t *m)
{
  chtype cbuf = 0, ebuf = 0;
  cmds_t *cur = (cmds_t *)m->d, *cur1, *cur2, *cur3;
  sock_t *t;
  int i, j, s, cx, n;
  
  t = findsock("server");
  if (!t)
    s = -1;
  else
    s = t->s;
  
  if (!cmdl)
  {
    cmdl = (cmds_t *)malloc(sizeof(cmds_t));
    cur = cmdl;
    cur->next = NULL;
    cur->prev = NULL;
    memset(cur->cmd, 0, sizeof(cur->cmd));
    curx = 0;
  }
  i = curx;
  cx = i;

  ebuf = wgetch(winput);
  if (ebuf == '\n' || ebuf == '\r')
  {
    curr = 0;
    ebuf = n = 0;
    if (*cur->cmd == '/')
    {
      if (strlen(cur->cmd) > 1)
      {
        n = parseout(s, cur->cmd+1, win);
        dstatus();
      }
    }
    else
    {
      if (!curchan)
      {
        wp(win, "%s* Not on a channel%s\n", RED, WHITE);
        drw(win);
      }
      else if (curchan->q == 1)
      {
        sendpack(s, F_TELL, "%s %s", fxp(curchan->nm), cur->cmd);
        recent = curchan;
        wp(win, "%s* --> (%s%s%s)%s %s\n", GREEN, WHITE, curchan->nm, GREEN, WHITE, cur->cmd);
        drw(win);
        recent = NULL;
      }
      else if (curchan->q == 2 && cur->cmd[0] != '\0')
      {
        ssock(ircsock, "PRIVMSG %s :%s\n", fxp(curchan->nm), cur->cmd);
        recent = curchan;
        wp(win, "%s<%s%s%s>%s %s\n", BRIGHT(MAGENTA), WHITE, mnick, BRIGHT(MAGENTA), WHITE, cur->cmd);
        drw(win);
        recent = NULL;
      }
      else if (cur->cmd[0] != '\0' && sendpack(s, F_SAY, "%s %s", fxp(curchan->nm), cur->cmd) == -1)
        delsock(s);
    }
    
    if (cmdcnt(cmdl) == (BACKLOG+1))
    {
      cur1 = cmdl->next;
      free(cmdl);
      cmdl = cur1;
      cmdl->prev = NULL;
    }

    cur1 = cmdend(cmdl);
    cur2 = cur;
    cur3 = cur1;
    
    if (!strlen(cur1->cmd))
      cur = cur1;
    else
    {
      cur = (cmds_t *)malloc(sizeof(cmds_t));
      cur->next = NULL;
      cur->prev = cur1;
      cur1->next = cur;
    }

    cur->cmd[0] = 1;
    
    if (cur2->cmd[0] != 1 && ccmd && ccmd->cmd[0])
    {
      cur1 = (cmds_t *)malloc(sizeof(cmds_t));
      cur3->prev->next = cur1;
      cur1->next = cur;
      memset(cur1->cmd, 0, sizeof(cur1->cmd));
      strcpy(cur1->cmd, cur2->cmd);
      cur1->prev = cur->prev;
      cur->prev = cur1;
    }
    
    for (cur1=cmdl;cur1;cur1=cur1->next)
    {
      if (cur1->cmd[0] == '\0')
      {
        if (cur1->prev)
          cur1->prev->next = cur1->next;
        if (cur1->next)
          cur1->next->prev = cur1->prev;
        free(cur1);
        cur1 = NULL;
        break;
      }
    }
    
    memset(cur->cmd, 0, sizeof(cur->cmd));
    
    ccmd = NULL;
    i = 0;
    cx = 0;
    curx = 0;
    indraw(cur->cmd, cx, winput);
    
    if (n == -1)
    {
      m->d = cbk = (void *)cur;
      curx = i;
      return(0);
    }
    else if (n == -2)
    {
      m->d = cbk = (void *)cur;
      curx = i;
      return(-1);
    }
  }
  else if (ebuf == '\e')
  {
    cbuf = wgetch(winput);
    if (cbuf == KEY_BACKSPACE || cbuf == 127 || cbuf == 8)
    {
      i--;
      while (i>=0&&cur->cmd[i]==' ')
      {
        strncpy(cur->cmd+i, cur->cmd+i+1, strlen(cur->cmd+i+1));
        cur->cmd[strlen(cur->cmd)-1] = 0;
        i--;
      }
      while (i>=0&&cur->cmd[i]!=' ')
      {
        strncpy(cur->cmd+i, cur->cmd+i+1, strlen(cur->cmd+i+1));
        cur->cmd[strlen(cur->cmd)-1] = 0;
        i--;
      }
      if (i < 0)
      {
        memset(cur->cmd, 0, sizeof(cur->cmd));
        i = 0;
      }
      else
        i++;
      if (!i)
        curr = 0;
      cx = curx = i;
      indraw(cur->cmd, cx, winput);
    }
  }
  else if (ebuf == KEY_BACKSPACE || ebuf == 127 || ebuf == 8)
  {
    if (i)
    {
      for (j=i-1;cur->cmd[j];j++)
        cur->cmd[j] = cur->cmd[j+1];
      i--;
      cx--;
      indraw(cur->cmd, cx, winput);
    }
    ebuf = 0;
  }
  else if (ebuf == 23)
  {
    for (j=i,cur->cmd[j]=0;j>=0;j--)
    {
      if (j == 0)
      {
        cur->cmd[j] = 0;
        i = cx = 0;
        break;
      }
      if (cur->cmd[j] == ' ')
      {
        i = cx = j+1;
        break;
      }
      cur->cmd[j] = 0;
    }
    indraw(cur->cmd, cx, winput);
  }
  else if (ebuf == 24)
  {
    cur->cmd[i] = cbuf;
    upchan(curchan);
    indraw(cur->cmd, cx, winput);
    if (wmode)
    {
      dscr(wchan);
      drw(wchan);
    }
    dstatus();
    ebuf = 0;
  }
  else if (ebuf == discard)
  {
    memset(cur->cmd, 0, strlen(cur->cmd)+1);
    cx = i = 0;
    indraw(cur->cmd, cx, winput);
    ebuf = 0;
  }
  else if (ebuf == 154)
    ebuf = 0;
  else if (ebuf == '\t')
  {
    int g;
    ebuf = 0;
    
    for (g=cx;cur->cmd[g]!=' '&&g>=0;g--);
    
    if (cur->cmd[g] != ' ')
      g = 0;
          
    if (*cur->cmd == '/' && !g)
    {
      alias_t *al;
      
      for (al=alhead;al;al=al->next)
        if (!strncasecmp(cur->cmd+1, al->nm, strlen(cur->cmd+1)))
        {
          memset(cur->cmd, 0, sizeof(cur->cmd));
          cur->cmd[0] = '/';
          strcpy(cur->cmd+1, al->nm);
          strcat(cur->cmd, " ");
          cx = curx = i = strlen(cur->cmd);
          indraw(cur->cmd, cx, winput);
          break;
        }

      for (j=0;out[j].func;j++)
        if (!strncasecmp(cur->cmd+1, out[j].nm, strlen(cur->cmd+1)) && out[j].help)
        {
          memset(cur->cmd, 0, sizeof(cur->cmd));
          cur->cmd[0] = '/';
          strcpy(cur->cmd+1, out[j].nm);
          strcat(cur->cmd, " ");
          cx = curx = i = strlen(cur->cmd);
          indraw(cur->cmd, cx, winput);
          break;
        }
    }
    else if (curchan && curchan->users)
    {
      user_t *usr;
      int k, z;
      
      for (k=cx;cur->cmd[k]!=' '&&k>=0;k--);
      
      if (cur->cmd[k] == ' ')
        z = k+1;
      else
        z = 0;
      
      for (usr=curchan->users;usr;usr=usr->next)
        if (!strncasecmp(cur->cmd+z, usr->nm, strlen(cur->cmd+z)))
        {
          for (k=z;k<=cx;k++)
            cur->cmd[k] = 0;
          strcpy(cur->cmd+z, usr->nm);
          if (!z)
            strcat(cur->cmd, ": ");
          cx = curx = i = strlen(cur->cmd);
          indraw(cur->cmd, cx, winput);
          break;
        }
    }
  }
  else if (ebuf == 20)
  {
    if (tind == (strlen(curchan->topic)-(COLS-(strlen(VERSION)+8)-4)))
      tind = 0;
    else
      tind += 5;
    if (!curchan || !curchan->topic || tind > strlen(curchan->topic))
      tind = 0;
    ebuf = 0;
    dstatus();
  }
  else if (ebuf == 5 || ebuf == KEY_PPAGE)
  {
    dscroll(win, 10);
    indraw(cur->cmd, cx, winput);
    ebuf = 0;
  }
  else if (ebuf == 4 || ebuf == KEY_NPAGE)
  {
    dscroll(win, -10);
    indraw(cur->cmd, cx, winput);
    ebuf = 0;
  }
  else if (ebuf == 12)
  {
    werase(wchan);
    drw(wchan);
    werase(sep);
    drw(sep);
    werase(winput);
    drw(winput);
    if (!notop)
    {
      werase(head);
      drw(head);
    }
    dstatus();
    dscr(wchan);
    drw(wchan);
    indraw(cur->cmd, cx, winput);
    ebuf = 0;
  }
  else if (ebuf == KEY_F(2))
  {
    sscr(search);
    return(0);
  }
  else if (ebuf == KEY_END || ebuf == KEY_SELECT)
  {
    scur = NULL;
    dscr(win);
    drw(win);
  }
  else if (ebuf == KEY_UP)
  {
    if (ccmd == NULL)
      cur1 = cur;
    else
      cur1 = ccmd;
    if (cur1->prev != NULL)
    {
      ccmd = cur1->prev;
      cx = i = strlen(ccmd->cmd);
      cur = ccmd;
      indraw(cur->cmd, cx, winput);
    }
  }
  else if (ebuf == KEY_DOWN)
  {
    if (ccmd != NULL)
    {
      cur1 = ccmd;
      if (cur1->next != NULL)
      {
        ccmd = cur1->next;
        cx = i = strlen(ccmd->cmd);
        cur = ccmd;
        indraw(cur->cmd, cx, winput);
      }
    }
  }
  else if (ebuf == KEY_RIGHT)
  {
    if (cur->cmd[i])
    {
      cx++;
      i++;
    }
    indraw(cur->cmd, cx, winput);
  }
  else if (ebuf == KEY_LEFT)
  {
    if (i)
    {
      cx--;
      i--;
    }
    indraw(cur->cmd, cx, winput);
  }
  else if (ebuf < 256)
  {
    cbuf = cur->cmd[i];
    for (j=i;cur->cmd[j];j++);
    memmove(cur->cmd+i+1, cur->cmd+i, j);
    cur->cmd[i] = ebuf;
    ebuf = 0;
    i++;
    cx++;
    curx++;
    indraw(cur->cmd, cx, winput);
  }
  m->d = (void *)cur;
  
  curx = i;
  
  return(1);
}

chans_t *findchan(chans_t *h, char *chan)
{
  chans_t *cur;

  for (cur=h;;cur=cur->next)
  {
    if (!cur)
      return(NULL);
    if (!strcasecmp(cur->nm, chan))
      return(cur);
  }
}

chans_t *findquery(chans_t *h, char *chan)
{
  chans_t *cur;

  for (cur=h;;cur=cur->next)
  {
    if (!cur)
      return(NULL);
    if (!strcasecmp(cur->nm, chan) && cur->q == 1)
      return(cur);
  }
}

int sendpack(int s, unsigned short op, char *fmt, ...)
{
  char *pack, *data;
  phead_t *hdr;
  va_list args;
  int r;
  
  if (s == -1)
  {
    wp(wchan, "%s* Not connected to the server%s\n", RED, WHITE);
    drw(wchan);
    return(-1);
  }
  
  if (!fmt)
  {
    hdr = (phead_t *)malloc(sizeof(phead_t));
    
    hdr->len = 0;
    hdr->op = op;

    if (info.d == 2)
    {
      if (wchan)
      {
        wp(wchan, "--> (%x)\n", hdr->op);
        drw(wchan);
      }
    }
    
    r = send(s, (char *)hdr, 4, 0);
    
    return(r);
  }
  
  data = (char *)malloc(4096);
  memset(data, 0, 4096);
  
  va_start(args, fmt);
  vsprintf(data, fmt, args);
  va_end(args);
  
  data = (char *)realloc(data, strlen(data)+1);
  
  pack = (char *)malloc(strlen(data)+5);
  memset(pack, 0, sizeof(pack));
  hdr = (phead_t *)pack;
  
  hdr->len = strlen(data);
  hdr->op = op;
  strcpy(pack+4, data);

  if (info.d == 2)
  {
    if (wchan)
    {
      wp(wchan, "--> (%x) |%s|\n", hdr->op, data);
      drw(wchan);
    }
  }
  
  r = send(s, pack, hdr->len+4, 0);
  free(pack);
  free(data);
  
  return(r);
}

int ssock(int s, char *fmt, ...)
{
  char *data;
  va_list args;
  int r;
  
  data = (char *)malloc(4096);
  memset(data, 0, 4096);
  
  va_start(args, fmt);
  vsprintf(data, fmt, args);
  va_end(args);
  
  data = (char *)realloc(data, strlen(data)+1);
  
  if (info.d == 2)
  {
    if (wchan)
    {
      wp(wchan, "--> |%s|\n", data);
      drw(wchan);
    }
  }

  r = write(s, data, strlen(data));
  free(data);
  
  return(r);
}

int recvpack(int s, char **buf, phead_t **hdr)
{
  int r = 0, i = 0;
  fd_set fs;
  struct timespec tv;
  
  if (!thdr)
  {
    thdr = (phead_t *)malloc(sizeof(phead_t));
    memset(thdr, 0, sizeof(phead_t));
  
    while (i < 4)
    {
      r = read(s, ((char *)thdr)+i, sizeof(phead_t)-i);
      if (r <= 0)
      {
        free(thdr);
        thdr = NULL;
        *hdr = NULL;
        *buf = NULL;
        return(-1);
      }
      i += r;
    }
  }
  
  if (!thdr->len)
  {
    *hdr = thdr;
    *buf = strdup("");
    thdr = NULL;
    return(1);
  }
  
  FD_ZERO(&fs);
  FD_SET(s, &fs);
  tv.tv_nsec = 0;
  tv.tv_sec = 0;
  
  if (!select(s+1, &fs, NULL, NULL, (struct timeval *)&tv))
    return(-2);
  
  if (!tdbuf)
  {
    tdbuf = (char *)malloc(thdr->len+1);
    memset(tdbuf, 0, thdr->len+1);
    tpos = 0;
  }
  
  r = read(s, tdbuf+tpos, thdr->len-tpos);
  if (r <= 0)
  {
    free(tdbuf);
    free(thdr);
    tdbuf = NULL;
    thdr = NULL;
    *buf = NULL;
    *hdr = NULL;
    return(-1);
  }
  if ((r+tpos) < thdr->len)
  {
    *buf = NULL;
    *hdr = NULL;
    tpos += r;
    return(-2);
  }
  
/*  memset(tdbuf, 0, 1024);
  r = read(s, tdbuf, thdr->len);
  if (r <= 0)
  {
    free(tdbuf);
    free(thdr);
    tdbuf = NULL;
    thdr = NULL;
    *buf = NULL;
    *hdr = NULL;
    return(-1);
  }
  
  tdbuf = (char *)realloc(tdbuf, strlen(tdbuf)+1); */
  
  *buf = tdbuf;
  *hdr = thdr;
  
  tdbuf = NULL;
  thdr = NULL;
  
#ifdef DEBUG
  wp(wchan, "|%s| (%x:%i)\n", *buf, (*hdr)->op, (*hdr)->len);
  if (wchan)
    drw(wchan);
#endif
  
  return(1);
}

int rsock(int s, char **buf)
{
  int r;
  fd_set fs;

  *buf = (char *)malloc(4096);
  memset(*buf, 0, 4096);
  
  FD_ZERO(&fs);
  FD_SET(s, &fs);
  
  if (select(s+1, &fs, NULL, NULL, NULL) == -1)
  {
    free(*buf);
    *buf = NULL;
    return(-1);
  }
  
  if ((r = read(s, *buf, 4096)) <= 0)
  {
    free(*buf);
    *buf = NULL;
    return(-1);
  }
  
  *buf = (char *)realloc(*buf, strlen(*buf)+1);
  
  return(r);
}

void indraw(char *buf, int s, WINDOW *win)
{
  int b = s, i;
  unsigned char *t1 = (char *)malloc(512), *t = (char *)malloc(513);
  char *p=NULL;

  strcpy(t, buf);
  strcat(t, " ");
  
  if (curchan)
  {
    p = strdup(curchan->nm);
    if (strlen(curchan->nm) > (COLS/4))
      p[COLS/4] = 0;
  }
  else
    p = NULL;
  
  if (!p)
  {
    if (b > (curr+COLS-1))
      curr = b-COLS+1;
    else if (b < curr)
    {
      curr-=COLS/4;
      if (curr < 0)
        curr = 0;
    }
    if ((curr+COLS) > strlen(t) && curr)
      curr = strlen(t)-COLS;
    if (curr < 0)
      curr = 0;
    sprintf(t1, fxp(t+curr));
  }
  else if (*buf)
  {
    i = strlen(p)+3;
    if (b > (curr+(COLS-i)-1))
      curr = b-(COLS-i)+1;
    else if (b < curr)
    {
      curr-=COLS/4;
      if (curr < 0)
        curr = 0;
    }
    if ((curr+(COLS-i)) > strlen(t) && curr)
      curr = strlen(t)-(COLS-i);
    if (curr < 0)
      curr = 0;
    if (curchan->q == 1)
      sprintf(t1, "(%s) %s", p, t+curr);
    else
      sprintf(t1, "[%s] %s", p, t+curr);
  }
  else
  {
    if (curchan->q == 1)
      sprintf(t1, "(%s) ", p);
    else
      sprintf(t1, "[%s] ", p);
  }
  
  werase(win);
  for (i=0;t1[i];i++)
  {
    if (t1[i] >= 1 && t1[i] <= 26)
      waddch(win, (t1[i]+64)|doesc(win, BOLD));
    else
      waddch(win, t1[i]);
  }
  if (p)
    b+=strlen(p)+3;
  b = b-curr;
  if (b < 0)
    b = 0;
  if (b >= COLS)
    b = COLS-1;
  wmove(win, 0, b);
  drw(win);
  free(t1);
  if (p)
    free(p);
}

char *fxp(char *in)
{
  int i=0, j=0;
  char *ret;
  
  if (!strchr(in, '%'))
    return(in);
  
  memset(tpbuf, 0, 512);
  ret = tpbuf;
  
  for (;in[i];i++,j++)
  {
    ret[j] = in[i];
    if (in[i] == '%')
    {
      j++;
      ret[j] = '%';
    }
  }
  
  ret[j+1] = '\0';
  
  return(ret);
}

int gconn(WINDOW *win, sock_t *m)
{
  int s = m->s, r;
  struct sockaddr_in frm;
  int frmlen = sizeof(frm);
  
  r = accept(s, (struct sockaddr *)&frm, &frmlen);
  if (r == -1)
    return(1);
  
  addsock(r, "conn", S_R, lsock);
  ssock(r, "1");
  
  return(1);
}

int lsock(WINDOW *win, sock_t *m)
{
  char buf[5];
  
  memset(buf, 0, sizeof(buf));
  recv(m->s, &buf, 4, MSG_PEEK);
  if (!strncmp(buf, "GET", 3))
  {
    recv(m->s, &buf, 3, 0);
    m->func = doget;
  }
  else if (!strncmp(buf, "SEND", 4))
  {
    recv(m->s, &buf, 4, 0);
    m->func = dosend;
  }
  else
    return(0);
    
  return(1);
}

int doget(WINDOW *win, sock_t *m)
{
  char *buf, *nm, *fn;
  unsigned long sz;
  file_t *cur;
  struct stat st;
  sock_t *sv;
  char **tok;
  int i, cnt;
  
  if (rsock(m->s, &buf) == -1)
    return(0);
    
  tok = form_toks(buf, &cnt);
  
  if (cnt < 3)
  {
    ssock(m->s, "INVALID REQUEST\n");
    free(buf);
    for (i=0;i<cnt;i++)
      free(tok[i]);
    free(tok);
    return(0);
  }
  
  nm = strdup(tok[0]);
  fn = strdup(tok[1]);
  sz = atof(tok[2]);
  free(buf);
  
  for (i=0;i<cnt;i++)
    free(tok[i]);
  free(tok);
  
  #ifdef DEBUG
    wp(win, "Request from %s: \"%s\"\n", nm, fn);
    drw(win);
  #endif
  
  cur = ffile(up, nm, fn);
  if (!cur)
  {
    ssock(m->s, "INVALID REQUEST\n");
    free(nm);
    free(fn);
    return(0);
  }
  free(nm);
  free(fn);
  
  if (!cur->f)
  {
    ssock(m->s, "FILE NOT FOUND\n");
    dupload(cur->nm, cur->fn);
    return(0);
  }
  
  fseek(cur->f, sz, SEEK_SET);
  
  m->d = (void *)cur;
  m->func = sfile;
  m->t = S_W;
  free(m->nm);
  m->nm = NULL;
  msprintf(&m->nm, "u %i", gnum(ux, 1));
  
  sv = findsock("server");
  sendpack(sv->s, F_UP, NULL);
  
  fstat(fileno(cur->f), &st);
  cur->csz = 0;
  cur->sz = st.st_size-sz;
  cur->bsz = sz;
  cur->g = 1;
  cur->sk = m;
  deltimer(cur->timer);
  
  ssock(m->s, "%lu", st.st_size);
  
  wp(win, "* Sending file \"%s\" to %s (%lu bytes)\n", cur->fn, cur->nm, cur->sz);
  drw(win);
  
  cur->st = time(NULL);
  
  return(1);
}

int dosend(WINDOW *win, sock_t *m)
{
  char *buf, *nm, *fn, *t;
  unsigned long sz;
  file_t *cur;
  struct stat st;
  sock_t *sv;
  char **tok;
  int i, cnt;
  
  if (rsock(m->s, &buf) == -1)
  {
    close(m->s);
    return(0);
  }
  
  tok = form_tokso(buf, &cnt);
  
  if (cnt < 3)
  {
    ssock(m->s, "INVALID REQUEST\n");
    free(buf);
    close(m->s);
    for (i=0;i<cnt;i++)
      free(tok[i]);
    free(tok);
    return(0);
  }
  
  nm = strdup(tok[0]);
  fn = strdup(tok[1]);
  sz = atof(tok[2]);
  free(buf);
  
  for (i=0;i<cnt;i++)
    free(tok[i]);
  free(tok);
  
  if (!strcasecmp(nm, "FILE"))
  {
    free(nm);
    free(fn);
    close(m->s);
    return(0);
  }
  
  if (strchr(fn, '/'))
    t = strrchr(fn, '/')+1;
  else if (strchr(fn, '\\'))
    t = strrchr(fn, '\\')+1;
  else
    t = fn;
  
  if (!sz)
  {
    cur = ffile(down, nm, t);
    if (cur)
      ddownload(cur->nm, cur->fn);
    rsock(m->s, &buf);
    wp(win, "* Error downloading \"%s\" from %s [%s]\n", t, nm, buf?buf:"");
    drw(win);
    free(fn);
    free(nm);
    close(m->s);
    return(0);
  }
  
  cur = ffile(down, nm, t);
  if (!cur)
  {
    free(nm);
    free(fn);
    return(0);
  }
  
  free(nm);
  free(fn);
  
  m->d = (void *)cur;
  m->func = gfile;
  free(m->nm);
  m->nm = NULL;
  msprintf(&m->nm, "d %i", gnum(dx, 0));
  
  sv = findsock("server");
  sendpack(sv->s, F_DOWN, NULL);
  
  fstat(fileno(cur->f), &st);
  cur->bsz = st.st_size;
  cur->csz = 0;
  cur->sz = sz - st.st_size;
  cur->g = 1;
  cur->sk = m;
  deltimer(cur->timer);
  
  ssock(m->s, "%lu", cur->bsz);
  
  wp(win, "* Getting file \"%s\" from %s (%lu bytes)\n", cur->fn, cur->nm, cur->sz);
  drw(win);
  
  cur->st = time(NULL);
  
  return(1);
}

int initfserv(int port)
{
  struct sockaddr_in me;
  int s, on = 1;
  
  if (!port)
    return(1);
  
  me.sin_addr.s_addr = INADDR_ANY;
  me.sin_port = htons(port);
  me.sin_family = AF_INET;
  
  s = socket(AF_INET, SOCK_STREAM, 0);
  if (s == -1)
    return(-1);
  
  setsockopt(s, SOL_SOCKET, SO_REUSEADDR, &on, sizeof(on));
  
  if (bind(s, (struct sockaddr *)&me, sizeof(me)) == -1)
    return(-1);
  
  listen(s, 5);
  
  addsock(s, "fserv", S_R, gconn);
  
  return(1);
}

void closefserv()
{
  sock_t *sk;
  
  sk = findsock("fserv");
  if (!sk)
    return;
  
  delsock(sk->s);
}

int initsend(WINDOW *win, sock_t *m)
{
  int s = m->s, r;
  file_t *cur = (file_t *)m->d;
  sock_t *sk;
  struct stat st;
  
  if (!cur->f)
    return(0);
  
  fstat(fileno(cur->f), &st);
  
  r = ssock(s, "SEND");
  
  r = ssock(s, "%s \"%s\" %lu", info.user, cur->rfn, st.st_size);
  
  if (r <= 0)
  {
    wp(win, "%s* %s is misconfigured%s\n", RED, cur->nm, WHITE);
    drw(win);
    sk = findsock("server");
    if (sk)
      sendpack(sk->s, F_MISCONFIGURE, cur->nm);
    dupload(cur->nm, cur->fn);
    close(m->s);
    return(0);
  }
  
  m->func = ssize;
  m->t = S_R;
  cur->sz = st.st_size;
  cur->csz = 0;
  
  return(1);
}

int ssize(WINDOW *win, sock_t *m)
{
  char *buf = (char *)malloc(32);
  int n=1, s = m->s;
  file_t *cur = (file_t *)m->d;
  struct timespec tv;
  fd_set fs;
  
  memset(buf, 0, 32);
  if (recv(s, &buf[0], 1, 0) <= 0)
  {
    wp(win, "%s* Error sending file to %s (they disconnected)%s\n", RED, cur->nm, WHITE);
    drw(win);
    dupload(cur->nm, cur->fn);
    return(0);
  }
  
  recv(s, buf, sizeof(buf), 0);
  
  while (n)
  {
    FD_ZERO(&fs);
    FD_SET(s, &fs);
  
    tv.tv_nsec = 50;
    tv.tv_sec = 0;
  
    n = select((s+1), &fs, NULL, NULL, (struct timeval *)&tv);
  
    if (FD_ISSET(s, &fs))
    {
      if (recv(s, buf+strlen(buf), sizeof(buf), 0) <= 0)
      {
        free(buf);
        wp(win, "%s* Error sending file to %s%s\n", RED, cur->nm, WHITE);
        drw(win);
        dupload(cur->nm, cur->fn);
        close(m->s);
        return(0);
      }
    }
  }
  
  fseek(cur->f, strtoul(buf, (char **)NULL, 10), SEEK_SET);
  
  cur->bsz = strtoul(buf, (char **)NULL, 10);
  cur->sz -= cur->bsz;
  cur->g = 1;
  deltimer(cur->timer);
  
  free(buf);
  
  m->func = sfile;
  m->t = S_W;
  
  wp(win, "* Sending file \"%s\" to %s (%lu bytes)\n", cur->fn, cur->nm, cur->sz);
  drw(win);
  
  cur->st = time(NULL);
  
  return(1);
}

int initget(WINDOW *win, sock_t *m)
{
  int s = m->s, r;
  file_t *cur = (file_t *)m->d;
  sock_t *sk;
  struct stat st;
  
  if (!cur)
    return(0);
  
  if (!cur->f)
    return(0);
  
  fstat(fileno(cur->f), &st);
  
  r = ssock(s, "GET");
  
  r = ssock(s, "%s \"%s\" %lu", info.user, cur->rfn, st.st_size);
  
  if (r <= 0)
  {
    wp(win, "%s* %s is misconfigured%s\n", RED, cur->nm, WHITE);
    drw(win);
    sk = findsock("server");
    if (sk)
      sendpack(sk->s, F_MISCONFIGURE, cur->nm);
    ddownload(cur->nm, cur->fn);
    close(m->s);
    return(0);
  }
  
  m->func = gsize;
  m->t = S_R;
  cur->bsz = st.st_size;
  cur->csz = 0;
  
  return(1);
}

int gsize(WINDOW *win, sock_t *m)
{
  char *buf = (char *)malloc(32);
  int i, s = m->s;
  file_t *cur = (file_t *)m->d;
  sock_t *sk;

  memset(buf, 0, 32);
  if (recv(s, &buf[0], 1, 0) <= 0)
  {
    wp(win, "%s* Error getting file from %s (they disconnected)%s\n", RED, cur->nm, WHITE);
    drw(win);
    ddownload(cur->nm, cur->fn);
    close(m->s);
    return(0);
  }
  
  for (i=0;;i++)
  {
    if (recv(s, &buf[i], 1, MSG_PEEK) <= 0 && !i)
    {
      wp(win, "%s* %s is misconfigured%s\n", RED, cur->nm, WHITE);
      drw(win);
      free(buf);
      sk = findsock("server");
      if (sk)
        sendpack(sk->s, F_MISCONFIGURE, cur->nm);
      ddownload(cur->nm, cur->rfn);
      close(m->s);
      return(0);
    }
    if (isalpha(buf[i]))
    {
      recv(s, buf, 32, 0);
      wp(win, "* Error getting \"%s\" from %s [%s]\n", cur->fn, cur->nm, buf);
      drw(win);
      ddownload(cur->nm, cur->fn);
      free(buf);
      close(m->s);
      return(0);
    }
    if (!isdigit(buf[i]))
      break;
    recv(s, &buf[i], 1, 0);
  }
  
  cur->sz = atol(buf)-cur->bsz;
  cur->g = 1;
  deltimer(cur->timer);
  
  free(buf);
  
  m->func = gfile;
  
  wp(win, "* Getting file \"%s\" from %s (%lu bytes)\n", cur->fn, cur->nm, cur->sz);
  drw(win);
  
  cur->st = time(NULL);
  
  return(1);
}

int sfile(WINDOW *win, sock_t *m)
{
  char *buf = (char *)malloc(2049), *st, *end;
  file_t *cur = (file_t *)m->d;
  int n;
  time_t t;
  
  memset(buf, 0, 2049);
  n = read(fileno(cur->f), buf, 2048);
  if (n <= 0)
  {
    t = time(NULL);
    wp(win, "* Finished sending \"%s\" to %s (%lu of %lu bytes) in %lu\n", cur->fn, cur->nm, cur->csz, cur->sz, time(NULL)-cur->st);
    drw(win);
    if (info.xfer)
    {
      st = strdup(ctime(&cur->st));
      st[strlen(st)-1] = 0;
      end = strdup(ctime(&t));
      end[strlen(end)-1] = 0;
      fprintf(info.xfer, "S %s %s \"%s\" %s\n", st, cur->nm, cur->fn, end);
      fflush(info.xfer);
      free(st);
      free(end);
    }
    dupload(cur->nm, cur->fn);
    free(buf);
    close(m->s);
    return(0);
  }
  
  if (send(m->s, buf, n, 0) <= 0)
  {
    t = time(NULL);
    wp(win, "%s* Transfer interrupted while sending \"%s\" to %s (%lu of %lu bytes)%s\n", RED, cur->fn, cur->nm, cur->csz, cur->sz, WHITE);
    drw(win);
    if (info.xfer)
    {
      st = strdup(ctime(&cur->st));
      st[strlen(st)-1] = 0;
      end = strdup(ctime(&t));
      end[strlen(end)-1] = 0;
      fprintf(info.xfer, "SI %s %s \"%s\" %s\n", st, cur->nm, cur->fn, end);
      fflush(info.xfer);
      free(st);
      free(end);
    }
    dupload(cur->nm, cur->fn);
    free(buf);
    close(m->s);
    return(0);
  }
  
  cur->csz += n;
  free(buf);
  
  return(1);
}

int gfile(WINDOW *win, sock_t *m)
{
  char *buf = (char *)malloc(2049), *st, *end;
  file_t *cur = (file_t *)m->d;
  int n;
  time_t t;
  
  memset(buf, 0, 2049);
  if ((n = recv(m->s, buf, 2048, 0)) <= 0)
  {
    t = time(NULL);
    wp(win, "* Finished transfer of \"%s\" (%lu of %lu bytes) in %lu\n", cur->fn, cur->csz, cur->sz, time(NULL)-cur->st);
    drw(win);
    if (info.xfer)
    {
      st = strdup(ctime(&cur->st));
      st[strlen(st)-1] = 0;
      end = strdup(ctime(&t));
      end[strlen(end)-1] = 0;
      if (cur->csz != cur->sz)
        fprintf(info.xfer, "RI %s %s \"%s\" %s\n", st, cur->nm, cur->fn, end);
      else
        fprintf(info.xfer, "R %s %s \"%s\" %s\n", st, cur->nm, cur->fn, end);
      fflush(info.xfer);
      free(st);
      free(end);
    }
    ddownload(cur->nm, cur->fn);
    free(buf);
    close(m->s);
    return(0);
  }
  write(fileno(cur->f), buf, n);
  free(buf);
  
  cur->csz += n;
  
  return(1);
}
