/* Copyright (c) 2000  Kevin Sullivan <nite@gis.net>
 *
 * Please refer to the COPYRIGHT file for more information.
 */

/* This file deal with dynamically linked libraries of user-defined
 * commands. This must be a relict from a time when nap was
 * closed-source. With an open source program, surely there is not
 * much benefit to having dynamically linked user-definable commands?
 * In any case, this seems like overkill for a napster client. What
 * kind of high-powered stuff are we supposed to plug in here? This
 * might be thrown out in a future release. For now, we'll make it
 * optional. -PS
 **/

#ifdef USERCMDS  /* if you ever use this feature, please let me know
		    at selinger@theory.stanford.edu. Otherwise it will
		    be thrown out in a not-too-distant release */

/* according to Christian "naddy" Weisgerber <naddy@mips.inka.de>:
 * "a.out systems need to prefix an underscore to symbol names, ELF
 * ones don't". "On some platforms (Linux, BSD) the preprocessor will
 * define __ELF__, which can be used for local fixes, but this isn't
 * available on other platforms such as Solaris."
 **/

#if (defined __OpenBSD__ || defined __linux__) && !defined __ELF__
  #define  dlsym(x,y) dlsym(x, "_" y)
#endif


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#ifndef MCURSES
#include <ncurses.h>
#endif
#include <stdarg.h>
#include <dlfcn.h>

#include "defines.h"
#include "codes.h"
#include "cmds.h"
#include "nap.h"
#include "winio.h"
#include "usercmds.h"

#ifdef MCURSES
#include "wind.h"
#endif


extern chans_t *curchan;
extern info_t info;

static int sock;
static WINDOW *w;
void *hnd = NULL;

ucmd_func_t ucmds[] = { 
  (ucmd_func_t)say, 
  (ucmd_func_t)dprint, 
  (ucmd_func_t)dcurchan, 
  (ucmd_func_t)call, 
  (ucmd_func_t)mynick 
};

optab_t cc[] = {
  { F_SAID, "chanmsg" },
  { F_NCHAN, "servmsg" },
  { 0x0, NULL }
};


int libload(char *nm)
{
  void (*setfunc)(ucmd_func_t *);

  hnd = dlopen(nm, RTLD_LAZY);
  if (!hnd)
    return(-1);
  
  if (!dlsym(hnd, "cmds") && !dlsym(hnd, "handlers"))
  {
    dlclose(hnd);
    return(0);
  }
  
  setfunc = dlsym(hnd, "setfunc");
  if (!setfunc)
  {
    dlclose(hnd);
    return(0);
  }
  
  setfunc(ucmds);
  
  return(1);
}

int libunload(void **hnd)
{
  if (!(*hnd))
    return(1);
  dlclose(*hnd);
  *hnd = NULL;
  return(1);
}
 
int usercmd(int s, char **tok, int num, char *buf, WINDOW *win)
{
  int i, r;
  cmd_t *cmds;
  
  if (!hnd)
    return(0);
  
  sock = s;
  w = win;
  
  cmds = (cmd_t *)dlsym(hnd, "cmds");
  if (!cmds)
    return(0);
  
  for (i=0;;i++)
  {
    if (!cmds[i].nm)
      return(0);
    
    if (!strcasecmp(cmds[i].nm, tok[0]))
    {
      r = cmds[i].func(buf, tok, num);
      return(r);
    }
  }
}

int userhandler(int s, unsigned short op, char **tok, int num, char *buf, WINDOW *win)
{
  int i, r;
  cmd_t *handlers;
  char *nm;
  
  if (!hnd)
    return(0);
  
  nm = findop(op);
  if (!nm)
    return(0);
  
  sock = s;
  w = win;
  
  handlers = (cmd_t *)dlsym(hnd, "handlers");
  if (!handlers)
    return(0);
  
  for (i=0;;i++)
  {
    if (!handlers[i].nm)
      return(0);
    
    if (!strcasecmp(handlers[i].nm, nm))
    {
      r = handlers[i].func(buf, tok, num);
      return(r);
    }
  }
}

char *findop(unsigned short op)
{
  int i;
  
  for (i=0;;i++)
  {
    if (cc[i].nm == NULL)
      return(NULL);
    if (cc[i].op == op)
      return(cc[i].nm);
  }
}

int say(char *chan, char *str)
{
  if (!chan || !str)
    return(0);

  str = strdup(str);

  sendpack(sock, F_SAY, "%s %s", chan, fixquotes(str));
  
  free(str);
  
  return(1);
}

int dprint(const char *fmt, ...)
{
  va_list args;
  char *str;
  int i;
  
  str = (char *)malloc(4096);
  memset(str, 0, 4096);
  
  va_start(args, fmt);
  vsprintf(str, fmt, args);
  va_end(args);
  
  if (str[strlen(str)-1] != '\n')
  {
    i = strlen(str);
    str[i] = '\n';
    str[i+1] = '\0';
  }
  str = (char *)realloc(str, strlen(str)+1);
  
  wp(w, str);
  wrefresh(w);
  
  free(str);
  
  return(1);
}

char *dcurchan()
{
  if (!curchan)
    return(NULL);
  return(curchan->nm);
}

int call(const char *fmt, ...)
{
  va_list args;
  char *str;
  int i;
  
  str = (char *)malloc(4096);
  memset(str, 0, 4096);
  
  va_start(args, fmt);
  vsprintf(str, fmt, args);
  va_end(args);
  
  str = (char *)realloc(str, strlen(str)+1);
  
  i = parseout(sock, str, w);  
  free(str);
  
  return(i);
}

char *mynick()
{
  return(info.user);
}

#endif /* USERCMDS */
