/* Copyright (c) 2000  Kevin Sullivan <nite@gis.net>
 *
 * Please refer to the COPYRIGHT file for more information.
 */

#include <stdio.h>
#include <string.h>
#include <ncurses.h>
#include <stdlib.h>
#include <ctype.h>
#include <sys/time.h>

#include "defines.h"
#include "colors.h"
#include "codes.h"
#include "scheck.h"
#include "nap.h"
#include "winio.h"
#include "sscr.h"


extern ssearch_t *search;
extern file_t *down;
extern unsigned char finp;
extern char *conns[];
extern int curx;
extern WINDOW *winput, *wchan;

int cx, cy, num, lk, max;
WINDOW *swin;
char *status;
void *cbk;
unsigned char finp = 0;

/* a flag indicating whether to display each respective category of
   information. Must be 0 or 1. */
int showconn=1;
int showping=1;
int showuser=0;
int showbrate=1;
int showlength=1;
int showfreq=0;
int showsize=0;

/* turn on result screen */
void sscr(ssearch_t *l)
{
  sock_t *cur;
  
  cur = findsock("input");
  if (cur) {
    cbk = cur->d;
    delsock(cur->fd);
  }
  finp = 1;
  
  addsock(0, "sinput", S_R, sinput);
  findsock("sinput")->d = (void *)l;
  
  swin = newwin("sscr", LINES, 0, 0, 0);
  wbkgdset(swin, COLOR_PAIR(CPW));
  keypad(swin, TRUE);
  werase(swin);
  drw(swin);
  
  cx = cy = 0;
  num = 1;
  max = 0;
  status = strdup("Use F1 or 'q' to return to the main screen");
  lk = 1;
  
  showconn = l && l->conn != -1;
  showping = l && l->ping != -2;

  plist(l);
}

void plist(ssearch_t *l)
{
  int j, c, ln=0, p, min, sec, col, titlecols, usercols=4;
  ssearch_t *cur;
  float mb;
  char *t=NULL;
  
  if (!finp)
    return;
  
  if (!l || l->num == -1)
  {
    werase(swin);
    mvwprintw(swin, LINES/2, (COLS-25)/2, "No search results to list");
    wmove(swin, LINES-1, 0);
    wrefresh(swin);
    return;
  }
  
  for (c=0,cur=l;cur;cur=cur->next) {
    if (cur->num > c)
      c = cur->num;
    if (strlen(cur->cmp) > usercols)
      usercols = strlen(cur->cmp);
  }

  if (num > c)
    num = c;
  if (cy > c-1)
    return;
  
  max = c;

  if (num > (LINES-3)+cy)
  {
    if (num-1 <= (LINES-3))
      cy++;
    else
      cy = num-(LINES-3);
  }
  if (num-1 < cy)
  {
    cy = num-1;
  }
  
  /* decide how many columns to use for filenames (=titlecols) */
  titlecols = COLS - 1;
  titlecols -= 8*showbrate;
  titlecols -= 9*showlength;
  titlecols -= (3+usercols)*showuser;
  titlecols -= 12*showconn;
  titlecols -= 7*showping;
  titlecols -= 8*showfreq;
  titlecols -= 8*showsize;
  if (titlecols < 14)
    titlecols = 14;
  
  werase(swin);
  
  msprintf(&t, "Filename (total = %d)", max);
  if (strlen(t) > titlecols)
    t[titlecols]=0;
  mvwprintw(swin, 0, 0, "%s", t);
  wmove(swin, 1, 0);
  whline(swin, ACS_HLINE, COLS);

  col = titlecols;
  if (showbrate) {
    mvwprintw(swin, 0, col+3, "BRate");
    wmove(swin, 0, col+1);
    wvline(swin, ACS_VLINE, LINES-1);
    col += 8;
  }
  if (showlength) {
    mvwprintw(swin, 0, col+3, "Length");
    wmove(swin, 0, col+1);
    wvline(swin, ACS_VLINE, LINES-1);
    col += 9;
  }
  if (showsize) {
    mvwprintw(swin, 0, col+3, "  MB");
    wmove(swin, 0, col+1);
    wvline(swin, ACS_VLINE, LINES-1);
    col += 8;
  }
  if (showfreq) {
    mvwprintw(swin, 0, col+3, "Freq");
    wmove(swin, 0, col+1);
    wvline(swin, ACS_VLINE, LINES-1);
    col += 8;
  }
  if (showuser) {
    mvwprintw(swin, 0, col+3, "User");
    wmove(swin, 0, col+1);
    wvline(swin, ACS_VLINE, LINES-1);
    col += 3+usercols;
  }
  if (showconn) {
    mvwprintw(swin, 0, col+3, "Speed");
    wmove(swin, 0, col+1);
    wvline(swin, ACS_VLINE, LINES-1);
    col += 12;
  }
  if (showping) {
    mvwprintw(swin, 0, col+3, "Ping");
    wmove(swin, 0, col+1);
    wvline(swin, ACS_VLINE, LINES-1);
    col += 7;
  }

  /* part of bug #214363 fix: j was c-i+1, cur->num was
     (c+1)-(cur->num).  j is the actual number displayed, and so is
     cur->num. Moreover, num is what used to be num+1, max is what
     used to be max+1, and c is what used to be c+1. In other words,
     since the display starts counting from one, so do we internally. -PS */

  for (j=cy+1,ln=0,p=1;ln<(LINES-3);j++,ln++,p=1)
  {
    for (cur=l;cur&&cur->num!=j;cur=cur->next);
    if (cur)
    {
      if (num == j)
      {
        wattron(swin, COLOR_PAIR(1)|A_BOLD);
        wmove(swin, ln+2, 0);
        whline(swin, ' ', COLS);
        if (!lk)
        {
          if (status)
            free(status);
          status = strdup(cur->fn);
        }
      }
        
      msprintf(&t, "%i) %s", j, cur->song);
      if (strlen(t) > titlecols)
	t[titlecols]=0;
      mvwprintw(swin, ln+2, 0, "%s", t);
      col = titlecols;

      if (showbrate) {
	mvwprintw(swin, ln+2, col+3, "%4i", cur->brate);
        if (num==j) {
	  wmove(swin, ln+2, col+1);
	  wvline(swin, ACS_VLINE, 1);
	}
	col+=8;
      }
      if (showlength) {
	min = cur->time/60;
	sec = cur->time%60;
	mvwprintw(swin, ln+2, col+3, "%3i:%02i", min, sec);
        if (num==j) {
	  wmove(swin, ln+2, col+1);
	  wvline(swin, ACS_VLINE, 1);
	}
	col+=9;
      }
      if (showsize) {
	mb = ((float)cur->sz)/1048576.0;
	mvwprintw(swin, ln+2, col+3, "%5.02f", mb);
        if (num==j) {
	  wmove(swin, ln+2, col+1);
	  wvline(swin, ACS_VLINE, 1);
	}
	col+=8;
      }
      if (showfreq) {
	mvwprintw(swin, ln+2, col+3, "%5i", cur->freq);
        if (num==j) {
	  wmove(swin, ln+2, col+1);
	  wvline(swin, ACS_VLINE, 1);
	}
	col+=8;
      }
      if (showuser) {
	mvwprintw(swin, ln+2, col+3, "%s", cur->cmp);
        if (num==j) {
	  wmove(swin, ln+2, col+1);
	  wvline(swin, ACS_VLINE, 1);
	}
	col+=3+usercols;
      }
      if (showconn) {
	mvwprintw(swin, ln+2, col+3, cur->conn != -1 ? conns[cur->conn] : "N/A");
        if (num==j) {
	  wmove(swin, ln+2, col+1);
	  wvline(swin, ACS_VLINE, 1);
	}
	col+=12;
      }
      if (showping) {
	mvwprintw(swin, ln+2, col+3, cur->ping>=0 ? "%4i" : " N/A", cur->ping);
        if (num==j) {
	  wmove(swin, ln+2, col+1);
	  wvline(swin, ACS_VLINE, 1);
	}
	col+=7;
      }
      
      if (num == j)
        wattroff(swin, COLOR_PAIR(1)|A_BOLD);

    } /* end if cur */
    else
      break;
  } /* end for j */
  
  if (status)
  {
    if (strlen(status) < COLS)
    {
      cx = 0;
      status = (char *)realloc(status, COLS+1);
    }
    if (strlen(status+cx) < COLS && cx)
      cx = strlen(status)-COLS;
    strncpy(status, status+cx, COLS);
    status[COLS] = 0;
  }
  
  wattron(swin, COLOR_PAIR(CPBR));
  if (!status)
    status = strdup("Status");
  wmove(swin, LINES-1, 0);
  whline(swin, ' ', COLS);
  waddstr(swin, status);
  wattroff(swin, COLOR_PAIR(CPBR));
  
  drw(swin);

  if (t)
    free(t);
}

int sinput(WINDOW *win, sock_t *m)
{
  chtype cbuf = 0;
  WINDOW *w = swin;
  sock_t *sk;
  ssearch_t *cur;
#ifdef QUEUE
  int ret;
#endif /* QUEUE */
  
  cbuf = wgetch(w);
  
  if (cbuf == KEY_F(1) || cbuf == 'q')
  {
    uscr();
    return(0);  /* note returning 0 will also cause the connection
		   "sinput" to be deleted */
  }
  else if (cbuf == KEY_UP)
  {
    num--;
    if (num <= 0)
      num = 1;
    cx = lk = 0;
  }
  else if (cbuf == KEY_DOWN)
  {
    num++;
    cx = lk = 0;
  }
  else if (cbuf == KEY_RIGHT)
    cx++;
  else if (cbuf == KEY_LEFT)
  {
    cx--;
    if (cx < 0)
      cx = 0;
  }
  else if (cbuf == KEY_PPAGE || cbuf == 5)
  {
    num -= LINES-4;
    if (num <= 0)
      num = 1;
    cx = lk = 0;
  }
  else if (cbuf == KEY_NPAGE || cbuf == 4)
  {
    num += LINES-4;
    cx = lk = 0;
  }
  else if (cbuf == '\n' && search)   /* fix bug #215003 - replaced max by 
				     max-1 throughout, except here. */
  {
    lk = 1;
    for (cur=search;cur;cur=cur->next)
      if (num == cur->num)
        break;
    if (cur)
    {
      if (status)
        free(status);
      status = NULL;
      sk = findsock("server");
      if (sk)
      {
#ifndef QUEUE
        if (ffile(down, cur->cmp, cur->song))
          msprintf(&status, "Already downloading %s", cur->song);
	else
	{
	  msprintf(&status, "Downloading %s", cur->song);
	  sendpack(sk->fd, F_DGET, "%s \"%s\"", cur->cmp, cur->fn);
	}
#else /* QUEUE */
        ret = enqdown(cur);
        if (ret == -1)
          msprintf(&status, "Already downloading %s", cur->song);
        else if (ret == -2)
          msprintf(&status, "Already enqueued %s", cur->song);
        else if (ret == 0)
          msprintf(&status, "Enqueuing %s", cur->song);
        else /* (ret == 1) */
        {
          msprintf(&status, "Downloading %s", cur->song);
          sendpack(sk->fd, F_DGET, "%s \"%s\"", cur->cmp, cur->fn);
        }
#endif /* QUEUE */
      }
      else /* !sk */
      {
        msprintf(&status, "Not connected to server");
      }
    } 
    else /* !cur */
    {
      msprintf(&status, "Error: No matching item in search list??");
    }
  }

  else if (cbuf == 'l')
    showlength = 1-showlength;
  else if (cbuf == 'b')
    showbrate = 1-showbrate;
  else if (cbuf == 'm')
    showsize = 1-showsize;
  else if (cbuf == 'f')
    showfreq = 1-showfreq;
  else if (cbuf == 'u')
    showuser = 1-showuser;
  else if (cbuf == 's')
    showconn = 1-showconn;
  else if (cbuf == 'p')
    showping = 1-showping;
  else if (cbuf == 'a') {
    showlength = showbrate = showsize = showfreq = 1;
    showconn = showping = showuser = 1;
  }
  plist(m->d);
  
  return(1);
}

void uscr()
{
  delwin(swin);
  
  addsock(0, "input", S_R, input);

  findsock("input")->d = cbk;
  finp = 0;
  
  if (status)
    free(status);
  
  dscr(wchan);
  drw(wchan);
  dstatus();
  indraw(cbk, curx, winput);
  drw(winput);
}
