/* Copyright (c) 2000  Kevin Sullivan <nite@gis.net>
 *
 * Please refer to the COPYRIGHT file for more information.
 */

#include <stdio.h>
#ifndef MCURSES
#include <ncurses.h>
#endif
#include <stdlib.h>
#include <time.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <sys/socket.h>
#include <errno.h>

#include "defines.h"
#include "codes.h"
#include "colors.h"
#include "scheck.h"
#include "timer.h"
#include "nap.h"
#include "winio.h"
#include "lists.h"

#ifdef MCURSES
#include "wind.h"
#endif

extern info_t info;
extern int srch;
extern WINDOW *wchan;

/* the global connection list */
static sock_t *socklist = NULL;

/* count the number of "u" and "d" connections in socklist */
int upsocks = 0, downsocks = 0; 

/* the following flag is set by the /tquit command to indicate that we
   should shut down nap as soon as upsocks=0 and downsocks=0, i.e.,
   all active uploads and downloads have been completed. Note that
   this does not include "queued" or "waiting" uploads and downloads,
   just those that are "in progress". The /unquit command resets this
   flag to 0, thus allowing the user to change her mind. */
int quit_after_transfers = 0;

/* set by /quit to cause an immediate unconditional shutdown of nap */
int quit_now = 0;

/* add a connection to end of global connection list. s is file
   descriptor, nm is name, t is type (S_R or S_W), func is handler
   function. d is left blank. nm is copied (strdup). 
   Also, if this is an upload or download connection, notify server (ugly!) */
/* Note that upload/dowload notification is done in event.c:dosend() and
   event.c:doget() for connections initiated by a remote client. */
void addsock(int s, char *nm, unsigned char t, int (*func)(WINDOW *, sock_t *))
{
  sock_t *cur, *sv;
  
  /* create a new socket */
  cur = (sock_t *)malloc(sizeof(sock_t));
  cur->fd = s;
  cur->t = t;
  cur->nm = strdup(nm);
  cur->func = func;
  cur->next = NULL;
  cur->d = NULL;

  /* an add it to the list */
  list_append(sock_t, socklist, cur);
  
  /* if it's an upload or a download, notify the server */
  sv = findsock("server");

  if (cur->nm[0] == 'u' && cur->nm[1] == ' ') {
    if (sv) {
      sendpack(sv->fd, F_UP, NULL);
    }
    upsocks++;
  } else if (cur->nm[0] == 'd' && cur->nm[1] == ' ') {
    if (sv) {
      sendpack(sv->fd, F_DOWN, NULL);
    }
    downsocks++;
  }
}

/* delete a connection. It is uniquely identified by its file
   descriptor s.  This routine also notifies the server if an upload
   or download is being shut down (this is ugly, since this should
   rather be done in connection with the task structure, not the
   connection structure). It also sets srch to 0 in case we are
   deleting the server connection. It also shuts down the socket of
   the connection, and closes it, if it is indeed a socket connection
   (file descriptor >2). */
void delsock(int s)
{
  sock_t *cur, *sv;
  
  /* unlink the first connection whose fd is s, if any */
  list_unlink_cond(sock_t, socklist, cur, cur->fd == s);
  
  if (!cur) {
    return;
  }

  sv = findsock("server");
  
  if (cur->nm[0] == 'd' && cur->nm[1] == ' ') {
    if (sv) {
      sendpack(sv->fd, F_DOWNDONE, NULL);
    }
    downsocks--;
  } else if (cur->nm[0] == 'u' && cur->nm[1] == ' ') {
    if (sv) {
      sendpack(sv->fd, F_UPDONE, NULL);
    }
    upsocks--;
  }

  if (!strcmp(cur->nm, "server"))
    srch = 0;

  free(cur->nm);
  free(cur);

  /* In many cases, the socket is closed by a connection handler before
   * the call to delsock(). (See the many places in event.c where a
   * connection handler calls close(m->fd) before return(0)).
   * It would be safer to only call close() once, but this second call 
   * to close() hasn't caused problems so far because delsock() is always
   * called immediately after a connection handler returns 0. i.e. there
   * is no opportunity for the socket descriptor to be reused in the mean
   * time (or is there?) */
  if (s > 2) {
    shutdown(s, 2);
    close(s);
  }
  return;
}

/* finds a connection by name in the connection list */
sock_t *findsock(const char *nm)
{
  sock_t *elt;
  
  list_find(elt, socklist, !strcasecmp(elt->nm, nm));  

  return(elt);
}

/* finds a connection by file descriptor in the connection list */
sock_t *findsockfd(int fd)
{
  sock_t *elt;
  
  list_find(elt, socklist, elt->fd == fd);  

  return(elt);
}

/* prints the list of connections to the specified window. */
void psocks(WINDOW *win)
{
  sock_t *cur;
  
  wp(win, "fd | nm\n");
  for (cur=socklist;cur!=NULL;cur=cur->next)
    wp(win, "%2d   %s\n", cur->fd, cur->nm);
  drw(win);
}

/* this is the main event loop of the nap program. It is here that we
 * wait for network activity and then tend to the various different
 * connections (including user input). In addition, we call tevent()
 * (in timer.c) once a second or so, to take care of scheduled events.
 *
 * This function is called precisely once, from nap.c:main(). Once
 * we leave here, nap shuts down and terminates.
 *
 * up is the input window (equal to the global variable winput), and
 * win is the output window of the terminal interface (equal to the
 * global variable wchan). Why not just refer to the global variables,
 * as do some of the procedures that are called from here?
 *
 * Here are the current conventions for what the socket readers
 * (cur->func) are supposed to return. They normally return 1. A
 * return value of 0 means the current socket should be deleted.
 * 
 **/

void sockfunc(WINDOW *win, WINDOW *up)
{
  fd_set fs, fw; /* sets of file descriptors to watch, see "man select" */
  int r;
  sock_t *cur;
  struct timeval sec; /* note originally used <time.h> timespec here
			 instead of <select.h> timeval - this was a
			 type error */
  
  while (1)
  {
    dochecks();  /* notify terminal that it has been resized, if
		    necessary. */

    drw(up);     /* redraw input window - why? */

    /* if the --autorestart option is set, and there is no server, try to 
       reconnect to a server. */

    if (info.autorestart) { 
      for (cur=socklist;cur!=NULL;cur=cur->next) {
	if (!strcasecmp(cur->nm, "server"))
	  break;
      }
      if (cur==NULL) {
	sleep(1);  /* sleep to avoid excessive load in case of infinite loop */
	wp(win, "Connection to server lost, reconnecting...\n");
	drw(win);
	dreconnect(-1, "reconnect", NULL, 0, win);
      }
    }
    
    /* we return if this was requested by the /quit command */
    /* actually this check is redundant here, but sometimes a little
       redundancy is good. */
    if (quit_now)
      return;

    /* we return conditionally if the was requested by /tquit */
    if (quit_after_transfers && upsocks==0 && downsocks==0)
      return;

    /* the following code was almost never used */
#if 0
    if (!socklist)
      return;
    else if (!strcasecmp(socklist->nm, "input") && !socklist->next)
    {
      delsock(socklist->fd);
      return;
    }
#endif    

    /* let fs be the set of "read" file descriptors, and fw the list
       of "write" file descriptors, from the connection list */

    FD_ZERO(&fs);
    FD_ZERO(&fw);
    for (cur=socklist;cur!=NULL;cur=cur->next)
    {
      if (cur->t == S_R)
        FD_SET(cur->fd, &fs);
      else if (cur->t == S_W)
        FD_SET(cur->fd, &fw);
    }
    
    /* prepare timeout of 1 second, then wait for activity on connections.
       On fs connections, wait for characters to be available for reading.
       On fw connections, wait for ability to write. */
    sec.tv_sec = 1;
    sec.tv_usec = 0;
    r = select(FD_SETSIZE, &fs, &fw, NULL, &sec);

    /* the reason the previous call to "select" needs a timeout is so that
       we can call tevent(0) once a second or so to tend to scheduled events */
    tevent(0);

    /* if there is an error, print it, but then continue. However,
       EINTR means select was interrupted. This happens e.g. when the
       user resizes the window.  */
    if (r == -1) {
      if (errno != EINTR) {  
	wp(win, ""RED"* Error while watching connections: %s"WHITE"\n", \
	   sys_errlist[errno]);
      }
      continue;
    }

    /* if there was no network activity, continue main loop */
    if (r == 0)
      continue;
    
    /* else do something for every connection that had activity. 
       Essentially, just call the handler (cur->func) for that connection. */
    for (cur=socklist;cur!=NULL;cur=cur->next)
    {
      if (cur->t == S_R && FD_ISSET(cur->fd, &fs))
      {
	FD_CLR(cur->fd, &fs);
        r = cur->func(win, cur);
	if (quit_now) {
	  goto quit;
	}
        if (!r) {
          delsock(cur->fd);
	}
	cur=socklist;  /* need to start from beginning since sockets might
		      have been deleted in the meantime */
      }
      else if (cur->t == S_W && FD_ISSET(cur->fd, &fw))
      {
	FD_CLR(cur->fd, &fw);
        r = cur->func(win, cur);
	if (quit_now) {
	  goto quit;
	}
        if (!r) {
          delsock(cur->fd);
        }
	cur=socklist;
      }
    }
  }

  /* end of loop not reachable */

 quit:                 
  /* we get here if the user issues the /quit command. */
  while (socklist!=NULL)
    delsock(socklist->fd);
  return;
}

/* this essentially seems to be a repetition of the above, except it
   returns after sec seconds. It is called only in one place in
   cmds.c, and can maybe be done better with timers? */

void sockfunct(WINDOW *win, WINDOW *up, int sec)
{
  fd_set fs, fw;
  int r, adj=0;
  sock_t *cur;
  struct timeval tv;
  time_t t, t1;
  
  while (1)
  {
    tv.tv_sec = sec-adj;
    tv.tv_usec = 0;
    if (tv.tv_sec <= 0)
      return;
      
    dochecks();
    drw(up);
    FD_ZERO(&fs);
    FD_ZERO(&fw);
    
    if (!socklist)
      return;
    else if (!strcasecmp(socklist->nm, "input") && !socklist->next)
    {
      delsock(socklist->fd);
      return;
    }
    
    for (cur=socklist;cur!=NULL;cur=cur->next)
    {
      if (cur->t == S_R)
        FD_SET(cur->fd, &fs);
      else if (cur->t == S_W)
        FD_SET(cur->fd, &fw);
    }
    
    time(&t);
    
    r = select(FD_SETSIZE, &fs, &fw, NULL, &tv);
    tevent(0);
    if (r == -1)
      continue;
    else if (!r)
      return;
    
    time(&t1);
    
    adj+=(t1-t);
    
    for (cur=socklist;cur!=NULL;cur=cur->next)
    {
      if (cur->t == S_R && FD_ISSET(cur->fd, &fs))
      {
	FD_CLR(cur->fd, &fs);
        r = cur->func(win, cur);
        if (!r)
          delsock(cur->fd);

        else if (r == -1)
	  goto quit;
	cur=socklist;
      }
      else if (cur->t == S_W && FD_ISSET(cur->fd, &fw))
      {
	FD_CLR(cur->fd, &fw);
        r = cur->func(win, cur);
        if (!r)
	  delsock(cur->fd);
        else if (r == -1)
	  goto quit;
	cur=socklist;
      }
    }
  }

  /* end of loop not reachable */
  
 quit:                 /* we get here if the user issues the quit
		       command. Note that we eventually return to the
		       'real' sockfunc, which will shut down because
		       there are no more sockets. */
  while (socklist!=NULL)
    delsock(socklist->fd);
  return;

}
