/* Copyright (c) 2000  Kevin Sullivan <nite@gis.net>
 *
 * Please refer to the COPYRIGHT file for more information.
 */

#include <stdio.h>
#include <time.h>
#include <stdarg.h>
#include <termios.h>
#include <dirent.h>
#include <fcntl.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#ifndef MCURSES
  #include <ncurses.h>
#endif
#include <stdlib.h>
#include <signal.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <sys/time.h>
#include <netdb.h>
#include <errno.h>

#include "getopt.h"
#include "defines.h"
#include "codes.h"
#include "colors.h"
#include "title.h"
#include "mp3s.h"
#include "alias.h"
#include "event.h"
#include "handlers.h"
#include "lists.h"
#include "nap.h"
#include "scheck.h"
#include "usercmds.h"
#include "winio.h"

#ifdef MCURSES
#include "wind.h"
#endif

/* standard napster servers */
char *std_serverlist = "server.napster.com;64.124.41.16;64.124.41.17;64.124.41.18;64.124.41.19";

extern unsigned char ircin, finp;
extern int ircsock;
extern unsigned char ups[];
extern scrls_t *mscroll, *scur;
extern chans_t *recent;
extern WINDOW *wchan, *winput;
extern file_t *up, *down;

chans_t *chanl = NULL, *curchan = NULL;
hotlist_t *hlist = NULL;
info_t info;          /* struct to hold some command line options */
int cloaked = 0;      /* whether we are cloaked */
unsigned char lpbrk=0, noresolv=0;
char tpbuf[512];
int tpos, ipcs[2];
unsigned char dresize = 0;

void doquit()
{
  scur = NULL;
  if (wchan)
  {
    dscr(wchan);
    drw(wchan);
  }
  if (ircin)
  {
    ssock(ircsock, "QUIT :Leaving\n");
    close(ircsock);
    ircsock = 0;
  }
  if (nvar("savechannels") == 1) {
    char *tmp = NULL, *fn;
    msprintf(&tmp, CHANNELFILE, info.user);
    fn = home_file(tmp);
    free(tmp);

    if (savechans(chanl, fn)==-1) {
      wp(wchan, ""RED"* Could not save channels to %s: %s"WHITE"\n", fn, \
	 sys_errlist[errno]);
    } else {
      wp(wchan, "* Saved channels to %s\n", fn);
    }
    free(fn);
  }

  /* close incomplete files, remove turds, etc */
  while (up)
    dupload(up);
  while (down)
    ddownload(wchan, down);
  endwin();

  exit(1);
}

void tresize(int dummy)
{
  dresize = 1;
}

/* catch interrupt signal, i.e. usually "Control-C". In interactive
   mode, quit program only after receiving two interrupts within one
   second. Interrupt also causes a "while" loop to be broken - see
   dwhile in cmds.c */

void sigint(int dummy)
{
  static time_t cct = 0;

  if ((time(NULL)-cct) <= 1)
    doquit();
  
  cct = time(NULL);
  lpbrk = 1;
}

void dochecks()
{
  if (dresize)
  {
    dresize = 0;
    resize();
  }
}

/* quotes a string so that it is printable. Returns a static object
   which is reused with the next call. */
char *quote(char *s) {
  static char buf[2048];
  int i;

  for (i=0; i<sizeof(buf)-5; s++) {
    if (*s == 0) {
      break;
    } else if (32 <= *s && *s < 127) { /* printable ASCII */
      buf[i++] = *s;
    } else if (*s == '\n') {
      i += sprintf(&buf[i], "\\n");
    } else if (*s == '\r') {
      i += sprintf(&buf[i], "\\r");
    } else {
      i += sprintf(&buf[i], "\\%03o", (unsigned char)*s);
    }
  }
  buf[i] = 0;
  return buf;
}

/* returns the full path of file fn relative to user's home
   directory. Always returns an allocated string. */
char *home_file(const char *fn) {
  char *home, *res;
  int len;

  if (fn[0] == '/')         /* absolute filename? */
    return strdup(fn);

  home = getenv("HOME");   /* else, try to find home directory */
  if (!home)
    return strdup(fn);

  len = strlen(home);
  res = (char *)malloc(len+strlen(fn)+2);
  strcpy(res, home);
  if (home[len-1] != '/')
    strcat(res, "/");
  strcat(res, fn);
  return res;
}

/* make a an allocated string from an integer, similar to strdup */
char *itoa(int n) {
  char buf[20];

  sprintf(buf, "%d", n);
  return strdup(buf);
}

/* strip whitespace from both ends of a string, descructively */
char *strip(char *s) {
  char *p;

  if (!s)
    return NULL;

  while (isspace(*s)) {
    s++;
  }

  p = s+strlen(s);
  while (p>s && isspace(p[-1]))
    p--;
  *p = 0;
  return s;
}

/* read an allocated line from file. Note: we could do better and
   avoid the static limit. */

char *getline(FILE *f) {
  char buf[512];
  char *res;
  
  res = fgets(buf, sizeof(buf), f);
  if (!res)
    return NULL;

  /* strip whitespace from both ends */
  res = strip(res);
  return strdup(res);
}

/* Read the config file FN. Values that were given on the command line
   override those in the config file - don't read them. Note: if user
   was specified on the command line, take special care *not* to use
   password and email address from config file. FN must be a non-null
   filename. Return 0. */

int readcfg(char *fn) {
  int other_user = 0;
  char *user=NULL, *pass=NULL, *email=NULL;
  int r;

  if (getval("user")) { /* user was specified on command line - do not
                           use password or email from config file */
    other_user = 1;
    user = strdup(getval("user"));
    pass = getrealval("pass");
    if (pass)
      pass = strdup(pass);
    email = getval("email");
    if (email)
      email = strdup(email);
  }
  
  /* read the config file - but in a wimpy way that does not overwrite
     existing values. This is because command line values should have
     priority. */
  wp(NULL, "Reading config file %s...\n", fn);
  r = loadsets(fn, NULL, 0, 0);

  if (r==0) {
    /* success */
  } else if (r==1) {
    wp(NULL, "There were some warnings, please edit your config file.\n", fn);
  } else if (errno == ENOENT) {   /* no such file or directory */
    wp(NULL, "Config file not found.\n", fn);
  } else {
    wp(NULL, "Error loading config file %s: %s\n", fn, sys_errlist[errno]);
  }

  /* restore password and email if necessary */
  if (other_user) {
    chset("user", user);
    chset("pass", pass);
    chset("email", email);
    free(user);
    free(pass);
    free(email);
  }
  
  return 0;
}

/* Try to guess reasonable defaults for vital configuration variables.
   Where appropriate, prompt the user. Return -1 if the user refused
   to supply a username or password, else 0. FN must be a non-null
   filename for the config file to be created / updated. If FN is
   null, nothing will be written to a file. Note: if info.daemon is
   set, then we never prompt the user for anything, and supply
   defaults as we can. */

int set_defaults_interactive(char *fn) {
  int changes = 0;
  char *ans;
  char *user, *pass;
  int conn;
  int r;
  struct stat st;

  ans = NULL;

  if (!getval("user")) {
    if (!info.daemon) {
      wp(NULL, "User: ");
      ans = getline(stdin);
    }
    if (!ans || !*ans) {
      wp(NULL, "No user name given\n");
      free(ans);
      return(-1);
    } else {
      chset("user", ans);
      changes = 1;
    }
  }

  free(ans);
  ans = NULL;

  user = getval("user");
  pass = getrealval("pass"); /* note: getval("pass") does not work - it always
				returns "?" */

  if (!pass || !strcmp(pass,"?")) {
    if (!info.daemon) {
      wp(NULL, "Password for user %s: ", user);
      ans = getpass(""); /* note: getpass(3) does not allocate its
			    result value, thus we should not free it. */
    }
    if (!ans || !*ans) {
      wp(NULL, "No password given\n");
      return(-1);
    } else {
      chset("pass", ans);
    }
  }

  ans = NULL;

  if (!getval("email")) {
    if (!info.daemon) {
      wp(NULL, "Email for user %s: ", user);
      ans = getline(stdin);
    }
    if (!ans || !*ans) {
      msprintf(&ans, "%s@localhost", user);
      chset("email", ans);
      wp(NULL, "No email given - trying %s\n", ans);
    } else {
      chset("email", ans);
      changes = 1;
    }
  }

  free(ans);
  ans = NULL;

  if (!getval("upload")) {
    if (!info.daemon) {
      wp(NULL, "Please enter a list of directories with files that you want to share.\n");
      wp(NULL, "You can enter several directories, separated by semicolons (';').\n");
      wp(NULL, "Upload path: ");
      ans = getline(stdin);
    }
    if (!ans || !*ans) {
      wp(NULL, "No upload directories given - unable to share files\n");
      free(ans);
    } else {
      chset("upload", ans);
      free(ans);
      changes = 1;
    }
  }

  if (!getval("download")) {
    if (!info.daemon) {
      wp(NULL, "Please enter the directory where you want to put downloaded files.\n");
      wp(NULL, "Download directory: ");
      ans = getline(stdin);
    }
    if (!ans || !*ans) {
      wp(NULL, "No download directory given - will use current working directory\n");
    } else {
      chset("download", ans);
      changes = 1;
    }
  }

  if (!getval("incomplete")) {
    if (!info.daemon) {
      wp(NULL, "Please enter the directory where you want to put incomplete files.\n");
      wp(NULL, "Incomplete directory: ");
      ans = getline(stdin);
    }
    if (!ans || !*ans) {
      wp(NULL, "No incomplete directory given - will use download directory\n");
    } else {
      chset("incomplete", ans);
      changes = 1;
    }
  }

  free(ans);
  ans = NULL;

  if (!getval("dataport")) {
    chset("dataport", "6699-6799");
    wp(NULL, "No dataport given - using %s\n", getval("dataport"));
  }

  if (!getval("connection")) {
    if (!info.daemon) {
      wp(NULL, "
          Connection | Number
          -------------------
          Unknown    |  0
          14.4       |  1
          28.8       |  2
          33.6       |  3
          56.7       |  4
          64K ISDN   |  5
          128K ISDN  |  6
          Cable      |  7
          DSL        |  8
          T1         |  9
          T3 or >    | 10
\n");
      wp(NULL, "How fast is your internet connection?\n");
      wp(NULL, "Please choose 0--10 from the chart: [4] ");
      ans = getline(stdin);
    }
    if (!ans || !*ans) {
      chset("connection", "4");
      changes = 1;
    } else {
      chset("connection", ans);
    }
  }

  free(ans);
  ans = NULL;

  conn = nvar("connection");
  if (conn < 0 || conn > 10) {
    wp(NULL, "Invalid connection given - using 0\n");
    chset("connection", NULL);
    conn = 0;
  }
  
  if (!getval("maxuploads")) {
    ans = itoa(ups[conn]);
    chset("maxuploads", ans);
    wp(NULL, "Allowing %s simultaneous uploads\n", ans);
    free(ans);
  }

  /* if anything changed, and if non-null fn was given, ask user if he
     wants to save changes. */

  if (fn && !info.daemon) {
    r = stat(fn, &st);  /* r!=0 if config file does not yet exist */
    
    if (r || changes) {
      if (r) {
	wp(NULL, "Create a config file (%s) with these settings? [y] ", fn);
      } else {
	wp(NULL, "Save these settings to your config file (%s)? [y] ", fn);
      }
      ans = getline(stdin);
      if (ans && (!strcmp(ans, "") || !strcasecmp(ans, "y") || !strcasecmp(ans, "yes"))) {
	r = savesets(fn);
	if (r==-1) {
	  wp(NULL, "Error saving config file: %s\n", sys_errlist[errno]);
	}
      } else {
	wp(NULL, "Not saving.\n");
      }
      free(ans);
    }
  }

  return 0;
}

/* print to a newly allocated string *str. If *str is non-NULL, free 
   the old content */
int msprintf(char **str, const char *fmt, ...)
{
  va_list args;
  char buf[4096];
  
  free(*str);
  
  va_start(args, fmt);
  vsprintf(buf, fmt, args);
  va_end(args);
  
  *str = strdup(buf);
  
  return(strlen(*str));
}

/* count the number of occurences of character t in string buf */
/* not used */
int strcnt(char *buf, char t)
{
  int i,r;
  
  for (i=0,r=0;buf[i];i++)
    if (buf[i] == t)
      r++;
  
  return(r);
}

static struct option longopts[] = {
  {"help",        0, 0, 'h'},
  {"version",     0, 0, 'v'},
  {"daemon",      0, 0, 'q'},
  {"autorestart", 0, 0, 'a'},
  {"build-only",  0, 0, 'B'},
  {"build",       0, 0, 'b'},
  {"nobuild",     0, 0, 'N'},
  {"notitle",     0, 0, 't'},
  {"reconnect",   0, 0, 'r'},
  {"nxterm",      0, 0, 'l'},
  {"create",      0, 0, 'm'},
  {"config",      1, 0, 'f'},
  {"server",      1, 0, 's'},
  {"debug",       1, 0, 'd'},
  {"log",         1, 0, 'x'},
  {"logall",      1, 0, 'g'},
  {"user",        1, 0, 'u'},
  {"pass",        1, 0, 'p'},
  {"password",    1, 0, 'p'},
  {"email",       1, 0, 'e'},
  {"upload",      1, 0, 'U'},
  {"download",    1, 0, 'D'},
  {"incomplete",  1, 0, 'I'},
  {"dataport",    1, 0, 'P'},
  {"connection",  1, 0, 'C'},
  {"maxuploads",    1, 0, 'M'},
  {"option",      1, 0, 'o'},
  {0, 0, 0, 0}
};

void phelp(char *nm)
{
  printf("Usage: %s [options]\n", nm);
  printf("Options:\n");
  printf("-h, --help         - print this help message\n");
  printf("-v, --version      - print version info and exit\n");
  printf("-b, --build        - build library of your current mp3s to send to server\n");
  printf("-B, --build-only   - build library and exit\n");
  printf("-N, --nobuild      - do not build library, even if it is out of date\n");
  printf("-m, --create       - create a new account with the napster server\n");
  printf("-r, --reconnect    - keep reconnecting until server connection established\n");
  printf("-a, --autorestart  - automatically reconnect when connection to server lost\n");
  printf("-q, --daemon       - run without user interface; file sharing only\n");
  printf("-t, --notitle      - do not display the title bar (fixes messed-up displays)\n");
  printf("-l, --nxterm       - try using a terminal which is compatible with most\n"
	 "                     systems (fixes some messed-up displays)\n");
  printf("-f fn, --config fn - specifies the config file to use (default $HOME/"CONFIGFILE")\n");
  printf("-x fn, --log fn    - log all transfers to a specific filename\n");
  printf("-g fn, --logall fn - log everything to a specific filename\n");
  printf("-s sv, --server sv - select a specific server (multiple -s opts possible)\n");
  printf("-d n, --debug n    - set debug level\n");
  printf("-u str, --user str     - specify napster username\n");
  printf("-p str, --pass str     - specify user's password\n");
  printf("-e str, --email str    - specify user's email address\n");
  printf("-U dir, --upload dir   - specify upload directory (multiple -U opts possible)\n");
  printf("-D dir, --download dir - specify download directory\n");
  printf("-I dir, --incomplete dir - specify directory for incomplete files\n");
  printf("-P n-m, --dataport n-m - specify port(s) to use for incoming upload requests\n");
  printf("-C n, --connection n   - specify connection speed number (see README)\n");
  printf("-M n, --maxuploads n   - specify maximum number of simultaneous uploads\n");
  printf("-o var=value, --option var=value  - set user variable\n");
}

void pversion() {
  printf("nap v"VERSION", a console napster client.\n"
	"Written by Kevin Sullivan. Modified by Peter Selinger and others.\n");
}

void dopts(int argc, char **argv)
{
  int c;
  char *p;
  
  /* defaults */
  info.user = NULL;
  info.pass = NULL;
  info.email = NULL;
  info.up = NULL;
  info.down = NULL;
  info.dataport = NULL;
  info.conn = NULL;
  info.d = "0";
  info.shared_filename = home_file(LIBRARYFILE);
  info.logallfile = NULL;
  info.logfile = NULL;
  info.daemon = 0;
  info.nxterm = 0;
  info.autorestart = 0;
  info.serverlist = NULL;
  info.maxuploads = NULL;
  info.build = 0;
  info.create = 0;
  info.reconnect = 0;
  info.notop = 0;

  while ((c = getopt_long(argc, argv, "h?vbBNmrqatlf:x:g:s:d:u:p:e:U:D:P:C:M:o:", 
			  longopts, NULL)) != -1)
  {
    switch (c)
    {
    case 'q':
      info.daemon=1;
      break;
    case 'a':
      info.autorestart=1;
      break;
    case 'd':
      chset("debug", optarg);
      break;
    case 'b':
      info.build = 1;
      break;
    case 'B':
      info.build = 2;
      break;
    case 'N':
      info.build = -1;
      break;
    case 'm':
      info.create = 1;
      break;
    case 'r':
      info.reconnect = 1;
      break;
    case 'f':
      chset("configfile", optarg);
      break;
    case 't':
      info.notop = 1;
      break;
    case 's':
      if (info.serverlist == NULL) {
	info.serverlist = strdup(optarg);
      } else {
	info.serverlist = (char *)realloc(info.serverlist, strlen(info.serverlist)+strlen(optarg)+2);
	strcat(info.serverlist, ";");
	strcat(info.serverlist, optarg);
      }
      break;
    case 'o':
      p = strchr(optarg, '=');
      if (!p) {
	chset(optarg, "1");
      } else {
	*p++ = 0;
	chset(optarg, p);
      }
      break;
    case 'l':
      info.nxterm = 1;
      break;
    case 'g':
      info.logallfile = strdup(optarg);
      break;
    case 'x':
      info.logfile = strdup(optarg);
      break;
    case '?':
      wp(NULL, "Try --help for more info\n");
      exit(1);
      break;
    case 'h':
      phelp(*argv);
      exit(0);
      break;
    case 'v':
      pversion();
      exit(1);
      break;
    case 'u':
      info.user = strdup(optarg);
      break;
    case 'p':
      info.pass = strdup(optarg);
      break;
    case 'e':
      info.email = strdup(optarg);
      break;
    case 'U':  /* note there can be several -U options, each adding a dir */
      if (!info.up) {
	info.up = strdup(optarg);
	break;
      }
      info.up = realloc(info.up, strlen(info.up)+strlen(optarg)+2);
      if (info.up) {
	strcat(info.up, ";");
	strcat(info.up, optarg);
      }
      break;	   
    case 'D':
      info.down = strdup(optarg);
      break;
    case 'I':
      info.incomplete = strdup(optarg);
      break;
    case 'P':
      info.dataport = strdup(optarg);
      break;
    case 'C':
      info.conn = strdup(optarg);
      break;
    case 'M':
      info.maxuploads = strdup(optarg);
      break;
    default:
      wp(NULL, "Invalid option -- %c\n", c);
      exit(1);
    }
  }

  if (optind < argc) {
    wp(NULL, "%s: unrecognized argument -- %s\n", *argv, argv[optind]);
    wp(NULL, "Try --help for more info\n");
    exit(1);
  }    

  return;
}

/* the main procedure: read command line, parse configuration file,
   check and create various things and then enter event loop; afterwards 
   shut down and quit */

int main(int argc, char *argv[])
{
  int s, n, c, r;
  char *srv;
  char *t = NULL;
  char *serverlist_dup;
  FILE *fl;
  char *fn;

  mscroll = scur = NULL;
  recent = NULL;
  
  dopts(argc, argv);
  
  /* open log files, if any. Note we do this before reading config file, 
   so that logging starts early. */

  fn = getval("configfile");
  if (!fn)
    fn = home_file(CONFIGFILE);
  else
    fn = strdup(fn);

  readcfg(fn);
  
  /* set defaults and prompt user for values that were left unset by
     command line and config file. Note: if --build-only is given, we
     do not need to generate this information. */
  
  if (info.build != 2) {
    r = set_defaults_interactive(fn);
    if (r==-1)
      return(1);
  }

  free(fn);

  if (info.serverlist == NULL) {
    info.serverlist = strdup(std_serverlist);
  }

  if (info.build != -1) {
    if (info.build)
      {
	wp(NULL, "Building library...\n");
	if (buildflist(info.shared_filename, info.up) == -1) {
	  wp(NULL, "There was an error building your library.\n");
	  return(1);
	}
      } else if (!up_to_date(info.shared_filename, info.up))
	{
	  wp(NULL, "Your library is not up-to-date. Rebuilding...\n");
	  if (buildflist(info.shared_filename, info.up) == -1) {
	    wp(NULL, "There was an error building your library.\n");
	    return(1);
	  }
	}
    
    if (info.build == 2)
      return(1);
  }

  fl = fopen(info.shared_filename, "r");
  if (!fl)
  {
    wp(NULL, "Error reading %s: %s\nTry running \"nap -b\".\n", \
       info.shared_filename, sys_errlist[errno]);
    return(1);
  }
  fclose(fl);

  /* check for news on the client */
  if (nvar("nonews")!=1 && !info.daemon) {
    wp(NULL, "Checking for new releases of nap...");
    r = checknv(1);  /* note: news are not currently written to logfile */
    if (r==1) {
      wp (NULL, "Press return to continue.");
      getline(stdin);
    } else if (r==0) {
      wp (NULL, "none found.\n");
    } else {
      wp (NULL, "failed.\n");
    }
  }
  
  /* note: we no longer check whether the download directory is
     writable until we actually try to write something to it */
  
  /* before connecting to the server, let's figure out our port number
     for incoming upload requests. We have to find a port that's not
     in use, and we have to communicate the port number to the server,
     so that's why we do this early rather than later. -PS */

  info.port = initfserv(info.dataport);
  if (info.port == -1) {
    if (strchr(info.dataport, '-')) {
      wp(NULL, "Warning: could not open a port in the range %s: %s\n", info.dataport, sys_errlist[errno]);
    } else {
      wp(NULL, "Warning: could not open port %s: %s\n", info.dataport, sys_errlist[errno]);
    }
    wp(NULL, "Unable to serve incoming upload requests; will proceed as if firewalled\n");
    info.port = 0;
  } else {
    wp(NULL, "Using port %d for incoming client connections\n", info.port);
  }

  serverlist_dup = strdup(info.serverlist);
  srv = strtok(serverlist_dup, ";");
  if (srv==NULL) {
    wp(NULL, "No server specified\n");
    return(1);
  }

  wp(NULL, "Getting best host...\n");
  for (;;)
  {
    /* if we have reached the end of the server list, go back to beginning, 
       if info.reconnect is set - else give up */
    if (srv==NULL) {
      if (info.reconnect) {
	free(serverlist_dup);
	serverlist_dup = strdup(info.serverlist);
	srv = strtok(serverlist_dup, ";");
	sleep(1);
      } else {
	return(1);
      }
    }

    srv = strip(srv);   /* remove whitespace on both ends */
    wp(NULL, "Trying %s\n", srv);
    s = 2;
    while (s == 2)
      s = conn(srv, PORT);
    if (s == -1)
    {
      srv = strtok(NULL, ";");
      continue;
    }
  
    c = info.conn ? atoi(info.conn) : 0;
    if (c < 0 || c > 10)
      c = 0;

    if (info.create)
    {
      wp(NULL, "Creating account...\n");
      n = makeact(s, info.user, info.pass, info.port, CLIENT, c, info.email);
      if (n == -1)
      {
        close(s);
	srv = strtok(NULL, ";");
        continue;
      }
      else 
      {
	info.create = 0;
        break; /* if we just created an account, we are already logged in */
      }
    }
    else
    {
      wp(NULL, "Logging in...\n");
      n = login(s, info.user, info.pass, info.port, CLIENT, c, info.email);
      if (n == -1)
      {
        close(s);
	srv = strtok(NULL, ";");
        continue;
      }
      else 
      {
        break;
      }
    }
  }
  free(serverlist_dup);
  
  /* announce our success */
  {
    struct sockaddr_in dst;
    int frmlen = sizeof(dst);
    if (!getpeername(s, (struct sockaddr *)&dst, &frmlen))
      wp(NULL, "Connected to %s:%i\n", inet_ntoa(dst.sin_addr), ntohs(dst.sin_port));
  }

  signal(SIGCHLD, SIG_IGN);
  signal(SIGPIPE, SIG_IGN);
  signal(SIGWINCH, tresize);
  if (!info.daemon)
    signal(SIGINT, sigint);
#ifdef MCURSES
  signal(SIGCONT, tresize);
#endif

  pipe(ipcs);
  
  addsock(ipcs[0], "ipc", S_R, inipc);
  addsock(s, "server", S_R, inserv);
  if (!info.daemon)
    addsock(0, "input", S_R, input);
  initwin(info.nxterm);
  t = glistn(info.user);
  checkhotlist(s, t);
  free(t);
  loadaliases(home_file(ALIASFILE));
  loadhandlers(home_file(HANDLERFILE));
  if (nvar("savechannels") == 1) {
    char *tmp = NULL, *fn;
    msprintf(&tmp, CHANNELFILE, info.user);
    fn = home_file(tmp);
    free(tmp);

    if (loadchans(s, fn)==-1) {
      wp(wchan, ""RED"* Could not load channels from %s: %s"WHITE"\n", fn, \
	 sys_errlist[errno]);
    } else {
      wp(wchan, "* Requested to join channels from %s\n", fn);
    }
    drw(wchan);
    free(fn);
  }
#ifndef __CYGWIN32__
#ifdef USERCMDS
  libload(USERLIB);
#endif
#endif
  wp(wchan, "%s\n", title);
  drw(wchan);
  drw(winput);
  usleep(400000);

  /* enter the main event loop. */
  sockfunc(wchan, winput);
  
  shutdown(s, 2);
  close(s);
  
  doquit();
  
  return(1);
}

/* ---------------------------------------------------------------------- */
/* functions that deal with reading and writing the library file. */

/* Note: the format of the library file changed in v1.4.4-ps5: two
   lines were added at the beginning: a title, and the "upload" path
   that was used to create the file. This allows us to check whether
   the library needs re-building, see up_to_date(). -PS */

/* "unshare" the old files with the server, then rebuild library, then
   "share" new files. */

int rebuild(int s, char *sd, char *path)
{
#if 0  /* we used to unshare each file individually (0x66), rather
	  than unsharing all of them at once (0x6e). I'll let the code
	  linger here for a while in case there was some reason this
	  was so -PS */

  FILE *f;
  char buf[1024];
  int i;
  char *r;

  f = fopen(sd, "r");
  if (!f) {
    return(-1);
  }

  while (1)
  {
    r = fgets(buf, sizeof(buf), f);
    if (!r)
      break;
    if (*r != '\"')
      continue; /* skip headers and junk */

    for (i=1;r[i];i++)
    {
      if (r[i] == '\"')
      {
        r[i] = 0;
        break;
      }
      if (r[i] == '/')
        r[i] = '\\';
    }
    if (strlen(r))
      sendpack(s, F_DFILE, "\"%s\"", r+1);
  }

#else  /* simpler way to do the same thing. */

  /* tell server to unshare files */
  sendpack(s, F_UNSHARE, NULL);

#endif
  
  if (buildflist(sd, path) == -1)
    return(-1);
  
  if (lfiles(s, sd) == -1)
    return(-1);
  
  return(1);
}

/* check whether the given filename FN is shared, by looking it up in
   the library file SD. This is used for security purposes before
   allowing an upload. Note: filename must use "/" and not "\". Return
   1 if shared, 0 if not. When in doubt, return 0. Note: we compare
   filenames here in a case sensitive manner. This may or may not be
   correct. Pros: we have exact control over which files are shared.
   Cons: dumb remote clients, or dumb servers, may convert filenames
   to lowercase or to uppercase, thus resulting in a request that we
   cannot fulfill. However, I have not seen this happen in practice. */
int isshared(char *fn, char *sd) {
  char buf[1024];
  FILE *f;
  char *r, *p;

  f = fopen(sd, "r");
  if (f == NULL) {
    return(0);
  }
  
  while (1)
  {
    r = fgets(buf, sizeof(buf), f);
    if (!r)
      break;
    if (*r != '\"')
      continue; /* skip headers and junk */
    p = strchr(r+1, '\"');
    if (!p)
      continue;
    *p = 0;
    if (!strcmp(r+1, fn)) {
      fclose(f);
      return 1;
    }
  }
  fclose(f);
  return 0;
}
  
/* build the library file. sd is the name of the library file, and
   path is the ";"-separated sequence of upload directories. */
int buildflist(char *sd, char *path)
{
  FILE *f;
  char sv[513], *t, *n; /* note: don't need malloc for sv */
  char *r;

  f = fopen(sd, "w");
  if (f == NULL)
  {
    wp(NULL, "Error opening \"%s\": %s\n", sd, sys_errlist[errno]);
    return(-1);
  }
  
  fprintf(f, NAP_LIBRARY_HEADER "\n");
  fprintf(f, "PATH=%s\n", path ? path : "");

  r = getcwd(sv, 512);
  if (r == NULL)
  {
    wp(NULL, "Error getting the current working directory: %s\n", 
       sys_errlist[errno]);
    return(-1);
  }
  
  /* note: if path==NULL, we just generate an empty file */

  if (path) {
    n = strdup(path);
    
    t = strtok(n, ";");
    
    while (t)
      {
	t = strip(t);  /* remove whitespace on both ends */
	dir(t, f);
	t = strtok(NULL, ";");
      }
    
    free(n);
  }

  fclose(f);
  
  chdir(sv);
  
  return(1);
}

/* add the files from directory nm to the library file f */
void dir(char *nm, FILE *f)
{
  DIR *t;
  char *p=NULL;
  
  if (nm[strlen(nm)-1] != '/')
    msprintf(&p, "%s/", nm);
  else
    p = strdup(nm);
  
  t = opendir(p);
  if (!t)
    return;
  
  ddir(p, t, f);
  closedir(t);
  free(p);
}

/* add the files from directory p (whose pathname is bpath, ending in
   "/"), to the library file f */
void ddir(char *bpath, DIR *p, FILE *f)
{
  struct dirent *ls;
  DIR *t;
  char *nbpath=NULL;
  mhdr_t *m;
  
  while ((ls = readdir(p)))
  {
    msprintf(&nbpath, "%s%s", bpath, ls->d_name);
    t = opendir(nbpath);  /* a bad way for testing for a directory */
    if (t)
    {
      closedir(t);
      if (strcmp(ls->d_name, "..") && strcmp(ls->d_name, "."))
      {
        free(nbpath);
        nbpath=NULL;
        msprintf(&nbpath, "%s%s/", bpath, ls->d_name);
        dir(nbpath, f);
      }
    }
    else
    {
      m = gethdr(nbpath, 293);
      if (m)
      {
        fprintf(f, "\"%s\" %s-%i %i %i %i %i\n", nbpath, m->check, m->sz1, m->sz, m->bitrate, m->freq, m->len);
        free(m);
      }
    }
    free(nbpath);
    nbpath=NULL;
  }
}

/* send to the server the list of files that we are sharing. SD is the
   library filename.  */
int lfiles(int s, char *sd)
{
  FILE *f;
  int i;
  char buf[512], *pt; /* note: don't need malloc for buf */
  char *r;

  f = fopen(sd, "r");
  if (f == NULL)
  {
    wp(NULL, "Error reading %s: %s\nTry running \"nap -b\".\n", sd,
       sys_errlist[errno]);
    return(-1);
  }
  
  while(1)
  {
    r = fgets(buf, sizeof(buf), f);
    if (!r)
      break;
    if (*r != '\"')
      continue; /* skip headers and junk */

    pt = strchr(buf, '\r');
    if (pt == NULL)
      pt = strchr(buf, '\n');
    if (pt)
      *pt = 0;
    for (i=0;buf[i];i++)
      if (buf[i] == '/')
        buf[i] = '\\';
    if (sendpack(s, F_SFILE, "%s", buf) == -1)  
    {
      wp(NULL, "Connection reset by peer\n");
      return(-1);
    }
  }

  fclose(f);
  
  return(1);
}  

/* decide whether the library file SD is up-to-date relative to PATH.
   Check: nap version number, path, and whether any files in the PATH
   have been modified more recently than library file. Return 1 if
   up-to-date, 0 if not. */
int up_to_date(char *sd, char *path) {
  FILE *f;
  char buf[512];
  char *r, *n, *t;
  struct stat st;
  time_t mtime;
  dev_inode_t *di_list, *e;
  int ret, res;

  if (!sd) {
    return 0;
  }

  f = fopen(sd, "r");
  if (!f) {
    return 0;
  }

  r = fgets(buf, sizeof(buf), f);
  if (!r) {
    fclose(f);
    return 0;
  }

  if (*r && r[strlen(r)-1]=='\n') {
    r[strlen(r)-1] = 0;
  }

  /* note strcmp returns non-zero if strings do *not* match */
  if (strcmp(r, NAP_LIBRARY_HEADER)) { 
    fclose(f);
    return 0;
  }

  r = fgets(buf, sizeof(buf), f);
  if (!r) {
    fclose(f);
    return 0;
  }

  if (*r && r[strlen(r)-1]=='\n') {
    r[strlen(r)-1] = 0;
  }

  if (strncmp(r, "PATH=", 5) || strcmp(r+5, path ? path : "")) { 
    fclose(f);
    return 0;
  }
  
  /* the info in the file seemed okay */
  fclose(f);
  
  /* now check the modification times. */
  ret = stat(sd, &st);
  if (ret)
    return 0;

  mtime = st.st_mtime; /* remember modification time of library file */

  di_list = NULL;
  res = 1;

  /* now check whether each directory on path was modified after mtime */
  if (path) {
    n = strdup(path);
    for (t=strtok(n, ";"); t; t=strtok(NULL, ";")) {
      t = strip(t);
      if (modified_after(mtime, t, &di_list)) {
	res = 0;
	break;
      }
    }
    free(n);
  }
  list_forall_unlink(e, di_list) {
    free(e);
  }
  
  return res;
}

/* check whether file or directory FN was modified at or after time
   MTIME. If it's a directory, also check files and subdirectories
   recursively. Be careful to avoid loops due to symlinks: DI_LIST is
   a linked list of device/inode pairs that we have already seen. */

int modified_after(time_t mtime, char *fn, dev_inode_t **di_list_p) {
  struct stat st;
  int ret;
  dev_inode_t *e;
  DIR *dir;
  struct dirent *dirent;

  ret = stat(fn, &st);
  if (ret) { /* file does not exist? Let's ignore it and hope for the
                best. Note this could be a problem if user deleted an
                entire upload directory. */
    return 0;
  }

  if (st.st_mtime >= mtime) {
    return 1;
  }

  if (!S_ISDIR(st.st_mode)) { /* if it's not a directory, we are done */
    return 0;
  }

  /* do not recurse if we've seen this dir before */
  list_find(e, *di_list_p, e->dev == st.st_dev && e->ino == st.st_ino);
  if (e) {
    return 0;
  }

  /* else remember dir, and recurse */
  e = (dev_inode_t *)malloc(sizeof(dev_inode_t));
  e->dev = st.st_dev;
  e->ino = st.st_ino;
  list_prepend(*di_list_p, e);

  /* now go through each file in this directory. Blah. */

  dir = opendir(fn); /* open directory */
  if (dir==NULL) {   /* if open fails, ignore gracefully */
    return 0;
  }
  while ((dirent = readdir(dir)) != NULL) { /* go through each entry */
    /* make sure to ignore ".." and "." */
    if (strcmp(dirent->d_name, "..")!=0 && strcmp(dirent->d_name, ".")!=0) {
      int len = strlen(fn)+strlen(dirent->d_name)+2;
      char filename[len]; /* Note: don't need alloc. This goes on the stack */
      strcpy(filename, fn);
      strcat(filename, "/");
      strcat(filename, dirent->d_name);
      if (modified_after(mtime, filename, di_list_p)) {
	closedir(dir);
	return 1;
      }
    }
  }

  closedir(dir);
  return 0;
}

/* end functions that deal with library file. */
/* ---------------------------------------------------------------------- */

/* Resolve the given string into an IP address in network byte
   order. The string may either be an IP address in dot notation
   ("127.0.0.1"), or a host name ("server.napster.com"). The answer is
   written into *inp. Return non-zero if the address is valid, 0 (with
   h_errno set) if not. */
int resolve(const char *host, struct in_addr *inp)
{
  struct hostent *hst;
  int r;

  r = inet_aton(host, inp);
  if (r) {
    return r;
  }

  hst = gethostbyname(host);
  if (!hst) {
    return 0;
  } else if (hst->h_addrtype != AF_INET || hst->h_length != 4) {
    /* not an IPv4 address??? */
    h_errno = HOST_NOT_FOUND;
    return 0;
  }
  memcpy(&inp->s_addr, hst->h_addr, 4);
  return 1;
}

/* return the formatted IP address of the remote peer for the given
   socket. The returned string is static and constant and will be
   overwritten with the next call, or the next call to ntoa. */
const char *getpeerip(int fd) {
  int r;
  struct sockaddr_in sa;
  socklen_t l = sizeof(sa);

  r = getpeername(fd, (struct sockaddr *)&sa, &l);
  if (r==-1 || l<sizeof(sa) || sa.sin_family != AF_INET)
    return "unknown";
  else 
    return inet_ntoa(sa.sin_addr);
}

/* swap byte order of a (32 bit) word */
unsigned long int swapl(unsigned long int x) {
  return ((x & 0xff) << 24) | ((x & 0xff00) << 8) | ((x & 0xff0000) >> 8) \
    | ((x & 0xff000000) >> 24);
}

int conn(char *host, unsigned short port)
{
  int s;
  struct sockaddr_in dst;
  char *buf, serv[16];
  int p, i, r;
  
  if (!strchr(host, ':'))
  {
    dst.sin_port = htons(port);
    dst.sin_family = AF_INET;
    r = resolve(host, &dst.sin_addr);
    if (!r)
    {
      noresolv = 1;
      wp(wchan, "Error resolving host %s: %s\n", host, hstrerror(h_errno));
      return(-1);
    }
    
    s = socket(AF_INET, SOCK_STREAM, 0);
    if (s == -1)
    {
      wp(wchan, "Error creating socket: %s\n", sys_errlist[errno]);
      return(-1);
    }
    
    if (connect(s, (struct sockaddr *)&dst, sizeof(dst)) == -1)
    {
      wp(wchan, "Error connecting socket: %s\n", sys_errlist[errno]);
      close(s);
      return(-1);
    }
  
    rsock(s, &buf);
    shutdown(s, 2);
    close(s);
    if (!buf)
    {
      wp(wchan, "Error finding best host\n");
      return(-1);
    }
  }
  else
    buf = strdup(host);

  if (!strncasecmp(buf, "wait", 4))
  {
    sscanf(buf, "%s %i", serv, &p);
    wp(wchan, "Waiting for %i seconds\n", p);
    sleep(p);
    return(2);
  }
  else if (!strncasecmp(buf, "busy", 4))
  {
    wp(wchan, "Busy\n");
    return(2);
  }
  p = atoi(strchr(buf, ':')+1);
  for (i=0;buf[i]!=':';i++)
    serv[i] = buf[i];
  serv[i] = '\0';
  free(buf);
  
  r = resolve(serv, &dst.sin_addr);
  if (!r)
  {
    wp(wchan, "Error resolving host %s: %s\n", serv, hstrerror(h_errno));
    return(-1);
  }
  dst.sin_port = htons(p);
  dst.sin_family = AF_INET;
  
  s = socket(AF_INET, SOCK_STREAM, 0);
  if (s == -1)
  {
    wp(wchan, "Error creating socket: %s\n", sys_errlist[errno]);
    return(-1);
  }
  
  wp(wchan, "Connecting...\n");
  
  if (connect(s, (struct sockaddr *)&dst, sizeof(dst)) == -1)
  {
    wp(wchan, "Error connecting socket: %s\n", sys_errlist[errno]);
    close(s);
    return(-1);
  }
  
  return(s);
}

int login(int s, char *user, char *pass, int data, char *client, int conn, char *email)
{
  char *t1=NULL;
  phead_t *rcv;
  int r;

  msprintf(&t1, "nap v%s", client);
  sendpack(s, F_LOGIN, "%s %s %i \"%s\" %i", user, pass, data, t1, conn);
  free(t1);
  t1 = NULL;
  if ((r = recvpack(s, &t1, &rcv)) == -1)
  {
    wp(wchan, "Error: Connection reset by peer\n");
    return(-1);
  }
  
  if (r == -2)
    while (recvpack(s, &t1, &rcv) == -2);
  
  if (!rcv)
    return(-1);
  
  if (rcv->op == F_LOGERROR)
  {
    wp(wchan, "Error: %s\n", t1);
    free(t1);
    free(rcv);
    return(-1);
  }
  else if (strcasecmp(t1, email))
    wp(wchan, "Email addresses did not match (%s and %s), proceeding anyway...\n", email, t1);
  
  free(t1);
  free(rcv);
  
  return(lfiles(s, info.shared_filename));
}

int makeact(int s, char *user, char *pass, int data, char *client, int conn, char *email)
{
  char *t1=NULL;
  phead_t *rcv;
  int r;
  
  sendpack(s, F_MKUSER, user);
  if ((r = recvpack(s, &t1, &rcv)) == -1)
  {
    wp(NULL, "Error: Connection reset by peer\n");
    return(-1);
  }
  
  if (r == -2)
    while (recvpack(s, &t1, &rcv) == -2);
  
  if (rcv->op == F_UNTK)
  {
    wp(NULL, "Error: Username taken\n");
    free(t1);
    free(rcv);
    return(-1);
  }
  else if (rcv->op == F_UNBAD)
  {
    wp(NULL, "Error: Invalid username\n");
    free(t1);
    free(rcv);
    return(-1);
  }
  else if (rcv->op == F_UNOK)
    wp(NULL, "Registered username\n");
  else
  {
    wp(NULL, "Unknown op: 0x%x\n", rcv->op);
    free(t1);
    free(rcv);
    return(-1);
  }
  
  free(t1);
  free(rcv);
  t1 = NULL;
  
  msprintf(&t1, "nap v%s", client);
  sendpack(s, F_REG, "%s %s %i \"%s\" %i %s", user, pass, data, t1, conn, email);
  free(t1);
  t1 = NULL;
  if ((r = recvpack(s, &t1, &rcv)) == -1)
  {
    wp(NULL, "Error: Connection reset by peer\n");
    return(-1);
  }
  
  if (r == -2)
    while (recvpack(s, &t1, &rcv) == -2);
  
  if (rcv->op == F_LOGERROR)
  {
    wp(NULL, "Error: %s\n", t1);
    free(t1);
    free(rcv);
    return(-1);
  }
  
  free(t1);
  free(rcv);
  
  return(lfiles(s, info.shared_filename));
}

/* calculate the name of the hotlist file */
char *glistn(char *t)
{
  char *r = strdup(t);
  char *r1 = NULL;
  char *r2;
  int i;
  
  for (i=0;r[i];i++)
    r[i] = tolower(r[i]);
  
  msprintf(&r1, HOTLISTFILE, r);
  free(r);

  r2 = home_file(r1);
  free(r1);

  return(r2);
}

/* send initial hotlist during the login process */
void checkhotlist(int s, char *fn)
{
  FILE *f;
  char rd[64];
  hotlist_t *elt;
  
  /* first delete old hlist, if any */
  list_forall_unlink(elt, hlist) {
    free(elt->nm);
    free(elt);
  }

  f = fopen(fn, "r");
  if (!f)
    return;
  
  while (fgets(rd, sizeof(rd), f) != NULL)
  {
    if (!strlen(rd))
      break;
    if (rd[strlen(rd)-1] == '\n')
      rd[strlen(rd)-1] = 0;

    /* create new hotlist entry */
    elt = (hotlist_t *)malloc(sizeof(hotlist_t));
    elt->nm = strdup(rd);
    elt->conn = 0;
    elt->on = 0;
    /* add it to the list */
    list_append(hotlist_t, hlist, elt);
    /* and register it with server */
    sendpack(s, NOTIFY_CHECK, "%s", rd);
  }
  fclose(f);
}

/* Get news on the client. Write it to the given fd. Used by the
   command /news, or on startup. This simply looks at a specific web
   page which is assumed to have such news. Forward compatibility is
   important here, thus every line of that page starts with a keyword,
   and we ignore lines whose keyword we don't. If VERBOSE is set,
   announce the absense of news as well, otherwise be quiet. Note:
   this function may be executed in a child process. Return 1 if there
   was news, 0 if there was none, and -1 or -2 if we could not gain
   access. In case of -1, errno is set, and in case -2, h_errno is
   set. If fd is 1, do some special formatting, write in black and
   white, and send the output through wp() for the log file's
   benefit. */
int checknv(int fd)
{
  int s, r;
  struct sockaddr_in dst;
  FILE *in;
  char buf[1024];
  int relevant, news;
  char *b;
  char *request;

  s = socket(AF_INET, SOCK_STREAM, 0);
  
  dst.sin_port = htons(CLIENTNEWSPORT);
  dst.sin_family = AF_INET;

  r = resolve(CLIENTNEWSHOST, &dst.sin_addr);
  if (r==0)
    return -2;
  r = connect(s, (struct sockaddr *)&dst, sizeof(dst));
  if (r==-1)
    return -1;
  request = "GET "CLIENTNEWSFILE" HTTP/1.0\n\n";
  r = write(s, request, strlen(request));
  if (r==-1)
    return -1;

  in = fdopen(s, "r"); /* it's easier to use fgets */
  
  /* skip HTTP headers */
  while (fgets(buf, sizeof(buf), in) != NULL) {
    b = strip(buf);
    if (*b==0)
      break;
  }
  
  relevant = 0;
  news = 0;
  
  /* parse each line of the file */
  while (fgets(buf, sizeof(buf), in) != NULL) {
    b = strip(buf);
    if (*b == 0 || !strncmp(b, "#", 1)) /* skip comments and blank lines */
      continue;
    if (b[strlen(b)-1] == '\n')
      b[strlen(b)-1] = 0;
    
    if (!strncmp(b, "VERSION ", 8) && !strcmp(b+8, VERSION)) {
      relevant = 1;      /* news for this client */
      continue;
    }
    if (!strncmp(b, "OTHER ", 6) || !strcmp(b, "OTHER")) {
      relevant = 1;
      continue;
    }
    if (relevant && (!strcmp(b, "END") || !strncmp(b, "END ", 4))) {
      break;
    }
    if (relevant && !strncmp(b, "ECHO ", 5)) {
      if (news==0 && fd==1) {
	wp(NULL, "\n**********************************************************************\n");
	wp(NULL, "NEWS:\n");
      }
      if (fd==1) {
	wp(NULL, "%s\n", b+5);
      } else {
	ssock(fd, ""MAGENTA"* %s"WHITE"\n", b+5);
      }
      news = 1;
      continue;
    }
  }
  if (news && fd==1) {
    wp(NULL, "**********************************************************************\n\n");
  }
  return news;
} 

/* save channels from channel list CL to file FN. Return -1 on i/o
   error with errno set, else 0 */
int savechans(chans_t *cl, char *fn)
{
  chans_t *cur;
  FILE *f;
  
  f = fopen(fn, "w");
  if (!f) {
    return -1;
  }

  for (cur=cl;cur;cur=cur->next)
    if (!cur->q)
      fprintf(f, "%s\n", cur->nm);
  
  fclose(f);
  return 0;
}

/* load channels from file FN and join them. Return -1 on i/o error with
   errno set, else 0. S is server's fd. */
int loadchans(int s, char *fn)
{
  char buf[256];
  FILE *f;
  
  f = fopen(fn, "r");
  if (!f) {
    return -1;
  }

  while (1)
  {
    if (fgets(buf, sizeof(buf), f) == NULL)
      break;
    if (buf[strlen(buf)-1] == '\n')
      buf[strlen(buf)-1] = 0;
    if (strlen(buf))
      sendpack(s, F_JOIN, "%s", buf);
  }
  
  fclose(f);
  return 0;
}

chans_t *findchan(chans_t *h, char *chan)
{
  chans_t *cur;

  for (cur=h;;cur=cur->next)
  {
    if (!cur)
      return(NULL);
    if (!strcasecmp(cur->nm, chan))
      return(cur);
  }
}

chans_t *findquery(chans_t *h, char *chan)
{
  chans_t *cur;

  for (cur=h;;cur=cur->next)
  {
    if (!cur)
      return(NULL);
    if (!strcasecmp(cur->nm, chan) && cur->q == 1)
      return(cur);
  }
}

/* for sending a packet to the napster server, whose file descriptor
   should be s. */
int sendpack(int s, unsigned short op, const char *fmt, ...)
{
  char pack[4+4096];  /* note: don't need malloc for data or pack */
  /* first four bytes of pack are header, remaining is data */
  phead_t *hdr;
  va_list args;
  int r;
  
  if (s == -1)
  {
    wp(wchan, "%s* Not connected to the server%s\n", RED, WHITE);
    drw(wchan);
    return(-1);
  }
  
  if (!fmt)
  {
    hdr = (phead_t *)malloc(sizeof(phead_t));
    
    hdr->len = 0;
    hdr->op = op;

    if (nvar("debug") == 2)
    {
      wp(wchan, ""DARK GREEN"--> (0x%x=%d)"WHITE"\n", hdr->op, hdr->op);
      drw(wchan);
    }
    
    r = send(s, (char *)hdr, 4, 0);
    
    return(r);
  }
  
  va_start(args, fmt);
  vsprintf(pack+4, fmt, args);
  va_end(args);
  
  hdr = (phead_t *)pack;
  
  hdr->len = strlen(pack+4);
  hdr->op = op;

  if (nvar("debug") == 2)
  {
    wp(wchan, ""DARK GREEN"--> (0x%x=%d) <%s>"WHITE"\n", hdr->op, hdr->op, quote(pack+4));
    drw(wchan);
  }
  
  r = send(s, pack, hdr->len+4, 0);
  
  return(r);
}

/* write formatted data to file descriptor, and possible log debugging
   info. Return the number of characters written, or -1 on error with
   errno set. */
int ssock(int s, const char *fmt, ...)
{
  char data[4096]; /* note: don't need malloc for data */
  va_list args;
  int r;
  
  va_start(args, fmt);
  vsprintf(data, fmt, args);
  va_end(args);
  
  if (nvar("debug") == 2)
  {
    sock_t *sk = findsockfd(s);
    wp(wchan, ""DARK GREEN"--> [to %d=%s] <%s>"WHITE"\n", s, sk ? sk->nm : "?", quote(data));
    drw(wchan);
  }

  r = write(s, data, strlen(data));
  
  return(r);
}

int recvpack(int s, char **buf, phead_t **hdr)
{
  int r = 0, i = 0;
  fd_set fs;
  struct timeval tv;
  static phead_t *thdr = NULL;
  static unsigned char *tdbuf = NULL;

  if (!thdr)
  {
    thdr = (phead_t *)malloc(sizeof(phead_t));
    memset(thdr, 0, sizeof(phead_t));
  
    while (i < 4)
    {
      r = read(s, ((char *)thdr)+i, sizeof(phead_t)-i);
      if (r <= 0)
      {
        free(thdr);
        thdr = NULL;
        *hdr = NULL;
        *buf = NULL;
        return(-1);
      }
      i += r;
    }
  }
  
  if (!thdr->len)
  {
    *hdr = thdr;
    *buf = strdup("");
    thdr = NULL;
    return(1);
  }
  
  FD_ZERO(&fs);
  FD_SET(s, &fs);
  tv.tv_usec = 0;
  tv.tv_sec = 0;
  
  if (!select(s+1, &fs, NULL, NULL, &tv))
    return(-2);
  
  if (!tdbuf)
  {
    tdbuf = (char *)malloc(thdr->len+1);
    memset(tdbuf, 0, thdr->len+1);
    tpos = 0;
  }
  
  r = read(s, tdbuf+tpos, thdr->len-tpos);
  if (r <= 0)
  {
    free(tdbuf);
    free(thdr);
    tdbuf = NULL;
    thdr = NULL;
    *buf = NULL;
    *hdr = NULL;
    return(-1);
  }
  if ((r+tpos) < thdr->len)
  {
    *buf = NULL;
    *hdr = NULL;
    tpos += r;
    return(-2);
  }
  
/*  memset(tdbuf, 0, 1024);
  r = read(s, tdbuf, thdr->len);
  if (r <= 0)
  {
    free(tdbuf);
    free(thdr);
    tdbuf = NULL;
    thdr = NULL;
    *buf = NULL;
    *hdr = NULL;
    return(-1);
  }
  
  tdbuf = (char *)realloc(tdbuf, strlen(tdbuf)+1); */
  
  *buf = tdbuf;
  *hdr = thdr;
  
  tdbuf = NULL;
  thdr = NULL;
  
  if (nvar("debug") == 2)
  {
    wp(wchan, ""DARK GREEN"<-- (0x%x=%d) <%s>"WHITE"\n", (*hdr)->op, (*hdr)->op, quote(*buf));
    drw(wchan);
  }
  
  return(1);
}

int rsock(int s, char **buf)
{
  int r;
  fd_set fs;

  *buf = (char *)malloc(4096+1);
  
  FD_ZERO(&fs);
  FD_SET(s, &fs);
  
  if (select(s+1, &fs, NULL, NULL, NULL) == -1)
  {
    free(*buf);
    *buf = NULL;
    return(-1);
  }
  
  if ((r = read(s, *buf, 4096)) <= 0)
  {
    free(*buf);
    *buf = NULL;
    return(-1);
  }
  (*buf)[r] = 0;  /* make sure it's 0 terminated */
  *buf = (char *)realloc(*buf, r+1);

  if (nvar("debug") == 2)
  {
    sock_t *sk = findsockfd(s); 
    wp(wchan, ""DARK GREEN"<-- [from %d=%s] <%s>"WHITE"\n", s, sk ? sk->nm : "?", quote(*buf));
    drw(wchan);
  }
  
  return(r);
}

