#ifndef _NAP_CMDS_H
#define _NAP_CMDS_H

/* Copyright (c) 2000  Kevin Sullivan <nite@gis.net>
 *
 * Please refer to the COPYRIGHT file for more information.
 */

#include <sys/time.h>
#include <netinet/in.h>

#include "scheck.h"


struct in_nap_cmd_s 
{
  unsigned short op;
  int (*func)(int, char *, char **, int, WINDOW *);
};
typedef struct in_nap_cmd_s in_nap_cmd_t;

struct out_nap_cmd_s 
{
  const char *nm;
  unsigned char a;
  int (*func)(int, char *, char **, int, WINDOW *);
  const char *help;
};
typedef struct out_nap_cmd_s out_nap_cmd_t;

struct user_s 
{
  char *nm, *addr;
  unsigned char conn;
  unsigned int scount;
  unsigned char flag;
  struct user_s *next;
};
typedef struct user_s user_t;

struct hotlist_s 
{
  char *nm;
  int conn;
  int on;
  struct hotlist_s *next;
};
typedef struct hotlist_s hotlist_t;

/* this is the type of channels as well as queries. A query is a
   (virtual) connection to a single user */
struct chans_s 
{
  char *nm;                   /* name of channel/query - case insensitive */
  char *topic;
  user_t *users;
  unsigned char q;            /* is it a query? */
  unsigned char flag;         /* ? */
  int l;                      /* ? */
  char *key;                  /* ? */
  unsigned char p;            /* ? */
  struct chans_s *next;
}; 
typedef struct chans_s chans_t;

/* the data structure for search result items. */
struct ssearch_s 
{
  int num;                  /* item number in search results list */
  char *song;               /* title with pathname componentes stripped */
  char *fn;                 /* title as sent by server */
  char *cmp;                /* nickname of remote user */
  int conn;                 /* connection speed of remote client */
  unsigned int sz;          /* total size */
  int brate;                /* bitrate (e.g. 128) */
  int freq;                 /* sample frequency (e.g. 44100) */
  int time;                 /* length, in seconds */
  unsigned long ip;         /* ip address of remote client */
  struct timeval s, r;      /* time ping sent, received */
  int ping;                 /* ping time */
  struct ssearch_s *next;   /* next item in list */
};
typedef struct ssearch_s ssearch_t;

/* The task list is one of the two "backbone" data structures of nap
   (the other is the connection list, type sock_t, see scheck.h). A
   "task" is an upload or a download. Tasks are kept in two global
   linked lists, "up" and "down".

   The following explanations are somewhat of a wish list, since they
   have not actually been implemented this way. In the interest of
   adding features such as queued uploads and downloads, the code will
   gradually converge to a clean state model as described here.

   A task can be in several different states. For download tasks,
   these are:
   
   QUEUED: requested by the user, but not yet requested from server.
   REQUESTED: requested from server, but not yet acknowledged by server.
   WAITING: waiting for remote client to contact us.
   CONNECTING: contacted remote client and waiting for a response.
   IN PROGRESS: reading file from the remote client.
   COMPLETED: file successfully received.
   INCOMPLETE: we were disconnected after receiving a partial file.
   FAILED: error before receiving file (e.g. it did not exist)
   TIMED OUT: waited too long for server or remote client.

   For upload tasks, the possible states are very similar:

   WAITING: waiting for remote client to contact us.
   CONNECTING: contacted remote client and waiting for a response.
   IN PROGRESS: sending file to the remote client.
   COMPLETED: file successfully sent.
   INCOMPLETE: disconnected after sending partial file.
   FAILED: error before sending file (e.g. it did not exist)
   TIMED OUT: waited too long for server or remote client.

   Note that there are no QUEUED or REQUESTED states for uploads,
   since uploads are always initiated by the server.

   Not all of the above states need to be explicitly represented in
   the data structures. For instance, one may choose to implement
   COMPLETED, INCOMPLETE, FAILED, and TIMED OUT simply by deleting the
   task. On the other hand, a clever client might want to try to
   something smart with an INCOMPLETE, for instance, try to resume it. 

   Much of the napster protocol has to do with what messages to expect
   and send in each state, and how to change states. The following
   model explains how to react to each condition in each state.
   This could be represented as a flow graph, but that's difficult to
   do in ASCII. 

   -------------------------------------------------------------------

   1. Downloads.

   [no state]: user download request (if limit exceeded) -> create
   task -> QUEUED

   [no state]: user download request (if limit ok) -> create task ->
   send 203 (download request) to server -> REQUESTED

   QUEUED: (when a slot becomes available) -> send 203 (download
   request) to server -> REQUESTED

   REQUESTED: receive 206 (get error) from server -> FAILED

   REQUESTED: receive 609 (accept failure) from server -> FAILED

   REQUESTED: receive 620 (queue limit) from server -> FAILED (or
   maybe back to QUEUED?)

   REQUESTED: receive 204 (download ack) from server (if remote party
   is firewalled) -> send 500 (alt download request) to server -> WAITING

   REQUESTED: receive 204 (download ack) from server (if remote party
   is not firewalled) -> contact remote client -> CONNECTING

   REQUESTED: (timed out) -> TIMED OUT

   WAITING: remote client connects to us (if negotiation fails) -> FAIL

   WAITING: remote client connects to us (if negotiation succeeds) ->
   send 218 (downloading) to sever -> IN PROGRESS

   WAITING: (timed out) -> TIMED OUT

   CONNECTING: remote client responds (if negotiation fails) -> FAIL

   CONNECTING: remote client responds (if negotiation succeeds) -> 
   send 218 (downloading) to sever -> IN PROGRESS

   CONNECTING: (timed out) -> TIMED OUT

   IN PROGRESS: receive data -> IN PROGRESS

   IN PROGRESS: (transfer complete) -> send 219 (download complete) to
   server -> COMPLETED

   IN PROGRESS: (transfer interrupted) -> send 219 (download complete) to
   server -> INCOMPLETE

   FAILED or COMPLETED or INCOMPLETE or TIMEOUT: -> remove task -> [no state]

   -------------------------------------------------------------------

   2. Uploads.

   [no state]: receive 607 (upload request) from server (if we don't
   have the file) -> send 608 (accept upload request) to server -> FAILED

   (Note: there is no "reject upload request" packet!)

   [no state]: receive 607 (upload request) from server (if upload
   limit exceeded) -> send 619 (queue limit) to server -> FAILED

   [no state]: receive 607 (upload request) from server (otherwise) ->
   create task -> send 608 (accept upload request) to server -> WAITING

   WAITING: remote client connects to us (if negotiation fails) -> FAILED

   WAITING: remote client connects to us (if negotiation succeeds) ->
   send 220 (uploading) to server and go to IN PROGRESS

   WAITING: receive 501 (alt download ack or "push request") from
   server -> contact remote client -> CONNECTING

   WAITING: (timed out) -> TIMED OUT

   (Note: apparently, 501 is only sent after 607. But we may also
   choose to accept 501 in [no state].)

   CONNECTING: remote client responds (if negotiation fails) -> FAILED

   (Note: there might be no harm in allowing an upload of a shared
   file even if it does not seem to have been initiated by the server,
   *provided* that we have space available. Usually this happens if a
   remote client sent an upload request, we received it, then we quit
   and restarted. The remote client will now try to connect to us, but
   we have erased our upload list!)

   CONNECTING: remote client responds (if negotiation succeeds) ->
   send 220 (uploading) to server -> IN PROGRESS

   CONNECTING: (timed out) -> TIMED OUT

   IN PROGRESS: send data -> IN PROGRESS

   IN PROGRESS: (transfer complete) -> send 221 (upload complete) to
   server -> COMPLETE

   IN PROGERSS: (disconnected) -> send 221 (upload complete) to server
   -> INCOMPLETE

   FAILED or COMPLETED or INCOMPLETE or TIMEOUT: -> remove task -> [no state]

   -------------------------------------------------------------------

   Some states may split into even more sub-states, for instance,
   CONNECTING usually involves first sending a GET and then waiting
   one more time for the remote client's response. This is implemented
   by remaining in the same state, but updating the (sock_t) "func"
   entry.

*/

/* this is the "old" task structure */

struct file_s 
{
  unsigned long addr;          /* remote IP (network byte order) */
  unsigned short port;         /* remote port (host byte order) */
  time_t st;                   /* time when ONGOING started, or 0 */ 
  FILE *f;                     /* local file, open for reading or writing */
  char *lfn;                   /* local file name, downloads only */
  char *fn;                    /* uploads: full name of local file,
				  downloads: stripped filename */
  char *nm;                    /* nick of remote user */
  char *rfn;                   /* uploads: stripped filename(?),
				  downloads: remote file name */ 
  char *check;                 /* remote client's md5 hash (downloads), or 
				  NULL */
  void *timer;                 /* timer for TIMEOUT in some states */
  size_t sz, csz, bsz;         /* sz: total bytes to send/get, csz: bytes
				  already sent/received, bsz: first
			          byte requested */
  unsigned char g;             /* apparently 1 iff timer is set */
  sock_t *sk;                  /* this task's client-client connection */
  struct file_s *next;         /* next item in list */
};
typedef struct file_s file_t;

/* this is the "new" task structure, implementing the above model.
   There is no harm and much benefit to having separate structures for
   uploads and downloads, even though they overlap somewhat. */

struct download_s
{
  int state;                   /* QUEUED, WAITING, etc */
  
  /* in all states: */

  ssearch_t *sres;             /* a copy of the original search result
				  which prompted this download. Note
				  this may or may not contain
				  linespeed or ping info. */
  char *nick;                  /* nickname of remote user, same as sres->nm */
  char *rfilename;             /* remote filename, same as sres->fn.
				  The fields (nick,rfilename) must
				  uniquely determine a download task,
				  since this is how the server names it. */
  char *basename;              /* filename without directory
				  components, same as sres->song */
  time_t cre_t;                /* time created */
  time_t mod_t;                /* time of the last state change */

  /* in any state after 204 was received (WAITING, CONNECTING, IN
     PROGRESS, FAILED, COMPLETED, INCOMPLETE, TIMEOUT): */

  struct sockaddr_in addr;     /* remote IP and port (see ip(4)) */
  unsigned char md5[16];       /* remote client's MD5 hash */
  int linespeed;               /* remote client's linespeed */

  /* in state CONNECTING and IN PROGRESS */
  
  sock_t sk;                   /* connection for client-to-client transfer */

  /* in state IN PROGRESS (filled during negotiation) */
  char *filename;              /* local filename to write to */
  FILE *f;                     /* local file to write to */
  size_t size;                 /* file size as reported by remote client */
  size_t first;                /* first byte to get */
  size_t pos;                  /* position of next byte to get */

  struct download_s *next;
};
typedef struct download_s download_t;

struct scount_s 
{
  unsigned long libraries, songs, gigs;
};
typedef struct scount_s scount_t;

#define ICMP_ECHO 8
struct icmphdr 
{
  unsigned char type, code;
  unsigned short checksum;
  union
  {
    struct
    {
      unsigned short id, sequence;
    } echo;
    unsigned long gateway;
    struct
    {
      unsigned short blah, mtu;
    } frag;
  } un;
};



#define I_NAP_FUNC(x) int (x)(int s, char *str, char **tok, int num, WINDOW *win)

I_NAP_FUNC(sjoin);
I_NAP_FUNC(spart);
I_NAP_FUNC(stopic);
I_NAP_FUNC(ssay);
I_NAP_FUNC(snchan);
I_NAP_FUNC(sjchan);
I_NAP_FUNC(nothing);
I_NAP_FUNC(snotice);
I_NAP_FUNC(sscount);
I_NAP_FUNC(stell);
I_NAP_FUNC(suser);
I_NAP_FUNC(swhois);
I_NAP_FUNC(soff);
I_NAP_FUNC(sget);
I_NAP_FUNC(snget);
#ifdef QUEUE
I_NAP_FUNC(snacc);
I_NAP_FUNC(sqlimit);
#endif /* QUEUE */
I_NAP_FUNC(sfreq);
I_NAP_FUNC(sry);
I_NAP_FUNC(ssf);
I_NAP_FUNC(smpart);
I_NAP_FUNC(suon);
I_NAP_FUNC(suoff);
I_NAP_FUNC(ssret);
I_NAP_FUNC(ssend);
I_NAP_FUNC(srbrowse);
I_NAP_FUNC(sdbrowse);
I_NAP_FUNC(sop);
I_NAP_FUNC(spchange);
I_NAP_FUNC(sbport);
I_NAP_FUNC(sannounce);
I_NAP_FUNC(sbanlist);
I_NAP_FUNC(scping);
I_NAP_FUNC(scpong);
I_NAP_FUNC(ssping);
I_NAP_FUNC(sredir);
I_NAP_FUNC(scycle);
I_NAP_FUNC(sclist);
I_NAP_FUNC(sclist2);
I_NAP_FUNC(snend);
I_NAP_FUNC(sblocklist);
I_NAP_FUNC(signorelist);
I_NAP_FUNC(signoreend);
I_NAP_FUNC(signoreadd);
I_NAP_FUNC(signoreremove);
I_NAP_FUNC(signoreunknown);
I_NAP_FUNC(signoreexists);
I_NAP_FUNC(signoreclear);
I_NAP_FUNC(signorefail);
I_NAP_FUNC(scbanlist);
I_NAP_FUNC(snerr);
I_NAP_FUNC(snadd);
I_NAP_FUNC(sme);


#define O_NAP_FUNC(x) int (x)(int s, char *str, char **tok, int num, WINDOW *win)

O_NAP_FUNC(ddebug);
O_NAP_FUNC(djoin);
O_NAP_FUNC(dpart);
O_NAP_FUNC(dquit);
O_NAP_FUNC(dtell);
O_NAP_FUNC(dwhois);
O_NAP_FUNC(dget);
O_NAP_FUNC(dpup);
O_NAP_FUNC(dpdown);
O_NAP_FUNC(dsearch);
O_NAP_FUNC(dpvars);
O_NAP_FUNC(dkill);
O_NAP_FUNC(dban);
O_NAP_FUNC(dunban);
O_NAP_FUNC(ddatap);
O_NAP_FUNC(dtopic);
O_NAP_FUNC(dlevel);
O_NAP_FUNC(dg);
#ifdef QUEUE
O_NAP_FUNC(dforceq);
#endif /* QUEUE */
O_NAP_FUNC(dopsay);
O_NAP_FUNC(ddns);
O_NAP_FUNC(dannounce);
O_NAP_FUNC(pchans);
O_NAP_FUNC(dbanlist);
O_NAP_FUNC(dlspeed);
O_NAP_FUNC(ddup);
O_NAP_FUNC(ddown);
O_NAP_FUNC(dpsocks);
O_NAP_FUNC(dmuzzle);
O_NAP_FUNC(dunmuzzle);
O_NAP_FUNC(ddisconnect);
O_NAP_FUNC(dreload);
O_NAP_FUNC(dreconnect);
O_NAP_FUNC(dbrowse);
O_NAP_FUNC(dhelp);
O_NAP_FUNC(dping);
O_NAP_FUNC(duserpass);
O_NAP_FUNC(dreloadconf);
O_NAP_FUNC(dsver);
O_NAP_FUNC(dsetconf);
O_NAP_FUNC(dchanclear);
O_NAP_FUNC(dsetlevel);
O_NAP_FUNC(dme);
O_NAP_FUNC(dusers);
O_NAP_FUNC(dgusers);
O_NAP_FUNC(dclist);
O_NAP_FUNC(dclist2);
O_NAP_FUNC(dclear);
O_NAP_FUNC(dnews);
O_NAP_FUNC(dsraw);
O_NAP_FUNC(dquery);
O_NAP_FUNC(dcloak);
O_NAP_FUNC(dblocklist);
O_NAP_FUNC(dblock);
O_NAP_FUNC(dunblock);
O_NAP_FUNC(dwindow);
O_NAP_FUNC(dignore);
O_NAP_FUNC(dunignore);
O_NAP_FUNC(dignoreclear);
O_NAP_FUNC(dignorelist);
O_NAP_FUNC(dalias);
O_NAP_FUNC(dunalias);
O_NAP_FUNC(daliaslist);
O_NAP_FUNC(dhandler);
O_NAP_FUNC(dunhandler);
O_NAP_FUNC(dhandlerlist);
O_NAP_FUNC(dcbanlist);
O_NAP_FUNC(dcban);
O_NAP_FUNC(dcunban);
O_NAP_FUNC(dcbanclear);
O_NAP_FUNC(dkick);
O_NAP_FUNC(dloadalias);
O_NAP_FUNC(dsavealias);
O_NAP_FUNC(dclearalias);
O_NAP_FUNC(dloadhandler);
O_NAP_FUNC(dsavehandler);
O_NAP_FUNC(dclearhandler);
O_NAP_FUNC(dwstats);
O_NAP_FUNC(dnotify);
O_NAP_FUNC(dunnotify);
O_NAP_FUNC(dhotlist);
O_NAP_FUNC(dsay);
O_NAP_FUNC(dhup);
O_NAP_FUNC(dtquit);
O_NAP_FUNC(dabout);
O_NAP_FUNC(dirc);
O_NAP_FUNC(dkickall);
O_NAP_FUNC(deval);
O_NAP_FUNC(dset);
O_NAP_FUNC(dunset);
O_NAP_FUNC(dsaveconfig);
O_NAP_FUNC(dloadconfig);
O_NAP_FUNC(dsavechannels);
O_NAP_FUNC(dloadchannels);
O_NAP_FUNC(decho);
O_NAP_FUNC(dresults);
O_NAP_FUNC(dexec);
O_NAP_FUNC(dtimer);
O_NAP_FUNC(dif);
O_NAP_FUNC(ddone);
O_NAP_FUNC(dwhile);
O_NAP_FUNC(dinc);
O_NAP_FUNC(ddec);
O_NAP_FUNC(dbreak);
O_NAP_FUNC(dlastlog);
O_NAP_FUNC(drebuild);
O_NAP_FUNC(dupdate);
O_NAP_FUNC(dchupload);
O_NAP_FUNC(dtlist);
O_NAP_FUNC(dnoprint);
O_NAP_FUNC(dstop);
O_NAP_FUNC(dserv);
O_NAP_FUNC(dserver);
O_NAP_FUNC(dupgrade);


char *fixquotes(char *s);
void upchan(chans_t *);
int in_cksum(u_short *, int);
void tvsub(register struct timeval *, register struct timeval *);
int icmpin(WINDOW *, sock_t *);
char **form_toks(char *, int *);
char **form_tokso(char *, int *);
int parsein(int, unsigned short, char *, WINDOW *);
char **fxv(char **, int, int *);
int parseout(int, char *, WINDOW *);
int parseoutn(int, char *, WINDOW *);
int gnum(unsigned char);
char *cstr(char *, int);
char *cspstr(char **, int, int);
int fcnt(file_t *);
void dupload(file_t *);
void ddownload(WINDOW *win, file_t *);
void move_to_downloaddir(WINDOW *win, file_t *task);
void remove_turd(WINDOW *win, file_t *task, int turdsize);
void mark_incomplete(WINDOW *win, file_t *task);
void tagincompletefile(char *fn, char *str);
file_t *ffile(file_t *, char *, char *);
void adduser(chans_t *, char *, int, unsigned char);
void deluser(chans_t *, char *);
user_t *finduser(chans_t *, char *);

#ifdef QUEUE
/* stuff added for download queuing */
int enqdown(ssearch_t *x);
void deqdown(ssearch_t *x);
int deqdown2(char *nm, char *fn);
int downqhook(char *nick);
/* end of download queuing stuff */
#endif /* QUEUE */
#endif
