/* Copyright (c) 2000  Kevin Sullivan <nite@gis.net>
 *
 * Please refer to the COPYRIGHT file for more information.
 */

#include <stdio.h>
#include <time.h>
#include <stdarg.h>
#include <termios.h>
#include <dirent.h>
#include <fcntl.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#ifndef MCURSES
  #include <ncurses.h>
#endif
#include <stdlib.h>
#include <signal.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <sys/time.h>
#include <netdb.h>
#include <errno.h>

#include "getopt.h"
#include "defines.h"
#include "codes.h"
#include "colors.h"
#include "title.h"
#include "mp3s.h"
#include "scheck.h"
#include "cmds.h"
#include "timer.h"
#include "alias.h"
#include "nap.h"
#include "winio.h"

#ifdef MCURSES
#include "wind.h"
#endif

/* standard napster servers */
char *std_serverlist = "server.napster.com;64.124.41.16;64.124.41.17;"
  "64.124.41.18;64.124.41.19";

extern file_t *up, *down;
extern unsigned char ircin, wmode, finp;
extern int ircsock;
extern unsigned char ups[];
extern int curx;
extern scrls_t *mscroll, *scur;
extern chans_t *recent;
extern cmds_t *cmdl, *ccmd;
extern WINDOW *wchan, *winput;

chans_t *chanl = NULL, *curchan = NULL;
struct link_t *hlist = NULL;
phead_t *thdr = NULL;
time_t cct = 0;
info_t info;          /* struct to hold some command line options */
int cloaked = 0;      /* whether we are cloaked */
unsigned char lpbrk=0, noprint=0, noresolv=0;
unsigned char *tdbuf = NULL;
char tpbuf[512];
int tpos, ipcs[2];
unsigned char dresize = 0, force = 0;

void ctrlc(int dummy)
{
  doquit();
}

void doquit()
{
  scur = NULL;
  if (wchan)
  {
    dscr(wchan);
    drw(wchan);
  }
  if (ircin)
  {
    ssock(ircsock, "QUIT :Leaving\n");
    close(ircsock);
    ircsock = 0;
  }
  endwin();
  if (nvar("savechannels") == 1)
    savechans(chanl);
  exit(1);
}

void tresize(int dummy)
{
  dresize = 1;
}

void sigint(int dummy)
{
  if ((time(NULL)-cct) <= 1)
    doquit();
  
  lpbrk = 1;
  cct = time(NULL);
}

void dochecks()
{
  if (dresize)
  {
    dresize = 0;
    resize();
  }
}

/* replaces all non-printable characters in a string s by '?'. s must
   be null-terminated. */
void to_printable(char *s) {
  while (*s) {
    if (*s<32 || *s>126)
      *s = '?';
    s++;
  }
}

/* returns the full path of file fn relative to user's home directory */
char *home_file(char *fn) {
  char *home, *res;
  int len;

  if (fn[0] == '/')         /* absolute filename? */
    return strdup(fn);

  home = getenv("HOME");   /* else, try to find home directory */
  if (!home)
    return strdup(fn);

  len = strlen(home);
  res = (char *)malloc(len+strlen(fn)+2);
  strcpy(res, home);
  if (home[len-1] != '/')
    strcat(res, "/");
  strcat(res, fn);
  return res;
}

/* read an allocated line from stdin */

char *getline() {
  char acc[513];
  char *res;
  
  res = fgets(acc, 512, stdin);
  if (!res)
    return NULL;

  /* strip trailing newline */
  if (acc[strlen(acc)-1]=='\n')
    acc[strlen(acc)-1] = 0;
  return strdup(acc);
}

/* Read the config file. Values that were given on the command line
   override those in the config file - don't read them. Try to prompt
   user for missing values, or assume reasonable defaults.
   Note: if user or password or email were given on command line, then
   do not read any of these three values from config file. */

int readcfg(char *fn)
{
  FILE *f;
  char *t;
  char *config_file = fn;
  
  /* if filename not given, use default */
  if (!fn)
    config_file = home_file(CONF);

  /* try to open config file */
  f = fopen(config_file, "r");

  /* complain of failure only if config filename was given explicitly */
  if (!f && fn) {
    wp(NULL, "Cannot read %s: %s\n", fn, sys_errlist[errno]);
    return(-1);
  }
    
  /* read what we can from config file */
  if (f)
  {
    wp(NULL, "Reading %s...\n", config_file);

    if (!info.user && !info.pass && !info.email) {
      info.user = findent(f, "user");
      info.pass = findent(f, "pass");
      info.email = findent(f, "email");
    }
    if (!info.up) {
      info.up = findent(f, "upload");
    }
    if (!info.down) {
      info.down = findent(f, "download");
    }
    if (info.port == -1) {
      t = findent(f, "dataport");
      if (t) {
	info.port = atoi(t);
	if (strchr(t, '-')) {
	  info.port_max = atoi(strchr(t, '-')+1);
	} else {
	  info.port_max = info.port;
	}
	free(t);
      }
    }
    if (info.conn == -1) {
      t = findent(f, "connection");
      if (t) {
	info.conn = atoi(t);
	free(t);
      }
    }
    if (info.maxuploads == -1) {
      t = findent(f, "maxuploads");
      if (!t)
	t = findent(f, "maxusers");   /* for backward compatibility */
      if (t) {
	info.maxuploads = atoi(t);
	free(t);
      }
    }
    fclose(f);
  }
  /* next, try to guess defaults or prompt user for any unknowns */
  if (!info.user || !*info.user) {
    printf("User: ");
    info.user = getline();
    if (!info.user || !*info.user) {
      wp(NULL, "No user name given\n");
      return(-1);
    }
  }
  if (!info.pass || !*info.pass || !strcmp(info.pass, "?")) {
    printf("Password for user %s: ", info.user);
    t = getpass("");
    if (!t || !*t) {
      wp(NULL, "No password given\n");
      return(-1);
    }
    info.pass = strdup(t);
    memset(t, 0, strlen(t));
  }
  if (!info.email || !*info.email) {
    printf("Email for user %s: ", info.user);
    info.email = getline();
    if (!info.email || !*info.email) {
      info.email=strdup("user@localhost");
      printf("No email given - trying %s\n", info.email);
    }
  }
  if (!info.up || !*info.up) {
    wp(NULL, "No upload directories given - unable to share files\n");
    info.up = strdup("");
  }
  if (!info.down || !*info.down) {
    wp(NULL, "No download directory given - using current working directory\n");
    info.down = (char *)malloc(513);
    getcwd(info.down, 512);
    if (!info.down) {
      wp(NULL, "Error getting the current working directory: %s\n", sys_errlist[errno]);
      return(-1);
    }
  }
  if (info.port == -1) {
    info.port = 6699;
    info.port_max = 6799;
    wp(NULL, "No dataport given - using %d-%d\n", info.port, info.port_max);
  }
  if (info.conn == -1) {
    printf("
          Connection | Number
          -------------------
          Unknown    |  0
          14.4       |  1
          28.8       |  2
          33.6       |  3
          56.7       |  4
          64K ISDN   |  5
          128K ISDN  |  6
          Cable      |  7
          DSL        |  8
          T1         |  9
          T3 or >    | 10
\n");
    printf("How fast is your internet connection? Please choose 0--10 from the list: ");
    t = getline();
    if (t) {
      info.conn = atoi(t);
      free(t);
    }
  }
  if (info.conn < 0 || info.conn > 10) {
    info.conn = 0;
    wp(NULL, "Invalid connection given - using 0\n");
  }
  if (info.maxuploads == -1) {
    info.maxuploads = ups[info.conn];
    wp(NULL, "No maxuploads given - allowing %d slots\n", info.maxuploads); 
  }
  return(1);
}

void checkhotlist(int s, char *fn)
{
  FILE *f;
  char rd[64];
  hotlist_t *cur;
  
  if (hlist)
    dellist(&hlist);
  
  f = fopen(fn, "r");
  if (!f)
    return;
  
  while (!feof(f))
  {
    memset(rd, 0, sizeof(rd));
    fgets(rd, sizeof(rd), f);
    if (!strlen(rd))
      break;
    rd[strlen(rd)-1] = 0;
    cur = (hotlist_t *)malloc(sizeof(hotlist_t));
    cur->nm = strdup(rd);
    cur->conn = 0;
    cur->on = 0;
    addlink(&hlist, cur);
    sendpack(s, NOTIFY_CHECK, "%s", rd);
  }
  fclose(f);
}

int isshared(char *nm)
{
  char buf[1024];  /* note: don't need malloc for buf */
  FILE *f;
  int i;
  
  f = fopen(info.shared_filename, "r");
  if (f == NULL)
  {
    return(0);
  }
  
  for (;;)
  {
    memset(buf, 0, 1024);
    if (fgets(buf, 1024, f) == NULL)
    {
      fclose(f);
      return(0);
    }
    for (i=1;buf[i]!='\"';i++)
      if (buf[i] == '\\')
        buf[i] = '/';
    if (!strncasecmp(buf+1, nm, strlen(nm)) && buf[1+strlen(nm)] == '\"')
    {
      fclose(f);
      return(1);
    }
  }
  
  return(0); /* not reached */
}

/* find value for given key in file f of key=value pairs. Ignore
   comments and whitespace */

char *findent(FILE *f, char *key)
{
  char buf[256]; /* don't need malloc - use stack! */
  char *pt, *pt2;
  int i, j;
  int keylen = strlen(key);

  fseek(f, 0, SEEK_SET);

  while (1)
  {
    memset(buf, 0, 256);
    if (fgets(buf, 256, f) == NULL)
      return(NULL);
    pt = buf;
    while (*pt == ' ' || *pt == '\t')   /* skip whitespace */
      pt++;
    if (*pt == '#')                     /* skip comments */
      continue;
    if (strncasecmp(pt, key, keylen))   /* skip non-matches */
      continue;
    pt += keylen;
    while (*pt == ' ' || *pt == '\t')   /* skip whitespace */
      pt++;
    if (*pt != '=')                     /* next character not '=', skip */
      continue;
    pt++;
    while (*pt == ' ' || *pt == '\t')   /* skip whitespace from value */
      pt++;
    pt2 = pt+strlen(pt)-1;
    while (*pt2 == ' ' || *pt2 == '\t' || *pt2 == '\n') 
      pt2--;                            /* trim trailing whitespace */

    pt2[1]=0;
    return strdup(pt);
  }
}

/* print to a newly allocated string *str. If *str is non-NULL, free 
   the old content */
int msprintf(char **str, char *fmt, ...)
{
  va_list args;
  char buf[4096];
  
  free(*str);
  
  va_start(args, fmt);
  vsprintf(buf, fmt, args);
  va_end(args);
  
  *str = strdup(buf);
  
  return(strlen(*str));
}

/* count the number of occurences of character t in string buf */
int strcnt(char *buf, char t)
{
  int i,r;
  
  for (i=0,r=0;buf[i];i++)
    if (buf[i] == t)
      r++;
  
  return(r);
}

static struct option longopts[] = {
  {"help",        0, 0, 'h'},
  {"version",     0, 0, 'v'},
  {"daemon",      0, 0, 'q'},
  {"autorestart", 0, 0, 'a'},
  {"build-only",  0, 0, 'B'},
  {"build",       0, 0, 'b'},
  {"notitle",     0, 0, 't'},
  {"reconnect",   0, 0, 'r'},
  {"nxterm",      0, 0, 'l'},
  {"create",      0, 0, 'm'},
  {"config",      1, 0, 'f'},
  {"server",      1, 0, 's'},
  {"debug",       1, 0, 'd'},
  {"log",         1, 0, 'x'},
  {"logall",      1, 0, 'g'},
  {"user",        1, 0, 'u'},
  {"pass",        1, 0, 'p'},
  {"password",    1, 0, 'p'},
  {"email",       1, 0, 'e'},
  {"upload",      1, 0, 'U'},
  {"download",    1, 0, 'D'},
  {"dataport",    1, 0, 'P'},
  {"connection",  1, 0, 'C'},
  {"maxuploads",    1, 0, 'M'},
  {0, 0, 0, 0}
};

void phelp(char *nm)
{
  printf("Usage: %s [options]\n", nm);
  printf("Options:\n");
  printf("-h, --help         - print this help message\n");
  printf("-v, --version      - print version info and exit\n");
  printf("-b, --build        - build library of your current mp3s to send to server\n");
  printf("-B, --build-only   - build library and exit\n");
  printf("-m, --create       - create new account from the information in config file\n");
  printf("-r, --reconnect    - keep reconnecting until server connection established\n");
  printf("-a, --autorestart  - automatically reconnect when connection to server lost\n");
  printf("-q, --daemon       - run without user interface; file sharing only\n");
  printf("-t, --notitle      - do not display the title bar (fixes messed-up displays)\n");
  printf("-l, --nxterm       - try using a terminal which is compatible with most\n"
	 "                     systems (fixes messed-up displays)\n");
  printf("-f fn, --config fn - specifies the config file to use (default $HOME/"CONF")\n");
  printf("-x fn, --log fn    - log all transfers to a specific filename\n");
  printf("-g fn, --logall fn - log everything to a specific filename\n");
  printf("-s sv, --server sv - select a specific server (multiple -s opts possible)\n");
  printf("-d n, --debug n    - set debug level\n");
  printf("-u str, --user str     - specify Napster username\n");
  printf("-p str, --pass str     - specify user's password\n");
  printf("-e str, --email str    - specify user's email address\n");
  printf("-U dir, --upload dir   - specify upload directory (multiple -U opts possible)\n");
  printf("-D dir, --download dir - specify download directory\n");
  printf("-P n-m, --dataport n-m - specify port(s) to use for incoming upload requests\n");
  printf("-C n, --connection n   - specify connection speed number (see README)\n");
  printf("-M n, --maxuploads n   - specify maximum number of simultaneous uploads\n");
}

void pversion() {
  printf("nap v"VERSION", written by Kevin Sullivan. "
	 "Modified by Peter Selinger and others.\n");
}

void dopts(int argc, char **argv)
{
  int c;
  
  /* defaults */
  info.user = NULL;
  info.pass = NULL;
  info.email = NULL;
  info.up = NULL;
  info.down = NULL;
  info.port = -1;
  info.port_max = -1;
  info.conn = -1;
  info.d = 0;
  info.shared_filename = home_file(SFILE);
  info.config_filename = NULL;
  info.logallfile = NULL;
  info.logfile = NULL;
  info.daemon = 0;
  info.autorestart = 0;
  info.serverlist = NULL;
  info.maxuploads = -1;
  info.build = 0;
  info.create = 0;
  info.reconnect = 0;
  info.notop = 0;

  while ((c = getopt_long(argc, argv, "h?vbBmrqatlf:x:g:s:d:u:p:e:U:D:P:C:M:", 
			  longopts, NULL)) != -1)
  {
    switch (c)
    {
      case 'q':
	info.daemon=1;
	break;
      case 'a':
	info.autorestart=1;
	break;
      case 'd':
        info.d = atoi(optarg);
        break;
      case 'B':
        info.build = 2;
        break;
      case 'b':
        info.build = 1;
        break;
      case 'm':
        info.create = 1;
        break;
      case 'r':
        info.reconnect = 1;
        break;
      case 'f':
        info.config_filename = strdup(optarg);
        break;
      case 't':
        info.notop = 1;
        break;
      case 's':
	if (info.serverlist == NULL) {
	  info.serverlist = strdup(optarg);
	} else {
	  info.serverlist = (char *)realloc(info.serverlist, strlen(info.serverlist)+strlen(optarg)+2);
	  strcat(info.serverlist, ";");
	  strcat(info.serverlist, optarg);
	}
        break;
      case 'l':
        force = 1;
        break;
      case 'g':
	info.logallfile = strdup(optarg);
        break;
      case 'x':
	info.logfile = strdup(optarg);
        break;
      case '?':
	wp(NULL, "Try --help for more info\n");
	exit(1);
	break;
      case 'h':
        phelp(*argv);
	exit(0);
        break;
      case 'v':
	pversion();
	exit(1);
	break;
      case 'u':
	info.user = strdup(optarg);
	break;
      case 'p':
	info.pass = strdup(optarg);
	break;
      case 'e':
	info.email = strdup(optarg);
	break;
      case 'U':  /* note there can be several -U options, each adding a dir */
	if (!info.up) {
	  info.up = strdup(optarg);
	  break;
	}
	info.up = realloc(info.up, strlen(info.up)+strlen(optarg)+2);
	if (info.up) {
	  strcat(info.up, ";");
	  strcat(info.up, optarg);
	}
	break;	   
      case 'D':
	info.down = strdup(optarg);
	break;
      case 'P':
	info.port = atoi(optarg);
	/* parse range */
	if (strchr(optarg, '-'))
	  info.port_max = atoi(strchr(optarg, '-')+1);
	else
	  info.port_max = info.port;
	break;
      case 'C':
	info.conn = atoi(optarg);
	break;
      case 'M':
	info.maxuploads = atoi(optarg);
	break;
      default:
        wp(NULL, "Invalid option -- %c\n", c);
        exit(1);
    }
  }

  if (optind < argc) {
    wp(NULL, "%s: unrecognized argument -- %s\n", *argv, argv[optind]);
    wp(NULL, "Try --help for more info\n");
    exit(1);
  }    

  return;
}

char *glistn(char *t)
{
  char *r = strdup(t);
  int i;
  
  for (i=0;r[i];i++)
    r[i] = tolower(r[i]);
  
  r = (char *)realloc(r, strlen(r)+9);
  strcat(r, ".hotlist");
  
  return(r);
}

/* the main procedure: read command line, parse configuration file,
   check and create various things and then enter event loop; afterwards 
   shut down and quit */

int main(int argc, char *argv[])
{
  int s, n, f, i;
  char *srv;
  char *t = NULL;
  char *serverlist_dup;
  FILE *fl;
  int port;

  mscroll = scur = NULL;
  cmdl = ccmd = NULL;
  curx = 0;
  wmode = 0;
  recent = NULL;
  
  dopts(argc, argv);
  
  /* open log files, if any. Note we do this before reading config file, 
   so that logging starts early. */

  if (readcfg(info.config_filename) == -1)
    return(1);

  /* set any defaults for values that were left unset by command line
     and config file */
  
  if (info.serverlist == NULL) {
    info.serverlist = strdup(std_serverlist);
  }

  /* note: prompting for password is now handled in readcfg */

  if (info.build)
  {
    wp(NULL, "Building library...\n");
    if (buildflist(info.shared_filename, info.up) == -1)
      return(1);
    if (info.build == 2)
      return(1);
  }
  
  fl = fopen(info.shared_filename, "r");
  if (!fl)
  {
    wp(NULL, "Error reading %s: %s\nTry running \"nap -b\".\n", info.shared_filename,
       sys_errlist[errno]);
    return(1);
  }
  fclose(fl);
  
  if (!opendir(info.down))
  {
    wp(NULL, "Error opening your download directory (%s): %s\n", info.down,
       sys_errlist[errno]);
    return(1);
  }
  
  if (info.conn < 0 || info.conn > 10)
    info.conn = 0;
  
  if (info.down[strlen(info.down)-1] == '/')
    msprintf(&t, "%s.tst", info.down);
  else
    msprintf(&t, "%s/.tst", info.down);
  f = open(t, O_CREAT);
  if (f == -1)
  {
    wp(NULL, "Error writing to your download directory (%s): %s\n", t, sys_errlist[errno]);
    free(t);
    return(1);
  }
  unlink(t);
  free(t);
  t = NULL;
  
  serverlist_dup = strdup(info.serverlist);
  srv = strtok(serverlist_dup, ";");
  if (srv==NULL) {
    wp(NULL, "No server specified\n");
    return(1);
  }

  /* before connecting to the server, let's figure out our port number
     for incoming upload requests. We have to find a port that's not
     in use, and we have to communicate the port number to the server,
     so that's why we do this early rather than later. -PS */

  port = initfserv(info.port, info.port_max);
  if (port == -1) {
    if (info.port == info.port_max) {
      wp(NULL, "Warning: could not open port %d: %s\n", info.port, sys_errlist[errno]);
    } else {
      wp(NULL, "Warning: could not open a port in the range %d-%d: %s\n", info.port, info.port_max, sys_errlist[errno]);
    }
    wp(NULL, "Unable to serve incoming upload requests; will proceed as if firewalled\n");
    info.port = 0;
  } else {
    wp(NULL, "Using port %d\n", port);
    info.port = port;
  }

  wp(NULL, "Getting best host...\n");
  for (;;)
  {
    /* if we have reached the end of the server list, go back to beginning, 
       if info.reconnect is set - else give up */
    if (srv==NULL) {
      if (info.reconnect) {
	free(serverlist_dup);
	serverlist_dup = strdup(info.serverlist);
	srv = strtok(serverlist_dup, ";");
	sleep(1);
      } else {
	return(1);
      }
    }

    wp(NULL, "Trying %s\n", srv);
    s = 2;
    while (s == 2)
      s = conn(srv, PORT);
    if (s == -1)
    {
      srv = strtok(NULL, ";");
      continue;
    }
  
    if (info.create)
    {
      wp(NULL, "Creating account...\n");
      n = makeact(s, info.user, info.pass, info.port, CLIENT, info.conn, info.email);
      if (n == -1)
      {
        close(s);
	srv = strtok(NULL, ";");
        continue;
      }
      else 
      {
	info.create = 0;
        break; /* if we just created an account, we are already logged in */
      }
    }
    else
    {
      wp(NULL, "Logging in...\n");
      n = login(s, info.user, info.pass, info.port, CLIENT, info.conn, info.email);
      if (n == -1)
      {
        close(s);
	srv = strtok(NULL, ";");
        continue;
      }
      else 
      {
        break;
      }
    }
  }
  free(serverlist_dup);
  
  /* announce our success */
  {
    struct sockaddr_in dst;
    int frmlen = sizeof(dst);
    if (!getpeername(s, (struct sockaddr *)&dst, &frmlen))
      wp(NULL, "Connected to %s:%i\n", inet_ntoa(dst.sin_addr), ntohs(dst.sin_port));
  }

  signal(SIGCHLD, SIG_IGN);
  signal(SIGPIPE, SIG_IGN);
  signal(SIGWINCH, tresize);
  if (!info.daemon)
    signal(SIGINT, sigint);
#ifdef MCURSES
  signal(SIGCONT, tresize);
#endif

  pipe(ipcs);
  
  addsock(ipcs[0], "ipc", S_R, inipc);
  addsock(s, "server", S_R, inserv);
  if (!info.daemon)
    addsock(0, "input", S_R, input);
/*  initssock(LPORT); */
  initwin(force);
  t = glistn(info.user);
  checkhotlist(s, t);
  free(t);
  loadaliases("alias");
  loadhandlers("handler");
  loadsets(".naprc");
  if (nvar("savechannels") == 1)
    loadchans(s);
#ifndef __CYGWIN32__
  libload(USERLIB);
#endif
/*  checknv(SHOST, SPORT, "VERSION"); */
  wp(wchan, "%s\n", title);
  drw(wchan);
  drw(winput);
  usleep(400000);

  /* enter the main event loop. */
  sockfunc(wchan, winput);
  
  shutdown(s, 2);
  close(s);
  
  doquit();
  
  return(1);
}

void savechans(chans_t *cl)
{
  chans_t *cur;
  FILE *f;
  char *fn = NULL;
  
  msprintf(&fn, "%s.channels", info.user);
  
  f = fopen(fn, "w");
  if (!f)
    return;
  free(fn);
  
  for (cur=cl;cur;cur=cur->next)
    if (!cur->q)
      fprintf(f, "%s\n", cur->nm);
  
  fclose(f);
}

void loadchans(int s)
{
  char buf[256], *fn = NULL;
  FILE *f;
  
  msprintf(&fn, "%s.channels", info.user);
  
  f = fopen(fn, "r");
  if (!f)
    return;
  free(fn);
  
  while (!feof(f))
  {
    memset(buf, 0, sizeof(buf));
    fgets(buf, sizeof(buf), f);
    
    if (buf[strlen(buf)-1] == '\n')
      buf[strlen(buf)-1] = 0;
    if (strlen(buf))
      sendpack(s, F_JOIN, "%s", buf);
  }
  
  fclose(f);
}

void initssock(int port)
{
  struct sockaddr_in frm;
  int s, on = 1;
  
  s = socket(AF_INET, SOCK_DGRAM, 0);
  
  frm.sin_addr.s_addr = INADDR_ANY;
  frm.sin_port = htons(port);
  frm.sin_family = AF_INET;
  
  setsockopt(s, SOL_SOCKET, SO_REUSEADDR, &on, sizeof(on));
  
  bind(s, (struct sockaddr *)&frm, sizeof(frm));
  
  addsock(s, "ssock", S_R, inssock);
}

int inssock(WINDOW *win, sock_t *m)
{
  char buf[1024];  /* note: don't need malloc */
  struct sockaddr_in frm;
  int frmlen = sizeof(frm);
  
  memset(buf, 0, 1024);
  
  recvfrom(m->s, buf, 1024, 0, (struct sockaddr *)&frm, &frmlen);
  if (!strncmp(buf, "VERSION", strlen("VERSION")))
  {
    if (strcmp(strchr(buf, ' ')+1, VERSION))
    {
      wp(win, "%s* Attention: Version %s of nap is out!%s\n", BRIGHT(BLUE), strchr(buf, ' ')+1, WHITE);
      wp(win, "%s* Attention: Please visit http://www.gis.net/~nite/ to upgrade immediately%s\n", BRIGHT(BLUE), WHITE);
      drw(win);
    }
  }
  else if (!strncmp(buf, "MSG", strlen("MSG")))
  {
    wp(win, "%s* %s%s\n", BRIGHT(BLUE), strchr(buf, ' ')+1, WHITE);
    drw(win);
  }
  
  return(1);
}

void checknv(char *host, int port, char *req)
{
  int s;
  struct sockaddr_in dst;
  
  s = socket(AF_INET, SOCK_DGRAM, 0);
  
  dst.sin_addr.s_addr = resolve(host);
  dst.sin_port = htons(port);
  dst.sin_family = AF_INET;
  
  sendto(s, req, strlen(req), 0, (struct sockaddr *)&dst, sizeof(dst));
}

int rebuild(int s, char *sd, char *path)
{
  FILE *f;
  char rd[1024];
  int i;
  
  f = fopen(sd, "r");
  if (!f)
    return(-1);
  
  while (!feof(f))
  {
    memset(rd, 0, sizeof(rd));
    fgets(rd, sizeof(rd), f);
    for (i=1;rd[i];i++)
    {
      if (rd[i] == '\"')
      {
        rd[i] = 0;
        break;
      }
      if (rd[i] == '/')
        rd[i] = '\\';
    }
    if (strlen(rd))
      sendpack(s, F_DFILE, "\"%s\"", rd+1);
  }
  
  if (buildflist(sd, path) == -1)
    return(-1);
  
  if (lfiles(s, sd) == -1)
    return(-1);
  
  return(1);
}

/* build the library file. sd is the name of the library file, and
   path is the ";"-separated sequence of upload directories. */
int buildflist(char *sd, char *path)
{
  FILE *f;
  char sv[513], *t, *n = strdup(path); /* note: don't need malloc for sv */
  char *r;

  f = fopen(sd, "w");
  if (f == NULL)
  {
    wp(NULL, "Error opening \"%s\": %s\n", sd, sys_errlist[errno]);
    free(n);
    return(-1);
  }
  
  r = getcwd(sv, 512);
  if (r == NULL)
  {
    wp(NULL, "Error getting the current working directory: %s\n", 
       sys_errlist[errno]);
    free(n);
    /* fixed a memory leak: used to forget to free() sv */
    return(-1);
  }
  
  t = strtok(n, ";");
  
  while (t)
  {
    dir(t, f);
    t = strtok(NULL, ";");
  }
  
  free(n);
  
  fclose(f);
  
  chdir(sv);
  
  return(1);
}

/* add the files from directory nm to the library file f */
void dir(char *nm, FILE *f)
{
  DIR *t;
  char *p=NULL;
  
  if (nm[strlen(nm)-1] != '/')
    msprintf(&p, "%s/", nm);
  else
    p = strdup(nm);
  
  t = opendir(p);
  if (!t)
    return;
  
  ddir(p, t, f);
  closedir(t);
  free(p);
}

/* add the files from directory p (whose pathname is bpath, ending in
   "/"), to the library file f */
void ddir(char *bpath, DIR *p, FILE *f)
{
  struct dirent *ls;
  DIR *t;
  char *nbpath=NULL;
  mhdr_t *m;
  
  while ((ls = readdir(p)))
  {
    msprintf(&nbpath, "%s%s", bpath, ls->d_name);
    t = opendir(nbpath);  /* a bad way for testing for a directory */
    if (t)
    {
      closedir(t);
      if (strcmp(ls->d_name, "..") && strcmp(ls->d_name, "."))
      {
        free(nbpath);
        nbpath=NULL;
        msprintf(&nbpath, "%s%s/", bpath, ls->d_name);
        dir(nbpath, f);
      }
    }
    else
    {
      m = gethdr(nbpath);
      if (m)
      {
        fprintf(f, "\"%s\" %s-%i %i %i %i %i\n", nbpath, m->check, m->sz, m->sz, m->bitrate, m->freq, m->len);
        free(m);
      }
    }
    free(nbpath);
    nbpath=NULL;
  }
}

int lfiles(int s, char *fn)
{
  FILE *f;
  int i;
  char buf[512], *pt; /* note: don't need malloc for buf */
  
  f = fopen(fn, "r");
  if (f == NULL)
  {
    wp(NULL, "Error reading %s: %s\nTry running \"nap -b\".\n", fn,
       sys_errlist[errno]);
    return(-1);
  }
  
  while(1)
  {
    memset(buf, 0, 512);
    if (fgets(buf, 512, f) == NULL)
      break;
    pt = strchr(buf, '\r');
    if (pt == NULL)
      pt = strchr(buf, '\n');
    if (pt)
      *pt = 0;
    for (i=0;buf[i];i++)
      if (buf[i] == '/')
        buf[i] = '\\';
    if (sendpack(s, F_SFILE, "%s", buf) == -1)  
    {
      wp(NULL, "Connection reset by peer\n");
      return(-1);
    }
  }

  fclose(f);
  
  return(1);
}  

/* this returns -1 and sets h_errno on error */
int32 resolve(char *host)
{
  struct hostent *hst;
  int32 ret;
  
  if ((ret = inet_addr(host)) != -1)
    return(ret);
  hst = gethostbyname(host);
  if (!hst)
    return(-1);
  
  return(*(int32 *)hst->h_addr);
}

int conn(char *host, unsigned short port)
{
  int s;
  struct sockaddr_in dst;
  char *buf, serv[16];
  int p, i;
  
  if (!strchr(host, ':'))
  {
    dst.sin_port = htons(port);
    dst.sin_family = AF_INET;
    dst.sin_addr.s_addr = resolve(host);
    if (dst.sin_addr.s_addr == -1)
    {
      noresolv = 1;
      wp(wchan, "Error resolving host %s: %s\n", host, hstrerror(h_errno));
      return(-1);
    }
    
    s = socket(AF_INET, SOCK_STREAM, 0);
    if (s == -1)
    {
      wp(wchan, "Error creating socket: %s\n", sys_errlist[errno]);
      return(-1);
    }
    
    if (connect(s, (struct sockaddr *)&dst, sizeof(dst)) == -1)
    {
      wp(wchan, "Error connecting socket: %s\n", sys_errlist[errno]);
      close(s);
      return(-1);
    }
  
    rsock(s, &buf);
    shutdown(s, 2);
    close(s);
    if (!buf)
    {
      wp(wchan, "Error finding best host\n");
      return(-1);
    }
  }
  else
    buf = strdup(host);

  if (!strncasecmp(buf, "wait", 4))
  {
    sscanf(buf, "%s %i", serv, &p);
    wp(wchan, "Waiting for %i seconds\n", p);
    sleep(p);
    return(2);
  }
  else if (!strncasecmp(buf, "busy", 4))
  {
    wp(wchan, "Busy\n");
    return(2);
  }
  p = atoi(strchr(buf, ':')+1);
  for (i=0;buf[i]!=':';i++)
    serv[i] = buf[i];
  serv[i] = '\0';
  free(buf);
  
  dst.sin_addr.s_addr = resolve(serv);
  if (dst.sin_addr.s_addr == -1)
  {
    wp(wchan, "Error resolving host %s: %s\n", serv, hstrerror(h_errno));
    return(-1);
  }
  dst.sin_port = htons(p);
  dst.sin_family = AF_INET;
  
  s = socket(AF_INET, SOCK_STREAM, 0);
  if (s == -1)
  {
    wp(wchan, "Error creating socket: %s\n", sys_errlist[errno]);
    return(-1);
  }
  
  wp(wchan, "Connecting...\n");
  
  if (connect(s, (struct sockaddr *)&dst, sizeof(dst)) == -1)
  {
    wp(wchan, "Error connecting socket: %s\n", sys_errlist[errno]);
    close(s);
    return(-1);
  }
  
  return(s);
}

int login(int s, char *user, char *pass, int data, char *client, int conn, char *email)
{
  char *t1=NULL;
  phead_t *rcv;
  int r;
  
  msprintf(&t1, "nap v%s", client);
  sendpack(s, F_LOGIN, "%s %s %i \"%s\" %i", user, pass, data, t1, conn);
  free(t1);
  t1 = NULL;
  if ((r = recvpack(s, &t1, &rcv)) == -1)
  {
    wp(wchan, "Error: Connection reset by peer\n");
    return(-1);
  }
  
  if (r == -2)
    while (recvpack(s, &t1, &rcv) == -2);
  
  if (!rcv)
    return(-1);
  
  if (rcv->op == F_LOGERROR)
  {
    wp(wchan, "Error: %s\n", t1);
    free(t1);
    free(rcv);
    return(-1);
  }
  else if (strcasecmp(t1, email))
    wp(wchan, "Email addresses did not match (%s and %s), proceeding anyway...\n", email, t1);
  
  free(t1);
  free(rcv);
  
  return(lfiles(s, info.shared_filename));
}

int makeact(int s, char *user, char *pass, int data, char *client, int conn, char *email)
{
  char *t1=NULL;
  phead_t *rcv;
  int r;
  
  sendpack(s, F_MKUSER, user);
  if ((r = recvpack(s, &t1, &rcv)) == -1)
  {
    wp(NULL, "Error: Connection reset by peer\n");
    return(-1);
  }
  
  if (r == -2)
    while (recvpack(s, &t1, &rcv) == -2);
  
  if (rcv->op == F_UNTK)
  {
    wp(NULL, "Error: Username taken\n");
    free(t1);
    free(rcv);
    return(-1);
  }
  else if (rcv->op == F_UNBAD)
  {
    wp(NULL, "Error: Invalid username\n");
    free(t1);
    free(rcv);
    return(-1);
  }
  else if (rcv->op == F_UNOK)
    wp(NULL, "Registered username\n");
  else
  {
    wp(NULL, "Unknown op: 0x%x\n", rcv->op);
    free(t1);
    free(rcv);
    return(-1);
  }
  
  free(t1);
  free(rcv);
  t1 = NULL;
  
  msprintf(&t1, "nap v%s", client);
  sendpack(s, F_REG, "%s %s %i \"%s\" %i %s", user, pass, data, t1, conn, email);
  free(t1);
  t1 = NULL;
  if ((r = recvpack(s, &t1, &rcv)) == -1)
  {
    wp(NULL, "Error: Connection reset by peer\n");
    return(-1);
  }
  
  if (r == -2)
    while (recvpack(s, &t1, &rcv) == -2);
  
  if (rcv->op == F_LOGERROR)
  {
    wp(NULL, "Error: %s\n", t1);
    free(t1);
    free(rcv);
    return(-1);
  }
  
  free(t1);
  free(rcv);
  
  return(lfiles(s, info.shared_filename));
}

int inipc(WINDOW *win, sock_t *m)
{
  int s = m->s;
  char *buf;
  
  rsock(s, &buf);
  if (!buf)
    return(1);
  
  addscroll(win, buf);
  dscr(win);
  drw(win);
  
  free(buf);
  
  return(1);
}

int inserv(WINDOW *win, sock_t *m)
{
  int s = m->s, j, n, r;
  phead_t *pk;
  char *data;
  sock_t *t;
  cmds_t *cur;
  
  r = recvpack(s, &data, &pk);
  
  if (r == -1)
  {
    wp(win, "%s* Error reading from socket: %s%s\n", RED, sys_errlist[errno], WHITE);
    drw(win);
    return(0);
  }
  else if (r == -2)
    return(1);

  for (j=0;data[j];j++)
    if (data[j] == 10)
    {
      strncpy(data+j, data+j+1, strlen(data+j+1));
      data[strlen(data)-1] = 0;
    }
    
  n = parsein(s, pk->op, data, win);
  noprint = 0;
  if (n == -1)
    return(0);
  else if (n == -2)
    return(-1);
  
  free(pk);
  free(data);
  
  dstatus();

  t = findsock("input");
  if (!t)
    return(1);
  if (!cmdl)
  {
    cmdl = (cmds_t *)malloc(sizeof(cmds_t));
    cur = cmdl;
    cur->next = NULL;
    cur->prev = NULL;
    memset(cur->cmd, 0, sizeof(cur->cmd));
    curx = 0;
    t->d = (void *)cur;
  }

  cur = (cmds_t *)t->d;
  indraw(cur->cmd, curx, winput);
  drw(winput);
  
  return(1);
}

unsigned char gchr(WINDOW *win)
{
  fd_set fs;
  struct timeval tv;
  unsigned char r;
  
  FD_ZERO(&fs);
  FD_SET(0, &fs);
  
  tv.tv_sec = 0;
  tv.tv_usec = 50000;
  
  if (!select(1, &fs, NULL, NULL, &tv))
    return(0);
  
  r = wgetch(win);
  if (r == '\e')
  {
    ungetch('\e');
    return(0);
  }
  
  return(r);
}

chans_t *findchan(chans_t *h, char *chan)
{
  chans_t *cur;

  for (cur=h;;cur=cur->next)
  {
    if (!cur)
      return(NULL);
    if (!strcasecmp(cur->nm, chan))
      return(cur);
  }
}

chans_t *findquery(chans_t *h, char *chan)
{
  chans_t *cur;

  for (cur=h;;cur=cur->next)
  {
    if (!cur)
      return(NULL);
    if (!strcasecmp(cur->nm, chan) && cur->q == 1)
      return(cur);
  }
}

int sendpack(int s, unsigned short op, char *fmt, ...)
{
  char pack[4+4096];  /* note: don't need malloc for data or pack */
  /* first four bytes of pack are header, remaining is data */
  phead_t *hdr;
  va_list args;
  int r;
  
  if (s == -1)
  {
    wp(wchan, "%s* Not connected to the server%s\n", RED, WHITE);
    drw(wchan);
    return(-1);
  }
  
  if (!fmt)
  {
    hdr = (phead_t *)malloc(sizeof(phead_t));
    
    hdr->len = 0;
    hdr->op = op;

    if (info.d == 2)
    {
      if (wchan)
      {
        wp(wchan, "--> (%x)\n", hdr->op);
        drw(wchan);
      }
    }
    
    r = send(s, (char *)hdr, 4, 0);
    
    return(r);
  }
  
  va_start(args, fmt);
  vsprintf(pack+4, fmt, args);
  va_end(args);
  
  hdr = (phead_t *)pack;
  
  hdr->len = strlen(pack+4);
  hdr->op = op;

  if (info.d == 2)
  {
    if (wchan)
    {
      wp(wchan, "--> (%x) |%s|\n", hdr->op, pack+4);
      drw(wchan);
    }
  }
  
  r = send(s, pack, hdr->len+4, 0);
  
  return(r);
}

int ssock(int s, char *fmt, ...)
{
  char data[4096]; /* note: don't need malloc for data */
  va_list args;
  int r;
  
  va_start(args, fmt);
  vsprintf(data, fmt, args);
  va_end(args);
  
  if (info.d == 2)
  {
    if (wchan)
    {
      wp(wchan, "--> |%s|\n", data);
      drw(wchan);
    }
  }

  r = write(s, data, strlen(data));
  
  return(r);
}

int recvpack(int s, char **buf, phead_t **hdr)
{
  int r = 0, i = 0;
  fd_set fs;
  struct timeval tv;
  
  if (!thdr)
  {
    thdr = (phead_t *)malloc(sizeof(phead_t));
    memset(thdr, 0, sizeof(phead_t));
  
    while (i < 4)
    {
      r = read(s, ((char *)thdr)+i, sizeof(phead_t)-i);
      if (r <= 0)
      {
        free(thdr);
        thdr = NULL;
        *hdr = NULL;
        *buf = NULL;
        return(-1);
      }
      i += r;
    }
  }
  
  if (!thdr->len)
  {
    *hdr = thdr;
    *buf = strdup("");
    thdr = NULL;
    return(1);
  }
  
  FD_ZERO(&fs);
  FD_SET(s, &fs);
  tv.tv_usec = 0;
  tv.tv_sec = 0;
  
  if (!select(s+1, &fs, NULL, NULL, &tv))
    return(-2);
  
  if (!tdbuf)
  {
    tdbuf = (char *)malloc(thdr->len+1);
    memset(tdbuf, 0, thdr->len+1);
    tpos = 0;
  }
  
  r = read(s, tdbuf+tpos, thdr->len-tpos);
  if (r <= 0)
  {
    free(tdbuf);
    free(thdr);
    tdbuf = NULL;
    thdr = NULL;
    *buf = NULL;
    *hdr = NULL;
    return(-1);
  }
  if ((r+tpos) < thdr->len)
  {
    *buf = NULL;
    *hdr = NULL;
    tpos += r;
    return(-2);
  }
  
/*  memset(tdbuf, 0, 1024);
  r = read(s, tdbuf, thdr->len);
  if (r <= 0)
  {
    free(tdbuf);
    free(thdr);
    tdbuf = NULL;
    thdr = NULL;
    *buf = NULL;
    *hdr = NULL;
    return(-1);
  }
  
  tdbuf = (char *)realloc(tdbuf, strlen(tdbuf)+1); */
  
  *buf = tdbuf;
  *hdr = thdr;
  
  tdbuf = NULL;
  thdr = NULL;
  
#ifdef DEBUG
  wp(wchan, "|%s| (%x:%i)\n", *buf, (*hdr)->op, (*hdr)->len);
  if (wchan)
    drw(wchan);
#endif
  
  return(1);
}

int rsock(int s, char **buf)
{
  int r;
  fd_set fs;

  *buf = (char *)malloc(4096);
  memset(*buf, 0, 4096);
  
  FD_ZERO(&fs);
  FD_SET(s, &fs);
  
  if (select(s+1, &fs, NULL, NULL, NULL) == -1)
  {
    free(*buf);
    *buf = NULL;
    return(-1);
  }
  
  if ((r = read(s, *buf, 4096)) <= 0)
  {
    free(*buf);
    *buf = NULL;
    return(-1);
  }
  
  *buf = (char *)realloc(*buf, strlen(*buf)+1);
  
  return(r);
}

char *fxp(char *in)
{
  int i=0, j=0;
  char *ret;
  
  if (!strchr(in, '%'))
    return(in);
  
  memset(tpbuf, 0, 512);
  ret = tpbuf;
  
  for (;in[i];i++,j++)
  {
    ret[j] = in[i];
    if (in[i] == '%')
    {
      j++;
      ret[j] = '%';
    }
  }
  
  ret[j+1] = '\0';
  
  return(ret);
}

int gconn(WINDOW *win, sock_t *m)
{
  int s = m->s, r;
  struct sockaddr_in frm;
  int frmlen = sizeof(frm);
  
  r = accept(s, (struct sockaddr *)&frm, &frmlen);
  if (r == -1)
    return(1);
  
  addsock(r, "conn", S_R, lsock);
  ssock(r, "1");
  
  return(1);
}

int lsock(WINDOW *win, sock_t *m)
{
  char buf[5];
  
  memset(buf, 0, sizeof(buf));
  recv(m->s, &buf, 4, MSG_PEEK);
  if (!strncmp(buf, "GET", 3))
  {
    recv(m->s, &buf, 3, 0);
    m->func = doget;
  }
  else if (!strncmp(buf, "SEND", 4))
  {
    recv(m->s, &buf, 4, 0);
    m->func = dosend;
  }
  else
    return(0);
    
  return(1);
}

/* remote client connected to our data port and issued a GET command */
int doget(WINDOW *win, sock_t *m)
{
  char *buf, *nm, *fn;
  unsigned long sz;
  file_t *cur;
  struct stat st;
  sock_t *sv;
  char **tok;
  int i, cnt;
  
  if (rsock(m->s, &buf) == -1) {
    close(m->s); /* -PS */
    return(0);
  }    

  if (info.d == 2)
  {
    wp(win, "<-- |%s| (from remote client)\n", buf);
    drw(win);
  }
  
  tok = form_tokso(buf, &cnt); 
  /* we should be able to use form_tokso() here instead of the more general
   * form_toks() */
  
  /* the client should have sent a string of the form
   * <remotenick> "<filename>" <offset> */
  
  if (cnt < 3)
  {
    wp(win, "%s* Bogus GET command from remote client%s\n", RED, WHITE);
    drw(win);
    ssock(m->s, "INVALID REQUEST\n");
    free(buf);
    for (i=0;i<cnt;i++)
      free(tok[i]);
    free(tok);
    close(m->s); /* -PS */
    return(0);
  }
  
  nm = strdup(tok[0]);
  fn = strdup(tok[1]);
  sz = atof(tok[2]);
  free(buf);
  
  for (i=0;i<cnt;i++)
    free(tok[i]);
  free(tok);
  
  #ifdef DEBUG
    wp(win, "Request from %s: \"%s\"\n", nm, fn);
    drw(win);
  #endif
  
  cur = ffile(up, nm, fn);
  free(nm);
  free(fn);

  if (!cur)
  {
    wp(win, "%s* %s requested upload of \"%s\" but it isn't " \
        "in the upload list! (GET)%s\n", RED, tok[0], tok[3], WHITE);
    drw(win);
    ssock(m->s, "INVALID REQUEST\n");
    close(m->s); /* -PS */
    return(0);
  }
  
  if (!cur->f)
  {
    wp(win, "%s* \"%s\" not found (requested by %s)%s\n", RED, \
        tok[3], tok[0], WHITE);
    drw(win);
    ssock(m->s, "FILE NOT FOUND\n");
    close(m->s); /* -PS */
    dupload(cur);
    return(0);
  }
  
  fseek(cur->f, sz, SEEK_SET);
  
  m->d = (void *)cur;
  m->func = sfile;
  m->t = S_W;
  free(m->nm);
  m->nm = NULL;
  msprintf(&m->nm, "u %i", gnum(1));
  
  sv = findsock("server");
  sendpack(sv->s, F_UP, NULL);
  
  fstat(fileno(cur->f), &st);
  cur->csz = 0;
  cur->sz = st.st_size-sz;
  cur->bsz = sz;
  cur->g = 1;
  cur->sk = m;
  deltimer(cur->timer);
  
  /* send the file size */
  ssock(m->s, "%lu", st.st_size);
  
  wp(win, "* Sending file \"%s\" to %s (%lu bytes)\n", cur->fn, cur->nm, cur->sz);
  drw(win);
  
  cur->st = time(NULL);
  
  return(1);
}

/* we requested a file from a firewalled client. The remote client has
 * connected to us and is about to start uploading the file */
int dosend(WINDOW *win, sock_t *m)
{
  char *buf, *nm, *fn, *t;
  unsigned long sz;
  file_t *cur;
  struct stat st;
  sock_t *sv;
  char **tok;
  int i, cnt;
  
  if (rsock(m->s, &buf) == -1)
  {
    close(m->s);
    return(0);
  }

  if (info.d == 2)
  {
    wp(win, "<-- |%s| (from remote client)\n", buf);
    drw(win);
  }
  
  tok = form_tokso(buf, &cnt);
  
  /* the remote client should have sent a string of the form 
   * <remotenick> "<filename>" <size> */

  if (cnt < 3)
  {
    wp(win, "%s* Bogus SEND command from remote client%s\n", RED, WHITE);
    /* maybe print IP address as well? */
    drw(win);
    ssock(m->s, "INVALID REQUEST\n");
    free(buf);
    close(m->s);
    for (i=0;i<cnt;i++)
      free(tok[i]);
    free(tok);
    return(0);
  }
  
  nm = strdup(tok[0]);
  fn = strdup(tok[1]);
  sz = atof(tok[2]);
  /* sz is unsigned long, why not use atol()? Is it because atol()
   * returns a (signed) long int, and we want that one extra bit of 
   * number space that an unsigned long int gives?? */

  free(buf);
  
  for (i=0;i<cnt;i++)
    free(tok[i]);
  free(tok);
  
  /* can remote client send "INVALID REQUEST" after a "SEND" ? */
  if (!strcasecmp(nm, "FILE")) /* "FILE NOT SHARED" or "FILE NOT FOUND" */
  {
    wp(win, "%s* Error downloading from firewalled remote client " \
        "[FILE NOT SHARED]%s\n", RED, WHITE);
    /* maybe print IP address and/or try to identify the nick by searching
     * through download list? */
    drw(win);
    /* the download this error message corresponds to will be removed from
     * the download list when the item's timer times out */
    free(nm);
    free(fn);
    close(m->s);
    return(0);
  }
  
  if (strchr(fn, '/'))
    t = strrchr(fn, '/')+1;
  else if (strchr(fn, '\\'))
    t = strrchr(fn, '\\')+1;
  else
    t = fn;
  
  if (!sz)
  {
    cur = ffile(down, nm, t);
    if (cur)
      ddownload(cur);
    rsock(m->s, &buf);
    if (buf)
      to_printable(buf);
    wp(win, "* Error downloading \"%s\" from %s (returned a size of %d) " \
        "[%s]\n", t, nm, sz, buf?buf:"");
    drw(win);
    free(fn);
    free(nm);
    close(m->s);
    return(0);
  }
  
  cur = ffile(down, nm, t);
  free(nm);
  free(fn);

  if (!cur)
  {
    wp(win, "%s* Error: %s tried to send (push) \"%s\" but the file is not "\
        "in your download list!%s\n", RED, nm, t, WHITE);
    drw(win);
    close(m->s);
    return(0);
  }
  
  m->d = (void *)cur;
  m->func = gfile;
  /* rename connection (i.e. sock_t *m) to "d #" (used to be "conn") */
  free(m->nm);
  m->nm = NULL;
  msprintf(&m->nm, "d %i", gnum(0));
  
  sv = findsock("server");
  sendpack(sv->s, F_DOWN, NULL);
  
  fstat(fileno(cur->f), &st);
  cur->bsz = st.st_size;
  cur->csz = 0;
  cur->sz = sz - st.st_size;
  cur->g = 1;
  cur->sk = m;
  deltimer(cur->timer);
  
  /* send offset */
  ssock(m->s, "%lu", cur->bsz);
  
  wp(win, "* Getting file \"%s\" from %s (%lu bytes)\n", cur->fn, cur->nm, cur->sz);
  drw(win);
  
  cur->st = time(NULL);
  
  return(1);
}

/* try to open a port in the range port..port_max for incoming upload
   requests. Return port number if successful, or -1 if failure (with
   errno set). If port 0 is requested, return 0 without opening
   anything - this probably was configured by the user if we're behind
   a firewall. */

int initfserv(int port, int port_max)
{
  struct sockaddr_in me;
  int s, on = 1;
  
  if (!port)
    return(0);
  
  s = socket(AF_INET, SOCK_STREAM, 0);
  if (s == -1)
    return(-1);
  
  setsockopt(s, SOL_SOCKET, SO_REUSEADDR, &on, sizeof(on));
  
  while (port <= port_max) {
    me.sin_addr.s_addr = INADDR_ANY;
    me.sin_port = htons(port);
    me.sin_family = AF_INET;
  
    if (bind(s, (struct sockaddr *)&me, sizeof(me)) == -1) { 
      /* failed, try next */
      port++;
      continue;
    } else {   /* success */
      listen(s, 5);
      addsock(s, "fserv", S_R, gconn);
      return(port);
    }
  }
  return(-1);  /* overall failure */
}

void closefserv()
{
  sock_t *sk;
  
  sk = findsock("fserv");
  if (!sk)
    return;
  
  delsock(sk->s);
}

/* We are about to upload a file to a remote client. We had to initiate
 * the connection because we are firewalled and cannot accept incoming
 * connections. */
int initsend(WINDOW *win, sock_t *m)
{
  int s = m->s, r;
  file_t *cur = (file_t *)m->d;
  sock_t *sk;
  struct stat st;
  
  if (!cur->f)
    return(0);
  
  fstat(fileno(cur->f), &st);
  
  r = ssock(s, "SEND");
  
  /* send "mynick filename offset" */
  r = ssock(s, "%s \"%s\" %lu", info.user, cur->rfn, st.st_size);
  
  if (r <= 0)
  {
    wp(win, "%s* %s is misconfigured%s\n", RED, cur->nm, WHITE);
    drw(win);
    sk = findsock("server");
    if (sk)
      sendpack(sk->s, F_MISCONFIGURE, "%s", cur->nm);
    dupload(cur);
    close(m->s);
    return(0);
  }
  
  m->func = ssize;
  m->t = S_R;
  cur->sz = st.st_size;
  cur->csz = 0;
  
  return(1);
}

int ssize(WINDOW *win, sock_t *m)
{
  char buf[32]; /* note: don't need malloc. */
  /* Use of "sizeof(buf)" below used to be erroneous, as buf was a pointer */
  int n=1, s = m->s;
  file_t *cur = (file_t *)m->d;
  struct timeval tv;
  fd_set fs;
  
  memset(buf, 0, 32);
  if (recv(s, &buf[0], 1, 0) <= 0)
  {
    wp(win, "%s* Error sending file to %s (they disconnected)%s\n", RED, cur->nm, WHITE);
    drw(win);
    dupload(cur);
    close(s); /* might as well close it. -PS */
    return(0);
  }
  
  recv(s, buf, sizeof(buf), 0);
  
  while (n)
  {
    FD_ZERO(&fs);
    FD_SET(s, &fs);
  
    tv.tv_usec = 50;
    tv.tv_sec = 0;
  
    n = select((s+1), &fs, NULL, NULL, &tv);
  
    if (FD_ISSET(s, &fs))
    {
      if (recv(s, buf+strlen(buf), sizeof(buf), 0) <= 0)
      {
        wp(win, "%s* Error sending file to %s%s\n", RED, cur->nm, WHITE);
        drw(win);
        dupload(cur);
        close(m->s);
        return(0);
      }
    }
  }
  
  fseek(cur->f, strtoul(buf, (char **)NULL, 10), SEEK_SET);
  
  cur->bsz = strtoul(buf, (char **)NULL, 10);
  cur->sz -= cur->bsz;
  cur->g = 1;
  deltimer(cur->timer);
  
  m->func = sfile;
  m->t = S_W;
  
  wp(win, "* Sending file \"%s\" to %s (%lu bytes)\n", cur->fn, cur->nm, cur->sz);
  drw(win);
  
  cur->st = time(NULL);
  
  return(1);
}

/* we have connected to a remote client and we are about to issue a request
 * for a file */
int initget(WINDOW *win, sock_t *m)
{
  int s = m->s, r;
  file_t *cur = (file_t *)m->d;
  sock_t *sk;
  struct stat st;
  
  if (!cur)
    return(0);
  
  if (!cur->f)
    return(0);
  
  fstat(fileno(cur->f), &st);
  
  /* note local file was already opened for append.  st.st_size is the
     file size; this is also the location of the first character to be
     sent by the remote host, in case of a resume. */

  r = ssock(s, "GET");
  r = ssock(s, "%s \"%s\" %lu", info.user, cur->rfn, st.st_size);
  
  if (r <= 0)
  {
    wp(win, "%s* %s is misconfigured%s\n", RED, cur->nm, WHITE);
    drw(win);
    sk = findsock("server");
    if (sk)
      sendpack(sk->s, F_MISCONFIGURE, "%s", cur->nm);
    ddownload(cur);
    close(m->s);
    return(0);
  }
  
  m->func = gsize;
  m->t = S_R;
  cur->bsz = st.st_size;
  cur->csz = 0;
  
  return(1);
}

/* continue the GET started by initget, now receive remote host's answer */

int gsize(WINDOW *win, sock_t *m)
{
  char buf[32];
  int i, j, s = m->s;
  file_t *cur = (file_t *)m->d;
  sock_t *sk;

  memset(buf, 0, 32);
  /* remote client should send a '1' (ASCII character 49) */
  /* (shouldn't this be moved into initget() before the GET packet is sent?) */
  if (recv(s, &buf[0], 1, 0) <= 0)
  {
    wp(win, "%s* Error getting file (%s) from %s (they disconnected)%s\n", RED, cur->fn, cur->nm, WHITE);
    drw(win);
    ddownload(cur);
    close(m->s);
    return(0);
  }
  
  /* read file size. drscholl's napster spec says:
   * "keep reading until you hit a character that is not a digit" */
  for (i=0; i<31;i++) /* but don't overflow buf[] ! */
  {
    if (recv(s, &buf[i], 1, MSG_PEEK) <= 0 && !i)
    {
      wp(win, "%s* %s is misconfigured%s\n", RED, cur->nm, WHITE);
      drw(win);
      sk = findsock("server");
      if (sk)
        sendpack(sk->s, F_MISCONFIGURE, "%s", cur->nm);
      ddownload(cur);
      close(m->s);
      return(0);
    }
    /* abort only if first character is not a digit.
     * this catches messages such as "INVALID REQUEST" or "FILE NOT SHARED"
     * but does not barf on ID3 tags that follow the file size.
     * (Before, an error was signaled if an alphabetic character 
     * followed 0 or more numeric characters. This caused a problem with
     * mp3 files that start with ID3 tags.) -NBL */
    if (!isdigit(buf[i]) && (i == 0))
    {
      recv(s, buf, 32, 0);
      buf[31]=0;          /* make sure it's 0 terminated */
      to_printable(buf);  /* eliminate non-printable chars */
      wp(win, "* Error getting \"%s\" from %s [%s]\n", cur->fn, cur->nm, buf);
      drw(win);
      ddownload(cur);
      close(m->s);
      return(0);
    }
    if (!isdigit(buf[i]))
      break;
    recv(s, &buf[i], 1, 0);
  }
  /* get rid of the non-digit that was just read
   * (or if i reached 31, make sure that buf[] is '\0'-terminated) */
  buf[i] = '\0'; 

  if (info.d == 2)
  {
    wp(win, "<-- |%s| (from remote client)\n", buf);
    drw(win);
  }
  
  cur->sz = atol(buf)-cur->bsz;
  cur->g = 1;
  deltimer(cur->timer);
  
  m->func = gfile;
  
  wp(win, "* Getting file \"%s\" from %s (%lu bytes)\n", cur->fn, cur->nm, cur->sz);
  drw(win);
  
  cur->st = time(NULL);
  
  return(1);
}

int sfile(WINDOW *win, sock_t *m)
{
  char buf[2049], *st, *end; /* note: don't need malloc for buf */
  file_t *cur = (file_t *)m->d;
  FILE *f;
  int n;
  time_t t;
  
  memset(buf, 0, 2049);
  /* is this mix of low-level and high-level i/o safe? Why use FILE*
   object if using low-level i/o? -PS */

  /* I once got a segfault on the following line, and it turned out
     (FILE *)(cur->f) pointed into unreadable memory - probably this
     file had been closed before we got here. Need to find and fix
     this bug. -PS */

  n = read(fileno(cur->f), buf, 2048); 
  if (n <= 0)
  {
    t = time(NULL);
    wp(win, "* Finished sending \"%s\" to %s (%lu of %lu bytes) in %lu\n", cur->fn, cur->nm, cur->csz, cur->sz, time(NULL)-cur->st);
    drw(win);
    if (info.logfile) {
      f = fopen(info.logfile, "a");
      if (f) {
	st = strdup(ctime(&cur->st));
	st[strlen(st)-1] = 0;
	end = strdup(ctime(&t));
	end[strlen(end)-1] = 0;
	fprintf(f, "S %s %s \"%s\" %s\n", st, cur->nm, cur->fn, end);
	fclose(f);
	free(st);
	free(end);
      }
    }
    dupload(cur);
    close(m->s);
    return(0);
  }
  
  if (send(m->s, buf, n, 0) <= 0)
  {
    t = time(NULL);
    wp(win, "%s* Transfer interrupted while sending \"%s\" to %s (%lu of %lu bytes)%s\n", RED, cur->fn, cur->nm, cur->csz, cur->sz, WHITE);
    drw(win);
    if (info.logfile) {
      f = fopen(info.logfile, "a");
      if (f) {
	st = strdup(ctime(&cur->st));
	st[strlen(st)-1] = 0;
	end = strdup(ctime(&t));
	end[strlen(end)-1] = 0;
	fprintf(f, "SI %s %s \"%s\" %s\n", st, cur->nm, cur->fn, end);
	fclose(f);
	free(st);
	free(end);
      }
    }
    dupload(cur);
    close(m->s);
    return(0);
  }
  
  cur->csz += n;
  
  return(1);
}

/* receive a raw file on socket, after initget and gsize did the
   initial communication. */
int gfile(WINDOW *win, sock_t *m)
{
  char buf[2049], *st, *end; /* note: don't need malloc for buf */
  file_t *cur = (file_t *)m->d;
  FILE *f;
  int n;
  time_t t;
  
  memset(buf, 0, 2049);
  if ((n = recv(m->s, buf, 2048, 0)) <= 0)
  {
    t = time(NULL);
    wp(win, "* Finished transfer of \"%s\" from %s (%lu of %lu bytes) in %lu\n", cur->fn, cur->nm, cur->csz, cur->sz, time(NULL)-cur->st);
    drw(win);
    if (info.logfile) {
      f = fopen(info.logfile, "a");
      if (f) {
	st = strdup(ctime(&cur->st));
	st[strlen(st)-1] = 0;
	end = strdup(ctime(&t));
	end[strlen(end)-1] = 0;
	if (cur->csz != cur->sz)
	  fprintf(f, "RI %s %s \"%s\" %s\n", st, cur->nm, cur->fn, end);
	else
	  fprintf(f, "R %s %s \"%s\" %s\n", st, cur->nm, cur->fn, end);
	fflush(f);
	free(st);
	free(end);
      }
    }
    ddownload(cur);
    close(m->s);
    return(0);
  }
  write(fileno(cur->f), buf, n);
  
  cur->csz += n;
  
  return(1);
}
