/* Copyright (c) 2000  Kevin Sullivan <nite@gis.net>
 *
 * Please refer to the COPYRIGHT file for more information.
 */

#ifndef MCURSES
#include <ncurses.h>
#endif
#include <dlfcn.h>
#include <pwd.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <errno.h>
#include <unistd.h>
#include <netdb.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <time.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <ctype.h>

#include "getopt.h"
#include "defines.h"
#include "colors.h"
#include "codes.h"
#include "lists.h"
#include "handlers.h"
#include "alias.h"
#include "nap.h"
#include "winio.h"
#include "timer.h"
#include "scheck.h"
#include "irc.h"
#include "cmds.h"

#ifdef MCURSES
#include "wind.h"
#endif


extern info_t info;
extern scrls_t *mscroll, *scur;
extern chans_t *chanl, *curchan, *recent;
extern ssearch_t *search;
extern int tind, ircsock;
extern unsigned char noprint, ircmode, lpbrk;
extern int ipcs[2];
extern char *mnick;
extern link_t *hlist;
extern WINDOW *winput;
extern void *hnd;
extern alias_t *alhead;
extern handler_t *hndhead;

char *conns[] = { "Unknown", "14.4", "28.8", "33.6", "56k", "ISDN-64K",
                  "ISDN-128K", "Cable", "DSL", "T1", "T3" };
unsigned char ups[] = { 3, 2, 3, 3, 4, 5, 5, 7, 7, 9, 9 };
unsigned char fl, wmode = 0, *dwi=NULL, srch=0;
char *tbuf = NULL;
int tin = 0;
char *ctbuf;
file_t *up, *down;
scount_t scount;

#ifdef QUEUE
ssearch_t *downq; /* download queue */
#endif /* QUEUE */

in_nap_cmd_t in[] = {
	 { F_SRET, ssret },
	 { F_SEND, ssend },
	 { F_RBROWSE, srbrowse },
	 { F_DBROWSE, sdbrowse },
	 { F_SGET, sget },
	 { F_NGET, snget },
#ifdef QUEUE
	 { F_NACC, snacc },
	 { F_RQLIMIT, sqlimit },
#endif /* QUEUE */
	 { F_FREQ, sfreq },
	 { F_SSF, ssf },
	 { F_COUNT, sscount },
	 { F_SWHOIS, swhois },
	 { F_SOFF, soff },
	 { F_UON, suon },
	 { F_UOFF, suoff },
	 { F_NOTICE, snotice },
	 { F_RY, sry },
	 { F_SJOIN, sjoin },
	 { F_SPART, spart },
	 { F_PART, smpart },
	 { F_TOPIC, stopic },
	 { F_SAID, ssay },
	 { F_NCHAN, snchan },
	 { F_JCHAN, sjchan },
	 { F_MNAME, snend },
	 { F_MNAME2, snend },
	 { F_TELL, stell },
	 { F_USER, suser },
	 { F_SOP, sop },
	 { F_PCHANGE, spchange },
	 { F_BPORT, sbport },
	 { F_ANNOUNCE, sannounce },
	 { F_SBANLIST, sbanlist },
	 { F_SBANLISTU, sbanlist },
	 { SERVER_PING, ssping },
	 { CLIENT_PING, scping },
	 { CLIENT_PONG, scpong },
	 { CLIENT_REDIRECT, sredir },
	 { CLIENT_CYCLE, scycle },
	 { F_SCLIST, sclist },
	 { CHANNEL_ENTRY2, sclist2 },
	 { F_NAMES, suser },
	 { F_SBLOCKLIST, sblocklist },
	 { IGNORE_LIST, signoreend },
	 { IGNORE_ENTRY, signorelist },
	 { IGNORE_ADD, signoreadd },
	 { IGNORE_REMOVE, signoreremove },
	 { IGNORE_UNKNOWN, signoreunknown },
	 { IGNORE_EXISTS, signoreexists },
	 { IGNORE_CLEAR, signoreclear },
	 { IGNORE_FAIL, signorefail },
	 { CHANNEL_BAN_ENTRY, scbanlist },
	 { NOTIFY_UNKNOWN, snerr },
	 { NOTIFY_EXISTS, snadd },
	 { CHAN_EMOTE, sme },
	 { 0x0, NULL },
};

out_nap_cmd_t out[] = {
          { "about", 1, dabout, NULL },
          { "alias", 1, dalias, "<name> <args> - Creates an alias" },
          { "aliaslist", 1, daliaslist, "- Returns a list of aliases created" },
          { "announce", 0, dannounce, "<msg> - Broadcasts a message to all users" },
	  { "ban", 0, dban, "<user/IP> - Bans the specified user or IP" },
          { "banlist", 0, dbanlist, "- Prints a list of the current bans on the server" },
          { "block", 0, dblock, "<IP> [reason] - Blocks the specified IP" },
          { "blocklist", 0, dblocklist, "- Gives a list of current blocked" },
          { "break", 1, dbreak, "- Breaks out of a loop" },
          { "browse", 0, dbrowse, "<user> - Browses user's files" },
          { "cban", 0, dcban, "<user> [reason] - Bans a user from a channel" },
          { "cbanlist", 0, dcbanlist, "- Returns a list of banned users in a channel" },
          { "chupload", 1, dchupload, "<path> - Changes your upload path (still need to /rebuild to update your files)" },
          { "clear", 1, dclear, "- Clears your screen buffer" },
          { "clearalias", 1, dclearalias, "- Clears all aliases" },
          { "clearhandler", 1, dclearhandler, "- Clears all handlers" },
          { "clist", 0, dclist, "- Gets a list of channels" },
          { "clist2", 0, dclist2, "- Gets a list of channels (includes user created)" },
          { "cloak", 0, dcloak, "- Cloaks yourself" },
          { "conf", 0, dsetconf, NULL },
          { "cunban", 0, dcunban, "<user> [reason] - Unbans a user from a channel" },
          { "ddown", 0, ddown, "<number> - Deletes download \"number\" returned from /pdown" },
          { "dec", 1, ddec, "- Decreases the variable by one" },
          { "debug", 1, ddebug, NULL },
          { "disconnect", 0, ddisconnect, "- Disconnects you from the server" },
          { "dns", 1, ddns, "<host/IP> - Attempts to resolve the specified address" },
          { "done", 1, ddone, "- Ends an alias" },
          { "dup", 0, ddup, "<number> - Deletes upload \"number\" returned from /pup" },
          { "echo", 1, decho, "<text> - Echos text to the screen" },
          { "eval", 1, deval, "<name> - Returns the value of a variable" },
          { "exec", 1, dexec, "[-o] <command> - Executes a command from a shell and redirects the input to the client" },
	  { "finger", 0, dwhois, "<user> - Gets information on the specified user" },
#ifdef QUEUE
	  { "forceq", 0, dforceq, "<user> - kickstart downloads from the specified user that are stuck in the queue" }, 
#endif /* QUEUE */
	  { "g", 0, dg, "<number> - Gets file number \"number\" returned from /search" },
	  { "get", 0, dg, NULL },
          { "gusers", 0, dgusers, "- Gets a global list of users" },
          { "handler", 1, dhandler, "<code> <args> - Adds a handler" },
          { "handlerlist", 1, dhandlerlist, "- Returns a list of handlers created" },
          { "help", 0, dhelp, "<command> - Returns help on the specified command" },
          { "hotlist", 0, dhotlist, "- Lists users on your hotlist that are on" },
          { "hup", 0, dhup, "- Scans to see which users on your hotlist are currently on" },
          { "if", 1, dif, "(<val> <op> <val>) <cmd> - Compares two values" },
          { "ignore", 0, dignore, "<user> - Ignores a user" },
          { "ignoreclear", 0, dignoreclear, "- Clears your ignore list" },
          { "ignorelist", 0, dignorelist, "- Lists ignored users" },
          { "inc", 1, dinc, "<var> - Increases the variable by 1" },
          { "irc", 0, dirc, NULL },
	  { "join", 0, djoin, "<chan> - Joins the specified channel" },
          { "kick", 0, dkick, "<user> [reason] - Kicks a user from a channel" },
          { "kickall", 0, dkickall, "<user> [reason] - Kicks a user from all channels you and the user are in" },
	  { "kill", 0, dkill, "<user> - Kills the specified user" },
	  { "lastlog", 1, dlastlog, "<str> - Returns all occurences of \"str\" that have been said or printed" },
          { "loadalias", 1, dloadalias, "<filename> - Loads a list of aliases from a filename" },
          { "loadconfig", 1, dloadconfig, "<filename> - Loads a list of settings from a filename" },
          { "loadhandler", 1, dloadhandler, "<filename> - Loads a list of handlers from a filename" },
          { "me", 0, dme, "<string> - Does an emotion" },
	  { "msg", 0, dtell, "<user> <msg> - Sends the user the message specified" },
          { "muzzle", 0, dmuzzle, "<user> <msg> - Muzzles the user with the specified message" },
          { "names", 0, dusers, "<channel> - Gets a list of channel users" },
          { "news", 1, dnews, "- Checks for any news on the client" },
          { "noprint", 1, dnoprint, "- Stops the client from echoing anything until the command returns" },
          { "notify", 0, dnotify, "<user> - Attempts to add a user to your hotlist" },
	  { "opsay", 0, dopsay, "<msg> - Broadcasts a message to all moderators/admins/elite" },
	  { "part", 0, dpart, "[chan/user] - Parts the specified or current channel or query" },
          { "pchans", 1, pchans, NULL },
	  { "pdown", 0, dpdown, "- Gives a listing of your current downloads" },
          { "ping", 0, dping, "<user> - Pings a user" },
          { "psocks", 1, dpsocks, NULL },
	  { "pup", 0, dpup, "- Gives a listing of your current uploads" },
	  { "pvars", 0, dpvars, "- Prints the values of all variables currently set" },
          { "query", 0, dquery, "<user> - Queries a user" },
	  { "quit", 0, dquit, "- Closes the program" }, 
	  { "rebuild", 1, drebuild, "- Rebuilds your library" },
          { "reconnect", 0, dreconnect, "- Reconnects you to the server" },
          { "reload", 0, dreloadconf, NULL },
          { "reloadm", 1, dreload, "- Reloads the user command module" },
          { "results", 1, dresults, "- Switches to the search results screen" },
          { "savealias", 1, dsavealias, "<filename> - Saves current aliases to a filename" },
          { "saveconfig", 1, dsaveconfig, "<filename> - Saves current settings to a filename" },
          { "savehandler", 1, dsavehandler, "<filename> - Saves current handlers to a filename" },
          { "say", 0, dsay, "<msg> - Sends msg to the current channel" },
	  { "search", 0, dsearch, "<query> - Searches the napster database for the specified query" },
	  { "serv", 0, dserv, "- Returns the current IP and port of the server you're connected to" },
	  { "server", 0, dserver, "<IP:port> - Connects to the specificed server and port" },
          { "set", 1, dset, "<name> <value> - Sets a variable to the specified value" },
	  { "setdataport", 0, ddatap, "<user> <port> - Sets a user's data port" },
          { "setlevel", 0, dsetlevel, "<channel> <level> - ?" },
          { "setlinespeed", 0, dlspeed, "<user> <speed> - Changes a user's linespeed" },
          { "setpassword", 0, duserpass, "<user> <password> - Sets a user's password" },
	  { "setuserlevel", 0, dlevel, "<user> <level> - Changes a user's userlevel" },
          { "sraw", 0, dsraw, NULL },
          { "stop", 1, dstop, "- Returns from the current command and stops all processing on it" },
          { "sver", 0, dsver, "- Returns the server version" },
	  { "tell", 0, dtell, "<user> <msg> - Sends the user the message specified" },
	  { "timer", 1, dtimer, "<min> <sec> <cmd> - Initiates a timer to execute in the specified time" },
	  { "tlist", 1, dtlist, "- Prints out a list of the current timers" },
	  { "topic", 0, dtopic, "<channel> <topic> - Changes a channel's topic" },
          { "tquit", 0, dtquit, "- Quits when all transfers have completed" },
          { "unalias", 1, dunalias, "<name> - Removes an alias" },
	  { "unban", 0, dunban, "<IP> - Unbans the specified IP" },
          { "unblock", 0, dunblock, "<IP> - Unblocks the specified IP" },
          { "unhandler", 1, dunhandler, "<code> - Removes a handler" },
          { "unignore", 0, dunignore, "<user> - Unignores a user" },
          { "unmuzzle", 0, dunmuzzle, "<user> - Unmuzzles the user" },
          { "unnotify", 0, dunnotify, "<user> - Removes a user from your hotlist" },
          { "unset", 1, dunset, "<name> - Unsets a variable" },
/*          { "update", 1, dupdate, NULL }, */
          { "while", 1, dwhile, "(<val> <op> <val>) cmd - Keeps executing cmd while the comparison is true" },
	  { "whois", 0, dwhois, "<user> - Gets information on the specified user" },
          { "window", 1, dwindow, "- Enables/disables window mode" },
          { "wstats", 1, dwstats, NULL },
	  { NULL, 0, NULL, NULL },
};


void upchan(chans_t *n)
{
  int i;
  chans_t *cur=NULL;
  
  tind = 0;
  
  if (!n || !chanl)
    return;
  
  if (curchan)
    cur = curchan->next;
  if (!cur)
    cur = chanl;
  for (i=0;;cur=cur->next)
  {
    if (i == 2)
      return;
    if (!cur)
      cur=chanl;
    if (!strcmp(cur->nm, n->nm))
      i++;
    else
      break;
  }
  curchan = cur;
  
  return;
}

static int
in_cksum(u_short *addr, int len)
{
	register int nleft = len;
	register u_short *w = addr;
	register int sum = 0;
	u_short answer = 0;

	/*
	 * Our algorithm is simple, using a 32 bit accumulator (sum), we add
	 * sequential 16 bit words to it, and at the end, fold back all the
	 * carry bits from the top 16 bits into the lower 16 bits.
	 */
	while (nleft > 1)  {
		sum += *w++;
		nleft -= 2;
	}

	/* mop up an odd byte, if necessary */
	if (nleft == 1) {
		*(u_char *)(&answer) = *(u_char *)w ;
		sum += answer;
	}

	/* add back carry outs from top 16 bits to low 16 bits */
	sum = (sum >> 16) + (sum & 0xffff);	/* add hi 16 to low 16 */
	sum += (sum >> 16);			/* add carry */
	answer = ~sum;				/* truncate to 16 bits */
	return(answer);
}

static void
tvsub(register struct timeval *out, register struct timeval *in)
{
	if ((out->tv_usec -= in->tv_usec) < 0) {
	  --out->tv_sec;
	  out->tv_usec += 1000000;
	}
	out->tv_sec -= in->tv_sec;
}

/* receive ping response */
int icmpin(WINDOW *win, sock_t *m)
{
  int s = m->s;
  ssearch_t *cur;
  char buf[256];
  struct sockaddr_in frm;
  int frmlen = sizeof(frm);
  struct timeval tv;
  
  gettimeofday(&tv, NULL);
  
  memset(buf, 0, sizeof(buf));
  recvfrom(s, buf, sizeof(buf), 0, (struct sockaddr *)&frm, &frmlen);
  
  for (cur=search;;cur=cur->next)
  {
    if (!cur)
      break;
    if (cur->ip == frm.sin_addr.s_addr && !timerisset(&cur->r))
      memcpy(&cur->r, &tv, sizeof(tv));
  }
  
  return(1);
}

char **form_toks(char *buf, int *cnt)
{
  int i, j, k, l = 0, c=0;
  char **ret;
  
  ret = (char **)malloc(4096);
  ret[0] = NULL;
  
  for (j=0,i=0;buf[j];i++)
  {
    while (buf[j] == ' ')
      j++;
    if (!buf[j])
      break;
    ret[i] = (char *)malloc(2048);
    memset(ret[i], 0, 2048);
    if (buf[j] == '\\')
    {
      for (k=0,j++;buf[j]!=' '&&buf[j]&&k!=2048;k++,j++)
        ret[i][k] = buf[j];
      ret[i][k] = 0;
    }
    else if (buf[j] == '\'')
    {
      j++;
      while (buf[j] && buf[j] == ' ')
        j++;
      for (k=0;buf[j]!='\''&&buf[j]&&k!=2048;k++,j++)
        ret[i][k] = buf[j];
      ret[i][k--] = 0;
      while (k >= 0 && ret[i][k] == ' ')
        ret[i][k--] = 0;
      j++;
    }
    else if (buf[j] == '\"')
    {
      for (k=0,j++;buf[j]!='\"'&&buf[j]&&k!=2048;k++,j++)
        ret[i][k] = buf[j];
      ret[i][k] = 0;
      j++;
    }
    else if (buf[j] == '(')
    {
      for (k=0,c=0;buf[j]&&k!=2047;k++,j++)
      {
        if (buf[j] == ')' && c == 1)
          break;
        if (c < 0)
          c = 0;
        if (buf[j] == '(')
          c++;
        else if (buf[j] == ')')
          c--;
        ret[i][k] = buf[j];
      }
      if (buf[j])
      {
        ret[i][k] = ')';
        ret[i][k+1] = 0;
        j++;
      }
    }
    else if (buf[j] == '{')
    {
      for (k=0,c=0;buf[j]&&k!=2047;k++,j++)
      {
        if (buf[j] == '}' && c == 1)
          break;
        if (c < 0)
          c = 0;
        if (buf[j] == '{')
          c++;
        else if (buf[j] == '}')
          c--;
        ret[i][k] = buf[j];
      }
      if (buf[j])
      {
        ret[i][k] = '}';
        ret[i][k+1] = 0;
        j++;
      }
    }
    else
    {
      for (k=0;buf[j]!=' '&&buf[j]&&k!=2048;k++,j++)
        ret[i][k] = buf[j];
      ret[i][k] = 0;
    }
    ret[i] = (char *)realloc(ret[i], strlen(ret[i])+1);
    l+=strlen(ret[i])+1;
  }
  
  ret[i] = NULL;
  
  *cnt = i;
  
  return(ret);
}

/* break up the contents of a server packet (or remote client packet)
 * into an array of tokens */
char **form_tokso(char *buf, int *cnt)
{
  int i, j, k, l = 0;
  char **ret;
  
  ret = (char **)malloc(4096);
  ret[0] = NULL;
  
  for (j=0,i=0;buf[j];i++)
  {
    while (buf[j] == ' ')
      j++;
    if (!buf[j])
      break;
    ret[i] = (char *)malloc(2048);
    memset(ret[i], 0, 2048);
    if (buf[j] == '\"')
    {
      /* tokens in double quotes (e.g filenames) get copied without the
       * double quotes */
      for (k=0,j++;buf[j]!='\"'&&buf[j]&&k!=2048;k++,j++)
        ret[i][k] = buf[j];
      ret[i][k] = 0;
      j++;
    }
    else
    {
      for (k=0;buf[j]!=' '&&buf[j]&&k!=2048;k++,j++)
        ret[i][k] = buf[j];
      ret[i][k] = 0;
    }
    ret[i] = (char *)realloc(ret[i], strlen(ret[i])+1);
    l+=strlen(ret[i])+1;
  }
  
  ret[i] = NULL;
  
/*  ret = (char **)realloc(ret, l+sizeof(char *)); */
  
  *cnt = i;
  
  return(ret);
}

int parsein(int s, unsigned short op, char *buf, WINDOW *win)
{
  int i, cnt, r=0;
  char **tok, *b, *t=NULL;
  handler_t *cur;
  
  for (cur=hndhead;cur;cur=cur->next)
  {
    if (op == cur->op)
    {
      msprintf(&t, "%i %s", op, buf);
      tok = form_toks(t, &cnt);
      free(t);
      t = NULL;
      b = dohandler(cur, tok, cnt);
      for (i=0;b[i]=='\n'||b[i]==' ';i++);
      msprintf(&t, "/%s", b+i);
      free(b);
      r = parseout(s, t, win);
      free(t);
      for (i=0;i<cnt;i++)
        free(tok[i]);
      free(tok);
      break;
    }
  }
  
  if (r == 2)
    return(1);
  
  if (noprint)
    noprint = 2;
  
  if (strlen(buf))
  {
    tok = form_toks(buf, &cnt);
    r = userhandler(s, op, tok, cnt, buf, win);
    for (i=0;i<cnt;i++)
      free(tok[i]);
    free(tok);
  }
  else
    r = userhandler(s, op, NULL, 0, NULL, win);
  if (r == 2)
    return(1);
  
  for (i=0;;i++)
  {
    if (in[i].func == NULL)
    {
      if (info.d == 1)
      {
        wp(win, "UNKNOWN OP: (0x%x - %i) |%s|\n", op, op, buf);
        drw(win);
      }
      return(1);
    }
    
    if (in[i].op == op)
    {
      if (info.d == 2)
      {
        wp(win, "|%s| (0x%x)\n", buf, op);
        drw(win);
      }
      if (strlen(buf))
      {
        tok = form_tokso(buf, &cnt);
        r = in[i].func(s, buf, tok, cnt, win);
        for (i=0;i<cnt;i++)
          free(tok[i]);
        free(tok);
      }
      else
        r = in[i].func(s, NULL, NULL, 0, win);
      return(r);
    }
  }
}

char **fxv(char **in, int cnt, int *rcnt)
{
  int i, k, j, n=cnt, c=0;
  char **ret;
  
  ret = (char **)malloc(4096);
  
  for (i=0,k=0,*rcnt=0;in[i];i++,k++)
  {
    if (*in[i] == '$' && strchr(in[i], '('))
    {
      ret[k] = (char *)malloc(512);
      ret[k+1] = NULL;
      memset(ret[k], 0, 512);
      do
      {
        if (!in[i])
          break;
        for (j=0;in[i][j];j++)
        {
          if (c < 0)
            c = 0;
          if (in[i][j] == '(')
            c++;
          else if (in[i][j] == ')')
            c--;
        }
        strcat(ret[k], in[i++]);
        strcat(ret[k], " ");
      } while (c);
      
      ret[k][strlen(ret[k])-1] = 0;
      ret[k] = (char *)realloc(ret[k], strlen(ret[k])+1);
      if (!in[i])
      {
        k++;
        break;
      }
      else
        i--;
    }
    else
    {
      ret[k] = strdup(in[i]);
      ret[k+1] = NULL;
    }
  }
  
  if (ret[k])
    k++;
  ret[k] = NULL;
  *rcnt = k;
  
  for (i=0;i<n;i++)
    free(in[i]);
  free(in);
  
  return(ret);
}

int parseout(int s, char *buf, WINDOW *win)
{
  char **tok, *st, t[2048], *b, *p, nb[2048];
  alias_t *cur;
  int i, j=0, k, cnt, c = 0, r=1, rn, sw=0, js=0, x, y, f=0, bc = 0;
  
  if (*buf == '/')
  {
    b = strdup(buf+1);
    sw = 1;
  }
  else
    b = strdup(buf);
  
  while (1)
  {
    if (!b[j])
      break;
    
    if (sw && !bc)
    {
      memset(t, 0, sizeof(t));
      while (b[j] && b[j] == ' ')
        j++;
      for (k=0,c=0,f=j;b[j];j++)
      {
        if (b[j] == '{')
          c++;
        else if (b[j] == '}')
          c--;
        else if ((b[j] == '|' || b[j] == '\n' || !b[j]) && !c)
          break;

        t[k] = b[j];
        k++;
        if (c < 0)
          c = 0;
      }
    }
    else
    {
      strcpy(t, b);
      j = strlen(t);
    }
    
    if (*t == '#')
    {
      if (b[j])
        j++;
      continue;
    }
    
    tok = form_toks(t, &cnt);
    if (tok[0] && *tok[0] == '#')
    {
      if (b[j])
        j++;
      for (i=0;i<cnt;i++)
        free(tok[i]);
      free(tok);
      continue;
    }
    
    if (*tok && *tok[0] == '.')
    {
      bc = 1;
      strcpy(tok[0], tok[0]+1);
      tok[0][strlen(tok[0])] = 0;
      strcpy(t, t+1);
      t[strlen(t)] = 0;
      f++;
    }
    
    for (cur=alhead;;cur=cur->next)
    {
      if (!cur || !tok[0])
      {
        if (sw && tok[0] && !bc)
        {
          js = 0;
          tok = fxv(tok, cnt, &cnt);
          for (i=0;i<cnt;i++)
          {
            if (*tok[i] == '$' && tok[i][1] && tok[i][1] != '+')
            {
              p = dovars(tok[i]+1);
              if (p)
              {
                free(tok[i]);
                tok[i] = p;
                js = 1;
              }
            }
          }
          for (i=0;i<cnt;i++)
          {
            if (*tok[i] == '$' && tok[i][1] == '+' && !tok[i][2])
            {
              if (i > 0 && tok[i-1] && tok[i+1])
              {
                for (x=1;tok[i-x]&&!(*tok[i-x]);x++);
                for (y=1;tok[i+y]&&!(*tok[i+y]);y++);
                if (!tok[i-x])
                  x--;
                if (!tok[i+y])
                  y--;
                tok[i] = (char *)realloc(tok[i], strlen(tok[i-x])+strlen(tok[i+y])+1);
                if (strcmp(tok[i-x], "$+"))
                  strcpy(tok[i], tok[i-x]);
                else
                  *tok[i] = 0;
                if (strcmp(tok[i+y], "$+"))
                  strcat(tok[i], tok[i+y]);
                *tok[i-x] = 0;
                *tok[i+y] = 0;
              }
              else
                *tok[i] = 0;
              js = 1;
            }
          }
          if (js)
          {
            memset(nb, 0, sizeof(nb));
            for (i=0;tok[i];i++)
            {
              if (*tok[i])
              {
                strcat(nb, tok[i]);
                strcat(nb, " ");
              }
            }
            nb[strlen(nb)-1] = 0;
            p = ins(b, nb, j-strlen(t), j);
            j = f;
            free(b);
            b = p;
            break;
          }
        }
        if (b[j])
          j++;
        break;
      }
      if (!strcasecmp(tok[0], cur->nm))
      {
        st = doalias(cur, tok, cnt);
        if (!st)
        {
          wp(win, "%s* Error: insufficient parameters%s\n", RED, WHITE);
          drw(win);
          free(b);
          free(st);
          for (i=0;i<cnt;i++)
            free(tok[i]);
          free(tok);
          return(1);
        }
        p = ins(b, st, j-strlen(t), j);
        free(b);
        b = p;
        free(st);
        j = f;
        sw = 1;
        break;
      }
    }
    
    for (i=0;i<cnt;i++)
      free(tok[i]);
    free(tok);
    
    if (j != f)
    {
      for (i=f,k=0;i<j;i++,k++)
        t[k] = b[i];
      t[k] = 0;
      if (sw)
      {
        if (t[strlen(t)-1] == '|' || t[strlen(t)-1] == '\n')
          t[strlen(t)-1] = 0;
        for (i=strlen(t)-1;t[i]==' '&&i>=0;i--)
          t[i] = 0;
      }
      rn = parseoutn(s, t, win);
      if (rn == -3) /* returned by ddone() */
      {
        free(b);
        return(-3);
      }
      else if (rn == -4) /* returned by dbreak() */
      {
        free(b);
        return(-4);
      }
      else if (rn == 2) /* returned by dstop() */
      {
        free(b);
        return(2);
      }
      if (rn < r)
        r = rn;
      if (bc == 1)
        bc = 0;
    }
  }
  
  free(b);
  return(r);
}

int parseoutn(int s, char *buf, WINDOW *win)
{
  int i, cnt, r = 1;
  char **tok;
  unsigned char *t = buf, f=0;
  
  tok = form_toks(t, &cnt);
  if (!tok[0])
  {
    for (i=0;i<cnt;i++)
      free(tok[i]);
    free(tok);
    return(1);
  }
  
  r = usercmd(s, tok, cnt, t, win);
  if (r == 2)
  {
    for (i=0;i<cnt;i++)
      free(tok[i]);
    free(tok);
    return(1);
  }
  
  if (ircmode && strcasecmp(tok[0], "server"))
  {
    if (!ircsock)
      return(1);
    if (!checkouts(ircsock, t, tok, cnt, win))
      f = 1;
    else
    {
      for (i=0;i<cnt;i++)
        free(tok[i]);
      free(tok);
      return(1);
    }
  }
  
  for (i=0;;i++)
  {
    if (out[i].func == NULL)
    {
      if (!r && !f)
      {
        wp(win, "%s* Unknown command: %s%s\n", RED, tok[0], WHITE);
        drw(win);
      }
      else if (f)
        ssock(ircsock, "%s\n", t);
      for (i=0;i<cnt;i++)
        free(tok[i]);
      free(tok);
      return(1);
    }
    
    /* original code:   
       if ((f && out[i].a && !strcasecmp(tok[0], out[i].nm)) 
           || (!f && !strcasecmp(tok[0], out[i].nm))) */
    if ((!f || out[i].a) && !strcasecmp(tok[0], out[i].nm))
    {
      r = out[i].func(s, t, tok, cnt, win);
      for (i=0;i<cnt;i++)
        free(tok[i]);
      free(tok);
      return(r);
    }
  }
 
  return(r);
}

/* return a unique number for (internal) identification of a
   particular upload/download. These are the numbers that make up the
   names of the connection, such as "u 5". They bear no relationship
   to the numbers that are displayed to the user with /pdown and /pup. 
   t is 1 for uploads and 0 for downloads - not that it really matters. 

   This function used to always return numbers between 0 and
   info.maxuploads/MAXDOWNLOADS. But the return value of -1 (no number
   available) was never checked for, so one had to check anyway that
   the number of up/downloads was bounded. (In the case of downloads,
   this check was omitted, leading to a potential segfault if two
   downloads both got the connection name "d -1".). I can see no harm
   in just numbering uploads and downloads sequentially, through the
   lifetime of nap. All that matters is that the numbers are
   unique. So we rotate through the integers mod 2^^32. Negative ones
   are fine, too. */
int gnum(unsigned char t)
{
  static int count[2] = { 0, 0 };
  
  return count[t]++;
}

char *cstr(char *str, int n)
{
  int i=0, c;
  
  if (ctbuf)
    free(ctbuf);
  
  while (str[i] == ' ' && str[i])
    i++;
    
  if (!str[i])
  {
    ctbuf = strdup("");
    return(ctbuf);
  }
  
  for (c=0;str[i]&&c!=n;i++)
  {
    if (str[i] == ' ')
    {
      c++;
      if (c == n)
        break;
      while (str[i] == ' ' && str[i])
        i++;
      if (!str[i])
        break;
    }
  }
  
  if (!str[i])
    ctbuf = strdup("");
  else
    ctbuf = strdup(str+i+1);
  
  return(ctbuf);
}

char *cspstr(char **tok, int cnt, int beg)
{
  int i;
  
  if (ctbuf)
    free(ctbuf);
  
  if (beg > cnt)
  {
    ctbuf = strdup("");
    return(ctbuf);
  }
  
  ctbuf = (char *)malloc(4096);
  memset(ctbuf, 0, 4096);
  
  for (i=beg;i<cnt;i++)
  {
    strcat(ctbuf, tok[i]);
    strcat(ctbuf, " ");
  }
  
  ctbuf[strlen(ctbuf)-1] = 0;
  ctbuf = (char *)realloc(ctbuf, strlen(ctbuf)+1);
  
  return(ctbuf);
}

int fcnt(file_t *t)
{
  file_t *cur;
  int i;

  for (i=0,cur=t;cur!=NULL;cur=cur->next)
      i++;
  
  return(i);
}

/* eliminated fcntr, since it was identical to fcnt -PS */
        
/* note that we now take a file_t* argument, not a (nm,fn) argument as
   in the original. -PS */
void dupload(file_t *task)
{
  file_t *cur, *cur1=NULL;
  
  for (cur=up;;cur=cur->next)
  {
    if (cur == NULL)
      return;
    if (cur == task)
      break;
    cur1 = cur;
  }
  
  if (cur1)
    cur1->next = cur->next;
  else if (cur->next)
    up = cur->next;
  else
    up = NULL;
  if (cur->f)
    fclose(cur->f);
  free(cur->nm);
  free(cur->fn);
  free(cur);
}

/* note that we now take a file_t* argument, not a (nm,fn) argument. -PS */
void ddownload(file_t *task)
{
  file_t *cur, *cur1=NULL;
  
  for (cur=down;;cur=cur->next)
  {
    if (cur == NULL)
      return;
    if (cur == task)
      break;
    cur1 = cur;
  }
  
#ifdef QUEUE
  /* added a hook here for dequeuing items from the download queue.
   * Everytime a download completes or is deleted we see if there are
   * any files from the same nick in the queue. */
  downqhook(cur->nm);
#endif /* QUEUE */

  if (cur1)
    cur1->next = cur->next;
  else if (cur->next)
    down = cur->next;
  else
    down = NULL;
  if (cur->f)
    fclose(cur->f);
  free(cur->nm);
  free(cur->fn);
  free(cur);
}

/* deleted function tlist since it was special case of ffile -PS */

/* find a file in the upload- or download list by username and filename,
   return NULL if not found. */
file_t *ffile(file_t *h, char *nm, char *fn)
{
  file_t *cur;
  
  for (cur=h;;cur=cur->next)
  {
    if (cur == NULL)
      return(NULL);
    if (!strcasecmp(cur->nm, nm) && !strcasecmp(cur->fn, fn))
      break;
  }
  
  return(cur);
}

#ifdef QUEUE
/* append a copy of ssearch_t x to the download queue downq
 * return -1 if there is a duplicate entry in the download list
 * return -2 if there is a duplicate entry in the download queue
 * return 0 if there is a download in progress or pending from the same nick
 *    (so we just enqueue the item)
 * return 1 if there is no download in progress or pending from the same nick
 *    (so we enqueue the item AND tell the calling function to send a download
 *    request for the file) 
 * Called by dg() */
int enqdown(ssearch_t *x)
{
  ssearch_t *z, *cur, *prev;
  file_t *fcur, *fprev;
  int nickindownlist, nickinqueue;

  /* first check the download list for a duplicate entry */
  nickindownlist = 0;
  for (fcur=down; fcur!=NULL; fcur=fcur->next)
  {
    if (!strcasecmp(fcur->nm, x->cmp))
    {
      nickindownlist = 1;
      if (!strcasecmp(fcur->fn, x->song))
        return(-1); /* found a duplicate entry in download list */
    }
  }
  /* at this point, nickindownlist tells us whether or not there is a
   * download in progress from the same nick */
  /* Now check the download queue for a duplicate entry */
  nickinqueue = 0;
  prev = NULL;
  for (cur=downq; cur!=NULL; cur=cur->next)
  {
    if (!strcasecmp(cur->cmp, x->cmp))
    {
      nickinqueue = 1;
      if (!strcasecmp(cur->song, x->song))
        return(-2); /* found a duplicate entry in download queue */
    }
    prev = cur;
  }
  /* prev now points to the end of the download queue */
  /* there were no duplicates, so enqueue a copy of x */
  z = (ssearch_t *) malloc(sizeof(ssearch_t));
  if (prev == NULL) /* (i.e. downq == NULL) queue was empty */
  {
    downq = z;
  }
  else
  {
    prev->next = z;
  }
  *z = *x; /* copy all the fields */
  z->num = -1; /* num field won't be used */
  z->next = NULL;
  z->song = strdup(x->song);
  z->fn = strdup(x->fn);
  z->cmp = strdup(x->cmp);
  if (!nickindownlist && !nickinqueue)
    return(1); 
  else
    return(0);
}

/* Remove ssearch_t x from the download queue downq.
 * Called by ddown() to remove items from the download queue. */
void deqdown(ssearch_t *x)
{
  ssearch_t *cur, *prev;

  prev = NULL;
  for (cur=downq; cur!=NULL; cur=cur->next)
  {
    if (cur == x)
      break;
    prev = cur;
  }
  if (cur == NULL) /* no match possibly because list was empty */
    return;
  else if (prev == NULL) /* removing first element in list */
    downq = cur->next;
  else 
    prev->next = cur->next;
  /* free the strings first */
  free(cur->song);
  free(cur->fn);
  free(cur->cmp);
  free(cur);
  
}

/* Remove file fn from nick nm from the download queue downq.
 * fn is the full filename (including path).
 * Called by sget() and snget() to remove an item from the download queue. */
int deqdown2(char *nm, char *fn)
{
  ssearch_t *cur, *prev;

  prev = NULL;
  for (cur=downq; cur!=NULL; cur=cur->next)
  {
    if (!strcasecmp(cur->cmp, nm) && !strcasecmp(cur->fn, fn))
      break;
    prev = cur;
  }
  if (cur == NULL) /* no match */
    return(0);
  else if (prev == NULL) /* removing first element in list */
    downq = cur->next;
  else 
    prev->next = cur->next;
  /* free the strings first */
  free(cur->song);
  free(cur->fn);
  free(cur->cmp);
  free(cur);
  return(1);
}

/* Search the download queue for a file from nick and send a download request
 * for it.
 * Called by ddownload() (whenever a download completes or is deleted)
 * and by dforceq() (to force a download request to be issued for an item 
 * that is stuck in the queue) */
int downqhook(char *nick)
{
  ssearch_t *cur, *prev;
  sock_t *sk;
  
  prev = NULL;
  for (cur=downq; cur!=NULL; cur=cur->next)
  {
    if (!strcasecmp(cur->cmp, nick))
      break;
    prev = cur;
  }
  if (cur == NULL) /* no files from nick in queue */
    return(0);
  
  sk = findsock("server");
  if (sk)
  {
    sendpack(sk->s, F_DGET, "%s \"%s\"", cur->cmp, cur->fn);
  }
  return(1);
  /* If we are not connected to the server, we have a problem...
   * If we can't issue a download request now, all the items in the 
   * queue from this nick will get stuck in the queue (even if the connection
   * to the server is re-established later). This is because a queued
   * item is only dequeued when a download from the _same nick_ completes
   * or is deleted.
   *   We could empty the queue of all files from this nick or 
   * maybe add a check for stuck queue items, i.e. items that might
   * remain stuck in the queue forever because there is no download in
   * progress from the same nick. 
   *   Currently, the user can either manually delete stuck items from 
   * the queue with /ddown or use /forceq to issue a download request 
   * for the first stuck item from a specified nick */
}
#endif /* QUEUE */

void adduser(chans_t *c, char *nm, int scount, unsigned char conn)
{
  user_t *cur, *cur1 = NULL;
  
  if (!c)
    return;
  
  if (!c->users)
  {
    c->users = (user_t *)malloc(sizeof(user_t));
    cur = c->users;
  }
  else
  {
    for (cur=c->users;cur;cur=cur->next)
    {
      if (!strcasecmp(cur->nm, nm))
        return;
      cur1 = cur;
    }
    cur = (user_t *)malloc(sizeof(user_t));
    cur1->next = cur;
  }
  
  cur->nm = strdup(nm);
  cur->addr = NULL;
  cur->conn = conn;
  cur->scount = scount;
  cur->flag = 0;
  cur->next = NULL;
}

void deluser(chans_t *c, char *nm)
{
  user_t *cur, *cur1 = NULL;
  
  for (cur=c->users;;cur=cur->next)
  {
    if (!cur)
      return;
    if (!strcasecmp(cur->nm, nm))
      break;
    cur1 = cur;
  }
  
  if (cur1)
    cur1->next = cur->next;
  else if (cur->next)
    c->users = cur->next;
  else
    c->users = NULL;
  
  free(cur->nm);
  if (cur->addr)
    free(cur->addr);
  free(cur);
}

/* find user or return NULL if not found */
user_t *finduser(chans_t *c, char *nm)
{
  user_t *cur;

  if (!nm)
    return(NULL);

  for (cur=c->users;;cur=cur->next)
  {
    if (!cur)
      return(NULL);
    if (!strcasecmp(cur->nm, nm))
      return(cur);
  }
}


I_NAP_FUNC(ssay)
{
  if (!(recent = findchan(chanl, tok[0])))
    return(1);
  if (!strcasecmp(info.user, tok[1]) && recent == curchan)
    wp(win, "%s<%s%s%s>%s %s\n", BRIGHT(MAGENTA), WHITE, tok[1], BRIGHT(MAGENTA), WHITE, str+strlen(tok[1])+strlen(tok[0])+2);
  else if (!strcasecmp(info.user, tok[1]))
    wp(win, "%s<%s%s%s:%s%s%s>%s %s\n", MAGENTA, WHITE, tok[1], DARK, WHITE, tok[0], MAGENTA, WHITE, str+strlen(tok[1])+strlen(tok[0])+2);
  else if (!strcasecmp(curchan->nm, tok[0]) || wmode)
    wp(win, "%s<%s%s%s>%s %s\n", BRIGHT(BLUE), WHITE, tok[1], BRIGHT(BLUE), WHITE, str+strlen(tok[1])+strlen(tok[0])+2);
  else
    wp(win, "%s<%s%s%s:%s%s%s>%s %s\n", BLUE, WHITE, tok[1], DARK, WHITE, tok[0], BLUE, WHITE, str+strlen(tok[1])+strlen(tok[0])+2);
  drw(win);
  recent = NULL;
  
  return(1);
}

I_NAP_FUNC(stell)
{
  recent = findquery(chanl, tok[0]);

  wp(win, "%s* [%s%s%s]%s %s\n", BRIGHT(GREEN), WHITE, tok[0], BRIGHT(GREEN), WHITE, str+strlen(tok[0])+1);
  drw(win);
  recent = NULL;
    
  return(1);
}

I_NAP_FUNC(sjoin)
{
  if (!(recent = findchan(chanl, tok[0])))
    return(1);
  if (!strcasecmp(tok[1], info.user))
    wp(win, "\n");
  wp(win, "%s* %s (%s%s%s) [%ssharing %i songs%s] has joined %s.%s\n", BRIGHT(GREEN), tok[1], WHITE, atoi(tok[3])>10?"Unknown":conns[atoi(tok[3])], BRIGHT(GREEN), WHITE, atoi(tok[2]), BRIGHT(GREEN), tok[0], WHITE);
  drw(win);
  adduser(recent, tok[1], atoi(tok[2]), atoi(tok[3]));
  recent = NULL;
  
  return(1);
}

I_NAP_FUNC(spart)
{
  if (!(recent = findchan(chanl, tok[0])))
    return(1);
  wp(win, "%s* %s (%s%s%s) [%ssharing %i songs%s] has left %s.%s\n", GREEN, tok[1], WHITE, conns[atoi(tok[3])], GREEN, WHITE, atoi(tok[2]), GREEN, tok[0], WHITE);
  drw(win);
  deluser(recent, tok[1]);
  recent = NULL;
  
  return(1);
}

I_NAP_FUNC(stopic)
{
  if (!(recent = findchan(chanl, tok[0])))
    return(1);
  wp(win, "%s* Topic for %s: %s%s\n", YELLOW, tok[0], str+strlen(tok[0])+1, WHITE);
  drw(win);
  
  if (recent->topic)
    free(recent->topic);
  recent->topic = strdup(str+strlen(tok[0])+1);
  if (recent == curchan)
    tind = 0;
  recent = NULL;
  
  return(1);
}

I_NAP_FUNC(sjchan)
{
  chans_t *cur, *cur1=NULL;

  if (!chanl)
  {
    chanl = (chans_t *)malloc(sizeof(chans_t));
    cur = chanl;
  }
  else
  {
    for (cur=chanl;cur!=NULL;cur=cur->next)
      cur1 = cur;
    cur = (chans_t *)malloc(sizeof(chans_t));
    cur1->next = cur;
  }
  
  cur->nm = strdup(tok[0]);
  cur->users = NULL;
  cur->q = 0;
  cur->topic = NULL;
  cur->l = 0;
  cur->flag = 0;
  cur->key = NULL;
  cur->p = 0;
  cur->next = NULL;
  curchan = cur;
  
  tin = -1;

  return(1);
}

/* 0x194 (404) server error message */
I_NAP_FUNC(snchan)
{
  if (dwi && ((!strcasecmp(tok[0], "information") && !strcasecmp(tok[2], dwi)) || (!strcasecmp(tok[0], "user") && !strcasecmp(tok[1], dwi))))
  {
    wp(win, "* Unable to resolve %s\n", dwi);
    drw(win);
    free(dwi);
    dwi = NULL;
  }

  wp(win, "%s* Server: %s%s\n", RED, str, WHITE);
  drw(win);
  
  if (!strcmp(str, "You are now cloaked.") || !strcmp(str, "You are cloaked."))
    cloaked = 1;
  else if (!strcmp(str, "You are no longer cloaked."))
    cloaked = 0;
  return(1);
}

I_NAP_FUNC(snotice)
{
  wp(win, "%s* %s%s\n", YELLOW, (str)?str:"", WHITE);
  drw(win);
  return(1);
}

I_NAP_FUNC(sscount)
{
  scount.libraries = atoi(tok[0]);
  scount.songs = atoi(tok[1]);
  scount.gigs = atoi(tok[2]);
  return(1);
}

I_NAP_FUNC(swhois)
{
  int hr, min, sec, t;
  struct hostent *d1, *d2;
  struct sockaddr_in in;
  
  if (dwi && !strcasecmp(dwi, tok[0]))
  {
    free(dwi);
    dwi = NULL;
    if (num < 13)
      return(1);
    if (fork())
      return(1);
      
    d1 = gethostbyname(tok[12]);
    if (!d1)
    {
      ssock(ipcs[1], "* Unable to resolve %s\n", tok[0]);
      exit(1);
    }
    
    memcpy(&in.sin_addr, d1->h_addr, d1->h_length);
    d2 = gethostbyaddr((char *)&in.sin_addr, d1->h_length, AF_INET);
    if (!d2)
    {
      ssock(ipcs[1], "* %s is %s\n", tok[0], inet_ntoa(in.sin_addr));
      exit(1);
    }
     
    ssock(ipcs[1], "* %s is %s [%s]\n", tok[0], d2->h_name, inet_ntoa(in.sin_addr));
    
    exit(1);
  }
  
  t = atoi(tok[2]);
  min = t/60;
  hr = min/60;
  sec = t%60;
  
  while (min >= 60)
    min-=60;

  wp(win, "%s ______________________%s\n", DARK, WHITE);
  wp(win, "%s|%s User: %s\n", DARK, WHITE, tok[0]);
  if (!strcasecmp(tok[0], "Ignitor"))
    wp(win, "%s|%s Class: Coder (%s)\n", DARK, WHITE, tok[1]);
  else if (!strcasecmp(tok[0], "nYtr0"))
    wp(win, "%s|%s Class: MoM (%s)\n", DARK, WHITE, tok[1]);
  else
    wp(win, "%s|%s Class: %s\n", DARK, WHITE, tok[1]);
  wp(win, "%s|%s Line: %s\n", DARK, WHITE, conns[atoi(tok[8])]);
  wp(win, "%s|%s Time: %i h %i m %i s\n", DARK, WHITE, hr, min, sec);
  wp(win, "%s|%s Channels: %s\n", DARK, WHITE, tok[3]);
  wp(win, "%s|%s Status: %s\n", DARK, WHITE, tok[4]);
  wp(win, "%s|%s Shared: %s\n", DARK, WHITE, tok[5]);
  wp(win, "%s|%s Client: %s\n", DARK, WHITE, tok[9]);
  wp(win, "%s|%s %s Downloading, %s Uploading\n", DARK, WHITE, tok[6], tok[7]);
  if (num > 10)
  {
    wp(win, "%s|%s %s Downloads, %s Uploads\n", DARK, WHITE, tok[10], tok[11]);
    wp(win, "%s|%s Address: %s\n", DARK, WHITE, tok[12]);
    wp(win, "%s|%s Data port: %s\n", DARK, WHITE, tok[14]);
  }
  drw(win);
  
  return(1);
}

I_NAP_FUNC(soff)
{
  time_t pt;
  char *t;
  
  if (dwi && !strcasecmp(dwi, tok[0]))
  {
    wp(win, "* Unable to resolve %s\n", tok[0]);
    drw(win);
    free(dwi);
    dwi = NULL;
  }
  
  pt = atol(tok[2]);
  t = ctime(&pt);
  t[strlen(t)-1] = '\0';
  t[strlen(t)] = 0;
  
  wp(win, "%s ______________________%s\n", DARK, WHITE);
  wp(win, "%s|%s User: %s\n", DARK, WHITE, tok[0]);
  if (!strcasecmp(tok[0], "Ignitor"))
    wp(win, "%s|%s Class: Coder (%s)\n", DARK, WHITE, tok[1]);
  else if (!strcasecmp(tok[0], "nYtr0"))
    wp(win, "%s|%s Class: MoM (%s)\n", DARK, WHITE, tok[1]);
  else
    wp(win, "%s|%s Class: %s\n", DARK, WHITE, tok[1]);
  wp(win, "%s|%s Last On: %s\n", DARK, WHITE, t);
  wp(win, "%s|%s Status: Offline\n", DARK, WHITE);
  drw(win);
  
  return(1);
}

#ifdef QUEUE
/* 620 (0x26c) remote client queue limit reached:
 * <nick> "<filename>" <filesize> <digit> */
I_NAP_FUNC(sqlimit)
{
  wp(win, "%s* Unable to get \"%s\" from %s (upload limit of %s reached)\n", \
      RED, tok[1], tok[0], tok[3], WHITE);
  drw(win);
  /* deqdown2(tok[0], tok[1]); */
  /* keep in queue and allow user to retry the download later with /forceq */

  return(1);
}
#endif /* QUEUE */

/* 206 (0xce) get error: <nick> "<filename>"  */
I_NAP_FUNC(snget)
{
  wp(win, "%s* Unable to get \"%s\" from %s (206 get error)\n", RED, \
      tok[1], tok[0], WHITE);
#ifdef QUEUE
  /* remove the corresponding item from the download queue */
  if (!deqdown2(tok[0], tok[1]))
  {
    wp(win, "%s* But it wasn't even in the download queue!%s\n", RED, WHITE);
  }
#endif /* QUEUE */
  drw(win);
  
  return(1);
}

#ifdef QUEUE
/* 609 (0x261) accept failed: <nick> "<filename>" */
I_NAP_FUNC(snacc)
{
  wp(win, "%s* Unable to get \"%s\" from %s (609 accept failed)\n", RED, \
      tok[1], tok[0], WHITE);
  /* remove the corresponding item from the download queue */
  if (!deqdown2(tok[0], tok[1]))
  {
    wp(win, "%s* But it wasn't even in the download queue!%s\n", RED, WHITE);
  }
  drw(win);
  
  return(1);
}
#endif /* QUEUE */

I_NAP_FUNC(nothing)
{
  return(1);
}

I_NAP_FUNC(smpart)
{
  chans_t *pt, *cur, *cur1=NULL;

  pt = findchan(chanl, tok[0]);
  
  if (!pt)
    return(1);
  
  wp(win, "%s* Left channel %s%s\n", GREEN, pt->nm, WHITE);
  drw(win);
  
  for (cur=chanl;cur!=pt;cur=cur->next)
    cur1 = cur;
  
  if (cur1)
    cur1->next = cur->next;
  else if (cur->next)
    chanl = cur->next;
  else
  {
    chanl = NULL;
    curchan = NULL;
  }
  if (curchan && !strcmp(pt->nm, curchan->nm))
    upchan(curchan);
  
  while (pt->users)
    deluser(pt, pt->users->nm);

  if (pt->topic)
    free(pt->topic);
  free(pt->nm);
  if (pt->key)
    free(pt->key);
  free(pt);
  
  if (wmode)
  {
    dscr(win);
    drw(win);
  }
  
  return(1);
}

I_NAP_FUNC(suser)
{
  int n, sp;
  char *c;
  
  if (num < 4)
    return(1);
  
  n = 16-strlen(tok[1]);
  sp = atoi(tok[3]);
  
  if (sp <= 3)
    c = RED;
  else if (sp <= 6)
    c = BRIGHT(YELLOW);
  else
    c = GREEN;
  
  recent = findchan(chanl, tok[0]);
  
  adduser(recent, tok[1], atoi(tok[2]), sp);
  
  if (tin == -1)
  {
    tin = 0;
    wp(win, "%s* Users:%s\n", YELLOW, WHITE);
  }

  if (tin == 4)
  {
    tin = 0;
    wp(win, "\n");
  }
  wp(win, "%s[%s%s", c, WHITE, tok[1]);
  for (;n>=0;n--)
    wp(win, " ");
  wp(win, "%s]%s", c, WHITE);
  
  recent = NULL;
  
  tin++;
  
  return(1);
}

/* server sends a 0x33e packet to end a list of users in a channel */
I_NAP_FUNC(snend)
{
  tin = -1;
  wp(win, "\n");
  drw(win);
  
  return(1);
}

/* 607 (0x25f) upload request: <nick> "<filename>" <speed> */
I_NAP_FUNC(sfreq)
{
  FILE *f;
  char *s1 = strdup(tok[1]), *nm=NULL;
  file_t *cur, *cur1=NULL;
  int i;
  
  for (i=0;i<strlen(s1);i++)
    if (s1[i] == '\\')
      s1[i] = '/';
  
  f = fopen(s1, "rb");
  if (!f)
  {
    sendpack(s, F_GFR, "%s", str);
    /* why accept the upload request?
     * According to drscholl's protocol spec there doesn't seem to be a
     * way for a client to reject an upload request. (Packet 609 "accept
     * failed" can only be sent by the server it seems.) Instead, we tell
     * the server that we are accepting the upload request, but when the
     * remote client connects to us, we will return an error message.
     * (ssf() and nap.c:doget() will send "INVALID REQUEST" because the file
     * won't be in the upload list.)
     * (If we send nothing at all, will the server timeout and send a 206
     * (get error) or 609 (accept failed) to the remote client?) */
    wp(win, "%s* File \"%s\" not found (requested by %s)%s\n", RED, s1, \
       tok[0], WHITE);
    drw(win);
    free(s1);
    return(1);
  }
  if (fcnt(up) == info.maxuploads)
  {
    /* sendpack(s, F_GFR, "%s", str); */ /* why is this here ? */
    sendpack(s, F_QLIMIT, "%s \"%s\" %i", tok[0], tok[1], info.maxuploads);
    wp(win, "%s* Too many uploads (can't send \"%s\" to %s)%s\n", RED, tok[1], \
        tok[0], WHITE);
    drw(win);
    fclose(f);
    free(s1);
    return(1);
  }
  else if (ffile(up, tok[0], tok[1]) != NULL)
  {
    /* we are already uploading the file to nick */
    wp(win, "%s* Already uploading \"%s\" to %s%s\n", RED, tok[1], tok[0], \
        WHITE);
    drw(win);
    /* send a F_QLIMIT or send nothing ? */
    fclose(f);
  }
  else
  {
    for (cur=up;cur!=NULL;cur=cur->next)
      cur1 = cur;
    if (!cur1)
    {
      up = (file_t *)malloc(sizeof(file_t));
      cur = up;
    }
    else
    {
      cur = (file_t *)malloc(sizeof(file_t));
      cur1->next = cur;
    }
    cur->rfn = strdup(s1);
    cur->fn = strdup(tok[1]);
    cur->nm = strdup(tok[0]);
    cur->f = f;
    cur->csz = cur->bsz = cur->sz = 0;
    cur->g = 0;
    cur->st = 0;
    /* the following is now okay because (nm,fn) uniquely determines task */
    msprintf(&nm, "dup \"%s\" \"%s\"", cur->nm, cur->fn);
    cur->timer = addtimer(90, s, nm, win);
    free(nm);
    cur->sk = NULL; /* assigned in ssf() or nap.c:doget() */
    cur->next = NULL;
    /* sendpack(s, F_SX, "%s", cur->nm); */ /* why send a link speed query? */
    /* accept upload request: <nick> "<filename>" */
    sendpack(s, F_GFR, "%s", str); 
  }

  free(s1);
  return(1);
}

/* link speed response: <nick> <linespeed> */
I_NAP_FUNC(sry)
{
  return(1);
}

/* 501 (0x1f5) alternate download ack: 
 * <nick> <ip> <port> "<filename>" <md5> <speed>
 * we are being asked to UPLOAD a file to a remote client (presumably
 * because we are firewalled and cannot accept incoming connections) */
I_NAP_FUNC(ssf)
{
  file_t *cur;
  sock_t *sk=NULL;
  int k;
  char *sname=NULL;
  struct sockaddr_in dst;
  
  cur = ffile(up, tok[0], tok[3]);
  if (!cur)
  {
    wp(win, "%s* %s requested alternate download of \"%s\" but the file " \
        "is not in the upload list! %s\n", RED, tok[0], tok[3], WHITE);
    drw(win);
    return(1);
  }
  
  dst.sin_addr.s_addr = strtoul(tok[1], (char **)NULL, 10);
  dst.sin_port = htons(atoi(tok[2]));
  dst.sin_family = AF_INET;
  
  k = socket(AF_INET, SOCK_STREAM, 0);
  
  switch (fork())
  {
    case 0:
      connect(k, (struct sockaddr *)&dst, sizeof(dst));
      exit(1);
      break;
    default:
      break;
  }
  
  msprintf(&sname, "u %i", gnum(1));
  
  addsock(k, sname, S_R, initsend);
  sk = findsock(sname);
  free(sname);
  
  sk->d = cur;
  cur->sk = sk;
  
  return(1);
}

/* 0xcc: got a download ack from the server. It is a string of the form:
 * <nick> <ip> <port> "<filename>" <md5> <linespeed>
 *   0      1     2        3          4       5       <-- token numbers */
I_NAP_FUNC(sget)
{
  file_t *cur, *cur1=NULL;
  struct sockaddr_in dst;
  int k;
  char *sname=NULL, *t=NULL, *nm=NULL;
  sock_t *sk=NULL;
  int remoteport;

  if (strchr(tok[3], '/'))
    t = strrchr(tok[3], '/')+1;
  else if (strchr(tok[3], '\\'))
    t = strrchr(tok[3], '\\')+1;
  else
    t = tok[3];
  
#ifdef QUEUE
  /* remove the (first matching) item from the download queue */
  if (!deqdown2(tok[0], tok[3]))
  {
    /* should abort if there is no match! */
    wp(win, "%s* Got a download ack for \"%s\" from %s, but there is no such" \
       " item in the download queue!%s\n", RED, tok[3], tok[0], WHITE);
    drw(win);
    return(1);
  }
#endif /* QUEUE */

  if (ffile(down, tok[0], t))
  {
    wp(win, "%s* Unable to get \"%s\" from %s because you are already " \
        "getting it%s\n", RED, tok[3], tok[0], WHITE);
    drw(win);
    return(1);
  }
  
  remoteport = atoi(tok[2]);
  if (!remoteport && !info.port)
  {
    wp(win, "%s* Unable to get \"%s\" from %s because both parties are " \
        "firewalled%s\n", RED, tok[3], tok[0], WHITE);
    drw(win);
    return(1);
  }
  else if (!remoteport)
  {
    /* remote client is firewalled (port=0), so send an alternate download
     * (push) request to server so that remote client will connect to us */
    sendpack(s, F_DSF, "%s \"%s\"", tok[0], tok[3]);
  }
  else
  {
    /* connect to the remote client at the given port */
    /* sendpack(s, F_SX, "%s", tok[0]); */ /* why send a link speed query? */
    
    k = socket(AF_INET, SOCK_STREAM, 0);
  
    dst.sin_addr.s_addr = strtoul(tok[1], (char **)NULL, 10);
    dst.sin_port = htons(remoteport);
    dst.sin_family = AF_INET;
    
    switch (fork())
    {
      case 0:
        connect(k, (struct sockaddr *)&dst, sizeof(dst));
        /* if connect returns an error, I assume it will be caught in
         * nap.c:initget() because ssock() will return an error (?) */
        exit(1);
        break;
      default:
        break;
    }

    msprintf(&sname, "d %i", gnum(0));
    addsock(k, sname, S_R, initget);
    sk = findsock(sname);
    free(sname);
  }
  
  /* append the file to the download list */
  for (cur=down;cur!=NULL;cur=cur->next)
    cur1 = cur;
  if (!cur1)
  {
    down = (file_t *)malloc(sizeof(file_t));
    cur = down;
  }
  else
  {
    cur = (file_t *)malloc(sizeof(file_t));
    cur1->next = cur;
  }
  
  cur->addr = atol(tok[1]);
  cur->port = remoteport;
  cur->fn = strdup(t);
  cur->nm = strdup(tok[0]);
  cur->rfn = strdup(tok[3]);
  cur->sz = 0;
  cur->csz = 0;
  cur->bsz = 0;
  cur->g = 0;
  cur->st = 0;
  msprintf(&nm, "ddown \"%s\" \"%s\"", cur->nm, cur->fn);
  cur->timer = addtimer(90, s, nm, win);
  /* the timer is cleared in nap.c:gsize() or in nap.c:dosend() once the
   * download begins */
  free(nm);
  cur->sk = NULL;
  cur->next = NULL;
  
  t = NULL;
  /* note: the next line is okay even if info.down ends in "/"; unix
     tolerates "//". */
  msprintf(&t, "%s/%s", info.down, cur->fn);
  cur->f = fopen(t, "ab");
  drw(win);
  free(t);

  if (sk) /* this is a normal download (the remote client isn't firewalled) */
  {
    cur->sk = sk;
    sk->d = cur;
  }
  /* for downloads from a firewalled remote client, a sock_t hasn't
   * been created yet, and we must wait for the remote client to connect
   * to us before cur->sk can be assigned a value. See nap.c:dosend() */
  
  return(1);
}

I_NAP_FUNC(suon)
{
  hotlist_t *cur;
  
  if (!hlist) {   /* fix bug #116254, patch #401946, by Ville Aikas */
    return(1);
  }
  cur = findlinks(hlist, tok[0], REF(hotlist_t *, hlist)->nm);
  if (cur)
  {
    cur->conn = atoi(tok[1]);
    cur->on = 1;
  }
  
  wp(win, "%s* %s (%s) is on napster%s\n", BRIGHT(GREEN), tok[0], conns[atoi(tok[1])], WHITE);
  drw(win);
  
  return(1);
}

I_NAP_FUNC(suoff)
{
  hotlist_t *cur;
  
  if (!hlist) {   /* fix bug #116254, by Ville Aikas */
    return(1);
  }
  cur = findlinks(hlist, tok[0], REF(hotlist_t *, hlist)->nm);
  
  if (cur && cur->on)
  {
    wp(win, "%s* %s has left napster%s\n", GREEN, tok[0], WHITE);
    drw(win);
    cur->on = 0;
  }
  else if (cur)
    cur->on = 0;
  
  return(1);
}

/* 201 (0xc9) search response:
 * "<filename>" <md5> <size> <bitrate> <frequency> <length> <nick> <ip>
 *  <link-type> [weight] */
I_NAP_FUNC(ssret)
{
  ssearch_t *elt, *tmp;
  char *t;
  
  if (num < 9)
    return(1);
  
  /* calculate short filename */
  if (strchr(tok[0], '/'))
    t = strrchr(tok[0], '/')+1;
  else if (strchr(tok[0], '\\'))
    t = strrchr(tok[0], '\\')+1;
  else
    t = tok[0];
  
  /* create a new search list element */
  elt = (ssearch_t *)malloc(sizeof(ssearch_t));
  
  elt->song = strdup(t);
  elt->num = -1;
  elt->fn = strdup(tok[0]);
  elt->sz = atoi(tok[2]);
  elt->brate = atoi(tok[3]);
  elt->freq = atoi(tok[4]);
  elt->time = atoi(tok[5]);
  elt->cmp = strdup(tok[6]);
  elt->ip = strtoul(tok[7], (char **)NULL, 10);
  elt->conn = atoi(tok[8]);
  timerclear(&elt->s);
  timerclear(&elt->r);
  elt->ping = -2;
  elt->next = NULL;
  
  /* and insert it into the list. Note that we keep the list ordered
     by decreasing connection speed. */

  list_insert_ordered(ssearch_t, search, elt, tmp, elt->conn >= tmp->conn);

  return(1);
}

/* 202 (0xca) end of search response from server */
I_NAP_FUNC(ssend)
{
  ssearch_t *cur, *low=NULL, *cur1, *a, *b;
  int min, sec, i, c, lp, havepings=0;
  /* Note: noresultscreen is set within nap with "/set noresultscreen 1" */
  float mb;
  char *d;
  sock_t *sk;
  struct timeval tv;
  struct icmphdr *icmp;
  struct sockaddr_in dst;
  
  sk = findsock("icmp");  /* icmp is the socket for ping results, 
			     if they were requested */
  
  if (sk) /* if pings are requested, send and receive pings, and
	     reorder the list */
  {
    havepings = 1;

    /* send out pings */
    for (cur=search;cur;cur=cur->next)
    {
      if (timerisset(&cur->s))
        continue;
    
      dst.sin_addr.s_addr = cur->ip;
      dst.sin_family = AF_INET;

      d = (char *)malloc(64);

      memset(d, 0, 64);
      icmp = (struct icmphdr *)d;
      icmp->type = ICMP_ECHO;
      icmp->code = 0;
      icmp->un.echo.id = (getpid()&0xffff);
      icmp->un.echo.sequence = 0;
      icmp->checksum = in_cksum((u_short *)icmp, 64);
      
      sendto(sk->s, d, 64, 0, (struct sockaddr *)&dst, sizeof(dst));
      gettimeofday(&cur->s, NULL);
      free(d);
      
      for (cur1=search;cur1;cur1=cur1->next)
        if (cur1->ip == cur->ip)
          memcpy(&cur1->s, &cur->s, sizeof(cur->s));
    }
    
    if (search)
    {
      wp(win, "* Waiting for ping results...\n");
      drw(win);
      sockfunct(win, winput, 3);  /* wait 3 seconds */
    }
    
    sk = findsock("icmp");
    if (sk)
      delsock(sk->s);

    /* compute ping times (cur->r - cur->s) */
    for (cur=search;cur;cur=cur->next)
    {
      if (timerisset(&cur->r))
      {
        memcpy(&tv, &cur->r, sizeof(tv));
        tvsub(&tv, &cur->s);
        cur->ping = (tv.tv_sec*10000+(tv.tv_usec/100))/10;
        timerclear(&tv);
      }
      else
        cur->ping = -1;
    }
    
    /* sort the search results in ascending order of ping time, expect
       that a ping time of -1 is considered "infinity", i.e., greater
       than any other ping time. */

    list_sort(ssearch_t, search, a, b, 
	      a->ping != -1 && (b->ping == -1 || a->ping < b->ping));

  } /* end if (sk) */
  
  /* number the search results; note they are already ordered. */
  i=1;
  list_forall (cur, search) {
    cur->num = i;
    i++;
  }

  /* show search result on main screen or result screen, depending
     on whether "noresultscreen" was selected. */
  if (nvar("noresultscreen")==1) 
  {
    wp(win, "# | Song | Bitrate | Frequency | Length | Size | User | Speed%s\n", havepings ? " | Ping" : "");
    wp(win, "-------------------------------------------------------------%s\n", havepings ? "-------" : "");
    list_forall (cur, search) {
      min = cur->time/60;
      sec = cur->time%60;
      mb = ((float)cur->sz)/1048576.0;
      
      wp(win, "%s%i.%s %s%s%s %ibps %ihz %s%i:%02i %.02fmb%s %s %s%s%s", BOLD, cur->num, WHITE, BOLD, (fl)?cur->fn:cur->song, WHITE, cur->brate, cur->freq, BOLD, min, sec, mb, WHITE, cur->cmp, BOLD, conns[cur->conn], WHITE);
      if (havepings && cur->ping == -1)
	wp(win, " N/A");
      else if (havepings)
	wp(win, " %i", cur->ping);
      wp(win, "\n");
    }
    drw(win);
  }
  else 
  {
    sscr(search);
  }
  
  srch = 0;
  
  return(1);
} 

/* received a browse response from server:
 * <nick> "<filename>" <md5> <size> <bitrate> <frequency> <time> 
 *   0        1          2     3       4          5         6    */
I_NAP_FUNC(srbrowse)
{
  ssearch_t *cur, *cur1;
  char *t;
  
  if (num < 7)
    return(1);
  
  if (search == NULL)
  {
    search = (ssearch_t *)malloc(sizeof(ssearch_t));
    cur = search;
  }
  else
  {
    for (cur1=NULL,cur=search;cur!=NULL;cur=cur->next)
      cur1 = cur;
    cur = (ssearch_t *)malloc(sizeof(ssearch_t));
    cur1->next = cur;
  }
  
  if (strchr(tok[1], '/'))
    t = strrchr(tok[1], '/')+1;
  else if (strchr(tok[1], '\\'))
    t = strrchr(tok[1], '\\')+1;
  else
    t = tok[1];
  
  cur->num = -1;
  cur->ping = -2;
  cur->song = strdup(t);
  cur->fn = strdup(tok[1]);
  cur->sz = atoi(tok[3]);
  cur->brate = atoi(tok[4]);
  cur->freq = atoi(tok[5]);
  cur->time = atoi(tok[6]);
  cur->cmp = strdup(tok[0]);
  cur->conn = -1;
  cur->next = NULL;
  
  return(1);
}

/* received an "end of browse list" packet from server:
 * <nick> [ip] */
I_NAP_FUNC(sdbrowse)
{
  ssearch_t *cur;
  int min, sec, i, r = nvar("noresultscreen");
  float mb;
  sock_t *sk;
  
  srch = 0;
  
  if (r == 1)
  {
    wp(win, "Number | Song | Bitrate | Frequency | Length | Size | User\n");
    wp(win, "----------------------------------------------------------\n");
  }
  
  for (cur=search,i=1;cur!=NULL;cur=cur->next,i++)
  {
    cur->num = i;
    min = cur->time/60;
    sec = cur->time%60;
    mb = ((float)cur->sz)/1048576.0;
    if (r == 1)
      wp(win, "%s%i.%s %s%s%s %i %i %s%i:%02i %.02fmb%s %s\n", BOLD, i, WHITE, BOLD, cur->song, WHITE, cur->brate, cur->freq, BOLD, min, sec, mb, WHITE, cur->cmp);
  }
  
  if (r == 1)
    drw(win);
  else
  {
    sscr(search);
  }
  
  return(1);
}

I_NAP_FUNC(sop)
{
  wp(win, "%s!%s%s%s!%s %s\n", GREEN, WHITE, tok[0], GREEN, WHITE, str+strlen(tok[0])+1);
  drw(win);
  
  return(1);
}

I_NAP_FUNC(spchange)
{
  wp(win, "%s* Your data port was changed to %s%s\n", RED, tok[0], WHITE);
  drw(win);
  
  closefserv();
  initfserv(atoi(tok[0]), atoi(tok[0]));
  
  return(1);
}

I_NAP_FUNC(sbport)
{
  wp(win, "%s* Your dataport is misconfigured%s\n", RED, WHITE);
  drw(win);
  
  return(1);
}

I_NAP_FUNC(sannounce)
{
  wp(win, "%s[%s%s%s]%s %s\n", RED, WHITE, tok[0], RED, WHITE, str+strlen(tok[0])+1);
  drw(win);
  
  return(1);
}

I_NAP_FUNC(sbanlist)
{
  wp(win, "%s* Banned: %s%s\n", BRIGHT(BLUE), str, WHITE);
  drw(win);
  
  return(1);
}

I_NAP_FUNC(ssping)
{
  sendpack(s, SERVER_PING, NULL);
  
  return(1);
}

I_NAP_FUNC(scping)
{
  wp(win, "%s* Received PING from %s%s\n", RED, tok[0], WHITE);
  drw(win);
  sendpack(s, CLIENT_PONG, "%s", str);
  
  return(1);
}

I_NAP_FUNC(scpong)
{
  wp(win, "%s* Received PONG from %s%s\n", RED, tok[0], WHITE);
  drw(win);

  return(1);
}

I_NAP_FUNC(sredir)
{
  int t;
  sock_t *sk;
  chans_t *cur;
  char *t1=NULL;
  
  wp(win, "%sYou've been redirected%s\n", RED, WHITE);
  drw(win);
  
  msprintf(&t1, "%s:%s", tok[0], tok[1]);
  
  t = conn(t1, PORT);
  if (t == -1)
  {
    drw(win);
    free(t1);
    return(1);
  }
  free(t1);
  wp(win, "Logging in...\n");
  drw(win);
  if (login(t, info.user, info.pass, info.port, CLIENT, info.conn, info.email) == -1)
  {
    drw(win);
    close(t);
    return(1);
  }
  
  sk = findsock("server");
  if (sk)
    delsock(sk->s);
  addsock(t, "server", S_R, inserv);
  sk = findsock("server");
  
  curchan=NULL;
  
  if (chanl)
  {
    for (cur=chanl;;cur=cur->next)
      if (!cur->q)
        sendpack(sk->s, F_JOIN, "%s", cur->nm);
    delnapchans();
  }
  
  upchan(chanl);
  
  return(1);
}

I_NAP_FUNC(scycle)
{
  int t;
  sock_t *sk;
  chans_t *cur;
  
  wp(win, "%sYou've been cycled%s\n", RED, WHITE);
  drw(win);
  
  wp(win, "Getting best host...\n");
  drw(win);
  t = conn(tok[0], PORT);
  if (t == -1)
  {
    drw(win);
    return(1);
  }
  wp(win, "Logging in...\n");
  drw(win);
  if (login(t, info.user, info.pass, info.port, CLIENT, info.conn, info.email) == -1)
  {
    drw(win);
    close(t);
    return(1);
  }
  
  sk = findsock("server");
  if (sk)
    delsock(sk->s);
  addsock(t, "server", S_R, inserv);
  sk = findsock("server");
  
  curchan=NULL;
  
  if (chanl)
  {
    for (cur=chanl;;cur=cur->next)
      if (!cur->q)
        sendpack(sk->s, F_JOIN, "%s", cur->nm);
    delnapchans();
  }
  
  upchan(chanl);
  
  return(1);
}

I_NAP_FUNC(sclist)
{
  wp(win, "%s%s%s - %s - %s\n", BOLD, tok[0], WHITE, tok[1], str+strlen(tok[0])+strlen(tok[1])+2);
  drw(win);
  
  return(1);
}

I_NAP_FUNC(sclist2)
{
  wp(win, "%s%s%s - %s - %s\n", BOLD, tok[0], WHITE, tok[1], tok[5] /* str+strlen(tok[0])+strlen(tok[1])+strlen(tok[2])+strlen(tok[3])+strlen(tok[4])+5 */ );
  drw(win);
  
  return(1);
}

I_NAP_FUNC(sblocklist)
{
  wp(win, "%s* Blocked: %s%s\n", BRIGHT(BLUE), str, WHITE);
  drw(win);
  
  return(1);
}

I_NAP_FUNC(signorelist)
{
  wp(win, "%s* Ignored: %s%s\n", BRIGHT(BLUE), str, WHITE);
  drw(win);
  
  return(1);
}

I_NAP_FUNC(signoreend)
{
  wp(win, "%s* Total users ignored: %s%s\n", BRIGHT(BLUE), str, WHITE);
  drw(win);
  
  return(1);
}

I_NAP_FUNC(signoreadd)
{
  wp(win, "%s* Added %s to the ignore list%s\n", BRIGHT(BLUE), str, WHITE);
  drw(win);
  
  return(1);
}

I_NAP_FUNC(signoreremove)
{
  wp(win, "%s* Removed %s from the ignore list%s\n", BRIGHT(BLUE), str, WHITE);
  drw(win);
  
  return(1);
}

I_NAP_FUNC(signoreunknown)
{
  wp(win, "%s* %s is not ignored or does not exist%s\n", RED, str, WHITE);
  drw(win);
  
  return(1);
}

I_NAP_FUNC(signoreexists)
{
  wp(win, "%s* %s is already on your ignore list%s\n", RED, str, WHITE);
  drw(win);
  
  return(1);
}

I_NAP_FUNC(signoreclear)
{
  wp(win, "%s* Cleared %s users from your ignore list%s\n", BRIGHT(BLUE), str, WHITE);
  drw(win);
  
  return(1);
}

I_NAP_FUNC(signorefail)
{
  wp(win, "%s* Ignoring %s failed!%s\n", RED, str, WHITE);
  drw(win);
  
  return(1);
}

I_NAP_FUNC(scbanlist)
{
  wp(win, "%s* Channel ban: %s%s\n", BRIGHT(BLUE), str, WHITE);
  drw(win);
  
  return(1);
}

I_NAP_FUNC(snerr)
{
  wp(win, "%s* Could not add %s to your hotlist%s\n", RED, tok[0], WHITE);
  drw(win);
  
  return(1);
}

I_NAP_FUNC(snadd)
{
  FILE *f;
  char *t = NULL, rd[64];
    
  t = glistn(info.user);
  
  f = fopen(t, "r");
  if (f)
  {
    while (!feof(f))
    {
      memset(rd, 0, sizeof(rd));
      fgets(rd, sizeof(rd), f);
      if (!strlen(rd))
        break;
      rd[strlen(rd)-1] = 0;
      if (!strcasecmp(rd, tok[0]))
      {
        wp(win, "%s* Error: %s is already on your hotlist%s\n", RED, tok[0], WHITE);
        drw(win);
        free(t);
        fclose(f);
        return(1);
      }
    }

    fclose(f);
  }
  
  f = fopen(t, "a");
  free(t);
  if (!f)
  {
    wp(win, "%s* Added %s to your hotlist, but was unable to open your hotlist for writing%s\n", RED, tok[0], WHITE);
    drw(win);
    return(1);
  }
  
  fprintf(f, "%s\n", tok[0]);
  fclose(f);
  
  wp(win, "%s* Added %s to your hotlist%s\n", BRIGHT(BLUE), tok[0], WHITE);
  drw(win);
  
  return(1);
}

I_NAP_FUNC(sme)
{
  if (!(recent = findchan(chanl, tok[0])))
    return(1);
  if (!strcasecmp(curchan->nm, tok[0]) || wmode)
  {
    wp(win, "%s* %s%s %s\n", CYAN, tok[1], WHITE, tok[2]);
    drw(win);
  }
  else
  {
    wp(win, "%s* %s%s/%s%s%s %s\n", CYAN, tok[1], WHITE, CYAN, tok[0], WHITE, tok[2]);
    drw(win);
  }
  recent = NULL;
  
  return(1);
}


O_NAP_FUNC(dquit)
{
  if (num > 1 && !strcasecmp(tok[1], "yes"))
    return(-2);
  if (fcnt(up) || fcnt(down))
  {
    wp(win, "%s* You are uploading or downloading songs, use \"/quit yes\" if you want to quit%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  return(-2);
}

O_NAP_FUNC(dtquit)
{
  sock_t *sk;
  
  sk = findsock("ssock");
  if (sk)
    delsock(sk->s);
  sk = findsock("fserv");
  if (sk)
    delsock(sk->s);
  sk = findsock("ipc");
  if (sk)
    delsock(sk->s);
  
  return(-1);
}

O_NAP_FUNC(djoin)
{
  chans_t *t;

  if (tok[1] == NULL)
  {
    wp(win, "%s* You didn't enter a channel name%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  t = findchan(chanl, tok[1]);
  if (t && !t->p)
  {
    curchan = t;
    return(1);
  }
                  
  sendpack(s, F_JOIN, "%s", tok[1]);
  
  return(1);
}

O_NAP_FUNC(dpart)
{
  chans_t *pt, *cur, *cur1=NULL;

  if (num == 1)
    pt = curchan;
  else
    pt = findchan(chanl, tok[1]);
  
  if (!pt && num > 1)
  {
    wp(win, "%s* You are not on %s channel %s\n", RED, tok[1], WHITE);
    drw(win);
    return(1);
  }
  else if (!pt)
  {
    wp(win, "%s* You are not on a channel %s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  if (!pt->q)
  {
    sendpack(s, F_PART, "%s", pt->nm);
    pt->p |= 1;
    return(1);
  }
  else if (pt->q == 2)
  {
    ssock(ircsock, "PART %s\n", pt->nm);
    pt->p |= 1;
    return(1);
  }

  wp(win, "%s* Ended query with %s%s\n", GREEN, pt->nm, WHITE);
  drw(win);
  
  for (cur=chanl;cur!=pt;cur=cur->next)
    cur1 = cur;
  
  if (cur1)
    cur1->next = cur->next;
  else if (cur->next)
    chanl = cur->next;
  else
  {
    chanl = NULL;
    curchan = NULL;
  }
  if (curchan && !strcmp(pt->nm, curchan->nm))
    upchan(curchan);

  if (pt->topic)
    free(pt->topic);
  free(pt->nm);
  free(pt);
  
  if (wmode)
  {
    dscr(win);
    drw(win);
  }
  
  return(1);
}

O_NAP_FUNC(dtell)
{
  if (num < 3)
  {
    wp(win, "%s* Not enough parameters%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  sendpack(s, F_TELL, "%s", fxp(cstr(str, 1)));
  recent = findchan(chanl, tok[1]);
  wp(win, "%s* --> (%s%s%s)%s %s\n", GREEN, WHITE, tok[1], GREEN, WHITE, cstr(str, 2));
  drw(win);
  recent = NULL;
  return(1);
}

O_NAP_FUNC(dwhois)
{
  if (num < 2)
    sendpack(s, F_WHOIS, "%s", info.user);
  else
    sendpack(s, F_WHOIS, "%s", tok[1]);
  return(1);
}

/* get command (currently not used). syntax is "/cmdname nick filename" */
O_NAP_FUNC(dget)
{
  if (num < 3)
  {
    wp(win, "%s* Not enough parameters%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  wp(win, "%s \"%s\"\n", tok[1], str+strlen(tok[0])+strlen(tok[1])+2);
  sendpack(s, F_DGET, "%s \"%s\"", tok[1], str+strlen(tok[0])+strlen(tok[1])+2);
  return(1);
}

/* get files by number e.g. "/g 1,3-4,5" */
O_NAP_FUNC(dg)
{
  ssearch_t *cur=search;
  int i, e, tmp;
  char *t, *n;
#ifdef QUEUE
  int ret;
#endif /* QUEUE */

  if (num < 2)
  {
    wp(win, "%s* Not enough parameters%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  if (num > 3)
  {
    wp(win, "%s* Don't use spaces when listing multiple files, use commas to seperate%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  if (!cur)
  {
    wp(win, "%s* No search performed yet%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  if (!strcasecmp(cur->cmp, info.user))
  {
    wp(win, "%s* Can't get a file from yourself!%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  n = strdup(tok[1]);
  
  for (t = strtok(n, ","); t; t = strtok(NULL, ","))
  {
    i = e = 0;
    if (strchr(t, '-')) /* assume token is a range */
    {
      sscanf(t, "%i-%i", &i, &e);
      if (i > e)
      {
        tmp = i; i = e; e = tmp;
      }
      /* at this point i <= e */
    }
    else /* token is a single number */
    {
      i = atoi(t);
      e = i;
    }
    if (i > 0 && e > 0)
    {
      for (;i!=(e+1);i++)
      {
	for (cur=search;cur!=NULL;cur=cur->next)
        {
	  if (cur->num == i)
	    break;
	}
        if (cur == NULL) /* no matching item */
        {
          wp(win, "%s* Error: no search result numbered %d\n", RED, i, WHITE);
          drw(win);
          /* maybe print "Invalid search item number" ? */
          break; /* i.e. skip rest of range */
        }
#ifndef QUEUE

	if (ffile(down, cur->cmp, cur->song))
        {
	  wp(win, "%s* Error: already getting \"%s\" from %s%s\n", RED, cur->song, cur->cmp, WHITE);
	  drw(win);
	}
	else
	  sendpack(s, F_DGET, "%s \"%s\"", cur->cmp, cur->fn);
#else /* QUEUE */
        /* We want to enforce a policy of a single download per nick.
         * If there already is a download in progress from the same nick, we
         * want to enqueue the file. If not, we want to immediately download
         * the file.
         *   It would seem natural to use the download list to determine
         * whether there is a download in progress from a given nick.
         * But there is a bit of trickiness because:
         *   a) this function processes several download requests at a time
         *   b) the download list isn't updated until sget() is called (when 
         *      a download ack is received from the server)
         * Say we want to get two files from the nick X with the command
         * "/g 1-2". We check the download list and see that there are no
         * downloads from X in progress. So we issue a request for file 1.
         * Now we consider file 2. If we just look at the download list,
         * it still tells us that there are no downloads from X in progress,
         * so we would mistakenly issue a request for file 2 when what
         * we really want to do is enqueue it.
         *    So we can't use the download list alone to determine whether
         * to issue a download request or not. The approach used is to
         * always enqueue the file, and to check BOTH the download list and
         * the download queue for downloads in progress or pending from a 
         * given nick. We only issue a download request if the nick doesn't
         * appear in either the download list or the download queue. Items are 
         * removed from the download queue in sget() or snget(). */
        ret = enqdown(cur);
        if (ret == -1)
        {
          wp(win, "%s* Already getting \"%s\" from %s%s\n", \
              RED, cur->song, cur->cmp, WHITE);
          drw(win);
        }
        else if (ret == -2)
        {
          wp(win, "%s* \"%s\" from %s is already enqueued%s\n", \
              RED, cur->song, cur->cmp, WHITE);
          drw(win);
        }
        else if (ret == 0)
        {
          /* There is a download pending or in progress from this nick */
          wp(win, "%s* Enqueuing \"%s\" from %s%s\n", \
              RED, cur->song, cur->cmp, WHITE);
          drw(win);
        }
        else /* if (ret == 1) */
        {
          /* There is no download pending or in progress from this nick so
           * we send a download request for the file. (The file has also been
           * enqueued.) */
          sendpack(s, F_DGET, "%s \"%s\"", cur->cmp, cur->fn);
          wp(win, "%s* Requesting \"%s\" from %s%s\n", \
              RED, cur->song, cur->cmp, WHITE);
          drw(win);
        } /* end if ret */
#endif /* QUEUE */
      } /* end for loop through range */
    } /* end if i,e > 0 */
  } /* end for t */
    
  drw(win);

  return(1);
}

#ifdef QUEUE
/* "/forceq nick" For use when items from a certain nick are stuck in
 * the download queue. If there is no download in progress from the
 * nick, this sends a download request for the first item in the download
 * queue from the nick. */
O_NAP_FUNC(dforceq)
{
  file_t *fcur;
  char *nick;
  int nickindownlist;
  
  if (num < 2)
  {
    wp(win, "%s* Not enough parameters%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  nick = tok[1];
  /* check if there is a download in progress from nick */
  nickindownlist = 0;
  for (fcur=down; fcur!=NULL; fcur=fcur->next)
  {
    if (!strcasecmp(fcur->nm, nick))
    {
      nickindownlist = 1;
      break;
    }
  }
  if (nickindownlist)
  {
    wp(win, "%s* There is already a download in progress from %s%s\n", \
        RED, nick, WHITE);
    drw(win);
    return(1);
  }
  /* send a request for the first item in the downq from nick */
  if (!downqhook(nick))
  {
    wp(win, "%s* There are no files from %s in the download queue%s\n", \
        RED, nick, WHITE);
    drw(win);
  }
  else
  {
    wp(win, "%s* Sent a request for first item from %s in download queue%s\n",\
        RED, nick, WHITE);
    drw(win);
  }

  return(1);
}
#endif /* QUEUE */

O_NAP_FUNC(dpup)
{
  int c, p;
  file_t *cur;
  
  for (c=1,cur=up;cur!=NULL;cur=cur->next,c++)
  {
    if (!((int)(cur->sz-cur->bsz)))
      p = 0;
    else
    {
      /* p = (int)((float)(cur->csz-cur->bsz)/(float)(cur->sz-cur->bsz)*100);*/
      /* See comments in dpdown() */
      p = (int)((float)(cur->bsz + cur->csz)/(float)(cur->bsz + cur->sz)*100);
    }
    if (p < 0 || p > 100)
      p = 0;
    if (!cur->st)
      wp(win, "%s%i.%s Waiting | %s | %s\n", BOLD, c, WHITE, cur->nm, cur->fn);
    else
      wp(win, "%s%i.%s Sending | %s | %s at %.2f k/s (%i\%%)\n", BOLD, c, WHITE, cur->nm, cur->fn, fabs((float)(cur->csz/1024)/((float)(time(NULL)-cur->st))), abs(p));
  }
  wp(win, "%s%i%s spots open\n", BOLD, info.maxuploads-c+1, WHITE);
  
  drw(win);
  
  return(1);
}

/* print download list and download queue, numbering the items in the order
 * they appear in the lists. */
O_NAP_FUNC(dpdown)
{
  int c, p;
  float rate;
  file_t *cur;
#ifdef QUEUE
  ssearch_t *qel;
  int min, sec, numdl, numqd;
  float mb;
#endif /* QUEUE */
  char *fw;
  
  for (c=1,cur=down;cur!=NULL;cur=cur->next,c++)
  {
    if (!((int)(cur->sz-cur->bsz)))
      p = 0;
    else
    {
      /* p = (int)((float)(cur->csz-cur->bsz)/(float)(cur->sz-cur->bsz)*100);*/
      /* the above formula only produces a meaningful result if bsz is 0.
       *    sz is the number of bytes to transfer (which may be less than
       *        the total final file size if we are resuming a download)
       *    csz is the number of bytes transferred so far
       *    bsz is the starting file size (i.e. the size of the file
       *        when we began the download, which may be > 0 if we are
       *        resuming a download) 
       * If we want to know how much of the transfer has completed, the
       * formula should either be csz/sz or (bsz+csz)/(bsz+sz) depending
       * on whether we want a percentage of sz or a percentage of the total
       * final file size. */
      p = (int)((float)(cur->bsz + cur->csz)/(float)(cur->bsz + cur->sz)*100);
    }
    if (p < 0 || p > 100)
      p = 0;
    fw = (cur->port) ? "" : " (firewalled)";
    if (!cur->st)
      wp(win, "%s%i.%s Waiting | %s%s | %s%s\n", BOLD, c, WHITE, \
          cur->nm, fw, BOLD, cur->fn);
    else
    {
      rate = fabs((float)(cur->csz/1024)/((float)(time(NULL)-cur->st)));
      wp(win, "%s%i.%s Getting | %s%s | %s%s%s at %.2f k/s (%i\%%)\n", BOLD, \
          c, WHITE, cur->nm, fw, BOLD, cur->fn, WHITE, rate, abs(p));
    }
  }
  wp(win, "%s%i%s downloads\n", BOLD, c-1, WHITE);

#ifdef QUEUE
  numdl = c-1;

  /* print download queue. Continue numbering from c. */
  wp(win, "Queued:\n");
  for (qel=downq; qel!=NULL; qel=qel->next)
  {
    min = qel->time/60;
    sec = qel->time%60;
    mb = ((float)qel->sz)/1048576.0;
    wp(win, "%s%i.%s %s%s%s %i %i %s%i:%02i %.02fmb%s %s %s%s%s\n", BOLD, c, \
        WHITE, BOLD, qel->song, WHITE, qel->brate, qel->freq, BOLD, min, sec, \
        mb, WHITE, qel->cmp, BOLD, conns[qel->conn], WHITE);
    /* wp(win, "%s%i.%s Queued | %s | %s\n", BOLD, c, WHITE, \
         qel->cmp, qel->song); */
    c++;
  }
  numqd = (c-1) - numdl;
  wp(win, "%s%d%s items queued\n", BOLD, numqd, WHITE);
#endif /* QUEUE */

  drw(win);
  
  return(1);
}

O_NAP_FUNC(dsearch)
{
  ssearch_t *cur, *cur1;
  char brate[256], freq[256], speed[256];
  int k, noping = 0, c, max, t=0, l = 0;
  struct sockaddr_in me;
  
  if (num < 2)
  {
    wp(win, "%s* Not enough parameters%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  if (srch)
  {
    wp(win, "%s* Another search is in progress%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  srch = 1;
  
  for (cur=search,cur1=cur;cur1;cur=cur1)
  {
    cur1 = cur->next;
    free(cur->song);
    free(cur->fn);
    free(cur->cmp);
    free(cur);
  }
  search = NULL;
  
  memset(brate, 0, sizeof(brate));
  memset(freq, 0, sizeof(freq));
  memset(speed, 0, sizeof(speed));
  
  opterr = 0;
  max = 250;
  optind = 0;
  
  while ((c = getopt(num, tok, "lfpb:c:r:m:")) != -1)
  {
    switch (c)
    {
      case 'f':
        fl = 1;
        break;
      case 'l':
        l = 1;
        break;
      case 'p':
        noping = 1;
        break;
      case 'b':
        if (*optarg == '=')
          strcat(brate, "BITRATE \"EQUAL TO\" \"");
        else if (*optarg == '<')
          strcat(brate, "BITRATE \"AT BEST\" \"");
        else if (*optarg == '>')
          strcat(brate, "BITRATE \"AT LEAST\" \"");
        else
        {
          strcat(brate, "BITRATE \"AT LEAST\" \"");
          t = 1;
        }
        
        if (t)
          strcat(brate, optarg);
        else
          strcat(brate, optarg+1);
        strcat(brate, "\"");
        t = 0;
        break;
      case 'c':
        if (*optarg == '=')
          strcat(speed, "LINESPEED \"EQUAL TO\" ");
        else if (*optarg == '<')
          strcat(speed, "LINESPEED \"AT BEST\" ");
        else if (*optarg == '>')
          strcat(speed, "LINESPEED \"AT LEAST\" ");
        else
        {
          strcat(speed, "LINESPEED \"AT LEAST\" ");
          t = 1;
        }
        
        if (t)
          strcat(speed, optarg);
        else
          strcat(speed, optarg+1);
        t = 0;
        break;
      case 'r':
        if (*optarg == '=')
          strcat(freq, "FREQ \"EQUAL TO\" \"");
        else if (*optarg == '<')
          strcat(freq, "FREQ \"AT BEST\" \"");
        else if (*optarg == '>')
          strcat(freq, "FREQ \"AT LEAST\" \"");
        else
        {
          strcat(freq, "FREQ \"AT LEAST\" \"");
          t = 1;
        }
        
        if (t)
          strcat(freq, optarg);
        else
          strcat(freq, optarg+1);
        strcat(freq, "\"");
        t = 0;
        break;
      case 'm':
        max = atoi(optarg);
        break;
      default:
        break;
    }
  }
  
  sendpack(s, F_SEARCH, "FILENAME CONTAINS \"%s\" MAX_RESULTS %i %s %s %s %s", cstr(str, optind), max, l?"LOCAL_ONLY":"", (*speed)?speed:"", (*brate)?brate:"", (*freq)?freq:"");
  
  wp(win, "* Searching for \"%s\"...\n", cstr(str, optind));
  drw(win);
  
  if (noping)
    return(1);
  
  /* this fails (returns -1) with errno=1 (EPERM) if you don't have root 
   * privileges, so the "icmp" sock_t never gets created and the ping times 
   * are not calculated in ssend() */ 
  k = socket(AF_INET, SOCK_RAW, IPPROTO_ICMP);
  if (k == -1)
    return(1);
  
  me.sin_addr.s_addr = INADDR_ANY;
  me.sin_family = AF_INET;
  
  if (bind(k, (struct sockaddr *)&me, sizeof(me)) == -1)
  {
    close(k);
    return(1);
  }
  
  addsock(k, "icmp", S_R, icmpin);
  
  return(1);
}

O_NAP_FUNC(dpvars)
{
  printsets(win);
  return(1);
}

O_NAP_FUNC(dkill)
{
  if (num < 2)
  {
    wp(win, "%s* Not enough parameters%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  if (num == 2)
    sendpack(s, F_KILL, "%s", tok[1]);
  else
    sendpack(s, F_KILL, "%s \"%s\"", tok[1], cstr(str, 2));
  
  return(1);
}

O_NAP_FUNC(dban)
{
  if (num < 2)
  {
    wp(win, "%s* Not enough parameters%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  if (num == 2)
    sendpack(s, F_BAN, "%s", tok[1]);
  else
    sendpack(s, F_BAN, "%s \"%s\"", tok[1], cstr(str, 2));
  
  return(1);
}

O_NAP_FUNC(dunban)
{
  if (num < 2)
  {
    wp(win, "%s* Not enough parameters%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  if (num == 2)
    sendpack(s, F_UNBAN, "%s", tok[1]);
  else
    sendpack(s, F_UNBAN, "%s \"%s\"", tok[1], cstr(str, 2));
  
  return(1);
}

O_NAP_FUNC(ddatap)
{
  if (num < 3)
  {
    wp(win, "%s* Not enough parameters%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  sendpack(s, F_DATAP, "%s %i", tok[1], atoi(tok[2]));
  
  return(1);
}

O_NAP_FUNC(dtopic)
{
  chans_t *cur;

  if (num == 1)
  {
    if (!curchan)
    {
      wp(win, "%s* Error: You're not on channel%s\n", RED, WHITE);
      drw(win);
      return(1);
    }
    wp(win, "%s* Topic for %s: %s%s\n", YELLOW, curchan->nm, curchan->topic, WHITE);
    drw(win);
    return(1);
  }
  else if (num == 2)
  {
    cur = findchan(chanl, tok[1]);
    if (!cur)
    {
      wp(win, "%s* Error: Not on channel %s%s\n", RED, tok[1], WHITE);
      drw(win);
      return(1);
    }
    wp(win, "%s* Topic for %s: %s%s\n", YELLOW, cur->nm, cur->topic, WHITE);
    drw(win);
    return(1);
  }
      
  sendpack(s, F_TOPIC, "%s", fxp(cstr(str, 1)));
  
  return(1);
}

O_NAP_FUNC(dlevel)
{
  if (num < 3)
  {
    wp(win, "%s* Not enough parameters%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  sendpack(s, F_LEVEL, "%s", cstr(str, 1));
  
  return(1);
}

O_NAP_FUNC(dopsay)
{
  if (num < 2)
  {
    wp(win, "%s* Not enough parameters%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  sendpack(s, F_SOP, "%s", fxp(cstr(str, 1)));
  
  return(1);
}

O_NAP_FUNC(ddns)
{
  struct hostent *d1, *d2;
  struct sockaddr_in in;
  
  if (num < 2)
  {
    wp(win, "%s* Not enough parameters%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  wp(win, "* Resolving %s...\n", tok[1]);
  drw(win);

  if (!strchr(tok[1], '.'))
  {
    dwi = strdup(tok[1]);
    sendpack(s, F_WHOIS, "%s", dwi);
    return(1);
  }
  
  if (fork())
    return(1);
  
  d1 = gethostbyname(tok[1]);
  if (!d1)
  {
    ssock(ipcs[1], "* Unable to resolve %s\n", tok[1]);
    exit(1);
  }
  
  memcpy(&in.sin_addr, d1->h_addr, d1->h_length);
  d2 = gethostbyaddr((char *)&in.sin_addr, d1->h_length, AF_INET);
  if (!d2)
  {
    ssock(ipcs[1], "* %s is %s\n", tok[1], inet_ntoa(in.sin_addr));
    exit(1);
  }
  
  ssock(ipcs[1], "* %s is %s [%s]\n", tok[1], d2->h_name, inet_ntoa(in.sin_addr));
  
  exit(1);
}

O_NAP_FUNC(dannounce)
{
  if (num < 2)
  {
    wp(win, "%s* Not enough parameters%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  sendpack(s, F_ANNOUNCE, "%s", fxp(cstr(str, 1)));
  
  return(1);
}

O_NAP_FUNC(pchans)
{
  chans_t *cur;
  
  if (!chanl)
    wp(win, "No channels\n");
  else if (curchan)
    wp(win, "Cur: %s\n", curchan->nm);
  else
    wp(win, "Cur: Null\n");
  for (cur=chanl;cur!=NULL;cur=cur->next)
    wp(win, "%s\n", cur->nm);
  
  drw(win);
  
  return(1);
}

O_NAP_FUNC(dbanlist)
{
  sendpack(s, F_DBANLIST, "%s", cstr(str, 1));
  
  return(1);
}

O_NAP_FUNC(dlspeed)
{
  if (num < 3)
  {
    wp(win, "%s* Not enough parameters%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  sendpack(s, F_LSCHANGE, "%s %s", tok[1], tok[2]);
  
  return(1);
}

O_NAP_FUNC(dpsocks)
{
  psocks(win);
  
  return(1);
}

/* delete an entry from the upload list */
O_NAP_FUNC(ddup)
{
  int c;
  file_t *cur;
  
  if (num < 2)
  {
    wp(win, "%s* Not enough parameters%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  /* "/dup nick filename". This syntax is used by the upload timer. 
   * This always deletes the _first_ item in the upload list that matches.
   * This can cause a problem.  If (for example) there are two uploads of
   * the same file to the same user and the second one times out (while in
   * the "Waiting" state), /dup will be run with the nick and filename
   * as args, and the wrong upload (the first one) will be deleted, usually
   * causing a seg fault later on. 
   * We try to prevent duplicates in the first place by not allowing
   * simultaneous uploads of the same file to the same user (see sfreq()). */
  if (!isdigit(*tok[1]) && num > 2)
  {
    cur = ffile(up, tok[1], tok[2]);
    if (cur)
    {
      wp(win, "* Deleted upload of \"%s\" to %s\n", cur->fn, cur->nm);
      drw(win);
      if (cur->sk)
	delsock(cur->sk->s);
      dupload(cur);
    }
    return(1);
  }
  
  /* "/dup upnum" */
  if (atoi(tok[1]) <= 0)
  {
    wp(win, "%s* Invalid upload number%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  for (c=1,cur=up;c!=atoi(tok[1]);cur=cur->next,c++)
  {
    if (cur == NULL)
    {
      wp(win, "%s* Invalid upload number%s\n", RED, WHITE);
      drw(win);
      return(1);
    }
  }
  
  if (cur == NULL)
  {
    wp(win, "%s* Invalid upload number%s\n", RED, WHITE);
    drw(win);
    return(1);
  }

  /* why is this not allowed? -PS */
  if (cur->csz)
  {
    wp(win, "%s* Cannot delete an uploading file%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  wp(win, "* Deleted upload of \"%s\" to %s\n", cur->fn, cur->nm);
  drw(win);

  /* bug fixed: dupload, which free()s cur, used to appear before cur->sk */
  
  if (cur->sk)
    delsock(cur->sk->s);
  
  dupload(cur);
  
  return(1);
}

/* delete an item from the download list or the download queue. 
 * Items are (implicitly) numbered according to their position in the list.
 * Use /pdown to view the numbering. */
O_NAP_FUNC(ddown)
{
  int c;
  file_t *cur;
  int dnum; /* can't use the name "num": it is an arg to this function! */
#ifdef QUEUE
  ssearch_t *qel;
#endif /* QUEUE */
  
  if (num < 2)
  {
    wp(win, "%s* Not enough parameters%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  /* when invoked as "ddown nick filename". This syntax is used by the
   * download timers. The problem is that if there are two or more items
   * in the download list with the same nick & filename, then this always
   * deletes the first matching item in the list (which may not correspond
   * to the item whose timer expired). We avoid this problem by not allowing
   * simultaneous downloads of the same file from the same user (in fact,
   * with download queueing, we only one download from each user). */
  /* (we don't support removing items from the queue with this syntax) */
  if (!isdigit(*tok[1]) && num > 2)
  {
    cur = ffile(down, tok[1], tok[2]);
    if (cur)
    {
      wp(win, "* Deleted download of \"%s\" from %s\n", cur->fn, cur->nm);
      drw(win);
      if (cur->sk)
	delsock(cur->sk->s);
      ddownload(cur);
    }
    return(1);
  }
  
  /* when invoked as "ddown dnum" */
  dnum = atoi(tok[1]);
  if (dnum <= 0)
  {
    wp(win, "%s* Invalid download number%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  /* find the dnum-th element in the download list */
  for (c=1,cur=down; cur; cur=cur->next,c++)
  {
    if (c == dnum)
      break;
  }
  
  if (cur)
  {
    wp(win, "* Deleted download of \"%s\" from %s\n", cur->fn, cur->nm);
    drw(win);
    if (cur->sk)
      delsock(cur->sk->s);
    ddownload(cur);
    return(1);
  }

#ifdef QUEUE
  /* dnum is greater than the number of items in the download list;
   * maybe it corresponds to an item in the download queue. (At this point,
   * c-1 represents the number of items in download list.) */
  for (qel=downq; qel != NULL; qel=qel->next, c++)
  {
    if (c == dnum)
      break;
  }

  if (qel)
  {
    wp(win, "* Dequeued download of \"%s\" from %s\n", qel->song, qel->cmp);
    drw(win);
    /* have to print the message first, because qel won't point to anything
     * after the call the deqdown() ! */
    deqdown(qel);
    return(1);
  }
#endif /* QUEUE */

  wp(win, "%s* Invalid download number%s\n", RED, WHITE);
  drw(win);

  return(1);
}

O_NAP_FUNC(dmuzzle)
{
  if (num < 2)
  {
    wp(win, "%s* Not enough parameters%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  if (num == 2)
    sendpack(s, F_MUZZLE, "%s", tok[1]);
  else
    sendpack(s, F_MUZZLE, "%s \"%s\"", tok[1], cstr(str, 2));
  
  return(1);
}

O_NAP_FUNC(dunmuzzle)
{
  if (num < 2)
  {
    wp(win, "%s* Not enough parameters%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  if (num == 2)
    sendpack(s, F_UNMUZZLE, "%s", tok[1]);
  else
    sendpack(s, F_UNMUZZLE, "%s \"%s\"", tok[1], cstr(str, 2));
  
  return(1);
}

O_NAP_FUNC(ddisconnect)
{
  sock_t *sk;

  curchan=NULL;
  
  wp(win, "%s* Disconnected from server%s\n", RED, WHITE);
  drw(win);

  if (chanl)
    delnapchans();
  
  upchan(chanl);
  
  sk = findsock("server");
  if (sk)
    delsock(sk->s);
  
  return(1);
}

O_NAP_FUNC(dreload)
{
  libunload(&hnd);
  if (libload(USERLIB) == -1)
  {
    wp(win, "%s* Error loading %s%s\n", RED, dlerror(), WHITE);
    drw(win);
  }
  
  return(1);
}

O_NAP_FUNC(dreconnect)
{
  int t;
  sock_t *sk;
  chans_t *cur;
  char *t1 = NULL;
  char *srv;
  char *serverlist_dup;

  wp(win, "Getting best host...\n");
  drw(win);

  serverlist_dup = strdup(info.serverlist);
  srv = strtok(serverlist_dup, ";");

  for (;  ; srv = strtok(NULL, ";")) {
    if (srv==NULL && info.reconnect) {  /* loop endlessly */
      sleep(1);
      free(serverlist_dup);
      serverlist_dup = strdup(info.serverlist);
      srv = strtok(serverlist_dup, ";");
    } else if (srv==NULL) {
      free(serverlist_dup);
      return(0);
    }
    wp(win, "Trying %s\n", srv);
    t = conn(srv, PORT);
    if (t == -1)
      {
	drw(win);
	continue;
      }
    
    sk = findsock("server");
    if (sk)
      {
	close(sk->s);
	delsock(sk->s);
      }
    wp(win, "Logging in...\n");
    drw(win);
    if (login(t, info.user, info.pass, info.port, CLIENT, info.conn, info.email) == -1)
      {
	drw(win);
	close(t);
	continue;
      }
    break;
  }
  free(serverlist_dup);

  {
    struct sockaddr_in dst;
    int frmlen = sizeof(dst);
    if (!getpeername(t, (struct sockaddr *)&dst, &frmlen))
      wp(win, "Connected to %s:%i\n", inet_ntoa(dst.sin_addr), ntohs(dst.sin_port));
  }

  addsock(t, "server", S_R, inserv);
  sk = findsock("server");
  
  t1 = glistn(info.user);
  checkhotlist(t, t1);
  free(t1);
  
  curchan = NULL;
  
  if (chanl)
  {
    for (cur=chanl;cur;cur=cur->next)
      if (!cur->q)
        sendpack(sk->s, F_JOIN, "%s", cur->nm);
    delnapchans();
  }
  
  upchan(chanl);
  
  return(1);
}

O_NAP_FUNC(dbrowse)
{
  ssearch_t *cur, *cur1;
  
  if (num < 2)
  {
    wp(win, "%s* Not enough parameters%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  if (srch)
  {
    wp(win, "%s* Another search is in progress%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  for (cur=search,cur1=cur;cur1!=NULL;cur=cur1)
  {
    cur1 = cur->next;
    free(cur->song);
    free(cur->fn);
    free(cur->cmp);
    free(cur);
  }
  search = NULL;
  
  sendpack(s, F_BROWSE, "%s", tok[1]);
  
  wp(win, "* Browsing %s...\n", tok[1]);
  drw(win);
  
  srch = 1;
  
  return(1);
}

O_NAP_FUNC(dhelp)
{
  int i, j, l;
  char buf[64];
  
  if (num < 2)
  {
    wp(win, "%s* Commands:%s\n", MAGENTA, WHITE);
    for (i=0,j=0;out[i].nm;i++,j++)
    {
      memset(buf, 0, sizeof(buf));
      if (j != 3)
      {
        for (l=COLS/4-strlen(out[i].nm);l>=0;l--)
          buf[l] = ' ';
      }
      wp(win, "%s%s%s%s", MAGENTA, out[i].nm, WHITE, buf);
      if (j == 3)
      {
        wp(win, "\n");
        j = -1;
      }
    }
    if (j)
      wp(win, "\n");
    drw(win);
    
    return(1);
  }
      
      
  for (i=0;;i++)
  {
    if (!out[i].nm)
    {
      wp(win, "%s* Could not find help on command: %s%s\n", RED, tok[1], WHITE);
      drw(win);
      return(1);
    }
    if (!strcasecmp(out[i].nm, tok[1]))
    {
      if (!out[i].help)
      {
        wp(win, "%s* No help available on %s%s\n", MAGENTA, out[i].nm, WHITE);
        drw(win);
        return(1);
      }
      wp(win, "%s* Help on %s:%s\n", MAGENTA, out[i].nm, WHITE);
      wp(win, "%s/%s %s%s\n", MAGENTA, out[i].nm, out[i].help, WHITE);
      drw(win);
      return(1);
    }
  }
}

O_NAP_FUNC(dping)
{
  sendpack(s, CLIENT_PING, "%s", cstr(str, 1));
  
  return(1);
};

O_NAP_FUNC(duserpass)
{
  sendpack(s, SET_USER_PASSWORD, "%s", cstr(str, 1));
  
  return(1);
}

O_NAP_FUNC(dreloadconf)
{
  sendpack(s, SERVER_RELOAD_CONFIG, "%s", cstr(str, 1));

  return(1);
}

O_NAP_FUNC(dsver)
{
  sendpack(s, SERVER_VERSION, "%s", cstr(str, 1));
  
  return(1);
}

O_NAP_FUNC(dchanclear)
{
  sendpack(s, CHANNEL_CLEAR, "%s", cstr(str, 1));
  
  return(1);
}

O_NAP_FUNC(dsetlevel)
{
  sendpack(s, CHANNEL_SETLEVEL, "%s", cstr(str, 1));
  
  return(1);
}

O_NAP_FUNC(dme)
{
  if (!curchan)
  {
    wp(win, "%sYou're not on a channel%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  sendpack(s, CHAN_EMOTE, "%s \"%s\"", curchan->nm, cstr(str, 1));
  
  return(1);
}

O_NAP_FUNC(dusers)
{
  if (num < 2)
  {
    if (!curchan)
    {
      wp(win, "%s* You're not on a channel%s\n", RED, WHITE);
      drw(win);
      return(1);
    }
    sendpack(s, CHANNEL_USERLIST, "%s", curchan->nm);
  }
  else
    sendpack(s, CHANNEL_USERLIST, "%s", cstr(str, 1));
  
  return(1);
}

O_NAP_FUNC(dgusers)
{
  sendpack(s, SERVER_USERLIST, "%s", cstr(str, 1));
  
  return(1);
}

O_NAP_FUNC(dsetconf)
{
  sendpack(s, SERVER_SET_CONFIG, "%s", cstr(str, 1));
  
  return(1);
}

O_NAP_FUNC(dclist)
{
  sendpack(s, F_CLIST, NULL);
  
  wp(win, "Channel | Users | Topic\n");
  wp(win, "-----------------------\n");
  
  return(1);
}

O_NAP_FUNC(dclist2)
{
  sendpack(s, CHANNEL_LIST2, NULL);
  
  wp(win, "Channel | Users | Topic\n");
  wp(win, "-----------------------\n");
  
  return(1);
}

O_NAP_FUNC(dclear)
{
  scrls_t *cur, *cur1;
  
  for (cur=mscroll,cur1=cur;;cur=cur1)
  {
    if (!cur)
      break;
    if (!cur->d)
    {
      cur1 = cur->next;
      continue;
    }
    if (wmode && cur->chan && cur->chan != curchan)
    {
      cur1 = cur->next;
      continue;
    }
    if (cur->prev)
      cur->prev->next = cur->next;
    cur1 = cur->next;
    if (cur1)
      cur1->prev = cur->prev;
    if (cur == mscroll)
      mscroll = cur1;
    free(cur);
  }
  
  scur = NULL;
  dscr(win);
  drw(win);
  
  return(1);
}

O_NAP_FUNC(dnews)
{
  checknv(SHOST, SPORT, "MSG");
  
  return(1);
}

O_NAP_FUNC(dsraw)
{
  if (num < 2)
    return(1);
  
  sendpack(s, atoi(tok[1]), "%s", cstr(str, 2));
  
  return(1);
}

O_NAP_FUNC(dquery)
{
  chans_t *cur, *cur1=NULL;
  
  if (num < 2)
  {
    wp(win, "%s* No username given%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  if ((cur = findchan(chanl, tok[1])))
  {
    curchan = cur;
    return(1);
  }

  if (!chanl)
  {
    chanl = (chans_t *)malloc(sizeof(chans_t));
    cur = chanl;
  }
  else
  {
    for (cur=chanl;cur!=NULL;cur=cur->next)
      cur1 = cur;
    cur = (chans_t *)malloc(sizeof(chans_t));
    cur1->next = cur;
  }
  
  cur->nm = strdup(tok[1]);
  cur->q = 1;
  cur->topic = NULL;
  cur->p = 0;
  cur->key = NULL;
  cur->next = NULL;
  cur->users = NULL;
  curchan = cur;
  
  recent = cur;
  wp(win, "%s* Querying %s%s\n", GREEN, cur->nm, WHITE);
  drw(win);
  recent = NULL;
  
  return(1);
}

O_NAP_FUNC(dcloak)
{
  sendpack(s, F_CLOAK, NULL);
  
  return(1);
}

O_NAP_FUNC(dblocklist)
{
  sendpack(s, F_BLOCKLIST, NULL);
  
  return(1);
}

O_NAP_FUNC(dblock)
{
  if (num < 2)
  {
    wp(win, "%s* Not enough parameters%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  if (num == 2)
    sendpack(s, F_BLOCK, "%s", tok[1]);
  else
    sendpack(s, F_BLOCK, "%s \"%s\"", tok[1], cstr(str, 2));
  
  return(1);
}

O_NAP_FUNC(dunblock)
{
  if (num < 2)
  {
    wp(win, "%s* Not enough parameters%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  if (num == 2)
    sendpack(s, F_UNBLOCK, "%s", tok[1]);
  else
    sendpack(s, F_UNBLOCK, "%s \"%s\"", tok[1], cstr(str, 2));
  
  return(1);
}

O_NAP_FUNC(dwindow)
{
  if (!wmode)
  {
    wp(win, "%s* Window mode enabled%s\n", GREEN, WHITE);
    wmode = 1;
  }
  else
  {
    wp(win, "%s* Window mode disabled%s\n", GREEN, WHITE);
    wmode = 0;
  }
  
  dscr(win);
  drw(win);
  
  return(1);
}

O_NAP_FUNC(dignore)
{
  sendpack(s, IGNORE_ADD, "%s", tok[1]);
  
  return(1);
}

O_NAP_FUNC(dunignore)
{
  sendpack(s, IGNORE_REMOVE, "%s", tok[1]);
  
  return(1);
}

O_NAP_FUNC(dignoreclear)
{
  sendpack(s, IGNORE_CLEAR, NULL);
  
  return(1);
}

O_NAP_FUNC(dignorelist)
{
  sendpack(s, IGNORE_LIST, NULL);
  
  return(1);
}

O_NAP_FUNC(dalias)
{
  alias_t *cur, *cur1=NULL;
  
  if (num < 3)
  {
    wp(win, "%s* Error: insufficient arguments supplied%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  if (!alhead)
  {
    cur = (alias_t *)malloc(sizeof(alias_t));
    alhead = cur;
  }
  else
  {
    for (cur=alhead;;cur=cur->next)
    {
      if (!cur)
        break;
      if (!strcasecmp(cur->nm, tok[1]))
      {
        wp(win, "%s* Error: an alias by that name already exists%s\n", RED, WHITE);
        drw(win);
        return(1);
      }
      cur1 = cur;
    }
    cur = (alias_t *)malloc(sizeof(alias_t));
    cur1->next = cur;
  }
  
  cur->nm = strdup(tok[1]);
  cur->args = strdup(cstr(str, 2));
  cur->next = NULL;
  
  wp(win, "%s* Added alias \"%s\"%s\n", BRIGHT(BLUE), cur->nm, WHITE);
  drw(win);
  
  return(1);
}

O_NAP_FUNC(dunalias)
{
  alias_t *cur, *cur1=NULL;
  
  if (num < 2)
  {
    wp(win, "%s* Error: You didn't specify an alias name%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  for (cur=alhead;;cur=cur->next)
  {
    if (!cur)
    {
      wp(win, "%s* Error: alias could not be found%s\n", RED, WHITE);
      drw(win);
      return(1);
    }
    if (!strcasecmp(cur->nm, tok[1]))
      break;
    cur1 = cur;
  }
  
  if (cur1)
    cur1->next = cur->next;
  else if (cur->next)
    alhead = cur->next;
  else
    alhead = NULL;
  free(cur->nm);
  free(cur->args);
  free(cur);
  
  wp(win, "%s* Removed alias \"%s\"%s\n", BRIGHT(BLUE), tok[1], WHITE);
  drw(win);
  
  return(1);
}

O_NAP_FUNC(dhandler)
{
  handler_t *cur, *cur1=NULL;
  
  if (num < 3)
  {
    wp(win, "%s* Error: insufficient arguments supplied%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  if (!hndhead)
  {
    cur = (handler_t *)malloc(sizeof(handler_t));
    hndhead = cur;
  }
  else
  {
    for (cur=hndhead;;cur=cur->next)
    {
      if (!cur)
        break;
      if (cur->op == atoi(tok[1]))
      {
        wp(win, "%s* Error: a handler by that name already exists%s\n", RED, WHITE);
        drw(win);
        return(1);
      }
      cur1 = cur;
    }
    cur = (handler_t *)malloc(sizeof(handler_t));
    cur1->next = cur;
  }
  
  cur->op = atoi(tok[1]);
  cur->args = strdup(cspstr(tok, num, 2));
  cur->next = NULL;
  
  wp(win, "%s* Added handler for code (%i)%s\n", BRIGHT(BLUE), cur->op, WHITE);
  drw(win);
  
  return(1);
}

O_NAP_FUNC(dunhandler)
{
  handler_t *cur, *cur1=NULL;
  
  if (num < 2)
  {
    wp(win, "%s* Error: You didn't specify a handler code%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  for (cur=hndhead;;cur=cur->next)
  {
    if (!cur)
    {
      wp(win, "%s* Error: handler could not be found%s\n", RED, WHITE);
      drw(win);
      return(1);
    }
    if (cur->op == atoi(tok[1]))
      break;
    cur1 = cur;
  }
  
  if (cur1)
    cur1->next = cur->next;
  else if (cur->next)
    hndhead = cur->next;
  else
    hndhead = NULL;
  free(cur->args);
  free(cur);
  
  wp(win, "%s* Removed handler for code (%i)%s\n", BRIGHT(BLUE), atoi(tok[1]), WHITE);
  drw(win);
  
  return(1);
}

O_NAP_FUNC(daliaslist)
{
  alias_t *cur;
  
  if (!alhead)
  {
    wp(win, "%s* No aliases%s\n", BRIGHT(BLUE), WHITE);
    drw(win);
  }
  
  for (cur=alhead;cur;cur=cur->next)
  {
    if (*cur->args == '{' && cur->args[1] == '\n')
      wp(win, "%s* Alias: %s ->%s\n%s\n", BRIGHT(BLUE), cur->nm, WHITE, cur->args);
    else
      wp(win, "%s* Alias: %s ->%s %s\n", BRIGHT(BLUE), cur->nm, WHITE, cur->args);
    drw(win);
  }
  
  return(1);
}

O_NAP_FUNC(dhandlerlist)
{
  handler_t *cur;
  
  if (!hndhead)
  {
    wp(win, "%s* No handlers%s\n", BRIGHT(BLUE), WHITE);
    drw(win);
  }
  
  for (cur=hndhead;cur;cur=cur->next)
  {
    if (*cur->args == '{' && cur->args[1] == '\n')
      wp(win, "%s* Handler: (%i) ->%s\n%s\n", BRIGHT(BLUE), cur->op, WHITE, cur->args);
    else
      wp(win, "%s* Handler: (%i) ->%s %s\n", BRIGHT(BLUE), cur->op, WHITE, cur->args);
    drw(win);
  }
  
  return(1);
}

O_NAP_FUNC(dkick)
{
  if (num < 2)
  {
    wp(win, "%s* Not enough parameters%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  if (!curchan)
  {
    wp(win, "%s* Not on a channel%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  if (num == 2)
    sendpack(s, CHANNEL_KICK, "%s %s", curchan->nm, tok[1]);
  else
    sendpack(s, CHANNEL_KICK, "%s %s \"%s\"", curchan->nm, tok[1], cstr(str, 2));

  return(1);
}

O_NAP_FUNC(dcbanlist)
{
  if (!curchan)
  {
    wp(win, "%s* Not on a channel%s\n", RED, WHITE);
    drw(win);
    return(1);
  }

  sendpack(s, CHANNEL_BAN_LIST, "%s", curchan->nm);
  
  return(1);
}

O_NAP_FUNC(dcban)
{
  if (num < 2)
  {
    wp(win, "%s* Not enough parameters%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  if (!curchan)
  {
    wp(win, "%s* Not on a channel%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  if (num == 2)
    sendpack(s, CHANNEL_BAN_ADD, "%s %s", curchan->nm, tok[1]);
  else
    sendpack(s, CHANNEL_BAN_ADD, "%s %s \"%s\"", curchan->nm, tok[1], cstr(str, 2));

  return(1);
}

O_NAP_FUNC(dcunban)
{
  if (num < 2)
  {
    wp(win, "%s* Not enough parameters%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  if (!curchan)
  {
    wp(win, "%s* Not on a channel%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  sendpack(s, CHANNEL_BAN_REMOVE, "%s %s", curchan->nm, tok[1]);

  return(1);
}

O_NAP_FUNC(dloadalias)
{
  if (num < 2)
  {
    wp(win, "%s* No filename specified%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  if (loadaliases(tok[1]) == -1)
  {
    wp(win, "%s* Error loading aliases%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  wp(win, "%s* Successfully loaded aliases%s\n", BRIGHT(BLUE), WHITE);
  drw(win);
  
  return(1);
}

O_NAP_FUNC(dsavealias)
{
  if (num < 2)
  {
    wp(win, "%s* No filename specified%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  if (savealiases(tok[1]) == -1)
  {
    wp(win, "%s* Error saving aliases%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  wp(win, "%s* Successfully saved aliases%s\n", BRIGHT(BLUE), WHITE);
  drw(win);
  
  return(1);
}

O_NAP_FUNC(dclearalias)
{
  alias_t *cur, *cur1=NULL;
  
  for (cur=alhead;;cur=cur1)
  {
    if (!cur)
    {
      if (!cur1)
        alhead = NULL;
      break;
    }
    cur1 = cur->next;
    free(cur->nm);
    free(cur->args);
    free(cur);
  }
  
  wp(win, "%s* Cleared all aliases%s\n", BRIGHT(BLUE), WHITE);
  drw(win);
  
  return(1);
}

O_NAP_FUNC(dloadhandler)
{
  if (num < 2)
  {
    wp(win, "%s* No filename specified%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  if (loadhandlers(tok[1]) == -1)
  {
    wp(win, "%s* Error loading handlers%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  wp(win, "%s* Successfully loaded handlers%s\n", BRIGHT(BLUE), WHITE);
  drw(win);
  
  return(1);
}

O_NAP_FUNC(dsavehandler)
{
  if (num < 2)
  {
    wp(win, "%s* No filename specified%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  if (savehandlers(tok[1]) == -1)
  {
    wp(win, "%s* Error saving handlers%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  wp(win, "%s* Successfully saved handlers%s\n", BRIGHT(BLUE), WHITE);
  drw(win);
  
  return(1);
}

O_NAP_FUNC(dclearhandler)
{
  handler_t *cur, *cur1=NULL;
  
  for (cur=hndhead;;cur=cur1)
  {
    if (!cur)
    {
      if (!cur1)
        hndhead = NULL;
      break;
    }
    cur1 = cur->next;
    free(cur->args);
    free(cur);
  }
  
  wp(win, "%s* Cleared all handlers%s\n", BRIGHT(BLUE), WHITE);
  drw(win);
  
  return(1);
}

O_NAP_FUNC(dnotify)
{
  FILE *f;
  char *t = NULL, rd[64];
  hotlist_t *cur;
  
  if (num < 2)
  {
    sendpack(s, ADD_NOTIFY, "%s", cstr(str, 1));
    return(1);
  }
  
  t = glistn(info.user);
  
  f = fopen(t, "r");
  free(t);
  if (!f)
  {
    cur = (hotlist_t *)malloc(sizeof(hotlist_t));
    cur->nm = strdup(cstr(str, 1));
    cur->on = 0;
    cur->conn = 0;
    addlink(&hlist, cur);
    sendpack(s, ADD_NOTIFY, "%s", cstr(str, 1));
    return(1);
  }
  
  while (!feof(f))
  {
    memset(rd, 0, sizeof(rd));
    fgets(rd, sizeof(rd), f);
    if (!strlen(rd))
      break;
    rd[strlen(rd)-1] = 0;
    if (!strcasecmp(rd, tok[1]))
    {
      wp(win, "%s* Error: %s is already on your hotlist%s\n", RED, tok[1], WHITE);
      drw(win);
      fclose(f);
      return(1);
    }
  }
  
  cur = (hotlist_t *)malloc(sizeof(hotlist_t));
  cur->nm = strdup(cstr(str, 1));
  cur->on = 0;
  cur->conn = 0;
  addlink(&hlist, cur);
  sendpack(s, ADD_NOTIFY, "%s", cstr(str, 1));
  
  return(1);
}

O_NAP_FUNC(dunnotify)
{
  FILE *f;
  char *t = NULL, rd[64], buf[4096];
  hotlist_t *cur;
  
  if (num < 2)
  {
    sendpack(s, REMOVE_NOTIFY, "%s", cstr(str, 1));
    return(1);
  }
  
  t = glistn(info.user);
  
  f = fopen(t, "r");
  if (!f)
  {
    wp(win, "%s* Error opening hotlist for reading%s\n", RED, WHITE);
    drw(win);

    return(1);
  }
  
  memset(buf, 0, sizeof(buf));
  while (!feof(f))
  {
    memset(rd, 0, sizeof(rd));
    fgets(rd, sizeof(rd), f);
    if (!strncasecmp(tok[1], rd, strlen(tok[1])))
      continue;
    strcat(buf, rd);
  }
  fclose(f);
  
  f = fopen(t, "w");
  free(t);
  if (!f)
  {
    wp(win, "%s* Error opening hotlist for writing%s\n", RED, WHITE);
    drw(win);

    return(1);
  }
  fputs(buf, f);
  fclose(f);
  
  sendpack(s, REMOVE_NOTIFY, "%s", cstr(str, 1));
  
  wp(win, "%s* Attempting to remove %s from your hotlist%s\n", BRIGHT(BLUE), tok[1], WHITE);
  drw(win); 
  
  cur = findlinks(hlist, cstr(str, 1), REF(hotlist_t *, hlist)->nm);
  dellink(&hlist, cur);
  free(cur);
  
  return(1);
}

O_NAP_FUNC(dhotlist)
{
  int i;
  hotlist_t *cur;
  
  if (!hlist)
  {
    wp(win, "%s* No users on your hotlist%s\n", BRIGHT(BLUE), WHITE);
    drw(win);
    return(1);
  }

  wp(win, "%s* Hotlist:%s\n", BRIGHT(BLUE), WHITE);
  
  for (i=0;;i++)
  {
    cur = linknum(hlist, i);
    if (!cur)
      break;
    wp(win, "%s* %s - %s%s\n", BRIGHT(BLUE), cur->nm, (cur->on)?"On":"Off", WHITE);
  }
  
  drw(win);
  return(1);
}

O_NAP_FUNC(dwstats)
{
  wstats(win);
  
  return(1);
}

O_NAP_FUNC(dsay)
{
  if (!curchan)
  {
    wp(win, "%s* Not on a channel%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  sendpack(s, F_SAY, "%s %s", fxp(curchan->nm), cstr(str, 1));
  
  return(1);
}

O_NAP_FUNC(dhup)
{
  char *t = NULL;
  
  t = glistn(info.user);
  
  checkhotlist(s, t);
  free(t);
  
  return(1);
}

O_NAP_FUNC(dabout)
{
  wp(win, "%s* nap v%s by Ignitor (Kevin Sullivan), modified by Peter Selinger and others%s\n", BRIGHT(RED), VERSION, WHITE);
  wp(win, "%s* Thanks: nocarrier, napster, fizban, amputee, nytr0, fletch, Plugh, pea and the rest of the admins and mods%s\n", BRIGHT(RED), WHITE);
  wp(win, "%s* Thanks: nyt for info on ncurses and drscholl for opennap (useful for testing)%s\n", BRIGHT(RED), WHITE);
  wp(win, "%s* Special Thanks: everyone that has emailed me with bug reports, comments and suggestions%s\n", BRIGHT(RED), WHITE);
  if (num == 5)
    wp(win, "%s* Extra Special Thanks: Brigand for being such a bitch :P%s\n", BRIGHT(RED), WHITE);
  drw(win);
  
  return(1);
}

O_NAP_FUNC(dirc)
{
  struct sockaddr_in dst;
  struct passwd *pw;
  char *t=NULL;
  int i;

  if (num < 2)
  {
    if (!ircsock)
      return(1);
    wp(win, "%s* IRC mode toggled: On%s\n", RED, WHITE);
    drw(win);
    ircmode = 1;
    return(1);
  }

  if (strcasecmp(tok[1], "server"))
  {
    if (!ircsock)
      return(1);
    if (!checkouts(ircsock, cstr(str, 1), tok+1, num-1, win))
      ssock(ircsock, "%s\n", cstr(str, 1));
    return(1);
  }
  
  if (num < 4)
    return(1);
  
  if (ircsock)
  {
    wp(win, "%s* Disconnect from your current IRC server first%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  ircsock = socket(AF_INET, SOCK_STREAM, 0);
  
  dst.sin_addr.s_addr = resolve(tok[2]);
  dst.sin_port = htons(atoi(tok[3]));
  dst.sin_family = AF_INET;
  
  if (connect(ircsock, (struct sockaddr *)&dst, sizeof(dst)) == -1)
  {
    wp(win, "%s* Error connecting socket%s\n", RED, WHITE);
    drw(win);
    close(ircsock);
    ircsock = 0;
    return(1);
  }
  
  addsock(ircsock, "irc", S_R, inirc);
  
  pw = getpwuid(getuid());
  if (pw)
  {
    msprintf(&t, "%s", pw->pw_gecos);
    for (i=0;t[i];i++)
    {
      if (t[i] == ',')
      {
        t[i] = 0;
        break;
      }
    }
  }
  
  ssock(ircsock, "NICK %s\nUSER %s NULL NULL :%s\n", info.user, info.user, (t)?t:info.user);
  if (t)
    free(t);
  mnick = strdup(info.user);
  ircmode = 1;
  
  return(1);
}

O_NAP_FUNC(dkickall)
{
  chans_t *cur;

  if (num < 2)
  {
    wp(win, "%s* Not enough parameters%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  if (!curchan)
  {
    wp(win, "%s* Not on a channel%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  for (cur=chanl;cur;cur=cur->next)
  {
    if (finduser(cur, tok[1]))
    {
      if (num == 2)
        sendpack(s, CHANNEL_KICK, "%s %s", cur->nm, tok[1]);
      else
        sendpack(s, CHANNEL_KICK, "%s %s \"%s\"", cur->nm, tok[1], cstr(str, 2));
    }
  }

  return(1);
}


O_NAP_FUNC(deval)
{
  unsigned char *p;

  if (num < 2)
    return(1);
  
  p = dovars(tok[1]);
  
  wp(win, "%s\n", p);
  drw(win);

  free(p);
  
  return(1);
}

O_NAP_FUNC(dset)
{
  if (num < 2)
  {
    wp(win, "%s* Insufficient parameters%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  if (num > 2)
    chset(tok[1], cstr(str, 2));
  else /* num == 2 */
    chset(tok[1], "");

  if (nvar("noechosets") != 1)
  {
    wp(win, "%s* Set variable %s to \"%s\"%s\n", BRIGHT(BLUE), tok[1], getval(tok[1], NULL), WHITE);
    drw(win);
  }
  
  return(1);
}

O_NAP_FUNC(dunset)
{
  if (num < 2)
  {
    wp(win, "%s* No variable name specified%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  if (!getval(tok[1], NULL))
  {
    if (nvar("noechosets") != 1)
    {
      wp(win, "%s* %s was not set%s\n", RED, tok[1], WHITE);
      drw(win);
    }
    return(1);
  }
  
  delset(tok[1]);
  
  if (nvar("noechosets") != 1)
  {
    wp(win, "%s* Unset variable %s%s\n", BRIGHT(BLUE), tok[1], WHITE);
    drw(win);
  }
  
  return(1);
}

O_NAP_FUNC(dloadconfig)
{
  if (num < 2)
  {
    wp(win, "%s* No filename specified%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  if (loadsets(tok[1]) == -1)
  {
    wp(win, "%s* Error loading config%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  wp(win, "%s* Successfully loaded config%s\n", BRIGHT(BLUE), WHITE);
  drw(win);
  
  return(1);
}

O_NAP_FUNC(dsaveconfig)
{
  if (num < 2)
  {
    wp(win, "%s* No filename specified%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  if (savesets(tok[1]) == -1)
  {
    wp(win, "%s* Error saving config%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  wp(win, "%s* Successfully saved config%s\n", BRIGHT(BLUE), WHITE);
  drw(win);
  
  return(1);
}

O_NAP_FUNC(decho)
{
  if (num < 2)
    return(1);
  
  wp(win, "%s\n", cstr(str, 1));
  drw(win);
  
  return(1);
}

O_NAP_FUNC(dresults)
{
  sscr(search);
  return(-1);
}

O_NAP_FUNC(dexec)
{
  char buf[512];
  FILE *f;
  int fl=0, i;

  if (num < 2)
  {
    wp(win, "%s* No command specified%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  if (tok[1][0] == '-' && tok[1][1] == 'o')
  {
    fl = 1;
    f = popen(cstr(str, 2), "r");
  }
  else
    f = popen(cstr(str, 1), "r");
  
  if (!f)
  {
    wp(win, "%s* Error executing command%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  memset(buf, 0, sizeof(buf));
  fread(&buf, sizeof(buf)-1, 1, f);
  pclose(f);
  for (i=0;buf[i];i++)
    if (buf[i] == '\n' || buf[i] == '\r')
    {
      strncpy(buf+i, buf+i+1, strlen(buf+i+1));
      buf[strlen(buf)-1] = 0;
    } 
  
  if (!fl)
  {
    wp(win, "%s\n", buf);
    drw(win);
    return(1);
  }
  
  if (!curchan)
  {
    wp(win, "%s* Not on a channel%s\n", RED, WHITE);
    drw(win);
  }
  else if (curchan->q == 1)
  {
    sendpack(s, F_TELL, "%s %s", fxp(curchan->nm), buf);
    recent = curchan;
    wp(win, "%s* --> (%s%s%s)%s %s\n", GREEN, WHITE, curchan->nm, GREEN, WHITE, buf);
    drw(win);
    recent = NULL;
  }
  else if (curchan->q == 2 && buf[0] != '\0')
  {
    ssock(ircsock, "PRIVMSG %s :%s\n", fxp(curchan->nm), buf);
    recent = curchan;
    wp(win, "%s<%s%s%s>%s %s\n", BRIGHT(MAGENTA), WHITE, mnick, BRIGHT(MAGENTA), WHITE, buf);
    drw(win);
    recent = NULL;
  }
  else if (buf[0] != '\0')
    sendpack(s, F_SAY, "%s %s", fxp(curchan->nm), buf);
  
  return(1);
}

O_NAP_FUNC(dtimer)
{
  int sec, i;
  unsigned char c=0;

  if (num < 4)
  {
    wp(win, "%s* Insufficient parameters%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  for (i=0;tok[1][i];i++)
    if (!isdigit(tok[1][i]))
      c = 1;
  
  for (i=0;tok[2][i];i++)
    if (!isdigit(tok[2][i]))
      c = 1;
  
  if (c)
  {
    wp(win, "%s* Invalid times to set%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  sec = atoi(tok[2])+atoi(tok[1])*60;
  
  addtimer(sec, s, cstr(str, 3), win);

  return(1);
}

O_NAP_FUNC(dif)
{
  char *t, *b;
  int i, c=0, r, f = 0;
  
  if (num < 2)
    return(1);
  
  t = (char *)malloc(512);
  memset(t, 0, 512);
  strcpy(t, tok[1]);
  
  b = strev(t, '(', ')', cmp);
  if (b)
    r = atoi(b);
  else
    r = -1;
  if (b)
    free(b);
  
  if (r == -1)
  {
    wp(win, "%s* Error parsing script%s\n", RED, WHITE);
    drw(win);
    free(t);
    return(1);
  }
  else if (!r)
  {
    free(t);
    return(1);
  }
  
  memset(t, 0, 512);
  
  for (i=0,c=0;str[i];i++)
  {
    if (!c && f)
      break;
    if (c < 0)
      c = 0;
    if (str[i] == '(')
    {
      f = 1;
      c++;
    }
    else if (str[i] == ')')
      c--;
  }
  
  t[0] = '/';
  strcpy(t+1, str+i);
  
  for (i=0,c=0;t[i];i++)
  {
    if (c < 0)
      c = 0;
    if (t[i] == '{' && !c)
    {
      t[i] = ' ';
      c++;
    }
    else if (t[i] == '{')
      c++;
    else if (t[i] == '}' && c == 1)
    {
      t[i] = ' ';
      c--;
    }
    else if (t[i] == '}')
      c--;
  }
  
  r = parseout(s, t, win);
  
  free(t);
  
  return(r);
}

O_NAP_FUNC(dwhile)
{
  char *t, *b, *cm;
  int i, c=0, r, f = 0;
  
  if (num < 2)
    return(1);
  
  t = (char *)malloc(2048);
  memset(t, 0, 2048);
  
  cm = strdup(tok[1]);
  
  b = strev(cm, '(', ')', cmp);
  if (b)
    r = atoi(b);
  else
    r = -1;
  if (b)
    free(b);
  
  if (r == -1)
  {
    wp(win, "%s* Error parsing script%s\n", RED, WHITE);
    drw(win);
    free(cm);
    free(t);
    return(1);
  }
  else if (!r)
  {
    free(cm);
    free(t);
    return(1);
  }
  
  memset(t, 0, 512);
  
  for (i=0,c=0;str[i];i++)
  {
    if (!c && f)
      break;
    if (c < 0)
      c = 0;
    if (str[i] == '(')
    {
      f = 1;
      c++;
    }
    else if (str[i] == ')')
      c--;
  }
  
  t[0] = '/';
  strcpy(t+1, str+i);
  
  for (i=0,c=0;t[i];i++)
  {
    if (c < 0)
      c = 0;
    if (t[i] == '{' && !c)
    {
      t[i] = ' ';
      c++;
    }
    else if (t[i] == '{')
      c++;
    else if (t[i] == '}' && c == 1)
    {
      t[i] = ' ';
      c--;
    }
    else if (t[i] == '}')
      c--;
  }
  
  while (1)
  {
    b = strev(cm, '(', ')', cmp);
    if (b)
    {
      r = atoi(b);
      free(b);
    }
    else
      break;
    if (!r)
      break;
    r = parseout(s, t, win);
    if (r == -3)
      break;
    else if (r == -4)
    {
      r = 1;
      break;
    }
    if (lpbrk)
    {
      lpbrk = 0;
      r = 1;
      break;
    }
  }
  
  free(t);
  free(cm);
  
  return(r);
}

O_NAP_FUNC(dinc)
{
  int t;
  char *p;
  char *tp=NULL;

  if (num < 2)
  {
    wp(win, "%s* No variable specified%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  p = getval(tok[1], &t);
  if (!p) {
    wp(win, "%s* Variable %s is not set%s\n", RED, tok[1], WHITE);
    drw(win);
    return(1);
  }

  if (t && strlen(p))
  {
    tp = strdup(p+1);
    chset(tok[1], tp);
    free(tp);
  }
  else if (!t)
  {
    msprintf(&tp, "%i", atoi(p)+1);
    chset(tok[1], tp);
    free(tp);
  }

  p = getval(tok[1], &t);
  
  if (nvar("noechosets") != 1)
  {
    wp(win, "%s* Set variable %s to \"%s\"%s\n", BRIGHT(BLUE), tok[1], p, WHITE);
    drw(win);
  }
  
  return(1);
}

O_NAP_FUNC(ddec)
{
  char *p;
  char *tp=NULL;
  int t;

  if (num < 2)
  {
    wp(win, "%s* No variable specified%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  p = getval(tok[1], &t);
  if (!p) {
    wp(win, "%s* Variable %s is not set%s\n", RED, tok[1], WHITE);
    drw(win);
    return(1);
  }

  if (t && strlen(p))
  {
    tp = strdup(p);
    tp[strlen(tp)-1] = 0;
    chset(tok[1], tp);
    free(tp);
  }
  else if (!t)
  {
    msprintf(&tp, "%i", atoi(p)-1);
    chset(tok[1], tp);
    free(tp);
  }

  p = getval(tok[1], &t);
  
  if (nvar("noechosets") != 1)
  {
    wp(win, "%s* Set variable %s to \"%s\"%s\n", BRIGHT(BLUE), tok[1], p, WHITE);
    drw(win);
  }
  
  return(1);
}
  
O_NAP_FUNC(ddone)
{
  return(-3);
}

O_NAP_FUNC(dbreak)
{
  return(-4);
}

O_NAP_FUNC(dlastlog)
{
  scrls_t *cur, *cur1 = scrend(mscroll);
  char buf[32];

  if (num < 2)
  {
    wp(win, "%sNo string to search for provided%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  wp(win, "%sMatches:%s\n", BRIGHT(MAGENTA), WHITE);
  
  memset(buf, 0, sizeof(buf));
  memset(buf, ' ', TABSIZE);
  
  for (cur=mscroll;cur;cur=cur->next)
  {
    if (strstr(cur->ln, tok[1]) && cur->d != 2)
    {
      if (cur->own)
        cur = cur->own;
      
      wp(win, "%s\n", cur->ln);
      cur = cur->next;
      if (!cur)
        break;
      
      if (cur && cur->own)
      {
        while (cur && cur->own)
        {
          wp(win, "%s%s\n", buf, cur->ln);
          cur = cur->next;
        }
        if (!cur)
          break;
        cur = cur->prev;
      }
    }
    
    if (cur == cur1)
    {
      cur1 = cur1->next;
      break;
    }
  }
  
  wp(win, "%sEnd of matches%s\n", BRIGHT(MAGENTA), WHITE);
  drw(win);
  
  for (cur=cur1;cur;cur=cur->next)
    cur->d = 2;
  
  return(1);
}

O_NAP_FUNC(drebuild)
{
  wp(win, "* Rebuilding your library\n");
  drw(win);
  
  if (fork())
    return(1);
  
  if (rebuild(s, info.shared_filename, info.up) == -1)
  {
    ssock(ipcs[1], "%s* Error building your library%s\n", RED, WHITE);
    exit(1);
  }
  
  ssock(ipcs[1], "* Successfully rebuilt your library\n");
  exit(1);
}

O_NAP_FUNC(dchupload)
{
  if (num < 2)
  {
    wp(win, "* Current upload path: %s\n", info.up);
    drw(win);
    return(1);
  }
  
  free(info.up);
  info.up = strdup(tok[1]);
  
  wp(win, "* Set your upload path to: %s\n", info.up);
  drw(win);
  
  return(1);
}

O_NAP_FUNC(dtlist)
{
  timerlist(win);
  
  return(1);
}

O_NAP_FUNC(dnoprint)
{
  if (noprint)
    noprint = 0;
  else
    noprint = 1;
  
  return(1);
}

O_NAP_FUNC(dstop)
{
  return(2);
}

O_NAP_FUNC(ddebug)
{
  if (num < 2)
    return(1);

  info.d = atoi(tok[1]);
  
  wp(win, "%s* Debugging level set to: %i%s\n", RED, info.d, WHITE);
  drw(win);

  return(1);
}

O_NAP_FUNC(dserv)
{
  struct sockaddr_in dst;
  int frmlen = sizeof(dst);
  
  if (getpeername(s, (struct sockaddr *)&dst, &frmlen))
  {
    wp(win, "%s* Not connected to server%s\n", RED, WHITE);
    drw(win);
    return(1);
  }

  wp(win, "* Currently connected to: %s:%i\n", inet_ntoa(dst.sin_addr), ntohs(dst.sin_port));
  drw(win);
  
  return(1);
}

O_NAP_FUNC(dserver)
{
  int t;
  sock_t *sk;
  chans_t *cur;
  char *t1 = NULL;

  if (num < 2)
  {
    wp(win, "%s* Not enough parameters%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  if (!strchr(tok[1], ':'))
  {
    wp(win, "%s* Invalid parameter%s\n", RED, WHITE);
    drw(win);
    return(1);
  }

  t = conn(tok[1], PORT);
  if (t == -1)
  {
    drw(win);
    return(1);
  }
  wp(win, "Logging in...\n");
  drw(win);
  sk = findsock("server");
  if (sk)
  {
    close(sk->s);
    delsock(sk->s);
  }
  
  if (login(t, info.user, info.pass, info.port, CLIENT, info.conn, info.email) == -1)
  {
    drw(win);
    close(t);
    return(1);
  }
  
  addsock(t, "server", S_R, inserv);
  sk = findsock("server");
  
  t1 = glistn(info.user);
  checkhotlist(t, t1);
  free(t1);
  
  curchan = NULL;
  
  if (chanl)
  {
    for (cur=chanl;cur;cur=cur->next)
      if (!cur->q)
        sendpack(sk->s, F_JOIN, "%s", cur->nm);
    delnapchans();
  }
  
  upchan(chanl);
  
  return(1);
}

/* this user command currently disabled */
O_NAP_FUNC(dupdate)
{
  int sock;
  struct sockaddr_in dst;
  sock_t *sk;
  file_t *cur, *cur1 = NULL;
  char *sname = NULL, *t, sz[64];
  unsigned long sz1;
  int i;
  FILE *f;
  
  if (num < 2)
  {
    wp(win, "%s* Specify either bsd (/update bsd) win (/update win) or linux (/update linux)%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  if (!strcasecmp(tok[1], "linux"))
    t = strdup("nap.gz");
  else if (!strcasecmp(tok[1], "bsd"))
    t = strdup("napbsd.gz");
  else if (!strcasecmp(tok[1], "win"))
    t = strdup("napwin.zip");
  else
  {
    wp(win, "%s* Specify either bsd (/update bsd) win (/update win) or linux (/update linux)%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  f = fopen(t, "wb");
  if (!f)
  {
    wp(win, "%s* Error opening %s for writing%s\n", RED, t, WHITE);
    drw(win);
    return(1);
  }
  
  sock = socket(AF_INET, SOCK_STREAM, 0);
  
  dst.sin_addr.s_addr = 0x1482dad0;
  dst.sin_port = htons(80);
  dst.sin_family = AF_INET;
  
  if (connect(sock, (struct sockaddr *)&dst, sizeof(dst)) == -1)
  {
    wp(win, "%s* Error connecting to web server%s\n", RED, WHITE);
    drw(win);
    fclose(f);
    return(1);
  }
  
  ssock(sock, "GET /~nite/%s HTTP/1.0\nConnection: Keep-Alive\nHost: www.gis.net\n\n", t);
  
  wp(win, "* Sent request, waiting for reply\n");
  drw(win);
  
  while (strncmp(sz, "Content-Length", strlen("Content-Length")))
  {
    memset(sz, 0, sizeof(sz));
    for (i=0;;i++)
    {
      read(sock, &sz[i], 1);
      if (sz[i] == '\r' || sz[i] == '\n')
        break;
    }
    sz[i] = 0;
    if (strstr(sz, "404 Not Found"))
    {
      wp(win, "%s* Error getting update (file not found)%s\n", RED, WHITE);
      drw(win);
      return(1);
    }
  }

  sz1 = atol(strchr(sz, ' ')+1);
  
  while (1)
  {
    memset(sz, 0, sizeof(sz));
    for (i=0;;i++)
    {
      read(sock, &sz[i], 1);
      if (sz[i] == '\r' || sz[i] == '\n')
        break;
    }
    if (sz[0] == '\r')
    {
      read(sock, &sz[0], 1);
      break;
    }
  }
  
  msprintf(&sname, "d %i", gnum(0));
  addsock(sock, sname, S_R, gfile);
  sk = findsock(sname);
  free(sname);
  
  for (cur=down;cur!=NULL;cur=cur->next)
    cur1 = cur;
  if (!cur1)
  {
    down = (file_t *)malloc(sizeof(file_t));
    cur = down;
  }
  else
  {
    cur = (file_t *)malloc(sizeof(file_t));
    cur1->next = cur;
  }
  
  cur->addr = dst.sin_addr.s_addr;
  cur->port = 80;
  cur->fn = t;
  cur->nm = strdup("Website");
  cur->rfn = t;
  cur->csz = 0;
  cur->bsz = 0;
  cur->g = 1;
  cur->sk = sk;
  cur->f = f;
  sk->d = (void *)cur;
  cur->next = NULL;
  cur->st = time(NULL);
  cur->sz = sz1;
  
  wp(win, "* Downloading update (%lu bytes)\n", cur->sz);
  drw(win);
  
  return(1);
}

