/******************************************************************************
*
* Nagios check_smtp plugin
*
* License: GPL
* Copyright (c) 1999-2006 nagios-plugins team
*
* Last Modified: $Date: 2006/10/19 23:53:28 $
*
* Description:
*
* This file contains the check_smtp plugin
*
*  This plugin will attempt to open an SMTP connection with the host.
*
*
* License Information:
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*
*
* $Id: check_smtp.c,v 1.54 2006/10/19 23:53:28 opensides Exp $
* 
******************************************************************************/

const char *progname = "check_smtp";
const char *revision = "$Revision: 1.54 $";
const char *copyright = "2000-2006";
const char *email = "nagiosplug-devel@lists.sourceforge.net";

#include "common.h"
#include "netutils.h"
#include "utils.h"

#ifdef HAVE_SSL
int check_cert = FALSE;
int days_till_exp;
#  define my_recv(buf, len) ((use_ssl && ssl_established) ? np_net_ssl_read(buf, len) : read(sd, buf, len))
#  define my_send(buf, len) ((use_ssl && ssl_established) ? np_net_ssl_write(buf, len) : send(sd, buf, len, 0))
#else /* ifndef HAVE_SSL */
#  define my_recv(buf, len) read(sd, buf, len)
#  define my_send(buf, len) send(sd, buf, len, 0)
#endif

enum {
	SMTP_PORT	= 25
};
#define SMTP_EXPECT "220"
#define SMTP_HELO "HELO "
#define SMTP_EHLO "EHLO "
#define SMTP_QUIT "QUIT\r\n"
#define SMTP_STARTTLS "STARTTLS\r\n"
#define SMTP_AUTH_LOGIN "AUTH LOGIN\r\n"

#ifndef HOST_MAX_BYTES
#define HOST_MAX_BYTES 255
#endif

#define EHLO_SUPPORTS_STARTTLS 1

int process_arguments (int, char **);
int validate_arguments (void);
void print_help (void);
void print_usage (void);
int my_close(void);

#include "regex.h"
char regex_expect[MAX_INPUT_BUFFER] = "";
regex_t preg;
regmatch_t pmatch[10];
char timestamp[20] = "";
char errbuf[MAX_INPUT_BUFFER];
int cflags = REG_EXTENDED | REG_NOSUB | REG_NEWLINE;
int eflags = 0;
int errcode, excode;

int server_port = SMTP_PORT;
char *server_address = NULL;
char *server_expect = NULL;
int smtp_use_dummycmd = 0;
char *mail_command = NULL;
char *from_arg = NULL;
int ncommands=0;
int command_size=0;
int nresponses=0;
int response_size=0;
char **commands = NULL;
char **responses = NULL;
char *authtype = NULL;
char *authuser = NULL;
char *authpass = NULL;
int warning_time = 0;
int check_warning_time = FALSE;
int critical_time = 0;
int check_critical_time = FALSE;
int verbose = 0;
int use_ssl = FALSE;
short use_ehlo = FALSE;
short ssl_established = 0;
char *localhostname = NULL;
int sd;
char buffer[MAX_INPUT_BUFFER];
enum {
  TCP_PROTOCOL = 1,
  UDP_PROTOCOL = 2,
  MAXBUF = 1024
};

/* written by lauri alanko */
static char *
base64 (const char *bin, size_t len)
{

	char *buf = (char *) malloc ((len + 2) / 3 * 4 + 1);
	size_t i = 0, j = 0;

	char BASE64_END = '=';
	char base64_table[64];
	strncpy (base64_table, "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/", 64);

	while (j < len - 2) {
		buf[i++] = base64_table[bin[j] >> 2];
		buf[i++] = base64_table[((bin[j] & 3) << 4) | (bin[j + 1] >> 4)];
		buf[i++] = base64_table[((bin[j + 1] & 15) << 2) | (bin[j + 2] >> 6)];
		buf[i++] = base64_table[bin[j + 2] & 63];
		j += 3;
	}

	switch (len - j) {
	case 1:
		buf[i++] = base64_table[bin[j] >> 2];
		buf[i++] = base64_table[(bin[j] & 3) << 4];
		buf[i++] = BASE64_END;
		buf[i++] = BASE64_END;
		break;
	case 2:
		buf[i++] = base64_table[bin[j] >> 2];
		buf[i++] = base64_table[((bin[j] & 3) << 4) | (bin[j + 1] >> 4)];
		buf[i++] = base64_table[(bin[j + 1] & 15) << 2];
		buf[i++] = BASE64_END;
		break;
	case 0:
		break;
	}

	buf[i] = '\0';
	return buf;
}

int
main (int argc, char **argv)
{
	short supports_tls=FALSE;
	int n = 0;
	double elapsed_time;
	long microsec;
	int result = STATE_UNKNOWN;
	char *cmd_str = NULL;
	char *helocmd = NULL;
	char *error_msg = NULL;
	struct timeval tv;
	struct hostent *hp;

	setlocale (LC_ALL, "");
	bindtextdomain (PACKAGE, LOCALEDIR);
	textdomain (PACKAGE);

	if (process_arguments (argc, argv) == ERROR)
		usage4 (_("Could not parse arguments"));

	/* initialize the HELO command with the localhostname */
	if(! localhostname){
		localhostname = malloc (HOST_MAX_BYTES);
		if(!localhostname){
			printf(_("malloc() failed!\n"));
			return STATE_CRITICAL;
		}
		if(gethostname(localhostname, HOST_MAX_BYTES)){
			printf(_("gethostname() failed!\n"));
			return STATE_CRITICAL;
		}
		hp = gethostbyname(localhostname);
		if(!hp) helocmd = localhostname;
		else helocmd = hp->h_name;
	} else {
		helocmd = localhostname;
	}
	if(use_ehlo)
		asprintf (&helocmd, "%s%s%s", SMTP_EHLO, helocmd, "\r\n");
	else
		asprintf (&helocmd, "%s%s%s", SMTP_HELO, helocmd, "\r\n");

	/* initialize the MAIL command with optional FROM command  */
	asprintf (&cmd_str, "%sFROM: %s%s", mail_command, from_arg, "\r\n");

	if (verbose && smtp_use_dummycmd)
		printf ("FROM CMD: %s", cmd_str);
	
	/* initialize alarm signal handling */
	(void) signal (SIGALRM, socket_timeout_alarm_handler);

	/* set socket timeout */
	(void) alarm (socket_timeout);

	/* start timer */
	gettimeofday (&tv, NULL);

	/* try to connect to the host at the given port number */
	result = my_tcp_connect (server_address, server_port, &sd);

	if (result == STATE_OK) { /* we connected */

		/* watch for the SMTP connection string and */
		/* return a WARNING status if we couldn't read any data */
		if (recv (sd, buffer, MAX_INPUT_BUFFER - 1, 0) == -1) {
			printf (_("recv() failed\n"));
			result = STATE_WARNING;
		}
		else {
			if (verbose)
				printf ("%s", buffer);
			/* strip the buffer of carriage returns */
			strip (buffer);
			/* make sure we find the response we are looking for */
			if (!strstr (buffer, server_expect)) {
				if (server_port == SMTP_PORT)
					printf (_("Invalid SMTP response received from host\n"));
				else
					printf (_("Invalid SMTP response received from host on port %d\n"),
									server_port);
				result = STATE_WARNING;
			}
		}

		/* send the HELO/EHLO command */
		send(sd, helocmd, strlen(helocmd), 0);

		/* allow for response to helo command to reach us */
		if(read (sd, buffer, MAXBUF - 1) < 0){
			printf (_("recv() failed\n"));
			return STATE_WARNING;
		} else if(use_ehlo){
			buffer[MAXBUF-1]='\0';
			if(strstr(buffer, "250 STARTTLS") != NULL ||
			   strstr(buffer, "250-STARTTLS") != NULL){
				supports_tls=TRUE;
			}
		}

		if(use_ssl && ! supports_tls){
			printf(_("WARNING - TLS not supported by server\n"));
			send (sd, SMTP_QUIT, strlen (SMTP_QUIT), 0);
			return STATE_WARNING;
		}

#ifdef HAVE_SSL
		if(use_ssl) {
		  /* send the STARTTLS command */
		  send(sd, SMTP_STARTTLS, strlen(SMTP_STARTTLS), 0);

		  recv(sd,buffer, MAX_INPUT_BUFFER-1, 0); /* wait for it */
		  if (!strstr (buffer, server_expect)) {
		    printf (_("Server does not support STARTTLS\n"));
		    send (sd, SMTP_QUIT, strlen (SMTP_QUIT), 0);
		    return STATE_UNKNOWN;
		  }
		  result = np_net_ssl_init(sd);
		  if(result != STATE_OK) {
		    printf (_("CRITICAL - Cannot create SSL context.\n"));
		    np_net_ssl_cleanup();
		    close(sd);
		    return STATE_CRITICAL;
		  } else {
			ssl_established = 1;
		  }

		/*
		 * Resend the EHLO command.
		 *
		 * RFC 3207 (4.2) says: ``The client MUST discard any knowledge
		 * obtained from the server, such as the list of SMTP service
		 * extensions, which was not obtained from the TLS negotiation
		 * itself.  The client SHOULD send an EHLO command as the first
		 * command after a successful TLS negotiation.''  For this
		 * reason, some MTAs will not allow an AUTH LOGIN command before
		 * we resent EHLO via TLS.
		 */
		if (my_send(helocmd, strlen(helocmd)) <= 0) {
			printf(_("SMTP UNKNOWN - Cannot send EHLO command via TLS.\n"));
			my_close();
			return STATE_UNKNOWN;
		}
		if (verbose)
			printf(_("sent %s"), helocmd);
		if ((n = my_recv(buffer, MAX_INPUT_BUFFER - 1)) <= 0) {
			printf(_("SMTP UNKNOWN - Cannot read EHLO response via TLS.\n"));
			my_close();
			return STATE_UNKNOWN;
		}
		if (verbose) {
			buffer[n] = '\0';
			printf("%s", buffer);
		}

#  ifdef USE_OPENSSL
		  if ( check_cert ) {
		    result = np_net_ssl_check_cert(days_till_exp);
		    if(result != STATE_OK){
		      printf (_("CRITICAL - Cannot retrieve server certificate.\n"));
		    }
		    my_close();
		    return result;
		  }
#  endif /* USE_OPENSSL */
		}
#endif
				
		/* sendmail will syslog a "NOQUEUE" error if session does not attempt
		 * to do something useful. This can be prevented by giving a command
		 * even if syntax is illegal (MAIL requires a FROM:<...> argument)
		 *
		 * According to rfc821 you can include a null reversepath in the from command
		 * - but a log message is generated on the smtp server.
		 *
		 * You can disable sending mail_command with '--nocommand'
		 * Use the -f option to provide a FROM address
		 */
		if (smtp_use_dummycmd) {
		  my_send(cmd_str, strlen(cmd_str));
		  my_recv(buffer, MAX_INPUT_BUFFER-1);
		  if (verbose) 
		    printf("%s", buffer);
		}

		while (n < ncommands) {
			asprintf (&cmd_str, "%s%s", commands[n], "\r\n");
			my_send(cmd_str, strlen(cmd_str));
			my_recv(buffer, MAX_INPUT_BUFFER-1);
			if (verbose) 
				printf("%s", buffer);
			strip (buffer);
			if (n < nresponses) {
				cflags |= REG_EXTENDED | REG_NOSUB | REG_NEWLINE;
				errcode = regcomp (&preg, responses[n], cflags);
				if (errcode != 0) {
					regerror (errcode, &preg, errbuf, MAX_INPUT_BUFFER);
					printf (_("Could Not Compile Regular Expression"));
					return ERROR;
				}
				excode = regexec (&preg, buffer, 10, pmatch, eflags);
				if (excode == 0) {
					result = STATE_OK;
				}
				else if (excode == REG_NOMATCH) {
					result = STATE_WARNING;
					printf (_("SMTP %s - Invalid response '%s' to command '%s'\n"), state_text (result), buffer, commands[n]);
				}
				else {
					regerror (excode, &preg, errbuf, MAX_INPUT_BUFFER);
					printf (_("Execute Error: %s\n"), errbuf);
					result = STATE_UNKNOWN;
				}
			}
			n++;
		}

		if (authtype != NULL) {
			if (strcmp (authtype, "LOGIN") == 0) {
				char *abuf;
				int ret;
				do {
					if (authuser == NULL) {
						result = STATE_CRITICAL;
						error_msg = _("no authuser specified, ");
						break;
					}
					if (authpass == NULL) {
						result = STATE_CRITICAL;
						error_msg = _("no authpass specified, ");
						break;
					}

					/* send AUTH LOGIN */
					my_send(SMTP_AUTH_LOGIN, strlen(SMTP_AUTH_LOGIN));
					if (verbose)
						printf (_("sent %s\n"), "AUTH LOGIN");

					if((ret = my_recv(buffer, MAXBUF - 1)) < 0){
						error_msg = _("recv() failed after AUTH LOGIN, \n");
						result = STATE_WARNING;
						break;
					}
					buffer[ret] = 0;
					if (verbose)
						printf (_("received %s\n"), buffer);

					if (strncmp (buffer, "334", 3) != 0) {
						result = STATE_CRITICAL;
						error_msg = _("invalid response received after AUTH LOGIN, ");
						break;
					}

					/* encode authuser with base64 */
					abuf = base64 (authuser, strlen(authuser));
					strcat (abuf, "\r\n");
					my_send(abuf, strlen(abuf));
					if (verbose)
						printf (_("sent %s\n"), abuf);

					if ((ret = my_recv(buffer, MAX_INPUT_BUFFER-1)) == -1) {
						result = STATE_CRITICAL;
						error_msg = _("recv() failed after sending authuser, ");
						break;
					}
					buffer[ret] = 0;
					if (verbose) {
						printf (_("received %s\n"), buffer);
					}
					if (strncmp (buffer, "334", 3) != 0) {
						result = STATE_CRITICAL;
						error_msg = _("invalid response received after authuser, ");
						break;
					}
					/* encode authpass with base64 */
					abuf = base64 (authpass, strlen(authpass));
					strcat (abuf, "\r\n");
					my_send(abuf, strlen(abuf));
					if (verbose) {
						printf (_("sent %s\n"), abuf);
					}
					if ((ret = my_recv(buffer, MAX_INPUT_BUFFER-1)) == -1) {
						result = STATE_CRITICAL;
						error_msg = _("recv() failed after sending authpass, ");
						break;
					}
					buffer[ret] = 0;
					if (verbose) {
						printf (_("received %s\n"), buffer);
					}
					if (strncmp (buffer, "235", 3) != 0) {
						result = STATE_CRITICAL;
						error_msg = _("invalid response received after authpass, ");
						break;
					}
					break;
				} while (0);
			} else {
				result = STATE_CRITICAL;
				error_msg = _("only authtype LOGIN is supported, ");
			}
		}

		/* tell the server we're done */
		my_send (SMTP_QUIT, strlen (SMTP_QUIT));

		/* finally close the connection */
		close (sd);
	}

	/* reset the alarm */
	alarm (0);

	microsec = deltime (tv);
	elapsed_time = (double)microsec / 1.0e6;

	if (result == STATE_OK) {
		if (check_critical_time && elapsed_time > (double) critical_time)
			result = STATE_CRITICAL;
		else if (check_warning_time && elapsed_time > (double) warning_time)
			result = STATE_WARNING;
	}

	printf (_("SMTP %s - %s%.3f sec. response time%s%s|%s\n"),
			state_text (result),
			(error_msg == NULL ? "" : error_msg),
			elapsed_time,
			verbose?", ":"", verbose?buffer:"",
			fperfdata ("time", elapsed_time, "s",
				(int)check_warning_time, warning_time,
				(int)check_critical_time, critical_time,
				TRUE, 0, FALSE, 0));

	return result;
}



/* process command-line arguments */
int
process_arguments (int argc, char **argv)
{
	int c;

	int option = 0;
	static struct option longopts[] = {
		{"hostname", required_argument, 0, 'H'},
		{"expect", required_argument, 0, 'e'},
		{"critical", required_argument, 0, 'c'},
		{"warning", required_argument, 0, 'w'},
		{"timeout", required_argument, 0, 't'},
		{"port", required_argument, 0, 'p'},
		{"from", required_argument, 0, 'f'},
		{"fqdn", required_argument, 0, 'F'},
		{"authtype", required_argument, 0, 'A'},
		{"authuser", required_argument, 0, 'U'},
		{"authpass", required_argument, 0, 'P'},
		{"command", required_argument, 0, 'C'},
		{"response", required_argument, 0, 'R'},
		{"nocommand", required_argument, 0, 'n'},
		{"verbose", no_argument, 0, 'v'},
		{"version", no_argument, 0, 'V'},
		{"use-ipv4", no_argument, 0, '4'},
		{"use-ipv6", no_argument, 0, '6'},
		{"help", no_argument, 0, 'h'},
		{"starttls",no_argument,0,'S'},
		{"certificate",required_argument,0,'D'},
		{0, 0, 0, 0}
	};

	if (argc < 2)
		return ERROR;

	for (c = 1; c < argc; c++) {
		if (strcmp ("-to", argv[c]) == 0)
			strcpy (argv[c], "-t");
		else if (strcmp ("-wt", argv[c]) == 0)
			strcpy (argv[c], "-w");
		else if (strcmp ("-ct", argv[c]) == 0)
			strcpy (argv[c], "-c");
	}

	while (1) {
		c = getopt_long (argc, argv, "+hVv46t:p:f:e:c:w:H:C:R:SD:F:A:U:P:",
		                 longopts, &option);

		if (c == -1 || c == EOF)
			break;

		switch (c) {
		case 'H':									/* hostname */
			if (is_host (optarg)) {
				server_address = optarg;
			}
			else {
				usage2 (_("Invalid hostname/address"), optarg);
			}
			break;
		case 'p':									/* port */
			if (is_intpos (optarg))
				server_port = atoi (optarg);
			else
				usage4 (_("Port must be a positive integer"));
			break;
		case 'F':
		/* localhostname */
			localhostname = strdup(optarg);
			break;
		case 'f':									/* from argument */
			from_arg = optarg;
			smtp_use_dummycmd = 1;
			break;
		case 'A':
			authtype = optarg;
			break;
		case 'U':
			authuser = optarg;
			break;
		case 'P':
			authpass = optarg;
			break;
		case 'e':									/* server expect string on 220  */
			server_expect = optarg;
			break;
		case 'C':									/* commands  */
			if (ncommands >= command_size) {
				commands = realloc (commands, command_size+8);
				if (commands == NULL)
					die (STATE_UNKNOWN,
					     _("Could not realloc() units [%d]\n"), ncommands);
			}
			commands[ncommands] = optarg;
			ncommands++;
			break;
		case 'R':									/* server responses */
			if (nresponses >= response_size) {
				responses = realloc (responses, response_size+8);
				if (responses == NULL)
					die (STATE_UNKNOWN,
					     _("Could not realloc() units [%d]\n"), nresponses);
			}
			responses[nresponses] = optarg;
			nresponses++;
			break;
		case 'c':									/* critical time threshold */
			if (is_intnonneg (optarg)) {
				critical_time = atoi (optarg);
				check_critical_time = TRUE;
			}
			else {
				usage4 (_("Critical time must be a positive integer"));
			}
			break;
		case 'w':									/* warning time threshold */
			if (is_intnonneg (optarg)) {
				warning_time = atoi (optarg);
				check_warning_time = TRUE;
			}
			else {
				usage4 (_("Warning time must be a positive integer"));
			}
			break;
		case 'v':									/* verbose */
			verbose++;
			break;
		case 't':									/* timeout */
			if (is_intnonneg (optarg)) {
				socket_timeout = atoi (optarg);
			}
			else {
				usage4 (_("Timeout interval must be a positive integer"));
			}
			break;
		case 'S':
		/* starttls */
			use_ssl = TRUE;
			use_ehlo = TRUE;
			break;
		case 'D':
		/* Check SSL cert validity */
#ifdef USE_OPENSSL
			if (!is_intnonneg (optarg))
				usage2 ("Invalid certificate expiration period",optarg);
				days_till_exp = atoi (optarg);
				check_cert = TRUE;
#else
				usage (_("SSL support not available - install OpenSSL and recompile"));
#endif
			break;
		case '4':
			address_family = AF_INET;
			break;
		case '6':
#ifdef USE_IPV6
			address_family = AF_INET6;
#else
			usage4 (_("IPv6 support not available"));
#endif
			break;
		case 'V':									/* version */
			print_revision (progname, revision);
			exit (STATE_OK);
		case 'h':									/* help */
			print_help ();
			exit (STATE_OK);
		case '?':									/* help */
			usage2 (_("Unknown argument"), optarg);
		}
	}

	c = optind;
	if (server_address == NULL) {
		if (argv[c]) {
			if (is_host (argv[c]))
				server_address = argv[c];
			else
				usage2 (_("Invalid hostname/address"), argv[c]);
		}
		else {
			asprintf (&server_address, "127.0.0.1");
		}
	}

	if (server_expect == NULL)
		server_expect = strdup (SMTP_EXPECT);

	if (mail_command == NULL)
		mail_command = strdup("MAIL ");

	if (from_arg==NULL)
		from_arg = strdup(" ");

	return validate_arguments ();
}



int
validate_arguments (void)
{
	return OK;
}


int 
my_close (void)
{
#ifdef HAVE_SSL
  np_net_ssl_cleanup();
#endif
  return close(sd);
}


void
print_help (void)
{
	char *myport;
	asprintf (&myport, "%d", SMTP_PORT);

	print_revision (progname, revision);

	printf ("Copyright (c) 1999-2001 Ethan Galstad <nagios@nagios.org>\n");
	printf (COPYRIGHT, copyright, email);

	printf("%s\n", _("This plugin will attempt to open an SMTP connection with the host."));

  printf ("\n\n");

	print_usage ();

	printf (_(UT_HELP_VRSN));

	printf (_(UT_HOST_PORT), 'p', myport);

	printf (_(UT_IPv46));

	printf (" %s\n", "-e, --expect=STRING");
  printf (_("    String to expect in first line of server response (default: '%s')\n"), SMTP_EXPECT);
  printf (" %s\n", "-n, nocommand");
  printf ("    %s\n", _("Suppress SMTP command"));
  printf (" %s\n", "-C, --command=STRING");
  printf ("    %s\n", _("SMTP command (may be used repeatedly)"));
  printf (" %s\n", "-R, --command=STRING");
  printf ("    %s\n", _("Expected response to command (may be used repeatedly)"));
  printf (" %s\n", "-f, --from=STRING");
  printf ("    %s\n", _("FROM-address to include in MAIL command, required by Exchange 2000")),
#ifdef HAVE_SSL
  printf (" %s\n", "-D, --certificate=INTEGER");
  printf ("    %s\n", _("Minimum number of days a certificate has to be valid."));
  printf (" %s\n", "-S, --starttls");
  printf ("    %s\n", _("Use STARTTLS for the connection."));
#endif

	printf (" %s\n", "-A, --authtype=STRING");
  printf ("    %s\n", _("SMTP AUTH type to check (default none, only LOGIN supported)"));
  printf (" %s\n", "-U, --authuser=STRING");
  printf ("    %s\n", _("SMTP AUTH username"));
  printf (" %s\n", "-P, --authpass=STRING");
  printf ("    %s\n", _("SMTP AUTH password"));

	printf (_(UT_WARN_CRIT));

	printf (_(UT_TIMEOUT), DEFAULT_SOCKET_TIMEOUT);

	printf (_(UT_VERBOSE));

	printf("\n");
	printf ("%s\n", _("Successul connects return STATE_OK, refusals and timeouts return"));
  printf ("%s\n", _("STATE_CRITICAL, other errors return STATE_UNKNOWN.  Successful"));
  printf ("%s\n", _("connects, but incorrect reponse messages from the host result in"));
  printf ("%s\n", _("STATE_WARNING return values."));

	printf (_(UT_SUPPORT));
}



void
print_usage (void)
{
  printf (_("Usage:"));
	printf ("%s -H host [-p port] [-e expect] [-C command] [-f from addr]", progname);
  printf ("[-A authtype -U authuser -P authpass] [-w warn] [-c crit] [-t timeout]\n");
  printf ("[-S] [-D days] [-n] [-v] [-4|-6]\n");
}

