<?php
// $Horde: nag/lib/Nag.php,v 1.39.2.3 2002/06/06 12:51:47 bjn Exp $

require_once NAG_BASE . '/lib/constants.php';

/**
 * Array for mapping the completed states to their string representations.
 * @var array $completion_map
 */
$completion_map = array(
    0 => _("No"),
    1 => _("Yes"));

/**
 * Nag Base Class.
 *
 * @author Jon Parise <jon@horde.org>
 * @version $Revision: 1.39.2.3 $
 * @package nag
 */
class Nag {

    /**
     * Retrieves the current user's task list from storage.
     * This function will also sort the resulting list, if requested.
     *
     * @param object Nag_Driver  $storage   The current storage object.
     * @param constant $sortby   (optional) The field by which to sort.
     *                           (NAG_SORT_PRIORITY, NAG_SORT_NAME
     *                            NAG_SORT_DUE, NAG_SORT_COMPLETION)
     * @param constant $sortdir  (optional) The direction by which to sort.
     *                           (NAG_SORT_ASC, NAG_SORT_DESC)
     *
     * @return array        Returns a list of the requested tasks.
     *
     * @see Nag_Driver::listTasks()
     */
    function listTasks($storage, $sortby = NAG_SORT_PRIORITY,
                       $sortdir = NAG_SORT_ASCEND)
    {
        global $prefs;

        /* Retrieve the task list for storage. */
        $tasks = $storage->listTasks();

        /* Filter completed tasks if the user doesn't want them shown. */
        if (!$prefs->getValue('show_completed')) {
            foreach ($tasks as $taskID => $task) {
                if (!empty($task['completed'])) {
                    unset($tasks[$taskID]);
                }
            }
        }

        /* Sort the task list. */
        $sort_functions = array(
            NAG_SORT_PRIORITY => 'ByPriority',
            NAG_SORT_NAME => 'ByName',
            NAG_SORT_CATEGORY => 'ByCategory',
            NAG_SORT_DUE => 'ByDue',
            NAG_SORT_COMPLETION => 'ByCompletion'
        );

        /* Sort the array if we have a sort function defined for this field. */
        if (isset($sort_functions[$sortby])) {
            $prefix = ($sortdir == NAG_SORT_DESCEND) ? '_rsort' : '_sort';
            uasort($tasks, $prefix . $sort_functions[$sortby]);
        }

        return $tasks;
    }

    /**
     * List a user's categories
     *
     * @return array A list of categories.
     */
    function listCategories()
    {
        global $prefs;
        
        static $catString, $categories;

        if (!isset($categories) || $catString != $prefs->getValue('task_categories')) {
            $catString = $prefs->getValue('task_categories');
            $cats = explode('|', $catString);
            $categories = array(0 => _("Unfiled"));
            foreach ($cats as $cat) {
                list($key, $val) = explode(':', $cat);
                $categories[$key] = $val;
            }
        }

        return $categories;
    }

    /**
     * Generates the HTML for a day selection widget.
     *
     * @param string $name         The name of the widget.
     * @param int    $timestamp    (optional) The timestamp to select by default.
     * @param string $javascript   (optional) Any onchange, onblur, etc. javascript to include.
     *
     * @return string       The HTML <select> widget.
     */
    function buildDayWidget($name, $timestamp = null, $javascript = null)
    {
        $selected = (isset($timestamp)) ? date('d', $timestamp) : null;

        $html = "<select name=\"$name\"";
        if ($javascript != null) {
            $html .= ' ' . $javascript;
        }
        $html .= '>';
        for ($day = 1; $day <= 31; $day++) {
            $html .= "<option value=\"$day\"";
            $html .= ($day == $selected) ? ' selected="selected">' : '>';
            $html .= "$day</option>";
        }
        $html .= "</select>\n";

        return $html;
    }

    /**
     * Generates the HTML for a month selection widget.
     *
     * @param string $name         The name of the widget.
     * @param int    $timestamp    (optional) The timestamp to select by default.
     * @param string $javascript   (optional) Any onchange, onblur, etc. javascript to include.
     *
     * @return string       The HTML <select> widget.
     */
    function buildMonthWidget($name, $timestamp = null, $javascript = null)
    {
        $selected = (isset($timestamp)) ? date('m', $timestamp) : null;

        $html = "<select name=\"$name\"";
        if ($javascript != null) {
            $html .= ' ' . $javascript;
        }
        $html .= '>';
        for ($month = 1; $month <= 12; $month++) {
            $html .= "<option value=\"$month\"";
            $html .= ($month == $selected) ? ' selected="selected">' : '>';
            $html .= strftime('%B', mktime(0, 0, 0, $month)) . '</option>';
        }
        $html .= "</select>\n";

        return $html;
    }

    /**
     * Generates the HTML for a year selection widget.
     *
     * @param int    $name         The name of the widget.
     * @param int    $years        The number of years into the future to include.
     * @param string $timestamp    (optional) The timestamp to select by default.
     * @param string $javascript   (optional) Any onchange, onblur, etc. javascript to include.
     *
     * @return string       The HTML <select> widget.
     */
    function buildYearWidget($name, $years, $timestamp = null, $javascript = null)
    {
        $selected = (isset($timestamp)) ? date('Y', $timestamp) : null;
        $startyear = (isset($selected) && ($selected < date('Y'))) ? $selected : date('Y');

        $html = "<select name=\"$name\"";
        if ($javascript != null) {
            $html .= ' ' . $javascript;
        }
        $html .= '>';
        for ($year = $startyear; $year < (date('Y') + $years); $year++) {
            $html .= "<option value=\"$year\"";
            $html .= ($year == $selected) ? ' selected="selected">' : '>';
            $html .= "$year</option>";
        }
        $html .= "</select>\n";

        return $html;
    }

    /**
     * Generates the HTML for an hour selection widget.
     *
     * @param string $name         The name of the widget.
     * @param int    $timestamp    (optional) The timestamp to select by default.
     * @param string $javascript   (optional) Any onchange, onblur, etc. javascript to include.
     *
     * @return string       The HTML <select> widget.
     */
    function buildHourWidget($name, $timestamp = null, $javascript = null)
    {
        $selected = (isset($timestamp)) ? date('H', $timestamp) : null;

        $html = "<select name=\"$name\"";
        if ($javascript != null) {
            $html .= ' ' . $javascript;
        }
        $html .= '>';
        for ($hour = 0; $hour <= 23; $hour++) {
            $html .= "<option value=\"$hour\"";
            $html .= ($hour == $selected) ? ' selected="selected">' : '>';
            $html .= sprintf("%02d", $hour) . '</option>';
        }
        $html .= "</select>\n";

        return $html;
    }

    /**
     * Generates the HTML for a minute selection widget.
     *
     * @param string   $name         The name of the widget.
     * @param integer  $increment    (optional) The increment between minutes.
     * @param integer  $timestamp    (optional) The timestamp to select by default.
     * @param string   $javascript   (optional) Any onchange, onblur, etc. javascript to include.
     *
     * @return string       The HTML <select> widget.
     */
    function buildMinuteWidget($name, $increment = 1, $timestamp = null, $javascript = null)
    {
        $selected = (isset($timestamp)) ? date('i', $timestamp) : null;

        $html = "<select name=\"$name\"";
        if ($javascript != null) {
            $html .= ' ' . $javascript;
        }
        $html .= '>';
        for ($minute = 0; $minute <= 60; $minute += $increment) {
            $html .= "<option value=\"$minute\"";
            $html .= ($minute == $selected) ? ' selected="selected">' : '>';
            $html .= sprintf("%02d", $minute) . '</option>';
        }
        $html .= "</select>\n";

        return $html;
    }

    /**
     * Builds the HTML for a priority selection widget.
     *
     * @param string  $name         The name of the widget.
     * @param integer $selected     (optional) The default selected priority.
     *
     * @return string       The HTML <select> widget.
     */
    function buildPriorityWidget($name, $selected = -1)
    {
        $html = "<select name=\"$name\">";

        for ($priority = 1; $priority <= 5; $priority++) {
            $html .= "<option value=\"$priority\"";
            $html .= ($priority == $selected) ? ' selected="selected">' : '>';
            $html .= $priority . '</option>';
        }
        $html .= "</select>\n";

        return $html;
    }

    /**                                                                        
     * Builds the HTML for a task completion state widget.
     *
     * @param string  $name         The name of the widget.
     * @param integer $selected     (optional) The default completion state.
     *
     * @return string       The HTML <select> widget.
     */
    function buildCompletionWidget($name, $selected = 0)
    {
        global $completion_map;

        $html = "<select name=\"$name\">";

        foreach ($completion_map as $completed => $name) {
            $html .= "<option value=\"$completed\"";
            $html .= ($completed == $selected) ? ' selected="selected">' : '>';
            $html .= $name . '</option>';
        }
        $html .= "</select>\n";

        return $html;
    }

    /**                                                                        
     * Builds the HTML for a task category widget.
     *
     * @param string  $name         The name of the widget.
     * @param integer $selected     (optional) The default category.
     *
     * @return string       The HTML <select> widget.
     */
    function buildCategoryWidget($name, $selected = 0)
    {
        $html = "<select name=\"$name\">";

        foreach (Nag::listCategories() as $id => $name) {
            $html .= "<option value=\"$id\"";
            $html .= ($id == $selected) ? ' selected="selected">' : '>';
            $html .= $name . '</option>';
        }
        $html .= '</select>';

        return $html;
    }

    /**
     * Formats the given Unix-style date string.
     *
     * @param string $unixdate     The Unix-style date value to format.
     *
     * @return string       The formatted due date string.
     */
    function formatDate($unixdate = '')
    {
        global $conf;

        if (empty($unixdate)) {
            return '';
        }

        $s  = strftime($conf['view']['date_format'], $unixdate);
        $s .= ' ' . _("at") . ' ';
        $s .= strftime($conf['view']['time_format'], $unixdate);

        if ($unixdate < time()) {
            $s = '<span class="overdue">' . $s . '</span>';
        }

        return $s;
    }

    /**
     * Returns the string matching the given category ID.
     *
     * @param integer $categoryID     The category ID to look up.
     *
     * @return string       The formatted category string.
     */
    function formatCategory($categoryID = 0)
    {
        $categories = Nag::listCategories();
        return isset($categories[$categoryID]) ?
            $categories[$categoryID] :
            $categories[0];
    }

    /**
     * Returns the string representation of the given completion status.
     *
     * @param int $completed    The completion value.
     *
     * @return string       The string representation of $completed.
     */
    function formatCompletion($completed)
    {
        global $completion_map;

        if (isset($completion_map[$completed])) {
            return $completion_map[$completed];
        }

        return _("Unknown");
    }
}

/**
 * Comparison function for sorting tasks by priority.
 *
 * @param array $a  Task one.
 * @param array $b  Task two.
 *
 * @return integer  1 if task one is greater, -1 if task two is greater; 0 if they are equal.
 */
function _sortByPriority($a, $b)
{
    if ($a['priority'] == $b['priority']) return 0;
    return ($a['priority'] > $b['priority']) ? 1 : -1;
}

/**
 * Comparison function for reverse sorting tasks by priority.
 *
 * @param array $a  Task one.
 * @param array $b  Task two.
 *
 * @return integer  -1 if task one is greater, 1 if task two is greater; 0 if they are equal.
 */
function _rsortByPriority($a, $b)
{
    if ($a['priority'] == $b['priority']) return 0;
    return ($a['priority'] < $b['priority']) ? -1 : 1;
}

/**
 * Comparison function for sorting tasks by name.
 *
 * @param array $a  Task one.
 * @param array $b  Task two.
 *
 * @return integer  1 if task one is greater, -1 if task two is greater; 0 if they are equal.
 */
function _sortByName($a, $b)
{
    return strcmp($a['name'], $b['name']);
}

/**
 * Comparison function for reverse sorting tasks by name.
 *
 * @param array $a  Task one.
 * @param array $b  Task two.
 *
 * @return integer  -1 if task one is greater, 1 if task two is greater; 0 if they are equal.
 */
function _rsortByName($a, $b)
{
    return strcmp($b['name'], $a['name']);
}

/**
 * Comparison function for sorting tasks by category.
 *
 * @param array $a  Task one.
 * @param array $b  Task two.
 *
 * @return integer  1 if task one is greater, -1 if task two is greater; 0 if they are equal.
 */
function _sortByCategory($a, $b)
{
    return strcmp($a['category'], $b['category']);
}

/**
 * Comparison function for reverse sorting tasks by category.
 *
 * @param array $a  Task one.
 * @param array $b  Task two.
 *
 * @return integer  -1 if task one is greater, 1 if task two is greater; 0 if they are equal.
 */
function _rsortByCategory($a, $b)
{
    return strcmp($b['category'], $a['category']);
}

/**
 * Comparison function for sorting tasks by due date.
 *
 * @param array $a  Task one.
 * @param array $b  Task two.
 *
 * @return integer  1 if task one is greater, -1 if task two is greater; 0 if they are equal.
 */
function _sortByDue($a, $b)
{
    if ($a['due'] == $b['due']) return 0;
    return ($a['due'] > $b['due']) ? -1 : 1;
}

/**
 * Comparison function for reverse sorting tasks by due date.
 *
 * @param array $a  Task one.
 * @param array $b  Task two.
 *
 * @return integer  -1 if task one is greater, 1 if task two is greater; 0 if they are equal.
 */
function _rsortByDue($a, $b)
{
    if ($a['due'] == $b['due']) return 0;
    return ($a['due'] < $b['due']) ? -1 : 1;
}

/**
 * Comparison function for sorting tasks by completion status.
 *
 * @param array $a  Task one.
 * @param array $b  Task two.
 *
 * @return integer  1 if task one is greater, -1 if task two is greater; 0 if they are equal.
 */
function _sortByCompletion($a, $b)
{
    if ($a['completed'] == $b['completed']) return 0;
    return ($a['completed'] > $b['completed']) ? -1 : 1;
}

/**
 * Comparison function for reverse sorting tasks by completion status.
 *
 * @param array $a  Task one.
 * @param array $b  Task two.
 *
 * @return integer  -1 if task one is greater, 1 if task two is greater; 0 if they are equal.
 */
function _rsortByCompletion($a, $b)
{
    if ($a['completed'] == $b['completed']) return 0;
    return ($a['completed'] < $b['completed']) ? -1 : 1;
}

?>
