/* Copyright (C) 2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/**
 * @file myx_gc_view.cpp 
 * @brief Implementation of the view class.
 * 
 */

#include "myx_gc_canvas.h"
#include "myx_gc_feedback.h"

//----------------- CFigureInstanceEnumerator --------------------------------------------------------------------------

/**
 * Constructor of the enumerator class.
 *
 * @param view The view which contains the layers which are to be enumerated.
 */
CFigureInstanceEnumerator::CFigureInstanceEnumerator(CGCView* view)
{
  FView = view;
  reset();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Determines if there is a next figure instance to enumerate.
 *
 * @return True if there is still a figure instance otherwise false.
 */
bool CFigureInstanceEnumerator::hasNext(void)
{
  return FLayerIterator != FView->FLayers.end();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Returns the next figure instance in the sequence.
 *
 * @return The next figure instance.
 */
CFigureInstance* CFigureInstanceEnumerator::next(void)
{
  CFigureInstance* result = *FFigureInstanceIterator;

  // Advance to next instance.
  ++FFigureInstanceIterator;
  while (true)
  {
    CLayer* layer = *FLayerIterator;
    if (FFigureInstanceIterator != layer->FInstances.end())
      break;

    // This layer is exhausted. Get the next one.
    ++FLayerIterator;
    if (FLayerIterator == FView->FLayers.end())
    {
      // No more layers.
      break;
    };

    if (!(*FLayerIterator)->FEnabled)
      FFigureInstanceIterator = (*FLayerIterator)->FInstances.end();
  };

  return result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Frees this enumerator instance. Usually called by non-C++ languages as memory is managed by the C++ runtime.
 */
void CFigureInstanceEnumerator::release(void)
{
  delete this;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Resets the enumerator to the first figure instance in the canvas.
 */
void CFigureInstanceEnumerator::reset(void)
{
  FLayerIterator = FView->FLayers.begin();
  while (FLayerIterator != FView->FLayers.end())
  {
    if ((*FLayerIterator)->FEnabled)
    {
      CLayer* layer = *FLayerIterator;
      FFigureInstanceIterator = layer->FInstances.begin();
      if (FFigureInstanceIterator == layer->FInstances.end())
      {
        ++FLayerIterator;
      }
      else
        break; // Found the first instance.
    }
    else
      ++FLayerIterator;
  };
}

//----------------- CLayerListener -----------------------------------------------------------------------------

void CGCView::CLayerListener::onAction(CGCBase* sender, CGCBase* origin, TActionType& actionType)
{
  view->action(origin, actionType);
}

//----------------------------------------------------------------------------------------------------------------------

void CGCView::CLayerListener::onChange(CGCBase* sender, CGCBase* origin, TGCChangeReason reason)
{
  view->handleChange(origin, reason);
};

//----------------------------------------------------------------------------------------------------------------------

void CGCView::CLayerListener::onDestroy(CGCBase* object)
{
  view->removeLayer((CLayer*) object);
}

//----------------------------------------------------------------------------------------------------------------------

void CGCView::CLayerListener::onError(CGCBase* sender, CGCBase* origin, const char* message)
{
  view->error(origin, message);
}

//----------------- CGCView --------------------------------------------------------------------------------------------

CGCView::CGCView(CGenericCanvas* canvas, string name): CGCBase(canvas)
{
  _className = "CGCView";
  FName = utf8ToUtf16(name);
  FColor[0] = 0;
  FColor[1] = 0;
  FColor[2] = 0;
  FColor[3] = 1;
  FZoom = 1;
  FOffsetX = 0;
  FOffsetY = 0;
  FWorkspaceWidth = 1000;
  FWorkspaceHeight = 1000;
  FNearPlane = -100;
  FFarPlane = 100;
  FJitter = 0;
  FListener.view = this;
  FStates = 0;
  FOrigin[0] = 0;
  FOrigin[1] = 0;
  FOrigin[2] = 0;

  FGrid = new CGridLayer(this);
  FFeedbackLayer = new CFeedbackLayer(this);
  FFeedbackLayer->addListener(&FListener);
  FConnectionLayer = new CConnectionLayer(this);
  FConnectionLayer->addListener(&FListener);
  FExposeFloater = NULL;
  FPrimitiveLayer = new CPrimitiveLayer(canvas);
  FCache = NULL;
}

//----------------------------------------------------------------------------------------------------------------------

CGCView::~CGCView()
{
  delete FCache;
  FCache = NULL;

  for (CLayers::const_iterator iterator = FLayers.begin(); iterator != FLayers.end(); ++iterator)
    (*iterator)->removeListener(&FListener);

  delete FGrid;
  delete FFeedbackLayer;
  delete FConnectionLayer;
  delete FPrimitiveLayer;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Used to set up things that need only to be done once if a view becomes active.
 */
void CGCView::activate(void)
{
  glClearColor(FColor[0], FColor[1], FColor[2], FColor[3]);

  // TODO: make this dynamically configurable (where applicable).
  glFrontFace(GL_CW);
  glDisable(GL_AUTO_NORMAL);
  glDisable(GL_CULL_FACE);
  glDisable(GL_DITHER);
  glDisable(GL_DEPTH_TEST);

  // The following is only useful if a multisampling pixel format was set up.
  if (canvas()->supportsExtension(GC_OE_MULTISAMPLING))
    glEnable(GL_MULTISAMPLE_ARB); 
  
  // Line and polygon smoothing is done via the multisample extension (if existant).
  glDisable(GL_LINE_SMOOTH);
  //glHint(GL_LINE_SMOOTH_HINT, GL_NICEST);
  glDisable(GL_POLYGON_SMOOTH);
  //glHint(GL_POLYGON_SMOOTH_HINT, GL_NICEST);
  
  glEnable(GL_BLEND);
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);

  glDisable(GL_FOG);
  glDisable(GL_LIGHTING);
  glDisable(GL_LOGIC_OP);
  glDisable(GL_STENCIL_TEST);
  glDisable(GL_TEXTURE_1D);
  glDisable(GL_TEXTURE_2D);
  
  // Disable extensions that could slow down glDrawPixels.
  if (canvas()->supportsExtension(GC_OE_CONVOLUTION))
  {
    // A recent glext.h include file is required for these constants.
    // Get one from www.opengl.org.
#ifdef GL_VERSION_1_3
    glDisable(GL_CONVOLUTION_1D);
    glDisable(GL_CONVOLUTION_2D);
    glDisable(GL_SEPARABLE_2D);
#else
    glDisable(GL_CONVOLUTION_1D_EXT);
    glDisable(GL_CONVOLUTION_2D_EXT);
    glDisable(GL_SEPARABLE_2D_EXT);
#endif
  };

  if (canvas()->supportsExtension(GC_OE_HISTOGRAM))
  {
#ifdef GL_VERSION_1_3
    glDisable(GL_HISTOGRAM);
    glDisable(GL_MINMAX);
#else
    glDisable(GL_HISTOGRAM_EXT);
    glDisable(GL_MINMAX_EXT);
#endif
  };

  if (canvas()->supportsExtension(GC_OE_TEXTURE3D))
  {
#ifdef GL_VERSION_1_3
    glDisable(GL_TEXTURE_3D);
#else
    glDisable(GL_TEXTURE_3D_EXT);
#endif
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Sets up the current projection and modelview matrices.
 *
 * @param doProjection A flag telling if also the project matrix should be set. Useful to avoid unnecessary applications.
 */
void CGCView::applyTransformations(bool doProjection)
{
  if (doProjection)
  {
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    glOrtho(-FJitter, FViewport.width - FJitter, FViewport.height - FJitter, -FJitter, FNearPlane, FFarPlane);
  };

  glViewport(FViewport.left, FViewport.top, FViewport.width, FViewport.height);

  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();
  glTranslated(FOffsetX, FOffsetY, 0);
  glScaled(FZoom, FZoom, 1);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Takes care for certain change events that must be considered special.
 */
void CGCView::handleChange(CGCBase* origin, TGCChangeReason reason)
{
  if (!destroying())
  {
    switch (reason)
    {
      case GC_CHANGE_LAYER_ADD_INSTANCE:
      case GC_CHANGE_LAYER_CLEAR:
      case GC_CHANGE_LAYER_REMOVE_INSTANCE:
      case GC_CHANGE_LAYER_ADD_GROUP:
      case GC_CHANGE_LAYER_REMOVE_GROUP:
        {
          if (FCache != NULL)
            FCache->invalidate();

          break;
        };
      default:
        change(origin, reason);
    };
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Main handler routine for mouse button down handling.
 *
 * @param button Which button has been pressed (left, middle, right).
 * @param modifiers Special flags that control the processing.
 * @param windowX Horizontal mouse coordinate in window space.
 * @param windowY Vertical mouse coordinate in window space.
 * @param viewCoords Mouse coordinates converted to view space.
 * @return True if the input was handled, otherwise false.
 */
bool CGCView::handleMouseDown(TMouseButton button, int modifiers, int windowX, int windowY, TVertex& viewCoords)
{
  bool result = false;
  TActionType lastAction = GC_ACTION_NONE;

  switch (button)
  {
    case GC_MOUSE_BUTTON_LEFT:
      {
        FStates |= GC_STATE_LBUTTON_DOWN;
        break;
      };
    case GC_MOUSE_BUTTON_MIDDLE:
      {
        FStates |= GC_STATE_MBUTTON_DOWN;
        break;
      };
    case GC_MOUSE_BUTTON_RIGHT:
      {
        FStates |= GC_STATE_RBUTTON_DOWN;
        break;
      };
  };

  if (button == GC_MOUSE_BUTTON_LEFT)
  {
    // First check special layers with interactive objects.
    CFigureInstance* instance;
    TFeedbackInfo feedbackInfo = FFeedbackLayer->getFeedbackInfo(viewCoords, FZoom, &instance);
    if (feedbackInfo == GC_FI_FLOATER)
    {
      lastAction = GC_ACTION_FLOATER_DRAG;
      action(this, lastAction);
      if (lastAction == GC_ACTION_FLOATER_DRAG)
      {
        FStates |= GC_STATE_OVERVIEW_DRAG;
        FLastWindowX = windowX;
        FLastWindowY= windowY;
        FLastViewX = viewCoords.x;
        FLastViewY = viewCoords.y;
        result = true;
      };
    };

    if (feedbackInfo == GC_FI_NONE)
      result = FConnectionLayer->handleMouseDown(button, modifiers, windowX, windowY, viewCoords);

    if (!result)
    {
      // Determine if an element is hit.
      updateCache();

      CGraphicElement* hit = FCache->findElement(viewCoords);
      // If an element was hit for which we have no further feedback info then just set the feedback to "on body".
      if (hit != NULL && feedbackInfo == GC_FI_NONE)
        feedbackInfo = GC_FI_ON_OBJECT;

      switch (feedbackInfo)
      {
        case GC_FI_SELECTION_BODY:
        case GC_FI_ON_OBJECT:
          {
            // Since there is a small gap between object and selection decoration
            // hit can be NULL.
            if (hit != NULL)
            {
              instance = (CFigureInstance*) hit;
              lastAction = instance->executeAssociatedActions(viewCoords);
              if (lastAction == GC_ACTION_DRAG_FIGURE)
              {
                FStates |= GC_STATE_DRAG_PENDING;
                if (modifiers == 0)
                  FStates |= GC_STATE_CLEAR_PENDING;
                else
                  if ((modifiers & GC_MODIFIER_ADD) != 0)
                    FFeedbackLayer->addToSelection(instance);
                  else
                    if ((modifiers & GC_MODIFIER_TOGGLE) != 0)
                      if (instance->selected())
                        FFeedbackLayer->removeFromSelection(instance);
                      else
                        FFeedbackLayer->addToSelection(instance);

                // If the hit is not yet selected then drag only this one instance,
                // otherwise the entire selection will be dragged.
                FDragSelection = instance->selected();
                FLastHit = instance;
                FLastWindowX = windowX;
                FLastWindowY= windowY;
                FLastViewX = viewCoords.x;
                FLastViewY = viewCoords.y;
              };
            };
            break;
          };
        case GC_FI_RESIZE_NORTH:
        case GC_FI_RESIZE_NORTH_EAST:
        case GC_FI_RESIZE_EAST:
        case GC_FI_RESIZE_SOUTH_EAST:
        case GC_FI_RESIZE_SOUTH:
        case GC_FI_RESIZE_SOUTH_WEST:
        case GC_FI_RESIZE_WEST:
        case GC_FI_RESIZE_NORTH_WEST:
          {
            lastAction = GC_ACTION_RESIZE;
            action(this, lastAction);
            if (lastAction == GC_ACTION_RESIZE)
            {
              FStates |= GC_STATE_RESIZING;
              FResizeInfo = feedbackInfo;
              FLastHit = instance;
              FLastWindowX = windowX;
              FLastWindowY= windowY;
              FLastViewX = viewCoords.x;
              FLastViewY = viewCoords.y;
            };
            break;
          };
      };
    };

    if (!result && lastAction == GC_ACTION_NONE)
    {
      // Nothing happend so far so start the rubber rectangle if allowed.
      lastAction = GC_ACTION_RUBBER_RECT;
      action(this, lastAction);
      if (lastAction = GC_ACTION_RUBBER_RECT)
      {
        rubberRectStart(GC_RRSTYLE_BLENDED_DIAGONALS, viewCoords, modifiers == 0);
        FStates |= GC_STATE_SELECTION_RECTANGLE;
        result = true;
      };
    };
  };

  return result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Main handler routine for mouse button move handling.
 *
 * @param modifiers Special flags that control the processing.
 * @param windowX Horizontal mouse coordinate in window space.
 * @param windowY Vertical mouse coordinate in window space.
 * @param viewCoords Mouse coordinates converted to view space.
 * @return True if the input was handled, otherwise false.
 */
bool CGCView::handleMouseMove(int modifiers, int windowX, int windowY, TVertex& viewCoords)
{
  bool result = false;

  if ((FStates & GC_STATE_RUBBER_RECTANGLE) != 0)
  {
    TRRSelectionAction action = GC_RRACTION_NONE;
    if ((FStates & GC_STATE_SELECTION_RECTANGLE) != 0)
    {
      action = GC_RRACTION_SELECT_REMOVE;
      if ((modifiers & GC_MODIFIER_ADD) != 0)
        action = GC_RRACTION_SELECT; // Always-add has higher prio than toggle.
      else
        if ((modifiers & GC_MODIFIER_TOGGLE) != 0)
          action = GC_RRACTION_TOGGLE;
    };

    FFeedbackLayer->rubberRectResize(viewCoords, action);
    result = true;
  };

  if ((FStates & GC_STATE_RUBBER_BAND) != 0)
  {
    FFeedbackLayer->rubberBandResize(viewCoords);
    result = true;
  };

  if ((FStates & GC_STATE_DRAG_PENDING) != 0)
  {
    // Drag is still waiting to start. Check if the user moved the mouse a bit.
    float threshold = 4; // Empirically found.
    if (abs(FLastWindowX - windowX) >= threshold || abs(FLastWindowY - windowY) >= threshold)
    {
      FStates &= ~(GC_STATE_CLEAR_PENDING | GC_STATE_DRAG_PENDING);
      FStates |= GC_STATE_DRAGGING;
    };
  };

  // Now the moved distance becomes important.
  float dX = viewCoords.x - FLastViewX;
  float dY = viewCoords.y - FLastViewY;
  FLastViewX = viewCoords.x;
  FLastViewY = viewCoords.y;
  
  if ((FStates & GC_STATE_DRAGGING) != 0)
  {
    if (FDragSelection)
      FFeedbackLayer->moveSelectedInstances(dX, dY, 0, true);
    else
      FLastHit->translate(dX, dY, 0, true);
    FCache->invalidate();
    result = true;
  }
  else
    if ((FStates & GC_STATE_RESIZING) != 0)
    {
      FLastHit->resize(dX, dY, FResizeInfo);
      FCache->invalidate();
      result = true;
    }
    else
      if ((FStates & GC_STATE_OVERVIEW_DRAG) != 0)
      {
        FLastOffsetX -= dX * FLastZoom;
        FLastOffsetY -= dY * FLastZoom;
        FExposeFloater->move(dX, dY);
        result = true;
      };

  if (!result)
    result = FConnectionLayer->handleMouseMove(modifiers, windowX, windowY, viewCoords);

  return result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Main handler routine for mouse button up handling.
 *
 * @param button Which button has been released (left, middle, right).
 * @param modifiers Special flags that control the processing.
 * @param windowX Horizontal mouse coordinate in window space.
 * @param windowY Vertical mouse coordinate in window space.
 * @param viewCoords Mouse coordinates converted to view space.
 * @return True if the input was handled, otherwise false.
 */
bool CGCView::handleMouseUp(TMouseButton button, int modifiers, int windowX, int windowY, TVertex& viewCoords)
{
  bool handled = false;

  if ((FStates & GC_STATE_SELECTION_RECTANGLE) != 0)
  {
    rubberRectStop();
    FStates &= ~GC_STATE_SELECTION_RECTANGLE;
    handled = true;
  };

  if (!handled)
    handled = FConnectionLayer->handleMouseUp(button, modifiers, windowX, windowY, viewCoords);

  // All following processing is always done regardless of the results from the special layers.
  if ((FStates & GC_STATE_CLEAR_PENDING) != 0)
  {
    // A pending clear selection action must be finished. If there is an instance hit from mouse down
    // then select this single instance.
    FStates &= ~GC_STATE_CLEAR_PENDING;
    FFeedbackLayer->clearSelection();
    if (FLastHit != NULL)
      FFeedbackLayer->addToSelection(FLastHit);
  };

  FLastHit = NULL;

  // Remove all states which can be stopped without further action.
  FStates &= ~(GC_STATE_DRAG_PENDING | GC_STATE_DRAGGING | GC_STATE_RESIZING | GC_STATE_OVERVIEW_DRAG);

  switch (button)
  {
    case GC_MOUSE_BUTTON_LEFT:
      {
        FStates &= ~GC_STATE_LBUTTON_DOWN;
        break;
      };
    case GC_MOUSE_BUTTON_MIDDLE:
      {
        FStates &= ~GC_STATE_MBUTTON_DOWN;
        break;
      };
    case GC_MOUSE_BUTTON_RIGHT:
      {
        FStates &= ~GC_STATE_RBUTTON_DOWN;
        break;
      };
  };

  return true;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * This is the main paint routine. It is called by the canvas if this view is the current view.
 */
void CGCView::render(void)
{
  applyTransformations(true);

  // Transform viewport coordinates into view space. They are used for occlusion culling.
  TBoundingBox visibleBounds;
  visibleBounds.upper.x = (FViewport.left - FOffsetX) / FZoom - FOrigin[0];
  visibleBounds.upper.y = (FViewport.top - FOffsetY) / FZoom - FOrigin[1];
  visibleBounds.lower.x = (FViewport.left + FViewport.width - FOffsetX) / FZoom - FOrigin[0];
  visibleBounds.lower.y = (FViewport.top + FViewport.height - FOffsetY) / FZoom - FOrigin[1];

  // TODO: complete origin implementation.
  glTranslated(FOrigin[0], FOrigin[1], FOrigin[2]);

  FGrid->render(FZoom, visibleBounds);

  for (CLayers::iterator iterator = FLayers.begin(); iterator != FLayers.end(); ++iterator)
    (*iterator)->render(FZoom, visibleBounds);

  FPrimitiveLayer->render(FZoom, visibleBounds);

  // A simple frame marks the virtual workspace size.
  glColor3ub(0, 0, 0);
  glBegin(GL_LINE_LOOP); 
    glVertex2f(0, 0);
    glVertex2f(FWorkspaceWidth, 0); 
    glVertex2f(FWorkspaceWidth, FWorkspaceHeight);
    glVertex2f(0, FWorkspaceHeight); 
  glEnd();

  FConnectionLayer->render(FZoom, visibleBounds);
  FFeedbackLayer->render(FZoom, visibleBounds);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Checks if the view's element cache must be created and validated.
 */
void CGCView::updateCache(void)
{
  if (FCache == NULL)
    FCache = new CElementLookupCache(FWorkspaceWidth, FWorkspaceHeight);

  if (!FCache->isValid())
  {
    CFigureInstanceEnumerator* enumerator = getFigureInstanceEnumerator();
    while (enumerator->hasNext())
      FCache->addElement(enumerator->next());
    delete enumerator;

    FCache->validate();
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Adds a layer to the internal list of layers that belong to this view (only if not yet there).
 *
 * @param layer The layer to add.
 */
void CGCView::addLayer(CLayer* layer)
{
  for (CLayers::const_iterator iterator = FLayers.begin(); iterator != FLayers.end(); ++iterator)
    if (*iterator == layer)
      return;

  layer->addListener(&FListener);
  FLayers.push_back(layer);
  
  change(this, GC_CHANGE_VIEW_ADD_LAYER);
  canvas()->refresh();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Adds the given instance to the current selection if it isn't already.
 *
 * @param instance The figure instance to add to the current selection. It must belong to one of the layers in this view
 *                 otherwise it will not be added.
 */
void CGCView::addToSelection(CFigureInstance* instance)
{
  if (instance != NULL && !instance->selected())
  {
    CLayer* layer = instance->layer();
    if (layer != NULL)
    {
      CLayers::iterator iterator = find(FLayers.begin(), FLayers.end(), layer);
      if (iterator != FLayers.end())
        FFeedbackLayer->addToSelection(instance);
    };
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Removes the current content of this view.
 *
 * @param removeLayers If true then all currently defined (normal) layers are removed from this view (but not the canvas)
 *                     in addition to being cleared.
 */
void CGCView::clearContent(bool removeLayers)
{
  beginUpdate();

  FCache->invalidate();

  for (CLayers::const_iterator iterator = FLayers.begin(); iterator != FLayers.end(); ++iterator)
    (*iterator)->clear();

  if (removeLayers)
    FLayers.clear();

  endUpdate();
  change(this, GC_CHANGE_VIEW_CLEAR);
  canvas()->refresh();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Removes all primitives.
 */
void CGCView::clearPrimitives(void)
{
  FPrimitiveLayer->clear();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Removes all currently selected figure instances from the selection set.
 */
void CGCView::clearSelection(void)
{
  FFeedbackLayer->clearSelection();
  change(this, GC_CHANGE_SELECTION_CLEAR);
  canvas()->refresh();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Sets the new background color of this view.
 *
 * @param red The red color component.
 * @param green The green color component.
 * @param blue The blue color component.
 * @param alpha The transparency component.
 */
void CGCView::color(float red, float green, float blue, float alpha)
{
  FColor[0] = red;
  FColor[1] = green;
  FColor[2] = blue;
  FColor[3] = alpha;

  change(this, GC_CHANGE_VIEW_PROPERTY);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Sets the new background color of this view
 *
 * @param newColor The new color to use.
 */
void CGCView::color(GLfloat* newColor)
{
  FColor[0] = newColor[0];
  FColor[1] = newColor[1];
  FColor[2] = newColor[2];
  FColor[3] = newColor[3];

  change(this, GC_CHANGE_VIEW_PROPERTY);
  canvas()->refresh();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Tells the caller whether this view contains a reference to the given layer.
 *
 * @param layer The layer to look for.
 * @param True if the layer is referenced in this view otherwise false.
 */
bool CGCView::contains(CLayer* layer)
{
  for (CLayers::const_iterator iterator = FLayers.begin(); iterator != FLayers.end(); ++iterator)
    if (*iterator == layer)
      return true;

  return false;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Creates a circle primitive and returns it.
 *
 * @param center The center position of the circle.
 * @param outerRadius The overall radius of the circle.
 * @param innerRadius The radius of the circle's hole. Can be 0 to not produce that hole.
 * @param color The fill and line color of the primitive.
 * @param filled True if filled, otherwise only the outline will be drawn.
 * @result The newly created primitive.
 */
CPrimitive* CGCView::createCircle(const TVertex& center, float outerRadius, float innerRadius, GLfloat* color, bool filled)
{
  CPrimitive* result = new CCircle(center, outerRadius, innerRadius, color, filled);
  FPrimitiveLayer->addPrimitive(result);

  return result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Proxy function for the connection layer. Read there for a description.
 */
CConnectionInstance* CGCView::createConnectionInstance(CConnection* connection, CFigureInstance* endPoint1, 
                                                       CFigureInstance* endPoint2)
{
  return FConnectionLayer->createInstance(connection, endPoint1, endPoint2);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Create a floater on the feedback layer. Simply forwards the call there.
 */
CFloater* CGCView::createFloater(const TBoundingBox& bounds, GLfloat* color1, GLfloat* color2, TFloaterDecoration decoration)
{
  return FFeedbackLayer->createFloater(bounds, color1, color2, decoration);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Creates a line primitive and returns it.
 *
 * @param start The start coordinate of the line.
 * @param end The end coordinate of the line.
 * @param color The fill and line color of the primitive.
 * @param width The line's thickness.
 * @param stipple If not 0 or 0xFFFF then it determinse the pattern used for the line.
 * @result The newly created primitive.
 */
CPrimitive* CGCView::createLine(const TVertex& start, const TVertex& end, GLfloat* color, GLfloat width, GLushort stipple)
{
  CPrimitive* result = new CLine(start, end, color, width, stipple);
  FPrimitiveLayer->addPrimitive(result);

  return result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Creates a rectangle primitive and returns it.
 *
 * @param bounds The outer bounds of the primitive.
 * @param color The fill and line color of the primitive.
 * @param filled True if filled, otherwise only the outline will be drawn.
 * @result The newly created primitive.
 */
CPrimitive* CGCView::createRectangle(const TBoundingBox& bounds, GLfloat* color, bool filled)
{
  CPrimitive* result = new CRectangle(bounds, color, filled);
  FPrimitiveLayer->addPrimitive(result);

  return result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Determines what action could be executed at the given position. Considered are any feedback state (selection, resize etc.)
 * from the feedback layer as well as actions defined in a figure element. The returned info is usually used to 
 * set an indicator (e.g. the mouse pointer) to a certain state to reflect what is possible at that point.
 *
 * @param windowX The horizontal target position in window coordinates.
 * @param windowY The vertical target position in window coordinate.
 * @return A flag indicating the possible action state.
 */
TFeedbackInfo CGCView::getFeedbackInfo(int windowX, int windowY)
{
  TFeedbackInfo result = GC_FI_NONE;

  updateCache();

  if ((FStates & GC_STATE_RESIZING) != 0)
    result = FResizeInfo;
  else
  {
    TVertex viewCoords; 
    windowToView(windowX, windowY, viewCoords);

    CFigureInstance* instance;
    result = FFeedbackLayer->getFeedbackInfo(viewCoords, FZoom, &instance);
    if (result == GC_FI_NONE)
      result = FConnectionLayer->getFeedbackInfo(viewCoords);
    
    if (result == GC_FI_NONE)
    {
      // If there is no action from one of the special layers then try to get one from the top
      // most figure instance in the top most layer if there is any.
      CGraphicElement* element = FCache->findElement(viewCoords);
      if (element != NULL)
      result = element->getFeedbackInfo(viewCoords);
    };
  };

  return result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Creates and returns a new figure instance enumerator instance. The caller is responsible for freeing
 * the returned instance.
 *
 * @return The new enumerator.
 */
CFigureInstanceEnumerator* CGCView::getFigureInstanceEnumerator(void)
{
  return new CFigureInstanceEnumerator(this);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Takes the given coordinates and tries to find a graphic element that was rendered at this position.
 * Positions must be given in view space.
 *
 * @param point The point to check given in view space. If necessary convert window coordinates first by using windowToView.
 * @param singleHit If true then search for hits is stopped after the first one was found.
 * @return A hit result class is returned regardless of the actual number of hits. It must be freed by the caller.
 */
CHitResults* CGCView::getHitTestInfoAt(TVertex point, bool singleHit)
{
  CHitResults* result = new CHitResults();
                      
  updateCache();

  FConnectionLayer->getHitTestInfoAt(result, point, singleHit);
  if (result->count() == 0)
  {
    if (singleHit)
    {
      CGraphicElement* element = FCache->findElement(point);
      if (element != NULL)
        result->addHit(element);
    }
    else
      FCache->findElements(point, result);
  };

  return result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Returns the current or last set rubber rect/band bounds.
 *
 * @param box [out] Receives the rubber bounds in view coordinates.
 */
void CGCView::getLastRubberBounds(TBoundingBox& box)
{
  box = FFeedbackLayer->rubberBounds();
  box.lower = FFeedbackLayer->layerToView(box.lower);
  box.upper = FFeedbackLayer->layerToView(box.upper);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Returns the current origin of the view.
 *
 * @param x [out] The x coordinate of the virtual origin.
 * @param y [out] The y coordinate of the virtual origin.
 * @param z [out] The z coordinate of the virtual origin.
 */
void CGCView::getOrigin(float* x, float* y, float* z)
{
  *x = FOrigin[0];
  *y = FOrigin[1];
  *z = FOrigin[2];
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Retrieves a selection enumerator from the feedback layer and returns it. The caller is reponsible for freeing
 * the returned instance.
 *
 * @param The new enumerator.
 */
CSelectionEnumerator* CGCView::getSelectionEnumerator(void)
{
  return FFeedbackLayer->getSelectionEnumerator();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Returns the current virtual workspace size within the view.
 *
 * @param width [out] The width of the workspace.
 * @param height [out] The height of the workspace.
 */
void CGCView::getWorkspace(float* width, float* height)
{
  *width = FWorkspaceWidth;
  *height = FWorkspaceHeight;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Called by the viewer to let the current view handle user input with the mouse.
 *
 * @param event The actual event (e.g. mouse down or up).
 * @param button Specifies the mouse button for which this event was triggered.
 * @param modifiers Any combination of TModifier. Specifies further how to handle the input.
 * @param x The horizontal window coordinate of the mouse pointer.
 * @param y The vertical window coordinate of the mouse pointer.
 * @return True if the mouse input was handled in some way, otherwise false.
 */
bool CGCView::handleMouseInput(TMouseEvent event, TMouseButton button, int modifiers, int x, int y)
{
  bool result = false;

  TVertex point;
  windowToView(x, y, point);

  beginUpdate();
  switch (event)
  {
    case GC_MOUSE_DOWN:
      {
        result = handleMouseDown(button, modifiers, x, y, point);
        break;
      };
    case GC_MOUSE_UP:
      {
        result = handleMouseUp(button, modifiers, x, y, point);
        break;
      };
    case GC_MOUSE_MOVE:
      {
        result = handleMouseMove(modifiers, x, y, point);
        break;
      };
  };
  endUpdate();

  return result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Jittering the viewport a little bit sometimes improves display quality (e.g. for thin lines). This function
 * sets this value for this view.
 *
 * @param value The new jitter value.
 */
void CGCView::jitter(float value)
{
  FJitter = value;
  change(this, GC_CHANGE_VIEW_PROPERTY);
  canvas()->refresh();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Returns the current jitter value.
 *
 * @return The current jitter value.
 */
float CGCView::jitter(void)
{
  return FJitter;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Sets the horizontal offset of the view. The offset is value about which the content of the view is moved.
 *
 * @param value The new horizontal offset.
 */
void CGCView::offsetX(float value)
{
  FOffsetX = value;
  canvas()->refresh();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Returns the current horizontal offset.
 *
 * @return The current horizontal offset.
 */
float CGCView::offsetX(void)
{
  return FOffsetX;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Sets the new vertical offset.
 *
 * @param value The new vertical offset.
 */
void CGCView::offsetY(float value)
{
  FOffsetY = value;
  canvas()->refresh();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Returns the current vertical offset.
 *
 * @return The curretn vertical offset.
 */
float CGCView::offsetY(void)
{
  return FOffsetY;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Tells the caller if this view is currently in overview mode.
 *
 * @result True if the overview mode is active.
 */
bool CGCView::overviewActive(void)
{
  return (FStates & GC_STATE_OVERVIEW) != 0;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Starts the overview mode for this view, that is, the view is zoomed out to workspace size and a floater is
 * displayed and can be moved to select a new display area.
 */
void CGCView::overviewStart(void)
{
  if ((FStates & GC_STATE_OVERVIEW) == 0)
  {
    FStates |= GC_STATE_OVERVIEW;
    FLastZoom = FZoom;

    // Convert the coordinates of the current viewport into view space. This is the area to which we return
    // after the overview mode is finished. It is indicated by a special floater on the feedback layer.
    TBoundingBox exposeBox;
    windowToView(FViewport.left, FViewport.top, exposeBox.upper);
    windowToView(FViewport.left + FViewport.width, FViewport.top + FViewport.height, exposeBox.lower);

    GLfloat color1[] = {0.0f, 0.0f, 0.0f, 0.04f};
    GLfloat color2[] = {0.0f, 0.0f, 0.0f, 0.3f};
    FExposeFloater = FFeedbackLayer->createFloater(exposeBox, color1, color2, GC_FLOATER_DECO_CUT_MARKERS);
    FFeedbackLayer->visibleParts(GC_FBPART_ALL);

    // Compute new zoom factor so that the full work area is displayed).
    float zoomX = FViewport.width / FWorkspaceWidth;
    float zoomY = FViewport.height / FWorkspaceHeight;
    FZoom = (zoomX < zoomY) ? zoomX : zoomY;

    FLastOffsetX = FOffsetX;
    FOffsetX = 0;
    FLastOffsetY = FOffsetY;
    FOffsetY = 0;

    change(this, GC_CHANGE_VIEW_OVERVIEW_START);
    canvas()->refresh();
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Stops the overview mode in this view (if it is active) and optionally returns to the last zoom level and display offset.
 *
 * @param overview If true then the view is set to the currently stored zoom level and display offset.
 */
void CGCView::overviewStop(bool returnToZoomed)
{
  if ((FStates & GC_STATE_OVERVIEW) != 0)
  {
    FStates &= ~GC_STATE_OVERVIEW;
    if (returnToZoomed)
    {
      FZoom = FLastZoom;
      FOffsetX = FLastOffsetX;
      FOffsetY = FLastOffsetY;
    };

    delete FExposeFloater;
    FExposeFloater = NULL;
    FFeedbackLayer->visibleParts(GC_FBPART_SELECTION | GC_FBPART_RUBBERRECT | GC_FBPART_RUBBERBAND);

    change(this, GC_CHANGE_VIEW_OVERVIEW_STOP);
    canvas()->refresh();
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Retrieves the value of the property given by path. The path syntax is must be something like (here expressed as regex)
 * (container)*(property), where container is a slash and the name of a container class (e.g. layers, figures) and
 * property is the name of a simple property of that container.
 *
 * @param name The name of the property.
 * @param index If the property is a list then this parameter gives the index into that list.
 * @return A description of the property value and, if the property is simple, the actual value.
 */
TGCVariant CGCView::property(const char* name, unsigned int index)
{
  TGCVariant result;

  switch (getContainerID(name))
  {
    case GC_CONTAINER_UNKNOWN:
      {
        switch (getPropertyID(name))
        {
          case GC_PROPERTY_NAME:
            {
              result = utf16ToUtf8(FName);
              break;
            };
          case GC_PROPERTY_DESCRIPTION:
            {
              result = "A view comprising a set of layers.";
              break;
            };
          case GC_PROPERTY_OWNER:
            {
              result = canvas();
              break;
            };
          case GC_PROPERTY_COLOR:
            {
              result = colorToString(FColor);
              break;
            };
          case GC_PROPERTY_ZOOM:
            {
              result = FZoom;
              break;
            };
          case GC_PROPERTY_X:
            {
              result = FOffsetX;
              break;
            };
          case GC_PROPERTY_Y:
            {
              result = FOffsetY;
              break;
            };
          case GC_PROPERTY_JITTER:
            {
              result = FJitter;
              break;
            };
        };
        break;
      };
    case GC_CONTAINER_LAYERS:
      {
        if (index < FLayers.size())
          result = (CGCBase*) FLayers[index];
        break;
      };
  };

  return result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Sets the value of the given property, which must be a simple property.
 *
 * @param name The name of the property.
 * @param index If the property is a list then this parameter gives the index into that list.
 * @param value The new value of the property. Automatic conversion is performed where possible.
 */
void CGCView::property(const char* name, unsigned int index, TGCVariant value)
{
  switch (getPropertyID(name))
  {
    case GC_PROPERTY_COLOR:
      {
        stringToColor(value, FColor);
        FColor[3] = 1;
        change(this, GC_CHANGE_VIEW_PROPERTY);
        break;
      };
    case GC_PROPERTY_ZOOM:
      {
        FZoom = value;
        change(this, GC_CHANGE_VIEW_PROPERTY);
        break;
      };
    case GC_PROPERTY_X:
      {
        FOffsetX = value;
        change(this, GC_CHANGE_VIEW_PROPERTY);
        break;
      };
    case GC_PROPERTY_Y:
      {
        FOffsetY = value;
        change(this, GC_CHANGE_VIEW_PROPERTY);
        break;
      };
    case GC_PROPERTY_JITTER:
      {
        FJitter = value;
        change(this, GC_CHANGE_VIEW_PROPERTY);
        break;
      };
  };
  canvas()->refresh();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Removes the given instance from the current selection if it selected.
 *
 * @param instance The figure instance to remove. It must belong to one of the layers in this view
 *                 otherwise it will not be removed.
 */
void CGCView::removeFromSelection(CFigureInstance* instance)
{
  if (instance != NULL && !instance->selected())
  {
    CLayer* layer = instance->layer();
    if (layer != NULL)
    {
      CLayers::iterator iterator = find(FLayers.begin(), FLayers.end(), layer);
      if (iterator != FLayers.end())
        FFeedbackLayer->removeFromSelection(instance);
    };
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 *  Removes the given layer from the list of layers that comprise this view.
 *
 * @param layer The layer to remove.
 */
void CGCView::removeLayer(CLayer* layer)
{
  if (!destroying())
  {
    for (CLayers::iterator iterator = FLayers.begin(); iterator != FLayers.end(); ++iterator)
      if (*iterator == layer)
      {
        if (!layer->destroying())
          layer->removeListener(&FListener);
        FLayers.erase(iterator);
        if (FCache)
          FCache->invalidate();

        break;
      };
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Starts the rubber band mode.
 *
 * @param style The visual style of the rubber band.
 * @param coords The start coordinates of the band in view space.
 */
void CGCView::rubberBandStart(TRubberBandStyle style, const TVertex& coords)
{
  // Implizitely stops any active rubber band.
  FFeedbackLayer->rubberBandStart(style, coords);
  FStates |= GC_STATE_RUBBER_BAND;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Stops the current rubber retangle action if there is one.
 */
void CGCView::rubberBandStop(void)
{
  FStates &= ~GC_STATE_RUBBER_BAND;
  FFeedbackLayer->rubberBandStop();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Starts the rubber rectangle mode.
 *
 * @param style The visual style of the rubber rectangle.
 * @param coords The start coordinates of the rectangle in view space.
 * @param removeSelection If true then the current selection is cleared.
 */
void CGCView::rubberRectStart(TRubberRectStyle style, const TVertex& coords, bool removeSelection)
{
  // Implizitely stops any active rubber rectangle.
  FFeedbackLayer->rubberRectStart(style, coords, removeSelection);
  FStates |= GC_STATE_RUBBER_RECTANGLE;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Stops the current rubber retangle action if there is one.
 */
void CGCView::rubberRectStop(void)
{
  FStates &= ~GC_STATE_RUBBER_RECTANGLE;
  FFeedbackLayer->rubberRectStop();
}

//----------------------------------------------------------------------------------------------------------------------

bool CGCView::saveAsPNG(const TGCViewport& viewport, const char *filename)
{
  unsigned char *pixels;
  
  pixels= (unsigned char*)malloc(viewport.width * viewport.height * 3);
  if (pixels)
  {
    TImage image;
    bool rc;
    
    glReadPixels(viewport.left, viewport.top, viewport.width, viewport.height,
                 GL_RGB, GL_UNSIGNED_BYTE, pixels);
    
    image.width= viewport.width;
    image.height= viewport.height;
    image.data= pixels;
    
    rc= savePNG(utf8ToUtf16(filename), &image, true);
    
    free(pixels);

    return rc;
  }
  return false;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Sets the virtual center of the view.
 *
 * @param x The x coordinate of the new center.
 * @param y The x coordinate of the new center.
 * @param z The x coordinate of the new center.
 */
void CGCView::setOrigin(float x, float y, float z)
{
  FOrigin[0] = x;
  FOrigin[1] = y;
  FOrigin[2] = z;

  canvas()->refresh();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Sets a new size for the current virtual workspace in this view.
 *
 * @param width The new workspace width.
 * @param height The new workspace height.
 */
void CGCView::setWorkspace(const float width, const float height)
{
  FWorkspaceWidth = width;
  FWorkspaceHeight = height;

  FConnectionLayer->workspaceChanged();

  // The workspace size may also directly influence the cache.
  #if USE_BSP_TREE
    delete FCache;
    FCache = NULL;
  #endif
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Hides or shows the current selection.
 *
 * @param visible If true the selection is shown, otherwise not.
 */
void CGCView::showSelection(bool visible)
{
  FFeedbackLayer->visible(visible);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Sets the new viewport for this view.
 *
 * @param NewViewport The new viewport to be used.
 */
void CGCView::viewport(const TGCViewport& newViewport)
{
  FViewport = newViewport;
  glViewport(FViewport.left, FViewport.top, FViewport.width, FViewport.height);
  change(this, GC_CHANGE_VIEW_PROPERTY);
  canvas()->refresh();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Converts the given window (viewer) coordinates into local (view) space.
 * @note This function resets the current projection and view matrices.
 *
 * @param x Horizontal window coordinate in pixels.
 * @param y Vertical window coordinate in pixels.
 * @param coords [out] A vertex getting the local coordiantes.
 */
void CGCView::windowToView(int x, int y, TVertex& coords)
{
  applyTransformations(true);

  GLint viewport[4];
  glGetIntegerv(GL_VIEWPORT, viewport);

  GLdouble modelviewMatrix[16];
  glGetDoublev(GL_MODELVIEW_MATRIX, modelviewMatrix);

  GLdouble projectionMatrix[16];
  glGetDoublev(GL_PROJECTION_MATRIX, projectionMatrix);

  double localX, localY, localZ;
  gluUnProject(x, viewport[3] - y, 0, modelviewMatrix, projectionMatrix, viewport, &localX, &localY, &localZ);

  coords = TVertex(localX - FOrigin[0], localY - FOrigin[1], localZ - FOrigin[2]);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Sets a new zoom factor.
 *
 * @param value The new zoom factor.
 */
void CGCView::zoom(float value)
{
  FZoom = value;
  change(this, GC_CHANGE_VIEW_PROPERTY);
  canvas()->refresh();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Returns the current zoom factor.
 *
 * @return The current zoom factor.
 */
float CGCView::zoom(void)
{
  return FZoom;
}

//----------------------------------------------------------------------------------------------------------------------

