/* Copyright (C) 2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/**
 * @file myx_gc_utilities.h 
 * @brief Some common utility functions.
 * 
 */

#ifndef __GC_GL_UTILITIES_H__
#define __GC_GL_UTILITIES_H__

#include <png.h>

#include "myx_gc_datatypes.h"

//----------------------------------------------------------------------------------------------------------------------

typedef enum tagColorType
{
  COLOR_TYPE_PALETTE =     PNG_COLOR_TYPE_PALETTE,
  COLOR_TYPE_GRAY =        PNG_COLOR_TYPE_GRAY,
  COLOR_TYPE_GRAY_ALPHA =  PNG_COLOR_TYPE_GRAY_ALPHA,
  COLOR_TYPE_RGB =         PNG_COLOR_TYPE_RGB,
  COLOR_TYPE_RGB_ALPHA =   PNG_COLOR_TYPE_RGB_ALPHA
} TColorType;

typedef struct tagImage
{
  unsigned int width;     // The width of the image in pixels.
  unsigned int height;    // The height of the image in pixels.
  unsigned char* data;    // The image data.
  TColorType colorType;    // The color format of the image (RGB, gray, palette etc.)
                           // Note: Palette images are not supported.
  unsigned int channels;  // Bytes per pixel.
  GLenum format;           // OpenGL color format specifier. Set by the image user.
} TImage;

//----------------------------------------------------------------------------------------------------------------------

/** Converts the given string, which is supposed to be an UTF-8 encoded text into an UTF-16 string. */
GENERIC_CANVAS_API wstring utf8ToUtf16(const string& Source);

/** Converts the given UTF-16 string into an UTF-8 string. */
GENERIC_CANVAS_API string utf16ToUtf8(const wstring& Source);

/** Converts the given string into an ANSI string using the current system locale. */
GENERIC_CANVAS_API string utf16ToANSI(const wstring& Source);

/** Converts the given string, which is supposed to be an UTF-8 encoded text into an ANSI string using the current system locale. */
GENERIC_CANVAS_API string utf8ToANSI(const string& Source);

/** Saves the given image as a PNG image. */
GENERIC_CANVAS_API bool savePNG(const wstring& filename, TImage *image, bool flipped);

/** Loads the given PNG image from disk. */
GENERIC_CANVAS_API TImage* loadPNG(const wstring& filename);

/** Releases the given image. */
GENERIC_CANVAS_API void freeImage(TImage* image);

/** Returns the current working folder. */
GENERIC_CANVAS_API string getCurrentDir(void);

/** Sets the current working folder. */
GENERIC_CANVAS_API void setCurrentDir(const string& folder);

/** Extracts the drive and path from the given file name. */
GENERIC_CANVAS_API string extractFilePath(const string& filename);

/** Platform neutral file open function. */
FILE* openFile(string filename, const char* openMode);
FILE* openFile(wstring filename, const wchar_t* openMode);

/** Read a text file into memory and perform $variable$ substitutions */
char *preprocessFile(const string &filename, map<string,string>& variables, size_t &size);

/** Sorts the given bounds so that left <= right and bottom <= top. */
TBoundingBox sortBounds(const TBoundingBox& bounds);

/** Determines whether both bounds overlap. */
bool boundsIntersect(const TBoundingBox& bounds1, const TBoundingBox& bounds2);

/** Determines whether both the second bounds are within the first bounds. */
bool boundsContainBounds(const TBoundingBox& bounds1, const TBoundingBox& bounds2);

/** Determines whether bounds are empty. */
bool boundsAreEmpty(const TBoundingBox& bounds);

/** Checks if a given point is within the given bounds. */
bool boundsContainPoint(const TBoundingBox& bounds, const float x, const float y);

/** Converts a container name to an identifier suitable for quick lookup. */
GENERIC_CANVAS_API TContainerID getContainerID(const string& container);

/** Returns an identifier for a given property name. */
GENERIC_CANVAS_API TPropertyID getPropertyID(const string& property);

/** Returns the index value for a given property. */
int getEntryIndex(string& Path);

/** Converts a (byte) color to a string. */
string colorToString(GLubyte* color);

/** Converts a (float) color to a string. */
string colorToString(GLfloat* color);

// Adds colors to the named color table.
void registerSystemColors(const CColorMap& colorMap);

// Find a color by name.
bool colorByName(string name, GLubyte* color);

/** Converts a string to color with byte members. */
int stringToColor(string ColorString, GLubyte* color);

/** Converts a string to color with float members. */
int stringToColor(string ColorString, GLfloat* color);

/** matrix code */
void matrixMultiply(TMatrix product, const TMatrix a, const TMatrix b);
void matrixRotate(TMatrix mat, GLfloat angle, GLfloat x, GLfloat y, GLfloat z);
void matrixScale(TMatrix mat, GLfloat x, GLfloat y, GLfloat z);
void matrixTranslate(TMatrix mat, GLfloat x, GLfloat y, GLfloat z);
TVertex matrixTransform(TMatrix m, const TVertex& v);
bool matrixInvert(const TMatrix& input, TMatrix& output);

//----------------------------------------------------------------------------------------------------------------------

/**
 * The bounding box computer is a neat little helper class to construct a final bound box out of an arbitrary number of
 * other boxes as well as (lists of) points and vertices.
 */
class CBoundingBoxComputer
{
private:
  TBoundingBox FCurrentBox;
  bool FEmpty;
protected:
  void includeVertex(const TVertex& vertex);
public:
  CBoundingBoxComputer(void);
  CBoundingBoxComputer(const TBoundingBox& initialBox);

  TBoundingBox boundingBox(void);
  void include(TMatrix matrix, const TBoundingBox& newBox);
  void include(const TVertex& offset, const TBoundingBox& newBox);
  void include(TMatrix matrix, const TVertex& vertex);
  void include(TMatrix matrix, const float& x, const float& y);
  void reset(void) { FEmpty = true; };
};
 
//----------------------------------------------------------------------------------------------------------------------

/**
 * Simple tokenizer class that works similar as Java's StringTokenizer.
 */
class StringTokenizer
{
  string FInput, FDelimiters;
  string::size_type FPosition;
  char FLastDelimiter;
public:
  StringTokenizer(string text, string delimiters): FInput(text), FDelimiters(delimiters)
  {
    FPosition = FInput.find_first_not_of(FDelimiters);
    FLastDelimiter = '\0';
  };
  
  bool hasMoreTokens()
  {
    return FPosition != string::npos;
  };
  
  char lastDelimiter(void) { return FLastDelimiter; };

  string nextToken()
  {
    string::size_type endPosition = FInput.find_first_of(FDelimiters, FPosition);
    if (FPosition == string::npos)
      FLastDelimiter = '\0';
    else
      FLastDelimiter = FInput[endPosition];
    string token = FInput.substr(FPosition, endPosition - FPosition);
    FPosition = FInput.find_first_not_of(FDelimiters, endPosition);

    return token;
  };

  float nextTokenAsFloat()
  {
    return (float) atof(nextToken().c_str());
  };

  int nextTokenAsInt()
  {
    return atoi(nextToken().c_str());
  };
};

//----------------------------------------------------------------------------------------------------------------------

#endif // __GC_GL_UTILITIES_H__
