/* Copyright (C) 2005 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

#include "MWReverseEngineering.h"

#include "MGGladeXML.h"
#include "myg_gtkutils.h"
#include "mywb.h"

#include <MySQLGRT/MGRTRevEngFilterPane.h>

/**
 * @file  MWReverseEngineering.cc
 * @brief 
 */

void MWReverseEngineering::update_advanced()
{
  MGRTWizardBase::update_advanced();
  
  bool has_details= false;
  Gtk::Widget *frame= _xml->get_widget("log_frame");
  switch (_section)
  {
  case 0: has_details= true; break;
  case 1:
    has_details= true;
    frame->reference();
    frame->get_parent()->remove(*frame);
    _xml->get_box("fetch_box")->pack_start(*frame);
    frame->unreference();
    break;
  case 3:
    has_details= true;
    frame->reference();
    frame->get_parent()->remove(*frame);
    _xml->get_box("reveng_box")->pack_start(*frame);
    frame->unreference();
    break;
  case 6:
    has_details= true;
    frame->reference();
    frame->get_parent()->remove(*frame);
    _xml->get_box("finish_box")->pack_start(*frame);
    frame->unreference();
    break;
  }
  _xml->get_button("show_more_button")->set_sensitive(has_details);
  
  if (!_advanced_shown)
    frame->hide();
  else
    frame->show();
}


void MWReverseEngineering::update_section(bool back)
{
  MGRTWizardBase::update_section(back);

  switch (_section)
  {
  case 0:
    _xml->get_label("top_label")->set_markup(_("<b>Source Connection</b>\nPlease specify the source connection."));
    _xml->get_note("notebook")->set_current_page(_section);
    _xml->get_button("next_button")->set_sensitive(true);
    break;
  case 1:
    perform_schemata_fetches(back);
    break;
  case 2:
    perform_schemata_selection(back);
    break;
  case 3:
    perform_reverse_engineering(back);
    break;
  case 4:
    perform_object_selection();
    break;
  case 5:
    perform_placement_selection();
    break;
  case 6:
    finalize_reverse_engineering();
    break;
  }
}


MWReverseEngineering::MWReverseEngineering(GtkWindow *win)
  : MGRTWizardBase(win)
{  
  _section= 0;
  _last_section= 6;
  
  _schema_list= 0;
  _catalog= 0;
}


MWReverseEngineering::~MWReverseEngineering()
{
  delete _schema_list;
  delete _catalog;
  delete _conn_panel;
}


void MWReverseEngineering::setup()
{
  MGRTWizardBase::setup();

  _xml->get_note("notebook")->set_current_page(0);
  
  _schema_store= Gtk::ListStore::create(_columns);
  _xml->get_tree("schema_tree")->get_selection()->signal_changed().connect(sigc::mem_fun(*this,&MWReverseEngineering::changed_schema_selection));
  _xml->get_tree("schema_tree")->set_model(_schema_store);
  _xml->get_tree("schema_tree")->get_selection()->set_mode(Gtk::SELECTION_MULTIPLE);
  _xml->get_tree("schema_tree")->append_column("", _columns.icon);
  _xml->get_tree("schema_tree")->append_column("", _columns.text);
  _xml->get_tree("schema_tree")->set_headers_visible(false);

  _xml->get_combo("place_algo")->signal_changed().connect(sigc::mem_fun(*this,&MWReverseEngineering::placement_changed));
  
  _xml->get_combo("place_algo")->set_active(1);
  _xml->get_combo("place_rows")->set_active(0);
  _xml->get_combo("place_space")->set_active(0);

  _conn_panel= new MGRTConnectDialog(_grt, "/rdbmsMgmt", "/workbench/connection", false);

  _conn_panel->set_selects_rdbms(true);

  _conn_panel->setup();
  
  _xml->get_box("connect_box")->pack_start(*_conn_panel->rdbms_panel(), false, false);
  
  _xml->get_box("connect_box")->pack_start(*_conn_panel->params_panel(), false, false);
  

  // force placement parameters to have a uniform layout
  _param_sizegroup= Gtk::SizeGroup::create(Gtk::SIZE_GROUP_HORIZONTAL);

  _param_sizegroup->add_widget(*_xml->get_widget("label28"));
  _param_sizegroup->add_widget(*_xml->get_widget("label29"));

/*
  _conn_panel->advanced_panel()->reference();
  _conn_panel->advanced_panel()->unparent();
  _xml->get_box("connect_box")->pack_start(*_conn_panel->advanced_panel());
  _conn_panel->advanced_panel()->unreference();
*/
  _xml->get_box("connect_box")->show_all();
  _conn_panel->advanced_panel()->hide();
}


MWReverseEngineering *MWReverseEngineering::create(MGRT *grt)
{
  MWReverseEngineering *dlg= 0;
  MGGladeXML *xml;
  
  xml= new MGGladeXML(get_app_file("reverse_engineer.glade"));
  
  xml->get_widget_derived("reveng_window", dlg);
  
  dlg->_xml= xml;
  dlg->set_grt(grt);
  dlg->setup();

  return dlg;
}


void MWReverseEngineering::set_grt(MGRT *grt)
{
  MGRTWizardBase::set_grt(grt);
  
   MGRTValue migration(MGRTValue::createObject(_grt->grt(), "db.migration.Migration", "Migration"));
  _grt->set_global_value("/migration", migration.grtValue());
}


void MWReverseEngineering::schemata_fetches_done(const MGRTValue &result, bool error, void *data)
{
  if (!error && result.isValid())
  {
    _xml->get_image("fetch_image1")->set(_task_checked);
    _xml->get_image("fetch_image2")->set(_task_checked);
    _xml->get_label("fetch_ok_label")->show();

    _schema_list= new MGRTValue(result);
    
    _grt->out_text(_("Schemata fetched.\n"));
    
    end_work();
  }
  else
  {
    _xml->get_image("fetch_image1")->set(_task_error);
    _xml->get_image("fetch_image2")->set(_task_error);

    _grt->out_text(ufmt(_("The list of schema names could not be retrieved (error: %i)\n%s"),
                        _grt->last_error(), _grt->last_error_description().c_str()));

    _xml->get_widget("log_frame")->show();

    end_work();

    _xml->get_button("back_button")->set_sensitive(true);
    _xml->get_button("next_button")->set_sensitive(false);
  }
}


void MWReverseEngineering::perform_schemata_fetches(bool back)
{
  const char *revEngModule;

  MGRTValue connection(_conn_panel->write_connection_to_target());

  _xml->get_label("top_label")->set_markup(_("<b>Source Schemata</b>\nThe list of available schemata is fetched."));

  revEngModule= connection["modules"]["ReverseEngineeringModule"].asString();

  _xml->get_image("fetch_image1")->set(_task_unchecked);
  _xml->get_image("fetch_image2")->set(_task_unchecked);

  _xml->get_label("fetch_ok_label")->hide();

  _xml->get_note("notebook")->set_current_page(_section);

  _xml->get_button("back_button")->set_sensitive(true);
  _xml->get_button("next_button")->set_sensitive(true);

  if (back)
    return;

  begin_work();

  _grt->out_text(_("Fetching schemata...\n"));

  if (_schema_list)
    delete _schema_list;
  _schema_list= 0;

  MGRTValue args(MGRTValue::createList(MYX_ANY_VALUE));
  args.append(MGRTValue("global::/workbench/connection"));
  _grt->call_async_function(revEngModule,
                            "getSchemata",
                            args,
                            NULL,
                            sigc::mem_fun(*this,&MWReverseEngineering::schemata_fetches_done),
                            false);
}



void MWReverseEngineering::refresh_schemata_list()
{
  Glib::RefPtr<Gdk::Pixbuf> icon= PIXCACHE->load("schema_32x32.png");

  _schema_store->clear();
  for (unsigned int i= 0; i < _schema_list->count(); i++)
  {
    Gtk::TreeIter iter= _schema_store->append();
    Gtk::TreeRow row= *iter;

    row[_columns.icon]= icon;
    row[_columns.text]= (*_schema_list)[i].asString();
  }

  _xml->get_button("next_button")->set_sensitive(false);
}


void MWReverseEngineering::changed_schema_selection()
{
  int c= _xml->get_tree("schema_tree")->get_selection()->count_selected_rows();

  _xml->get_button("next_button")->set_sensitive(c > 0);

  switch (c)
  {
  case 0:
    _xml->get_label("schemata_label")->set_text(ufmt(_("%i items selected."), 0));
    break;
  case 1:
    _xml->get_label("schemata_label")->set_text(_("1 item selected."));
    break;
  default:
    _xml->get_label("schemata_label")->set_text(ufmt(_("%i items selected."), c));
    break;
  }
}


void MWReverseEngineering::perform_schemata_selection(bool back)
{
  MGRTValue connection(_conn_panel->write_connection_to_target());

  _xml->get_note("notebook")->set_current_page(_section);

  _xml->get_label("top_label")->set_markup(_("<b>Schema Selection</b>\nPlease select the schemata to reverse engineer."));

  _xml->get_button("back_button")->set_sensitive(true);
  _xml->get_button("next_button")->set_sensitive(true);

  if (back)
    return;

  refresh_schemata_list();

  _xml->get_label("schemata_label")->set_text(ufmt(_("%i items selected."), 0));
}


bool MWReverseEngineering::build_object_selection()
{
  //Clear Filter Frames if there are some already
  ((Gtk::ScrolledWindow*)_xml->get_widget("filter_scrollwin"))->remove();
  _filter_panes.clear();
  
  Gtk::VBox *vbox= Gtk::manage(new Gtk::VBox(false, 8));

  MGRTValue schemata((*_catalog)["schemata"]);

  //Get the struct type of the schemata
  MYX_GRT_STRUCT *gstruct= myx_grt_dict_struct_get(_grt->grt(), schemata[0].grtValue());

  //Clear the Mig.SourceObjects list
  MGRTValue sourceObjects(MGRTValue::fromGlobal(_grt->grt(), "/migration/sourceObjects"));
  sourceObjects.clear();

  //Create a filter frame for each member deriving from db.DatabaseObject
  for (int i= 0; i < myx_grt_struct_get_member_count_total(_grt->grt(), gstruct); i++)
  {
    MYX_GRT_STRUCT_MEMBER *member= myx_grt_struct_get_member_by_index_total(_grt->grt(), gstruct, i);
    const char *memberName= myx_grt_struct_get_member_name(member);
    if (myx_grt_struct_member_get_type(member) != MYX_LIST_VALUE)
      continue;
    const char *contentStructName= myx_grt_struct_member_get_content_struct_name(member);

    //Only consider structs that derive from db.DatabaseObject
    if (contentStructName &&
        myx_grt_struct_inherits_from(_grt->grt(),contentStructName, "db.DatabaseObject"))
    { 
      int count= 0;
      
      //Collect all objects for the sourceObject list
      //Loop over all schemata
      for (unsigned int j= 0; j < schemata.count(); j++)
      {
        MGRTValue schema(schemata[j]);
        MGRTValue schemaItemList(schema[memberName]);

        if (schemaItemList.isValid())
        {
          //Add objects to sourceObject list
          for (unsigned int k= 0; k < schemaItemList.count(); k++)
          {
            MGRTValue schemaItem(schemaItemList[k]);

            sourceObjects.append(MGRTValue(schemaItem.dictId()));
          }
          count+= schemaItemList.count();
        }
      }

      //Create FilterFrame
      if (count > 0)
      {
        MGRTRevEngFilterPane *filterFrame;
        
        filterFrame= new MGRTRevEngFilterPane(_grt, contentStructName);

        _filter_panes.push_back(filterFrame);

        filterFrame->show();
        vbox->pack_start(*filterFrame, true, true);

        if (g_str_has_suffix(contentStructName, ".Table")
            || g_str_has_suffix(contentStructName, ".View"))
        {
          filterFrame->set_selected(true);
        }
      }
    }
  }
  
  if (vbox->children().size() == 0)
  {
    Gtk::Label *label = new Gtk::Label(_("<b><big>There were no objects in the reverse engineered schema.</big></b>"));
    label->set_use_markup(true);
    vbox->pack_start(*Gtk::manage(label));
    label->show();
  }

  ((Gtk::ScrolledWindow*)_xml->get_widget("filter_scrollwin"))->add(*vbox);
  vbox->show();

  for (unsigned int i= 0; i < _filter_panes.size(); i++)
    _filter_panes[i]->refresh();
  
  return true;
}

void MWReverseEngineering::reverse_engineering_done(const MGRTValue &result, bool error, void *data)
{
  if (!error && result.isValid())
  {
    _catalog = new MGRTValue(result);

    _grt->set_global_value("/migration/sourceCatalog", result);

    _xml->get_image("reveng_image1")->set(_task_checked);

    if (build_object_selection())
    {
      _xml->get_image("reveng_image2")->set(_task_checked);

      _xml->get_label("reveng_ok_label")->show();
    }
    else
      _xml->get_image("reveng_image2")->set(_task_error);

    _grt->out_text(_("Reverse engineering finished.\n"));
  }
  else
  {
    _xml->get_image("reveng_image1")->set(_task_error);

    _grt->out_text(ufmt(_("The schema(ta) could not be reverse engineered (error: %i)\n%s\n"),
                        _grt->last_error(), _grt->last_error_description().c_str()));
  }

  end_work();
}


void MWReverseEngineering::perform_reverse_engineering(bool back)
{
  const char *revEngModule;
  MGRTValue result;
  
  MGRTValue connection(_conn_panel->write_connection_to_target());

  _xml->get_label("top_label")->set_markup(_("<b>Reverse Engineering</b>\nThe selected schemata is fetched and reverse engineered."));

  revEngModule= connection["modules"]["ReverseEngineeringModule"].asString();

  _xml->get_image("reveng_image1")->set(_task_unchecked);
  _xml->get_image("reveng_image2")->set(_task_unchecked);
  _xml->get_label("reveng_ok_label")->hide();

  _xml->get_button("back_button")->set_sensitive(true);
  _xml->get_note("notebook")->set_current_page(_section);
  if (back)
  {
    _xml->get_button("next_button")->set_sensitive(true);
    return;
  }

  MGRTValue selectedSchemas(MGRTValue::createList(MYX_STRING_VALUE));

  begin_work();

  std::list<Gtk::TreePath> selected= _xml->get_tree("schema_tree")->get_selection()->get_selected_rows();
  for (std::list<Gtk::TreePath>::const_iterator iter= selected.begin();
       iter != selected.end(); ++iter)
  {
    Gtk::TreeRow row= *_schema_store->get_iter(*iter);
    Glib::ustring name= row[_columns.text];

    selectedSchemas.append(MGRTValue(name.c_str()));
  }

  if (_catalog)
    delete _catalog;
  _catalog= 0;

  _grt->out_text(_("Reverse engineering schema..."));

  MGRTValue args(MGRTValue::createList(MYX_ANY_VALUE));;

  args.append(MGRTValue("global::/workbench/connection"));
  args.append(selectedSchemas);

  _grt->call_async_function(revEngModule,
                            "reverseEngineer",
                            args,
                            NULL,
                            sigc::mem_fun(*this,&MWReverseEngineering::reverse_engineering_done),
                            false);
}


void MWReverseEngineering::perform_object_selection()
{
  _xml->get_label("top_label")->set_markup(_("<b>Object Selection</b>\nAdd objects to the ignore list."));
  
  _xml->get_note("notebook")->set_current_page(_section);

  _xml->get_button("next_button")->set_sensitive(true);
}


void MWReverseEngineering::perform_placement_selection()
{
  _xml->get_label("top_label")->set_markup(_("<b>Placement Options</b>\nSelect options for placement of reverse engineered objects on the model."));

  _xml->get_note("notebook")->set_current_page(_section);

  _xml->get_button("next_button")->set_sensitive(true);
  
  MGRTValue schemaList(MGRTValue::fromGlobal(_grt->grt(), "/migration/sourceCatalog/schemata"));
  unsigned int objCount= 0;
  for (int i= schemaList.count()-1; i>=0; i--)
  {
    objCount+= schemaList[i]["tables"].count();
    objCount+= schemaList[i]["views"].count();
  }

  if (objCount > 300)
    _xml->get_combo("place_rows")->set_active(4);
  else if (objCount > 200)
    _xml->get_combo("place_rows")->set_active(3);
  else if (objCount > 100)
    _xml->get_combo("place_rows")->set_active(2);
  else if (objCount > 50)
    _xml->get_combo("place_rows")->set_active(1);
  else
    _xml->get_combo("place_rows")->set_active(0);
}



void MWReverseEngineering::merge_catalogs_done(const MGRTValue &result, bool error, void *data)
{
  _xml->get_image("fin_image2")->set(_task_checked);
  switch (_xml->get_combo("place_algo")->get_active_row_number())
  {
  case 1: // grid
    {
      _xml->get_label("finish_ok_label")->set_text(_("Placing objects on the canvas"));
    
      MGRTValue model(MGRTValue::fromGlobal(_grt->grt(), "/workbench/model"));
      MGRTValue schemaList(MGRTValue::fromGlobal(_grt->grt(), "/migration/sourceCatalog/schemata"));
      int maxColumns;
      int spacing;
      
      switch (_xml->get_combo("place_space")->get_active_row_number())
      {
      default:
      case 0: spacing= 40; break;
      case 1: spacing= 70; break;
      case 2: spacing= 100; break;
      }

      switch (_xml->get_combo("place_rows")->get_active_row_number())
      {
      default:
      case 0: maxColumns= 5; break;
      case 1: maxColumns= 10; break;
      case 2: maxColumns= 15; break;
      case 3: maxColumns= 20; break;
      case 4: maxColumns= 30; break;
      }

      for (unsigned int i= 0; i < schemaList.count(); i++)
      {
        MGRTValue schema(schemaList[i]);
        //check if the current view is empty
        MGRTValue currentView(MGRTValue::refObject(_grt->grt(), model["currentView"].asString()));
        MGRTValue viewElementList(currentView["elements"]);
        MGRTValue args(MGRTValue::createList());
        
        if (viewElementList.count() > 0)
        {
          args.append(model);
          args.append(schema["name"]);
          
          // if not, add a new view
          _grt->call_procedure("Workbench",
                               "addView",
                               args);
        }
        args.clear();
        args.append(schema);
        args.append(MGRTValue(20));
        args.append(MGRTValue(20));
        args.append(MGRTValue(maxColumns));
        args.append(MGRTValue(spacing));
        _grt->call_procedure("Workbench",
                             "arrangeSchemaOnCurrentView",
                             args);
      }
    }
    break;

  case 2: // auto
    {
      MGRTValue args(MGRTValue::createList());
      MGRTValue model(MGRTValue::fromGlobal(_grt->grt(), "/workbench/model"));
      MGRTValue currentView(MGRTValue::refObject(_grt->grt(), model["currentView"].asString()));
      MGRTValue schemaList(MGRTValue::fromGlobal(_grt->grt(), "/migration/sourceCatalog/schemata"));
      for (unsigned int i= 0; i < schemaList.count(); i++)
      {
        MGRTValue schema(schemaList[i]);
        
        //check if the current view is empty
        MGRTValue currentView(MGRTValue::refObject(_grt->grt(), model["currentView"].asString()));
        
        MGRTValue viewElementList(currentView["elements"]);
        if (viewElementList.count() > 0)
        {
          args.clear();
          args.append(model);
          args.append(schema["name"]);
          
          // if not, add a new view
          _grt->call_procedure("Workbench",
                               "addView",
                               args);
        }
        args.clear();
        args.append(schema);
        args.append(currentView);
        _grt->call_procedure("Workbench", "loadAndAutoArrange",
                             args);
      }
    }
    break;
  }

  _xml->get_image("fin_image3")->set(_task_checked);
  _xml->get_label("finish_ok_label")->set_text(_("Reverse engineering completed successfully."));

  _grt->out_text(_("Finished.\n"));

  end_work();

  _xml->get_button("next_button")->set_sensitive(true);
  _xml->get_button("cancel_button")->set_sensitive(false);
}


void MWReverseEngineering::finalize_reverse_engineering()
{
  _xml->get_label("top_label")->set_markup(_("<b>Finalization</b>\nFinalizing the reverse engineering process."));

  _xml->get_image("fin_image1")->set(_task_unchecked);
  _xml->get_image("fin_image2")->set(_task_unchecked);
  if (_xml->get_combo("place_algo")->get_active_row_number() == 0)
    _xml->get_image("fin_image3")->set(_task_disabled);
  else
    _xml->get_image("fin_image3")->set(_task_unchecked);
  _xml->get_label("finish_ok_label")->set_text("");


  _xml->get_note("notebook")->set_current_page(_section);

  _xml->get_button("next_button")->set_sensitive(false);
  _xml->get_button("back_button")->set_sensitive(false);
  _xml->get_button("cancel_button")->set_sensitive(true);

  begin_work();

  _grt->out_text(_("Finalizing..."));
  _xml->get_label("finish_ok_label")->set_text(_("Removing ignored objects from catalog"));

  MGRTValue args(MGRTValue::createList(MYX_ANY_VALUE));
  
  args.append(_grt->global_value("/migration/sourceCatalog"));
  args.append(_grt->global_value("/migration/ignoreList"));
  _grt->call_procedure("DbUtils",
                       "removeIgnoredObjectsFromCatalog",
                       args);

  args.clear();
  args.append(_grt->global_value("/workbench/catalog"));
  _grt->call_procedure("DbUtils",
                       "removeEmptySchemataFromCatalog",
                       args);

  _xml->get_image("fin_image1")->set(_task_checked);

  args.clear();
  args.append(_grt->global_value("/migration/sourceCatalog"));
  args.append(_grt->global_value("/workbench/catalog"));
  _grt->call_async_function("DbUtils",
                            "mergeCatalogs",
                            args,
                            0,
                            sigc::mem_fun(*this,&MWReverseEngineering::merge_catalogs_done),
                            true);
}


void MWReverseEngineering::placement_changed()
{
  _xml->get_note("param_note")->set_current_page(_xml->get_combo("place_algo")->get_active_row_number());
}
