unit GrtWb;

// Copyright (C) 2003, 2004 MySQL AB
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

interface

uses
  gnugettext, TntSystem,
  Windows, SysUtils, Classes, TntClasses, Forms, TntSysUtils, Messages,
  myx_public_interface,
  myx_grt_public_interface, myx_grt_builtin_module_public_interface,
  myx_grt_wb_public_interface,
  UniCodeConsole, SyncObjs, AuxFuncs,
  Grt;

type
  TGrtWb = class;
  TGrtEngineWb = class;

  GrtWbCallbackReason =
  (
    GrtWbChangeAdded = Ord(MYX_WBGC_ADDED),
    GrtWbChangeRemove = Ord(MYX_WBGC_REMOVED),
    GrtWbChangeModified = Ord(MYX_WBGC_MODIFIED),
    GrtWbChangeSwitched = Ord(MYX_WBGC_SWITCHED),

    GrtWbRefresh = Ord(MYX_WBGC_REFRESH),

    GrtWbChangeRubberbandStarted = Ord(MYX_WBGC_RUBBERBAND_STARTED),
    GrtWbChangeRubberbandStopped = Ord(MYX_WBGC_RUBBERBAND_STOPPED),

    GrtWbChangeTaskCompleted = Ord(MYX_WBGC_TASK_COMPLETED),
    GrtWbChangeTaskCanceled = Ord(MYX_WBGC_TASK_CANCELED),

    GrtWbChangeSeletionChange = Ord(MYX_WBGC_SELECTION_CHANGE),
    GrtWbChangeLayoutChanged = Ord(MYX_WBGC_LAYOUT_CHANGED)
  );

  TGrtGcViewChange = procedure(View, ViewObject: Pointer; Reason: GrtWbCallbackReason) of object;
  TGrtGcLayerChange = procedure(Layer, LayerObject: Pointer; Reason: GrtWbCallbackReason) of object;
  TGrtGcFigureChange = procedure(Figure, FigureObject: Pointer; Reason: GrtWbCallbackReason) of object;

  TGrtWb = class(TGrt)
  protected
    function CreateGrtEngine: TGrtEngine; override;
  public
    procedure InitializeGcWbBridge(Canvas: Pointer; Path: WideString);
    procedure FreeGcWbBridge(Canvas: Pointer; Path: WideString);
    procedure RegisterGcWbBridgeCallbacks(
      ViewChangeCallback: TGrtGcViewChange;
      LayerChangeCallback: TGrtGcLayerChange;
      FigureChangeCallback: TGrtGcFigureChange);

    procedure ProcessPendingWbBridge;
    procedure ProcessRelocations;
    function LoadStylesFromFile(Canvas: Pointer; FileName: WideString; Variables: Pointer): Integer;

    function GetWbObjectForView(View: Pointer): Pointer;
  end;

  PCallbackEntry = ^TCallbackEntry;
  TCallbackEntry = record
    Type_: Integer;
    GrtValue: Pointer;            // The GRT value that has had the change.
    ValueObject: Pointer;         // The associated GC object.
    Reason: GrtWbCallbackReason;
  end;

  TGrtEngineWb = class(TGrtEngine)
  public
    constructor Create(CreateSuspended: Boolean; Grt: TGrt); override;
    destructor Destroy; override;

    procedure AddCallbackEntry(AType: Integer; GrtValue, ValueObj: Pointer; AReason: GrtWbCallbackReason);
  protected
    function ApplicationHook(var Message: TMessage): Boolean; override;
    procedure InitializeLoadersAndModules; override;
    procedure FinalizeLoadersAndModules; override;

    procedure DoWbChangeCallback;
  private
    FWbLoaderState: GrtLoaderState;
    FCallbackEntries: TThreadList;      // A list of yet to complete callbacks.

    FViewChangeCallback: TGrtGcViewChange;
    FLayerChangeCallback: TGrtGcLayerChange;
    FFigureChangeCallback: TGrtGcFigureChange;

    procedure ClearCallbackEntries;
  end;

procedure ProcessViewChangeCallback(Runtime: Pointer; view, viewObject: Pointer; change: MYX_WB_GC_CHANGE; data: Pointer) cdecl;
procedure ProcessLayerChangeCallback(Runtime: Pointer; layer, layerObject: Pointer; change: MYX_WB_GC_CHANGE; data: Pointer) cdecl;
procedure ProcessFigureChangeCallback(Runtime: Pointer; figure, figureObject: Pointer; change: MYX_WB_GC_CHANGE; data: Pointer) cdecl;

//----------------------------------------------------------------------------------------------------------------------

implementation

const
  WM_PROCESS_CALLBACK = WM_APP + 10101;

//----------------------------------------------------------------------------------------------------------------------

procedure ProcessViewChangeCallback(Runtime: Pointer; view, viewObject: Pointer; change: MYX_WB_GC_CHANGE; data: Pointer) cdecl;

var
  GrtEngine: TGrtEngineWb;

begin
  GrtEngine := data;

  if (Assigned(GrtEngine.FViewChangeCallback)) then
    GrtEngine.AddCallbackEntry(0, view, viewObject, GrtWbCallbackReason(change));
end;

//----------------------------------------------------------------------------------------------------------------------

procedure ProcessLayerChangeCallback(Runtime: Pointer; layer, layerObject: Pointer; change: MYX_WB_GC_CHANGE; data: Pointer) cdecl;

var
  GrtEngine: TGrtEngineWb;

begin
  GrtEngine := data;

  if (Assigned(GrtEngine.FLayerChangeCallback)) then
    GrtEngine.AddCallbackEntry(1, layer, layerObject, GrtWbCallbackReason(change));
end;

//----------------------------------------------------------------------------------------------------------------------

procedure ProcessFigureChangeCallback(Runtime: Pointer; figure, figureObject: Pointer; change: MYX_WB_GC_CHANGE; data: Pointer) cdecl;

var
  GrtEngine: TGrtEngineWb;

begin
  GrtEngine := data;

  if (Assigned(GrtEngine.FFigureChangeCallback)) then
    GrtEngine.AddCallbackEntry(2, figure, figureObject, GrtWbCallbackReason(change));
end;

//----------------------------------------------------------------------------------------------------------------------

function TGrtWb.CreateGrtEngine: TGrtEngine;

begin
  Result := TGrtEngineWb.Create(False, self);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGrtWb.InitializeGcWbBridge(Canvas: Pointer; Path: WideString);

begin
  myx_grt_wb_bridge_initialize(NativeGrt, Canvas, Path);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGrtWb.FreeGcWbBridge(Canvas: Pointer; Path: WideString);

begin
  myx_grt_wb_bridge_free(NativeGrt, Canvas, Path);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGrtWb.RegisterGcWbBridgeCallbacks(
  ViewChangeCallback: TGrtGcViewChange;
  LayerChangeCallback: TGrtGcLayerChange;
  FigureChangeCallback: TGrtGcFigureChange);

begin
  if (Assigned(FGrtEngine)) then
  begin
    TGrtEngineWb(FGrtEngine).FViewChangeCallback := ViewChangeCallback;
    TGrtEngineWb(FGrtEngine).FLayerChangeCallback := LayerChangeCallback;
    TGrtEngineWb(FGrtEngine).FFigureChangeCallback := FigureChangeCallback;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGrtWb.ProcessPendingWbBridge;

begin
  myx_grt_wb_bridge_process_pending(NativeGrt);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGrtWb.ProcessRelocations;

begin
  myx_grt_wb_bridge_process_relocations(NativeGrt);
end;

//----------------------------------------------------------------------------------------------------------------------

function TGrtWb.LoadStylesFromFile(Canvas: Pointer;
  FileName: WideString; Variables: Pointer): Integer;

begin
  Result := myx_grt_wb_load_style_file(Canvas, FileName, Variables);
end;

//----------------------------------------------------------------------------------------------------------------------

function TGrtWb.GetWbObjectForView(View: Pointer): Pointer;

begin
  Result := myx_grt_wb_object_for_view(View);
end;

//----------------------------------------------------------------------------------------------------------------------
// Grt Engine
//----------------------------------------------------------------------------------------------------------------------

constructor TGrtEngineWb.Create(CreateSuspended: Boolean; Grt: TGrt);

begin
  inherited Create(CreateSuspended, Grt);

  FWbLoaderState := GrtLsNotInitialized;
  FCallbackEntries := TThreadList.Create;

  FViewChangeCallback := nil;
  FLayerChangeCallback := nil;
  FFigureChangeCallback := nil;
end;

//----------------------------------------------------------------------------------------------------------------------

destructor TGrtEngineWb.Destroy;

begin
  ClearCallbackEntries;
  inherited;
end;

// -----------------------------------------------------------------------------

procedure TGrtEngineWb.FinalizeLoadersAndModules;

begin
  inherited;

  // Unload canvas module
  myx_unregister_grt_wb_module(NativeGrt);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGrtEngineWb.InitializeLoadersAndModules;

begin
  inherited;
  
  //Load canvas module
  FWbLoaderState := GrtLsInitializeFailed;
  if (myx_register_grt_wb_module(NativeGrt) = MYX_GRT_NO_ERROR) then
  begin
    FWbLoaderState := GrtLsModulesLoaded;

    OutputModuleStatus('Workbench', 1, MYX_GRT_NO_ERROR);

    myx_grt_wb_bridge_set_callbacks(NativeGrt, self, ProcessViewChangeCallback, ProcessLayerChangeCallback,
      ProcessFigureChangeCallback);
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGrtEngineWb.DoWbChangeCallback;

var
  Entry: PCallbackEntry;

begin
  repeat
    with FCallbackEntries, LockList do
    begin
      if Count > 0 then
      begin
        Entry := First;
        Remove(Entry);
      end
      else
        Entry := nil;
      UnlockList;
    end;

    if Entry = nil then
      Break;

    case Entry.Type_ of
      0:
        FViewChangeCallback(Entry.GrtValue, Entry.ValueObject, Entry.Reason);
      1:
        FLayerChangeCallback(Entry.GrtValue, Entry.ValueObject, Entry.Reason);
      2:
        FFigureChangeCallback(Entry.GrtValue, Entry.ValueObject, Entry.Reason);
    end;
    myx_grt_value_release(Entry.GrtValue);
    
    Dispose(Entry);
  until False;

  TGrtWb(Grt).ProcessPendingWbBridge;
end;

// -----------------------------------------------------------------------------

function TGrtEngineWb.ApplicationHook(var Message: TMessage): Boolean;

begin
  Result := inherited ApplicationHook(Message);
  if not Result then
    case Message.Msg of
      WM_PROCESS_CALLBACK:
        begin
          DoWbChangeCallback;
          Result := True;
        end;
    end;
end;

// -----------------------------------------------------------------------------

procedure TGrtEngineWb.ClearCallbackEntries;

var
  I: Integer;
  Entry: PCallbackEntry;

begin
  with FCallbackEntries, LockList do
  try
    for I := 0 to Count - 1 do
    begin
      Entry := Items[I];
      myx_grt_value_release(Entry.GrtValue);
      Dispose(Entry);
    end;
    Clear;
  finally
    UnlockList;
  end;
end;

// -----------------------------------------------------------------------------

procedure TGrtEngineWb.AddCallbackEntry(AType: Integer; GrtValue, ValueObj: Pointer; AReason: GrtWbCallbackReason);

var
  Entry: PCallbackEntry;
  LastEntry: PCallbackEntry;

begin
  with FCallbackEntries, LockList do
  try
    if Count > 0 then
      LastEntry := Last
    else
      LastEntry := nil;

    if (LastEntry = nil) or (LastEntry.Type_ <> AType) or (LastEntry.GrtValue <> GrtValue) or
      (LastEntry.ValueObject <> ValueObj) or (LastEntry.Reason <> AReason) then
    begin
      // Add a new entry only if there is none yet or the last one we added differs from the new one.
      New(Entry);
      Entry.Type_ := AType;
      Entry.GrtValue := GrtValue;
      Entry.ValueObject := ValueObj;
      myx_grt_value_retain(GrtValue);
      Entry.Reason := AReason;
      FCallbackEntries.Add(Entry);
    end;
  finally
    UnlockList;
  end;

  PostMessage(Application.Handle, WM_PROCESS_CALLBACK, 0, 0);
end;

//----------------------------------------------------------------------------------------------------------------------

initialization
  SetGrtClass(TGrtWb);
end.

