/* Copyright (C) 2006 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

#include "MWSynchronizeDatabase.h"

#include "MGGladeXML.h"
#include "myg_gtkutils.h"
#include "mywb.h"

#include <MySQLGRT/MGRTRevEngFilterPane.h>


/**
 * @file  MWSynchronizeDatabase.cc
 * @brief 
 */

void MWSynchronizeDatabase::update_advanced()
{
  MGRTWizardBase::update_advanced();
  
  bool has_details= false;
  Gtk::Widget *frame= _xml->get_widget("log_frame");
  switch (_section)
  {
  case 0: has_details= true; break;
  case 1:
    has_details= true;
    frame->reference();
    frame->get_parent()->remove(*frame);
    _xml->get_box("fetch_box")->pack_start(*frame);
    frame->unreference();
    break;
  case 3:
    has_details= true;
    frame->reference();
    frame->get_parent()->remove(*frame);
    _xml->get_box("reveng_box")->pack_start(*frame);
    frame->unreference();
    break;
  case 6:
    has_details= true;
    frame->reference();
    frame->get_parent()->remove(*frame);
    _xml->get_box("finish_box")->pack_start(*frame);
    frame->unreference();
    break;
  }
  _xml->get_button("show_more_button")->set_sensitive(has_details);
  
  if (!_advanced_shown)
    frame->hide();
  else
    frame->show();
}


void MWSynchronizeDatabase::update_section(bool back)
{
  MGRTWizardBase::update_section(back);

  switch (_section)
  {
  case 0:
    _xml->get_label("top_label")->set_markup(_("<b>Source Connection</b>\nPlease specify the source connection."));
    _xml->get_note("notebook")->set_current_page(_section);
    break;
  case 1:
    perform_schemata_fetches(back);
    break;
  case 2:
    perform_schemata_selection(back);
    break;
  case 3:
    perform_reverse_engineering(back);
    break;
  case 4:
    perform_action_selection(back);
    break;
  case 5:
    show_script_preview();
    break;
  case 6:
    finalize_synchronization();
    break;
  }
}


MWSynchronizeDatabase::MWSynchronizeDatabase(GtkWindow *win)
  : MGRTWizardBase(win)
{  
  _section= 0;
  _last_section= 6;

  _changes_tree= 0;
  _schema_list= 0;
  _catalog= 0;

  _conn_panel= 0;
  
  _connect_ok= false;
}


MWSynchronizeDatabase::~MWSynchronizeDatabase()
{
  _grt->unset_global_value("/databaseSync");

  delete _changes_tree;
  delete _schema_list;
  delete _catalog;
  delete _conn_panel;
}


void MWSynchronizeDatabase::connect_ready_changed(bool ready)
{
  _connect_ok= ready;
  update_section(false);
}


bool MWSynchronizeDatabase::validate_section()
{
  if (_section == 0)
    return _connect_ok;

  return MGRTWizardBase::validate_section();
}


void MWSynchronizeDatabase::setup()
{
  MGRTWizardBase::setup();

  _xml->get_note("notebook")->set_current_page(0);
  
  _schema_store= Gtk::ListStore::create(_columns);
  _xml->get_tree("schema_tree")->get_selection()->signal_changed().connect(sigc::mem_fun(*this,&MWSynchronizeDatabase::changed_schema_selection));
  _xml->get_tree("schema_tree")->set_model(_schema_store);
  _xml->get_tree("schema_tree")->get_selection()->set_mode(Gtk::SELECTION_MULTIPLE);
  _xml->get_tree("schema_tree")->append_column("", _columns.icon);
  _xml->get_tree("schema_tree")->append_column("", _columns.text);
  _xml->get_tree("schema_tree")->set_headers_visible(false);


  _conn_panel= new MGRTConnectDialog(_grt, "/rdbmsMgmt", "/workbench/connection", false);

  _conn_panel->signal_ready_changed().connect(sigc::mem_fun(*this, &MWSynchronizeDatabase::connect_ready_changed));
  
  _conn_panel->set_selects_rdbms(false);

  _conn_panel->setup();
    
  _xml->get_box("connect_box")->pack_start(*_conn_panel->params_panel(), false, false);
  

/*
  _conn_panel->advanced_panel()->reference();
  _conn_panel->advanced_panel()->unparent();
  _xml->get_box("connect_box")->pack_start(*_conn_panel->advanced_panel());
  _conn_panel->advanced_panel()->unreference();
*/
  _xml->get_box("connect_box")->show_all();
  _conn_panel->advanced_panel()->hide();
  
  
  Gtk::TreeView *tree= _xml->get_tree("sync_tree");

  Gtk::TreeViewColumn *column= new Gtk::TreeView::Column(_("Database Object"));
  column->pack_start(_acolumns.icon, false);
  column->pack_start(_acolumns.db_object);
  tree->append_column(*Gtk::manage(column));

  MGCellRendererText *rend= Gtk::manage(new MGCellRendererText());
  column= new Gtk::TreeView::Column(_("Action"), *rend);
  column->add_attribute(rend->property_text(), _acolumns.action_text);
  tree->append_column(*Gtk::manage(column));
  rend->property_visible()= true;
  rend->property_weight()= Pango::WEIGHT_BOLD;
  rend->property_mode()= Gtk::CELL_RENDERER_MODE_ACTIVATABLE;
  rend->signal_clicked().connect(sigc::mem_fun(*this,&MWSynchronizeDatabase::action_toggled));

  tree->append_column(_("Model Object"), _acolumns.model_object);
  
  tree->get_column(0)->set_resizable(true);
  tree->get_column(1)->set_resizable(true);
  tree->get_column(2)->set_resizable(true);
  
  _action_store= Gtk::TreeStore::create(_acolumns);
  tree->set_model(_action_store);
}


MWSynchronizeDatabase *MWSynchronizeDatabase::create(MGRT *grt)
{
  MWSynchronizeDatabase *dlg= 0;
  MGGladeXML *xml;
  
  xml= new MGGladeXML(get_app_file("sync_db.glade"));
  
  xml->get_widget_derived("sync_window", dlg);
  
  dlg->_xml= xml;
  dlg->set_grt(grt);
  dlg->setup();

  return dlg;
}


void MWSynchronizeDatabase::set_grt(MGRT *grt)
{
  MGRTWizardBase::set_grt(grt);
  
  MGRTValue dbSync(MGRTValue::createObject(_grt->grt(), "db.DatabaseSync", "DatabaseSync"));
  _grt->set_global_value("/databaseSync", dbSync);
}


void MWSynchronizeDatabase::schemata_fetches_done(const MGRTValue &result, bool error, void *data)
{
  if (!error && result.isValid())
  {
    _xml->get_image("fetch_image1")->set(_task_checked);
    _xml->get_image("fetch_image2")->set(_task_checked);

    _xml->get_widget("fetch_ok_label")->show();
    
    _schema_list= new MGRTValue(result);

    _grt->set_global_value("/databaseSync/dbCatalog", *_schema_list);

    _grt->out_text(_("Schemata fetched.\n"));

    end_work();
  }
  else
  {
    _xml->get_image("fetch_image1")->set(_task_error);
    _xml->get_image("fetch_image2")->set(_task_error);

    _grt->out_text(_("The list of schema names could not be retrieved"));

    _xml->get_widget("log_frame")->show();

    end_work();

    _xml->get_button("back_button")->set_sensitive(true);
    _xml->get_button("next_button")->set_sensitive(false);
  }
}


void MWSynchronizeDatabase::perform_schemata_fetches(bool back)
{
  const char *revEngModule;

  MGRTValue connection(_conn_panel->write_connection_to_target());

  _xml->get_label("top_label")->set_markup(_("<b>Source Schemata</b>\nThe list of available schemata is fetched."));

  revEngModule= connection["modules"]["ReverseEngineeringModule"].asString();

  _xml->get_note("notebook")->set_current_page(_section);

  _xml->get_button("back_button")->set_sensitive(true);
  _xml->get_button("next_button")->set_sensitive(true);

  if (back)
    return;

  _xml->get_image("fetch_image1")->set(_task_unchecked);
  _xml->get_image("fetch_image2")->set(_task_unchecked);

  _xml->get_label("fetch_ok_label")->hide();

  begin_work();

  _grt->out_text(_("Fetching schemata...\n"));

  if (_schema_list)
    delete _schema_list;
  _schema_list= 0;

  MGRTValue args(MGRTValue::createList(MYX_ANY_VALUE));
  args.append(MGRTValue("global::/workbench/connection"));
  _grt->call_async_function(revEngModule,
                            "getSchemata",
                            args,
                            NULL,
                            sigc::mem_fun(*this,&MWSynchronizeDatabase::schemata_fetches_done),
                            false);
}



void MWSynchronizeDatabase::refresh_schemata_list()
{
  Glib::RefPtr<Gdk::Pixbuf> icon= PIXCACHE->load("schema_32x32.png");

  _schema_store->clear();
  for (unsigned int i= 0; i < _schema_list->count(); i++)
  {
    Gtk::TreeIter iter= _schema_store->append();
    Gtk::TreeRow row= *iter;

    row[_columns.icon]= icon;
    row[_columns.text]= (*_schema_list)[i].asString();
  }

  _xml->get_button("next_button")->set_sensitive(false);
}


void MWSynchronizeDatabase::changed_schema_selection()
{
  int c= _xml->get_tree("schema_tree")->get_selection()->count_selected_rows();

  _xml->get_button("next_button")->set_sensitive(c > 0);

  switch (c)
  {
  case 0:
    _xml->get_label("schemata_label")->set_text(ufmt(_("%i items selected."), 0));
    break;
  case 1:
    _xml->get_label("schemata_label")->set_text(_("1 item selected."));
    break;
  default:
    _xml->get_label("schemata_label")->set_text(ufmt(_("%i items selected."), c));
    break;
  }
}


void MWSynchronizeDatabase::perform_schemata_selection(bool back)
{
  MGRTValue connection(_conn_panel->write_connection_to_target());

  _xml->get_note("notebook")->set_current_page(_section);

  _xml->get_label("top_label")->set_markup(_("<b>Schema Selection</b>\nPlease select the schemata to reverse engineer."));

  _xml->get_button("back_button")->set_sensitive(true);
  _xml->get_button("next_button")->set_sensitive(true);

  if (back)
    return;

  refresh_schemata_list();

  _xml->get_label("schemata_label")->set_text(ufmt(_("%i items selected."), 0));
}




void MWSynchronizeDatabase::changes_get_done(const MGRTValue &result, bool error, void *data)
{
  if (!error && result.isValid())
  {    
    _xml->get_image("reveng_image2")->set(_task_checked);

    _grt->out_text(_("Changes list fetched.\n"));
    
    if (_changes_tree)
      delete _changes_tree;
    _changes_tree= new MGRTValue(result);

    _xml->get_label("reveng_ok_label")->show();

    end_work();
  }
  else
  {
    _xml->get_image("reveng_image2")->set(_task_error);

    _grt->out_text(_("The list of changes could not be processed"));

    _xml->get_widget("log_frame")->show();

    end_work();
   
    _xml->get_button("back_button")->set_sensitive(true);
    _xml->get_button("next_button")->set_sensitive(false);
  }
}


void MWSynchronizeDatabase::reverse_engineering_done(const MGRTValue &result, bool error, void *data)
{
  if (!error && result.isValid())
  {
    _catalog = new MGRTValue(result);

    
    _grt->set_global_value("/databaseSync/dbCatalog", *_catalog);

    _xml->get_image("reveng_image1")->set(_task_checked);
    _xml->get_image("reveng_image2")->set(_task_unchecked);

    MGRTValue args(MGRTValue::createList(MYX_ANY_VALUE));
    args.append(MGRTValue::fromGlobal(_grt->grt(), "/workbench/catalog"));
    args.append(MGRTValue::fromGlobal(_grt->grt(), "/databaseSync/dbCatalog"));
    
    _grt->call_async_function("TransformationMysql",
                              "getCatalogsChanges",
                              args,
                              NULL,
                              sigc::mem_fun(*this,&MWSynchronizeDatabase::changes_get_done),
                              false);
  }
  else
  {
    _xml->get_image("reveng_image1")->set(_task_error);

    _grt->out_text(ufmt(_("The schema(ta) could not be reverse engineered (error: %i)\n%s\n"),
                        _grt->last_error(), _grt->last_error_description().c_str()));
    
    end_work();
    _xml->get_button("next_button")->set_sensitive(false);
  }
}


void MWSynchronizeDatabase::perform_reverse_engineering(bool back)
{
  const char *revEngModule;
  MGRTValue result;
  
  MGRTValue connection(_conn_panel->write_connection_to_target());

  _xml->get_label("top_label")->set_markup(_("<b>Reverse Engineering</b>\nThe selected schemata is fetched and reverse engineered."));

  revEngModule= connection["modules"]["ReverseEngineeringModule"].asString();

  _xml->get_image("reveng_image1")->set(_task_unchecked);
  _xml->get_image("reveng_image2")->set(_task_disabled);
  _xml->get_label("reveng_ok_label")->hide();

  _xml->get_button("back_button")->set_sensitive(true);
  _xml->get_note("notebook")->set_current_page(_section);
  if (back)
  {
    _xml->get_button("next_button")->set_sensitive(true);
    return;
  }

  MGRTValue selectedSchemas(MGRTValue::createList(MYX_STRING_VALUE));

  begin_work();

  std::list<Gtk::TreePath> selected= _xml->get_tree("schema_tree")->get_selection()->get_selected_rows();
  for (std::list<Gtk::TreePath>::const_iterator iter= selected.begin();
       iter != selected.end(); ++iter)
  {
    Gtk::TreeRow row= *_schema_store->get_iter(*iter);
    Glib::ustring name= row[_columns.text];

    selectedSchemas.append(MGRTValue(name.c_str()));
  }

  if (_catalog)
    delete _catalog;
  _catalog= 0;

  _grt->out_text(_("Reverse engineering schema..."));

  MGRTValue args(MGRTValue::createList(MYX_ANY_VALUE));;

  args.append(MGRTValue("global::/workbench/connection"));
  args.append(selectedSchemas);

  _grt->call_async_function(revEngModule,
                            "reverseEngineer",
                            args,
                            NULL,
                            sigc::mem_fun(*this,&MWSynchronizeDatabase::reverse_engineering_done),
                            false);
}


void MWSynchronizeDatabase::set_action_text(Gtk::TreeRow &row,
                                            const MGRTValue &value)
{
  if (value["changed"].asInt())
  {
    if (value.getValue("dbObject").isValid() && value.getValue("modelObject").isValid())
    {
      if (!value["alterDirection"].isValid() || value["alterDirection"].asInt()==0)
        row[_acolumns.action_text]= _("Update DB");
      else
        row[_acolumns.action_text]= _("Update Model");
    }
    else if (!value.getValue("modelObject").isValid())
    {
      if (!value["alterDirection"].isValid() || value["alterDirection"].asInt()==0)
        row[_acolumns.action_text]= _("Drop Object");
      else
        row[_acolumns.action_text]= _("Fetch Object");
    }
    else
    {
      if (!value["alterDirection"].isValid() || value["alterDirection"].asInt()==0)
        row[_acolumns.action_text]= _("Create Object");
      else
        row[_acolumns.action_text]= _("Delete in Model");
    }
  }
  else
    row[_acolumns.action_text]= "";
}


void MWSynchronizeDatabase::fill_action_tree(Gtk::TreeIter iter,
                                             MGRTValue value)
{
  Gtk::TreeRow row;
  MGRTValue dbObject, modelObject, object;

  if (!iter)
    iter= _action_store->append();
  else
    iter= _action_store->append((*iter)->children());

  row= *iter;
  if (value["dbObject"].isValid())
  {
    dbObject= MGRTValue::refObject(_grt->grt(), value["dbObject"].asString());
    row[_acolumns.db_object]= dbObject["name"].asString();
  }
  else
    row[_acolumns.db_object]= "-";

  if (value["modelObject"].isValid())
  {
    modelObject=  MGRTValue::refObject(_grt->grt(), value["modelObject"].asString());
    row[_acolumns.model_object]= modelObject["name"].asString();
  }
  else
    row[_acolumns.model_object]= "-";

  set_action_text(row, value);

  row[_acolumns.value]= value;

  if (dbObject.isValid())
    object= dbObject;
  else
    object= modelObject;

  if (object.isKindOf(_grt->grt(), "db.Table"))
    row[_acolumns.icon]= PIXCACHE->load("db.Table.16x16.png");
  else if (object.isKindOf(_grt->grt(), "db.View"))
    row[_acolumns.icon]= PIXCACHE->load("db.View.16x16.png");
  else if (object.isKindOf(_grt->grt(), "db.Routine"))
    row[_acolumns.icon]= PIXCACHE->load("db.Routine.16x16.png");
  else if (object.isKindOf(_grt->grt(), "db.Schema"))
    row[_acolumns.icon]= PIXCACHE->load("db.Schema.16x16.png");
  else if (object.isKindOf(_grt->grt(), "db.Column"))
    row[_acolumns.icon]= PIXCACHE->load("16x16_Field.png");

  MGRTValue children(value["children"]);
  
  for (unsigned int i= 0; i < children.count(); i++)
  {
    fill_action_tree(iter, children[i]);
  }
}


void MWSynchronizeDatabase::refresh_action_list()
{  
  _action_store->clear();

  fill_action_tree(Gtk::TreeIter(), *_changes_tree);
  
  
  // expand everything up to tables
  Gtk::TreeView *tree= _xml->get_tree("sync_tree");
  Gtk::TreeIter siter, titer;
  siter= _action_store->children().begin();
  while (siter != _action_store->children().end())
  {
    Gtk::TreeRow srow= *siter;
    tree->expand_row(Gtk::TreePath(siter), false);
    titer= srow.children().begin();
    while (titer != srow.children().end())
    {
      tree->expand_row(Gtk::TreePath(titer), false);
      ++titer;
    }
    ++siter;
  }
}


#if 0
void MWSynchronizeDatabase::perform_change_processing(bool back)
{
  _xml->get_note("notebook")->set_current_page(_section);

  _xml->get_label("top_label")->set_markup(_("<b>Synchronize</b>\nSpecify changes to be applied to the database and model."));

  _xml->get_button("back_button")->set_sensitive(true);
  _xml->get_button("next_button")->set_sensitive(true);

  if (back)
    return;
  
  MGRTValue args(MGRTValue::createList());
  
  args.append(*_changes_tree);

  _grt->call_async_function("TransformationMysql", "getSqlChanges", args, 
                            NULL, sigc::mem_fun(*this,&MWSynchronizeDatabase::sync_data_done),
                            true);

  begin_work();
}
#endif

void MWSynchronizeDatabase::sync_data_done(const MGRTValue &result, bool error, void *data)
{
  if (error)
  {
    _xml->get_image("process_image")->set(_task_error);

    end_work();
    _xml->get_button("next_button")->set_sensitive(false);
    
    _xml->get_widget("log_frame")->show();
  }
  else
  {
    _xml->get_image("process_image")->set(_task_checked);

    end_work();
        
    _xml->get_label("process_label")->set_markup(_("<b>No changes in the model.</b>"));
    _xml->get_label("process_label")->show();

    _grt->out_text(_("There are no differences to be synchronized."));
    
    _xml->get_button("back_button")->set_sensitive(false);
    _xml->get_button("next_button")->hide();
    _xml->get_button("fin_button")->show();
  }
}


void MWSynchronizeDatabase::perform_action_selection(bool back)
{
  _xml->get_note("notebook")->set_current_page(_section);

  _xml->get_label("top_label")->set_markup(_("<b>Synchronization Actions</b>\nSpecify changes to be applied to the database and model."));

  _xml->get_button("back_button")->set_sensitive(true);
  _xml->get_button("next_button")->set_sensitive(true);

  if (back)
    return;

  
  refresh_action_list();
}



void MWSynchronizeDatabase::show_script_preview()
{
  _xml->get_note("notebook")->set_current_page(_section);

  _xml->get_label("top_label")->set_markup(_("<b>Script Preview</b>\nPreview script to be executed for performing synchronization."));

  _xml->get_button("back_button")->set_sensitive(true);
  _xml->get_button("next_button")->set_sensitive(true);


  MGRTValue args(MGRTValue::createList());
  args.append(*_changes_tree);

  MGRTValue result= _grt->call_function("TransformationMysql", "getSqlChanges", args);
  
  
  Glib::ustring script= collect_sql_script(*_changes_tree);
  _xml->get_text("textview1")->get_buffer()->set_text(script);
}



Glib::ustring MWSynchronizeDatabase::collect_sql_script(MGRTValue tree)
{
  MGRTValue obj;
  Glib::ustring script;

  obj= tree["dbObject"];
  if (!obj.isValid())
    obj= tree["modelObject"];
  
  if (obj.isValid())
  {
    obj= MGRTValue::refObject(_grt->grt(), obj.asString());
    Glib::ustring text= obj["sql"].isValid() ? obj["sql"].asString() : "";
    
    if (text != "")
      script= script + text + "\n";
    
    MGRTValue children(tree["children"]);

    if (children.isValid())
      for (unsigned int i= 0; i < children.count(); i++)
      {
        script= script + collect_sql_script(children[i]);
      }
  }
  return script;
}



bool MWSynchronizeDatabase::collect_and_show_errors(MGRTValue tree)
{
  bool res= true;
  MGRTValue children(tree["children"]);
  MGRTValue slog;
  
  for (unsigned int i= 0; i < children.count(); i++)
    res= collect_and_show_errors(children[i]) || res;

  slog= tree["syncLog"];
  if (slog.isValid())
  {
    MGRTValue entries(slog["entries"]);
    if (entries.isValid() && entries.count() > 0)
    {
      res= false;
      for (unsigned int i= 0; i < entries.count(); i++)
      {
        MGRTValue item(entries[i]);
        
        _xml->get_text("log_text")->get_buffer()->insert(_xml->get_text("log_text")->get_buffer()->end(),
                                                         item["name"].asString());
      }
    }
  }
  return res;
}


void MWSynchronizeDatabase::finalize_done(const MGRTValue &result, bool error, void *data)
{
  end_work();
  
  if (collect_and_show_errors(*_changes_tree))
  {
    _xml->get_image("fin_image")->set(_task_checked);
    _xml->get_label("finish_ok_label")->show();
  }
  else
  {
    _xml->get_image("fin_image")->set(_task_error);
    _xml->get_label("finish_ok_label")->show();
    _xml->get_label("finish_ok_label")->set_text(_("<b>Error applying changes.</b>"));
  }
}


void MWSynchronizeDatabase::finalize_synchronization()
{
  _xml->get_note("notebook")->set_current_page(_section);

  
  MGRTValue args(MGRTValue::createList());
  args.append(*_changes_tree);
  args.append(_conn_panel->write_connection_to_target());

  _xml->get_image("fin_image")->set(_task_unchecked);
  begin_work();
  
  _grt->call_async_function("TransformationMysql",
                            "applySqlChanges",
                            args,
                            0,
                            sigc::mem_fun(*this,&MWSynchronizeDatabase::finalize_done),
                            true);
}




void MWSynchronizeDatabase::action_toggled(const Glib::ustring &path)
{
  Gtk::TreeIter iter= _action_store->get_iter(path);
  Gtk::TreeRow row= *iter;
  MGRTValue value= row[_acolumns.value];
  bool flag;
  
  if (!value["changed"].asInt())
    return;
  
  if (!value["alterDirection"].isValid())
    flag= true;
  else
    flag= !value["alterDirection"].asInt();

  value.set("alterDirection", flag);

  set_action_text(row, value);
}
