/* Copyright (C) 2002 MySQL AB & Jorge del Conde

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU Library General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.
    
  You should have received a copy of the GNU Library General Public
  License along with this library; if not, write to the Free
  Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
  MA 02111-1307, USA 
*/

#include "CTableTools.h"
#include <qmessagebox.h>
#include <qpopmenu.h>
#include <qpushbutton.h>
#include "CConfig.h"
#include "Config.h"

CTableTools::CTableTools(int type, QPtrList<ToolOptions> *options, QWidget* parent, const char* name, QWidgetStack * w, CMySQLConnection *m, const QString &tableName, int serverid)
:CMyWindow( parent, name, WDestructiveClose, !g_isMDI), serverID(serverid)
{
  if (!name)
    setName("CTableTools");
  setSizePolicy( QSizePolicy( (QSizePolicy::SizeType)0, (QSizePolicy::SizeType)0, sizePolicy().hasHeightForWidth() ) );
  setMinimumSize( QSize( 280, 210 ) );
  setIcon(getPixmapIcon("applicationIcon"));
  
  CTableToolsLayout = new QVBoxLayout( this ); 
  Q_CHECK_PTR(CTableToolsLayout);
  CTableToolsLayout->setSpacing( 2 );
  CTableToolsLayout->setMargin( 2 );
  
  m_pTopFrame = new QFrame( this, "m_pTopFrame" );
  Q_CHECK_PTR(m_pTopFrame);
  m_pTopFrame->setFrameShape( QFrame::Box );
  m_pTopFrame->setFrameShadow( QFrame::Sunken );
  widgetStack = w;
  mysql = m;
  m_tableName = tableName;
  Options = options;
  Type = type;
  switch (Type)
  {
  case ANALYZE_TABLE: setCaption(tr("Analyze Table"));
    break;
  case CHECK_TABLE: setCaption(tr("Check Table"));
    break;
  case OPTIMIZE_TABLE: setCaption(tr("Optimize Table"));
    break;
  case REPAIR_TABLE: setCaption(tr("Repair Table"));
    break;
  case SHOW_CREATE_TABLE: setCaption(tr("Show Create Table"));
    break;
  }
 
  create();
  CTableToolsLayout->addWidget( m_pTopFrame );
  
  Layout14 = new QHBoxLayout;
  Q_CHECK_PTR(Layout14);
  Layout14->setSpacing( 6 );
  Layout14->setMargin( 0 );

  PushButton4 = new QPushButton( this, "PushButton4" );
  Q_CHECK_PTR(PushButton4);
  PushButton4->setPixmap(getPixmapIcon("contextHelpIcon"));
  Layout14->addWidget( PushButton4 );

  QSpacerItem* spacer = new QSpacerItem( 20, 20, QSizePolicy::Expanding, QSizePolicy::Minimum );
  Q_CHECK_PTR(spacer);
  Layout14->addItem( spacer );
  
  m_pExecutePushButton = new QPushButton( this, "m_pExecutePushButton" );
  Q_CHECK_PTR(m_pExecutePushButton);
  m_pExecutePushButton->setMinimumSize( QSize( 60, 0 ) );
  m_pExecutePushButton->setMaximumSize( QSize( 60, 32767 ) );
  m_pExecutePushButton->setText(tr("&Execute", "" ) );    
  Layout14->addWidget( m_pExecutePushButton );
  
  m_pCancelPushButton = new QPushButton( this, "m_pCancelPushButton" );
  Q_CHECK_PTR(m_pCancelPushButton);
  m_pCancelPushButton->setMinimumSize( QSize( 60, 0 ) );
  m_pCancelPushButton->setMaximumSize( QSize( 60, 32767 ) );
  m_pCancelPushButton->setText(tr("&Cancel", "" ) );
  
  Layout14->addWidget( m_pCancelPushButton );
  CTableToolsLayout->addLayout( Layout14 );
  
  if (g_isMDI)
    move(mapFromGlobal(QCursor::pos()).x() - (int)width() / 2, mapFromGlobal(QCursor::pos()).y() - (int)height() / 2);
  else
    move(QCursor::pos().x() - (int)width() / 2, QCursor::pos().y() - (int)height() / 2);
  init();
  myResize(280, 210);
}

CTableTools::~CTableTools()
{
  delete Options;
}

bool CTableTools::inRange(int res)
{
  return ((res >= ANALYZE_TABLE) && (res <= SHOW_CREATE_TABLE));
}

void CTableTools::create()
{
  m_pTopFrameLayout = new QVBoxLayout( m_pTopFrame );
  Q_CHECK_PTR(m_pTopFrameLayout);
  m_pTopFrameLayout->setSpacing( 4 );
  m_pTopFrameLayout->setMargin( 8 );
  
  tablesListBox = new CTablesListBox( m_pTopFrame, "tablesListBox", mysql, m_tableName);
  Q_CHECK_PTR(tablesListBox);
  m_pTopFrameLayout->addWidget( tablesListBox);
  
  if (!Options->isEmpty())
  {
    pOptions = new QGroupBox( m_pTopFrame, "pOptions" );
    Q_CHECK_PTR(pOptions);
    pOptions->setTitle(tr("Options", "" ) );
    pOptions->setColumnLayout(0, Qt::Vertical );
    pOptions->layout()->setSpacing( 0 );
    pOptions->layout()->setMargin( 4 );
    
    pOptionsLayout = new QGridLayout(pOptions->layout() );
    Q_CHECK_PTR(pOptionsLayout);
    pOptionsLayout->setAlignment( Qt::AlignTop );
    pOptionsLayout->setSpacing( 2 );
    pOptionsLayout->setMargin( 2 );
    
    Layout = new QGridLayout;
    Q_CHECK_PTR(Layout);
    Layout->setSpacing( 6 );
    Layout->setMargin( 0 );
    
    ToolOptions *opt;
    int col=0, row=0;
    for (opt = Options->first(); opt; opt = Options->next())
    {
      opt->checkBox = new QCheckBox(pOptions);
      Q_CHECK_PTR(opt->checkBox);
      opt->checkBox->setText(opt->Display);        
      Layout->addWidget(opt->checkBox, row, col++);
      if (col >= 3)
      {
        col = 0;
        row++;
      }
    }
    pOptionsLayout->addLayout(Layout, 0, 0 );
    m_pTopFrameLayout->addWidget(pOptions );
  }
}

void CTableTools::processMenu(QWidgetStack * w, CMySQLConnection *m, int res, const QString & tableName, int serverid)
{ 
  QPtrList<ToolOptions> *options = new QPtrList<ToolOptions>;
  Q_CHECK_PTR(options);
  switch (res)  //SETS OPTIONS.  Currently, only CHECK and REPAIR have options.
  {
  case CHECK_TABLE:	{
    options->setAutoDelete(true);
    ToolOptions *quick = new ToolOptions;
    Q_CHECK_PTR(quick);
    quick->Display = "Quick";
    quick->Value = "QUICK";
    options->append(quick);
    
    ToolOptions *fast = new ToolOptions;
    Q_CHECK_PTR(fast);
    fast->Display = "Fast";
    fast->Value = "FAST";
    options->append(fast);
    
    ToolOptions *medium = new ToolOptions;
    Q_CHECK_PTR(medium);
    medium->Display = "Medium";
    medium->Value = "MEDIUM";
    options->append(medium);
    
    ToolOptions *extended = new ToolOptions;
    Q_CHECK_PTR(extended);
    extended->Display = "Extended";
    extended->Value = "EXTENDED";
    options->append(extended);
    
    ToolOptions *changed = new ToolOptions;
    Q_CHECK_PTR(changed);
    changed->Display = "Changed";
    changed->Value = "CHANGED";
    options->append(changed);
    break;
                    }
    
  case REPAIR_TABLE:	{
    options->setAutoDelete(true);      
    ToolOptions *quick = new ToolOptions;
    Q_CHECK_PTR(quick);
    quick->Display = "Quick";
    quick->Value = "QUICK";
    options->append(quick);
    
    ToolOptions *extended = new ToolOptions;
    Q_CHECK_PTR(extended);
    extended->Display = "Extended";
    extended->Value = "EXTENDED";
    options->append(extended);
    break;
                      }
  }    
  CTableTools * c = new CTableTools(res, options, (QWidget *)g_WorkSpace, 0, w, m, tableName, serverid);
  Q_CHECK_PTR(c);
  myShowWindow(c);
}

void CTableTools::execute()
{
  QString selectedTables = tablesListBox->selectedTables();
  if (selectedTables.isNull())
    QMessageBox::warning(0,tr("Error"), tr("You must select at least one Table."));
  else
  {    
    if ((Type >= ANALYZE_TABLE) && (Type <= REPAIR_TABLE))
    {
      QString qry;
      switch (Type)
      {
      case ANALYZE_TABLE: qry = "ANALYZE TABLE";
        break;
      case CHECK_TABLE: qry = "CHECK TABLE";
        break;
      case OPTIMIZE_TABLE: qry = "OPTIMIZE TABLE";
        break;
      case REPAIR_TABLE: qry = "REPAIR TABLE";
        break;
      }
      qry += "\n" + selectedTables + "\n";
      if (!Options->isEmpty())
      {
        qry += " ";
        ToolOptions *opt;
        for (opt = Options->first(); opt; opt = Options->next())
          if (opt->checkBox->isChecked())
            qry += opt->Value + " ";
      }
      CQueryWindow *w = new CQueryWindow((QWidget *)g_WorkSpace, caption() + ": " +
        selectedTables, mysql, qry, RESULTS_PANEL, serverID);
      Q_CHECK_PTR(w);
      myShowWindow(w);
      w->setFocus();
      w->raise();
      w->ExecuteQuery();
    }
    else
    {
      switch (Type)
      {
      case SHOW_CREATE_TABLE:
        {
          QString sql = "";
          QStringList tables = tablesListBox->selectedTablesList();
          for (QStringList::Iterator i = tables.begin(); i != tables.end(); i++)
          {                         
            CMySQLQuery query(mysql);
            QString qry = "SHOW CREATE TABLE " + mysql->Quote(*i);
            if (query.exec(qry, false))
            {
              sql += "# Host: " + mysql->getHostName() + "\n" +
                "# Database: " + mysql->getDatabaseName() + "\n" +
                "# Table: '" + (*i) + "'\n" +
                "#";
              query.next();
              sql += "\n" + query.Row(1);
            }
            sql += "; \n\n";
          }                      
          CQueryWindow *w = new CQueryWindow((QWidget *)g_WorkSpace, tr("Show Create Table for") + ": " +
            tables.join(", "), mysql, sql, SQL_PANEL, serverID);
          Q_CHECK_PTR(w);
          myShowWindow(w);
          w->setFocus();
          w->raise();
        }
      }
    }
    close();  //Close this Dialog
  }
}

QPopupMenu * CTableTools::tableToolsMenu()
{
  QPopupMenu *p_toolsMenu = new QPopupMenu();
  Q_CHECK_PTR(p_toolsMenu);
  p_toolsMenu->insertItem(tr("Analyze Table"), ANALYZE_TABLE);
  p_toolsMenu->insertItem(tr("Check Table"), CHECK_TABLE);
  p_toolsMenu->insertItem(tr("Optimize Table"), OPTIMIZE_TABLE);
  p_toolsMenu->insertItem(tr("Repair Table"), REPAIR_TABLE);
  p_toolsMenu->insertSeparator();
  p_toolsMenu->insertItem(tr("Show Create"), SHOW_CREATE_TABLE);
  return p_toolsMenu;
}

void CTableTools::NoTablesLoaded()
{
  m_pExecutePushButton->setEnabled(false);
}

void CTableTools::init()
{
  QWhatsThis::add( (QWidget *)this,tr("Select one or more Tables.", "" ) );
  QWhatsThis::add( (QWidget *) m_pExecutePushButton,tr("Click to execute.", "" ) );
  QWhatsThis::add( (QWidget *) m_pCancelPushButton,tr("Close this Dialog without executing.", "" ) );
  
  connect(m_pCancelPushButton, SIGNAL(clicked()), this, SLOT(close()));
  connect(m_pExecutePushButton, SIGNAL(clicked()), this, SLOT(execute()));
  connect(PushButton4, SIGNAL(clicked()), this, SLOT(whatsThis()));
}



