/* Copyright (C) 2002 MySQL AB & Jorge del Conde

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU Library General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.
    
  You should have received a copy of the GNU Library General Public
  License along with this library; if not, write to the Free
  Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
  MA 02111-1307, USA 
*/

#include "CApplicationWindow.h"
#include "CSqlTable.h"
#include "CTextFieldEditor.h"
#include "CImageFieldEditor.h"
#include <qmessagebox.h>

privateEllipseBox::privateEllipseBox( QWidget* parent,  const char* name, WFlags fl )
: QWidget( parent, name, fl ), TableItem(0)
{
  if ( !name )
    setName( "privateEllipseBox" );
  Form1Layout = new QHBoxLayout( this, 0, 0, "Form1Layout"); 
  
  LineEdit = new privateEllipseBoxLineEdit(this, "LineEdit");
  LineEdit->setReadOnly(true);
  LineEdit->setFrame(false);
  Form1Layout->addWidget(LineEdit);
  
  EllipseButton = new QPushButton( this, "EllipseButton");
  EllipseButton->setText("...");
  EllipseButton->setAccel(0);
  Form1Layout->addWidget(EllipseButton);
  connect(EllipseButton, SIGNAL(clicked()), this, SLOT(Clicked()));
  connect(LineEdit, SIGNAL(doubleClicked(int)), this, SLOT(DoubleClicked(int)));
}

privateEllipseBox::~privateEllipseBox()
{
}

void privateEllipseBox::Clicked()
{
  emit buttonClicked(TableItem);
}

void privateEllipseBox::DoubleClicked(int button)
{  
  emit doubleClicked(button, TableItem);
}

void privateEllipseBox::setTableItem(QTableItem *i)
{
  TableItem = i;
}

void privateEllipseBox::setText(const QString &t)
{
  LineEdit->setText(t);
}

QString privateEllipseBox::text() const
{
  return LineEdit->text();
}

CEllipseTableItem::CEllipseTableItem( QTable *table, const QString &txt )
: QTableItem( table, WhenCurrent, txt ), isBinary(false), isConnected(false), isConnected2(false)
{
  rowOffset = 0;
  fieldLength = 0;
  fieldIndex = 0;
  fieldName = QString::null;
}

void CEllipseTableItem::setContentFromEditor( QWidget *w )
{
  if ( w->inherits( "privateEllipseBox" ) )      
    setText(((privateEllipseBox*)w)->text());  
}

void CEllipseTableItem::signalConnectButton(const QString & sig, QObject *rec, const QString & s)
{
  signal = sig;
  receiver = rec;
  slot = s;
  isConnected = true; 
}

void CEllipseTableItem::signalConnectItem(const QString & sig, QObject *rec, const QString & s)
{
  signal2 = sig;
  receiver2 = rec;
  slot2 = s;
  isConnected2 = true; 
}

QWidget *CEllipseTableItem::createEditor() const
{  
  ((CEllipseTableItem*)this )->eb = new privateEllipseBox( table()->viewport(), "privateEllipseBox");
  Q_CHECK_PTR(((CEllipseTableItem*)this )->eb);
  eb->setTableItem((QTableItem *)this);
  eb->setText(text());
  eb->setBackgroundColor( table()->viewport()->backgroundColor() );  
  if (isConnected)
    QObject::connect(eb, signal.latin1(), receiver, slot.latin1());
  if (isConnected2)
    QObject::connect(eb, signal2.latin1(), receiver2, slot2.latin1());

  //QObject::connect( eb, SIGNAL( toggled( bool ) ), table(), SLOT( doValueChanged() ) );  
  return eb;  
}

void CEllipseTableItem::paint( QPainter *p, const QColorGroup &cg, const QRect &cr, bool selected)
{
  p->fillRect(0, 0, cr.width(), cr.height(), selected ? cg.brush(QColorGroup::Highlight ) : cg.brush(QColorGroup::Base ));  
  p->setPen(selected ? cg.highlightedText() : cg.text());  
  p->drawText(2, 0, cr.width() - 4, cr.height(), wordWrap() ? (alignment() | WordBreak) : alignment(), text());
}

void CEllipseTableItem::setReadOnly(bool b)
{
  eb->LineEdit->setReadOnly(b);
}

int CEllipseTableItem::rtti() const
{
  return 2001;
}

void CEllipseTableItem::setFieldIndex(unsigned int idx)
{
  fieldIndex = idx;
}

void CEllipseTableItem::setFieldLength(unsigned int len)
{
  fieldLength = len;
}

void CEllipseTableItem::setRowOffset(unsigned int offset)
{
  rowOffset = offset;
}

void CEllipseTableItem::setBinaryField(bool b)
{
  isBinary = b;
}

void CEllipseTableItem::setFieldProperties(const QString &name, unsigned int idx, unsigned int offset)
{
  setFieldName(name);
  setFieldIndex(idx);
  setRowOffset(offset);
}

void CEllipseTableItem::setFieldName(const QString &name)
{
  fieldName = name;
}

QPoint CEllipseTableItem::EllipsePoint()
{
  return eb->EllipseButton->mapToGlobal(QPoint( eb->EllipseButton->pos().x() - eb->LineEdit->width(),
    eb->EllipseButton->pos().y() + eb->EllipseButton->height()));
}

CCheckTableItem::CCheckTableItem( QTable *table, const QString &txt )
: QTableItem(table, QTableItem::WhenCurrent, txt), checked(false)
{  
}

QWidget *CCheckTableItem::createEditor() const
{
  ((CCheckTableItem*)this )->cb = new QCheckBox(table()->viewport(), "CCheckTableItem");
  Q_CHECK_PTR(((CCheckTableItem*)this )->cb);
  cb->setChecked(checked);
  cb->setText(text());
  cb->setBackgroundColor(table()->viewport()->backgroundColor() );
  QObject::connect(cb, SIGNAL(toggled(bool)), table(), SLOT(doValueChanged()));
  return cb;
}

void CCheckTableItem::setContentFromEditor( QWidget *w )
{
  if (w->inherits("QCheckBox"))
  {
    QCheckBox *cb = (QCheckBox*)w;
    checked = cb->isChecked();
  }
}

void CCheckTableItem::paint(QPainter *p, const QColorGroup &cg, const QRect &cr, bool selected)
{
  p->fillRect( 0, 0, cr.width(), cr.height(),
    selected ? cg.brush( QColorGroup::Highlight )
    : table()->viewport()->backgroundBrush() );
  
  int w = cr.width();
  int h = cr.height();
  QSize sz = QSize( table()->style().pixelMetric( QStyle::PM_IndicatorWidth ),
    table()->style().pixelMetric( QStyle::PM_IndicatorHeight ) );
  QColorGroup c( cg );
  c.setBrush( QColorGroup::Background, c.brush( QColorGroup::Base ) );
  table()->style().drawPrimitive( QStyle::PE_Indicator, p,
    QRect( 0, ( cr.height() - sz.height() ) / 2, sz.width(), sz.height() ), c,
    isChecked() ? QStyle::Style_On : QStyle::Style_Off );
  int x = sz.width() +6;
  w = w - x - 2;
  if ( selected )
    p->setPen( cg.highlightedText() );
  else
    p->setPen( cg.text() );
  p->drawText( x, 0, w, h, wordWrap() ? ( alignment() | WordBreak ) : alignment(), text() );
}

void CCheckTableItem::setChecked( bool b )
{
  checked = b;
  table()->updateCell( row(), col() );
}

bool CCheckTableItem::isChecked() const
{
  return checked;
}

int CCheckTableItem::rtti() const
{
  return 1001;
}

privateCheckListItem::privateCheckListItem(QListView * parent, const QString & text, int columnid)
: QCheckListItem(parent, text, CheckBox)
{
  created = false;
  setOn (true);
  columnID = columnid;
}

void privateCheckListItem::stateChange (bool s)
{  
  if (created)  
    ((privateListView *)listView())->CheckBoxClicked(columnID, s);
  else
    created = true;  
}


privateCSqlTableColumns::privateCSqlTableColumns(QWidget * parent, const char * name, WFlags f)
:QDockWindow(OutsideDock, parent, name, f)
{
  setMovingEnabled (true);
  setResizeEnabled (true);
  setHorizontallyStretchable (true);
  setVerticallyStretchable (true);
  setOpaqueMoving (false);
  setCloseMode(CMessageWindow::Always);

  setCaption(tr("Table Columns" ));

  Columns = new privateListView( this, "Columns" );  
  Columns->addColumn( tr("Columns" ) );
  Columns->header()->setResizeEnabled( FALSE, Columns->header()->count() - 1 ); 
  Columns->setFrameShape( QListView::Box );
  Columns->setLineWidth( 1 );
  Columns->setShowSortIndicator( TRUE );
  Columns->setResizeMode( QListView::AllColumns );
  QWhatsThis::add(Columns, tr("Select the columns that you want visible in the grid." ) );
  setWidget(Columns);
}

privateListView::privateListView( QWidget * parent, const char * name, WFlags f)
:QListView(parent, name, f)
{
  connect(this, SIGNAL(contextMenuRequested(QListViewItem *, const QPoint &, int)), this, SLOT(displayMenu( QListViewItem *, const QPoint &, int )));
}

void privateListView::displayMenu( QListViewItem *, const QPoint &pos, int )
{
  QPopupMenu *p_itemMenu = new QPopupMenu();
  Q_CHECK_PTR(p_itemMenu);  
  p_itemMenu->insertItem(getPixmapIcon("checkedIcon"), tr("Check All"), 1);
  p_itemMenu->insertItem(getPixmapIcon("uncheckedIcon"), tr("Clear All"), 2);  
  int res = p_itemMenu->exec(pos);
  delete p_itemMenu;
  bool b = (res == 1);
  QListViewItemIterator it(this);
  for ( ; it.current(); ++it)
    ((privateCheckListItem *)it.current())->setOn(b);
}

void privateListView::CheckBoxClicked(int c, bool s)
{  
  emit showColumn(c, s);  
}

CSqlTable::CSqlTable(QWidget * parent, const char *name) : QTable(parent, name)
{
  Query = NULL;
  init();
}

CSqlTable::CSqlTable(QWidget * parent, CMySQLConnection *conn, const QString & query) : QTable(parent)
{
  Query = NULL;
  setName("CSqlTable");
  setDatabaseConnection(conn);
  setQuery(query);
  init();
}

CSqlTable::CSqlTable(QWidget * parent, CMySQLConnection *conn) : QTable(parent)
{
  Query = NULL;
  setName("CSqlTable");
  setDatabaseConnection(conn);
  init();
}

CSqlTable::~CSqlTable()
{ 
  closeEditors();
  delete EditorList;
  enableColumnsWindow(false);
  delete Query;  
}

void CSqlTable::setNumCols(int r)
{    
  if (!headerSort.isEmpty())
    headerSort.clear();
  headerSort.resize(r);
  for (int i = 0; i < r; i++)
    headerSort.insert(i, new bool (true));    
  QTable::setNumCols(r);  
}

void CSqlTable::setSaveTitle(const QString &title)
{
  saveTitle = title;
  hasSaveTitle = true;
}

QString CSqlTable::getSaveTitle()
{
  QString ret = QString::null;
  if (hasSaveTitle)
  {
    ret = "# " + saveTitle + getDatabaseConnection()->getLineTerminator(true);
    ret += "# " + tr("Connection") + ": " + getDatabaseConnection()->getConnectionName() + getDatabaseConnection()->getLineTerminator(true);
    ret += "# " + tr("Host") + ": " + getDatabaseConnection()->getHostName() + getDatabaseConnection()->getLineTerminator(true);
    ret += "# " + tr("Saved") + ": " + QDateTime::currentDateTime().toString("yyyy-MM-dd hh:mm:ss") + getDatabaseConnection()->getLineTerminator(true);
    ret += "#" + getDatabaseConnection()->getLineTerminator(true);
  }
  return ret;
}

QString CSqlTable::getSaveContents()
{
  QString contents = getSaveTitle();
  QString line = QString::null;
  emit executing(true);
  for (unsigned long c = 0; c <= (unsigned long) numCols() -1; c++)
  {
    if (numCols() <= 0)
      break;
    if (headerDict.find(c) == NULL)
    {
      line += getDatabaseConnection()->getFieldEncloser(true) + horizontalHeader()->label(c) + 
        getDatabaseConnection()->getFieldEncloser(true) + getDatabaseConnection()->getFieldSeparator(true);
    }    
  }  
  line = line.mid(0,line.length() - getDatabaseConnection()->getFieldSeparator(true).length());
  contents += line + getDatabaseConnection()->getLineTerminator(true);
    
  for (unsigned long r = 0; r <= (unsigned long) numRows() - 1; r++)
  {
    QString line = QString::null;
    if (numRows() <= 0)
        break;
    for (unsigned long c = 0; c <= (unsigned long) numCols() - 1 ; c++)
    {
      if (numCols() <= 0)
        break;
      if (headerDict.find(c) == NULL)
      {
        QString txt = ((text(r,c).isEmpty()) ? getDatabaseConnection()->getReplaceEmpty(true) : text(r,c));      
        line += getDatabaseConnection()->getFieldEncloser(true) + txt + getDatabaseConnection()->getFieldEncloser(true)  
        + getDatabaseConnection()->getFieldSeparator(true);          
      }
    }
    line = line.mid(0,line.length() - getDatabaseConnection()->getFieldSeparator(true).length());
    contents += line + getDatabaseConnection()->getLineTerminator(true);
  }
  emit executing(false);
  return contents;
}

void CSqlTable::saveQueryContents()
{
  if (isExecuting || Query->isNull())
    return;
  isExecuting = true;
  tmpFileName = getSaveName(tmpFileName, "txt",
			    tr("Text Files (*.txt);;All Files(*.*)"));  
  if (tmpFileName.isEmpty())
    return;
  
  QFile file( tmpFileName );    
  if (file.exists() && g_confirmCritical)
    if ((QMessageBox::warning(0,
			      qApp->translate("SaveToFile", "Replace File"),
			      qApp->translate("SaveToFile",
	                      "The specified file name already exists.\
\nDo you want to replace it ?"),
      QMessageBox::Yes, QMessageBox::No) != QMessageBox::Yes))
      return;
  if ( !file.open( IO_WriteOnly ) )
  {
    if (getDatabaseConnection()->messagePanel)
      getDatabaseConnection()->messagePanel->Critical(tr("An error occurred while saving the file"));
    return;
  } 
  emit executing(true);
  QTextStream ts( &file );  
  ts << getSaveTitle();
  ts << "# " << tr("Query") << ":" << getDatabaseConnection()->getLineTerminator(true);
  ts << "# " << m_Query.replace(QRegExp("\n"), getDatabaseConnection()->getLineTerminator(true) + "# ");
  ts << getDatabaseConnection()->getLineTerminator(true) << "#" << getDatabaseConnection()->getLineTerminator(true);
  QString line = QString::null;    
  Query->dataSeek(0);
  setCursor (Qt::waitCursor);
  CMySQLConnection* con = getDatabaseConnection();
  QString encl = con->getFieldEncloser(true),
    sep = con->getFieldSeparator(true),
    term = con->getLineTerminator(true),
    repl_empty = con->getReplaceEmpty(true);
  
  unsigned int num_fields = Query->numFields();
  memset(Query->field_mask, 0, num_fields);
  Query->last_field_ind = 0;

  for (unsigned int i = 0; i < num_fields; i++)
  {
    if (headerDict.find(i) == NULL)
    {
      Query->field_mask[i] = QUERY_FIELD_HIDDEN;
      Query->last_field_ind++;
    }
  }

  for (unsigned int j = 0; j < num_fields; j++)
  {
    if (Query->field_mask[j] == QUERY_FIELD_HIDDEN)
    {   
      ts << encl << Query->Fields[j].name << encl;
      if (j < Query->last_field_ind)
        ts << sep;
      else
        break;
    }
  }
  ts << term;

  if ((numRows() <= 0) || (numCols() <= 0))
  {
    emit executing(false);
    return;
  }
  while (Query->next(true, false))
    Query->export_current_row(ts,sep,encl,term,repl_empty);
  file.close();
  setCursor(Qt::ArrowCursor);
  isExecuting = false;
  emit executing(false);
  if (getDatabaseConnection()->messagePanel)      
    getDatabaseConnection()->messagePanel->Information(qApp->translate("saveToFile", "Successfully saved:") + " " + tmpFileName);  
}

void CSqlTable::save()
{
  saveToFile(tmpFileName, "txt", tr("Text Files (*.txt);;All Files(*.*)" ), getSaveContents(), getDatabaseConnection()->messagePanel);
}

void CSqlTable::headerPressed(int)
{
  qDebug("headerPressed()");
  isExecuting = true;
}

void CSqlTable::headerReleased(int)
{
  qDebug("headerReleased()");
  isExecuting = false;
}

void CSqlTable::processMenu(int res, int, int)
{
  switch (res)
  {
    case MENU_SAVE:
      saveQueryContents();
      break;
    case MENU_CLEAR_GRID:
      reset();
      break;
    case MENU_REFRESH:
      exec();
      break;
    case MENU_COLUMNS_WINDOW:
      if (columnsWindowEnabled)
      {
        if (!columnWindowFirstTimeShown)
        {
          ColumnsWindow->move(QCursor::pos().x(), QCursor::pos().y());
          columnWindowFirstTimeShown = true;
        }
        ColumnsWindow->show();        
      }
      break;
  }
}

int CSqlTable::displayMenu(const QPoint &pos)
{
  QPopupMenu *p_itemMenu = new QPopupMenu();  
  p_itemMenu->insertItem(getPixmapIcon("chooseFieldsIcon"), tr("Choose Fields"), MENU_COLUMNS_WINDOW);

  if (columnsWindowEnabled)
    if (ColumnsWindow->isVisible())
      p_itemMenu->setItemEnabled (MENU_COLUMNS_WINDOW, false);

  p_itemMenu->insertSeparator();  
  p_itemMenu->insertItem(getPixmapIcon("saveGridResultsIcon"), tr("Save"), MENU_SAVE);
  p_itemMenu->insertSeparator();  
  p_itemMenu->insertItem(getPixmapIcon("refreshTablesIcon"), tr("Refresh"), MENU_REFRESH);
  p_itemMenu->insertSeparator();  
  p_itemMenu->insertItem(getPixmapIcon("clearGridIcon"), tr("Clear Grid"), MENU_CLEAR_GRID);
  p_itemMenu->setItemEnabled (MENU_CLEAR_GRID, !isExecuting);
  
  int res = p_itemMenu->exec(pos);
  delete p_itemMenu;	  
  return res;  
}

void CSqlTable::ContextMenuRequested(int row, int col, const QPoint & pos)
{	
  processMenu(displayMenu(pos), row, col);
}

struct SortableTableItem
{
    QTableItem *item;
};

#if defined(Q_C_CALLBACKS)
extern "C" {
#endif
  
#ifdef Q_OS_TEMP
  static int _cdecl cmpTableItems(const void *n1, const void *n2)
#else
  static int cmpTableItems(const void *n1, const void *n2)
#endif
  {
    if (!n1 || !n2)
      return 0;
    
    SortableTableItem *i1 = (SortableTableItem *)n1;
    SortableTableItem *i2 = (SortableTableItem *)n2;
    
    return i1->item->key().localeAwareCompare(i2->item->key());
  }
  
#ifdef Q_OS_TEMP
  static int _cdecl cmpIntTableItems(const void *n1, const void *n2)
#else
  static int cmpIntTableItems(const void *n1, const void *n2)
#endif
  {    
    if (!n1 || !n2)
      return 0;
    
    SortableTableItem *i1 = (SortableTableItem *)n1;
    SortableTableItem *i2 = (SortableTableItem *)n2;
    
    bool ok;
    double v1 = i1->item->key().toDouble(&ok);
    if (!ok)
      return cmpTableItems(n1, n2);

    double v2 = i2->item->key().toDouble(&ok);
    if (!ok)
      return cmpTableItems(n1, n2);

    if (v1 < v2)
      return -1;
    else
      if (v1 > v2)
        return 1;
      else
        return 0;
  }
  
#if defined(Q_C_CALLBACKS)
}
#endif

void CSqlTable::sortColumn(int col, bool ascending, bool wholeRows)
{  
  int filledRows = 0, i;
  for (i = 0; i < numRows(); ++i)
  {
    QTableItem *itm = item(i, col);
    if (itm)
      filledRows++;
  }
  
  if (!filledRows)
    return;
  
  SortableTableItem *items = new SortableTableItem[ filledRows ];
  int j = 0;
  for (i = 0; i < numRows(); ++i)
  {
    QTableItem *itm = item(i, col);
    if (!itm)
      continue;
    items[ j++ ].item = itm;
  }
  
  if (Query != NULL)
    qsort(items, filledRows, sizeof(SortableTableItem), (IS_NUM(Query->Fields[col].type) ? cmpIntTableItems : cmpTableItems));
  else
    qsort(items, filledRows, sizeof(SortableTableItem), cmpTableItems);
  
  bool updatesEnabled = isUpdatesEnabled();
  bool viewportUpdatesEnabled = viewport()->isUpdatesEnabled();
  setUpdatesEnabled(false);
  viewport()->setUpdatesEnabled(false);
  for (i = 0; i < numRows(); ++i)
  {
    if (i < filledRows)
    {
      if (ascending)
      {
        if (items[ i ].item->row() == i)
          continue;
        if (wholeRows)
          swapRows(items[ i ].item->row(), i);
        else
          swapCells(items[ i ].item->row(), col, i, col);
      }
      else
      {
        if (items[ i ].item->row() == filledRows - i - 1)
          continue;
        if (wholeRows)
          swapRows(items[ i ].item->row(), filledRows - i - 1);
        else
          swapCells(items[ i ].item->row(), col,
			       filledRows - i - 1, col);
      }
    }
  }
  setUpdatesEnabled(updatesEnabled);
  viewport()->setUpdatesEnabled(viewportUpdatesEnabled);
  
  if (!wholeRows)
    repaintContents(columnPos(col), contentsY(),
			 columnWidth(col), visibleHeight(), false);
  else
    repaintContents(contentsX(), contentsY(),
			 visibleWidth(), visibleHeight(), false);
  
  delete [] items;
}

void CSqlTable::Sort(int col)
{  
  if (isExecuting)
    return;
  setCursor (Qt::waitCursor);
  horizontalHeader()->setSortIndicator (col, *headerSort[col]);
  sortColumn (col, *headerSort[col], true);
  *headerSort[col] = !(*headerSort[col]);
  setCursor (Qt::arrowCursor);
}

void CSqlTable::setQuery(const QString & query)
{
  m_Query = query;
}

void CSqlTable::cancelQuery()
{
  m_cancel = true;  
}

void CSqlTable::setDatabaseConnection(CMySQLConnection *conn)
{
  m_pDatabaseConnection = conn;
  if (Query)
  {
    Query->result_in_use = 0;
    delete Query;
  }
  Query = new CMySQLQuery(conn);
}


void CSqlTable::reset()
{
  if (isExecuting)
    return;
  isExecuting = true;
  emit executing(true);  
  if (Query != NULL)
  {
    Query->result_in_use = 0;
    Query->freeResult();
  }  
  m_cancel = false;
  closeEditors();
  if (!headerSort.isEmpty())
    headerSort.clear();
  clearCellWidget(currentRow(), currentColumn());
  ensureVisible(0, 0);
  verticalScrollBar()->setValue(0);  
  setNumRows(0);  
  setNumCols(0);
  for (int i = 0; i <= horizontalHeader()->count(); i++)  
    horizontalHeader()->setLabel(i, QString::null);  
  if (sorting())
    horizontalHeader()->setSortIndicator( -1 );
  if (columnsWindowEnabled)
    ColumnsWindow->Columns->clear();
  emit executing(false);
  isExecuting = false;
}

void CSqlTable::exec(const QString & qry, bool printAffected)
{
  if (isExecuting)
    return;  
  reset();
  isExecuting = true;  
  emit startquery();
  setCursor (Qt::waitCursor);
  setQuery(qry);  
  if (Query->exec(m_Query, printAffected))
  {
    emit executing(true);
    unsigned long nFields = Query->numFields();
    setNumRows(Query->numRows());    
    setNumCols(nFields);
    headerDict.clear();
    headerDict.resize(nFields);
    QString icon;    
    for (unsigned int i = 0; i < nFields; i++)
    {	      
      if (IS_PRI_KEY(Query->Fields[i].flags))
        icon = "pkIcon";        
      else
        if (IS_MUL_KEY(Query->Fields[i].flags))
          icon = "mulIcon";          
        else
          if (IS_UNI_KEY(Query->Fields[i].flags))
            icon = "uniIcon";            
          else
            icon = "nothingIcon";            
          horizontalHeader()->setLabel(i,getPixmapIcon(icon), QString::fromLatin1(Query->Fields[i].name));
          if (columnsWindowEnabled)
            new privateCheckListItem(ColumnsWindow->Columns, Query->Fields[i].name, i);
          
          if (hasAutoFieldEditors)
            switch (Query->Fields[i].type)
          {
           case FIELD_TYPE_TINY_BLOB:
           case FIELD_TYPE_MEDIUM_BLOB:
           case FIELD_TYPE_LONG_BLOB:
           case FIELD_TYPE_BLOB :
             if (myReadOnly)
               setColumnReadOnly (i, false);
             break;
           default:
             setColumnReadOnly (i, myReadOnly);
             break;          
          }
          qApp->processEvents();
    }
    unsigned long j = 0;   
    Query->result_in_use = true;
    while (Query->next(true, false))
    {
      if (m_cancel)
        Query->cancel();
      else
      {
        for (unsigned int i = 0; i < nFields; i++)
        {
          char* text = Query->row[i];
          unsigned int text_len;
          if (!text) text = "NULL";
          text_len = strlen(text);
          if ((maxCellTextSize > 0) && (text_len > (uint)maxCellTextSize))
          {
            char* p = text + text_len - 1;
            *p-- = '.'; *p-- = '.'; *p-- = '.';
          }
          QString tmp(text);
          
          if (hasAutoFieldEditors)
          {
            switch (Query->Fields[i].type)
            {
            case FIELD_TYPE_TINY_BLOB:
            case FIELD_TYPE_MEDIUM_BLOB:
            case FIELD_TYPE_LONG_BLOB:
            case FIELD_TYPE_BLOB:
              {
                CEllipseTableItem *x = new CEllipseTableItem(this, text);
                setItem (j, i, x);
                x->setFieldProperties(Query->Fields[i].name, i, j);
                x->setBinaryField(IS_BINARY_FIELD(Query->Fields[i].flags));
                x->setFieldLength(Query->getFieldLength(i));
                x->signalConnectButton(SIGNAL(buttonClicked(QTableItem *)), this,
                  SLOT(EllipseButtonClicked(QTableItem *)));
                x->signalConnectItem(SIGNAL(doubleClicked(int, QTableItem *)), this,
                  SLOT(EllipseItemDoubleClicked(int, QTableItem *)));
                break;
              }            
            default: setText(j,i, text);
              break;
            }
          }
          else
            setText(j,i,text);
        }
        j++;
      }
    }
    emit executing(false);
  }
  setCursor(Qt::ArrowCursor);
  emit endquery();
  isExecuting = false;
}

void CSqlTable::OpenTextEditor(QTableItem *x)
{  
  CTextFieldEditor *TextEditor = new CTextFieldEditor(g_WorkSpace, EditorList, "CTextFieldEditor", m_pDatabaseConnection, myReadOnly);
  TextEditor->setCaption(tr("Text Editor - Column ") + "'" + ((CEllipseTableItem *)x)->getFieldName() + "' Row " + QString::number(currentRow() + 1));
  TextEditor->setText(Query->getFieldValue(((CEllipseTableItem *)x)->getRowOffset(), ((CEllipseTableItem *)x)->getFieldIndex()));
  myShowWindow(TextEditor);
}

void CSqlTable::OpenImageEditor(QTableItem *x)
{
  CImageFieldEditor *ImageEditor = new CImageFieldEditor(g_WorkSpace, EditorList, "CImageFieldEditor", m_pDatabaseConnection, myReadOnly);
  ImageEditor->setCaption(tr("Image Editor - Column ") + "'" + ((CEllipseTableItem *)x)->getFieldName() + "' Row " + QString::number(currentRow() + 1));
  ImageEditor->loadImage(((CEllipseTableItem *)x)->getFieldName(),
    Query->getFieldValue(((CEllipseTableItem *)x)->getRowOffset(), ((CEllipseTableItem *)x)->getFieldIndex()),
    ((CEllipseTableItem *)x)->getFieldLength());
  ImageEditor->show();  
}

void CSqlTable::DoubleClicked (int row, int col, int button, const QPoint &)
{
  if (button == 1)
  {
    switch (Query->Fields[col].type)
    {
    case FIELD_TYPE_TINY_BLOB:
    case FIELD_TYPE_MEDIUM_BLOB:
    case FIELD_TYPE_LONG_BLOB:
    case FIELD_TYPE_BLOB:
      break;
    default:
      {
        CTextFieldEditor *TextEditor = new CTextFieldEditor(g_WorkSpace, EditorList, "CTextFieldEditor", m_pDatabaseConnection, myReadOnly);
        TextEditor->setCaption(tr("Text Editor - Column ") + "'" + Query->Fields[col].name + "' Row " + QString::number(row + 1));
        TextEditor->setText(text(row, col));
        myShowWindow(TextEditor);
      }
      break;
    }
  }
}

void CSqlTable::P_saveToFile(QTableItem *x)
{
  CEllipseTableItem * item = (CEllipseTableItem *)x;
  QString tmp, ext = QString::null;
  QString ext_desc = tr("All Files(*.*)");
  bool writeBinary = item->isBinaryField();
  if (!writeBinary)
  {    
    ext = "txt";
    ext_desc = tr("Text Files (*.txt)") + ";;" + ext_desc;
  }  
  saveToFile(tmp, ext, ext_desc, Query->getFieldValue(item->getRowOffset(), item->getFieldIndex()), getDatabaseConnection()->messagePanel, writeBinary, item->getFieldLength());
}

void CSqlTable::EllipseItemDoubleClicked(int button, QTableItem *x)
{
  if ((button != 1) && (x != 0))
    return;
  if (((CEllipseTableItem *)x)->isBinaryField())
    OpenImageEditor(x);
  else
    OpenTextEditor(x);
}

void CSqlTable::EllipseButtonClicked(QTableItem *x)
{
  QPopupMenu *p_itemMenu = new QPopupMenu();  
  QPopupMenu *p_openAsMenu = new QPopupMenu();
  if (((CEllipseTableItem *)x)->isBinaryField())
  {
    p_openAsMenu->insertItem(tr("Image"), MENU_OPEN_IMAGE);
    p_openAsMenu->insertItem(tr("Text"), MENU_OPEN_TEXT);
  }
  else
  {
    p_openAsMenu->insertItem(tr("Text"), MENU_OPEN_TEXT);
    p_openAsMenu->insertItem(tr("Image"), MENU_OPEN_IMAGE);
  }  
  p_openAsMenu->setItemEnabled (MENU_OPEN_IMAGE, ((CEllipseTableItem *)x)->isBinaryField());
  
  p_itemMenu->insertItem(getPixmapIcon("openIcon"), tr("Open As"), p_openAsMenu);
  
  p_itemMenu->insertSeparator();  
  p_itemMenu->insertItem(getPixmapIcon("loadIcon"), tr("Load from File"), MENU_LOAD);
  p_itemMenu->setItemEnabled (MENU_LOAD, !myReadOnly);  
  p_itemMenu->insertItem(getPixmapIcon("saveIcon"), tr("Save to File"), MENU_SAVE);
  int res = p_itemMenu->exec(((CEllipseTableItem *)x)->EllipsePoint());
  delete p_itemMenu;
  delete p_openAsMenu;

  switch (res) {
  case MENU_OPEN_TEXT:
    OpenTextEditor(x);
    break;
  case MENU_OPEN_IMAGE:
    OpenImageEditor(x);
    break;
  case MENU_LOAD:  //TODO ... needs to be coded when MyCC supports inplace editing for fields
    break;
  case MENU_SAVE:
    P_saveToFile(x);
    break;
  }  
}

void CSqlTable::exec(bool printAffected)
{
  if (!m_Query.isEmpty())
    exec(m_Query, printAffected);
}

void CSqlTable::ShowColumn(int c, bool s)
{
  if (s)
  {
    Headers *h = headerDict.find(c);
    if (h != NULL)
    {
      QIconSet icon(h->icon);
      QString label(h->label);
      int size = h->size;
      horizontalHeader()->setLabel(c, icon, label, size);    
      headerDict.remove(c);
      showColumn(c);
    }
  }
  else
  {    
    Headers *h = new Headers(horizontalHeader()->sectionSize(c), horizontalHeader()->label(c), *horizontalHeader()->iconSet(c));    
    if (headerDict.find(c) != NULL)
      headerDict.remove(c);
    headerDict.insert(c, h);    
    horizontalHeader()->setLabel(c, getPixmapIcon("nothingIcon"), QString::null, 0);
    hideColumn(c);
  }
}

void CSqlTable::closeEditors(bool force)
{
  if (!autoCloseEditors && !force)
    return;
  if (!EditorList->isEmpty())
  {
    CFieldEditorWindow *e;
    QPtrListIterator<CFieldEditorWindow> it(*EditorList);
    while ((e = it.current()) != 0 )
    {
      ++it;
      e->close();
      delete e;
    }
    EditorList->clear();
#ifdef DEBUG
    qDebug("closeEditors() - Done");
#endif
  }
}

void CSqlTable::enableColumnsWindow(bool b)
{  
  if (b && (ColumnsWindow == 0))
  {
    ColumnsWindow = new privateCSqlTableColumns(0, "ColumnsWindow", WStyle_StaysOnTop);
    connect(ColumnsWindow->Columns, SIGNAL(showColumn(int, bool)), this, SLOT (ShowColumn(int, bool)));
    columnsWindowEnabled = true;
  }
  else  
    if (!b && (ColumnsWindow != 0))
    {
      ColumnsWindow->close();
      disconnect(ColumnsWindow->Columns, SIGNAL(showColumn(int, bool)), this, SLOT (ShowColumn(int, bool)));
      delete ColumnsWindow;
      ColumnsWindow = 0;
      columnsWindowEnabled = false;
    }  
}

bool CSqlTable::eventFilter( QObject *o, QEvent *e )
{  
  if (strcmp(o->className(),"QTableHeader") == 0)
  {    
    if ( e->type() == QEvent::MouseButtonPress)   
      isExecuting = true;    
    else
      if ( e->type() == QEvent::MouseButtonRelease)      
        isExecuting = false;      
  }
  return QTable::eventFilter( o, e );  
}

void CSqlTable::init()
{
  autoCloseEditors = true;
  horizontalHeader()->installEventFilter(this);
  verticalHeader()->installEventFilter(this);
  maxCellTextSize = 0;
  ColumnsWindow = 0;
  columnsWindowEnabled = false;
  isExecuting = false;
  EditorList = new QPtrList<CFieldEditorWindow>();
  headerSort.setAutoDelete(true);
  headerDict.setAutoDelete(true);
  hasAutoFieldEditors = false;
  myReadOnly = false;
  m_cancel = false;
  setCursor(Qt::ArrowCursor);
  setMargin(0);
  setLineWidth(1);
  hasSaveTitle = false;
  columnWindowFirstTimeShown = false;  
  saveTitle = QString::null;
  tmpFileName = QString::null;
  connect(this,SIGNAL(clicked(int, int, int, const QPoint &)), this, SLOT(Clicked(int, int, int, const QPoint &)));
  connect(this,SIGNAL(doubleClicked(int, int, int, const QPoint &)), this, SLOT(DoubleClicked(int, int, int, const QPoint &)));
  connect(this,SIGNAL(contextMenuRequested(int, int, const QPoint &)), this, SLOT(ContextMenuRequested(int, int, const QPoint &)));
  connect(horizontalHeader(), SIGNAL(clicked(int)), this, SLOT (Sort(int)));
  connect(horizontalHeader(), SIGNAL(pressed(int)), this, SLOT (headerPressed(int)));
  connect(horizontalHeader(), SIGNAL(released(int)), this, SLOT (headerReleased(int)));
  connect(verticalHeader(), SIGNAL(pressed(int)), this, SLOT (headerPressed(int)));
  connect(verticalHeader(), SIGNAL(released(int)), this, SLOT (headerReleased(int)));
}
