/* Copyright (C) 2002 MySQL AB & Jorge del Conde

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU Library General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.
    
  You should have received a copy of the GNU Library General Public
  License along with this library; if not, write to the Free
  Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
  MA 02111-1307, USA 
*/

#include "syntaxhighliter_sql.h"
#include "paragdata.h"
#include <qstring.h>
#include <qstringlist.h>
#include <qapplication.h>
#include <qregexp.h>
#include <qintdict.h>
#include "CGeneralConfig.h"
#include "Config.h"

SqlFormat::SqlFormat(CConfig *cfg, const QString & displayname, const QString & realname, SQLKeyWord::Highlight_Type id, const QString & defaultfont, const QString & defaultcolor)
{
  displayName = displayname;
  realName = realname;
  ID = id;
  update(&dFont, &dColor, defaultfont, defaultcolor);
  update(&Font, &Color, cfg->readStringEntry(realname + " Highlight", defaultfont),
         cfg->readStringEntry(realname + " Highlight Color", defaultcolor));
  if (g_SqlEditorStyles->size() <= (g_SqlEditorStyles->count() +1))
    g_SqlEditorStyles->resize(g_SqlEditorStyles->count() + 1);
  g_SqlEditorStyles->insert((uint)ID, this);
}

SqlFormat::~SqlFormat()
{
}

void SqlFormat::setDefaultValues()
{
  m_font.setFamily(dFont.family());
  m_font.setBold(dFont.bold());
  m_font.setUnderline(dFont.underline());
  m_font.setItalic(dFont.italic());
  m_font.setPointSize(dFont.pointSize());  
  m_color = dColor;
}

void SqlFormat::setDefaults()
{
  m_font.setFamily(Font.family());
  m_font.setBold(Font.bold());
  m_font.setUnderline(Font.underline());
  m_font.setItalic(Font.italic());
  m_font.setPointSize(Font.pointSize());  
  m_color = Color;
}

void SqlFormat::update(QFont *f, QColor *c, const QString &font, const QString &color)
{  
  f->fromString(font);
  QRegExp rx("^(\\d+),(\\d+),(\\d+)$");
  
  int r=0, g=0, b=0;
  if (rx.search(color) != -1)
  {
    r = rx.cap(1).toInt();
    g = rx.cap(2).toInt();
    b = rx.cap(3).toInt();
  }
  c->setRgb (r, g, b);
}

bool SqlFormat::saveToCfg(CConfig *cfg)
{
  bool err;
  Font.setFamily(m_font.family());
  Font.setBold(m_font.bold());
  Font.setUnderline(m_font.underline());
  Font.setItalic(m_font.italic());
  Font.setPointSize(m_font.pointSize());
  Color = m_color;
  QString color = QString::number(Color.red()) + "," + QString::number(Color.green()) + "," + QString::number(Color.blue());
  err = cfg->writeEntry(realName + " Highlight", Font.toString());
  err &= cfg->writeEntry(realName + " Highlight Color", color);
  return err;
}

void SyntaxHighlighter_SQL::loadSqlKeywords(QDict<SQLKeyWord> *keyword_dict)
{
  KeywordDict = *keyword_dict;  
  hasSqlSyntaxHighlighting = true;
}

void SyntaxHighlighter_SQL::loadDatabaseKeywords(int serverid)
{  
  DBKeyWord *l= g_DBKeywordDict->find(serverid);
  if (l != NULL)
  {
    FieldsList.clear();
    FieldsList.resize(l->Databases.size() + l->Tables.size() + l->Fields.size());

    QDictIterator<QString> it(l->Databases);
    for( ; it.current(); ++it )
      FieldsList.insert(it.currentKey(), it.current());

    QDictIterator<QString> it2(l->Tables);  
    for( ; it2.current(); ++it2 )
      FieldsList.insert(it2.currentKey(), it2.current());

    QDictIterator<QString> it3(l->Fields);
    for( ; it3.current(); ++it3 )
      FieldsList.insert(it3.currentKey(), it3.current());
  }  
  hasDatabaseSyntaxHighlighting = true;
}

SyntaxHighlighter_SQL::SyntaxHighlighter_SQL()
: QTextPreProcessor(), lastFormat( 0 ), lastFormatId( -1 )
{
  formats.setAutoDelete(true);
  hasSqlSyntaxHighlighting = false;
  hasDatabaseSyntaxHighlighting = false;
  QIntDictIterator<SqlFormat> it(*g_SqlEditorStyles);
  for ( ; it.current(); ++it )  
    if (it.current()->ID != SQLKeyWord::highlight_PARENTHESES_MATCHING)
      addFormat(it.current()->ID, new QTextFormat(it.current()->Font, it.current()->Color));  
}

SyntaxHighlighter_SQL::~SyntaxHighlighter_SQL()
{    
}

void SyntaxHighlighter_SQL::process( QTextDocument *doc, QTextParag *string, int, bool invalidate )
{
  QTextFormat *formatStandard = format(SQLKeyWord::highlight_STANDARD);
  QTextFormat *formatString = format(SQLKeyWord::highlight_STRINGS);
  QTextFormat *formatNumber = format(SQLKeyWord::highlight_NUMBERS);
  QTextFormat *formatComment = format(SQLKeyWord::highlight_COMMENTS);
  QTextFormat *formatVariable = format(SQLKeyWord::highlight_VARIABLES);
  QTextFormat *formatTablesFields = format(SQLKeyWord::highlight_TABLES_AND_FIELDS);  
  
  const int StateStandard = 0;
  const int StateCommentStart1 = 1;
  const int StateCCommentStart2 = 2;
  const int StateCComment = 3;  
  const int StateCCommentEnd1 = 4;
  const int StateCCommentEnd2 = 5;
  const int StateStringStart = 6;
  const int StateString = 7;
  const int StateStringEnd = 8;
  const int StateString2Start = 9;
  const int StateString2 = 10;
  const int StateString2End = 11;
  const int StateHashComment = 12;
  const int StateEscStart = 13;
  const int StateEsc = 14;
  const int StateEscEnd = 15;
  
  const int InputAlpha = 0;  
  const int InputAsterix = 1;
  const int InputSlash = 2;
  const int InputParen = 3;
  const int InputSpace = 4;
  const int InputHash = 5;
  const int InputQuotation = 6;
  const int InputApostrophe = 7;
  const int InputSep = 8;
  const int InputEsc = 9;
  
  static uchar table[ 16 ][ 10 ] = {
    { StateStandard,    StateStandard,       StateCommentStart1, StateStandard,    StateStandard,    StateHashComment, StateStringStart, StateString2Start, StateStandard,    StateEscStart },    // StateStandard
    { StateStandard,    StateCCommentStart2, StateStandard,      StateStandard,    StateStandard,    StateHashComment, StateStringStart, StateString2Start, StateStandard,    StateEscStart },    // StateCommentStart1
    { StateCComment,    StateCCommentEnd1,   StateCComment,      StateCComment,    StateCComment,    StateCComment,    StateCComment,    StateCComment,     StateCComment,    StateCComment },    // StateCCommentStart2    
    { StateCComment,    StateCCommentEnd1,   StateCComment,      StateCComment,    StateCComment,    StateCComment,    StateCComment,    StateCComment,     StateCComment,    StateCComment },    // StateCComment    
    { StateCComment,    StateCCommentEnd1,   StateCCommentEnd2,  StateCComment,    StateCComment,    StateCComment,    StateCComment,    StateCComment,     StateCComment,    StateCComment },    // StateCCommentEnd1
    { StateStandard,    StateStandard,       StateCommentStart1, StateStandard,    StateStandard,    StateHashComment, StateStringStart, StateString2Start, StateStandard,    StateEscStart },    // StateCCommentEnd2
    { StateString,      StateString,         StateString,        StateString,      StateString,      StateString,      StateStringEnd,   StateString,       StateString,      StateString },      // StateStringStart
    { StateString,      StateString,         StateString,        StateString,      StateString,      StateString,      StateStringEnd,   StateString,       StateString,      StateString },      // StateString
    { StateStandard,    StateStandard,       StateCommentStart1, StateStandard,    StateStandard,    StateHashComment, StateStringStart, StateString2Start, StateStandard,    StateEscStart },    // StateStringEnd
    { StateString2,     StateString2,        StateString2,       StateString2,     StateString2,     StateString2,     StateString2,     StateString2End,   StateString2,     StateString2 },     // StateString2Start
    { StateString2,     StateString2,        StateString2,       StateString2,     StateString2,     StateString2,     StateString2,     StateString2End,   StateString2,     StateString2 },     // StateString2
    { StateStandard,    StateStandard,       StateCommentStart1, StateStandard,    StateStandard,    StateHashComment, StateStringStart, StateString2Start, StateStandard,    StateEscStart },    // StateString2End    
    { StateHashComment, StateHashComment,    StateHashComment,   StateHashComment, StateHashComment, StateHashComment, StateHashComment, StateHashComment,  StateHashComment, StateHashComment }, // StateHashComment    
    { StateEsc,         StateEsc,            StateEsc,           StateEsc,         StateEsc,         StateHashComment, StateEsc,         StateEsc,          StateEsc,         StateStandard },    // StateEscStart
    { StateEsc,         StateEsc,            StateEsc,           StateEsc,         StateEsc,         StateHashComment, StateEsc,         StateEsc,          StateEsc,         StateStandard },    // StateEsc
    { StateStandard,    StateStandard,       StateCommentStart1, StateStandard,    StateStandard,    StateHashComment, StateStringStart, StateString2Start, StateStandard,    StateStandard }     // StateEscEnd
  };
  
  QString buffer;
  
  int state = StateStandard;
  if ( string->prev() )
  {
    if ( string->prev()->endState() == -1 )
      process( doc, string->prev(), 0, false);
    state = string->prev()->endState();
  }
  int input=0;
  int i = 0;
  bool lastWasBackSlash = false;
  bool makeLastStandard = false;
  
  ParagData *paragData = (ParagData*)string->extraData();
  if ( paragData )
    paragData->parenList.clear();
  else
    paragData = new ParagData;
  
  static QString alphabeth = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ";
  static QString mathChars = "xXeE";
  static QString numbers = "0123456789";
  QChar lastChar;  
  bool b = true;  
  while (b)
  {
    QChar c = string->at( i )->c;    
    if ( lastWasBackSlash )
    {
      input = InputSep;
    }
    else
    {      
      switch ( c )
      {
      case '*':
        input = InputAsterix;
        break;
      case '`':
        input = InputEsc;      
        break;
      case '/':
        input = InputSlash;
        break;
      case '(':
        input = InputParen;        
        if ( state == StateStandard ||       
          state == StateHashComment ||
          state == StateCCommentEnd2 ||
          state == StateCCommentEnd1 ||
          state == StateString2End ||
          state == StateStringEnd )
          paragData->parenList << Paren( Paren::Open, c, i );
        break;
      case ')':
        input = InputParen;        
        if ( state == StateStandard ||        
          state == StateHashComment ||
          state == StateCCommentEnd2 ||
          state == StateCCommentEnd1 ||
          state == StateString2End ||
          state == StateStringEnd )
          paragData->parenList << Paren( Paren::Closed, c, i );
        break;
      case '#':              
        input = InputHash;
        break;
      case '"':
        input = InputQuotation;        
        break;        
      case '\'':
        input = InputApostrophe;
        break;
      case ' ':
        input = InputSpace;        
        break;        
      default:
        {          
          if ( c.isNumber() || c.isLetter() || c == '_' || c == '@' || c == '.')
            input = InputAlpha;          
          else            
            input = InputSep;            
        }
        break;
      }
    }
    
    lastWasBackSlash = !lastWasBackSlash && c == '\\';
    
    if ( input == InputAlpha )
      buffer += c;
    
    state = table[ state ][ input ];
    
    switch (state)
    {      
    case StateStandard:
      {        
        string->setFormat( i, 1, formatStandard, false);
        if (makeLastStandard)
          string->setFormat( i - 1, 1, formatStandard, false);
        makeLastStandard = false;
        if ( buffer.length() > 0 && input != InputAlpha )
        { 
          QRegExp rx( "^(\\d+(\\.)?(\\.\\d+)?)$");
          if (rx.search(buffer) != -1)         
            string->setFormat( i - buffer.length(), buffer.length(), formatNumber, false);
          else         
            if (buffer[0] == '@')
              string->setFormat( i - buffer.length(), buffer.length(), formatVariable, false);
            else
            {
              bool found = false;
              if (hasSqlSyntaxHighlighting)
              {
                SQLKeyWord *tmp = KeywordDict.find(buffer.lower());
                if (tmp != NULL)
                {
                  string->setFormat( i - buffer.length(), buffer.length(), format(tmp->Highlight), false);                
                  found = true;
                }
              }
              
              if (hasDatabaseSyntaxHighlighting  && !found)
              {                
                if (FieldsList.find(buffer.lower()) != NULL)
                  string->setFormat( i - buffer.length(), buffer.length(), formatTablesFields, false);                
              }
            }
            buffer = QString::null;
        }
      }
      break;
    case StateCommentStart1:
      if ( makeLastStandard )
        string->setFormat( i - 1, 1, formatStandard, false);
      makeLastStandard = true;
      buffer = QString::null;
      break;
    case StateCCommentStart2:
      string->setFormat( i - 1, 2, formatComment, false);
      makeLastStandard = false;
      buffer = QString::null;
      break;
    case StateCComment:    
      if ( makeLastStandard )
        string->setFormat( i - 1, 1, formatStandard, false);
      makeLastStandard = false;
      string->setFormat( i, 1, formatComment, false);
      buffer = QString::null;
      break;      
    case StateCCommentEnd1:
      if ( makeLastStandard )
        string->setFormat( i - 1, 1, formatStandard, false);
      makeLastStandard = false;
      string->setFormat( i, 1, formatComment, false);
      buffer = QString::null;
      break;
    case StateCCommentEnd2:
      if ( makeLastStandard )
        string->setFormat( i - 1, 1, formatStandard, false);
      makeLastStandard = false;
      string->setFormat( i, 1, formatComment, false);
      buffer = QString::null;
      break;
    case StateStringStart:
      if ( makeLastStandard )
        string->setFormat( i - 1, 1, formatStandard, false);
      makeLastStandard = false;
      string->setFormat( i, 1, formatStandard, false);
      buffer = QString::null;
      break;
    case StateString:
      if ( makeLastStandard )
        string->setFormat( i - 1, 1, formatStandard, false);
      makeLastStandard = false;
      string->setFormat( i, 1, formatString, false);
      buffer = QString::null;
      break;
    case StateStringEnd:
      if ( makeLastStandard )
        string->setFormat( i - 1, 1, formatStandard, false);
      makeLastStandard = false;
      string->setFormat( i, 1, formatStandard, false);
      buffer = QString::null;
      break;
    case StateString2Start:
      if ( makeLastStandard )
        string->setFormat( i - 1, 1, formatStandard, false);
      makeLastStandard = false;
      string->setFormat( i, 1, formatStandard, false);
      buffer = QString::null;
      break;
    case StateString2:
      if ( makeLastStandard )
        string->setFormat( i - 1, 1, formatStandard, false);
      makeLastStandard = false;
      string->setFormat( i, 1, formatString, false);
      buffer = QString::null;
      break;
    case StateString2End:
      if ( makeLastStandard )
        string->setFormat( i - 1, 1, formatStandard, false);
      makeLastStandard = false;
      string->setFormat( i, 1, formatStandard, false);
      buffer = QString::null;
      break;
      
    case StateHashComment:
      if ( makeLastStandard )
        string->setFormat( i - 1, 1, formatStandard, false);
      makeLastStandard = false;
      string->setFormat( i, 1, formatComment, false);
      buffer = QString::null;
      break;
      
    case StateEscStart:
      if ( makeLastStandard )
        string->setFormat( i - 1, 1, formatStandard, false);
      makeLastStandard = false;
      string->setFormat( i, 1, formatStandard, false);
      buffer = QString::null;      
      break;
      
    case StateEsc:
      if ( makeLastStandard )
        string->setFormat( i - 1, 1, formatStandard, false);
      makeLastStandard = false;
      string->setFormat( i, 1, formatTablesFields, false);
      buffer = QString::null;    
      break;
      
    case StateEscEnd:
      if ( makeLastStandard )
        string->setFormat( i - 1, 1, formatStandard, false);
      makeLastStandard = false;
      string->setFormat( i, 1, formatStandard, false);
      buffer = QString::null;      
      break;
  }
  
  lastChar = c;
  i++;
  if ( i >= string->length() )
    break;
  }
  
  string->setExtraData( paragData );
  
  if ( state == StateCComment || state == StateCCommentEnd1 )  
    string->setEndState( StateCComment );  
  else
    if ( state == StateString )    
      string->setEndState( StateString );    
    else
      if ( state == StateString2 )      
        string->setEndState( StateString2 );      
      else      
        string->setEndState( StateStandard );      
      
      string->setFirstPreProcess( false);
      
      if ( invalidate && string->next() && !string->next()->firstPreProcess() && string->next()->endState() != -1 )
      {
        QTextParag *p = string->next();
        while ( p )
        {
          if ( p->endState() == -1 )
            return;
          p->setEndState( -1 );
          p = p->next();
        }
      }
}

QTextFormat *SyntaxHighlighter_SQL::format( int id )
{
  if ( lastFormatId == id  && lastFormat )
    return lastFormat;  
  QTextFormat *f = formats[ id ];
  lastFormat = f ? f : formats[ 0 ];
  lastFormatId = id;
  return lastFormat;
}

void SyntaxHighlighter_SQL::addFormat( int id, QTextFormat *f )
{
  formats.insert( id, f );
}

void SyntaxHighlighter_SQL::removeFormat( int id )
{
  formats.remove( id );
}

