/* Copyright (C) 2002 MySQL AB & Jorge del Conde

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU Library General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.
    
  You should have received a copy of the GNU Library General Public
  License along with this library; if not, write to the Free
  Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
  MA 02111-1307, USA 
*/

#include "CQueryWindow.h"
#include "CHistoryView.h"
#include <qlayout.h>

CQueryWindow::CQueryWindow( QWidget* parent, QString caption, CMySQLConnection *m, const QString & sql, unsigned short display, int serverid)
: CMyWindow( parent, "CQueryWindow", WDestructiveClose, !g_isMDI, true), serverID(serverid)
{
  createForm(m, SQL_QUERY, "", display);
  setCaption(caption);
  if (!sql.isEmpty())
    sqlView->setText(sql);  
}


CQueryWindow::CQueryWindow( QWidget* parent, CMySQLConnection *m, int querytype, const QString & table, int limit, unsigned short display, int serverid)
: CMyWindow( parent, "CQueryWindow", WDestructiveClose, !g_isMDI, true), serverID(serverid)
{  
  createForm(m, querytype, table, display);
  QString c = ((m_queryType == SQL_QUERY) ? tr("Query in Database") + ": " : tr("Data in Table") + ": '" + table + "' " + tr("in") + " '");
  c += mysql->getDatabaseName() + "' [" + mysql->getConnectionName() + "]";	
  setCaption(tr(c, "" ) ); 	

  if (!m_defaultTable.isEmpty())
  {
    QString Sql = "SELECT *\nFROM " + mysql->Quote(m_defaultTable);
    if (m_queryType == SQL_LIMIT)
      Sql += "\nLIMIT " + QString::number(limit);
    sqlView->setText(Sql);
  }  
}

void CQueryWindow::createForm(CMySQLConnection *m, int querytype, const QString & table, unsigned short display)
{  
  canClose = true;
  mysql = new CMySQLConnection(m->getConnectionName(), messagePanel());
  Q_CHECK_PTR(mysql);
  mysql->connect(m->getDatabaseName()); 
  messageWindow()->setCarriageReturn(mysql->getLineTerminator(true));  
  m_defaultTable = table;
  m_queryType = querytype;
  (void)statusBar();  
  setCursor(Qt::ArrowCursor);
  selectQuery = getPixmapIcon("selectQueryIcon");
  newTableQuery = getPixmapIcon("newTableQueryIcon");
  updateQuery = getPixmapIcon("updateQueryIcon");
  appendQuery = getPixmapIcon("appendQueryIcon");
  deleteQuery = getPixmapIcon("deleteQueryIcon");

  setName("CQueryWindow");  
  setIcon(getPixmapIcon("applicationIcon"));

  setCentralWidget( new QWidget( this, "qt_central_widget" ) );
  CQueryWindowLayout = new QHBoxLayout( centralWidget() );
  Q_CHECK_PTR(CQueryWindowLayout);
  CQueryWindowLayout->setSpacing( 2 );
  CQueryWindowLayout->setMargin( 2 );
  QHBox * pTop = new QHBox(centralWidget());
  Q_CHECK_PTR(pTop);
  QSplitter* pBottom = new QSplitter( QSplitter::Vertical, pTop);
  Q_CHECK_PTR(pBottom);      
  sqlView = new CSqlEditor(pBottom, (mysql->hasCompletion() ? serverID : -1 ), "sqlView", mysql);
  Q_CHECK_PTR(sqlView);
  results = new CSqlTable(pBottom, mysql);
  Q_CHECK_PTR(results);
  results->enableColumnsWindow(true);
  
  //TODO: Remove this line when CSqlTable supports editing!
  results->setSaveTitle(tr("Query Results"));
  results->setAutomaticFieldEditors(true);
  results->setMyReadOnly(true);
  results->setMaxCellTextSize(MAX_BLOB_SIZE);
  
  pBottom->setOpaqueResize(true);
  pBottom->setResizeMode (sqlView, QSplitter::KeepSize);
  pBottom->setResizeMode (results, QSplitter::KeepSize);    
  CQueryWindowLayout->addWidget(pTop);
  sqlView->setMinimumSize(10, 50);
  results->setMinimumSize(10, 50);	
  fileOpenAction = new QAction( this, "fileOpenAction" );
  Q_CHECK_PTR(fileOpenAction);
  fileOpenAction->setText(tr("Open", "" ) );
  fileOpenAction->setIconSet( QIconSet(getPixmapIcon("openIcon") ) );
  fileOpenAction->setMenuText(tr("&Open", "" ) );
  fileOpenAction->setAccel( 0 );

  showMessagesAction = new QAction(this, "showMessagesAction" );
  Q_CHECK_PTR(showMessagesAction);
  showMessagesAction->setText(tr("Show Messages Panel", "" ) );  
  showMessagesAction->setMenuText(tr("Show &Messages Panel", "" ) );
  showMessagesAction->setAccel( 0 );
  showMessagesAction->setToggleAction(true);
  showMessagesAction->setOn(messageWindow()->isVisible());

  filePrintAction = new QAction( this, "filePrintAction" );
  Q_CHECK_PTR(filePrintAction);
  filePrintAction->setIconSet( QIconSet(getPixmapIcon("printIcon") ) );
  filePrintAction->setText(tr("Print", "" ) );
  filePrintAction->setMenuText(tr("&Print", "" ) );
  filePrintAction->setAccel( 4194384 );
  fileCloseAction = new QAction( this, "fileCloseAction" );
  Q_CHECK_PTR(fileCloseAction);
  fileCloseAction->setText(tr("Close", "" ) );
  fileCloseAction->setMenuText(tr("&Close", "" ) );
  fileCloseAction->setIconSet( QIconSet(getPixmapIcon("closeIcon") ) );
  fileCloseAction->setAccel( 0 );
  editUndoAction = new QAction( this, "editUndoAction" );
  Q_CHECK_PTR(editUndoAction);
  editUndoAction->setIconSet( QIconSet(getPixmapIcon("undoIcon") ) );
  editUndoAction->setText(tr("Undo", "" ) );
  editUndoAction->setMenuText(tr("&Undo", "" ) );
  editUndoAction->setAccel( 4194394 );
  editRedoAction = new QAction( this, "editRedoAction" );
  Q_CHECK_PTR(editRedoAction);
  editRedoAction->setIconSet( QIconSet(getPixmapIcon("redoIcon") ) );
  editRedoAction->setText(tr("Redo", "" ) );
  editRedoAction->setMenuText(tr("&Redo", "" ) );
  editRedoAction->setAccel( 4194393 );
  editCutAction = new QAction( this, "editCutAction" );
  Q_CHECK_PTR(editCutAction);
  editCutAction->setIconSet( QIconSet(getPixmapIcon("cutIcon") ) );
  editCutAction->setText(tr("Cut", "" ) );
  editCutAction->setMenuText(tr("&Cut", "" ) );
  editCutAction->setAccel( 4194392 );
  editCopyAction = new QAction( this, "editCopyAction" );
  Q_CHECK_PTR(editCopyAction);
  editCopyAction->setIconSet( QIconSet(getPixmapIcon("copyIcon") ) );
  editCopyAction->setText(tr("Copy", "" ) );
  editCopyAction->setMenuText(tr("C&opy", "" ) );
  editCopyAction->setAccel( 4194371 );
  editPasteAction = new QAction( this, "editPasteAction" );
  Q_CHECK_PTR(editPasteAction);
  editPasteAction->setIconSet( QIconSet(getPixmapIcon("pasteIcon") ) );
  editPasteAction->setText(tr("Paste", "" ) );
  editPasteAction->setMenuText(tr("&Paste", "" ) );
  editPasteAction->setAccel( 4194390 );
  
  toolBar = new QToolBar( "", this, Top );
  Q_CHECK_PTR(toolBar);
  toolBar->setLabel(tr("Tools", "" ) );	
  
  fileOpenAction->addTo( toolBar );

  p_saveTypesMenu = new QPopupMenu();
  Q_CHECK_PTR(p_saveTypesMenu);
  p_saveTypesMenu->insertItem(getPixmapIcon("saveIcon"), tr("&Query"), 1);	
  p_saveTypesMenu->insertItem(getPixmapIcon("saveGridResultsIcon"), tr("&Results"), 2);	
    
  saveTypeButton = new QToolButton(toolBar);
  Q_CHECK_PTR(saveTypeButton);
  saveTypeButton->setPopup(p_saveTypesMenu);
  saveTypeButton->setPixmap(getPixmapIcon("saveIcon"));
  saveTypeButton->setTextLabel("Save", true);
  saveTypeButton->setPopupDelay (0);
  
  filePrintAction->addTo( toolBar );
  toolBar->addSeparator();
  editUndoAction->addTo( toolBar );
  editRedoAction->addTo( toolBar );
  editCopyAction->addTo( toolBar );
  editCutAction->addTo( toolBar );
  editPasteAction->addTo( toolBar );
  
  menubar = new QMenuBar( this, "menubar" );
  Q_CHECK_PTR(menubar);
  
  fileMenu = new QPopupMenu( this );
  Q_CHECK_PTR(fileMenu);
  fileOpenAction->addTo( fileMenu );
  fileMenu->insertItem(getPixmapIcon("saveIcon"), tr("&Save"), p_saveTypesMenu);  

  fileMenu->insertSeparator();
  showMessagesAction->addTo( fileMenu );
  fileMenu->insertSeparator();
  filePrintAction->addTo( fileMenu );
  fileMenu->insertSeparator();
  fileCloseAction->addTo( fileMenu );
  menubar->insertItem(tr("&File", "" ), fileMenu );
  
  editMenu = new QPopupMenu( this );
  Q_CHECK_PTR(editMenu);
  editUndoAction->addTo( editMenu );
  editRedoAction->addTo( editMenu );
  editMenu->insertSeparator();
  editCutAction->addTo( editMenu );
  editCopyAction->addTo( editMenu );
  editPasteAction->addTo( editMenu );
  menubar->insertItem(tr("&Edit", "" ), editMenu );
  
  sqlAction = new QAction( this, "sqlAction" );
  Q_CHECK_PTR(sqlAction);
  sqlAction->setText(tr("SQL Pane", "" ) );
  sqlAction->setIconSet( QIconSet(getPixmapIcon("sqlIcon") ) );
  sqlAction->setMenuText(tr("&SQL Pane", "" ) );
  sqlAction->setAccel( 0 );
  sqlAction->setToggleAction(true);
  
  gridAction = new QAction( this, "gridAction" );
  Q_CHECK_PTR(gridAction);
  gridAction->setText(tr("Results Pane", "" ) );
  gridAction->setIconSet( QIconSet(getPixmapIcon("gridIcon") ) );
  gridAction->setMenuText(tr("Results &Pane", "" ) );
  gridAction->setAccel( 0 );
  gridAction->setToggleAction(true);
  
  executeQueryAction = new QAction( this, "executeQueryAction" );
  Q_CHECK_PTR(executeQueryAction);
  executeQueryAction->setText(tr("Run", "" ) );
  executeQueryAction->setIconSet( QIconSet(getPixmapIcon("executeQueryIcon") ) );
  executeQueryAction->setMenuText(tr("&Run", "" ) );
  executeQueryAction->setAccel( 0 );
  
  cancelQueryAction = new QAction( this, "cancelQueryAction" );
  Q_CHECK_PTR(cancelQueryAction);
  cancelQueryAction->setText(tr("Cancel Execution and Clear Results", "" ) );
  cancelQueryAction->setIconSet( QIconSet(getPixmapIcon("cancelQueryIcon") ) );
  cancelQueryAction->setMenuText(tr("&Cancel", "" ) );
  cancelQueryAction->setAccel( 0 );
  
  queryBar = new QToolBar( "", this, Top );
  Q_CHECK_PTR(queryBar);
  queryBar->setLabel(tr("Query", "" ) );
  sqlAction->addTo(queryBar);
  gridAction->addTo(queryBar);
  queryBar->addSeparator();
  p_queryTypesMenu = new QPopupMenu();
  Q_CHECK_PTR(p_queryTypesMenu);
  p_queryTypesMenu->insertItem(selectQuery, tr("Select Query"), 1);	
  p_queryTypesMenu->insertItem(newTableQuery, tr("Make-Table Query"), 2);	
  p_queryTypesMenu->insertItem(updateQuery, tr("Update Query"), 3);	
  p_queryTypesMenu->insertItem(appendQuery, tr("Append Query"), 4);	
  p_queryTypesMenu->insertItem(deleteQuery, tr("Delete Query"), 5);
  
  if (!m_defaultTable.isEmpty())
  {
    queryTypeButton = new QToolButton(queryBar);
    Q_CHECK_PTR(queryTypeButton);
    queryTypeButton->setPopup(p_queryTypesMenu);
    queryTypeButton->setPixmap(selectQuery);
    queryTypeButton->setTextLabel("Query Type", true);
    queryTypeButton->setPopupDelay (0);
    queryBar->addSeparator();
  }
  executeQueryAction->addTo(queryBar);
  cancelQueryAction->addTo(queryBar);
  queryMenu = new QPopupMenu( this );
  Q_CHECK_PTR(queryMenu);
  sqlAction->addTo(queryMenu);
  gridAction->addTo(queryMenu);
  
  if (!m_defaultTable.isEmpty())
  {
    queryMenu->insertSeparator();
    queryMenu->insertItem(getPixmapIcon("queryTypesIcon"), tr("&Query Type"), p_queryTypesMenu);
  }
  
  queryMenu->insertSeparator();
  executeQueryAction->addTo(queryMenu);
  cancelQueryAction->addTo(queryMenu);
  menubar->insertItem(tr("&Query", "" ), queryMenu );

  HistoryPanel = new CHistoryPanel(tr("History"));
  messageWindow()->addPanel(HistoryPanel);
  HistoryPanel->setMaxDisplaySize(259);
  HistoryPanel->setMessagePanel(messagePanel());
  ReloadHistoryList();
  init();
  showSqlPanel((display & SQL_PANEL) == SQL_PANEL);  
  showGridPanel((display & RESULTS_PANEL) == RESULTS_PANEL);
  myResize(520, 419);  
  autoPlace();
}

void CQueryWindow::ReloadHistoryList()
{
  HistoryPanel->clear();
  g_HistoryList->loadTo(HistoryPanel);
}

CQueryWindow::~CQueryWindow()
{
  delete p_queryTypesMenu;
  delete p_saveTypesMenu;
  delete mysql;
}

void CQueryWindow::SelectionChanged()
{
  bool s = sqlView->hasSelectedText();
  editCutAction->setEnabled(s);
  editCopyAction->setEnabled(s);
}

void CQueryWindow::showSqlPanel(bool s)
{	
  if (s)
    sqlView->show();
  else
    sqlView->hide();
  sqlAction->setOn(s);
  gridAction->setEnabled(s);
}

void CQueryWindow::showGridPanel(bool s)
{	
  if (s)
    results->show();  
  else  
    results->hide();      
  gridAction->setOn(s);
  sqlAction->setEnabled(s);
}

void CQueryWindow::SqlAction()
{
  showSqlPanel(!sqlView->isVisible());
}

void CQueryWindow::GridAction()
{
  showGridPanel(!results->isVisible());
}

void CQueryWindow::doExecuteQuery(const QString &qry)
{
  sqlView->setText(qry);
  ExecuteQuery();
}

void CQueryWindow::setQuery(const QString &qry)
{
  sqlView->setText(qry);
  showSqlPanel(true);
}

void CQueryWindow::ExecuteQuery()
{  
  if (mysql->isConnected())
  {    
    showGridPanel(true);
    HistoryPanel->History(sqlView->text());    
    results->exec(sqlView->text());
  }
}

void CQueryWindow::closeEvent ( QCloseEvent * e )
{
  if (!canClose)
    e->ignore();
  else
  {
    beforeClose();    
    e->accept();
  }
}

void CQueryWindow::disableButtons(bool e)
{
  if (e)
    setCursor (Qt::waitCursor);
  else
    setCursor (Qt::arrowCursor);
  canClose = !e;
  executeQueryAction->setEnabled(!e);
  
  saveTypeButton->setEnabled(!e);
  p_saveTypesMenu->setEnabled(!e);  
  filePrintAction->setEnabled(!e);
  fileCloseAction->setEnabled(!e);
}

void CQueryWindow::CancelQuery()
{
  results->cancelQuery();
  results->reset();
}

void CQueryWindow::SelectQuery()
{  
  showSqlPanel(true);  
  sqlView->setText("SELECT *\nFROM " + m_defaultTable);
  queryTypeButton->setPixmap(selectQuery);
}

void CQueryWindow::MakeTableQuery()
{  
  showSqlPanel(true);
  QString Sql = "TODO !!";	
  queryTypeButton->setPixmap(newTableQuery);
}

void CQueryWindow::UpdateQuery()
{  
  showSqlPanel(true);  
  sqlView->setText("UPDATE " + m_defaultTable + "\nSET [" + tr("Field1") + " = " + tr("Value1") + ",...N],");
  queryTypeButton->setPixmap(updateQuery);
}

void CQueryWindow::AppendQuery()
{
  showSqlPanel(true);    
  sqlView->setText("INSERT INTO " + m_defaultTable + "\n( [" + tr("Field1") + ",...N] )\nVALUES( [" + tr("Value1") + ",...N] )");
  queryTypeButton->setPixmap(appendQuery);
}

void CQueryWindow::DeleteQuery()
{	  
  showSqlPanel(true);
  QString Sql = "DELETE\nFROM " + m_defaultTable + " WHERE ";	
  sqlView->setText(Sql);
  queryTypeButton->setPixmap(deleteQuery);
}

void CQueryWindow::processShowMessagesSlot()
{  
  if (!messageWindow()->isHidden())
    messageWindow()->hide();  
  else  
    messageWindow()->show();  
  showMessagesAction->setOn(!messageWindow()->isHidden());
}

void CQueryWindow::init()
{
  connect( fileCloseAction, SIGNAL( activated() ), this, SLOT( close() ) );	
  connect( sqlView, SIGNAL( selectionChanged() ), this, SLOT( SelectionChanged() ) );
  connect( sqlView, SIGNAL( undoAvailable(bool) ), editUndoAction, SLOT( setEnabled(bool) ) );
  connect( sqlView, SIGNAL( redoAvailable(bool) ), editRedoAction, SLOT( setEnabled(bool) ) );
  connect( sqlView, SIGNAL( copyAvailable(bool) ), editCopyAction, SLOT( setEnabled(bool) ) );

  connect( fileOpenAction, SIGNAL( activated() ), sqlView, SLOT( openFile() ) );

  connect( results, SIGNAL( executing(bool) ), this, SLOT( disableButtons(bool) ) );
  connect( results, SIGNAL( executing(bool) ), HistoryPanel, SLOT( disableMenuItems(bool) ) );

  connect( HistoryPanel, SIGNAL(executeQuery(const QString &)), this, SLOT( doExecuteQuery(const QString &) ) );
  connect( HistoryPanel, SIGNAL(doubleClicked(const QString &)), this, SLOT( setQuery(const QString &) ) );
  connect( HistoryPanel, SIGNAL(reloadHistoryList()), this, SLOT( ReloadHistoryList() ) );

  p_saveTypesMenu->connectItem(1,sqlView, SLOT(save()));
  p_saveTypesMenu->connectItem(2,results, SLOT(saveQueryContents()));
  
  connect(showMessagesAction, SIGNAL(activated()), this, SLOT(processShowMessagesSlot()));
  connect(messageWindow(), SIGNAL(visibilityChanged(bool)), showMessagesAction, SLOT(setOn(bool)));

  connect( editUndoAction, SIGNAL( activated() ), sqlView, SLOT( undo() ) );
  connect( editRedoAction, SIGNAL( activated() ), sqlView, SLOT( redo() ) );
  connect( editCutAction, SIGNAL( activated() ), sqlView, SLOT( cut() ) );
  connect( editCopyAction, SIGNAL( activated() ), sqlView, SLOT( copy() ) );
  connect( editPasteAction, SIGNAL( activated() ), sqlView, SLOT( paste() ) );
  
  connect( sqlAction, SIGNAL( activated() ), this, SLOT( SqlAction() ) );
  connect( gridAction, SIGNAL( activated() ), this, SLOT( GridAction() ) );

  connect( executeQueryAction, SIGNAL( activated() ), this, SLOT( ExecuteQuery() ) );
  connect( cancelQueryAction, SIGNAL( activated() ), this, SLOT( CancelQuery() ) );
		
  p_queryTypesMenu->connectItem(1,this, SLOT(SelectQuery()));
  p_queryTypesMenu->connectItem(2,this, SLOT(MakeTableQuery()));
  p_queryTypesMenu->connectItem(3,this, SLOT(UpdateQuery()));
  p_queryTypesMenu->connectItem(4,this, SLOT(AppendQuery()));
  p_queryTypesMenu->connectItem(5,this, SLOT(DeleteQuery()));
  
#ifdef QT_NO_PRINTER
  filePrintAction->setEnabled(false);
#else
  connect( filePrintAction, SIGNAL( activated() ), sqlView, SLOT( print() ) );
#endif
  
  SelectionChanged();
}

