/* Copyright (C) 2002 MySQL AB & Jorge del Conde

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU Library General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.
    
  You should have received a copy of the GNU Library General Public
  License along with this library; if not, write to the Free
  Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
  MA 02111-1307, USA 
*/

#include "CMySQLQuery.h"
#include <qdatetime.h>
#include <qapplication.h>
#include "CConsoleWindow.h"

#define PENDING_SLICE  20

CMySQLQuery::CMySQLQuery(CMySQLConnection *conn):
 field_mask(0), num_fields(0), m_pMySQLConnection(conn), res(0), m_cancel(true)
{
   checkPendingEvt = 0;
   result_in_use = 0;
}

CMySQLQuery::~CMySQLQuery()
{
  result_in_use = 0;
  freeResult();
}

void CMySQLQuery::cancel()
{
  m_cancel = true;
}

bool CMySQLQuery::isNull()
{
  return (res == NULL);
}

unsigned long CMySQLQuery::getFieldLength(unsigned int index)
{
  return !isNull() ? *(mysql_fetch_lengths(res) + index) : 0;  
}

const char * CMySQLQuery::getFieldValue(unsigned int offset, unsigned int index)
{ 
  dataSeek(offset);  
  return ((row=mysql_fetch_row(res)) != NULL) ? row[index] : 0;  
}

bool CMySQLQuery::next(bool /*showError*/, bool autoFree)
{      
  bool r;
  checkPendingEvt++;
  r = ((row=mysql_fetch_row(res)) != NULL);
  if (checkPendingEvt == PENDING_SLICE)
  {
    if (!m_pMySQLConnection->isBlockingQueries() && qApp->hasPendingEvents())
    {
      qApp->processEvents();
      if (m_cancel)
      	r = false;       
    }
    checkPendingEvt = 0;
  }
  if (!r && autoFree )
    freeResult();
  return r;
}

bool CMySQLQuery::ColumnQuery(QStringList &list, CMySQLConnection *m, const QString &qry, unsigned int r, bool blocking, bool showError)
{
  bool b = m->isBlockingQueries();
  bool hasblocking = false;
  bool ret = false;
  if ((blocking != b) && (blocking == true))
  {
    m->setBlockingQueries(blocking);
    hasblocking = true;
  }
  if (m->isConnected())
  {
    CMySQLQuery query(m);
    if (query.exec(qry, false, showError))
    {
      ret = true;
      while (query.next(showError))		
        list.append( query.Row(r) );
    }
  }  
  if (hasblocking)
    m->setBlockingQueries(!blocking);
  return ret;
}

void CMySQLQuery::dataSeek(unsigned long r)
{
  checkPendingEvt = 0;
  if (!isNull())
    mysql_data_seek(res, r);  
}
 

void CMySQLQuery::export_current_row(QTextStream& ts, QString& sep,
                                     QString& encl,QString& term,
                                     QString& repl_empty)
{
  for (unsigned int i = 0; i < num_fields; i++)
  {
    if (field_mask[i] == QUERY_FIELD_HIDDEN)
    {
      const char* r = row[i];
      if (!r)
        r = "NULL";
      if (!*r)
        r = (const char*)repl_empty;
      ts << encl << r << encl;
      if (i < last_field_ind)
        ts << sep;
      else
        break;
    }
  }
  ts << term;  
}

unsigned long CMySQLQuery::numRows()
{
  return ((!isNull()) ? mysql_num_rows(res) : 0);
}

bool CMySQLQuery::printAffectedRows(bool isSelectQuery, bool printAffected, bool showError, const QString &time)
{  
  long r = mysql_affected_rows(m_pMySQLConnection->mysql);   
  
  int flag = (isSelectQuery) ? INFORMATION : WARNING;

  QString msg = QString::number(r) + " " + ((isSelectQuery) ? 
    ((r == 1) ? QObject::tr("row found") : QObject::tr("rows found")) :
    ((r == 1) ? QObject::tr("affected row") : QObject::tr("affected rows")));
  if (!time.isNull())
    msg += " " + time;

  if ((r >= 0) && (printAffected))
    m_pMySQLConnection->PrintError(flag, msg);
  else
    if ((r == -1) && (showError))
      m_pMySQLConnection->PrintError(CRITICAL, m_pMySQLConnection->getError());

  return (r >= 0);  
}

void CMySQLQuery::freeResult()
{
  checkPendingEvt = 0;
  if (!isNull())
  {
    if (!result_in_use)
    {
      mysql_free_result(res);
      res = NULL;
    }
    delete [] field_mask;
    field_mask = 0;
  }  
}

bool CMySQLQuery::exec_static(const QString &qry, bool showAffected, bool *printedMessage, bool showError)
{
  checkPendingEvt = 0;
  m_cancel = false;
  result_in_use = 0;
  freeResult();
  CConsoleWindow::sqlDebugPanel->SqlDebug("[" + m_pMySQLConnection->getConnectionName() + "] " + qry);
  if (!m_pMySQLConnection->isConnected())
  {  
    if (showError)
      m_pMySQLConnection->PrintError(CRITICAL, m_pMySQLConnection->getError());
    if (printedMessage != 0)
      *printedMessage = true;
    return false;	
  }  

  if((mysql_query(m_pMySQLConnection->mysql, qry.latin1())) != 0)
  {
    if (showError)
      m_pMySQLConnection->PrintError(CRITICAL, m_pMySQLConnection->getError());
    if (printedMessage != 0)
      *printedMessage = true;
    return false;
  }

  if ((!printAffectedRows(false, showAffected)))
    if (printedMessage != 0)
      *printedMessage = true;  

  return true;
}

bool CMySQLQuery::exec(const QString &qry, bool showAffected, bool showError)
{
  checkPendingEvt = 0;
  m_cancel = false;
  result_in_use = 0;
  freeResult();
  CConsoleWindow::sqlDebugPanel->SqlDebug("[" + m_pMySQLConnection->getConnectionName() + "] " + qry);
  if (!m_pMySQLConnection->isConnected())
  {
    if (showError)
      m_pMySQLConnection->PrintError(CRITICAL, m_pMySQLConnection->getError());
    return false;
  }
  m_Query = qry.stripWhiteSpace();  
  QTime tm;
  tm.start();
  if((mysql_query(m_pMySQLConnection->mysql, m_Query.latin1())) != 0)    
  {    
    if (showError)
      m_pMySQLConnection->PrintError(CRITICAL, m_pMySQLConnection->getError());
    return false;
  }
  QString t = "(" + QString::number((double)tm.elapsed() / 1000L, 'f', 2) + ") " + QObject::tr("sec");
  if ((res=mysql_store_result(m_pMySQLConnection->mysql)) !=  NULL)
  {
    Fields = mysql_fetch_fields(res);
    field_mask = new char[(num_fields=mysql_num_fields(res))];
    return printAffectedRows(true, showAffected, showError, t);
  }

  printAffectedRows(!((m_pMySQLConnection->getErrno() == 0) && (res == NULL)), showAffected, showError, t);

  return false;
}


const QString CMySQLQuery::Row(unsigned int idx)
{
  return QString::fromLatin1(row[idx]);
}

