/* Copyright (C) 2002 MySQL AB & Jorge del Conde

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU Library General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.
    
  You should have received a copy of the GNU Library General Public
  License along with this library; if not, write to the Free
  Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
  MA 02111-1307, USA 
*/
#include "CImageFieldEditor.h"
#include <qstatusbar.h>

CImageFieldEditor::CImageFieldEditor(QWidget* parent, QPtrList<CFieldEditorWindow> *editorList, const char* name, CMySQLConnection *m, bool ro)
: CFieldEditorWindow(parent, editorList, name, m, ro)
{  
  if (!name)
    setName("CImageFieldEditor");
  (void)statusBar();
  editor = new CImageEditor(centralWidget(), this, "editor");

  CFieldEditorWindowLayout->addWidget(editor, 0, 0 );
  
  fileOpenAction = new QAction( this, "fileOpenAction" );
  Q_CHECK_PTR(fileOpenAction);
  fileOpenAction->setText(tr("Open", "" ) );
  fileOpenAction->setIconSet( QIconSet(getPixmapIcon("openIcon") ) );
  fileOpenAction->setMenuText(tr("&Open", "" ) );
  fileOpenAction->setAccel( 0 );
  fileOpenAction->setEnabled(!ro);
  
  editCopyAction = new QAction( this, "editCopyAction" );
  Q_CHECK_PTR(editCopyAction);
  editCopyAction->setIconSet( QIconSet(getPixmapIcon("copyIcon") ) );
  editCopyAction->setText(tr("Copy", "" ) );
  editCopyAction->setMenuText(tr("C&opy", "" ) );
  editCopyAction->setAccel( 4194371 );  

  editPasteAction = new QAction( this, "editPasteAction" );
  Q_CHECK_PTR(editPasteAction);
  editPasteAction->setIconSet( QIconSet(getPixmapIcon("pasteIcon") ) );
  editPasteAction->setText(tr("Paste", "" ) );
  editPasteAction->setMenuText(tr("&Paste", "" ) );
  editPasteAction->setAccel( 4194390 );
  editPasteAction->setEnabled(!ro);

  fileCloseAction = new QAction( this, "fileCloseAction" );
  Q_CHECK_PTR(fileCloseAction);
  fileCloseAction->setText(tr("Close", "" ) );
  fileCloseAction->setMenuText(tr("&Close", "" ) );
  fileCloseAction->setIconSet( QIconSet(getPixmapIcon("closeIcon")));
  fileCloseAction->setAccel( 0 );
  
  toolBar = new QToolBar(tr("Tools"), this, DockTop );
  Q_CHECK_PTR(toolBar);

  imageBar = new QToolBar(tr("Image Tools"), this, DockTop );
  Q_CHECK_PTR(imageBar);
  
  fileOpenAction->addTo( toolBar ); 
  
  p_saveImageTypeMenu = new QPopupMenu(this);
  Q_CHECK_PTR(p_saveImageTypeMenu);
  p_savePixmapTypeMenu = new QPopupMenu(this);
  Q_CHECK_PTR(p_savePixmapTypeMenu);

  QStrList fmt = QImage::outputFormats();  
  for (const char* f = fmt.first(); f; f = fmt.next())
  {
    p_saveImageTypeMenu->insertItem(f);
    p_savePixmapTypeMenu->insertItem(f);
  }

  saveImageTypeButton = new QToolButton(toolBar);
  Q_CHECK_PTR(saveImageTypeButton);
  saveImageTypeButton->setPopup(p_saveImageTypeMenu);
  saveImageTypeButton->setPixmap(getPixmapIcon("saveIcon"));
  saveImageTypeButton->setTextLabel("Save Image As", true);
  saveImageTypeButton->setPopupDelay (0);

  savePixmapTypeButton = new QToolButton(toolBar);
  Q_CHECK_PTR(savePixmapTypeButton);
  savePixmapTypeButton->setPopup(p_savePixmapTypeMenu);
  savePixmapTypeButton->setPixmap(getPixmapIcon("savePixmapIcon"));
  savePixmapTypeButton->setTextLabel("Save Pixmap As", true);
  savePixmapTypeButton->setPopupDelay (0);

  toolBar->addSeparator();
  editCopyAction->addTo( toolBar );  
  editPasteAction->addTo( toolBar );
  
  menubar = new QMenuBar( this, "menubar" );
  Q_CHECK_PTR(menubar);
  
  fileMenu = new QPopupMenu( this );
  Q_CHECK_PTR(fileMenu);
  fileOpenAction->addTo( fileMenu );  

  fileMenu->insertItem(getPixmapIcon("saveIcon"), tr("&Save Image As"), p_saveImageTypeMenu, 1);
  fileMenu->insertItem(getPixmapIcon("savePixmapIcon"), tr("Save &Pixmap As"), p_savePixmapTypeMenu, 2);

  fileMenu->insertSeparator();
  fileCloseAction->addTo( fileMenu );
  menubar->insertItem(tr("&File", "" ), fileMenu );
  
  editMenu = new QPopupMenu( this );
  Q_CHECK_PTR(editMenu);    
  editCopyAction->addTo( editMenu );
  editPasteAction->addTo( editMenu );
  menubar->insertItem(tr("&Edit", "" ), editMenu );

  colorOptions = new QActionGroup( this, "colorOptions", true );

  imageAutoColorAction = new QAction( colorOptions, "imageAutoColorAction", true );
  Q_CHECK_PTR(imageAutoColorAction);
  imageAutoColorAction->setText(tr("AutoColor") );
  imageAutoColorAction->setMenuText(tr("&AutoColor", "" ) );  
  imageAutoColorAction->setOn(true);

  imageColorOnlyAction = new QAction( colorOptions, "imageColorOnlyAction", true );
  Q_CHECK_PTR(imageColorOnlyAction);
  imageColorOnlyAction->setText(tr("ColorOnly") );
  imageColorOnlyAction->setMenuText(tr("&ColorOnly", "" ) );  

  imageMonoColorAction = new QAction( colorOptions, "imageMonoColorAction", true );
  Q_CHECK_PTR(imageMonoColorAction);
  imageMonoColorAction->setText(tr("MonoColor") );
  imageMonoColorAction->setMenuText(tr("&MonoColor", "" ) );  

  imageHFlipAction = new QAction( this, "imageHFlipAction" );
  Q_CHECK_PTR(imageHFlipAction);
  imageHFlipAction->setText(tr("Horizontal flip", "" ) );
  imageHFlipAction->setIconSet( QIconSet(getPixmapIcon("flipHIcon") ) );
  imageHFlipAction->setMenuText(tr("&Horizontal flip", "" ) );

  imageVFlipAction = new QAction( this, "imageVFlipAction" );
  Q_CHECK_PTR(imageVFlipAction);
  imageVFlipAction->setText(tr("Vertical flip", "" ) );
  imageVFlipAction->setIconSet( QIconSet(getPixmapIcon("flipVIcon") ) );
  imageVFlipAction->setMenuText(tr("&Vertical flip", "" ) );

  imageRotateAction = new QAction( this, "imageRotateAction" );
  Q_CHECK_PTR(imageRotateAction);
  imageRotateAction->setText(tr("Rotate 180", "" ) );
  imageRotateAction->setIconSet( QIconSet(getPixmapIcon("rotateIcon") ) );
  imageRotateAction->setMenuText(tr("&Rotate 180", "" ) );

  imageTo1BitAction = new QAction( this, "imageTo1BitAction" );
  Q_CHECK_PTR(imageTo1BitAction);
  imageTo1BitAction->setText(tr("Convert to 1 bit", "" ) );
  imageTo1BitAction->setIconSet( QIconSet(getPixmapIcon("bit1Icon")) );
  imageTo1BitAction->setMenuText(tr("Convert to &1 bit", "" ) );

  imageTo8BitAction = new QAction( this, "imageTo8BitAction" );
  Q_CHECK_PTR(imageTo8BitAction);
  imageTo8BitAction->setText(tr("Convert to 8 bit", "" ) );
  imageTo8BitAction->setIconSet( QIconSet(getPixmapIcon("bits8Icon") ) );
  imageTo8BitAction->setMenuText(tr("Convert to &8 bit", "" ) );

  imageTo32BitAction = new QAction( this, "imageTo32BitAction" );
  Q_CHECK_PTR(imageTo32BitAction);
  imageTo32BitAction->setText(tr("Convert to 32 bit", "" ) );
  imageTo32BitAction->setIconSet( QIconSet(getPixmapIcon("bits32Icon") ) );
  imageTo32BitAction->setMenuText(tr("Convert to &32 bit", "" ) );
   
  imageHFlipAction->addTo(imageBar);
  imageVFlipAction->addTo(imageBar);
  imageRotateAction->addTo(imageBar);
  imageBar->addSeparator();
  imageTo1BitAction->addTo(imageBar);
  imageTo8BitAction->addTo(imageBar);
  imageTo32BitAction->addTo(imageBar);

  imageMenu = new QPopupMenu( this );
  Q_CHECK_PTR(imageMenu);
  imageAutoColorAction->addTo( imageMenu );
  imageColorOnlyAction->addTo( imageMenu );
  imageMonoColorAction->addTo( imageMenu );
  imageMenu->insertSeparator();
  imageHFlipAction->addTo( imageMenu );
  imageVFlipAction->addTo( imageMenu );
  imageRotateAction->addTo( imageMenu );
  imageMenu->insertSeparator();
  imageTo1BitAction->addTo( imageMenu );
  imageTo8BitAction->addTo( imageMenu );
  imageTo32BitAction->addTo( imageMenu );

  menubar->insertItem(tr("&Image", "" ), imageMenu );
  setSaveEnabled(false);

  init();
}

CImageFieldEditor::~CImageFieldEditor()
{  
}

void CImageFieldEditor::loadImage(const char *fieldname, const char *data, uint len)
{
  editor->loadImage(fieldname, data, len);
}

void CImageFieldEditor::doResize(int w, int h)
{
  int nw = 2 + w + leftDock()->width() + rightDock()->width();
  int nh = 6 + h + topDock()->height() + bottomDock()->height() + Frame5->height() + statusBar()->height() + menubar->height(); 
  resize (nw, nh);
}

void CImageFieldEditor::setColorType(QAction * type)
{
  if (type == imageColorOnlyAction)
    editor->doColorChange(ColorOnly);
  else
    if (type == imageMonoColorAction)
      editor->doColorChange(MonoOnly);
    else    
      editor->doColorChange(AutoColor);
}

void CImageFieldEditor::myMessage(int type, const QString &msg)
{
  mysql->PrintError(type, msg);
}

void CImageFieldEditor::updateStatusBar(const QString &msg)
{
  statusBar()->clear();
  statusBar()->message(msg);
}

void CImageFieldEditor::setSaveEnabled(bool b)
{
  saveImageTypeButton->setEnabled(b);
  savePixmapTypeButton->setEnabled(b);
  fileMenu->setItemEnabled (1, b);
  fileMenu->setItemEnabled (2, b);
}

void CImageFieldEditor::saveImage(int id)
{
  QString f = p_saveImageTypeMenu->text(id);
  if (f.lower() == "jpg")
    f = "JPEG";
  editor->saveImage(f);
}

void CImageFieldEditor::savePixmap(int id)
{
  QString f = p_savePixmapTypeMenu->text(id);
  if (f.lower() == "jpg")
    f = "JPEG";
  editor->savePixmap(f);
}

void CImageFieldEditor::init()
{
  connect( fileCloseAction, SIGNAL( activated() ), this, SLOT( close() ) );
  connect( colorOptions, SIGNAL( selected( QAction * ) ), this, SLOT( setColorType( QAction * ) ) );
  connect( imageHFlipAction, SIGNAL( activated() ), editor, SLOT( hFlip() ) );
  connect( imageVFlipAction, SIGNAL( activated() ), editor, SLOT( vFlip() ) );
  connect( imageRotateAction, SIGNAL( activated() ), editor, SLOT( rot180() ) );
  connect( imageTo1BitAction, SIGNAL( activated() ), editor, SLOT( to1Bit() ) );
  connect( imageTo8BitAction, SIGNAL( activated() ), editor, SLOT( to8Bit() ) );
  connect( imageTo32BitAction, SIGNAL( activated() ), editor, SLOT( to32Bit() ) );

  connect( editor, SIGNAL( enableDepth1(bool) ), imageTo1BitAction, SLOT( setEnabled(bool) ) );
  connect( editor, SIGNAL( enableDepth8(bool) ), imageTo8BitAction, SLOT( setEnabled(bool) ) );
  connect( editor, SIGNAL( enableDepth32(bool) ), imageTo32BitAction, SLOT( setEnabled(bool) ) );

  connect( editor, SIGNAL( statusMessage(const QString &) ), this, SLOT( updateStatusBar(const QString &) ) );
  connect( editor, SIGNAL( myErr(int, const QString &) ), this, SLOT( myMessage(int, const QString &) ) );
  
  connect( editor, SIGNAL( canSave(bool) ), this, SLOT( setSaveEnabled(bool) ) );

  connect(p_saveImageTypeMenu, SIGNAL(activated(int)), this, SLOT(saveImage(int)));
  connect(p_savePixmapTypeMenu, SIGNAL(activated(int)), this, SLOT(savePixmap(int)));
 
  if (!readOnly)
  {
    connect( fileOpenAction, SIGNAL( activated() ), editor, SLOT( openFile() ) );
    connect( editPasteAction, SIGNAL( activated() ), editor, SLOT( paste() ) );
  }
}

