/* Copyright (C) 2002 MySQL AB & Jorge del Conde

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU Library General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.
    
  You should have received a copy of the GNU Library General Public
  License along with this library; if not, write to the Free
  Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
  MA 02111-1307, USA 
*/

#include "CImageEditor.h"
#include "CFieldEditorWindow.h"
#include "Globals.h"
#include <qmenubar.h>
#include <qfiledialog.h>
#include <qpopupmenu.h>
#include <qlabel.h>
#include <qpainter.h>
#include <qapplication.h>
#include <qclipboard.h>
#include <qmessagebox.h>

#include <ctype.h> /* for tolower() */

QString g_img_fname_prompt;
static inline void append_img_ext(QString& s, const char* ext);

CImageEditor::CImageEditor(QWidget *parent, QWidget *parentwidget, const char *name)
: QWidget(parent, name, Qt::WDestructiveClose | Qt::WResizeNoErase), conversion_flags(PreferDither), parentWidget(parentwidget), filename(0)
{
  pickx = -1;
  picky = -1;
  clickx = -1;
  clicky = -1;
  alloc_context = 0;  

  setSizePolicy(QSizePolicy((QSizePolicy::SizeType)7, (QSizePolicy::SizeType)7, 0, 0));
  setMenuItemFlags();
  setMouseTracking(true);
}

CImageEditor::~CImageEditor()
{
  if (alloc_context)
    QColor::destroyAllocContext(alloc_context);
  if (other == this)
    other = 0;
}

void CImageEditor::doColorChange(ImageConversionFlags f)
{
  int ocf = conversion_flags;
  conversion_flags = conversion_flags & ~ColorMode_Mask | f;
  if (ocf != conversion_flags)
  {
    setMenuItemFlags();	
    reconvertImage();
    repaint(image.hasAlphaBuffer());
  }  
}

void CImageEditor::setMenuItemFlags()
{  
  bool valid_image = pm.size() != QSize(0, 0);
  emit canSave(valid_image);   
  emit enableDepth1(image.depth() != 1);
  emit enableDepth8(image.depth() != 8);
  emit enableDepth32(image.depth() != 32);
}

void CImageEditor::updateStatus()
{
  if (pm.size() == QSize(0, 0))
  {
    if (filename)
      emit myErr(CRITICAL, tr("Could not load image"));
    else
      emit myErr(WARNING, tr("No image - select Open from File menu."));
  }
  else
  {
    QString message, moremsg;
    message.sprintf("%dx%d", image.width(), image.height());    
    if (pm.size() != pmScaled.size())
    {
      moremsg.sprintf(" [%dx%d]", pmScaled.width(), pmScaled.height());
      message += moremsg;
    }
    moremsg.sprintf(", %d ", image.depth());
    message += moremsg + tr("bits") + " ";
    if (image.valid(pickx,picky))
    {
      moremsg.sprintf("(%d,%d)=#%0*x ", pickx, picky, image.hasAlphaBuffer() ? 8 : 6, image.pixel(pickx,picky));
      message += moremsg;
    }
    if (image.numColors() > 0)
    {
      if (image.valid(pickx,picky))
        moremsg.sprintf(", %d/%d ", image.pixelIndex(pickx,picky), image.numColors());
      else
        moremsg.sprintf(", %d ", image.numColors());
      message += moremsg + tr("colors");
    }
    if (image.hasAlphaBuffer())
    {
      if (image.depth() == 8)
      {
        int i;
        bool alpha[256];
        int nalpha=0;
        
        for (i=0; i<256; i++)
          alpha[i] = false;
        
        for (i=0; i<image.numColors(); i++)
        {
          int alevel = image.color(i) >> 24;
          if (!alpha[alevel])
          {
            alpha[alevel] = true;
            nalpha++;
          }
        }
        moremsg.sprintf(", %d ", nalpha);
        moremsg += tr("alpha levels");
      }
      else
        moremsg = ", " + tr("8-bit alpha channel");
      message += moremsg;
    }
    emit statusMessage(message);
  }
}

void CImageEditor::saveImage(const QString & fmt)
{
  doSave(fmt, TYPE_IMAGE);  
}

void CImageEditor::savePixmap(const QString & fmt)
{
  doSave(fmt, TYPE_PIXMAP);  
}

void CImageEditor::doSave(const QString & fmt, int type)
{
  QString ext;
  append_img_ext(ext,(const char*)fmt);
  QString savefilename = getSaveName(fname, ext, fmt + "(*." + ext + ")");
  if (g_confirmCritical)
  {
    QFile file(savefilename);
    if (file.exists())
      if ((QMessageBox::warning(0, tr("Replace File"), tr("The specified file name already exists.\nDo you want to replace it ?"),
        QMessageBox::Yes, QMessageBox::No) != QMessageBox::Yes))              
        return;
    file.close();
  }
  if (!savefilename.isEmpty())
  {
    QString Type;
    bool ok = false;
    switch (type) {
      case TYPE_IMAGE:
        {
          Type = tr("Image");
          ok = image.save(savefilename, fmt.latin1());          
        }
        break;
      case TYPE_PIXMAP:
        {
          Type = tr("Pixmap");
          ok = pmScaled.save(savefilename, fmt.latin1());
        }
        break;
    }    
    if (!ok)
      emit myErr(CRITICAL, tr("Error saving file") + ": " + savefilename);
    else
      emit myErr(INFORMATION, Type + " " + savefilename + " " + tr("saved successfully."));
  }
}

static inline void append_img_ext(QString& s, const char* ext)
{
  const char *p;
  char c;
  for (p=ext; (c=tolower(*p)); p++)
  {
    if (c == 'e' && p == ext+2 && tolower(*ext) == 'j' &&
	tolower(ext[1]) == 'p')
      continue;
    s += c;
  }
}

void init_img_fname_prompt()
{
  g_img_fname_prompt = QObject::tr("Image Files",0);
  g_img_fname_prompt += '(';
  QStrList fmt = QImage::outputFormats();  
  for (const char* f = fmt.first(); f; f = fmt.next())
  {
    g_img_fname_prompt += '*';
    g_img_fname_prompt += '.';
    append_img_ext(g_img_fname_prompt,f);
    g_img_fname_prompt += ' ';
  }  
  g_img_fname_prompt += ");;";
  g_img_fname_prompt += QObject::tr("All Files");
  g_img_fname_prompt += " (*.*)";

}

void CImageEditor::openFile()
{
  QString newfilename = QFileDialog::getOpenFileName("", g_img_fname_prompt);
  if (!newfilename.isEmpty())
  {
    loadImage(newfilename) ;
    repaint();
  }
}

bool CImageEditor::loadImage(const char *fileName, const char *data, uint len)
{
  filename = fileName;
  bool ok = false;
  if (filename)
  {
    setCursor (Qt::waitCursor);
    ok = (data == 0 && len == 0) ? image.load(filename) : image.loadFromData ((const unsigned char *)data, len);
    pickx = -1;
    clickx = -1;
    if (ok)
      ok = reconvertImage();
    if (ok)
    {
      setCaption(filename);
      int w = pm.width();
      int h = pm.height();
      
      const int reasonable_width = 128;
      if (w < reasonable_width)
      {		
        int multiply = (reasonable_width + w - 1) / w;
        w *= multiply;
        h *= multiply;
      }           
      ((CFieldEditorWindow *)parentWidget)->doResize(w, h);
    }
    else
    {
      pm.resize(0,0);
      update();
    }
    setCursor(Qt::ArrowCursor);
  }
  updateStatus();
  setMenuItemFlags();
  return ok;
}

bool CImageEditor::reconvertImage()
{
  bool success = false;
  
  if (image.isNull()) return false;
  
  if (alloc_context)
  {
    QColor::destroyAllocContext(alloc_context);
    alloc_context = 0;
  }
  setCursor (Qt::waitCursor);  
  if (pm.convertFromImage(image, conversion_flags))
  {
    pmScaled = QPixmap();
    scale();
    resize(width(), height());
    success = true;
  }
  else    
    pm.resize(0,0);    
  updateStatus();
  setMenuItemFlags();
  setCursor(Qt::ArrowCursor);
  return success;
}

void CImageEditor::scale()
{
  int h = height();
  if (image.isNull())
    return;
  setCursor (Qt::waitCursor);
  if (width() == pm.width() && h == pm.height())    
    pmScaled = pm;
  else
  {
    QWMatrix m;
    m.scale(((double)width())/pm.width(),((double)h)/pm.height());
    pmScaled = pm.xForm(m);
  }
  setCursor(Qt::ArrowCursor);
}

void CImageEditor::resizeEvent(QResizeEvent *)
{
  if (pm.size() == QSize(0, 0))
    return;
  
  int h = height();
  if (width() != pmScaled.width() || h != pmScaled.height())
  {
    scale();
    updateStatus();
  }
  if (image.hasAlphaBuffer())
    erase();
}

bool CImageEditor::convertEvent(QMouseEvent* e, int& x, int& y)
{
  if (pm.size() != QSize(0, 0))
  {
    int h = height();
    int nx = e->x() * image.width() / width();
    int ny = e->y() * image.height() / h;
    if (nx != x || ny != y)
    {
      x = nx;
      y = ny;
      updateStatus();
      return true;
    }
  }
  return false;
}

void CImageEditor::mouseMoveEvent(QMouseEvent *e)
{
  if (convertEvent(e,pickx,picky))
  {
    updateStatus();
    if ((e->state()&LeftButton))
    {
      may_be_other = false;
      if (clickx >= 0 && other)      
        copyFrom(other);	    
    }
  }
}

void CImageEditor::paintEvent(QPaintEvent *e)
{
  if (pm.size() != QSize(0, 0))
  {
    QPainter painter(this);
    painter.setClipRect(e->rect());
    painter.drawPixmap(0, 0, pmScaled);
  }
}

void CImageEditor::copyFrom(CImageEditor* s)
{
  if (clickx >= 0)
  {
    int dx = clickx;
    int dy = clicky;
    int sx = s->clickx;
    int sy = s->clicky;
    int sw = QABS(clickx - pickx)+1;
    int sh = QABS(clicky - picky)+1;
    if (clickx > pickx)
    {
      dx = pickx;
      sx -= sw-1;
    }
    if (clicky > picky)
    {
      dy = picky;
      sy -= sh-1;
    }
    bitBlt(&image, dx, dy, &s->image, sx, sy, sw, sh);
    reconvertImage();
    repaint(image.hasAlphaBuffer());
  }
}

CImageEditor* CImageEditor::other = 0;

void CImageEditor::hFlip()
{
  setImage(image.mirror(true,false));
}

void CImageEditor::vFlip()
{
  setImage(image.mirror(false,true));
}

void CImageEditor::rot180()
{
  setImage(image.mirror(true,true));
}

void CImageEditor::copy()
{
#ifndef QT_NO_MIMECLIPBOARD
  QApplication::clipboard()->setImage(image);
#endif
}

void CImageEditor::paste()
{
#ifndef QT_NO_MIMECLIPBOARD
  QImage p = QApplication::clipboard()->image();
  if (!p.isNull())
  {
    filename = "pasted";
    setImage(p);
  }
#endif
}

void CImageEditor::setImage(const QImage& newimage)
{
  image = newimage;
  
  pickx = -1;
  clickx = -1;
  setCaption(filename);
  int w = image.width();
  int h = image.height();
  if (!w)
    return;
  
  const int reasonable_width = 128;
  if (w < reasonable_width)
  {	
    int multiply = (reasonable_width + w - 1) / w;
    w *= multiply;
    h *= multiply;
  }  
  reconvertImage();
  repaint(image.hasAlphaBuffer());
  
  updateStatus();
  setMenuItemFlags();
}

void CImageEditor::to1Bit()
{
  toBitDepth(1);
}

void CImageEditor::to8Bit()
{
  toBitDepth(8);
}

void CImageEditor::to32Bit()
{
  toBitDepth(32);
}

void CImageEditor::toBitDepth(int d)
{
  image = image.convertDepth(d);
  reconvertImage();
  repaint(image.hasAlphaBuffer());
}
