/* Copyright(C) 2002 MySQL AB & Jorge del Conde

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU Library General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or(at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.
    
  You should have received a copy of the GNU Library General Public
  License along with this library; if not, write to the Free
  Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
  MA 02111-1307, USA 
*/

#include "CDatabaseTree.h"
#include "CConfig.h"
#include "CConnectionDialog.h"
#include "CConsoleWindow.h"

CDatabaseTree::CDatabaseTree(QWidget * myparent, QWidget * parent, bool ismainwidget, const char * name, WFlags f) : QListView(parent, name, f)
{
  isMainWidget = ismainwidget;
  ConsoleWindow =(CConsoleWindow *)myparent;  
  cfgname = QString(CODENAME) + ".cfg";
  CConfig *cfg = new CConfig(cfgname);
  Q_CHECK_PTR(cfg);
  if(cfg->exists())
  {
    cfg->prepare();    
    automaticToolbars = strtobool(cfg->readStringEntry("Automatic Toolbars", "true"));
  }
  delete cfg;
  actionMenu = new QPopupMenu( this);
  Q_CHECK_PTR(actionMenu);

  if((ismainwidget) ||(!((CConsoleWindow *)myparent)->getCanClose()))
  {
    registerServerAction = new QAction(parent, "registerServerAction");
    Q_CHECK_PTR(registerServerAction);
    registerServerAction->setText(tr( "Register Server"));
    registerServerAction->setIconSet( QIconSet(getPixmapIcon("registerServerIcon")));
    registerServerAction->setMenuText(tr( "R&egister Server"));
    registerServerAction->setAccel( 0);
  
    refreshAction = new QAction(parent, "refreshAction");
    Q_CHECK_PTR(refreshAction);
    refreshAction->setText(tr( "Refresh"));
    refreshAction->setIconSet( QIconSet(getPixmapIcon("refreshIcon")));
    refreshAction->setMenuText(tr( "&Refresh"));
    refreshAction->setAccel( 0);
  
    actionToolBar = new QToolBar((QMainWindow *) myparent);
    Q_CHECK_PTR(actionToolBar);
    actionToolBar->setLabel(tr( "Tools"));
    
    registerServerAction->addTo(actionToolBar);
    actionToolBar->addSeparator();
    refreshAction->addTo( actionToolBar);    
    
    registerServerAction->addTo(actionMenu);
    
    actionMenu->insertSeparator();
    refreshAction->addTo(actionMenu);
    actionMenu->insertSeparator();
    connect(registerServerAction, SIGNAL(activated()), this, SLOT(processRegisterServerSlot()));	
    connect(refreshAction, SIGNAL(activated()), this, SLOT(processRefreshSlot()));
  }  
  showTreeAction = new QAction(parent, "showTreeAction");
  Q_CHECK_PTR(showTreeAction);
  showTreeAction->setText(tr( "Show Tree"));  
  showTreeAction->setMenuText(tr( "&Show Tree"));
  showTreeAction->setAccel( 0);
  showTreeAction->setToggleAction(true);
  showTreeAction->setOn(parentWidget()->isVisible());      
  
  showMessagesAction = new QAction(parent, "showMessagesAction");
  Q_CHECK_PTR(showMessagesAction);
  showMessagesAction->setText(tr( "Show Messages Panel"));  
  showMessagesAction->setMenuText(tr( "Show &Messages Panel"));
  showMessagesAction->setAccel( 0);
  showMessagesAction->setToggleAction(true);
  showMessagesAction->setOn(ConsoleWindow->messageWindow()->isVisible());

  ServerItemToolBar = new CServerItemToolbar((QMainWindow *)myparent);
  Q_CHECK_PTR(ServerItemToolBar);
  toolBars.append(ServerItemToolBar);
  
  DatabaseGroupToolBar = new CDatabaseGroupToolbar((QMainWindow *)myparent);
  Q_CHECK_PTR(DatabaseGroupToolBar);
  toolBars.append(DatabaseGroupToolBar);  
  
  DatabaseItemToolBar = new CDatabaseItemToolbar((QMainWindow *)myparent);
  Q_CHECK_PTR(DatabaseItemToolBar);
  toolBars.append(DatabaseItemToolBar);  
  
  TableGroupToolBar = new CTableGroupToolbar((QMainWindow *)myparent);
  Q_CHECK_PTR(TableGroupToolBar);
  toolBars.append(TableGroupToolBar);  
  
  TableItemToolBar = new CTableItemToolbar((QMainWindow *)myparent);
  Q_CHECK_PTR(TableItemToolBar);
  toolBars.append(TableItemToolBar);
  
  ServerAdministrationItemToolBar = new CServerAdministrationItemToolbar((QMainWindow *)myparent);
  Q_CHECK_PTR(ServerAdministrationItemToolBar);
  toolBars.append(ServerAdministrationItemToolBar);

  UserAdminItemToolBar = new CUserAdminItemToolbar((QMainWindow *)myparent);
  Q_CHECK_PTR(UserAdminItemToolBar);
  toolBars.append(UserAdminItemToolBar);

  UserItemToolBar = new CUserItemToolbar((QMainWindow *)myparent);
  Q_CHECK_PTR(UserItemToolBar);
  toolBars.append(UserItemToolBar);
  
  enableToolbars(false);

  showTreeAction->addTo(actionMenu);
  showMessagesAction->addTo(actionMenu);
  
  if(!ismainwidget)
  {    
    actionMenu->insertSeparator();
    closeAction = new QAction(parent, "closeAction");
    Q_CHECK_PTR(closeAction);
    closeAction->setMenuText(tr("&Close"));
    closeAction->setAccel( 0);
    closeAction->setIconSet( QIconSet(getPixmapIcon("closeIcon")));
    connect(closeAction, SIGNAL(activated()), myparent, SLOT(close()));
    closeAction->addTo(actionMenu);
  }
  
  toolBarMenu = new QPopupMenu( this);
  toolBarMenu->setCheckable( true);
  
 ((CConsoleWindow *)myparent)->menuBar()->insertItem(tr("&Action"), actionMenu);
 ((CConsoleWindow *)myparent)->menuBar()->insertItem(tr("&Toolbars"), toolBarMenu);  
  
  addColumn(tr("MySQL Servers"));
  setRootIsDecorated(false);
  setResizeMode(QListView::AllColumns);
  setColumnWidth(0, 150);
  init();
  defaultMenu = true;
  setMargin(0);
  setShowSortIndicator(true);
}

void CDatabaseTree::enableToolbars(bool enable)
{
  CToolBar *t;
  for(t = toolBars.first(); t; t = toolBars.next())  
    t->setEnabled(enable);
}

void CDatabaseTree::automaticToolbarsSlot()
{
  automaticToolbars = !automaticToolbars;
  if(g_saveWorkspace)
  {
    CConfig *cfg = new CConfig(cfgname);
    Q_CHECK_PTR(cfg);
    if(cfg->exists())
    {
      cfg->prepare();    
      cfg->writeEntry("Automatic Toolbars", booltostr(automaticToolbars));
      cfg->flush();
    }
    delete cfg;
  }
  configureToolbars((CDatabaseTreeItem *)currentItem());
  
  if(!automaticToolbars)
  {
    CToolBar *t;
    for(t = toolBars.first(); t; t = toolBars.next())
      t->readConfig();
  }
}

void CDatabaseTree::toolbarMenuAboutToShow()
{
  toolBarMenu->clear();
  int id = toolBarMenu->insertItem(tr("Automatic Toolbars"), this, SLOT(automaticToolbarsSlot()));
  toolBarMenu->setItemChecked(id, automaticToolbars);
  toolBarMenu->insertSeparator();
  CToolBar *t;
  for(t = toolBars.first(); t; t = toolBars.next())
  {
    if(t->getMyVisible())
    {
      int id = toolBarMenu->insertItem(t->label(), t,((automaticToolbars) ? SLOT(myShow()) : SLOT(myShowWrite())));
      toolBarMenu->setItemChecked(id, t->isVisible());    
      toolBarMenu->setItemEnabled(id, !automaticToolbars);
    }    
  }
}

void CDatabaseTree::ToolbarButtonClicked(int type, int result)
{
  if((currentItem() == NULL) ||(currentItem() == 0))
    return;
  CDatabaseTreeItem *item =(CDatabaseTreeItem *)currentItem(); 
  while((item != NULL) ||(item != 0))
  {    
    if(item->getType() == type)
    {
      if(result == NEW_WINDOW_FROM_HERE)
        OpenInNewWindow(item);
      else
        item->processMenu(result);
      break;
    }
    else
      item =(CDatabaseTreeItem *) item->parent();
  }
  configureToolbars((CDatabaseTreeItem *)currentItem());
}

void CDatabaseTree::setVisibleToolbar(CToolBar *v)
{
  CToolBar *t;
  for(t = toolBars.first(); t; t = toolBars.next())  
    t->setVisible((t->getType() == v->getType()));
}

void CDatabaseTree::configureToolbars(CDatabaseTreeItem * item)
{
  if((item == NULL) ||(item == 0))
    return;
  
  if((currentItem() == NULL) ||(currentItem() == 0))
    return;

  if(!item->isVisible())
    return;
  
  int major, minor;  
  item->getServerVersion(major, minor);
  switch(item->getType())
  {
  case SERVER_ITEM: {
    if(automaticToolbars)
      setVisibleToolbar(ServerItemToolBar);
    ServerItemToolBar->setMyEnabled(true, major, minor);
    DatabaseGroupToolBar->setMyEnabled(false, major, minor);
    DatabaseItemToolBar->setMyEnabled(false, major, minor);
    TableGroupToolBar->setMyEnabled(false, major, minor);
    TableItemToolBar->setMyEnabled(false, major, minor);
    ServerAdministrationItemToolBar->setMyEnabled(false, major, minor);
    UserAdminItemToolBar->setMyEnabled(false, major, minor);
    UserItemToolBar->setMyEnabled(false, major, minor);
    ServerItemToolBar->setConnectedToolbar(item->isConnected());
                    }
    break;
  case DATABASE_GROUP: {
    if(automaticToolbars)
      setVisibleToolbar(DatabaseGroupToolBar);
    ServerItemToolBar->setMyEnabled(true, major, minor);
    DatabaseGroupToolBar->setMyEnabled(true, major, minor);
    DatabaseItemToolBar->setMyEnabled(false, major, minor);
    TableGroupToolBar->setMyEnabled(false, major, minor);
    TableItemToolBar->setMyEnabled(false, major, minor);
    ServerAdministrationItemToolBar->setMyEnabled(false, major, minor);
    UserAdminItemToolBar->setMyEnabled(false, major, minor);
    UserItemToolBar->setMyEnabled(false, major, minor);

    ServerItemToolBar->setConnectedToolbar(item->isConnected());
                       }
    break;
  case DATABASE_ITEM: {
    if(automaticToolbars)
      setVisibleToolbar(DatabaseItemToolBar);
    ServerItemToolBar->setMyEnabled(true, major, minor);
    DatabaseGroupToolBar->setMyEnabled(true, major, minor);
    DatabaseItemToolBar->setMyEnabled(true, major, minor);
    TableGroupToolBar->setMyEnabled(false, major, minor);
    TableItemToolBar->setMyEnabled(false, major, minor);    
    ServerAdministrationItemToolBar->setMyEnabled(false, major, minor);
    UserAdminItemToolBar->setMyEnabled(false, major, minor);
    UserItemToolBar->setMyEnabled(false, major, minor);
    DatabaseItemToolBar->setConnectedToolbar(item->isConnected());

    ServerItemToolBar->setConnectedToolbar(true);
                      }
    break;
  case TABLE_GROUP: {
    if(automaticToolbars)
      setVisibleToolbar(TableGroupToolBar);
    ServerItemToolBar->setMyEnabled(true, major, minor);
    DatabaseGroupToolBar->setMyEnabled(true, major, minor);
    DatabaseItemToolBar->setMyEnabled(true, major, minor);
    TableGroupToolBar->setMyEnabled(true, major, minor);
    TableItemToolBar->setMyEnabled(false, major, minor);
    ServerAdministrationItemToolBar->setMyEnabled(false, major, minor);
    UserAdminItemToolBar->setMyEnabled(false, major, minor);
    UserItemToolBar->setMyEnabled(false, major, minor);

    DatabaseItemToolBar->setConnectedToolbar(true);
    ServerItemToolBar->setConnectedToolbar(true);
                    }
    break;
  case TABLE_ITEM: {
    if(automaticToolbars)
      setVisibleToolbar(TableItemToolBar);
    ServerItemToolBar->setMyEnabled(true, major, minor);
    DatabaseGroupToolBar->setMyEnabled(true, major, minor);
    DatabaseItemToolBar->setMyEnabled(true, major, minor);
    TableGroupToolBar->setMyEnabled(true, major, minor);
    TableItemToolBar->setMyEnabled(true, major, minor);
    ServerAdministrationItemToolBar->setMyEnabled(false, major, minor);
    UserAdminItemToolBar->setMyEnabled(false, major, minor);
    UserItemToolBar->setMyEnabled(false, major, minor);

    DatabaseItemToolBar->setConnectedToolbar(true);
    ServerItemToolBar->setConnectedToolbar(true);
                   }
    break;
    
  case SERVER_ADMINISTRATION_ITEM: {
    if(automaticToolbars)
      setVisibleToolbar(ServerAdministrationItemToolBar);
    ServerItemToolBar->setMyEnabled(true, major, minor);
    DatabaseGroupToolBar->setMyEnabled(false, major, minor);
    DatabaseItemToolBar->setMyEnabled(false, major, minor);
    TableGroupToolBar->setMyEnabled(false, major, minor);
    TableItemToolBar->setMyEnabled(false, major, minor);
    ServerAdministrationItemToolBar->setMyEnabled(true, major, minor);
    UserAdminItemToolBar->setMyEnabled(false, major, minor);
    UserItemToolBar->setMyEnabled(false, major, minor);
    
    ServerItemToolBar->setConnectedToolbar(true);
                                   }
    break;
  case USER_ADMINISTRATION_ITEM: {
    if(automaticToolbars)
      setVisibleToolbar(UserAdminItemToolBar);
    ServerItemToolBar->setMyEnabled(true, major, minor);
    DatabaseGroupToolBar->setMyEnabled(false, major, minor);
    DatabaseItemToolBar->setMyEnabled(false, major, minor);
    TableGroupToolBar->setMyEnabled(false, major, minor);
    TableItemToolBar->setMyEnabled(false, major, minor);
    ServerAdministrationItemToolBar->setMyEnabled(false, major, minor);
    UserAdminItemToolBar->setMyEnabled(true, major, minor);
    UserItemToolBar->setMyEnabled(false, major, minor);
    UserAdminItemToolBar->setConnectedToolbar(item->isConnected());
    
    ServerItemToolBar->setConnectedToolbar(true);
                                 }
    break;
  case USER_ITEM: {
    if(automaticToolbars)
      setVisibleToolbar(UserItemToolBar);
    ServerItemToolBar->setMyEnabled(true, major, minor);
    DatabaseGroupToolBar->setMyEnabled(false, major, minor);
    DatabaseItemToolBar->setMyEnabled(false, major, minor);
    TableGroupToolBar->setMyEnabled(false, major, minor);
    TableItemToolBar->setMyEnabled(false, major, minor);
    ServerAdministrationItemToolBar->setMyEnabled(false, major, minor);
    UserAdminItemToolBar->setMyEnabled(true, major, minor);
    UserItemToolBar->setMyEnabled(true, major, minor);
    
    ServerItemToolBar->setConnectedToolbar(true);
                  }
    break;
  }
}

void CDatabaseTree::drawItems(CDatabaseTreeItem * itemRoot)
{
  setCursor(Qt::waitCursor);
  if((itemRoot != 0) &&(widgetStack != NULL))
  {
    isNewWindow = true;
    defaultMenu = false;
    switch(itemRoot->getType()) {
    case SERVER_ITEM: {
			new CServerItem(ConsoleWindow, itemRoot->getServerID(), this, itemRoot->text(0), itemRoot->isConnected(), widgetStack);						
      break;
                      }
    case DATABASE_GROUP: {
      ServerItemToolBar->setMyVisible(false);            
			CMySQLConnection * m = new CMySQLConnection(itemRoot->m_pMySQLServer->getConnectionName(), ConsoleWindow->messagePanel());
      Q_CHECK_PTR(m);
      new CDatabaseGroupItem(ConsoleWindow, itemRoot->getServerID(), this, m, widgetStack);
      break;
                         }
    case DATABASE_ITEM: {

      ServerItemToolBar->setMyVisible(false);
      DatabaseGroupToolBar->setMyVisible(false);
			CMySQLConnection * m = new CMySQLConnection(itemRoot->m_pMySQLServer->getConnectionName(), ConsoleWindow->messagePanel());
      Q_CHECK_PTR(m);
      QString tmpname = itemRoot->text(0);
      new CDatabaseItem(ConsoleWindow, itemRoot->getServerID(), this, m, tmpname, itemRoot->isConnected(), widgetStack);						
      break;
                        }
    case TABLE_GROUP: {
      ServerItemToolBar->setMyVisible(false);
      DatabaseGroupToolBar->setMyVisible(false);
      DatabaseItemToolBar->setMyVisible(false);
			CMySQLConnection * m = new CMySQLConnection(itemRoot->m_pMySQLServer->getConnectionName(), ConsoleWindow->messagePanel());
      Q_CHECK_PTR(m);      
      CMySQLConnection * d = new CMySQLConnection(itemRoot->m_pMySQLServer->getConnectionName(), ConsoleWindow->messagePanel());
      Q_CHECK_PTR(d);
      d->connect(itemRoot->m_pMySQLDatabase->getDatabaseName());
      CTableGroupItem * p = new CTableGroupItem(ConsoleWindow, itemRoot->getServerID(), this, m, itemRoot->m_pMySQLDatabase->getDatabaseName(), d, widgetStack);
      Q_CHECK_PTR(p);
      p->setOpen(true);
      break;
                      }
    case SERVER_ADMINISTRATION_ITEM: {
      ServerItemToolBar->setMyVisible(false);      
			CMySQLConnection * m = new CMySQLConnection(itemRoot->m_pMySQLServer->getConnectionName(), ConsoleWindow->messagePanel());
      Q_CHECK_PTR(m);
      new CServerAdministrationItem(ConsoleWindow, this, m, widgetStack);
      break;
                                     }
    case USER_ADMINISTRATION_ITEM: {
      ServerItemToolBar->setMyVisible(false);      
			CMySQLConnection * m = new CMySQLConnection(itemRoot->m_pMySQLServer->getConnectionName(), ConsoleWindow->messagePanel());
      Q_CHECK_PTR(m);
      new CUserAdminItem(ConsoleWindow, this, itemRoot->isConnected(), m, widgetStack);
      break;
                                     }  
    }
  }
  else
  {
    isNewWindow = false;
    addServers(true);  
  }  
  hideItemToolbars();
  setCursor(Qt::ArrowCursor);
  if(currentItem() == 0)
    setCurrentItem(firstChild());
  configureToolbars((CDatabaseTreeItem *)currentItem());
}

CDatabaseTree::~CDatabaseTree()
{
#ifdef DEBUG
    qDebug( "~CDatabaseTree() ");
#endif
}

void CDatabaseTree::ConnectionRenamed(CDatabaseTreeItem *item, const QString & newName)
{
  item->setText(0, newName);
}

void CDatabaseTree::hideItemToolbars()
{
  QListViewItemIterator it( this);
  bool hasVisibleItem = false;
  for(; it.current(); ++it)
  {
    CDatabaseTreeItem * p_Item =(CDatabaseTreeItem *) it.current();
    if(p_Item->parent() == NULL)
    {
      if(p_Item->isVisible())
      {
        setCurrentItem(p_Item);
        setSelected(p_Item, true);
        Clicked(1, p_Item, QPoint(), 0);
        hasVisibleItem = true;
        break;
      }
    }
  }

  if(!hasVisibleItem)
  {
    CToolBar *t;
    for(t = toolBars.first(); t; t = toolBars.next())
    {
      t->setVisible(false);
      t->setEnabled(false);
    }
    widgetStack->raiseWidget(0);
  }
}

void CDatabaseTree::refreshServers()
{  
  QListViewItemIterator it( this);
  bool del = false;
  for(; it.current(); ++it)
  {
    CDatabaseTreeItem * p_Item =(CDatabaseTreeItem *) it.current();
    if(p_Item->parent() == NULL)  //parent is this therefore item == SERVER_ITEM
    {       
      QString *serverName = g_ServersDict->find(((CServerItem *) p_Item)->getServerID());
      if((serverName == NULL) ||(!CConfig::exists(*serverName)))
      {
       ((CServerItem *)p_Item)->removeServer(); //Item has been deleted.
        del = true;
      }
      else
      {
        if(p_Item->text(0) != *serverName)
          p_Item->setText(0, *serverName);  //Item was renamed
        p_Item->m_pMySQLServer->update();
      }
    }
  }  
  if(del)
    hideItemToolbars();
  addServers(false);
}

void CDatabaseTree::addServers(bool all)
{
  QStringList dbServerList;
  CConfig::list(dbServerList, g_CONNECTION_PATH);
  bool add = all;
  for(QStringList::Iterator j = dbServerList.begin(); j != dbServerList.end(); j++)
  {   
    add =(!all) ?(findItem((*j), 0) == 0) : true;
    if(add)
      new CServerItem(ConsoleWindow, g_serverID++, this,(*j), false, widgetStack);
  }
}

void CDatabaseTree::processMenu(int res)
{  
  switch(res)
  {
		case MENU_NEW: {      
      CConnectionDialog* pConnectionDialog = new CConnectionDialog(ConsoleWindow,(QWidget *)g_WorkSpace, "Connection Dialog");
      Q_CHECK_PTR(pConnectionDialog);
      connect(pConnectionDialog, SIGNAL(newConnection(bool)), this, SLOT(addServers(bool)));
      myShowWindow(pConnectionDialog);      	
      break;
                   }      
    case MENU_REFRESH: {
      refreshServers();
      ConsoleWindow->messagePanel()->Information(tr("Refresh successful"));
      break;
                       }
      
    case MENU_HIDE_TREE: {
      showTreeAction->setOn(false);
      parentWidget()->hide();
      break;
                         }
    case MENU_SHOW_TREE: {
      showTreeAction->setOn(true);
      parentWidget()->show();
      break;
                         }
    case MENU_HIDE_MESSAGES: {
      showMessagesAction->setOn(false);
      ConsoleWindow->messageWindow()->hide();
      break;
                             }
    case MENU_SHOW_MESSAGES: {
      showMessagesAction->setOn(true);
      ConsoleWindow->messageWindow()->show();
      break;
                             }
      
  }
}

void CDatabaseTree::displayMenu(const QPoint & pos)
{
  QPopupMenu *p_itemMenu = new QPopupMenu();
  Q_CHECK_PTR(p_itemMenu);
  p_itemMenu->insertItem(registerServerAction->iconSet(), registerServerAction->text(), MENU_NEW);  
  p_itemMenu->insertItem(refreshAction->iconSet(), refreshAction->text(), MENU_REFRESH);
  p_itemMenu->insertSeparator();
  p_itemMenu->insertItem(tr("Hide"), MENU_HIDE_TREE);
  int res = p_itemMenu->exec(pos);
  delete p_itemMenu;
  processMenu(res);
}

void CDatabaseTree::Collapsed(QListViewItem *item)
{  
  if((item == 0) ||(item == NULL))
    return;
  CDatabaseTreeItem * p_Item =(CDatabaseTreeItem *) item;
  p_Item->collapsed();
}

void CDatabaseTree::Expanded( QListViewItem *item)
{  
  if((item == 0) ||(item == NULL))
    return;
  CDatabaseTreeItem * p_Item =(CDatabaseTreeItem *) item;
  p_Item->expanded();
}

void CDatabaseTree::DoubleClicked( QListViewItem *item)
{
 if((item == 0) ||(item == NULL))
    return;
  CDatabaseTreeItem * p_Item =(CDatabaseTreeItem *) item;  
  p_Item->setCurrentText(p_Item->text(0));
  p_Item->doubleClicked();
  configureToolbars((CDatabaseTreeItem *)currentItem());
}

void CDatabaseTree::ItemRenamed( QListViewItem * item, int, const QString & text)
{  
  if((item == 0) ||(item == NULL))
    return;
  CDatabaseTreeItem * p_Item =(CDatabaseTreeItem *) item;
  p_Item->renamed(p_Item->getCurrentText(), text);
}

void CDatabaseTree::CurrentChanged( QListViewItem *item)
{
  if((item == 0) ||(item == NULL))
    return;  
  CDatabaseTreeItem * p_Item =(CDatabaseTreeItem *) item;
  CDatabaseTreeItem * p_Item_tmp = p_Item;  
  p_Item->setCurrentText(p_Item->text(0));
  p_Item->clicked();
  if(!isNewWindow)
    while((p_Item != NULL) ||(p_Item != 0))
    {
      configureToolbars(p_Item);
      p_Item =(CDatabaseTreeItem *) p_Item->parent();
    }
  configureToolbars(p_Item_tmp);
}

void CDatabaseTree::ReturnPressed( QListViewItem *item)
{  
  if((item == 0) ||(item == NULL))
    return;
  CDatabaseTreeItem * p_Item =(CDatabaseTreeItem *) item;  
  p_Item->setCurrentText(p_Item->text(0));
  p_Item->setOpen(!p_Item->isOpen());
  p_Item->doubleClicked();
}

void CDatabaseTree::SpacePressed( QListViewItem *item)
{
  if((item == 0) ||(item == NULL))
    return;
  CDatabaseTreeItem * p_Item =(CDatabaseTreeItem *) item;  
  p_Item->setCurrentText(p_Item->text(0));
  p_Item->setOpen(!p_Item->isOpen());
  p_Item->startRename(0);
}

void CDatabaseTree::Clicked( int button, QListViewItem * item, const QPoint &, int)
{
  if((item == 0) ||(item == NULL) ||(button != 1))
    return;	
  CDatabaseTreeItem * p_Item =(CDatabaseTreeItem *) item;  
  p_Item->setCurrentText(p_Item->text(0));
  p_Item->clicked();
}

void CDatabaseTree::startDrag()
{
 ((CDatabaseTreeItem *)currentItem())->startDrag();
}

void CDatabaseTree::RightButtonClicked( QListViewItem *item, const QPoint & pos, int)
{
  if(!item)
  {
    if(defaultMenu)
      displayMenu(pos);
    return;
  }  
  int res =((CDatabaseTreeItem *) item)->displayMenu(pos);
  
  if(res == NEW_WINDOW_FROM_HERE)
    OpenInNewWindow(((CDatabaseTreeItem *) item));
  else  
   ((CDatabaseTreeItem *) item)->processMenu(res);
  configureToolbars((CDatabaseTreeItem *)currentItem());
}

void CDatabaseTree::OpenInNewWindow(CDatabaseTreeItem *p_Item)
{
  CConsoleWindow * consoleWindow = new CConsoleWindow(g_WorkSpace, "CConsoleWindow2", WDestructiveClose, p_Item, !g_isMDI);
  Q_CHECK_PTR(consoleWindow);
  consoleWindow->setCaption(p_Item->getTitle());  
  consoleWindow->statusBar()->message("[" + p_Item->m_pMySQLServer->getConnectionName() + "]  "
    +((p_Item->m_pMySQLServer->getConnectionName() == p_Item->text(0)) ? QString("") : p_Item->text(0)));
  consoleWindow->m_pDatabaseTree->setColumnText(0,p_Item->text(0));
  if(g_isMDI)
    connect(((QMainWindow *)(qApp->mainWidget())), SIGNAL(undockwindows()), consoleWindow, SLOT(UndockWindows()));
  consoleWindow->messageWindow()->setCarriageReturn(p_Item->m_pMySQLServer->getLineTerminator(true));    
  myShowWindow(consoleWindow);
  consoleWindow->setFocus();
  consoleWindow->raise();  
}

void CDatabaseTree::processRegisterServerSlot()
{
  processMenu(MENU_NEW);
}

void CDatabaseTree::processRefreshSlot()
{
 processMenu(MENU_REFRESH);
}

void CDatabaseTree::processShowTreeSlot()
{  
  processMenu((showTreeAction->isOn()) ? MENU_SHOW_TREE : MENU_HIDE_TREE);
}

void CDatabaseTree::processShowMessagesSlot()
{  
  processMenu((showMessagesAction->isOn()) ? MENU_SHOW_MESSAGES : MENU_HIDE_MESSAGES);
}    

void CDatabaseTree::init()
{  
  connect(this, SIGNAL(contextMenuRequested(QListViewItem *, const QPoint &, int)), this, SLOT(RightButtonClicked( QListViewItem *, const QPoint &, int)));
  connect(this, SIGNAL(mouseButtonClicked(int, QListViewItem *, const QPoint &, int)), this, SLOT(Clicked(int, QListViewItem *, const QPoint &, int)));
  connect(this, SIGNAL(doubleClicked(QListViewItem *)), this, SLOT(DoubleClicked( QListViewItem *)));
  connect(this, SIGNAL(expanded(QListViewItem *)), this, SLOT(Expanded( QListViewItem *)));
  connect(this, SIGNAL(collapsed(QListViewItem *)), this, SLOT(Collapsed( QListViewItem *)));
  connect(this, SIGNAL(itemRenamed(QListViewItem *, int, const QString &)), this, SLOT(ItemRenamed(QListViewItem *, int, const QString &)));
  connect(this, SIGNAL(currentChanged(QListViewItem *)), this, SLOT(CurrentChanged( QListViewItem *)));
  connect(this, SIGNAL(returnPressed(QListViewItem *)), this, SLOT(ReturnPressed( QListViewItem *)));
  connect(this, SIGNAL(spacePressed(QListViewItem *)), this, SLOT(SpacePressed( QListViewItem *)));
  connect(showTreeAction, SIGNAL(activated()), this, SLOT(processShowTreeSlot()));  
  connect(toolBarMenu, SIGNAL(aboutToShow()), this, SLOT(toolbarMenuAboutToShow()));
  connect(parentWidget(), SIGNAL(visibilityChanged(bool)), showTreeAction, SLOT(setOn(bool)));
  connect(showMessagesAction, SIGNAL(activated()), this, SLOT(processShowMessagesSlot()));
  connect(ConsoleWindow->messageWindow(), SIGNAL(visibilityChanged(bool)), showMessagesAction, SLOT(setOn(bool)));
  CToolBar *t;
  for(t = toolBars.first(); t; t = toolBars.next())
    connect(t, SIGNAL(buttonClicked(int, int)), this, SLOT(ToolbarButtonClicked(int, int)));   
}
