/*+*****************************************************************************
*                                                                              *
* File: xp.c                                                                   *
*                                                                              *
* Description: X protocol handling                                             *
*                                                                              *
**-****************************************************************************/

#ifndef __lint
char xp_vers[] = "@(#)xp.c	1.16	02/18/99	Written by Lionel Cons";
#endif /* __lint */

/******* Include Files ********************************************************/

#include "xp.h"
#include "util.h"
#include "io.h"
#include "mc.h"
#include "xu.h"
#include "xn.h"
#include "host.h"
#include "xmd.h"
#include "config.h"

/******* Constants ************************************************************/

/*
 * from Xproto.h
 */
#define X_Error                          0
#define X_Reply	                         1

#define X_ChangeWindowAttributes         2
#define X_GetAtomName                   17
#define X_ChangeProperty                18 
#define X_SendEvent                     25
#define X_GrabPointer                   26
#define X_GrabButton                    28
#define X_GrabKeyboard                  31
#define X_GrabKey                       33
#define X_GrabServer                    36      
#define X_QueryKeymap                   44       
#define X_ChangeKeyboardMapping        100
#define X_ChangeHosts                  109
#define X_ListHosts                    110
#define X_SetAccessControl             111
#define X_NoOperation                  127

/*
 * from X.h
 */
#define CWEventMask               (1L<<11)
#define CWEventIndex			11 /* index of CWEventMask */
#define PropertyChangeMask	  (1L<<22) 
#define Expose                          12
#define SelectionClear                  29
#define SelectionRequest                30
#define SelectionNotify                 31
#define ClientMessage                   33

/*
 * from Xatom.h
 */
#define XA_ATOM		       ((Atom)  4)
#define XA_CUT_BUFFER0         ((Atom)  9)
#define XA_CUT_BUFFER7         ((Atom) 16)
#define XA_RESOURCE_MANAGER    ((Atom) 23)

/*
 * maximum number of arguments to ChangeWindowAttributes
 */
#define MAX_CWA_ARGS	16

/******* Macros ***************************************************************/

/* handy macros to extract packet info */
#define BUF8(pos)	getCARD8(buf, pos)
#define BUF16(pos)	getCARD16(xip->sex, &buf[pos])
#define BUF32(pos)	getCARD32(xip->sex, &buf[pos])

/* test if the window belongs to the connection */
#define OWNING(win)	((win & ~xip->resid_mask) == xip->resid_base)

/******* External Stuff *******************************************************/

/******* Local Stuff **********************************************************/

/*
 * initialise X info
 */
void xp_init(XINFO *xip)
{
  xip->sex = '?'; /* dummy sex */
  xip->seq = 0;
  xip->resid_base = 0;
  xip->resid_mask = 0;
  xip->sel_req = 0;
  xip->sel_prop = 0;
  xip->pkt_count = 0;
  xip->pkt_size = 0;
  xip->alert = NULL;
  xip->alert_count = 0;
  xip->rep_first = 0;
  xip->rep_count = 0;
  xip->fake_request = NULL;
  xip->fake_packet = NULL;
}

/*
 * add a fake packet to the output queue
 */
static void queue_fake_packet(XINFO *xip, MC *fp)
{
  MC *mcp;

  if (xip->fake_packet == NULL) {
    /* empty queue */
    xip->fake_packet = fp;
    return;
  }
  /* go to the end of the queue */
  for (mcp=xip->fake_packet; mcp->next; mcp=mcp->next) ;
  mcp->next = fp;
}

/*
 * create a fake NoOperation request
 */
static MC *create_noop_request(XINFO *xip)
{
  MC *mcp;

  mcp = mc_new(4);
  putCARD8(mcp->base, 0, X_NoOperation); /* type */
  putCARD8(mcp->base, 1, 0);	         /* unused */
  putCARD16(xip->sex, &mcp->base[2], 1); /* length */
  return(mcp);
}

/*
 * create a fake GetAtomName request (to trigger a reply)
 */
static MC *create_gan_request(XINFO *xip)
{
  MC *mcp;

  mcp = mc_new(8);
  putCARD8(mcp->base, 0, X_GetAtomName); /* type */
  putCARD8(mcp->base, 1, 0);	         /* unused */
  putCARD16(xip->sex, &mcp->base[2], 2); /* length */
  putCARD32(xip->sex, &mcp->base[4], XA_ATOM); /* atom */
  return(mcp);
}

/*
 * replace the current packet by an inoffensive one by queuing
 * a fake NoOp request (to replace a request without reply) or
 * a fake GetAtomName (to replace a request with reply);
 * for the later, also insert a fake server reply 
 *
 * all the data structures come from Xproto.h
 */
void xp_noop(char *buf, XINFO *xip)
{
  U8BITS type;
  int i;
  MC *mcp;

  type = BUF8(0);
  switch (type) {
  /*
   * requests without a reply
   */
  case X_ChangeWindowAttributes:
  case X_ChangeProperty:
  case X_SendEvent:
  case X_GrabButton:
  case X_GrabKey:
  case X_GrabServer:
  case X_ChangeKeyboardMapping:
  case X_ChangeHosts:
  case X_SetAccessControl:
    xip->fake_request = create_noop_request(xip);
    break;
  /*
   * requests with a reply
   */
  case X_GrabPointer:
  case X_GrabKeyboard:
    xip->fake_request = create_gan_request(xip);
    /* xGrabPointerReply and xGrabKeyboardReply:
       BYTE type;
       BYTE status;
       CARD16 sequenceNumber B16;
       CARD32 length B32;
       */
    mcp = mc_new(32);
    putCARD8(mcp->base, 0, X_Reply);
    putCARD8(mcp->base, 1, 0);	/* status = success */
    putCARD16(xip->sex, &mcp->base[2], xip->seq);
    putCARD32(xip->sex, &mcp->base[4], 0);
    queue_fake_packet(xip, mcp);
    break;
  case X_QueryKeymap:
    xip->fake_request = create_gan_request(xip);
    /* xQueryKeymapReply:
       BYTE type;
       BYTE pad1;
       CARD16 sequenceNumber B16;
       CARD32 length B32;
       BYTE map[32];
       */
    mcp = mc_new(40);
    putCARD8(mcp->base, 0, X_Reply);
    putCARD8(mcp->base, 1, 0);
    putCARD16(xip->sex, &mcp->base[2], xip->seq);
    putCARD32(xip->sex, &mcp->base[4], 2);
    for (i=8; i<40; i++) mcp->base[i] = 0; /* null keymap */
    queue_fake_packet(xip, mcp);
    break;
  case X_ListHosts:
    xip->fake_request = create_gan_request(xip);
    /* xListHostsReply:
       BYTE type;
       BOOL enabled;
       CARD16 sequenceNumber B16;
       CARD32 length B32;
       CARD16 nHosts B16;
       */
    mcp = mc_new(32);
    putCARD8(mcp->base, 0, X_Reply);
    putCARD8(mcp->base, 1, 1); /* access = enabled */
    putCARD16(xip->sex, &mcp->base[2], xip->seq);
    putCARD32(xip->sex, &mcp->base[4], 0);
    putCARD16(xip->sex, &mcp->base[8], 0);
    queue_fake_packet(xip, mcp);
    break;
  default:
    die("bad request type in xp_noop: %d", type);
  }
  /* better be safe than sorry */
  if (!xip->fake_request)
    die("xp_noop failed to create a fake request for %d", type);
}

/*
 * check the X request
 * all the data structures come from Xproto.h
 */
void xp_check(char *buf, XINFO *xip)
{
  U8BITS type, etype;
  U32BITS window, mask, prop, values[MAX_CWA_ARGS];
  int pos, i, len;
  char *atom;

  xip->alert = NULL;
  type = BUF8(0);
  switch (type) {
  /****************************************************************************/
  case X_ChangeWindowAttributes:
    /* xChangeWindowAttributesReq:
       CARD8 reqType;
       BYTE pad;
       CARD16 length B16;
       Window window B32;
       CARD32 valueMask B32; 
       */
    window = BUF32(4);
    /* ignore if it's on a window belonging to the connection */
    if (OWNING(window)) break;
    mask = BUF32(8);
    /* decode the packet */
    len = BUF16(2) - 3; /* last argument in the request */
    pos = 12;
    for (i=0; i<MAX_CWA_ARGS && i<len; i++) {
      if (BITTST(mask, 1 << i)) {
	values[i] = BUF32(pos);
	pos += 4;
      } else {
	values[i] = 0;
      }
    }
#ifndef FORBID_ICC
    if (BITTST(mask, CWEventMask)) {
      /* change event mask */
      if (((values[CWEventIndex] & ~PropertyChangeMask) == 0) &&
	  (window == xip->sel_req)) {
	/* only change (at most) PropertyChange on the window requesting the
	   selection, this is used by ICC so we allow it... */
	BITCLR(mask, CWEventMask);
      }
    }
#endif /* FORBID_ICC */
    /* no suspicious attributes changed */
    if (mask == 0) break;
#ifdef DEBUG
    if (DEBUGGING(DBG_ALERT)) {
      dprintf("alrt", "ChangeWindowAttributes window=" PRIx32BITS " mask=" PRIx32BITS,
	      window, mask);
      for (i=0; i<MAX_CWA_ARGS && i<len; i++) {
	if (BITTST(mask, 1 << i)) {
	  dprintf("    ", " argument %d (%s) is " PRIx32BITS,
		  i, xn_wattr_name(i), values[i]);
	}
      }
    }
#endif
    xip->alert = "Change the window attributes of a foreign window";
    xip->alert_level = AL_SEVERE;
    break;
  /****************************************************************************/
  case X_ChangeProperty:
    /* xChangePropertyReq:
       CARD8 reqType;
       CARD8 mode;
       CARD16 length B16;
       Window window B32;
       Atom property B32, type B32;
       CARD8 format;
       BYTE pad[3];
       CARD32 nUnits B32;
       */
    window = BUF32(4);
    /* ignore if it's on a window belonging to the connection */
    if (OWNING(window)) break;
    prop = BUF32(8);
    atom = xu_atom_name((Atom)prop);
#ifndef FORBID_ICC
    /* ignore if it's on the window requesting the selection and
       the property is the one expected, see ICCCM */
    if ((window == xip->sel_req) && (prop == xip->sel_prop)) break;
    /* cut and paste modify CUT_BUFFER? of the root window... */
    if ((window == (U32BITS)xu_root_window) &&
	(prop   >= (U32BITS)XA_CUT_BUFFER0) &&
	(prop   <= (U32BITS)XA_CUT_BUFFER7)) break;
#endif /* FORBID_ICC */
    /* Motif modifies properties on the Motif window... but why? */;
    if ((window == (U32BITS)xu_motif_window) && !strncmp(atom, "_MOTIF_", 7)) break;
#ifdef DEBUG
    if (DEBUGGING(DBG_ALERT)) {
      U8BITS mode = BUF8(1);
      char *cp;
      switch (mode) {
      case 0:
	cp = "Replace";
	break;
      case 1:
	cp = "Prepend";
	break;
      case 2:
	cp = "Append";
	break;
      default:
	cp = "???";
	break;
      }
      dprintf("alrt", "ChangeProperty mode=%s[%d] window=" PRIx32BITS " prop=%s[%d]",
	      cp, mode, window, atom, prop);
    }
#endif
    xip->alert = "Change a property of a foreign window";
    xip->alert_level = AL_DUBIOUS;
    break;
  /****************************************************************************/
  case X_SendEvent:
    /* xSendEventReq:
       CARD8 reqType;
       BOOL propagate;
       CARD16 length B16;
       Window destination B32;
       CARD32 eventMask B32;
       xEvent event;
       */
    window = BUF32(4);
    mask = BUF32(8);
    etype = BUF8(12) & 0x7F;
    /* some programs (e.g. netscape) send UnmapNotify events to the root window
       and even Expose events to random windows! */
    if ((window == (U32BITS)xu_root_window) && (etype == UnmapNotify)) break;
    if (etype == Expose) break;
    /* some programs (e.g. ghostview) send fake ClientMessage events */
    if (etype == ClientMessage) break;
#ifndef FORBID_ICC
    /* cut and paste send fake SelectionNotify events, see ICCCM */
    if ((etype == SelectionNotify) && (window == xip->sel_req)) break;
#endif /* FORBID_ICC */
#ifdef DEBUG
    if (DEBUGGING(DBG_ALERT))
      dprintf("alrt", "SendEvent window=" PRIx32BITS " mask=" PRIx32BITS " event=%s[%d]",
	      window, mask, xn_event_name(etype), etype);
#endif
    xip->alert = "Send a fake event";
    xip->alert_level = AL_DUBIOUS;
    break;
  /****************************************************************************/
  case X_GrabPointer:
    /* xGrabPointerReq:
       CARD8 reqType;
       BOOL ownerEvents;
       CARD16 length B16;
       Window grabWindow B32;
       CARD16 eventMask B16;
       BYTE pointerMode, keyboardMode;
       Window confineTo B32;
       Cursor cursor B32;
       Time time B32;
       */
#ifdef DEBUG
    if (DEBUGGING(DBG_ALERT))
      dprintf("alrt", "GrabPointer pmode=%s kmode=%s",
	      (buf[10] ? "Async" : "Sync"),
	      (buf[11] ? "Async" : "Sync"));
#endif
    xip->alert = "Grab the pointer";
    xip->alert_level = AL_INTERNAL;
    break;
  /****************************************************************************/
#ifdef FORBID_ALL_GRABS
  case X_GrabButton:
    /* xGrabButtonReq:
       CARD8 reqType;
       BOOL ownerEvents;
       CARD16 length B16;
       Window grabWindow B32;
       CARD16 eventMask B16;
       BYTE pointerMode, keyboardMode;
       Window confineTo B32;
       Cursor cursor B32;
       CARD8 button;
       BYTE pad;
       CARD16 modifiers B16;
       */
#ifdef DEBUG
    if (DEBUGGING(DBG_ALERT))
      dprintf("alrt", "GrabButton pmode=%s kmode=%s",
	      (buf[10] ? "Async" : "Sync"),
	      (buf[11] ? "Async" : "Sync"));
#endif
    xip->alert = "Grab the mouse button";
    xip->alert_level = AL_INTERNAL;
    break;
#endif /* FORBID_ALL_GRABS */
  /****************************************************************************/
  case X_GrabKeyboard:
    /* xGrabKeyboardReq:
       CARD8 reqType;
       BOOL ownerEvents;
       CARD16 length B16;
       Window grabWindow B32;
       Time time B32;
       BYTE pointerMode, keyboardMode;  
       CARD16 pad B16;
       */
#ifdef DEBUG
    if (DEBUGGING(DBG_ALERT))
      dprintf("alrt", "GrabKeyboard pmode=%s kmode=%s",
	      (buf[12] ? "Async" : "Sync"),
	      (buf[13] ? "Async" : "Sync"));
#endif
    xip->alert = "Grab the keyboard";
    xip->alert_level = AL_INTERNAL;
    break;
  /****************************************************************************/
#ifdef FORBID_ALL_GRABS
  case X_GrabKey:
    /* xGrabKeyReq:
       CARD8 reqType;
       BOOL ownerEvents;
       CARD16 length B16;
       Window grabWindow B32;
       CARD16 modifiers B16;
       CARD8 key;
       BYTE pointerMode, keyboardMode;  
       BYTE pad1, pad2, pad3;
       */
#ifdef DEBUG
    if (DEBUGGING(DBG_ALERT))
      dprintf("alrt", "GrabKey keycode=%d pmode=%s kmode=%s", BUF8(10),
	      (buf[11] ? "Async" : "Sync"),
	      (buf[12] ? "Async" : "Sync"));
#endif
    xip->alert = "Grab a key";
    xip->alert_level = AL_INTERNAL;
    break;
#endif /* FORBID_ALL_GRABS */
  /****************************************************************************/
  case X_GrabServer:
#ifdef DEBUG
    if (DEBUGGING(DBG_ALERT))
      dprintf("alrt", "GrabServer");
#endif
    xip->alert = "Grab the server";
    xip->alert_level = AL_INTERNAL;
    break;
  /****************************************************************************/
  case X_QueryKeymap:
#ifdef DEBUG
    if (DEBUGGING(DBG_ALERT))
      dprintf("alrt", "QueryKeymap");
#endif
    xip->alert = "Query the current state of the keyboard";
    xip->alert_level = AL_DUBIOUS;
    break;
  /****************************************************************************/
  case X_ChangeKeyboardMapping:
    /* xChangeKeyboardMappingReq:
       CARD8 reqType;
       CARD8 keyCodes;
       CARD16 length B16;
       KeyCode firstKeyCode;
       CARD8 keySymsPerKeyCode;
       CARD16 pad1 B16;
       */
#ifdef DEBUG
    if (DEBUGGING(DBG_ALERT))
      dprintf("alrt", "ChangeKeyboardMapping length=%d", BUF8(1));
#endif
    xip->alert = "Change the keyboard mapping";
    xip->alert_level = AL_DUBIOUS;
    break;
  /****************************************************************************/
  case X_ChangeHosts:
    /* xChangeHostsReq:
       CARD8 reqType;
       BYTE mode;
       CARD16 length B16;
       CARD8 hostFamily;
       BYTE pad;
       CARD16 hostLength B16;
       */
#ifdef DEBUG
    if (DEBUGGING(DBG_ALERT)) {
      U8BITS family = BUF8(4);
      char *cp;
      switch (family) {
      case 0:
	cp = "Internet";
	break;
      case 1:
	cp = "DECnet";
	break;
      case 2:
	cp = "Chaos";
	break;
      default:
	cp = "???";
	break;
      }
      dprintf("alrt", "ChangeHosts mode=%s family=%s[%d] address=%s",
	      (buf[1] ? "Delete" : "Insert"), cp, family,
	      ((family == 0) ? host_name(BUF32(8)) : "???"));
    }
#endif
    xip->alert = "Change the access control list";
    xip->alert_level = AL_SEVERE;
    break;
  /****************************************************************************/
  case X_ListHosts:
#ifdef DEBUG
    if (DEBUGGING(DBG_ALERT))
      dprintf("alrt", "ListHosts");
#endif
    xip->alert = "Read the access control list";
    xip->alert_level = AL_SEVERE;
    break;
  /****************************************************************************/
  case X_SetAccessControl:
    /* xSetAccessControlReq:
       CARD8 reqType;
       BYTE mode;
       CARD16 length B16;
       */
#ifdef DEBUG
    if (DEBUGGING(DBG_ALERT))
      dprintf("alrt", "SetAccessControl mode=%s", (buf[1] ? "Enable" : "Disable"));
#endif
    xip->alert = "Disable the access control";
    xip->alert_level = AL_SEVERE;
    break;
  }

  /* update the alert counter */
  if (xip->alert && (xip->alert_level != AL_INTERNAL)) xip->alert_count++;
}

/*
 * trace X server packets to follow ICC events and to trace reply types
 * (see also xp_print for more info on packet decoding)
 */
void xp_trace(char type, char *buf, char *tag, XINFO *xip)
{
  U8BITS n;
  U16BITS s;
  char *rn;
  int i, count, last;
#ifndef DEBUG
  tag = tag; /* unused parameter */
#endif

  switch (type) {
  case XP_REQUEST:
    /*
     * we track requests requiring a reply
     */
    /* xReq:
       CARD8 reqType;
       */
    n = BUF8(0);
    rn = xn_request_name(n);
    if (*rn == '*' || *rn == '+') {
      /*
       * this request probably needs a reply
       */
      if (xip->rep_count < XP_REPMAX) {
	/* we can store it */
	last = (xip->rep_first + xip->rep_count) & (XP_REPMAX - 1);
	xip->rep_seq[last] = xip->seq;
	xip->rep_type[last] = n;
	xip->rep_count++;
#ifdef DEBUG
      } else {
	/* not enough space to store it */
	if (DEBUGGING(DBG_X) && DEBUGGING(DBG_WARNING))
	  dprintf("warn", "can't store request %s[%d] seq=%d",
		  xn_request_name(n), n, xip->seq);
#endif
      }
    }
    break;
  case XP_PACKET:
    /* xGenericReply:
       BYTE type;
       BYTE data1;
       CARD16 sequenceNumber B16;
       CARD32 length B32;
       */
    n = BUF8(0);
    s = BUF16(2);
    if (n == X_Error) {
      /*
       * easy: nothing to do...
       */
    } else if (n == X_Reply) {
      /*
       * we track replies
       */
      if (xip->rep_count && xip->rep_seq[xip->rep_first] == s) {
	/*
	 * easy, the reply is the first one expected
	 */
	xip->rep_first++;
	if (xip->rep_first == XP_REPMAX) xip->rep_first = 0;
	xip->rep_count--;
      } else {
	/*
	 * check if we indeed expect this reply
	 */
	last = -1;
	for (i=xip->rep_first,count=xip->rep_count; count>0; i++,count--) {
	  if (i == XP_REPMAX) i = 0; /* wrap */
	  if (xip->rep_seq[i] == s) {
	    last = i;
	    break;
	  }
	}
	if (last < 0) {
	  /* really unexpected reply */
#ifdef DEBUG
	  if (DEBUGGING(DBG_X) && DEBUGGING(DBG_WARNING))
	    dprintf("warn", "unexpected reply seq=%d", xip->seq);
#endif
	} else {
	  /* expected reply but out of sync */
	  for (i=xip->rep_first,count=xip->rep_count; count>0; i++,count--) {
	    if (i == XP_REPMAX) i = 0; /* wrap */
	    if (i == last) break;
	    xip->rep_count--;
#ifdef DEBUG
	    n = xip->rep_type[i];
	    if (DEBUGGING(DBG_X) && DEBUGGING(DBG_WARNING))
	      dprintf("warn", "lost reply %s[%d] seq=%d",
		      xn_request_name(n), n, xip->rep_seq[i]);
#endif
	  }
	  /* resync */
	  xip->rep_first = last;
	  /* discard the expected reply */
	  xip->rep_first++;
	  if (xip->rep_first == XP_REPMAX) xip->rep_first = 0;
	  xip->rep_count--;
	}
       }
    } else { /* event */
#ifndef FORBID_ICC
      /*
       * we track selection request for ICCCM handling
       */
      n &= 0x7F; /* clear high bit */
      switch (n) {
      case SelectionRequest:
	/* selectionRequest:
	   CARD32 pad00 B32;
	   Time time B32;    
	   Window owner B32, requestor B32;
	   Atom selection B32, target B32, property B32;
	   */
	xip->sel_req  = BUF32(12);
	xip->sel_prop = BUF32(24);
#ifdef DEBUG
	if (DEBUGGING(DBG_X)) {
	  dprintf(tag, "trace SelectionRequest req=" PRIx32BITS " prop=%s[%d]",
		  xip->sel_req, xu_atom_name((Atom)xip->sel_prop), xip->sel_prop);
	}
#endif /* DEBUG */
	break;
      case SelectionClear:
	xip->sel_req = 0;
	xip->sel_prop = 0;
	break;
      }
#endif /* FORBID_ICC */
    }
    break;
  default:
    die("bad type in xp_trace: %c", type);
  }
}

#ifdef DEBUG
/*
 * print the given X data (packet, event, reply...)
 * all the data structures come from Xproto.h
 */
void xp_print(char type, char *buf, char *tag, XINFO *xip)
{
  U8BITS t;
  U16BITS n, p, s;
  char *cp;

  switch (type) {
  /****************************************************************************/
  case XP_CSETUP:
    /* xConnClientPrefix:
       CARD8	byteOrder;
       BYTE	pad;
       CARD16	majorVersion B16, minorVersion B16;
       CARD16	nbytesAuthProto B16;
       CARD16	nbytesAuthString B16;
       CARD16	pad2 B16;
       */
    n = BUF16(6);
    p = BUF16(8);
    if (n == 0) {
      cp = "none";
    } else {
      cp = str_ncopy((char *)&buf[12], n);
    }
    dprintf(tag, "setup sex=%c proto=%d.%d auth=%s authlen=%d",
	    xip->sex, BUF16(2), BUF16(4), cp, p);
    if (n != 0) str_free(cp);
    break;
  /****************************************************************************/
  case XP_REQUEST:
    /* xReq:
       CARD8 reqType;
       CARD8 data;
       CARD16 length B16;
       */
    t = BUF8(0);
    p = BUF16(2);
    dprintf(tag, "request %s[%d] seq=%d length=%d", xn_request_name(t), t, xip->seq, 4*p);
    break;
  /****************************************************************************/
  case XP_SSETUP:
    /* xConnSetupPrefix:
       BOOL           success;
       BYTE           lengthReason;
       CARD16         majorVersion B16, minorVersion B16;
       CARD16         length B16;
       */
    if (buf[0] == 0) {
      /* failure */
      t = BUF8(1);
      if (t == 0) {
	cp = "none";
      } else {
	cp = str_ncopy((char *)&buf[8], t);
      }
      dprintf(tag, "failed setup proto=%d.%d reason=%s", BUF16(2), BUF16(4), cp);
      if (t != 0) str_free(cp);
    } else {
      /* success */
      /* xConnSetup:
	 CARD32         release B32;
	 CARD32         ridBase B32, ridMask B32;
	 CARD32         motionBufferSize B32;
	 CARD16         nbytesVendor B16;
	 CARD16         maxRequestSize B16;
	 CARD8          numRoots;
	 CARD8          numFormats;
	 CARD8          imageByteOrder;
	 CARD8          bitmapBitOrder;
	 CARD8          bitmapScanlineUnit, bitmapScanlinePad;
	 KeyCode	minKeyCode, maxKeyCode;
	 CARD32	        pad2 B32;
	 */
      n = BUF16(24);
      if (n == 0) {
	cp = "none";
      } else {
	cp = str_ncopy((char *)&buf[40], n);
      }
      dprintf(tag, "setup proto=%d.%d rel=%d rid=%08X/%08X vendor=%s",
	      BUF16(2), BUF16(4), BUF32(8), BUF32(12), BUF32(16), cp);
      if (n != 0) str_free(cp);
    }
    break;
  /****************************************************************************/
  case XP_PACKET:
    /* xGenericReply:
       BYTE type;
       BYTE data1;
       CARD16 sequenceNumber B16;
       CARD32 length B32;
       */
    t = BUF8(0);
    s = BUF16(2);
    if (t == X_Error) {
      /* xError:
	 BYTE type;
	 BYTE errorCode;
	 CARD16 sequenceNumber B16;
	 CARD32 resourceID B32;
	 CARD16 minorCode B16;
	 CARD8 majorCode;
	 */
      t = BUF8(1);
      dprintf(tag, "error %s[%d] seq=%d id=%08X code=%d/%d", xn_error_name(t),
	      t, s, BUF32(4), BUF8(10), BUF16(8));
    } else if (t == X_Reply) {
      /* xReply */
      if (xip->rep_count && xip->rep_seq[xip->rep_first] == s) {
	/* expected reply */
	t = xip->rep_type[xip->rep_first];
	dprintf(tag, "reply %s[%d] seq=%d length=%d", xn_request_name(t), t,
		s, 32 + 4*BUF32(4));
      } else {
	/* unexpected reply (or we didn't take care to trace requests) */
	dprintf(tag, "reply seq=%d length=%d", s, 32 + 4*BUF32(4));
      }
    } else {
      /* xEvent:
	 BYTE type;
	 BYTE detail;
	 CARD16 sequenceNumber B16;
	 */
      t &= 0x7F; /* clear high bit */
      dprintf(tag, "event %s[%d] synth=%s detail=%d seq=%d", xn_event_name(t), t,
	      ((buf[0] & 0x80) ? "yes" : "no"), BUF8(1), s);
    }
    break;
  default:
    die("bad type in xp_print: %c", type);
  }
}
#endif
