/*+*****************************************************************************
*                                                                              *
* File: popup.c                                                                *
*                                                                              *
* Description: handle the popup menu                                           *
*                                                                              *
**-****************************************************************************/

#ifndef __lint
char popup_vers[] = "@(#)popup.c	1.9	02/19/99	Written by Lionel Cons";
#endif /* __lint */

/******* Include Files ********************************************************/

#include <time.h>
#include <X11/Xlib.h>
#include <Xm/LabelG.h>
#include <Xm/PushBG.h>
#include <Xm/SeparatoG.h>
#include <Xm/ToggleBG.h>
#include <Xm/RowColumn.h>

#include "popup.h"
#include "conn.h"
#include "input.h"
#include "xu.h"
#include "xp.h"
#include "sb.h"

/******* Constants ************************************************************/

/******* Macros ***************************************************************/

/******* External Stuff *******************************************************/

/******* Local Stuff **********************************************************/

static void (*popup_kill_connection)(CONN *);

/*
 * callback to change the frozen flag
 */
static void popup_frozen_cb(Widget w, CONN *c, XmToggleButtonCallbackStruct *cbs)
{
  w = w; cbs = cbs; /* unused parameters */

  BITCHG(c->flags, CF_FROZEN);
  /* update the input */
  input_update_connection(c);
}

/*
 * callback to change the checked flag
 */
static void popup_checked_cb(Widget w, CONN *c, XmToggleButtonCallbackStruct *cbs)
{
  w = w; cbs = cbs; /* unused parameters */

  BITCHG(c->flags, CF_CHECKED);
}

/*
 * callback to change the safe flag
 */
static void popup_safe_cb(Widget w, CONN *c, XmToggleButtonCallbackStruct *cbs)
{
  w = w; cbs = cbs; /* unused parameters */

  BITCHG(c->flags, CF_SAFE);
}

/*
 * dismiss the info box
 */
static void popup_info_dismiss(Widget w, CONN *c)
{
  xu_dismiss(w, NULL);
  c->ui.info = (Widget)NULL;
}

/*
 * put the text to display in a static buffer
 */
static char *popup_info_text(CONN *c)
{
  int n, i;

  /* init */
  sb_discard();

  /* basic info */
  sb_add_strings("Connection ", c->id, " from ", c->name,
		 " [", c->host->ip_name, "]\n", 0);
  sb_add_strings("Started on: ", ctime(&c->start), 0);
  sb_add_strings("Idle since: ", ctime(&c->idle_since), 0);
  sprintf(sb_line, PRId32BITS " X packets (" PRId32BITS " KB) with %d alert%s\n",
	  c->xinfo.pkt_count, (c->xinfo.pkt_size >> 10),
	  c->xinfo.alert_count, (c->xinfo.alert_count > 1 ? "s" : ""));
  sb_add_string(sb_line);
  sprintf(sb_line, "Resource id base " PRIx32BITS ", mask " PRIx32BITS "\n",
	  c->xinfo.resid_base, c->xinfo.resid_mask);
  sb_add_string(sb_line);

  /* windows */
  n = xu_scan(c->xinfo.resid_base, c->xinfo.resid_mask);
  if (n == 0) {
    sb_add_string("No window displayed\n");
  } else {
    sb_add_strings("Window", (n > 1 ? "s" : ""), ":\n", 0);
    for (i=0; i<n; i++) {
      sprintf(sb_line, "   " PRIx32BITS " ", (U32BITS)xu_window_info[i].id);
      sb_add_strings(sb_line, xu_window_info[i].name,
		     " (", xu_window_info[i].geom, ")\n", 0);
    }
  }
  
  return(sb_buffer);
}

/*
 * callback to show the info box
 */
static void popup_info_cb(Widget w, CONN *c, XmPushButtonCallbackStruct *cbs)
{
  ANSWER ok;
  char *text;
  w = w; cbs = cbs; /* unused parameters */

  /* init */
  text = popup_info_text(c);

  /* already displayed? */
  if (c->ui.info) {
    xu_text(c->ui.info, text);
    xu_raise(c->ui.info);
    return;
  }

  /* display it */
  ok.text = "Ok";
  ok.callback = (XtCallbackProc)popup_info_dismiss;
  ok.data = (XtPointer)c;
  c->ui.info = xu_info(text, &ok);
}

/*
 * callback to confirm a connection
 */
static void popup_kill_cb(Widget w, CONN *c, XmPushButtonCallbackStruct *cbs)
{
  w = w; cbs = cbs; /* unused parameters */

  popup_kill_connection(c);
}

/*
 * event handler for the popup menu
 */
void popup_handler(Widget w, CONN *c, XButtonPressedEvent *event)
{
  /* create menu if needed */
  if (c->ui.menu == NULL) {
    Arg args[2];
    Widget item;

    c->ui.menu = XmCreatePopupMenu(w, "Popup Menu", args, 0);
    item = XmCreateLabelGadget(c->ui.menu, c->id, args, 0);
    XtManageChild(item);
    XtSetArg(args[0], XmNseparatorType, XmDOUBLE_LINE);
    item = XmCreateSeparatorGadget(c->ui.menu, "sep 1", args, 1);
    XtManageChild(item);
    item = XmCreateToggleButtonGadget(c->ui.menu, "Frozen", args, 0);
    XtManageChild(item);
    XtAddCallback(item, XmNvalueChangedCallback,
		(XtCallbackProc)popup_frozen_cb, (XtPointer)c);
    c->ui.menu_frozen = item;
    item = XmCreateToggleButtonGadget(c->ui.menu, "Checked", args, 0);
    XtManageChild(item);
    XtAddCallback(item, XmNvalueChangedCallback,
		(XtCallbackProc)popup_checked_cb, (XtPointer)c);
    c->ui.menu_checked = item;
    item = XmCreateToggleButtonGadget(c->ui.menu, "Safe", args, 0);
    XtManageChild(item);
    XtAddCallback(item, XmNvalueChangedCallback,
		(XtCallbackProc)popup_safe_cb, (XtPointer)c);
    c->ui.menu_safe = item;
    XtSetArg(args[0], XmNseparatorType, XmSINGLE_LINE);
    item = XmCreateSeparatorGadget(c->ui.menu, "sep 2", args, 1);
    XtManageChild(item);
    item = XmCreatePushButtonGadget(c->ui.menu, "Info...", args, 0);
    XtManageChild(item);
    XtAddCallback(item, XmNactivateCallback,
		(XtCallbackProc)popup_info_cb, (XtPointer)c);
    item = XmCreatePushButtonGadget(c->ui.menu, "Kill...", args, 0);
    XtManageChild(item);
    XtAddCallback(item, XmNactivateCallback,
		(XtCallbackProc)popup_kill_cb, (XtPointer)c);
  }

  /* adjust menu state and display it */
  XtVaSetValues(c->ui.menu_frozen,
		XmNset, (BITTST(c->flags, CF_FROZEN) ? True : False),
		NULL);
  XtVaSetValues(c->ui.menu_checked,
		XmNset, (BITTST(c->flags, CF_CHECKED) ? True : False),
		NULL);
  XtVaSetValues(c->ui.menu_safe,
		XmNset, (BITTST(c->flags, CF_SAFE) ? True : False),
		NULL);
  XmMenuPosition(c->ui.menu, event);
  XtManageChild(c->ui.menu);
}

/*
 * setup
 */
void popup_setup(void (*kc)(CONN *))
{
  popup_kill_connection = kc;
}
