/*+*****************************************************************************
*                                                                              *
* File: opt.c                                                                  *
*                                                                              *
* Description: option handling                                                 *
*                                                                              *
**-****************************************************************************/

#ifndef __lint
char opt_vers[] = "@(#)opt.c	1.15	07/22/99	Written by Lionel Cons";
#endif /* __lint */

/******* Include Files ********************************************************/

#include "opt.h"
#include "util.h"
#include "conn.h"
#include "config.h"
#include "xs.h"
#include "vers.h"

#include <ctype.h>
#include <X11/StringDefs.h>

/******* Constants ************************************************************/

#define MXCONNS_CLASS	"Mxconns"      /* mxconns' application class */

/******* Macros ***************************************************************/

#define YORN(x)		((x) ? "yes" : "no")

/******* External Stuff *******************************************************/

MXCONNS_OPTS opt;

/******* Local Stuff **********************************************************/

/*
 * X resources with their default values
 */
static XtResource resources[] = {
  { "verbose", "Verbose", XtRBoolean, sizeof(Boolean),
      XtOffsetOf(MXCONNS_OPTS, verbose), XtRImmediate, False },
  { "hunt", "Hunt", XtRBoolean, sizeof(Boolean),
      XtOffsetOf(MXCONNS_OPTS, hunt), XtRImmediate, False },
  { "fork", "Fork", XtRBoolean, sizeof(Boolean),
      XtOffsetOf(MXCONNS_OPTS, fork), XtRImmediate, False },
  { "ident", "Ident", XtRBoolean, sizeof(Boolean),
      XtOffsetOf(MXCONNS_OPTS, ident), XtRImmediate, False },
  { "dispno", "Dispno", XtRInt, sizeof(int),
      XtOffsetOf(MXCONNS_OPTS, dispno), XtRImmediate, (XtPointer)5 },
#ifdef DEBUG
  { "debug", "Debug", XtRString, sizeof(String),
      XtOffsetOf(MXCONNS_OPTS, debug_s), XtRImmediate, "" },
#endif
  { "icf", "Icf", XtRString, sizeof(String),
      XtOffsetOf(MXCONNS_OPTS, icf_s), XtRImmediate, "" },
  { "iname", "Iname", XtRString, sizeof(String),
      XtOffsetOf(MXCONNS_OPTS, iname), XtRImmediate, "" },
  { "config", "Config", XtRString, sizeof(String),
      XtOffsetOf(MXCONNS_OPTS, config), XtRImmediate, "" },
  { "xsock", "Xsock", XtRString, sizeof(String),
    XtOffsetOf(MXCONNS_OPTS, xsock), XtRImmediate, DEFAULT_XSOCK },
  { "xflags", "Xflags", XtRString, sizeof(String),
    XtOffsetOf(MXCONNS_OPTS, xflags_s), XtRImmediate, "TU" },
};

/*
 * command line options
 */
XrmOptionDescRec options[] = {
  { "-verbose", "*verbose", XrmoptionNoArg, "True" },
  { "-hunt",    "*hunt",    XrmoptionNoArg, "True" },
  { "-fork",    "*fork",    XrmoptionNoArg, "True" },
  { "-ident",   "*ident",   XrmoptionNoArg, "True" },
  { "-dispno",  "*dispno",  XrmoptionSepArg, NULL  },
#ifdef DEBUG
  { "-debug",   "*debug",   XrmoptionSepArg, NULL  },
#endif
  { "-icf",     "*icf",     XrmoptionSepArg, NULL  },
  { "-iname",   "*iname",   XrmoptionSepArg, NULL  },
  { "-config",  "*config",  XrmoptionSepArg, NULL  },
  { "-xsock",   "*xsock",   XrmoptionSepArg, NULL  },
  { "-xflags",  "*xflags",  XrmoptionSepArg, NULL  },
};

#ifdef DEBUG
/*
 * parse the given debug string
 */
static void parse_debug(void)
{
  opt.debug = 0;
  if (!opt.debug_s[0]) return;
  if (isalpha((int)opt.debug_s[0])) {
    /* assume a string */
    char *cp;
    for (cp=opt.debug_s; *cp; cp++) {
      switch (*cp) {
      case 'g': BITCHG(opt.debug, DBG_WARNING);  break;
      case 'e': BITCHG(opt.debug, DBG_ERROR);    break;
      case 's': BITCHG(opt.debug, DBG_MISC);     break;
      case 'c': BITCHG(opt.debug, DBG_CONN);     break;
      case 'u': BITCHG(opt.debug, DBG_UI);       break;
      case 'x': BITCHG(opt.debug, DBG_X);        break;
      case 'f': BITCHG(opt.debug, DBG_FSA);      break;
      case 'i': BITCHG(opt.debug, DBG_IO);       break;
      case 'r': BITCHG(opt.debug, DBG_IO_READ);  break;
      case 'w': BITCHG(opt.debug, DBG_IO_WRITE); break;
      case 't': BITCHG(opt.debug, DBG_STATS);    break;
      case 'l': BITCHG(opt.debug, DBG_LOG);      break;
      case 'm': BITCHG(opt.debug, DBG_MC);       break;
      case 'a': BITCHG(opt.debug, DBG_ALERT);    break;
      case 'd': BITCHG(opt.debug, DBG_DISPATCH); break;
      case 'o': BITCHG(opt.debug, DBG_CONFIG);   break;
      case 'n': BITCHG(opt.debug, DBG_IDENT);    break;
      case 'F': /* few */
	BITSET(opt.debug, DBG_WARNING|DBG_ERROR|DBG_MISC|DBG_CONN|DBG_ALERT|DBG_CONFIG);
	break;
      case 'M': /* many */
	BITSET(opt.debug, DBG_ALL);
	BITCHG(opt.debug, DBG_IO_READ|DBG_IO_WRITE|DBG_LOG);
	break;
      case 'A': /* all */
	BITSET(opt.debug, DBG_ALL);
	break;
      default:
	die("bad debug character in '%s': %c", opt.debug_s, *cp);
	break;
      }
    }
  } else {
    /* assume a number */
    opt.debug = atoi(opt.debug_s);
  }
}
#endif /* DEBUG */

/*
 * parse the given initial connection flags string
 */
static void parse_icf(void)
{
  opt.icf = 0;
  if (!opt.icf_s[0]) return;
  if (isalpha((int)opt.icf_s[0])) {
    /* assume a string */
    char *cp;
    for (cp=opt.icf_s; *cp; cp++) {
      switch (*cp) {
      case 'f': BITSET(opt.icf, CF_FROZEN);    break;
      case 'c': BITSET(opt.icf, CF_CHECKED);   break;
      case 's': BITSET(opt.icf, CF_SAFE);      break;
      default:
	die("bad connection flag character in '%s': %c", opt.icf_s, *cp);
	break;
      }
    }
  } else {
    /* assume a number */
    opt.icf = atoi(opt.icf_s);
  }
}

/*
 * parse the given X socket flags string
 */
static void parse_xflags(void)
{
  opt.xflags = 0;
  if (!opt.xflags_s[0]) return;
  if (isalpha((int)opt.xflags_s[0])) {
    /* assume a string */
    char *cp;
    for (cp=opt.xflags_s; *cp; cp++) {
      switch (*cp) {
      case 'h': BITSET(opt.xflags, XS_HUNT);       break;
      case 't': BITSET(opt.xflags, XS_WANT_TCP);   break;
      case 'T': BITSET(opt.xflags, XS_NEED_TCP);   break;
      case 'u': BITSET(opt.xflags, XS_WANT_UNIX);  break;
      case 'U': BITSET(opt.xflags, XS_NEED_UNIX);  break;
      default:
	die("bad X socket flag character in '%s': %c", opt.xflags_s, *cp);
	break;
      }
    }
  } else {
    /* assume a number */
    opt.xflags = atoi(opt.xflags_s);
  }
}

/*
 * print usage information
 */
static void usage(char *progname)
{

  fprintf(stderr, "This is %s\n", &mxconns_vers[4]);
  fprintf(stderr, "Usage: %s [options]\n", progname);
  fprintf(stderr, "   -verbose     : print the display created on stdout (%s)\n",
	  YORN(opt.verbose));
  fprintf(stderr, "   -hunt        : hunt for a display (%s)\n",
	  YORN(opt.hunt));
  fprintf(stderr, "   -fork        : put itself into the background (%s)\n",
	  YORN(opt.fork));
  fprintf(stderr, "   -ident       : use the IDENT protocol (%s)\n",
	  YORN(opt.ident));
  fprintf(stderr, "   -dispno NUM  : use the display number NUM (%d)\n",
	  opt.dispno);
#ifdef DEBUG
  fprintf(stderr, "   -debug STR   : set the debugging flags to STR (%s)\n",
	  opt.debug_s);
#endif
  fprintf(stderr, "   -icf STR     : set the initial connection flags to STR (%s)\n",
	  opt.icf_s);
  fprintf(stderr, "   -iname STR   : set the icon name to STR (%s)\n",
	  opt.iname);
  fprintf(stderr, "   -config PATH : use PATH as configuration file (%s)\n",
	  opt.config);
  fprintf(stderr, "   -xsock PATH  : use PATH for the X sockets (%s)\n",
	  opt.xsock);
  fprintf(stderr, "   -xflags STR  : set the X socket flags to STR (%s)\n",
	  opt.xflags_s);
  exit(1);
}

/*
 * initialise X and parse the options, return the toplevel widget
 */
Widget opt_init(int argc, char *argv[])
{
  Widget top;
  XtAppContext ac;

  /* initialize X and parse the command line options */
  top = XtVaAppInitialize(&ac, MXCONNS_CLASS, options, XtNumber(options),
			  &argc, argv, NULL, NULL, 0);

  /* merge default X resources */
  XtGetApplicationResources(top, &opt, resources, XtNumber(resources), NULL, 0);

  /* extra options are buggy */
  if (argc > 1) usage(argv[0]);

  /* special treatments */
#ifdef DEBUG
  parse_debug();
#endif
  parse_icf();
  parse_xflags();

  /* -hunt versus -xflags conflicts */
  if (opt.hunt) BITSET(opt.xflags, XS_HUNT);

  /* that's all folks! */
  return(top);
}
