/*+*****************************************************************************
*                                                                              *
* File: list.c                                                                 *
*                                                                              *
* Description: scrolled list handling                                          *
*                                                                              *
**-****************************************************************************/

#ifndef __lint
char list_vers[] = "@(#)list.c	1.6	02/01/99	Written by Lionel Cons";
#endif /* __lint */

/******* Include Files ********************************************************/

#include "list.h"
#include "util.h"
#include "conn.h"

#include <Xm/List.h>

/******* Constants ************************************************************/

#define VISIBLE_ITEMS	7	/* number of visible items in the list */
#define ALL_MODS	(Mod1Mask|Mod2Mask|Mod3Mask|Mod4Mask|Mod5Mask)

/******* Macros ***************************************************************/

/******* External Stuff *******************************************************/

/******* Local Stuff **********************************************************/

static Widget list_widget;
static void (*list_activate)(CONN *);
static void (*list_popup)(Widget, CONN *, XButtonPressedEvent *);

/*
 * add a CONN to the list, at the end
 */
void list_add(CONN *c)
{
  char *name;
  XmString str;

  name = (char *)safe_malloc(2 + strlen(c->id) + strlen(c->name));
  sprintf(name, "%s:%s", c->id, c->name);
  str = XmStringCreateSimple(name);
  XmListAddItemUnselected(list_widget, str, 0);
  XmStringFree(str);
  str_free(name);
}

/*
 * remove a CONN from the list, which must be in sync with the connection list
 */
void list_remove(CONN *c)
{
  int pos;

  pos = conn_position(c);
  if (pos > 0) XmListDeletePos(list_widget, pos);
#ifdef DEBUG
  else if (DEBUGGING(DBG_UI))
    dprintf(c->tag, "connection not in list!");
#endif
}

/*
 * callback called when a list item is activated
 */
static void list_activate_cb(Widget w, XtPointer data, XmListCallbackStruct *cbs)
{
  CONN *c;
  w = w; data = data; /* unused parameters */

  c = conn_at_position(cbs->item_position);
  if (c == NULL) {
#ifdef DEBUG
    if (DEBUGGING(DBG_UI))
      dprintf("list", "unknown connection %d!", cbs->item_position);
#endif
    return;
  }
  (*list_activate)(c);
}

/*
 * event handler called when the mouse button is pressed on the list
 */
static void list_button_handler(Widget w, caddr_t data, XButtonPressedEvent *event)
{
  int count, *list;
  CONN *c;
  data = data; /* unused parameters */

  /* care only about mouse button 3 or mouse button 1 plus any modifier */
  if ((event->button != 3) &&
      (event->button != 1 || !BITTST(event->state, ALL_MODS))) return;

  /* check selection, should be only 1 */
  XmListGetSelectedPos(w, &list, &count);
  if (count != 1) return;

  /* find the connection */
  c = conn_at_position(list[0]);
  if (c == NULL) {
#ifdef DEBUG
    if (DEBUGGING(DBG_UI))
      dprintf("list", "unknown connection %d!", list[0]);
#endif
    return;
  }

  (*list_popup)(w, c, event);
}

/*
 * create the list widget and hook the given functions
 */
void list_setup(Widget toplevel, void (*activate)(CONN *),
		void (*popup)(Widget, CONN *, XButtonPressedEvent *))
{
  Arg args[2];
  XtCallbackRec cb[2];

  list_activate = activate;
  list_popup = popup;
  XtSetArg(args[0], XmNvisibleItemCount, VISIBLE_ITEMS);
  cb[0].callback = (XtCallbackProc)list_activate_cb;
  cb[0].closure  = (XtPointer)NULL;
  cb[1].callback = (XtCallbackProc)NULL;
  XtSetArg(args[1], XmNdefaultActionCallback, cb);
  list_widget = XmCreateScrolledList(toplevel, "Scrolled List", args, 2);
  XtAddEventHandler(list_widget, ButtonPressMask, False,
		    (XtEventHandler)list_button_handler, (XtPointer)NULL);
  XtManageChild(list_widget);
}
