/*+*****************************************************************************
*                                                                              *
* File: io.c                                                                   *
*                                                                              *
* Description: input / output handling                                         *
*                                                                              *
**-****************************************************************************/

#ifndef __lint
char io_vers[] = "@(#)io.c	1.10	02/18/99	Written by Lionel Cons";
#endif /* __lint */

/******* Include Files ********************************************************/

#include "io.h"
#include "util.h"
#include "mc.h"
#include "xmd.h"
#include "err.h"

#include <sys/uio.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#ifdef SOLARIS
#include <sys/filio.h> /* for FIONBIO */
#endif
#include <signal.h>

/******* Constants ************************************************************/

/*
 * byte sex used by X11, hard-coded in the X sources :-(
 */
#define SEX_MSB		((U8BITS)0x42)	/* ASCII B */
#define SEX_LSB		((U8BITS)0x6c)	/* ASCII l */

/*
 * try to figure out MAXIOV
 */
#ifndef MAXIOV
#define MAXIOV		16
#endif

/******* Macros ***************************************************************/

/*
 * get an unsigned char out of some number
 */
#define UCHAR(x)		((unsigned char)((x) & 0xFF))

/******* External Stuff *******************************************************/

/******* Local Stuff **********************************************************/

static struct iovec io_iovec[MAXIOV]; /* I/O vector used by io_write */

/*
 * put a CARD16
 */
void putCARD16(U8BITS sex, char *cp, U16BITS val)
{
  unsigned char *ucp = (unsigned char *)cp;

  switch (sex) {
  case SEX_MSB:
    ucp[0] = UCHAR(val >> 8);
    ucp[1] = UCHAR(val);
    return;
  case SEX_LSB:
    ucp[1] = UCHAR(val >> 8);
    ucp[0] = UCHAR(val);
    return;
  }
  die("bad sex in putCARD16: %c", sex);
}

/*
 * get a CARD16
 */
U16BITS getCARD16(U8BITS sex, char *cp)
{
  unsigned char *ucp = (unsigned char *)cp;

  switch (sex) {
  case SEX_MSB:
    return(ucp[1] | (ucp[0] << 8));
  case SEX_LSB:
    return(ucp[0] | (ucp[1] << 8));
  }
  die("bad sex in getCARD16: %c", sex);
  return(0); /* lint */
}

/*
 * put a CARD32
 */
void putCARD32(U8BITS sex, char *cp, U32BITS val)
{
  unsigned char *ucp = (unsigned char *)cp;

  switch (sex) {
  case SEX_MSB:
    ucp[0] = UCHAR(val >> 24);
    ucp[1] = UCHAR(val >> 16);
    ucp[2] = UCHAR(val >> 8);
    ucp[3] = UCHAR(val);
    return;
  case SEX_LSB:
    ucp[3] = UCHAR(val >> 24);
    ucp[2] = UCHAR(val >> 16);
    ucp[1] = UCHAR(val >> 8);
    ucp[0] = UCHAR(val);
    return;
  }
  die("bad sex in putCARD32: %c", sex);
}

/*
 * get a CARD32
 */
U32BITS getCARD32(U8BITS sex, char *cp)
{
  unsigned char *ucp = (unsigned char *)cp;

  switch (sex) {
  case SEX_MSB:
    return(ucp[3] | (ucp[2] << 8) | (ucp[1] << 16) | (ucp[0] << 24));
  case SEX_LSB:
    return(ucp[0] | (ucp[1] << 8) | (ucp[2] << 16) | (ucp[3] << 24));
  }
  die("bad sex in getCARD32: %c", sex);
  return(0); /* lint */
}

/*
 * initialise an IO structure
 */
void io_init(IO *iop, int fd_in, int fd_out)
{
  iop->tag[0] = '\0';
  iop->state = '?'; /* dummy state */
  iop->fd_in  = fd_in;
  iop->fd_out = fd_out;
  iop->flags = 0;
  iop->pos = 0;
  iop->has = 0;
  iop->needs = 0;
  iop->outq_first = NULL;
  iop->outq_last  = NULL;
}

/*
 * close an IO structure, including its file descriptors
 */
void io_close(IO *iop)
{
  MC *mcp, *next;

  /* close fds */
  shutdown(iop->fd_in, 2);
  close(iop->fd_in);
  shutdown(iop->fd_out, 2);
  close(iop->fd_out);
  /* get rid of MCs */
  mcp = iop->outq_first;
  while (mcp) {
    next = mcp->next;
    mc_old(mcp);
    mcp = next;
  }
}

/*
 * read something into the buffer
 */
void io_read(IO *iop)
{
  int want, got;

  /* preliminary cleanup */
  if (!BITTST(iop->flags, IO_READ)) return;
  if (iop->has == 0) iop->pos = 0;
  if ((iop->pos + iop->needs >  IO_BUFSIZE) &&
      (iop->has + iop->needs <= IO_BUFSIZE)) {
#ifdef DEBUG
    if (DEBUGGING(DBG_IO))
      dprintf(iop->tag, "move data pos=%d has=%d needs=%d",
	      iop->pos, iop->has, iop->needs);
#endif
    /* moving data inside the buffer to allow a full read */
    memmove(iop->buf, &iop->buf[iop->pos], iop->has);
    iop->pos = 0;
  }
  want = IO_BUFSIZE - iop->pos - iop->has;
  if (want == 0) {
    BITCLR(iop->flags, IO_READ);
    return;
  }
  /* really read */
  got = read(iop->fd_in, &iop->buf[iop->pos+iop->has], want);
#ifdef DEBUG
  if (DEBUGGING(DBG_IO))
    dprintf(iop->tag, "read pos=%d has=%d needs=%d want=%d got=%d",
	    iop->pos, iop->has, iop->needs, want, got);
#endif
  if (got == -1) {
    /* an error */
    if (errno == EWOULDBLOCK) {
#ifdef DEBUG
      if (DEBUGGING(DBG_IO) && DEBUGGING(DBG_WARNING))
	dprintf(iop->tag, "read aborted: %s", ERROR);
#endif
    } else {
      err_report(iop->tag, "read failed: ", ERROR, 0);
      BITSET(iop->flags, IO_ERROR);
    }
    return;
  } else if (got == 0) {
    /* end of file */
#ifdef DEBUG
    if (DEBUGGING(DBG_IO) && DEBUGGING(DBG_WARNING))
      dprintf(iop->tag, "read failed: EOF");
#endif
    BITSET(iop->flags, IO_ERROR);
    return;
  }
  /* successful read */
#ifdef DEBUG
  if (DEBUGGING(DBG_IO_READ)) hexdump(&iop->buf[iop->pos+iop->has], got);
#endif
  iop->has += got;
  if ((iop->has >= iop->needs) ||          /* enough data */
      (iop->pos + iop->has == IO_BUFSIZE)) /* buffer full */
    BITCLR(iop->flags, IO_READ);
}

/*
 * write as many MCs as possible
 */
void io_write(IO *iop)
{
  int want, wrote;
  int niov, try_again;
  MC *mcp, *next;

  /* preliminary cleanup */
  if (!BITTST(iop->flags, IO_WRITE)) return;
  if (!iop->outq_first) {
    BITCLR(iop->flags, IO_WRITE);
    return;
  }
top:
  want = 0;
  for (mcp=iop->outq_first,niov=0; mcp&&(niov<MAXIOV); mcp=mcp->next,niov++) {
    io_iovec[niov].iov_base = (caddr_t)mcp->base;
    io_iovec[niov].iov_len = mcp->len;
    want += mcp->len;
  }
  /* really write */
  wrote = writev(iop->fd_out, io_iovec, niov);
#ifdef DEBUG
  if (DEBUGGING(DBG_IO))
    dprintf(iop->tag, "write niov=%d want=%d wrote=%d", niov, want, wrote);
#endif
  if (wrote == -1) {
    /* an error */
    if (errno == EWOULDBLOCK) {
#ifdef DEBUG
      if (DEBUGGING(DBG_IO) && DEBUGGING(DBG_WARNING))
	dprintf(iop->tag, "write aborted: %s", ERROR);
#endif
    } else {
      err_report(iop->tag, "write failed: ", ERROR, 0);
      BITSET(iop->flags, IO_ERROR);
    }
    return;
  }
  /* successful write */
  try_again = (wrote == want);
  for (mcp=iop->outq_first; mcp&&(wrote>0); mcp=next) {
    next = mcp->next;
    if (wrote >= mcp->len) {
      /* chunk completely written */
#ifdef DEBUG
      if (DEBUGGING(DBG_IO_WRITE)) hexdump(mcp->base, mcp->len);
#endif
      wrote -= mcp->len;
      mc_old(mcp);
    } else {
      /* chunk partially written */
#ifdef DEBUG
      if (DEBUGGING(DBG_IO_WRITE)) hexdump(mcp->base, wrote);
#endif
      mcp->base += wrote;
      mcp->len  -= wrote;
      break;
    }
  }
  iop->outq_first = mcp;
  if (!iop->outq_first) {
    /* nothing else to write */
    BITCLR(iop->flags, IO_WRITE);
    iop->outq_last = NULL;
  } else {
    /* still something to write */
    if (try_again) goto top;
  }
}

/*
 * change state and needs, update flags
 */
void io_needs(IO *iop, char state, int needs)
{
  iop->state = state;
  iop->needs = needs;
  if (iop->has >= iop->needs)
    BITCLR(iop->flags, IO_READ);
  else
    BITSET(iop->flags, IO_READ);
}

/*
 * discard the current data in the input buffer
 */
void io_discard(IO *iop)
{
  int len;

  len = (iop->has < iop->needs) ? iop->has : iop->needs;
  /* update */
  iop->pos   += len;
  iop->has   -= len;
  iop->needs -= len;
}

/*
 * copy the current data to the output queue
 */
void io_flush(IO *iop)
{
  int len;
  MC *mcp;

  len = (iop->has < iop->needs) ? iop->has : iop->needs;
  /* allocate */
  mcp = mc_new(len);
  memmove(mcp->base, &iop->buf[iop->pos], len);
  /* update */
  iop->pos   += len;
  iop->has   -= len;
  iop->needs -= len;
  /* queue */
  if (iop->outq_last) iop->outq_last->next = mcp;
  else iop->outq_first = mcp;
  iop->outq_last = mcp;
  BITSET(iop->flags, IO_WRITE);
}

/*
 * add a memory chunk to the output queue
 */
void io_queue(IO *iop, MC *mcp)
{
  if (iop->outq_last) iop->outq_last->next = mcp;
  else iop->outq_first = mcp;
  iop->outq_last = mcp;
  BITSET(iop->flags, IO_WRITE);
}

/*
 * unblock a file descritor
 */
void io_unblock(int fd)
{
  int one = 1;

#ifdef FIOSNBIO
  ioctl(fd, FIOSNBIO, &one);
#else
#ifdef FIONBIO
  ioctl(fd, FIONBIO, &one);
#endif /* FIONBIO */
#endif /* FIOSNBIO */
}

/*
 * dummy SIGALRM handler to interrupt read and write
 */
static void io_alarm(int sig) { sig = sig; }

/*
 * read with a timeout
 */
int io_timeout_read(int timeout, int fd, void *buf, int len)
{
  struct sigaction new, old;
  int olderrno, res;

  memset((char *)&new, '\0', sizeof(new));
  new.sa_handler = io_alarm;
  sigaction(SIGALRM, &new, &old);
  alarm(timeout);
  res = read(fd, buf, len);
  olderrno = errno;
  alarm(0);
  sigaction(SIGALRM, &old, &new);
  errno = olderrno;
  return(res);
}

/*
 * write with a timeout
 */
int io_timeout_write(int timeout, int fd, void *buf, int len)
{
  struct sigaction new, old;
  int olderrno, res;

  memset((char *)&new, '\0', sizeof(new));
  new.sa_handler = io_alarm;
  sigaction(SIGALRM, &new, &old);
  alarm(timeout);
  res = write(fd, buf, len);
  olderrno = errno;
  alarm(0);
  sigaction(SIGALRM, &old, &new);
  errno = olderrno;
  return(res);
}
