/*+*****************************************************************************
*                                                                              *
* File: ident.c                                                                *
*                                                                              *
* Description: IDENT (RFC-1413) handling                                       *
*                                                                              *
**-****************************************************************************/

#ifndef __lint
char ident_vers[] = "@(#)ident.c	1.9	02/19/99	Written by Lionel Cons";
#endif /* __lint */

/******* Include Files ********************************************************/

#include "ident.h"
#include "util.h"
#include "io.h"
#include "sb.h"

#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <ctype.h>

/******* Constants ************************************************************/

/******* Macros ***************************************************************/

#define SKIP_SPACES	while (*cp &&  isspace((int)*cp)) cp++
#define SKIP_WORD	while (*cp && !isspace((int)*cp)) cp++
#define SKIP_TO_COLON	while (*cp && *cp != ':')    cp++; if (*cp) cp++

/******* External Stuff *******************************************************/

/******* Local Stuff **********************************************************/

char *ident_name(int fd)
{
  struct sockaddr_in laddr, raddr, ident_laddr, ident_raddr;
  int res, len, ident_fd;
  char *cp, *name;

  /* init */
  ident_fd = -1;
  len = sizeof(raddr);
  if (getpeername(fd, (struct sockaddr *)&raddr, &len) < 0) {
#ifdef DEBUG
    if (DEBUGGING(DBG_IDENT))
      dprintf("iden", "can't getpeername on fd %d: %s", fd, ERROR);
#endif
    goto unavailable;
  }
  len = sizeof(laddr);
  if (getsockname(fd, (struct sockaddr *)&laddr, &len) < 0) {
#ifdef DEBUG
    if (DEBUGGING(DBG_IDENT))
      dprintf("iden", "can't getsockname on fd %d: %s", fd, ERROR);
#endif
    goto unavailable;
  }

  /* create socket */
  ident_fd = socket(AF_INET, SOCK_STREAM, 0);
  if (ident_fd < 0) {
#ifdef DEBUG
    if (DEBUGGING(DBG_IDENT))
      dprintf("iden", "can't create IDENT socket: %s", ERROR);
#endif
    goto unavailable;
  }
  
  /* bind socket */
  memset((char *)&ident_laddr, '\0', sizeof(ident_laddr));
  ident_laddr.sin_family = AF_INET;
  ident_laddr.sin_addr = laddr.sin_addr;
  ident_laddr.sin_port = 0;
  res = bind(ident_fd, (void *)&ident_laddr, sizeof(ident_laddr));
  if (res < 0) {
#ifdef DEBUG
    if (DEBUGGING(DBG_IDENT))
      dprintf("iden", "can't bind IDENT socket on %s:0: %s",
	      inet_ntoa(laddr.sin_addr), ERROR);
#endif
    goto unavailable;
  }

  /* connect socket */
  memset((char *)&ident_raddr, '\0', sizeof(ident_raddr));
  ident_raddr.sin_family = AF_INET;
  ident_raddr.sin_addr = raddr.sin_addr;
  ident_raddr.sin_port = htons(113);
  res = connect(ident_fd, (void *)&ident_raddr, sizeof(ident_raddr));
  if (res < 0) {
#ifdef DEBUG
    if (DEBUGGING(DBG_IDENT))
      dprintf("iden", "can't connect IDENT socket to %s:113: %s",
	      inet_ntoa(raddr.sin_addr), ERROR);
#endif
    goto unavailable;
  }

  /* send query */
  sprintf(sb_line, "%d , %d", ntohs(raddr.sin_port), ntohs(laddr.sin_port));
#ifdef DEBUG
  if (DEBUGGING(DBG_IDENT)) dprintf("iden", "ident query is '%s\\r\\n'", sb_line);
#endif
  strcat(sb_line, "\r\n");
  res = io_timeout_write(3, ident_fd, sb_line, strlen(sb_line));
  res -= strlen(sb_line);
  if (res < 0) {
#ifdef DEBUG
    if (DEBUGGING(DBG_IDENT))
      dprintf("iden", "can't send IDENT query: %s", ERROR);
#endif
    goto error;
  }

  /* read reply */
  res = io_timeout_read(3, ident_fd, sb_line, SB_LINE_LENGTH-1);
  if (res <= 0) {
#ifdef DEBUG
    if (DEBUGGING(DBG_IDENT))
      dprintf("iden", "can't read IDENT reply: %s", ERROR);
#endif
    goto error;
  }
#ifdef DEBUG
  close(ident_fd);
  ident_fd = -1;
  sb_line[res] = '\0';
  if (DEBUGGING(DBG_IDENT)) {
    dprintf("iden", "ident reply is %d bytes", res);
    hexdump(sb_line, res);
  }
#endif
  
  /* parse reply: 'lport , fport : USERID : opsys : name\n\r' */
  cp = sb_line;
  SKIP_TO_COLON;
  SKIP_SPACES;
  if (strncasecmp(cp, "userid", 6)) goto error;
  cp += 6;
  SKIP_SPACES;
  if (*cp != ':') goto error;
  cp++;
  SKIP_TO_COLON;
  SKIP_SPACES;
  if (*cp == '\0') goto error;
  name = cp;
  SKIP_WORD;
  *cp = '\0';

  /* clean the name */
  cp = name;
  while (*cp) {
    if (isprint((int)*cp))
      *cp = tolower((int)*cp);
    else
      *cp = '_';
    cp++;
  }
  if (strlen(name) > 8)
    return(str_ncopy(name, 8));
  else
    return(str_copy(name));

  /* can't connect to the ident server, maybe there's none */
unavailable:
  if (ident_fd >= 0) close(ident_fd);
  return(NULL);

  /* error while connected to the ident server */
error:
  if (ident_fd >= 0) close(ident_fd);
  return(str_copy("???"));
}
